from __future__ import with_statement
import xml.dom.minidom
import os
import shutil
import hashlib
import sys
import optparse


def checkAndInstallCheckoutHook(base_dir, project_dir):
	hook_dir_path = os.path.join(base_dir, '.git', 'hooks')
	checkout_hook_path = os.path.join(hook_dir_path, 'post-checkout')
	is_update = False

	# Remove old version of the hook, if it hasn't been modified.
	if os.path.isfile(checkout_hook_path):
		hash = hashlib.md5(file(checkout_hook_path).read()).hexdigest()
		# print hash
		if hash == "c9639203583f173f7719f5aa599c61b1" or hash == "9f86c113c51f762a01530130d9ebd2b5":
			os.remove(checkout_hook_path)
			is_update = True
	if os.path.isdir(hook_dir_path) and not os.path.isfile(checkout_hook_path):
		shutil.copy(os.path.join(project_dir, 'git_checkout_hook.sh'), checkout_hook_path)
		if is_update:
			print "Git checkout hook updated."
		else:
			print "Git checkout hook installed."


class SourcesCollection:
	def __init__(self, config):
		# Read list of source files generated by core.
		self.sources = SourcesCollection.loadSourcesConfigurations(config.base_dir)
		self.sources_config = self.sources['plain' if config.no_jumbo else 'jumbo']
		if config.no_jumbo:
			self.sources_all = sorted(set(self.sources['plain']['all']))
		else:
			self.sources_all = sorted(set(self.sources['plain']['all']) | set(self.sources['jumbo']['all']))

	@staticmethod
	def loadSourcesConfigurations(rootDir):
		"""Read list of source files generated by core."""
		sources = {}
		for type in ('plain', 'jumbo'):
			sources[type] = {}
			for configuration in ('all', 'component_framework', 'component_plugin', 'component_mapi', 'nopch', 'pch', 'pch_jumbo', 'pch_system_includes', 'component_opera_autoupdatechecker'):
				sources[type][configuration] = SourcesCollection.loadSourcesFile(rootDir, type, configuration)
		return sources

	@staticmethod
	def loadSourcesFile(root, type, name):
		"""
		Loads the contents of the sourceslist with the specified name.
		@param type can be 'plain' or 'jumbo'
		@param name can be one of 'all', 'nopch', 'pch', 'pch_jumbo' or
		'pch_system_includes'.
		@return a list of source-files.
		"""
		filename = os.path.join(root, 'modules', 'hardcore', 'setup', type, 'sources', 'sources.%s' % name)
		if not os.path.exists(filename):
			print "Opening '%s' failed. Possibly checkout hook is outdated. Problem should resolve itself the next time solution is built." % filename
			return []

		with open(filename) as f:
			return f.read().splitlines()

	def getSourceComponent(self, source):
		if source in self.sources_config['component_framework']:
			return "component_framework"
		if source in self.sources_config['component_plugin']:
			return "component_plugin"
		if source in self.sources_config['component_mapi']:
			return "component_mapi"
		if source in self.sources_config['component_opera_autoupdatechecker']:
			return "component_opera_autoupdatechecker"
		return None

	def getSourcePch(self, source):
		if source in self.sources_config['pch']:
			return "core/pch.h"
		if source in self.sources_config['nopch']:
			return None
		if source in self.sources_config['pch_jumbo']:
			return "core/pch_jumbo.h"
		elif source in self.sources_config['pch_system_includes']:
			return "core/pch_system_includes.h"
		return "core/pch.h"

	def isSourceIncludedInBuild(self, source):
		return (source in self.sources_config['pch']
			or source in self.sources_config['nopch']
			or source in self.sources_config['pch_jumbo']
			or source in self.sources_config['pch_system_includes'])


class ProjectDocument:
	def __init__(self, project_file, project_filters):
		self.vcxdom = xml.dom.minidom.parse(project_file)
		self.filterdom = xml.dom.minidom.parse(project_filters)
		self.project_file = project_file
		self.project_filters_file = project_filters
		self.filterList = []

	def preprocessTemplate(self, project_name):
		"""
		Processes parsed DOM tree and removes all the elements that have
		"exclude_from" attribute that matches name of the current project.
		"""
		elements = self.vcxdom.getElementsByTagName('*')

		for element in elements:
			if element.hasAttribute('exclude_from'):
				# Exclude from the project if attribute value matches project name.
				if element.getAttribute('exclude_from') == project_name:
					# Remove indentation left after removing the element. Not strictly necessary.
					if element.previousSibling.nodeType == xml.dom.Node.TEXT_NODE:
						element.parentNode.removeChild(element.previousSibling)

					# Remove the element.
					element.parentNode.removeChild(element)

				# Remove the attribute.
				element.removeAttribute('exclude_from')

	def createPadding(self, spaces):
		"""Creates a text node with line-break followed by a given number of spaces"""
		return self.vcxdom.createTextNode("\n".ljust(spaces + 1))

	def createElement(self, elementName):
		return self.vcxdom.createElement(elementName)

	def createFilterTextNode(self, data):
		return self.filterdom.createTextNode(data)

	def createFilterElement(self, elementName):
		return self.filterdom.createElement(elementName)

	def toXml(self):
		return "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" + self.vcxdom.documentElement.toxml("utf-8")

	def filtersToXml(self):
		return "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" + self.filterdom.documentElement.toxml("utf-8")

	def addFile(self, src_node, filter_node, file_name, pch_name, exclude_from_build, optimize_speed):
		OPTIMIZE_SPEED_CONDITION = "'$(Configuration)'=='Release'"

		rel_name = file_name
		# Note: we have to use a backslash in the filenames in the
		# solution file and not the local path separator:
		if os.sep != '\\': rel_name = rel_name.replace(os.sep, '\\')

		src_node.appendChild(self.createPadding(4))
		fnode = self.createElement("ClCompile")

		fnode.setAttribute("Include", rel_name)

		if pch_name != "core/pch.h":
			if pch_name is None:
				fnode.appendChild(self.createPadding(6))
				preproc = self.createElement("PreprocessorDefinitions")
				preproc.appendChild(self.createFilterTextNode("ALLOW_SYSTEM_INCLUDES;%(PreprocessorDefinitions)"))
				fnode.appendChild(preproc)

			if pch_name is None or pch_name == "core/pch_system_includes.h":
				fnode.appendChild(self.createPadding(6))
				pchnode = self.createElement("PrecompiledHeader")
				pchnode.appendChild(self.createFilterTextNode(""))
				fnode.appendChild(pchnode)
			elif pch_name == "core/pch_jumbo.h":
				fnode.appendChild(self.createPadding(6))
				pchfilenode = self.createElement("PrecompiledHeaderFile")
				pchfilenode.appendChild(self.createFilterTextNode(pch_name))
				fnode.appendChild(pchfilenode)

		if not exclude_from_build and optimize_speed:
			fnode.appendChild(self.createPadding(6))
			optnode = self.createElement("Optimization")
			optnode.setAttribute("Condition", OPTIMIZE_SPEED_CONDITION)
			optnode.appendChild(self.createFilterTextNode("MaxSpeed"))
			fnode.appendChild(optnode)

		# Exclude files that should not be compiled but still in the project.
		if exclude_from_build:
			fnode.appendChild(self.createPadding(6))
			excludefilenode = self.createElement("ExcludedFromBuild")
			excludefilenode.appendChild(self.createFilterTextNode("true"))
			fnode.appendChild(excludefilenode)

		# If any childs were added, indent closing tag properly.
		if fnode.childNodes.length:
			fnode.appendChild(self.createPadding(4))

		# Add the file
		src_node.appendChild(fnode)

		# Update the filter node and filter list
		filter = rel_name.replace("src\\", "")
		filter = filter.replace("..\\", "")
		idx = filter.rfind("\\")
		if idx > 0:
			idx_temp = idx
			filter_temp = filter
			while idx_temp > 0:
				filter_temp = filter_temp[0:idx_temp]
				self.filterList.append("Source Files\\"+filter_temp)
				idx_temp = filter_temp.rfind("\\")

			filter = filter[0:idx]
			filter = "Source Files\\" + filter
		else:
			filter = "Source Files"

		fnode = self.createFilterElement("ClCompile")
		fnode.setAttribute("Include", rel_name)
		fnamenode = self.createFilterElement("Filter")
		fnamenode.appendChild(self.createFilterTextNode(filter))
		fnode.appendChild(self.createPadding(6))
		fnode.appendChild(fnamenode)
		fnode.appendChild(self.createPadding(4))

		filter_node.appendChild(self.createPadding(4))
		filter_node.appendChild(fnode)


class ProjectDotUserDocument:
	def __init__(self, file_path):
		self.file_path = file_path
		self.dom_document = xml.dom.minidom.parse(file_path)

	def isEmpty(self):
		return len(self.dom_document.getElementsByTagName("LocalDebuggerWorkingDirectory")) == 0


class ProjectTask:
	def initDefaults(self):
		self.name = None
		self.template = None
		self.include_non_component = False
		self.components = []
		self.update_vcxproj_and_filters = True
		self.update_dotuser = True
		self.preprocess_template = True

	def __init__(self, field_values):
		self.initDefaults()
		for field_name in field_values:
			self.__dict__[field_name] = field_values[field_name]

	def updateProjectAndFilters(self, config, project, sources_collection):
		filterIgnoreList = []
		vcxitems = project.vcxdom.getElementsByTagName("ItemGroup")
		# Remove all old files
		while vcxitems[1].firstChild:
			vcxitems[1].removeChild(vcxitems[1].firstChild)

		filteritems = project.filterdom.getElementsByTagName("ItemGroup")

		# Remove all old files and filters
		while filteritems[1].firstChild:
			filteritems[1].removeChild(filteritems[1].firstChild)

		# Clear the filter list
		filter = filteritems[0].firstChild
		while filter:
			nf = filter.nextSibling
			if filter.nodeType == filter.ELEMENT_NODE and filter.getAttribute("Include").count("\\") > 0:
				if len(filter.getElementsByTagName("UniqueIdentifier")) > 0:
					# Add the manually added filter to the filter ignore list
					# No need to iterate up since the parent filters must also be manually added
					filterIgnoreList.append(filter.getAttribute("Include"))
				else:
					# Remove the filter added by the script
					if filter.previousSibling.nodeType == filter.TEXT_NODE:
						filteritems[0].removeChild(filter.previousSibling)
					filteritems[0].removeChild(filter)
			filter = nf

		for source in sources_collection.sources_all:
			module_group = source.split('/')[0]
			module_name = source.split('/')[1]
			full_module_name = os.path.join(module_group, module_name)
			if not config.isDisabledModule(module_name):
				component = sources_collection.getSourceComponent(source)
				if (component is None) and (not self.include_non_component):
					continue
				if (component is not None) and (len(self.components) > 0) and (not component in self.components):
					continue
				path = os.path.join(config.REL_PATH_PREFIX, *source.split('/'))
				pch = sources_collection.getSourcePch(source)
				exclude_from_build = not sources_collection.isSourceIncludedInBuild(source)
				optimize_speed = full_module_name in config.optimize_speed_modules
				project.addFile(vcxitems[1], filteritems[1], path, pch, exclude_from_build, optimize_speed)

		# Add the final padding in the vcxproj
		vcxitems[1].appendChild(project.createPadding(2))

		# List all filters
		filterList = project.filterList
		filterList.sort()
		filterList.append("")
		for f in range(len(filterList)-1):
			if filterList[f] != filterList[f+1] and filterList[f] not in filterIgnoreList:
				filteritems[0].appendChild(project.createPadding(4))
				fnode = project.createFilterElement("Filter")
				fnode.setAttribute("Include", filterList[f])
				filteritems[0].appendChild(fnode)

		# Add the final padding in the filters
		filteritems[0].appendChild(project.createPadding(2))
		filteritems[1].appendChild(project.createPadding(2))

		changed = False
		projfile_data = project.toXml()
		with open(project.project_file, "r") as projfile:
			if projfile.read() != projfile_data:
				changed = True

		if not changed:
			return

		with open(project.project_file, "w") as projfile:
			print "Writing %s..." % project.project_file
			projfile.write(projfile_data)
		with open(project.project_filters_file, "w") as filters_file:
			print "Writing %s..." % project.project_filters_file
			filters_file.write(project.filtersToXml())
		print "ALERT: a new version of %s has been created and must be reloaded into Visual Studio" % project.project_file

	def process(self, config, sources_collection):
		# Project file path.
		project_file = os.path.join(config.project_dir, self.name + ".vcxproj")
		project_filters = project_file + ".filters"
		project_dotuser_file = project_file + ".user"

		# Project template path.
		project_template = os.path.join(config.project_dir, self.template + ".vcxproj.template")
		project_filters_template = os.path.join(config.project_dir, self.template + ".vcxproj.filters.template")
		project_dotuser_template = os.path.join(config.project_dir, self.template + ".vcxproj.user.template")

		# Create project files from the templates.
		if self.update_vcxproj_and_filters:
			if os.path.isfile(project_template) and not os.path.isfile(project_file):
				shutil.copy(project_template, project_file)
			if (os.path.isfile(project_filters_template)
					and (not os.path.isfile(project_filters) or (os.path.getsize(project_filters) == 0))):
				shutil.copy(project_filters_template, project_filters)
		if (self.update_dotuser
				and os.path.isfile(project_dotuser_template)
				and (not os.path.isfile(project_dotuser_file) or ProjectDotUserDocument(project_dotuser_file).isEmpty())):
			shutil.copy(project_dotuser_template, project_dotuser_file)
			# "touch" a vcxproj to force reload
			if os.path.isfile(config.solution_file):
				os.utime(config.solution_file, None)

		project = None
		if os.path.isfile(project_file) and os.path.isfile(project_filters):
			project = ProjectDocument(project_file, project_filters)

		if self.preprocess_template:
			project.preprocessTemplate(self.name)

		if self.update_vcxproj_and_filters and project is not None:
			self.updateProjectAndFilters(config, project, sources_collection)

class Config:
	"""Relative path used for generated projects."""
	REL_PATH_PREFIX = "..\\..\\"

	def __init__(self):
		# Base directory of the project file.
		self.project_dir = os.path.dirname(sys.argv[0])
		# Root directory of the repo.
		self.base_dir = os.path.normpath(os.path.join(self.project_dir, "..", ".."))

		options = Config.parseOptions()
		self.no_jumbo = options.no_jumbo

		self.solution_file = os.path.join(self.project_dir, "Opera_vc2010.sln")
		self.optimize_speed_modules = Config.loadOptimizeSpeedModules(self.project_dir)

	def isDisabledModule(self, module_name):
		return module_name in ("coredoc", "build", "mac", "paxage", "posix", "quix", "unix", "utilix", "viewix", "x11api", "posix_ipc")

	@staticmethod
	def parseOptions():
		option_parser = optparse.OptionParser(
			usage="Usage: %prog [options]",
			description="".join(
				["Generates project files for Visual Studio from templates"]))

		option_parser.add_option(
			"--no_jumbo", action="store_true",
			help=" ".join(
				["to disable compiling with jumbo compile units.",
				 "\n"
				 "Note: the jumbo-compile-units are generated by operasetup.py.",
				 "This script only enables or disables the jumbo compile units.",
				 "To enable jumbo compilation, don't use the --no_jumbo argument."]))

		option_parser.set_defaults(no_jumbo=False)
		(options, args) = option_parser.parse_args()
		return options

	@staticmethod
	def loadOptimizeSpeedModules(project_dir):
		optimize_txt = os.path.join(project_dir, "optimize_speed.txt")
		if not os.path.isfile(optimize_txt):
			return []
		with open(optimize_txt) as f:
			return f.read().splitlines()

################
# main program #
################

config = Config()

checkAndInstallCheckoutHook(config.base_dir, config.project_dir)

projects = [
	ProjectTask({
		# project name
		"name" : "Opera",
		# project template
		"template" : "Opera",
		# compile non-component files
		"include_non_component" : True,
		# compile files belonging to the specified component
		"components" : ["component_framework", "component_mapi"],
		# pre-process plugin wrapper template
		"preprocess_template" : False,
	}),
	ProjectTask({
		"name" : "plugin_wrapper",
		"template" : "plugin_wrapper",
		"components" : ["component_framework", "component_plugin"]
	}),
	ProjectTask({
		"name" : "plugin_wrapper_32",
		"template" : "plugin_wrapper",
		"components" : ["component_framework", "component_plugin"]
	}),
	ProjectTask({
		"name" : 'OperaMAPI',
		"template" : 'OperaMAPI',
		"components" : ["component_mapi", "component_framework"]
	}),
	ProjectTask({
		"name" : 'OperaMAPI_32',
		"template" : 'OperaMAPI',
		"components" : ["component_mapi", "component_framework"]
	}),
	ProjectTask({
		"name" : 'desktop_starter/desktop_starter',
		"template" : 'desktop_starter/desktop_starter',
		'update_vcxproj_and_filters' : False,
		"preprocess_template" : False,
	})
]

# Process projects.
sources_collection = SourcesCollection(config)
for task in projects:
	task.process(config, sources_collection)
