/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*-
**
** Copyright (C) 2011 Opera Software AS.  All rights reserved.
**
** This file is part of the Opera web browser.  It may not be distributed
** under any circumstances.
*/

/**
 * @file op_functions.h
 *
 * Substitution of standard C calls with op_ calls, if needed.
 *
 * This file introduces an exception to Opera Core op_ prefix rule.
 * This exception is only introduced for libopeay module, because it
 * contains mostly 3rd party OpenSSL code and we can't force OpenSSL
 * developers to adopt our op_ prefix rule. If this exception will
 * not work on all supported platforms - it will be removed.
 *
 * If you are curious and want more info about the problem, please read
 * the relevant discussion in developer@opera.com maillist, August 2011,
 * subject "op_ prefix". Here is the a copy of the exception proposal
 * from that discussion:
 *

<quote>
From: "Alexei Khlebnikov" <alexei.khlebnikov@opera.com>
Subject: Re: [Developer] op_ prefix
Date: Mon, 08 Aug 2011 14:43:44 +0200

...

I want to introduce an exception to the op-prefix rule for at least some
3rd-party code, because we can't make upstream developers to adopt our
op-prefix rule. I want to apply this exception to libopeay (OpenSSL),
but it can also be useful for lea_malloc, libfreetype and zlib,
if their owners will want it.

We now have op_calls everywhere in libopeay. It means we have a lot of local
patches that only exist to satisfy our op-prefix rule. We currently have
3.7 MB of local patches between "vanilla" OpenSSL and libopeay
("diff -Nura" with default 3 lines of context). Of course, our local patches do not consist entirely of changing non-op_calls to op_calls, but such changes are a big share of all patches, more than half of them.

I am now upgrading OpenSSL and have a lot merge conflicts because of that.
Also, I need to replace newly-added non-op_calls by op_calls. I can write
a script for such replacements, but I am upgrading directory-by-directory,
so I'll need to run it many times anyway, and this script will not save me
from all those manual merge conflict resolutions.

Thus I want to make a header, local to libopeay, that will contain blocks like

    #if SYSTEM_MEMSET == NO
    # undef memset
    # define memset op_memset
    #endif

for every function which is a part of system system and has a SYSTEM_* define.
I am going to include this header into every (autogenerated) .cpp file
of libopeay, but none of its .h files, thus other modules won't be "poisoned".
Those rare system function calls from .h files, if they exist, will be
converted to op_calls anyway, so that other modules including headers
from libopeay will be able to compile. Other arrangements can be done,
the point is to remove as many local patches as possible, even if it won't be
possible to remove all of them.

Why it should work for all projects. All projects have to define SYSTEM_MEMSET
correctly to YES or NO. Otherwise they'll get a compilation error immediately,
according to this code in modules/hardcore/base/system.h:


    #if !defined SYSTEM_MEMSET || (SYSTEM_MEMSET != YES && SYSTEM_MEMSET != NO)
    #   error "You need to decide if you want SYSTEM_MEMSET or not."
    #   error " The system has the following function:"
    #   error ""
    #   error " void* memset(void* s1, int n, size_t nbytes);"
    #   error ""
    #   error " If SYSTEM_MEMSET==NO then the macro op_memset need not be defined."
    #   error ""
    #endif // SYSTEM_MEMSET

And these definitions will be visible in all the Core code. Thus my code

    #if SYSTEM_MEMSET == NO
    # undef memset
    # define memset op_memset
    #endif

will behave the following way:

1) If system memset is used and SYSTEM_MEMSET == YES:
#undef and #define will be ignored, memset will be defined to whatever it was
previously defined on the system.

2) If system memset is not used and SYSTEM_MEMSET == NO:
memset will be defined as op_memset, which in turn can be defined as any
(weird) platform-specific function or Opera's implementation of this function
can be taken.

Regarding Eddy's point about exotic functions like wcsnicoll. I haven't seen
exotic function calls in OpenSSL. It is mostly functions from string.h: str*()
and mem*(). So what will happen if such functions still exist or will be
introduced? As I said, I am only going to introduce the mentioned code blocks
for functions in system system, which have corresponding SYSTEM_* define.
So those exotic functions are not affected by my proposal and their linking
will be solved another way, as it would be without this proposal.

Regarding Jakub's point about memory allocation. AFAIU, you are talking about
op_malloc and op_free. When using own memory allocator, you define,
for example, op_malloc as own_malloc and op_free as own_free. You don't use
system's malloc and free and have SYSTEM_MALLOC == NO. In this case "plain"
malloc will be substituted by the preprocessor by op_malloc and then
by own_malloc. So it's transparent. OP_NEW and OP_DELETE are unaffected
by this proposal, because OpenSSL is pure C code and OP_NEW/OP_DELETE
do not have corresponding SYSTEM_* macros.

I'd like to repeat, I want to introduce this exception only for one module,
containing 3rd-party code. Not for the whole Core. If it won't work, we can
switch to "total op-prefix" again.
</quote>

 * Documentation update 2012-04-13:

It turned out that YES, NO and SYSTEM_* defines are undefined in the end
of modules/hardcore/base/system.h, which is included by core/pch.h,
i.e. before this file (op_functions.h). Thus we can't rely on SYSTEM_*
defines. But before each SYSTEM_X is undefined, each SYSTEM_X is converted
into the corresponding HAVE_X:

#if SYSTEM_MEMSET == YES
#	define HAVE_MEMSET
#endif // SYSTEM_MEMSET == YES

Thus we can use HAVE_* defines.

 *
 * @author Alexei Khlebnikov <alexeik@opera.com>
 *
 */

#ifdef OP_FUNCTIONS_H
#error This file must be included exactly once, from modules/libopeay/core_includes.h.
#endif
#define OP_FUNCTIONS_H

#ifndef HAVE_ATOI
# undef  atoi
# define atoi op_atoi
#endif

#ifndef HAVE_GMTIME
# undef  gmtime
# define gmtime op_gmtime
#endif

#ifndef HAVE_MEMCMP
# undef  memcmp
# define memcmp op_memcmp
#endif

#ifndef HAVE_MEMCPY
# undef  memcpy
# define memcpy op_memcpy
#endif

#ifndef HAVE_MEMMOVE
# undef  memmove
# define memmove op_memmove
#endif

#ifndef HAVE_MEMSET
# undef  memset
# define memset op_memset
#endif

#ifndef HAVE_SSCANF
# undef  sscanf
# define sscanf op_sscanf
#endif

#ifndef HAVE_STRCHR
# undef  strchr
# define strchr op_strchr
#endif

#ifndef HAVE_STRCMP
# undef  strcmp
# define strcmp op_strcmp
#endif

#ifndef HAVE_STRCPY
# undef  strcpy
# define strcpy op_strcpy
#endif

#ifndef HAVE_STRICMP
# undef  stricmp
# define stricmp op_stricmp
# undef  strcasecmp
# define strcasecmp op_stricmp
#endif

#ifndef HAVE_STRLEN
# undef  strlen
# define strlen op_strlen
#endif

#ifndef HAVE_STRNCMP
# undef  strncmp
# define strncmp op_strncmp
#endif

#ifndef HAVE_STRNCPY
# undef  strncpy
# define strncpy op_strncpy
#endif

#ifndef HAVE_STRNICMP
# undef  strnicmp
# define strnicmp op_strnicmp
# undef  strncasecmp
# define strncasecmp op_strnicmp
#endif

#ifndef HAVE_STRSTR
# undef  strstr
# define strstr op_strstr
#endif

#ifndef HAVE_STRTOUL
# undef  strtoul
# define strtoul op_strtoul
#endif

#ifndef HAVE_TOLOWER
# undef  tolower
# define tolower op_tolower
#endif
