#!/usr/bin/env python
#
# Copyright (C) 1995-2012 Opera Software ASA.  All rights reserved.
#
# This file is part of the Opera web browser.
# It may not be distributed under any circumstances.

#
# Parse a viewers declaration file, like 'viewers/module.viewers' and
# create the source code files used to initialize the viewers database.
#

# TODO: 1) Support deprecation
#       2) Allow including of module viewer declarations from other places
#       3) Make it possible to cherry-pick in a project delivery

import os
import os.path
import sys
import re
import datetime

#
# There is one Viewer object for each content type declared in the
# viewers file.
#
class Viewer(object):

    def __init__(self, name, owner):
        self.name = name
        self.prop = {}

    def CreateIf(self, f):
        if 'Depends on' in self.prop and self.prop['Depends on'] != 'nothing':
            print >> f, "#if %s" % self.prop['Depends on']

    def CreateEndif(self, f):
        if 'Depends on' in self.prop and self.prop['Depends on'] != 'nothing':
            print >> f, "#endif // %s" % self.prop['Depends on']

#
# The Viewers object is used to hold a collection of Viewer objects, and
# contains all the methods to manipulate such a collection.
#
class Viewers(object):

    def __init__(self):
        self.viewers = []
        self.comment = re.compile(r'^\s*(\#.*)?$')
        self.viewerheader = re.compile(r'^(VIEWER_\S+)\s+(\S+)$')
        self.viewerdata = re.compile(r'^\s*(Action|Type|CType|Ext|Depends\son|Web\shandler\sallowed|Allow\sany\sextension|Container\stype):\s+(\S.*)$')

    def Parse(self, file):
        f = open(file, 'r')
        while True:
            line = f.readline()
            if not line: break
            if self.comment.match(line): continue
            m = self.viewerheader.match(line)
            if m:
                current = Viewer(m.group(1), m.group(2))
                self.viewers.append(current)
                continue
            m = self.viewerdata.match(line)
            if m:
                prp = m.group(1)
                val = m.group(2)
                if not current:
                    print >> sys.stderr, "Property %s needs viewer" % prp
                    sys.exit(1)
                if val.strip() == "VIEWER_REDIRECT":
                    print >> sys.stderr, "Invalid Action: %s\n" % val.strip()
                    sys.exit(1)
                if prp.strip() == "Web handler allowed" and val.strip() != '0' and val.strip() != '1':
                    print >> sys.stderr, "Invalid Web handler allowed value: %s\n" % val.strip()
                    sys.exit(1)
                if prp.strip() == "Allow any extension" and val.strip() != '0' and val.strip() != '1':
                    print >> sys.stderr, "Invalid Allow any extension value: %s\n" % val.strip()
                    sys.exit(1)
                current.prop[prp] = val.strip()
                continue
            print >> sys.stderr, "Invalid line:\n", line
            sys.exit(1)
            break
        f.close()

    def CreateEnums(self, file, macro):
        now = datetime.datetime.now()
        f = open(file, "w")
        print >> f, '''/*
 * Copyright (C) 1995-%s Opera Software ASA.  All rights reserved.
 *
 * This file is part of the Opera web browser.
 * It may not be distributed under any circumstances.
 */

/*
 * Please do not modify this file.
 *
 * This file is automatically generated by viewers.py
 *
 * To change this file, change modules/viewers/module.viewers, and run
 * the script modules/viewers/src/viewers.py
 */

#ifndef %s
#define %s

enum ViewersEnum {''' % (now.year, macro, macro)

        for v in self.viewers:
            v.CreateIf(f)
            print >> f, "\t%s," % v.name
            v.CreateEndif(f)
        print >> f, "\tVIEWER_LAST"
        print >> f, "};\n\n#endif // %s" % macro
        f.close()

    def CreateData(self, file):
        now = datetime.datetime.now()
        f = open(file, "w")
        print >> f, '''/*
 * Copyright (C) 1995-%s Opera Software ASA.  All rights reserved.
 *
 * This file is part of the Opera web browser.
 * It may not be distributed under any circumstances.
 */

/*
 * Please do not modify this file.
 *
 * This file is automatically generated by viewers.py
 *
 * To change this file, change modules/viewers/module.viewers, and run
 * the operasetup.py script.
 */

#ifdef HAS_COMPLEX_GLOBALS

const ViewerTypes Viewers::defaultOperaViewerTypes[] =
{
''' % (now.year)
        for v in self.viewers:
            if not 'Type' in v.prop:
                print >> sys.stderr, "Viewer %s has no Type" % v.name
                sys.exit(1)
            if not 'Ext' in v.prop:
                print >> sys.stderr, "Viewer %s has no Ext" % v.name
                sys.exit(1)
            if not 'Action' in v.prop:
                print >> sys.stderr, "Viewer %s has no Action" % v.name
                sys.exit(1)
            if not 'CType' in v.prop:
                print >> sys.stderr, "Viewer %s has no CType" % v.name
                sys.exit(1)
            if not 'Web handler allowed' in v.prop:
                print >> sys.stderr, "Viewer %s has no Web handler allowed" % v.name
                sys.exit(1)
            if not 'Allow any extension' in v.prop:
                print >> sys.stderr, "Viewer %s has no Allow any extension" % v.name
                sys.edit(1)
            if v.prop['Web handler allowed'] == '0' and v.prop['Action'] == 'VIEWER_WEB_APPLICATION':
                print >> sys.stderr, "Viewer %s has contradictory configuration of \'Action\' and \'Web handler allowed\' settings" % v.name
                sys.exit(1)
            if not 'Container type' in v.prop:
                print >> sys.stderr, "Viewer %s has no Container type" % v.name
                sys.exit(1)
            if v.prop['Container type'] == 'none':
                container = 'VIEWER_NULL'
            else:
                container = v.prop['Container type']
            v.CreateIf(f)
            decl = "\t{ " + v.prop['Type'] + ", " + v.prop['Ext'] + ", "
            decl += v.prop['Action'] + ", FROM_RANGED_ENUM(URLContentType, " + v.prop['CType'] + "), "
            decl += container + ", "
            if v.prop['Web handler allowed'] == '1':
	            decl += "true"
            else:
                decl += "false"
            decl += ", "
            if v.prop['Allow any extension'] == '1':
	            decl += "true"
            else:
                decl += "false"
            decl += " },"
            print >> f, decl
            v.CreateEndif(f)
        print >> f, '''};

#else //HAS_COMPLEX_GLOBALS

void Viewers::init_defaultOperaViewerTypes()
{
	ViewerTypes *local = defaultOperaViewerTypes;
	int i = 0;
'''
        for v in self.viewers:
            v.CreateIf(f)
            decl = "\tlocal[i].type=" + v.prop['Type'] + ", "
            decl += "local[i].ext=" + v.prop['Ext'] + ", "
            decl += "local[i].action=" + v.prop['Action'] + ", "
            decl += "local[i].ctype=FROM_RANGED_ENUM(URLContentType," + v.prop['CType'] + "), "
            decl += "local[i].container=" + container + ";"
            decl += "local[i].web_handler_allowed="
            if v.prop['Web handler allowed'] == '1':
                decl += "true;"
            else:
                decl += "false;"
            decl += "local[i++].allow_any_extension="
            if v.prop['Allow any extension'] == '1':
                decl += "true;"
            else:
                decl += "false;"
            print >> f, decl
            v.CreateEndif(f)
        print >> f, '''
	OP_ASSERT(i == defaultOperaViewerTypes_SIZE);
}

#endif //HAS_COMPLEX_GLOBALS'''
        f.close()

#
# Main program: Parse the module.viewers file, and create the two
# source code files based on it.
#
def BuildViewers(src, enumdest, datadest):
    v = Viewers()
    v.Parse(src)
    v.CreateEnums(enumdest, 'GENERATED_VIEWERS_ENUM_H')
    v.CreateData(datadest)

if __name__ == "__main__":
    d = os.path.dirname(sys.argv[0])
    if d:
        os.chdir(d)
    BuildViewers('../module.viewers',
                 os.path.join(d, 'generated_viewers_enum.h'),
                 os.path.join(d, 'generated_viewers_data.inc'))
