#!/usr/bin/pike
array(int) make_surrogates(int cc)
{
	if (cc < 0x10000)
		error("%x: Invalid call to make_surrogates", cc);
	int high = 0xD800 | ((cc - 0x10000) >> 10);
	int low  = 0xDC00 | ((cc - 0x10000) & 0x3FF);
	return ({ high, low });
}

int main()
{
	// Read composition exclusion table
	Stdio.File exclusions = Stdio.File();
	multiset(int) dont = (< >);
	exclusions->open("../data/CompositionExclusions.txt", "r");
	int in_nonstarter = 0;
	foreach (exclusions->read()/"\n", string line)
	{
		if (!strlen(line))
		{
			continue;
		}
		// We need the "Non-Starter Decomposition" list, which is
		// commented out in the CompositionExclusions.txt file.
		if (search(line, "Non-Starter Decompositions") != -1)
		{
			in_nonstarter = 1;
		}
		int codepoint = 0;
		if (in_nonstarter)
		{
			if (!sscanf(line, "# %x", codepoint) || codepoint < 256)
			{
				continue;
			}
		}
		else
		{
			if (line[0] == '#' || !sscanf(line, "%x", codepoint) || codepoint < 256)
			{
				continue;
			}
		}
		dont += (< codepoint >);
	}
	exclusions->close();


	// Read the Unicode character database to find mappings
	mapping decompose = ([]);
	mapping compose = ([]);
	mapping compat = ([]);
	mapping narrow_wide = ([]);

	int last_faked = 0xf300;
	
	int fake_decompose(int c, array(int) cc, int iscompat)
	{
		compat[c] = iscompat;
		if (sizeof(cc) == 2)
		{
			decompose[c] = cc;
			return c;
		}
		last_faked++;
		decompose[c] = ({ cc[0], fake_decompose(last_faked, cc[1..], iscompat) });
		if (last_faked >= 0xf800)
			error("Error: Faked decompose overflow!\n");
		return c;
	};

	// Read UnicodeData.txt
	foreach(Stdio.read_file("../data/UnicodeData.txt") / "\n"; int lno; string line) 
	//foreach (Stdio.stdin.read()/"\n", string line)
	{
		// Remove comments and split into semi-colon separated fields
		sscanf(line, "%s#", line);
		if (!sizeof(line))
			continue;
		array data = line / ";";
		if (sizeof(data) != 15)
			continue;

		// Get codepoint
		int c;
		sscanf(data[0], "%x", c);

		// Read the Decomposition_Mapping to find how to decompose this
		// codepoint, if there is anything
		catch {
			int strip_lt(string cd)
			{
				if (search(cd, "<narrow>") >= 0 || search(cd, "<wide>") >= 0)
					narrow_wide[c]=1;
				// <tags> indicate compatibility mappings					
				if (cd[0] == '<')	compat[c]=1;
				return cd[0] != '<';
			};

			// Parse data for this entry
			array(int) cc = map(filter(data[5]/" "-({""}), strip_lt),Gmp.mpz, 16)-({0});
			if (sizeof(cc))
			{
				// Store the data for composition
				if (sizeof(cc)==2 && !dont[c] && !compat[c])
				{
					compose[(cc[0]<<32)|cc[1]] = c;
				}

				// Recode decomposition into UTF-16 code units
				array(int) utf16_cc = ({});
				foreach (cc;; int cc_char)
				{
					if (cc_char < 65536)
					{
						utf16_cc += ({ cc_char });
					}
					else
					{
						utf16_cc += make_surrogates(cc_char);
					}
				}

				if (sizeof(utf16_cc) > 2)
				{
					// Fake long decompositions through the PUA.
					fake_decompose(c, utf16_cc, compat[c]);
				}
				else
				{
					decompose[c] = utf16_cc+({ 0 });
				}
			}
		};
	}

	mapping top=([]);
	int i, sz_dc, sz_dcnbmp;
	int compat_int, compatnbmp_int;
	array(int) narrow_wide_indexes = ({}); 
	write(#"/** @file decompositions.inl
 * This file is auto-generated by modules/unicode/scripts/make_decompose.pike.
 * DO NOT EDIT THIS FILE MANUALLY.
 */\n
#ifdef USE_UNICODE_INC_DATA\n");
	write("static const Decomposition decompositions[] = {");

	// Write the canonical decompositions to the data file
	int decompositions_have_nonbmp = 0;
	foreach ((sort(indices(decompose))), int c)
	{
		if (c <= 65535 && decompose[c][0]<=65535 && decompose[c][1]<=65535)
		{
			if (sz_dc)
				write(", ");
			if (!(sz_dc & 1))
				write("\n\t");
			
			int q = decompose[c][1];
			compat_int |= compat[c]<<sz_dc; // Bit array of compat flags
			if (narrow_wide[c]>0)
				narrow_wide_indexes += ({sz_dc});
			sz_dc++;
			write("{0x%04X,{0x%04X,0x%04X}}", c, decompose[c][0], q);
		}
		else if (c <= 65535 && (decompose[c][0]>=65536 || decompose[c][1]>=65536))
		{
			error("%x cannot expand BMP to two non-BMP\n", c);
			return 1;
		}
		else
		{
			if (!decompositions_have_nonbmp)
			{
				decompositions_have_nonbmp = 1;
				write("\n};\n\n");
				write("static const DecompositionNonBMP decompositionsnonbmp[] = {\n");
			}
			if (sz_dcnbmp)
				write(",\n");
			// Encode non-BMP character as a UTF-16 surrogate pair
			// in a 32-bit integer
			array(int) surrogate_c = make_surrogates(c);
			int shift_c = surrogate_c[0] << 16 | surrogate_c[1];
			compatnbmp_int |= compat[c]<<sz_dcnbmp; // Bit array of compat flags
			sz_dcnbmp++;
			write("\t{0x%08XU,{0x%04X,0x%04X}} /* U+%X */", shift_c, decompose[c][0], decompose[c][1], c);
		}
	}
	write("\n};\n\n");

	// Write the canonical decomposition bit flags to the data file
	write("static const unsigned char compat_decomposition["+(sz_dc/8+1)+"] = {");
	string q = compat_int->digits(256);
	q = "\0"*(sz_dc/8-sizeof(q)+1) + q;
	for(int i=0; i<sizeof(q); i++)
	{
		if (i)
			write(", ");
		if (!(i & 7))
			write("\n\t");
		write("0x%02x", q[-(i+1)]);
	}
	write("\n};\n\n");

	// Write the <narrow> <wide> decomposition indexes
	write("static const unsigned short narrow_wide_decomposition_indexes["+sizeof(narrow_wide_indexes)+"] = {");
	int j=0;
	foreach(narrow_wide_indexes, int n_w_ind)
	{
		if (j)
			write(", ");
		if (!(j & 7))
			write("\n\t");
		write("%d", n_w_ind);
		j++;
	}
	write("\n};\n\n");

	foreach ((sort(indices(narrow_wide))), int c)
	{
		
	}

	write("static const unsigned char compat_decomposition_nonbmp["+(sz_dcnbmp/8+1)+"] = {");
	string qnbmp = compatnbmp_int->digits(256);
	qnbmp = "\0"*(sz_dcnbmp/8-sizeof(qnbmp)+1) + qnbmp;
	for(int i=0; i<sizeof(qnbmp); i++)
	{
		if (i)
			write(", ");
		if (!(i & 7))
			write("\n\t");
		write("0x%02x", qnbmp[-(i+1)]);
	}
	write("\n};\n\n");

	// Write compositions to the data file
	write("static const Composition compositions[] = {");
	top=([]);
	i=0;

	int compositions_have_nonbmp = 0;
	int sz_cc, sz_ccnbmp;
	foreach (sort(indices(compose)), int c)
	{
		int c1 = (int)(c>>32);
		int c2 = (int)(c&0xffffffff);
		if (c1 <= 65535 && c2 <= 65535 && ((int)compose[c]) <= 65535)
		{
			if (sz_cc)
				write(", ");
			if (!(sz_cc & 1))
				write("\n\t");
			sz_cc++;
			write("{0x%04X,0x%04X,0x%04X}", c1, c2, (int)compose[c]);
		}
		else if ((c1 <= 65535 || c2 <= 65535) && ((int)compose[c]) > 65535)
		{
			error("%x cannot compose initial BMP to non-BMP\n", compose[c]);
			return 1;
		}
		else
		{
			if (!compositions_have_nonbmp)
			{
				write("\n};\n");
				write("static const CompositionNonBMP compositionsnonbmp[] = {\n");
				compositions_have_nonbmp = 1;
			}
			if (sz_ccnbmp)
				write(",\n");
			// Encode non-BMP character as a UTF-16 surrogate pair
			// in a 32-bit integer
			array(int) surrogate_c1 = make_surrogates(c1);
			int shift_c1 = surrogate_c1[0] << 16 | surrogate_c1[1];
			array(int) surrogate_c2 = make_surrogates(c2);
			int shift_c2 = surrogate_c2[0] << 16 | surrogate_c2[1];
			array(int) surrogate_comp = make_surrogates((int)compose[c]);
			int shift_comp = surrogate_comp[0] << 16 | surrogate_comp[1];
			sz_ccnbmp++;
			write("\t{0x%08X,0x%08X,0x%08X} /* U+%X + U+%X -> U+%X */", shift_c1, shift_c2, shift_comp, c1, c2, (int)compose[c]);
			// write("\t{0x%08X,0x%08X,0x%08X}", c1, c2, (int)compose[c]);
		}
	}
	write("\n};\n#endif // USE_UNICODE_INC_DATA\n");
	return 0;
}
