/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4; c-file-style:"stroustrup" -*-
**
** Copyright (C) 1995-2012 Opera Software ASA.  All rights reserved.
**
** This file is part of the Opera web browser.  It may not be distributed
** under any circumstances.
**
** Espen Sand
*/

#ifndef X11__INPUTUTILS_H__
#define X11__INPUTUTILS_H__

#include "modules/hardcore/keys/opkeys.h"
#include "platforms/utilix/x11_all.h" // For KeySym, otherwise x11types.h would be fine


namespace InputUtils
{
	struct KeyContent
	{
		KeyContent() : m_opera_key(OP_KEY_INVALID), m_location(OpKey::LOCATION_STANDARD) {}

		/**
		 * Set virtual key (OP_KEY_...) and location
		 *
		 * @param opera_key The virtual key
		 * @param location Keyboard area
		 */
		void SetVirtualKey(OpVirtualKey opera_key, OpKey::Location location) { m_opera_key = opera_key; m_location = location;}

		/**
		 * Set text that represets a key
		 *
		 * @param text Key text
		 *
		 * @return OpStatus::OK on success, otherwise OpStatus::ERR_NO_MEMORY
		 */
		OP_STATUS SetText(OpStringC text) { return m_utf8_text.SetUTF8FromUTF16(text); }

		/**
		 * Resets state to initial state
		 */
		void Reset() { m_opera_key = OP_KEY_INVALID; m_location = OpKey::LOCATION_STANDARD; m_utf8_text.Empty(); }

		/**
		 * Return virtual key code.
		 */
		OpVirtualKey GetVirtualKey() const
		{
			OP_ASSERT(m_opera_key != OP_KEY_UNICODE || m_utf8_text.HasContent());
			return m_opera_key;
		}

		/**
		 * Return entered key text in utf8 format
		 */
		const OpStringC8 GetUtf8Text() const { return m_utf8_text; }

		/**
		 * Return keyboard area where key is located
		 */
		OpKey::Location GetLocation() const { return m_location; }

	private:
		OpVirtualKey m_opera_key;
		OpString8 m_utf8_text;
		OpKey::Location m_location;
	};

	/**
	 * Translate an X11 key press or key release event to readable text
	 * (if the event represents a regular character) and to an OP_KEY symbol.
	 * Either text or symbol or both has to be present for a successful return.
	 *
	 * @param event X11 event
	 * @param content Event data is saved here on return
	 *
	 * @param true on success, otherwise false on memory error, if
	 * content is empty or if the event is not a key press or key release
	 * event
	 */
	bool TranslateKeyEvent(XEvent* event, KeyContent& content);

	/**
	 * Convert virtual_key to X11 keysym. The conversion is a reversal of the
	 * conversion made in @ref TranslateKeyEvent
	 *
	 * @param virtual_key The virtual key
	 * @param location Keyboard location. Only OpKey::LOCATION_STANDARD is supported now
	 *
	 * @return The X11 keysym or NoSymbol in no match
	 */
	KeySym VirtualKeyToKeySym(OpVirtualKey virtual_key, OpKey::Location location);

	/**
	 * Translate button flags from X to Opera.
	 * @param xbutton expected value: 1, 2 or 3. Don't pass anything else.
	 */
	MouseButton X11ToOpButton(int xbutton);

	/**
	 * Return the native mouse button that will, when pressed, start a drag
	 */
	inline unsigned int GetX11DragButton() { return Button1; }

	/**
	 * Return true if button is a mouse wheel button, otherwise false
	 *
	 * @param button The button type to examine
	 *
	 * @return See above
	 */
	inline bool IsX11WheelButton(int button) { return button == Button4 || button == Button5; }

	/**
	 * Examined the key event to determine if the O-menu can be opnened as
	 * a result of the event. For menu activation this function should be called
	 * for both key press and key release events. A failure on any should block
	 * the menu.
	 *
	 * @param e The X event
	 *
	 * @return true if the X event allows the O-menu to be opened, otherwise false.

	 */
	bool IsOMenuToggleKey(XEvent* e);

	/**
	 * Returns true if the event was eaten (which means that no further
	 * processing should happen.
	 *
	 * @param e The X event
	 */
	bool HandleEvent(XEvent* e);

	/**
	 * Extracts keysym and text from a keyevent.
	 *
	 * @param event Should be of type KeyPress of KeyRelease
	 * @param keysym Returns the keysym of the key.  Returns NoSymbol
	 * on errors or if the event does not represent a valid keysym.
	 * @param text Returns the text generated by the key.  Returns ""
	 * on errors or if the event does not generate any text.
	 */
	void GetKey(XEvent* event, KeySym& keysym, OpString& text);

	/**
	 * Notifies input code that focus has changed.
	 */
	void HandleFocusChange(XEvent* event);

	/**
	 * Sets the current modifier mask. This must be done before any code 
	 * that can block further event processing
	 */
	void SetX11ModifierMask(XEvent* event);

	/**
	 * Return the X11 modifier mask. Opera core and quick will not understand this 
	 * format. It is for platform only
	 */
	unsigned int GetX11ModifierMask();

	/**
	 * Return the modifier flags (shift keys and so on) - in the Opera (not X) format.
	 */
	ShiftKeyState GetOpModifierFlags();

	/**
	 * Sets the current button flags. This must be done before any code 
	 * that can block further event processing
	 */
	void SetOpButtonFlags(XEvent* event);

	/**
	 * Return the (mouse) button flags - in the Opera (not X) format.
	 */
	int GetOpButtonFlags();

	/**
	 * Returns the pressed state of a given button.
	 *
	 * @param button The mouse button to test
	 * @param exact Only the given button can be pressed if true, otherwise
	 *              any button combination is allowed
	 *
	 * @return true if the button is pressed, otherwise false
	 */
	bool IsOpButtonPressed(MouseButton button, bool exact);

	/**
	 * Returns the flag state when all "normal" X11 modifier keys are pressed
	 */
	inline UINT32 AllXKeyModifierMasks() 
	{
		return ShiftMask|ControlMask|Mod1Mask|Mod4Mask|Mod5Mask;
	}

	/**
	 * Updates the current mouse button click count. This function should only
	 * be called once per keypress
	 *
	 * @param event The mouse press event
	 */
	void UpdateClickCount(XEvent* event);

	/**
	 * Return the number of successive clicks in the same window with the same
	 * button within a reasonably short amount of time. Used for tracking
	 * double-clicks, triple-clicks and so on.
	 */
	int GetClickCount();

	/**
	 * Returns the screen number (0 and upwards) where the mouse pointer is 
	 * currently located or -1 on error
	 */
	int GetScreenUnderPointer();

	/**
	 * Returns the global mouse position (with respect to the root window) on
	 * the screen the mouse is located
	 *
	 * @param pos The postion
	 * 
	 * @return TRUE on success, otherwise FALSE (should not happen)
	 */
	BOOL GetGlobalPointerPos(OpPoint& pos);

	/**
	 * Prints information about key event. Works for press and release
	 *
	 * @param e The key event to inspect
	 */
	void DumpKeyEvent(XEvent* e);

	/**
	 * Prints information about button (mouse) event. Works for press and release
	 *
	 * @param e The button event to inspect
	 */
	void DumpButtonEvent(XEvent* e);

}

#endif // X11__INPUTUTILS_H__
