#include "core/pch.h"
#ifdef SPC_USE_HUNSPELL_ENGINE
#include "modules/spellchecker/hunspell_3p/src/hunspell/hunspell_common.h"
#include "modules/spellchecker/hunspell_3p/src/hunspell/license.hunspell"
#include "modules/spellchecker/hunspell_3p/src/hunspell/license.myspell"

#include "modules/spellchecker/hunspell_3p/src/hunspell/csutil.h"
#include "modules/spellchecker/hunspell_3p/src/hunspell/atypes.h"
#include "modules/spellchecker/hunspell_3p/src/hunspell/langnum.h"

// Unicode character encoding information
struct unicode_info {
  unsigned short c;
  unsigned short cupper;
  unsigned short clower;
};


#if defined(HUNSPELL_USE_UTF_TBL) && !defined(OPENOFFICEORG) && !defined(MOZILLA_CLIENT)
#  include "modules/spellchecker/hunspell_3p/src/hunspell/utf_info.cpp"
#  define UTF_LST_LEN ((int)(sizeof(utf_lst) / (sizeof(unicode_info))))
#endif

#ifdef MOZILLA_CLIENT
#include "nsCOMPtr.h"
#include "nsServiceManagerUtils.h"
#include "nsIUnicodeEncoder.h"
#include "nsIUnicodeDecoder.h"
#include "nsUnicharUtils.h"
#include "nsICharsetConverterManager.h"

static NS_DEFINE_CID(kCharsetConverterManagerCID, NS_ICHARSETCONVERTERMANAGER_CID);
#endif

struct unicode_info2 {
  char cletter;
  unsigned short cupper;
  unsigned short clower;
};

#ifdef HUNSPELL_USE_UTF_TBL
static struct unicode_info2 * utf_tbl = NULL;
static int utf_tbl_count = 0; // utf_tbl can be used by multiple Hunspell instances
#endif

/* only UTF-16 (BMP) implementation */
char * u16_u8(char * dest, int size, const w_char * src, int srclen) {
    signed char * u8 = (signed char *)dest;
    signed char * u8_max = (signed char *)(u8 + size);
    const w_char * u2 = src;
    const w_char * u2_max = src + srclen;
    while ((u2 < u2_max) && (u8 < u8_max)) {
        if (u2->h) { // > 0xFF
            // XXX 4-byte haven't implemented yet.
            if (u2->h >= 0x08) {   // >= 0x800 (3-byte UTF-8 character)
                *u8 = 0xe0 + (u2->h >> 4);
                u8++;
                if (u8 < u8_max) {
                    *u8 = 0x80 + ((u2->h & 0xf) << 2) + (u2->l >> 6);
                    u8++;
                    if (u8 < u8_max) {
                        *u8 = 0x80 + (u2->l & 0x3f);
                        u8++;
                    }
                }
            } else { // < 0x800 (2-byte UTF-8 character)
                *u8 = 0xc0 + (u2->h << 2) + (u2->l >> 6);
                u8++;
                if (u8 < u8_max) {
                    *u8 = 0x80 + (u2->l & 0x3f);
                    u8++;
                }
            }
        } else { // <= 0xFF
            if (u2->l & 0x80) { // >0x80 (2-byte UTF-8 character)
                *u8 = 0xc0 + (u2->l >> 6);
                u8++;
                if (u8 < u8_max) {
                    *u8 = 0x80 + (u2->l & 0x3f);
                    u8++;
                }
            } else { // < 0x80 (1-byte UTF-8 character)
                *u8 = u2->l;
                u8++;
            }
        }
        u2++;
    }
    *u8 = '\0';
    return dest;
}


/* only UTF-16 (BMP) implementation */
int u8_u16(w_char * dest, int size, const char * src) {
    const signed char * u8 = (const signed char *)src;
    w_char * u2 = dest;
    w_char * u2_max = u2 + size;
    
    while ((u2 < u2_max) && *u8) {
    switch ((*u8) & 0xf0) {
        case 0x00:
        case 0x10:
        case 0x20:
        case 0x30:
        case 0x40:
        case 0x50:
        case 0x60:
        case 0x70: {
            u2->h = 0;
            u2->l = *u8;
            break;
        }
        case 0x80:
        case 0x90:
        case 0xa0:
        case 0xb0: {
            HUNSPELL_WARNING(hunspell_stderr, "UTF-8 encoding error. Unexpected continuation bytes in %ld. character position\n%s\n", static_cast<long>(u8 - (signed char *)src), src);    
            u2->h = 0xff;
            u2->l = 0xfd;
            break;
        }
        case 0xc0:
        case 0xd0: {    // 2-byte UTF-8 codes
            if ((*(u8+1) & 0xc0) == 0x80) {
                u2->h = (*u8 & 0x1f) >> 2;
                u2->l = (*u8 << 6) + (*(u8+1) & 0x3f);
                u8++;
            } else {
                HUNSPELL_WARNING(hunspell_stderr, "UTF-8 encoding error. Missing continuation byte in %ld. character position:\n%s\n", static_cast<long>(u8 - (signed char *)src), src);
                u2->h = 0xff;
                u2->l = 0xfd;
            }
            break;
        }
        case 0xe0: {    // 3-byte UTF-8 codes
            if ((*(u8+1) & 0xc0) == 0x80) {
                u2->h = ((*u8 & 0x0f) << 4) + ((*(u8+1) & 0x3f) >> 2);
                u8++;
                if ((*(u8+1) & 0xc0) == 0x80) {
                    u2->l = (*u8 << 6) + (*(u8+1) & 0x3f);
                    u8++;
                } else {
                    HUNSPELL_WARNING(hunspell_stderr, "UTF-8 encoding error. Missing continuation byte in %ld. character position:\n%s\n", static_cast<long>(u8 - (signed char *)src), src);
                    u2->h = 0xff;
                    u2->l = 0xfd;
                }
            } else {
                HUNSPELL_WARNING(hunspell_stderr, "UTF-8 encoding error. Missing continuation byte in %ld. character position:\n%s\n", static_cast<long>(u8 - (signed char *)src), src);
                u2->h = 0xff;
                u2->l = 0xfd;
            }
            break;
        }
        case 0xf0: {    // 4 or more byte UTF-8 codes
            HUNSPELL_WARNING(hunspell_stderr, "This UTF-8 encoding can't convert to UTF-16:\n%s\n", src);
            u2->h = 0xff;
            u2->l = 0xfd;
            return -1;
        }
    }
    u8++;
    u2++;
    }
    return (int)(u2 - dest);
}

void flag_qsort(unsigned short flags[], int begin, int end) {
    unsigned short reg;
    if (end > begin) {
        unsigned short pivot = flags[begin];
        int l = begin + 1;
        int r = end;
        while(l < r) {
            if (flags[l] <= pivot) {
                l++;
            } else {
             r--;
             reg = flags[l];
             flags[l] = flags[r];
             flags[r] = reg;
          }
       }
       l--;
       reg = flags[begin];
       flags[begin] = flags[l];
       flags[l] = reg;

       flag_qsort(flags, begin, l);
       flag_qsort(flags, r, end);
    }
 }

int flag_bsearch(unsigned short flags[], unsigned short flag, int length) {
   int mid;
   int left = 0;
   int right = length - 1;
   while (left <= right) {
      mid = (left + right) / 2;
      if (flags[mid] == flag) return 1;
      if (flag < flags[mid]) right = mid - 1;
      else left = mid + 1;
   }
   return 0;
}

 // strip strings into token based on single char delimiter
 // acts like strsep() but only uses a delim char and not
 // a delim string
 // default delimiter: white space characters
 
 char * mystrsep(char ** stringp, const char delim)
 {
   char * mp = *stringp;
   if (*mp != '\0') {
      char * dp;
      if (delim) {
        dp = op_strchr(mp, delim);
      } else {
        // don't use op_isspace() here, the string can be in some random charset
        // that's way different than the locale's
        for (dp = mp; (*dp && *dp != ' ' && *dp != '\t'); dp++);
        if (!*dp) dp = NULL;
      }
      if (dp) {
         *stringp = dp+1;
         *dp = '\0';
      } else {
         *stringp = mp + op_strlen(mp);
      }
      return mp;
   }
   return NULL;
 }

 // replaces strdup with ansi version
 char * mystrdup(const char * s)
 {
   char * d = NULL;
   if (s) {
      int sl = op_strlen(s);
      d = (char *) op_malloc(((sl+1) * sizeof(char)));
      if (d) {
         op_memcpy(d,s,((sl+1)*sizeof(char)));
         return d;
      }
      HUNSPELL_WARNING(hunspell_stderr, "Can't allocate memory.\n");
   }
   return d;
 }

 // strcat for limited length destination string
 char * mystrcat(char * dest, const char * st, int max) {
   int len;
   int len2;
   if (dest == NULL || st == NULL) return dest;
   len = op_strlen(dest);
   len2 = op_strlen(st);
   if (len + len2 + 1 > max) return dest;
   op_strcpy(dest + len, st);
   return dest;
 }

 // remove cross-platform text line end characters
 void mychomp(char * s)
 {
   size_t k = op_strlen(s);
   if ((k > 0) && ((*(s+k-1)=='\r') || (*(s+k-1)=='\n'))) *(s+k-1) = '\0';
   if ((k > 1) && (*(s+k-2) == '\r')) *(s+k-2) = '\0';
 }
 
 
 //  does an ansi strdup of the reverse of a string
 char * myrevstrdup(const char * s)
 {
     char * d = NULL;
     if (s) {
        size_t sl = op_strlen(s);
        d = (char *) op_malloc(sl+1);
        if (d) {
          const char * p = s + sl - 1;
          char * q = d;
          while (p >= s) *q++ = *p--;
          *q = '\0';
        } else {
          HUNSPELL_WARNING(hunspell_stderr, "Can't allocate memory.\n");
        }
     }
     return d;
 }

#ifdef MORPHOLOGICAL_FUNCTIONS
// break text to lines
// return number of lines
int line_tok(const char * text, char *** lines, char breakchar) {
    int linenum = 0;
    if (!text) {
        return linenum;
    }
    char * dup = mystrdup(text);
    char * p = op_strchr(dup, breakchar);
    while (p) {
        linenum++;
        *p = '\0';
        p++;
        p = op_strchr(p, breakchar);
    }
    linenum++;
    *lines = (char **) op_malloc(linenum * sizeof(char *));
    if (!(*lines)) {
        op_free(dup);
        return 0;
    }

    p = dup;
    int l = 0;
    for (int i = 0; i < linenum; i++) {
        if (*p != '\0') {
            (*lines)[l] = mystrdup(p);
            if (!(*lines)[l]) {
                for (i = 0; i < l; i++) op_free((*lines)[i]);
                op_free(dup);
                return 0;
            }
            l++;
        }
        p += op_strlen(p) + 1;
    }
    op_free(dup);
    if (!l) op_free(*lines);
    return l;
}

// uniq line in place
char * line_uniq(char * text, char breakchar) {
    char ** lines;
    int linenum = line_tok(text, &lines, breakchar);
    int i;
    op_strcpy(text, lines[0]);
    for ( i = 1; i < linenum; i++ ) {
        int dup = 0;
        for (int j = 0; j < i; j++) {
            if (op_strcmp(lines[i], lines[j]) == 0) dup = 1;
        }
        if (!dup) {
            if ((i > 1) || (*(lines[0]) != '\0')) {
                op_sprintf(text + op_strlen(text), "%c", breakchar);
            }
            op_strcat(text, lines[i]);
        }
    }
    for ( i = 0; i < linenum; i++ ) {
        if (lines[i]) op_free(lines[i]);
    }
    if (lines) op_free(lines);
    return text;
}

// uniq and boundary for compound analysis: "1\n\2\n\1" -> " ( \1 | \2 ) "
char * line_uniq_app(char ** text, char breakchar) {
    if (!op_strchr(*text, breakchar)) {
        return *text;
    }
    
    char ** lines;
    int i;
    int linenum = line_tok(*text, &lines, breakchar);
    int dup = 0;
    for (i = 0; i < linenum; i++) {
        for (int j = 0; j < (i - 1); j++) {
            if (op_strcmp(lines[i], lines[j]) == 0) {
                *(lines[i]) = '\0';
                dup++;
                break;
            }
        }
    }
    if ((linenum - dup) == 1) {
        op_strcpy(*text, lines[0]);
        freelist(&lines, linenum);
        return *text;
    }
    char * newtext = (char *) op_malloc(op_strlen(*text) + 2 * linenum + 3 + 1);
    if (newtext) {
        op_free(*text);
        *text = newtext;
    } else {
        freelist(&lines, linenum);
        return *text;
    }    
    op_strcpy(*text," ( ");
    for (i = 0; i < linenum; i++) if (*(lines[i])) {
        op_sprintf(*text + op_strlen(*text), "%s%s", lines[i], " | ");
    }
    (*text)[op_strlen(*text) - 2] = ')'; // " ) "
    freelist(&lines, linenum);
    return *text;
}
#endif

 // append s to ends of every lines in text
 void strlinecat(char * dest, const char * s)
 {
    char * dup = mystrdup(dest);
    char * source = dup;
    int len = op_strlen(s);
    if (dup) {
        while (*source) {
            if (*source == '\n') {
                hunspell_strncpy(dest, s, len);
                dest += len;
            }
            *dest = *source;
            source++; dest++;
        }
        op_strcpy(dest, s);
        op_free(dup);
    }
 }

// change \n to char c
char * tr(char * text, char oldc, char newc) {
    char * p;
    for (p = text; *p; p++) if (*p == oldc) *p = newc;
    return text;
}


#ifdef MORPHOLOGICAL_FUNCTIONS

// morphcmp(): compare MORPH_DERI_SFX, MORPH_INFL_SFX and MORPH_TERM_SFX fields
// in the first line of the inputs
// return 0, if inputs equal
// return 1, if inputs may equal with a secondary suffix
// otherwise return -1
int morphcmp(const char * s, const char * t)
{
    int se = 0;
    int te = 0;
    const char * sl;
    const char * tl;    
    const char * olds;
    const char * oldt;
    if (!s || !t) return 1;
    olds = s;
    sl = op_strchr(s, '\n');
    s = op_strstr(s, MORPH_DERI_SFX);
    if (!s || (sl && sl < s)) s = op_strstr(olds, MORPH_INFL_SFX);
    if (!s || (sl && sl < s)) {
        s= op_strstr(olds, MORPH_TERM_SFX);
        olds = NULL;
    }
    oldt = t;
    tl = op_strchr(t, '\n');
    t = op_strstr(t, MORPH_DERI_SFX);
    if (!t || (tl && tl < t)) t = op_strstr(oldt, MORPH_INFL_SFX);
    if (!t || (tl && tl < t)) {
        t = op_strstr(oldt, MORPH_TERM_SFX);
        oldt = NULL;
    }
    while (s && t && (!sl || sl > s) && (!tl || tl > t)) {
        s += MORPH_TAG_LEN;
        t += MORPH_TAG_LEN;
        se = 0;
        te = 0;
        while ((*s == *t) && !se && !te) {
            s++;
            t++;
            switch(*s) {
                case ' ':
                case '\n':
                case '\t':
                case '\0': se = 1;
            }
            switch(*t) {
                case ' ':
                case '\n':
                case '\t':
                case '\0': te = 1;
            }
        }
        if (!se || !te) {
            // not terminal suffix difference
            if (olds) return -1;
            return 1;
        }
        olds = s;
        s = op_strstr(s, MORPH_DERI_SFX);
        if (!s || (sl && sl < s)) s = op_strstr(olds, MORPH_INFL_SFX);
        if (!s || (sl && sl < s)) {
            s = op_strstr(olds, MORPH_TERM_SFX);
            olds = NULL;
        }
        oldt = t;
        t = op_strstr(t, MORPH_DERI_SFX);
        if (!t || (tl && tl < t)) t = op_strstr(oldt, MORPH_INFL_SFX);
        if (!t || (tl && tl < t)) {
            t = op_strstr(oldt, MORPH_TERM_SFX);
            oldt = NULL;
        }
    }
    if (!s && !t && se && te) return 0;
    return 1;
}

int get_sfxcount(const char * morph)
{
    if (!morph || !*morph) return 0;
    int n = 0;
    const char * old = morph;
    morph = op_strstr(morph, MORPH_DERI_SFX);
    if (!morph) morph = op_strstr(old, MORPH_INFL_SFX);
    if (!morph) morph = op_strstr(old, MORPH_TERM_SFX);
    while (morph) {
        n++;
        old = morph;
        morph = op_strstr(morph + 1, MORPH_DERI_SFX);
        if (!morph) morph = op_strstr(old + 1, MORPH_INFL_SFX);
        if (!morph) morph = op_strstr(old + 1, MORPH_TERM_SFX);
    }
    return n;
}
#endif


int fieldlen(const char * r)
{
    int n = 0;
    while (r && *r != ' ' && *r != '\t' && *r != '\0' && *r != '\n') {
        r++;
        n++;
    }
    return n;
}

char * copy_field(char * dest, size_t size, const char * morph, const char * var)
{
    if (!morph) return NULL;
    const char * beg = op_strstr(morph, var);
    if (beg) {
       char * d = dest;
	   char* dend = dest + size;
       for (beg += MORPH_TAG_LEN; *beg != ' ' && *beg != '\t' &&
            *beg != '\n' && *beg != '\0' && d < dend; d++, beg++) {
         *d = *beg;
       }
	   if ( d < dend ) {
		   *d = '\0';
		   return dest;
	   }
  }
  return NULL;
}

char * mystrrep(char * word, const char * pat, const char * rep) {
    char * pos = op_strstr(word, pat);
    if (pos) {
      int replen = op_strlen(rep);
      int patlen = op_strlen(pat);
      while (pos) {
        if (replen < patlen) {
            char * end = word + op_strlen(word);
            char * next = pos + replen;
            char * prev = pos + op_strlen(pat);
            for (; prev < end; *next = *prev, prev++, next++);
            *next = '\0';
        } else if (replen > patlen) {
            char * end = pos + patlen;
            char * next = word + op_strlen(word) + replen - patlen;
            char * prev = next - replen + patlen;
            for (; prev >= end; *next = *prev, prev--, next--);
        }
        hunspell_strncpy(pos, rep, replen);
        pos = op_strstr(word, pat);
      }
    }
    return word;
}

 // reverse word 
 int reverseword(char * word) {
   char r;
   for (char * dest = word + op_strlen(word) - 1; word < dest; word++, dest--) {
     r=*word;
     *word = *dest;
     *dest = r;
   }
   return 0;
 }

 // reverse word (error: 1)
 int reverseword_utf(char * word) {
   w_char w[MAXWORDLEN];// ARRAY OK 2009-02-09 jb
   w_char * p;
   w_char r;
   int l = u8_u16(w, MAXWORDLEN, word);
   if (l == -1) return 1;
   p = w;
   for (w_char * dest = w + l - 1; p < dest; p++, dest--) {
     r=*p;
     *p = *dest;
     *dest = r;
   }
   u16_u8(word, op_strlen(word) + 1, w, l);
   return 0;
 }

 int uniqlist(char ** list, int n) {
   int i;
   if (n < 2) return n;
   for (i = 0; i < n; i++) {
     for (int j = 0; j < i; j++) {
        if (list[j] && list[i] && (op_strcmp(list[j], list[i]) == 0)) {
            op_free(list[i]);
            list[i] = NULL;
            break;
        }
     }
   } 
   int m = 1;  
   for (i = 1; i < n; i++) if (list[i]) {
        list[m] = list[i];
        m++;
    }
   return m;
 }
 
 void freelist(char *** list, int n) {
   if (list && *list && n > 0) {
     for (int i = 0; i < n; i++) if ((*list)[i]) op_free((*list)[i]);
     op_free(*list);
     *list = NULL;
   }
 }
 
 // convert null terminated string to all caps
 void mkallcap(char * p, const struct cs_info * csconv)
 {
   while (*p != '\0') {
     *p = csconv[((unsigned char) *p)].cupper;
     p++;
   }
 }
  
 // convert null terminated string to all little
 void mkallsmall(char * p, const struct cs_info * csconv)
 {
   while (*p != '\0') {
     *p = csconv[((unsigned char) *p)].clower;
     p++;
   }
 }

void mkallsmall_utf(w_char * u, int nc, int langnum) {
    for (int i = 0; i < nc; i++) {
        unsigned short idx = (u[i].h << 8) + u[i].l;
        if (idx != unicodetolower(idx, langnum)) {
            u[i].h = (unsigned char) (unicodetolower(idx, langnum) >> 8);
            u[i].l = (unsigned char) (unicodetolower(idx, langnum) & 0x00FF);
        }
    }
}

void mkallcap_utf(w_char * u, int nc, int langnum) {
    for (int i = 0; i < nc; i++) {
        unsigned short idx = (u[i].h << 8) + u[i].l;
        if (idx != unicodetoupper(idx, langnum)) {
            u[i].h = (unsigned char) (unicodetoupper(idx, langnum) >> 8);
            u[i].l = (unsigned char) (unicodetoupper(idx, langnum) & 0x00FF);
        }
    }
}
 
 // convert null terminated string to have initial capital
 void mkinitcap(char * p, const struct cs_info * csconv)
 {
   if (*p != '\0') *p = csconv[((unsigned char)*p)].cupper;
 }

 // conversion function for protected memory
 void store_pointer(char * dest, char * source)
 {
    op_memcpy(dest, &source, sizeof(char *));
 }

 // conversion function for protected memory
 char * get_stored_pointer(const char * s)
 {
    char * p;
    op_memcpy(&p, s, sizeof(char *));
    return p;
 }

#ifndef MOZILLA_CLIENT
 // convert null terminated string to all caps using encoding
 void enmkallcap(char * d, const char * p, const char * encoding)
 
 {
   struct cs_info * csconv = get_current_cs(encoding);
   while (*p != '\0') {
     *d++ = csconv[((unsigned char) *p)].cupper;
     p++;
   }
   *d = '\0';
 }

 // convert null terminated string to all little using encoding
 void enmkallsmall(char * d, const char * p, const char * encoding)
 {
   struct cs_info * csconv = get_current_cs(encoding);
   while (*p != '\0') {
     *d++ = csconv[((unsigned char) *p)].clower;
     p++;
   }
   *d = '\0';
 }

 // convert null terminated string to have initial capital using encoding
 void enmkinitcap(char * d, const char * p, const char * encoding)
 {
   struct cs_info * csconv = get_current_cs(encoding);
   op_memcpy(d,p,(op_strlen(p)+1));
   if (*p != '\0') *d= csconv[((unsigned char)*p)].cupper;
 }

// these are simple character mappings for the 
// encodings supported
// supplying isupper, tolower, and toupper

#ifndef HAS_COMPLEX_GLOBALS
# define iso1_tbl g_hunspell_iso1_tbl
# define iso2_tbl g_hunspell_iso2_tbl
# define iso3_tbl g_hunspell_iso3_tbl
# define iso4_tbl g_hunspell_iso4_tbl
# define iso5_tbl g_hunspell_iso5_tbl
# define iso6_tbl g_hunspell_iso6_tbl
# define iso7_tbl g_hunspell_iso7_tbl
# define iso8_tbl g_hunspell_iso8_tbl
# define iso9_tbl g_hunspell_iso9_tbl
# define iso10_tbl g_hunspell_iso10_tbl
# define koi8r_tbl g_hunspell_koi8r_tbl
# define koi8u_tbl g_hunspell_koi8u_tbl
# define cp1251_tbl g_hunspell_cp1251_tbl
# define iso13_tbl g_hunspell_iso13_tbl
# define iso14_tbl g_hunspell_iso14_tbl
# define iso15_tbl g_hunspell_iso15_tbl
# define iscii_devanagari_tbl g_hunspell_iscii_devanagari_tbl
# define tis620_tbl g_hunspell_tis620_tbl
#endif


CS_ARRAY(struct cs_info, iso1_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x01, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END


CS_ARRAY(struct cs_info, iso2_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x01, 0xb1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x01, 0xb3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x01, 0xb5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x01, 0xb6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x01, 0xb9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x01, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x01, 0xbb, 0xab),
CS_ARRAY_INFO_ENTRY(0x01, 0xbc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x01, 0xbe, 0xae),
CS_ARRAY_INFO_ENTRY(0x01, 0xbf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x01, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END


CS_ARRAY(struct cs_info, iso3_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x01, 0xb1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x01, 0xb6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0xa9),
CS_ARRAY_INFO_ENTRY(0x01, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x01, 0xbb, 0xab),
CS_ARRAY_INFO_ENTRY(0x01, 0xbc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x01, 0xbf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xc3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso4_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x01, 0xb1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x01, 0xb3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x01, 0xb5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x01, 0xb6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x01, 0xb9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x01, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x01, 0xbb, 0xab),
CS_ARRAY_INFO_ENTRY(0x01, 0xbc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x01, 0xbe, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x01, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso5_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x01, 0xf7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xab),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xae),
CS_ARRAY_INFO_ENTRY(0x01, 0xff, 0xaf),
CS_ARRAY_INFO_ENTRY(0x01, 0xd0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x01, 0xd1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x01, 0xd2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x01, 0xd3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x01, 0xd4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x01, 0xd5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x01, 0xd6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x01, 0xd7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x01, 0xd8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x01, 0xd9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x01, 0xda, 0xba),
CS_ARRAY_INFO_ENTRY(0x01, 0xdb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x01, 0xdc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x01, 0xdd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x01, 0xde, 0xbe),
CS_ARRAY_INFO_ENTRY(0x01, 0xdf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xd1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xd3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xd4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xd5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xd6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xd8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xd9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xda, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xdb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xaf)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso6_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xc0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xc1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xc2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xc3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xc4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xc5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xc6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xc7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xc8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xc9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xca, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xcb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xcc, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xcd, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xce, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xcf, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xd1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xd3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xd4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xd5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xd6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x00, 0xd8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xd9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xda, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xdb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xea),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xec),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xed),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xee),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xef),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xfa),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xfe),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso7_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x01, 0xdc, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x01, 0xdd, 0xb8),
CS_ARRAY_INFO_ENTRY(0x01, 0xde, 0xb9),
CS_ARRAY_INFO_ENTRY(0x01, 0xdf, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0xbe),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xc0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x01, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x01, 0xf7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso8_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xc0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xc1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xc2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xc3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xc4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xc5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xc6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xc7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xc8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xc9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xca, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xcb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xcc, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xcd, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xce, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xcf, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xd1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xd3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xd4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xd5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xd6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x00, 0xd8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xd9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xda, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xdb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xea),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xec),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xed),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xee),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xef),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xfa),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xfe),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso9_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x01, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0xdd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso10_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xc0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xc1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xc2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xc3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xc4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xc5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xc6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xc7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xc8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xc9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xca, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xcb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xcc, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xcd, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xce, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xcf, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xd1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xd3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xd4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xd5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xd6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x00, 0xd8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xd9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xda, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xdb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xea),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xec),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xed),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xee),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xef),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xfa),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xfe),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, koi8r_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x01, 0xa3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xc0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x00, 0xc1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x00, 0xc2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x00, 0xc3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x00, 0xc4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x00, 0xc5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x00, 0xc6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x00, 0xc7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x00, 0xc8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x00, 0xc9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x00, 0xca, 0xea),
CS_ARRAY_INFO_ENTRY(0x00, 0xcb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x00, 0xcc, 0xec),
CS_ARRAY_INFO_ENTRY(0x00, 0xcd, 0xed),
CS_ARRAY_INFO_ENTRY(0x00, 0xce, 0xee),
CS_ARRAY_INFO_ENTRY(0x00, 0xcf, 0xef),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xd1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x00, 0xd3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x00, 0xd4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x00, 0xd5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x00, 0xd6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xd8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x00, 0xd9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x00, 0xda, 0xfa),
CS_ARRAY_INFO_ENTRY(0x00, 0xdb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xfe),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xff),
CS_ARRAY_INFO_ENTRY(0x01, 0xc0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x01, 0xc1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x01, 0xc2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x01, 0xc3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x01, 0xc4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x01, 0xc5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x01, 0xc6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x01, 0xc7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x01, 0xc8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x01, 0xc9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x01, 0xca, 0xea),
CS_ARRAY_INFO_ENTRY(0x01, 0xcb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x01, 0xcc, 0xec),
CS_ARRAY_INFO_ENTRY(0x01, 0xcd, 0xed),
CS_ARRAY_INFO_ENTRY(0x01, 0xce, 0xee),
CS_ARRAY_INFO_ENTRY(0x01, 0xcf, 0xef),
CS_ARRAY_INFO_ENTRY(0x01, 0xd0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x01, 0xd1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x01, 0xd2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x01, 0xd3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x01, 0xd4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x01, 0xd5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x01, 0xd6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x01, 0xd7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x01, 0xd8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x01, 0xd9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x01, 0xda, 0xfa),
CS_ARRAY_INFO_ENTRY(0x01, 0xdb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x01, 0xdc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x01, 0xdd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x01, 0xde, 0xfe),
CS_ARRAY_INFO_ENTRY(0x01, 0xdf, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, koi8u_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xb4), /* ie */
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xb6), /* i */
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xb7), /* ii */
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xbd), /* g'' */
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x01, 0xa3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4), /* IE */
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6), /* I */
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7), /* II */
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xc0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x00, 0xc1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x00, 0xc2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x00, 0xc3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x00, 0xc4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x00, 0xc5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x00, 0xc6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x00, 0xc7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x00, 0xc8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x00, 0xc9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x00, 0xca, 0xea),
CS_ARRAY_INFO_ENTRY(0x00, 0xcb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x00, 0xcc, 0xec),
CS_ARRAY_INFO_ENTRY(0x00, 0xcd, 0xed),
CS_ARRAY_INFO_ENTRY(0x00, 0xce, 0xee),
CS_ARRAY_INFO_ENTRY(0x00, 0xcf, 0xef),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xd1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x00, 0xd3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x00, 0xd4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x00, 0xd5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x00, 0xd6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xd8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x00, 0xd9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x00, 0xda, 0xfa),
CS_ARRAY_INFO_ENTRY(0x00, 0xdb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xfe),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xff),
CS_ARRAY_INFO_ENTRY(0x01, 0xc0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x01, 0xc1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x01, 0xc2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x01, 0xc3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x01, 0xc4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x01, 0xc5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x01, 0xc6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x01, 0xc7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x01, 0xc8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x01, 0xc9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x01, 0xca, 0xea),
CS_ARRAY_INFO_ENTRY(0x01, 0xcb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x01, 0xcc, 0xec),
CS_ARRAY_INFO_ENTRY(0x01, 0xcd, 0xed),
CS_ARRAY_INFO_ENTRY(0x01, 0xce, 0xee),
CS_ARRAY_INFO_ENTRY(0x01, 0xcf, 0xef),
CS_ARRAY_INFO_ENTRY(0x01, 0xd0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x01, 0xd1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x01, 0xd2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x01, 0xd3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x01, 0xd4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x01, 0xd5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x01, 0xd6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x01, 0xd7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x01, 0xd8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x01, 0xd9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x01, 0xda, 0xfa),
CS_ARRAY_INFO_ENTRY(0x01, 0xdb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x01, 0xdc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x01, 0xdd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x01, 0xde, 0xfe),
CS_ARRAY_INFO_ENTRY(0x01, 0xdf, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, cp1251_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x01, 0x90, 0x80),
CS_ARRAY_INFO_ENTRY(0x01, 0x83, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x01, 0x9a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x01, 0x9c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x01, 0x9d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x01, 0x9e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x01, 0x9f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x01, 0xa2, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa1),
CS_ARRAY_INFO_ENTRY(0x01, 0xbc, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x01, 0xb4, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x01, 0xb8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x01, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x01, 0xbf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x01, 0xb3, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xa3),
CS_ARRAY_INFO_ENTRY(0x01, 0xbe, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x01, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x01, 0xf7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x01, 0xff, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xdf)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso13_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0A, 0x0A),
CS_ARRAY_INFO_ENTRY(0x00, 0x0B, 0x0B),
CS_ARRAY_INFO_ENTRY(0x00, 0x0C, 0x0C),
CS_ARRAY_INFO_ENTRY(0x00, 0x0D, 0x0D),
CS_ARRAY_INFO_ENTRY(0x00, 0x0E, 0x0E),
CS_ARRAY_INFO_ENTRY(0x00, 0x0F, 0x0F),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1A, 0x1A),
CS_ARRAY_INFO_ENTRY(0x00, 0x1B, 0x1B),
CS_ARRAY_INFO_ENTRY(0x00, 0x1C, 0x1C),
CS_ARRAY_INFO_ENTRY(0x00, 0x1D, 0x1D),
CS_ARRAY_INFO_ENTRY(0x00, 0x1E, 0x1E),
CS_ARRAY_INFO_ENTRY(0x00, 0x1F, 0x1F),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2A, 0x2A),
CS_ARRAY_INFO_ENTRY(0x00, 0x2B, 0x2B),
CS_ARRAY_INFO_ENTRY(0x00, 0x2C, 0x2C),
CS_ARRAY_INFO_ENTRY(0x00, 0x2D, 0x2D),
CS_ARRAY_INFO_ENTRY(0x00, 0x2E, 0x2E),
CS_ARRAY_INFO_ENTRY(0x00, 0x2F, 0x2F),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3A, 0x3A),
CS_ARRAY_INFO_ENTRY(0x00, 0x3B, 0x3B),
CS_ARRAY_INFO_ENTRY(0x00, 0x3C, 0x3C),
CS_ARRAY_INFO_ENTRY(0x00, 0x3D, 0x3D),
CS_ARRAY_INFO_ENTRY(0x00, 0x3E, 0x3E),
CS_ARRAY_INFO_ENTRY(0x00, 0x3F, 0x3F),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6A, 0x4A),
CS_ARRAY_INFO_ENTRY(0x01, 0x6B, 0x4B),
CS_ARRAY_INFO_ENTRY(0x01, 0x6C, 0x4C),
CS_ARRAY_INFO_ENTRY(0x01, 0x6D, 0x4D),
CS_ARRAY_INFO_ENTRY(0x01, 0x6E, 0x4E),
CS_ARRAY_INFO_ENTRY(0x01, 0x6F, 0x4F),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7A, 0x5A),
CS_ARRAY_INFO_ENTRY(0x00, 0x5B, 0x5B),
CS_ARRAY_INFO_ENTRY(0x00, 0x5C, 0x5C),
CS_ARRAY_INFO_ENTRY(0x00, 0x5D, 0x5D),
CS_ARRAY_INFO_ENTRY(0x00, 0x5E, 0x5E),
CS_ARRAY_INFO_ENTRY(0x00, 0x5F, 0x5F),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6A, 0x4A),
CS_ARRAY_INFO_ENTRY(0x00, 0x6B, 0x4B),
CS_ARRAY_INFO_ENTRY(0x00, 0x6C, 0x4C),
CS_ARRAY_INFO_ENTRY(0x00, 0x6D, 0x4D),
CS_ARRAY_INFO_ENTRY(0x00, 0x6E, 0x4E),
CS_ARRAY_INFO_ENTRY(0x00, 0x6F, 0x4F),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7A, 0x5A),
CS_ARRAY_INFO_ENTRY(0x00, 0x7B, 0x7B),
CS_ARRAY_INFO_ENTRY(0x00, 0x7C, 0x7C),
CS_ARRAY_INFO_ENTRY(0x00, 0x7D, 0x7D),
CS_ARRAY_INFO_ENTRY(0x00, 0x7E, 0x7E),
CS_ARRAY_INFO_ENTRY(0x00, 0x7F, 0x7F),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8A, 0x8A),
CS_ARRAY_INFO_ENTRY(0x00, 0x8B, 0x8B),
CS_ARRAY_INFO_ENTRY(0x00, 0x8C, 0x8C),
CS_ARRAY_INFO_ENTRY(0x00, 0x8D, 0x8D),
CS_ARRAY_INFO_ENTRY(0x00, 0x8E, 0x8E),
CS_ARRAY_INFO_ENTRY(0x00, 0x8F, 0x8F),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9A, 0x9A),
CS_ARRAY_INFO_ENTRY(0x00, 0x9B, 0x9B),
CS_ARRAY_INFO_ENTRY(0x00, 0x9C, 0x9C),
CS_ARRAY_INFO_ENTRY(0x00, 0x9D, 0x9D),
CS_ARRAY_INFO_ENTRY(0x00, 0x9E, 0x9E),
CS_ARRAY_INFO_ENTRY(0x00, 0x9F, 0x9F),
CS_ARRAY_INFO_ENTRY(0x00, 0xA0, 0xA0),
CS_ARRAY_INFO_ENTRY(0x00, 0xA1, 0xA1),
CS_ARRAY_INFO_ENTRY(0x00, 0xA2, 0xA2),
CS_ARRAY_INFO_ENTRY(0x00, 0xA3, 0xA3),
CS_ARRAY_INFO_ENTRY(0x00, 0xA4, 0xA4),
CS_ARRAY_INFO_ENTRY(0x00, 0xA5, 0xA5),
CS_ARRAY_INFO_ENTRY(0x00, 0xA6, 0xA6),
CS_ARRAY_INFO_ENTRY(0x00, 0xA7, 0xA7),
CS_ARRAY_INFO_ENTRY(0x01, 0xB8, 0xA8),
CS_ARRAY_INFO_ENTRY(0x00, 0xA9, 0xA9),
CS_ARRAY_INFO_ENTRY(0x01, 0xBA, 0xAA),
CS_ARRAY_INFO_ENTRY(0x00, 0xAB, 0xAB),
CS_ARRAY_INFO_ENTRY(0x00, 0xAC, 0xAC),
CS_ARRAY_INFO_ENTRY(0x00, 0xAD, 0xAD),
CS_ARRAY_INFO_ENTRY(0x00, 0xAE, 0xAE),
CS_ARRAY_INFO_ENTRY(0x01, 0xBF, 0xAF),
CS_ARRAY_INFO_ENTRY(0x00, 0xB0, 0xB0),
CS_ARRAY_INFO_ENTRY(0x00, 0xB1, 0xB1),
CS_ARRAY_INFO_ENTRY(0x00, 0xB2, 0xB2),
CS_ARRAY_INFO_ENTRY(0x00, 0xB3, 0xB3),
CS_ARRAY_INFO_ENTRY(0x00, 0xB4, 0xB4),
CS_ARRAY_INFO_ENTRY(0x00, 0xB5, 0xB5),
CS_ARRAY_INFO_ENTRY(0x00, 0xB6, 0xB6),
CS_ARRAY_INFO_ENTRY(0x00, 0xB7, 0xB7),
CS_ARRAY_INFO_ENTRY(0x00, 0xB8, 0xA8),
CS_ARRAY_INFO_ENTRY(0x00, 0xB9, 0xB9),
CS_ARRAY_INFO_ENTRY(0x00, 0xBA, 0xAA),
CS_ARRAY_INFO_ENTRY(0x00, 0xBB, 0xBB),
CS_ARRAY_INFO_ENTRY(0x00, 0xBC, 0xBC),
CS_ARRAY_INFO_ENTRY(0x00, 0xBD, 0xBD),
CS_ARRAY_INFO_ENTRY(0x00, 0xBE, 0xBE),
CS_ARRAY_INFO_ENTRY(0x00, 0xBF, 0xAF),
CS_ARRAY_INFO_ENTRY(0x01, 0xE0, 0xC0),
CS_ARRAY_INFO_ENTRY(0x01, 0xE1, 0xC1),
CS_ARRAY_INFO_ENTRY(0x01, 0xE2, 0xC2),
CS_ARRAY_INFO_ENTRY(0x01, 0xE3, 0xC3),
CS_ARRAY_INFO_ENTRY(0x01, 0xE4, 0xC4),
CS_ARRAY_INFO_ENTRY(0x01, 0xE5, 0xC5),
CS_ARRAY_INFO_ENTRY(0x01, 0xE6, 0xC6),
CS_ARRAY_INFO_ENTRY(0x01, 0xE7, 0xC7),
CS_ARRAY_INFO_ENTRY(0x01, 0xE8, 0xC8),
CS_ARRAY_INFO_ENTRY(0x01, 0xE9, 0xC9),
CS_ARRAY_INFO_ENTRY(0x01, 0xEA, 0xCA),
CS_ARRAY_INFO_ENTRY(0x01, 0xEB, 0xCB),
CS_ARRAY_INFO_ENTRY(0x01, 0xEC, 0xCC),
CS_ARRAY_INFO_ENTRY(0x01, 0xED, 0xCD),
CS_ARRAY_INFO_ENTRY(0x01, 0xEE, 0xCE),
CS_ARRAY_INFO_ENTRY(0x01, 0xEF, 0xCF),
CS_ARRAY_INFO_ENTRY(0x01, 0xF0, 0xD0),
CS_ARRAY_INFO_ENTRY(0x01, 0xF1, 0xD1),
CS_ARRAY_INFO_ENTRY(0x01, 0xF2, 0xD2),
CS_ARRAY_INFO_ENTRY(0x01, 0xF3, 0xD3),
CS_ARRAY_INFO_ENTRY(0x01, 0xF4, 0xD4),
CS_ARRAY_INFO_ENTRY(0x01, 0xF5, 0xD5),
CS_ARRAY_INFO_ENTRY(0x01, 0xF6, 0xD6),
CS_ARRAY_INFO_ENTRY(0x00, 0xD7, 0xD7),
CS_ARRAY_INFO_ENTRY(0x01, 0xF8, 0xD8),
CS_ARRAY_INFO_ENTRY(0x01, 0xF9, 0xD9),
CS_ARRAY_INFO_ENTRY(0x01, 0xFA, 0xDA),
CS_ARRAY_INFO_ENTRY(0x01, 0xFB, 0xDB),
CS_ARRAY_INFO_ENTRY(0x01, 0xFC, 0xDC),
CS_ARRAY_INFO_ENTRY(0x01, 0xFD, 0xDD),
CS_ARRAY_INFO_ENTRY(0x01, 0xFE, 0xDE),
CS_ARRAY_INFO_ENTRY(0x00, 0xDF, 0xDF),
CS_ARRAY_INFO_ENTRY(0x00, 0xE0, 0xC0),
CS_ARRAY_INFO_ENTRY(0x00, 0xE1, 0xC1),
CS_ARRAY_INFO_ENTRY(0x00, 0xE2, 0xC2),
CS_ARRAY_INFO_ENTRY(0x00, 0xE3, 0xC3),
CS_ARRAY_INFO_ENTRY(0x00, 0xE4, 0xC4),
CS_ARRAY_INFO_ENTRY(0x00, 0xE5, 0xC5),
CS_ARRAY_INFO_ENTRY(0x00, 0xE6, 0xC6),
CS_ARRAY_INFO_ENTRY(0x00, 0xE7, 0xC7),
CS_ARRAY_INFO_ENTRY(0x00, 0xE8, 0xC8),
CS_ARRAY_INFO_ENTRY(0x00, 0xE9, 0xC9),
CS_ARRAY_INFO_ENTRY(0x00, 0xEA, 0xCA),
CS_ARRAY_INFO_ENTRY(0x00, 0xEB, 0xCB),
CS_ARRAY_INFO_ENTRY(0x00, 0xEC, 0xCC),
CS_ARRAY_INFO_ENTRY(0x00, 0xED, 0xCD),
CS_ARRAY_INFO_ENTRY(0x00, 0xEE, 0xCE),
CS_ARRAY_INFO_ENTRY(0x00, 0xEF, 0xCF),
CS_ARRAY_INFO_ENTRY(0x00, 0xF0, 0xD0),
CS_ARRAY_INFO_ENTRY(0x00, 0xF1, 0xD1),
CS_ARRAY_INFO_ENTRY(0x00, 0xF2, 0xD2),
CS_ARRAY_INFO_ENTRY(0x00, 0xF3, 0xD3),
CS_ARRAY_INFO_ENTRY(0x00, 0xF4, 0xD4),
CS_ARRAY_INFO_ENTRY(0x00, 0xF5, 0xD5),
CS_ARRAY_INFO_ENTRY(0x00, 0xF6, 0xD6),
CS_ARRAY_INFO_ENTRY(0x00, 0xF7, 0xF7),
CS_ARRAY_INFO_ENTRY(0x00, 0xF8, 0xD8),
CS_ARRAY_INFO_ENTRY(0x00, 0xF9, 0xD9),
CS_ARRAY_INFO_ENTRY(0x00, 0xFA, 0xDA),
CS_ARRAY_INFO_ENTRY(0x00, 0xFB, 0xDB),
CS_ARRAY_INFO_ENTRY(0x00, 0xFC, 0xDC),
CS_ARRAY_INFO_ENTRY(0x00, 0xFD, 0xDD),
CS_ARRAY_INFO_ENTRY(0x00, 0xFE, 0xDE),
CS_ARRAY_INFO_ENTRY(0x00, 0xFF, 0xFF)
CS_ARRAY_END


CS_ARRAY(struct cs_info, iso14_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x01, 0xa2, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x01, 0xa5, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa4),
CS_ARRAY_INFO_ENTRY(0x01, 0xa6, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x01, 0xb8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x01, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xa6),
CS_ARRAY_INFO_ENTRY(0x01, 0xbc, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x01, 0xff, 0xaf),
CS_ARRAY_INFO_ENTRY(0x01, 0xb1, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb0),
CS_ARRAY_INFO_ENTRY(0x01, 0xb3, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb2),
CS_ARRAY_INFO_ENTRY(0x01, 0xb5, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x01, 0xb9, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xaa),
CS_ARRAY_INFO_ENTRY(0x01, 0xbf, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xac),
CS_ARRAY_INFO_ENTRY(0x01, 0xbe, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbb),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x01, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x01, 0xf7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iso15_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x01, 0xa8, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x01, 0xb8, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x01, 0xbd, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbc),
CS_ARRAY_INFO_ENTRY(0x01, 0xff, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x01, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x01, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x01, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x01, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x01, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x01, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x01, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x01, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x01, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x01, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x01, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x01, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x01, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x01, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x01, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x01, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x01, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x01, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x01, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x01, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x01, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x01, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x01, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x01, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x01, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x01, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x01, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x01, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x01, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x01, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xbe)
CS_ARRAY_END

CS_ARRAY(struct cs_info, iscii_devanagari_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xc0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xc1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xc2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xc3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xc4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xc5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xc6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xc7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xc8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xc9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xca, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xcb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xcc, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xcd, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xce, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xcf, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xd1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xd3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xd4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xd5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xd6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x00, 0xd8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xd9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xda, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xdb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xea),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xec),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xed),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xee),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xef),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xfa),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xfe),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

CS_ARRAY(struct cs_info, tis620_tbl)
CS_ARRAY_INFO_ENTRY(0x00, 0x00, 0x00),
CS_ARRAY_INFO_ENTRY(0x00, 0x01, 0x01),
CS_ARRAY_INFO_ENTRY(0x00, 0x02, 0x02),
CS_ARRAY_INFO_ENTRY(0x00, 0x03, 0x03),
CS_ARRAY_INFO_ENTRY(0x00, 0x04, 0x04),
CS_ARRAY_INFO_ENTRY(0x00, 0x05, 0x05),
CS_ARRAY_INFO_ENTRY(0x00, 0x06, 0x06),
CS_ARRAY_INFO_ENTRY(0x00, 0x07, 0x07),
CS_ARRAY_INFO_ENTRY(0x00, 0x08, 0x08),
CS_ARRAY_INFO_ENTRY(0x00, 0x09, 0x09),
CS_ARRAY_INFO_ENTRY(0x00, 0x0a, 0x0a),
CS_ARRAY_INFO_ENTRY(0x00, 0x0b, 0x0b),
CS_ARRAY_INFO_ENTRY(0x00, 0x0c, 0x0c),
CS_ARRAY_INFO_ENTRY(0x00, 0x0d, 0x0d),
CS_ARRAY_INFO_ENTRY(0x00, 0x0e, 0x0e),
CS_ARRAY_INFO_ENTRY(0x00, 0x0f, 0x0f),
CS_ARRAY_INFO_ENTRY(0x00, 0x10, 0x10),
CS_ARRAY_INFO_ENTRY(0x00, 0x11, 0x11),
CS_ARRAY_INFO_ENTRY(0x00, 0x12, 0x12),
CS_ARRAY_INFO_ENTRY(0x00, 0x13, 0x13),
CS_ARRAY_INFO_ENTRY(0x00, 0x14, 0x14),
CS_ARRAY_INFO_ENTRY(0x00, 0x15, 0x15),
CS_ARRAY_INFO_ENTRY(0x00, 0x16, 0x16),
CS_ARRAY_INFO_ENTRY(0x00, 0x17, 0x17),
CS_ARRAY_INFO_ENTRY(0x00, 0x18, 0x18),
CS_ARRAY_INFO_ENTRY(0x00, 0x19, 0x19),
CS_ARRAY_INFO_ENTRY(0x00, 0x1a, 0x1a),
CS_ARRAY_INFO_ENTRY(0x00, 0x1b, 0x1b),
CS_ARRAY_INFO_ENTRY(0x00, 0x1c, 0x1c),
CS_ARRAY_INFO_ENTRY(0x00, 0x1d, 0x1d),
CS_ARRAY_INFO_ENTRY(0x00, 0x1e, 0x1e),
CS_ARRAY_INFO_ENTRY(0x00, 0x1f, 0x1f),
CS_ARRAY_INFO_ENTRY(0x00, 0x20, 0x20),
CS_ARRAY_INFO_ENTRY(0x00, 0x21, 0x21),
CS_ARRAY_INFO_ENTRY(0x00, 0x22, 0x22),
CS_ARRAY_INFO_ENTRY(0x00, 0x23, 0x23),
CS_ARRAY_INFO_ENTRY(0x00, 0x24, 0x24),
CS_ARRAY_INFO_ENTRY(0x00, 0x25, 0x25),
CS_ARRAY_INFO_ENTRY(0x00, 0x26, 0x26),
CS_ARRAY_INFO_ENTRY(0x00, 0x27, 0x27),
CS_ARRAY_INFO_ENTRY(0x00, 0x28, 0x28),
CS_ARRAY_INFO_ENTRY(0x00, 0x29, 0x29),
CS_ARRAY_INFO_ENTRY(0x00, 0x2a, 0x2a),
CS_ARRAY_INFO_ENTRY(0x00, 0x2b, 0x2b),
CS_ARRAY_INFO_ENTRY(0x00, 0x2c, 0x2c),
CS_ARRAY_INFO_ENTRY(0x00, 0x2d, 0x2d),
CS_ARRAY_INFO_ENTRY(0x00, 0x2e, 0x2e),
CS_ARRAY_INFO_ENTRY(0x00, 0x2f, 0x2f),
CS_ARRAY_INFO_ENTRY(0x00, 0x30, 0x30),
CS_ARRAY_INFO_ENTRY(0x00, 0x31, 0x31),
CS_ARRAY_INFO_ENTRY(0x00, 0x32, 0x32),
CS_ARRAY_INFO_ENTRY(0x00, 0x33, 0x33),
CS_ARRAY_INFO_ENTRY(0x00, 0x34, 0x34),
CS_ARRAY_INFO_ENTRY(0x00, 0x35, 0x35),
CS_ARRAY_INFO_ENTRY(0x00, 0x36, 0x36),
CS_ARRAY_INFO_ENTRY(0x00, 0x37, 0x37),
CS_ARRAY_INFO_ENTRY(0x00, 0x38, 0x38),
CS_ARRAY_INFO_ENTRY(0x00, 0x39, 0x39),
CS_ARRAY_INFO_ENTRY(0x00, 0x3a, 0x3a),
CS_ARRAY_INFO_ENTRY(0x00, 0x3b, 0x3b),
CS_ARRAY_INFO_ENTRY(0x00, 0x3c, 0x3c),
CS_ARRAY_INFO_ENTRY(0x00, 0x3d, 0x3d),
CS_ARRAY_INFO_ENTRY(0x00, 0x3e, 0x3e),
CS_ARRAY_INFO_ENTRY(0x00, 0x3f, 0x3f),
CS_ARRAY_INFO_ENTRY(0x00, 0x40, 0x40),
CS_ARRAY_INFO_ENTRY(0x01, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x01, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x01, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x01, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x01, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x01, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x01, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x01, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x01, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x01, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x01, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x01, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x01, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x01, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x01, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x01, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x01, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x01, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x01, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x01, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x01, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x01, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x01, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x01, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x01, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x01, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x5b, 0x5b),
CS_ARRAY_INFO_ENTRY(0x00, 0x5c, 0x5c),
CS_ARRAY_INFO_ENTRY(0x00, 0x5d, 0x5d),
CS_ARRAY_INFO_ENTRY(0x00, 0x5e, 0x5e),
CS_ARRAY_INFO_ENTRY(0x00, 0x5f, 0x5f),
CS_ARRAY_INFO_ENTRY(0x00, 0x60, 0x60),
CS_ARRAY_INFO_ENTRY(0x00, 0x61, 0x41),
CS_ARRAY_INFO_ENTRY(0x00, 0x62, 0x42),
CS_ARRAY_INFO_ENTRY(0x00, 0x63, 0x43),
CS_ARRAY_INFO_ENTRY(0x00, 0x64, 0x44),
CS_ARRAY_INFO_ENTRY(0x00, 0x65, 0x45),
CS_ARRAY_INFO_ENTRY(0x00, 0x66, 0x46),
CS_ARRAY_INFO_ENTRY(0x00, 0x67, 0x47),
CS_ARRAY_INFO_ENTRY(0x00, 0x68, 0x48),
CS_ARRAY_INFO_ENTRY(0x00, 0x69, 0x49),
CS_ARRAY_INFO_ENTRY(0x00, 0x6a, 0x4a),
CS_ARRAY_INFO_ENTRY(0x00, 0x6b, 0x4b),
CS_ARRAY_INFO_ENTRY(0x00, 0x6c, 0x4c),
CS_ARRAY_INFO_ENTRY(0x00, 0x6d, 0x4d),
CS_ARRAY_INFO_ENTRY(0x00, 0x6e, 0x4e),
CS_ARRAY_INFO_ENTRY(0x00, 0x6f, 0x4f),
CS_ARRAY_INFO_ENTRY(0x00, 0x70, 0x50),
CS_ARRAY_INFO_ENTRY(0x00, 0x71, 0x51),
CS_ARRAY_INFO_ENTRY(0x00, 0x72, 0x52),
CS_ARRAY_INFO_ENTRY(0x00, 0x73, 0x53),
CS_ARRAY_INFO_ENTRY(0x00, 0x74, 0x54),
CS_ARRAY_INFO_ENTRY(0x00, 0x75, 0x55),
CS_ARRAY_INFO_ENTRY(0x00, 0x76, 0x56),
CS_ARRAY_INFO_ENTRY(0x00, 0x77, 0x57),
CS_ARRAY_INFO_ENTRY(0x00, 0x78, 0x58),
CS_ARRAY_INFO_ENTRY(0x00, 0x79, 0x59),
CS_ARRAY_INFO_ENTRY(0x00, 0x7a, 0x5a),
CS_ARRAY_INFO_ENTRY(0x00, 0x7b, 0x7b),
CS_ARRAY_INFO_ENTRY(0x00, 0x7c, 0x7c),
CS_ARRAY_INFO_ENTRY(0x00, 0x7d, 0x7d),
CS_ARRAY_INFO_ENTRY(0x00, 0x7e, 0x7e),
CS_ARRAY_INFO_ENTRY(0x00, 0x7f, 0x7f),
CS_ARRAY_INFO_ENTRY(0x00, 0x80, 0x80),
CS_ARRAY_INFO_ENTRY(0x00, 0x81, 0x81),
CS_ARRAY_INFO_ENTRY(0x00, 0x82, 0x82),
CS_ARRAY_INFO_ENTRY(0x00, 0x83, 0x83),
CS_ARRAY_INFO_ENTRY(0x00, 0x84, 0x84),
CS_ARRAY_INFO_ENTRY(0x00, 0x85, 0x85),
CS_ARRAY_INFO_ENTRY(0x00, 0x86, 0x86),
CS_ARRAY_INFO_ENTRY(0x00, 0x87, 0x87),
CS_ARRAY_INFO_ENTRY(0x00, 0x88, 0x88),
CS_ARRAY_INFO_ENTRY(0x00, 0x89, 0x89),
CS_ARRAY_INFO_ENTRY(0x00, 0x8a, 0x8a),
CS_ARRAY_INFO_ENTRY(0x00, 0x8b, 0x8b),
CS_ARRAY_INFO_ENTRY(0x00, 0x8c, 0x8c),
CS_ARRAY_INFO_ENTRY(0x00, 0x8d, 0x8d),
CS_ARRAY_INFO_ENTRY(0x00, 0x8e, 0x8e),
CS_ARRAY_INFO_ENTRY(0x00, 0x8f, 0x8f),
CS_ARRAY_INFO_ENTRY(0x00, 0x90, 0x90),
CS_ARRAY_INFO_ENTRY(0x00, 0x91, 0x91),
CS_ARRAY_INFO_ENTRY(0x00, 0x92, 0x92),
CS_ARRAY_INFO_ENTRY(0x00, 0x93, 0x93),
CS_ARRAY_INFO_ENTRY(0x00, 0x94, 0x94),
CS_ARRAY_INFO_ENTRY(0x00, 0x95, 0x95),
CS_ARRAY_INFO_ENTRY(0x00, 0x96, 0x96),
CS_ARRAY_INFO_ENTRY(0x00, 0x97, 0x97),
CS_ARRAY_INFO_ENTRY(0x00, 0x98, 0x98),
CS_ARRAY_INFO_ENTRY(0x00, 0x99, 0x99),
CS_ARRAY_INFO_ENTRY(0x00, 0x9a, 0x9a),
CS_ARRAY_INFO_ENTRY(0x00, 0x9b, 0x9b),
CS_ARRAY_INFO_ENTRY(0x00, 0x9c, 0x9c),
CS_ARRAY_INFO_ENTRY(0x00, 0x9d, 0x9d),
CS_ARRAY_INFO_ENTRY(0x00, 0x9e, 0x9e),
CS_ARRAY_INFO_ENTRY(0x00, 0x9f, 0x9f),
CS_ARRAY_INFO_ENTRY(0x00, 0xa0, 0xa0),
CS_ARRAY_INFO_ENTRY(0x00, 0xa1, 0xa1),
CS_ARRAY_INFO_ENTRY(0x00, 0xa2, 0xa2),
CS_ARRAY_INFO_ENTRY(0x00, 0xa3, 0xa3),
CS_ARRAY_INFO_ENTRY(0x00, 0xa4, 0xa4),
CS_ARRAY_INFO_ENTRY(0x00, 0xa5, 0xa5),
CS_ARRAY_INFO_ENTRY(0x00, 0xa6, 0xa6),
CS_ARRAY_INFO_ENTRY(0x00, 0xa7, 0xa7),
CS_ARRAY_INFO_ENTRY(0x00, 0xa8, 0xa8),
CS_ARRAY_INFO_ENTRY(0x00, 0xa9, 0xa9),
CS_ARRAY_INFO_ENTRY(0x00, 0xaa, 0xaa),
CS_ARRAY_INFO_ENTRY(0x00, 0xab, 0xab),
CS_ARRAY_INFO_ENTRY(0x00, 0xac, 0xac),
CS_ARRAY_INFO_ENTRY(0x00, 0xad, 0xad),
CS_ARRAY_INFO_ENTRY(0x00, 0xae, 0xae),
CS_ARRAY_INFO_ENTRY(0x00, 0xaf, 0xaf),
CS_ARRAY_INFO_ENTRY(0x00, 0xb0, 0xb0),
CS_ARRAY_INFO_ENTRY(0x00, 0xb1, 0xb1),
CS_ARRAY_INFO_ENTRY(0x00, 0xb2, 0xb2),
CS_ARRAY_INFO_ENTRY(0x00, 0xb3, 0xb3),
CS_ARRAY_INFO_ENTRY(0x00, 0xb4, 0xb4),
CS_ARRAY_INFO_ENTRY(0x00, 0xb5, 0xb5),
CS_ARRAY_INFO_ENTRY(0x00, 0xb6, 0xb6),
CS_ARRAY_INFO_ENTRY(0x00, 0xb7, 0xb7),
CS_ARRAY_INFO_ENTRY(0x00, 0xb8, 0xb8),
CS_ARRAY_INFO_ENTRY(0x00, 0xb9, 0xb9),
CS_ARRAY_INFO_ENTRY(0x00, 0xba, 0xba),
CS_ARRAY_INFO_ENTRY(0x00, 0xbb, 0xbb),
CS_ARRAY_INFO_ENTRY(0x00, 0xbc, 0xbc),
CS_ARRAY_INFO_ENTRY(0x00, 0xbd, 0xbd),
CS_ARRAY_INFO_ENTRY(0x00, 0xbe, 0xbe),
CS_ARRAY_INFO_ENTRY(0x00, 0xbf, 0xbf),
CS_ARRAY_INFO_ENTRY(0x00, 0xc0, 0xc0),
CS_ARRAY_INFO_ENTRY(0x00, 0xc1, 0xc1),
CS_ARRAY_INFO_ENTRY(0x00, 0xc2, 0xc2),
CS_ARRAY_INFO_ENTRY(0x00, 0xc3, 0xc3),
CS_ARRAY_INFO_ENTRY(0x00, 0xc4, 0xc4),
CS_ARRAY_INFO_ENTRY(0x00, 0xc5, 0xc5),
CS_ARRAY_INFO_ENTRY(0x00, 0xc6, 0xc6),
CS_ARRAY_INFO_ENTRY(0x00, 0xc7, 0xc7),
CS_ARRAY_INFO_ENTRY(0x00, 0xc8, 0xc8),
CS_ARRAY_INFO_ENTRY(0x00, 0xc9, 0xc9),
CS_ARRAY_INFO_ENTRY(0x00, 0xca, 0xca),
CS_ARRAY_INFO_ENTRY(0x00, 0xcb, 0xcb),
CS_ARRAY_INFO_ENTRY(0x00, 0xcc, 0xcc),
CS_ARRAY_INFO_ENTRY(0x00, 0xcd, 0xcd),
CS_ARRAY_INFO_ENTRY(0x00, 0xce, 0xce),
CS_ARRAY_INFO_ENTRY(0x00, 0xcf, 0xcf),
CS_ARRAY_INFO_ENTRY(0x00, 0xd0, 0xd0),
CS_ARRAY_INFO_ENTRY(0x00, 0xd1, 0xd1),
CS_ARRAY_INFO_ENTRY(0x00, 0xd2, 0xd2),
CS_ARRAY_INFO_ENTRY(0x00, 0xd3, 0xd3),
CS_ARRAY_INFO_ENTRY(0x00, 0xd4, 0xd4),
CS_ARRAY_INFO_ENTRY(0x00, 0xd5, 0xd5),
CS_ARRAY_INFO_ENTRY(0x00, 0xd6, 0xd6),
CS_ARRAY_INFO_ENTRY(0x00, 0xd7, 0xd7),
CS_ARRAY_INFO_ENTRY(0x00, 0xd8, 0xd8),
CS_ARRAY_INFO_ENTRY(0x00, 0xd9, 0xd9),
CS_ARRAY_INFO_ENTRY(0x00, 0xda, 0xda),
CS_ARRAY_INFO_ENTRY(0x00, 0xdb, 0xdb),
CS_ARRAY_INFO_ENTRY(0x00, 0xdc, 0xdc),
CS_ARRAY_INFO_ENTRY(0x00, 0xdd, 0xdd),
CS_ARRAY_INFO_ENTRY(0x00, 0xde, 0xde),
CS_ARRAY_INFO_ENTRY(0x00, 0xdf, 0xdf),
CS_ARRAY_INFO_ENTRY(0x00, 0xe0, 0xe0),
CS_ARRAY_INFO_ENTRY(0x00, 0xe1, 0xe1),
CS_ARRAY_INFO_ENTRY(0x00, 0xe2, 0xe2),
CS_ARRAY_INFO_ENTRY(0x00, 0xe3, 0xe3),
CS_ARRAY_INFO_ENTRY(0x00, 0xe4, 0xe4),
CS_ARRAY_INFO_ENTRY(0x00, 0xe5, 0xe5),
CS_ARRAY_INFO_ENTRY(0x00, 0xe6, 0xe6),
CS_ARRAY_INFO_ENTRY(0x00, 0xe7, 0xe7),
CS_ARRAY_INFO_ENTRY(0x00, 0xe8, 0xe8),
CS_ARRAY_INFO_ENTRY(0x00, 0xe9, 0xe9),
CS_ARRAY_INFO_ENTRY(0x00, 0xea, 0xea),
CS_ARRAY_INFO_ENTRY(0x00, 0xeb, 0xeb),
CS_ARRAY_INFO_ENTRY(0x00, 0xec, 0xec),
CS_ARRAY_INFO_ENTRY(0x00, 0xed, 0xed),
CS_ARRAY_INFO_ENTRY(0x00, 0xee, 0xee),
CS_ARRAY_INFO_ENTRY(0x00, 0xef, 0xef),
CS_ARRAY_INFO_ENTRY(0x00, 0xf0, 0xf0),
CS_ARRAY_INFO_ENTRY(0x00, 0xf1, 0xf1),
CS_ARRAY_INFO_ENTRY(0x00, 0xf2, 0xf2),
CS_ARRAY_INFO_ENTRY(0x00, 0xf3, 0xf3),
CS_ARRAY_INFO_ENTRY(0x00, 0xf4, 0xf4),
CS_ARRAY_INFO_ENTRY(0x00, 0xf5, 0xf5),
CS_ARRAY_INFO_ENTRY(0x00, 0xf6, 0xf6),
CS_ARRAY_INFO_ENTRY(0x00, 0xf7, 0xf7),
CS_ARRAY_INFO_ENTRY(0x00, 0xf8, 0xf8),
CS_ARRAY_INFO_ENTRY(0x00, 0xf9, 0xf9),
CS_ARRAY_INFO_ENTRY(0x00, 0xfa, 0xfa),
CS_ARRAY_INFO_ENTRY(0x00, 0xfb, 0xfb),
CS_ARRAY_INFO_ENTRY(0x00, 0xfc, 0xfc),
CS_ARRAY_INFO_ENTRY(0x00, 0xfd, 0xfd),
CS_ARRAY_INFO_ENTRY(0x00, 0xfe, 0xfe),
CS_ARRAY_INFO_ENTRY(0x00, 0xff, 0xff)
CS_ARRAY_END

// struct ent_entry definition moved to a more global place.

//static struct enc_entry encds[] = {
CS_ARRAY(struct enc_entry, encds)
CS_ARRAY_ENC_ENTRY("iso88591",        iso1_tbl),
CS_ARRAY_ENC_ENTRY("iso88592",        iso2_tbl),
CS_ARRAY_ENC_ENTRY("iso88593",        iso3_tbl),
CS_ARRAY_ENC_ENTRY("iso88594",        iso4_tbl),
CS_ARRAY_ENC_ENTRY("iso88595",        iso5_tbl),
CS_ARRAY_ENC_ENTRY("iso88596",        iso6_tbl),
CS_ARRAY_ENC_ENTRY("iso88597",        iso7_tbl),
CS_ARRAY_ENC_ENTRY("iso88598",        iso8_tbl),
CS_ARRAY_ENC_ENTRY("iso88599",        iso9_tbl),
CS_ARRAY_ENC_ENTRY("iso885910",       iso10_tbl),
CS_ARRAY_ENC_ENTRY("tis620",           tis620_tbl),
CS_ARRAY_ENC_ENTRY("tis6202533",           tis620_tbl),
CS_ARRAY_ENC_ENTRY("iso885911",           tis620_tbl),
CS_ARRAY_ENC_ENTRY("iso885913",           iso13_tbl),
CS_ARRAY_ENC_ENTRY("iso885914",           iso14_tbl),
CS_ARRAY_ENC_ENTRY("iso885915",           iso15_tbl),
CS_ARRAY_ENC_ENTRY("koi8r",           koi8r_tbl),
CS_ARRAY_ENC_ENTRY("koi8u",           koi8u_tbl),
CS_ARRAY_ENC_ENTRY("cp1251", cp1251_tbl),
CS_ARRAY_ENC_ENTRY("microsoftcp1251", cp1251_tbl),
CS_ARRAY_ENC_ENTRY("xisciias", iscii_devanagari_tbl),
CS_ARRAY_ENC_ENTRY("isciidevanagari", iscii_devanagari_tbl)
CS_ARRAY_END

#ifdef HAS_COMPLEX_GLOBALS
# define NUM_ENCDS (sizeof(encds) / sizeof(encds[0]))
#else
# define NUM_ENCDS HUNSPELL_NUM_ENCDS
# define encds g_hunspell_encds
#endif

/* map to lower case and remove non alphanumeric chars */
static void toAsciiLowerAndRemoveNonAlphanumeric( const char* pName, char* pBuf )
{
    while ( *pName )
    {
        /* A-Z */
        if ( (*pName >= 0x41) && (*pName <= 0x5A) )
        {
            *pBuf = (*pName)+0x20;  /* toAsciiLower */
            pBuf++;
        }
        /* a-z, 0-9 */
        else if ( ((*pName >= 0x61) && (*pName <= 0x7A)) ||
                  ((*pName >= 0x30) && (*pName <= 0x39)) )
        {
            *pBuf = *pName;
            pBuf++;
        }

        pName++;
    }

    *pBuf = '\0';
}

struct cs_info * get_current_cs(const char * es) {
  char *normalized_encoding = OP_NEWA(char, op_strlen(es)+1);
  toAsciiLowerAndRemoveNonAlphanumeric(es, normalized_encoding);

  struct cs_info * ccs = NULL;
  int n = NUM_ENCDS;
  OP_ASSERT(NUM_ENCDS == HUNSPELL_NUM_ENCDS);
  for (int i = 0; i < n; i++) {
    if (op_strcmp(normalized_encoding,encds[i].enc_name) == 0) {
      ccs = encds[i].cs_table;
      break;
    }
  }

  OP_DELETEA(normalized_encoding);

  if (!ccs) {
    HUNSPELL_WARNING(hunspell_stderr, "error: unknown encoding %s: using %s as fallback\n", es, encds[0].enc_name);
    ccs = encds[0].cs_table;
  }

  return ccs;
}
#else
// XXX This function was rewritten for mozilla. Instead of storing the
// conversion tables static in this file, create them when needed
// with help the mozilla backend.
struct cs_info * get_current_cs(const char * es) {
  struct cs_info *ccs;

  nsCOMPtr<nsIUnicodeEncoder> encoder; 
  nsCOMPtr<nsIUnicodeDecoder> decoder; 

  nsresult rv;
  nsCOMPtr<nsICharsetConverterManager> ccm = do_GetService(kCharsetConverterManagerCID, &rv);
  if (NS_FAILED(rv))
    return nsnull;

  rv = ccm->GetUnicodeEncoder(es, getter_AddRefs(encoder));
  if (NS_FAILED(rv))
    return nsnull;
  encoder->SetOutputErrorBehavior(encoder->kOnError_Signal, nsnull, '?');
  rv = ccm->GetUnicodeDecoder(es, getter_AddRefs(decoder));
  if (NS_FAILED(rv))
    return nsnull;
  decoder->SetInputErrorBehavior(decoder->kOnError_Signal);

  if (NS_FAILED(rv))
    return nsnull;

  ccs = new cs_info[256];

  for (unsigned int i = 0; i <= 0xff; ++i) {
    PRBool success = PR_FALSE;
    // We want to find the upper/lowercase equivalents of each byte
    // in this 1-byte character encoding.  Call our encoding/decoding
    // APIs separately for each byte since they may reject some of the
    // bytes, and we want to handle errors separately for each byte.
    char lower, upper;
    do {
      if (i == 0)
        break;
      const char source = char(i);
      PRUnichar uni, uniCased;
      PRInt32 charLength = 1, uniLength = 1;

      rv = decoder->Convert(&source, &charLength, &uni, &uniLength);
      // Explicitly check NS_OK because we don't want to allow
      // NS_OK_UDEC_MOREOUTPUT or NS_OK_UDEC_MOREINPUT.
      if (rv != NS_OK || charLength != 1 || uniLength != 1)
        break;
      uniCased = ToLowerCase(uni);
      rv = encoder->Convert(&uniCased, &uniLength, &lower, &charLength);
      // Explicitly check NS_OK because we don't want to allow
      // NS_OK_UDEC_MOREOUTPUT or NS_OK_UDEC_MOREINPUT.
      if (rv != NS_OK || charLength != 1 || uniLength != 1)
        break;

      uniCased = ToUpperCase(uni);
      rv = encoder->Convert(&uniCased, &uniLength, &upper, &charLength);
      // Explicitly check NS_OK because we don't want to allow
      // NS_OK_UDEC_MOREOUTPUT or NS_OK_UDEC_MOREINPUT.
      if (rv != NS_OK || charLength != 1 || uniLength != 1)
        break;

      success = PR_TRUE;
    } while (0);

    if (success) {
      ccs[i].cupper = upper;
      ccs[i].clower = lower;
    } else {
      ccs[i].cupper = i;
      ccs[i].clower = i;
    }

    if (ccs[i].clower != (unsigned char)i)
      ccs[i].ccase = true;
    else
      ccs[i].ccase = false;
  }

  return ccs;
}
#endif

// primitive op_isalpha() replacement for tokenization
char * get_casechars(const char * enc) {
    struct cs_info * csconv = get_current_cs(enc);
    char expw[257]; // ARRAY OK 2009-02-09 jb
    char * p =  expw;
    for (int i = 0; i <= 255; i++) {
        if ((csconv[i].cupper != csconv[i].clower)) {
    	    *p = (char) i;
    	    p++;
        }
    }
    *p = '\0';
#ifdef MOZILLA_CLIENT
    OP_DELETEA( csconv);
#endif
    return mystrdup(expw);
}

// language to encoding default map

// struct lang_map definition moved to a more global place

//struct lang_map lang2enc[] = {
CS_ARRAY(struct lang_map, lang2enc)
CS_ARRAY_LANG_MAP_ENTRY("ar", LANG_ar),
CS_ARRAY_LANG_MAP_ENTRY("az", LANG_az),
CS_ARRAY_LANG_MAP_ENTRY("bg", LANG_bg),
CS_ARRAY_LANG_MAP_ENTRY("ca", LANG_ca),
CS_ARRAY_LANG_MAP_ENTRY("cs", LANG_cs),
CS_ARRAY_LANG_MAP_ENTRY("da", LANG_da),
CS_ARRAY_LANG_MAP_ENTRY("de", LANG_de),
CS_ARRAY_LANG_MAP_ENTRY("el", LANG_el),
CS_ARRAY_LANG_MAP_ENTRY("en", LANG_en),
CS_ARRAY_LANG_MAP_ENTRY("es", LANG_es),
CS_ARRAY_LANG_MAP_ENTRY("eu", LANG_eu),
CS_ARRAY_LANG_MAP_ENTRY("gl", LANG_gl),
CS_ARRAY_LANG_MAP_ENTRY("fr", LANG_fr),
CS_ARRAY_LANG_MAP_ENTRY("hr", LANG_hr),
CS_ARRAY_LANG_MAP_ENTRY("hu", LANG_hu),
CS_ARRAY_LANG_MAP_ENTRY("hu_HU", LANG_hu),
CS_ARRAY_LANG_MAP_ENTRY("it", LANG_it),
CS_ARRAY_LANG_MAP_ENTRY("la", LANG_la),
CS_ARRAY_LANG_MAP_ENTRY("lv", LANG_lv),
CS_ARRAY_LANG_MAP_ENTRY("nl", LANG_nl),
CS_ARRAY_LANG_MAP_ENTRY("pl", LANG_pl),
CS_ARRAY_LANG_MAP_ENTRY("pt", LANG_pt),
CS_ARRAY_LANG_MAP_ENTRY("sv", LANG_sv),
CS_ARRAY_LANG_MAP_ENTRY("tr_TR", LANG_tr),
CS_ARRAY_LANG_MAP_ENTRY("tr", LANG_tr),
CS_ARRAY_LANG_MAP_ENTRY("ru", LANG_ru),
CS_ARRAY_LANG_MAP_ENTRY("uk", LANG_uk)
CS_ARRAY_END

#ifdef HAS_COMPLEX_GLOBALS
# define NUM_LANGS (sizeof(lang2enc) / sizeof(lang2enc[0]))
#else
# define NUM_LANGS HUNSPELL_NUM_LANGS
# define lang2enc g_hunspell_lang2enc
#endif

int get_lang_num(const char * lang) {
  int n = sizeof(lang2enc) / sizeof(lang2enc[0]);
  for (int i = 0; i < n; i++) {
    if (op_strcmp(lang, lang2enc[i].lang) == 0) {
      return lang2enc[i].num;
    }
  }
  return LANG_xx;
}

#ifdef HUNSPELL_USE_UTF_TBL
#ifndef OPENOFFICEORG
#ifndef MOZILLA_CLIENT
int initialize_utf_tbl() {
  utf_tbl_count++;
  if (utf_tbl) return 0;
  utf_tbl = (unicode_info2 *) op_malloc(CONTSIZE * sizeof(unicode_info2));
  if (utf_tbl) {
    size_t j;
    for (j = 0; j < CONTSIZE; j++) {
      utf_tbl[j].cletter = 0;
      utf_tbl[j].clower = (unsigned short) j;
      utf_tbl[j].cupper = (unsigned short) j;
    }
    for (j = 0; j < UTF_LST_LEN; j++) {
      utf_tbl[utf_lst[j].c].cletter = 1;
      utf_tbl[utf_lst[j].c].clower = utf_lst[j].clower;
      utf_tbl[utf_lst[j].c].cupper = utf_lst[j].cupper;
    }
  } else return 1;
  return 0;
}
#endif
#endif

void free_utf_tbl() {
  if (utf_tbl_count > 0) utf_tbl_count--;
  if (utf_tbl && (utf_tbl_count == 0)) {
    op_free(utf_tbl);
    utf_tbl = NULL;
  }
}
#endif //HUNSPELL_USE_UTF_TBL


unsigned short unicodetoupper(unsigned short c, int langnum)
{
  // In Azeri and Turkish, I and i dictinct letters:
  // There are a dotless lower case i pair of upper `I',
  // and an upper I with dot pair of lower `i'. 
  if (c == 0x0069 && ((langnum == LANG_az) || (langnum == LANG_tr)))
    return 0x0130;
#ifdef OPENOFFICEORG
  return u_toupper(c);
#elif defined(MOZILLA_CLIENT)
  return ToUpperCase((PRUnichar) c);
#elif defined(HUNSPELL_USE_UTF_TBL)
  return (utf_tbl) ? utf_tbl[c].cupper : c;
#else
  return hunspell_u_toupper(c);
#endif //HUNSPELL_USE_UTF_TBL
}

unsigned short unicodetolower(unsigned short c, int langnum)
{
  // In Azeri and Turkish, I and i dictinct letters:
  // There are a dotless lower case i pair of upper `I',
  // and an upper I with dot pair of lower `i'. 
  if (c == 0x0049 && ((langnum == LANG_az) || (langnum == LANG_tr)))
    return 0x0131;
#ifdef OPENOFFICEORG
  return u_tolower(c);
#elif defined(MOZILLA_CLIENT)
  return ToUpperCase((PRUnichar) c);
#elif defined(HUNSPELL_USE_UTF_TBL)
  return (utf_tbl) ? utf_tbl[c].clower : c;
#else
  return hunspell_u_tolower(c);
#endif
}

int unicodeisalpha(unsigned short c)
{
#ifdef OPENOFFICEORG
  return u_isalpha(c);
#elif defined(HUNSPELL_USE_UTF_TBL)
  return (utf_tbl) ? utf_tbl[c].cletter : 0;
#else
  return hunspell_u_isalpha(c);
#endif
}

/* get type of capitalization */
int get_captype(char * word, int nl, cs_info * csconv) {
   // now determine the capitalization type of the first nl letters
   int ncap = 0;
   int nneutral = 0;
   int firstcap = 0;
   if (csconv == NULL) return NOCAP;
   for (char * q = word; *q != '\0'; q++) {
      if (csconv[*((unsigned char *)q)].ccase) ncap++;
      if (csconv[*((unsigned char *)q)].cupper == csconv[*((unsigned char *)q)].clower) nneutral++;
   }
   if (ncap) {
     firstcap = csconv[*((unsigned char *) word)].ccase;
   }

   // now finally set the captype
   if (ncap == 0) {
        return NOCAP;
   } else if ((ncap == 1) && firstcap) {
        return INITCAP;
   } else if ((ncap == nl) || ((ncap + nneutral) == nl)) {
        return ALLCAP;
   } else if ((ncap > 1) && firstcap) {
        return HUHINITCAP;
   }
   return HUHCAP;
}

int get_captype_utf8(w_char * word, int nl, int langnum) {
   // now determine the capitalization type of the first nl letters
   int ncap = 0;
   int nneutral = 0;
   int firstcap = 0;
   unsigned short idx;
   // don't check too long words
   if (nl >= MAXWORDLEN) return 0;
   // big Unicode character (non BMP area)
   if (nl == -1) return NOCAP;
   for (int i = 0; i < nl; i++) {
     idx = (word[i].h << 8) + word[i].l;
     if (idx != unicodetolower(idx, langnum)) ncap++;
     if (unicodetoupper(idx, langnum) == unicodetolower(idx, langnum)) nneutral++;
   }
   if (ncap) {
      idx = (word[0].h << 8) + word[0].l;
      firstcap = (idx != unicodetolower(idx, langnum));
  }

   // now finally set the captype
   if (ncap == 0) {
        return NOCAP;
   } else if ((ncap == 1) && firstcap) {
        return INITCAP;
   } else if ((ncap == nl) || ((ncap + nneutral) == nl)) {
        return ALLCAP;
   } else if ((ncap > 1) && firstcap) {
        return HUHINITCAP;
   }
   return HUHCAP;
}


// strip all ignored characters in the string
void remove_ignored_chars_utf(char * word, unsigned short ignored_chars[], int ignored_len)
{
   w_char w[MAXWORDLEN]; // ARRAY OK 2009-02-09 jb
   w_char w2[MAXWORDLEN]; // ARRAY OK 2009-02-09 jb
   int i;
   int j;
   int len = u8_u16(w, MAXWORDLEN, word);
   for (i = 0, j = 0; i < len; i++) {
      if (!flag_bsearch(ignored_chars, ((unsigned short *) w)[i], ignored_len)) {
         w2[j] = w[i];
         j++;
      }
   }
   if (j < i) u16_u8(word, op_strlen(word) + 1, w2, j);
}

// strip all ignored characters in the string
void remove_ignored_chars(char * word, char * ignored_chars)
{
   for (char * p = word; *p != '\0'; p++) {
      if (!op_strchr(ignored_chars, *p)) {
        *word = *p;
        word++;
      }
   }
   *word = '\0';
}

int parse_string(char * line, char ** out, int ln)
{
   char * tp = line;
   char * piece;
   int i = 0;
   int np = 0;
   if (*out) {
      HUNSPELL_WARNING(hunspell_stderr, "error: line %d: multiple definitions\n", ln);
      return 1;
   }
   piece = mystrsep(&tp, 0);
   while (piece) {
      if (*piece != '\0') {
          switch(i) {
              case 0: { np++; break; }
              case 1: { 
                *out = mystrdup(piece);
                if (!*out) return 1;
                np++;
                break;
              }
              default: break;
          }
          i++;
      }
      // op_free(piece);
      piece = mystrsep(&tp, 0);
   }
   if (np != 2) {
      HUNSPELL_WARNING(hunspell_stderr, "error: line %d: missing data\n", ln);
      return 1;
   } 
   return 0;
}

int parse_array(char * line, char ** out, unsigned short ** out_utf16,
       int * out_utf16_len, int utf8, int ln) {
   if (parse_string(line, out, ln)) return 1;
   if (utf8) {
        w_char w[MAXWORDLEN]; // ARRAY OK 2009-02-09 jb
        int n = u8_u16(w, MAXWORDLEN, *out);
        if (n > 0) {
            flag_qsort((unsigned short *) w, 0, n);
            *out_utf16 = (unsigned short *) op_malloc(n * sizeof(unsigned short));
            if (!*out_utf16) return 1;
            op_memcpy(*out_utf16, w, n * sizeof(unsigned short));
        }
        *out_utf16_len = n;
   }
   return 0;
}
#endif //SPC_USE_HUNSPELL_ENGINE
