# -*- Mode: python; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*-

import datetime
import re
import sys
import os
import glob
import xml.sax

#
# global variables
#

verbosity = 0
table_files = []
module_tables_file_name = "string_tables.h"
module_globals_file_name = "string_globals.h"

#
# definition of functions and classes used
#

def verbose(s, level = 0):
	global verbosity
	if verbosity >= level:
		print s

# String table entry
class StringEntry:
	def __init__(self):
		self.string = ""
		self.cond = ""
		self.comment = ""

# String table
class StringTable:
	def __init__(self):
		self.size = 0
		self.name = ""
		self.global_name = ""
		self.data_type = ""
		self.inc_name = ""
		self.cond = ""
		self.has_conds = 0
		self.entries = []
	def dump(self, dump_strings):
		dump_str = "\n%s, %s\n" % (self.name, self.data_type)
		dump_str += "------------------------------------------\n"
		dump_str += "inc_file: %s\n" % self.inc_name
		dump_str += "size: %s\n" % self.size
		if dump_strings:
			for s in self.entries:
				dump_str += "  %s\n" % s.string
		dump_str += "\n"
		verbose(dump_str, 2)
	def dumpToFile(self, inc_file):
		inc_file.write("CONST_ARRAY(%s, %s)\n" % (self.name, self.data_type))
		for s in self.entries:
			if s.cond != "":
				inc_file.write("#if %s\n" % s.cond)
			inc_file.write("\tCONST_ENTRY(UNI_L(\"%s\"))," % s.string)
			if s.comment != "":
				inc_file.write(" // %s" % s.comment)
			inc_file.write("\n")
			if s.cond != "":
				inc_file.write("#endif // %s\n" % s.cond)
		inc_file.write("CONST_END(%s)\n\n" % self.name)
		if self.has_conds:
			inc_file.write("enum %s_COUNTER\n{\n" % self.name)
			for s in self.entries:
				if s.cond != "":
					inc_file.write("#if %s\n" % s.cond)
				inc_file.write("\tDUMMY_ENTRY_%s_%s,\n" % (self.name, s.string))
				if s.cond != "":
					inc_file.write("#endif // %s\n" % s.cond)
			inc_file.write("\t%s_COUNTER_LAST\n" % self.name)
			inc_file.write("}\n")
	def addEntry(self, str_entry):
		new_entry = StringEntry()
		new_entry.string = str_entry.string
		if str_entry.cond != "":
			new_entry.cond = str_entry.cond
			self.has_conds = 1
		self.entries.append(new_entry)
		self.size += 1

class StringTableFile:
	def __init__(self):
		self.name = ""
		self.cond = ""
		self.tables = []
		self.has_globals = 0
	def dump(self, dump_strings):
		for t in self.tables:
			t.dump(dump_strings)
	def dumpToFile(self):
		if self.name != "" and len(self.tables) > 0:
			inc_file = open(self.name, "w")
			inc_file.write("""\
/*******************************************************************
 * This file is generated by the script %s
 * found in logdoc/scripts
 *
 *     !!! DO NOT EDIT THIS FILE BY HAND !!!
 *
 * All changes will be lost when the files are generated again.
 * Generated: %s
 ******************************************************************/\n\n"""
% (sys.argv[0], datetime.date.today().isoformat()))
			if self.cond != "":
				inc_file.write("#if %s\n\n" % self.cond)
			for t in self.tables:
				t.dumpToFile(inc_file)
			if self.cond != "":
				inc_file.write("#endif // %s\n" % self.cond)
			verbose("Wrote file %s" % self.name, 0)
	def addTable(self, table):
		self.tables.append(table)
		
# XML handlers
class StrTblHandlers(xml.sax.ContentHandler):
	NO_STATE = 0
	IN_FILE = 1
	IN_TABLE = 2
	IN_STRING = 4
	def __init__(self, strfile):
		self.strfile = strfile
		self.state = self.NO_STATE
		self.current_string = None
		self.current_table = None
	def endDocument(self):
		self.strfile.dump(True)
		self.strfile.dumpToFile()
	def startElement(self, name, attrs):
		verbose("    Start of element " + name, 3)
		if name == "file":
			if self.state == self.NO_STATE:
				self.state = self.IN_FILE
				if "name" in attrs:
					self.strfile.name = attrs["name"]
				if "cond" in attrs:
					self.strfile.cond = attrs["cond"]
			else:
				verbose("  Weird file start", 2)
		elif name == "tbl":
			if self.state == self.IN_FILE:
				self.state = self.IN_TABLE
				self.current_table = StringTable()
				if "name" in attrs:
					self.current_table.name = attrs["name"]
				if "type" in attrs:
					self.current_table.data_type = attrs["type"]
				if "cond" in attrs:
					self.current_table.cond = attrs["cond"]
				if "globalname" in attrs:
					self.strfile.has_globals = 1
					self.current_table.global_name = attrs["globalname"]
			else:
				verbose("  Weird table start", 2)
		elif name == "str":
			if self.state == self.IN_TABLE:
				self.state = self.IN_STRING
				self.current_string = StringEntry()
				if "cond" in attrs:
					self.current_string.cond = attrs["cond"]
				if "comment" in attrs:
					self.current_string.comment = attrs["comment"]
			else:
				verbose("  Weird string start", 2)
		else:
			verbose(" Weird start tag found", 2)
	def endElement(self, name):
		verbose("    End of element " + name, 3)
		if self.state == self.IN_FILE and name == "file":
			self.state = self.NO_STATE
		if self.state == self.IN_TABLE and name == "tbl":
			if self.current_table == None:
				verbose("  Weird table end", 2)
			self.state = self.IN_FILE
			self.strfile.addTable(self.current_table)
			self.current_table = None
		elif self.state == self.IN_STRING and name == "str":
			if self.current_string == None:
				verbose("  Weird string end", 2)
			self.state = self.IN_TABLE
			self.current_table.addEntry(self.current_string)
			self.current_string = None
		else:
			verbose("  Weird endtag: " + name, 2)
	def characters(self, content):
		verbose("     content{%s}" % content, 4)
		if self.state == self.IN_STRING:
			self.current_string.string = content

def parseFile(filename):
	verbose("  Parsing " + filename, 1)
	xml_file = open(filename, "r")
	table_file = StringTableFile()
	table_files.append(table_file)
	content_handler = StrTblHandlers(table_file)
	xml.sax.parse(xml_file, content_handler)

def scanForFiles(arg, dirname, names):
	verbose("Scanning directory: " + dirname, 1)
	fil_reg = re.compile("\.string_tbl")
	
	str_tbl_files = glob.glob(dirname + "/*.string_tbl")
	for filename in str_tbl_files:
		verbose(" Found " + filename, 1)
		parseFile(filename)


# make the root of logdoc the current working dir
if not os.path.isdir("logdoc"):
    while not os.path.isdir("logdoc"):
        prevdir = os.getcwd()
        os.chdir("..")
        if prevdir == os.getcwd():
            break

if not os.path.isdir("logdoc"):
	verbose("error: You must be in the logdoc module to run this script", 0)
	exit

os.chdir("logdoc");

verbose("Scanning for string table files...", 0)

# find the string table files
if len(sys.argv) > 1:
	parseFile(sys.argv[1])
else:
	os.path.walk(".", scanForFiles, None)

# write the list of string tables to be used in logdoc_module.h
verbose("Found %d table files" % len(table_files), 1)
tables_file = open(module_tables_file_name, "w")
globals_file = open(module_globals_file_name, "w")
for tbl_file in table_files:
	if tbl_file.cond != "":
		tables_file.write("#if %s\n" % tbl_file.cond)
		if tbl_file.has_globals:
			globals_file.write("#if %s\n" % tbl_file.cond)
	for str_tbl in tbl_file.tables:
		if str_tbl.cond != "":
			tables_file.write("#if %s\n" % str_tbl.cond)
			if str_tbl.global_name:
				globals_file.write("#if %s\n" % str_tbl.cond)
		tables_file.write("\tconst %s\t\tm_%s[%d];\n" % (str_tbl.data_type, str_tbl.name, str_tbl.size))
		if str_tbl.global_name:
			globals_file.write("# define %s	(g_opera->logdoc_module.%s)\n" % (str_tbl.global_name, str_tbl.name))
		if str_tbl.cond != "":
			tables_file.write("#endif // %s\n" % str_tbl.cond);
			if str_tbl.global_name:
				globals_file.write("#endif // %s\n" % str_tbl.cond);
	if tbl_file.cond != "":
		tables_file.write("#endif // %s\n" % tbl_file.cond);
		if tbl_file.has_globals:
			globals_file.write("#endif // %s\n" % tbl_file.cond);
tables_file.close()
globals_file.close()
verbose("\nWrote file %s" % module_tables_file_name, 0)
verbose("Wrote file %s" % module_globals_file_name, 0)
	
verbose("Finished!", 0)
