/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*-
 *
 * Copyright (C) 2002-2011 Opera Software AS.  All rights reserved.
 *
 * This file is part of the Opera web browser.
 * It may not be distributed under any circumstances.
 */
#ifndef _DOCUMENTORIGIN_H_
#define _DOCUMENTORIGIN_H_

#include "modules/url/url2.h"
#include "modules/util/opstring.h"

class DocumentReferrer;

enum DomainOverrides
{
	DOMAIN_NOT_MODIFIED,
	DOMAIN_OVERRIDDEN
};

/**
 * This class represents the security context of a document and is
 * closely related to the Origin concept in the HTML5 specification.
 * The information kept here includes:
 *
 * * An URL - The main security decider.
 *
 * * A flag "generated by opera" - Saying if this document is the Internet
 *   resource from |url| or something we generated as a placeholder/replacement
 *   for that resource. If this is TRUE we're typically displaying some kind
 *   of error page rather than the real resource and that means that it is in
 *   a different security context than otherwise.
 *
 * * An "effective domain" - Used in scripting instead of |url| to decide
 *   access rights. The effective domain of a document from "www.opera.com"
 *   might for instance by a script be set to "opera.com" which
 *   would allow it some access to a document from "bugs.opera.com" that has
 *   also set its effective domain (by setting document.domain in a script) to
 *   "opera.com".
 *
 * The object is reference counted and shared between some owners. The main
 * and initial owner is FramesDocument, but DOM_Runtimes (script environments)
 * also own a reference each (there can be more than one script environment
 * if there are extensions involved).
 *
 * It's important to make a Copy of this object when creating a new document
 * or mutations of the initial document's Origin (i.e. setting effective domain)
 * will affect other documents as well.
 */
class DocumentOrigin
{
	friend class DocumentReferrer;
public:
	/**
	 * Main/initial way to create a DocumentOrigin object using the url
	 * as security context, assuming "not generated by opera" and
	 * having the same effective domain as the url's host name.
	 *
	 * Reference counted object. Starts with reference count 1.
	 *
	 * @param[in] url The url forming the basis of the security. This should
	 * typically not be a url without hostname unless that is all information
	 * there is. That can happen when loading a document at the top level
	 * with an about: or data: or javascript: url.
	 *
	 * @returns The DocumentOrigin or NULL if there was an out of
	 * memory situation.
	 *
	 * @see MakeUniqueOrigin()
	 */
	static DocumentOrigin* Make(URL url);

	/**
	 * Creates a DocumentOrigin object that is guarenteed to not be
	 * compatible with any other DocumentOrigin ever created with this
	 * method or otherwise (except clones).
	 *
	 * Reference counted object. Starts with reference count 1.
	 *
	 * @param[in] context The context for this unique DocumentOrigin.
	 * @param[in] in_sandbox The document is in a HTML 5 sandbox.
	 *
	 * @returns The DocumentOrigin or NULL if there was an out of
	 * memory situation.
	 *
	 * @see Make()
	 */
	static DocumentOrigin* MakeUniqueOrigin(URL_CONTEXT_ID context, BOOL in_sandbox);

	/**
	 * Increases the reference counter by 1.
	 */
	void IncRef() { IncRefInternal(); }

	/**
	 * Decreases the reference counter by 1. If the reference counter
	 * reaches 0 then the object is deleted by this method so the
	 * pointer to this object should be set to NULL or go out of scope
	 * as soon as this has been called (normal reference counted
	 * pointer behaviour).
	 */
	void DecRef() { DecRefInternal(); }

	/**
	 * Returns the effective domain. Always non-null but may be the empty string
	 * in case a document has no domain.
	 */
	const uni_char* GetEffectiveDomain() const { return effective_domain.CStr(); }

	/**
	 * Is this what HTML calls a "unique origin", i.e. an origin that
	 * isn't like any other and can be identified as such?
	 *
	 * @returns TRUE if this is a "unique origin".
	 */
	BOOL IsUniqueOrigin() const { return !!security_context.GetAttribute(URL::KIsUniqueOrigin); }

	/**
	 * Called to change the effective domain as a result of modifying
	 * document.domain from script. No checking of reasonability of the new
	 * domain is done. has_overridden_domain will be DOMAIN_OVERRIDDEN after the call.
	 *
	 * @param[in] new_domain The new domain, a non-null string.
	 *
	 * @returns OpStatus::OK if the domain was changed or
	 * OpStatus::ERR_NO_MEMORY if the operation couldn't be performed due to
	 * lack of memory.
	 */
	OP_STATUS OverrideEffectiveDomain(const uni_char* new_domain) { OP_ASSERT(new_domain); RETURN_IF_ERROR(effective_domain.Set(new_domain)); has_overridden_domain = DOMAIN_OVERRIDDEN; return OpStatus::OK;}

	/** The main security context. */
	URL security_context;
	/** If this document shows an "opera generated" page rather than
		the Internet resource. */
	BOOL is_generated_by_opera;
	/** If the effective domain has been set by scripts. */
	DomainOverrides has_overridden_domain;
	/** The document is in an HTML 5 sandbox */
	BOOL in_sandbox;
private:
	void IncRefInternal() const
	{
		const_cast<DocumentOrigin*>(this)->ref_count++;
	}
	void DecRefInternal() const
	{
		DocumentOrigin* _this = const_cast<DocumentOrigin*>(this);
		if (!--_this->ref_count)
			OP_DELETE(_this);
	}
	OpString effective_domain;
	DocumentOrigin(URL url) : security_context(url), is_generated_by_opera(FALSE), has_overridden_domain(DOMAIN_NOT_MODIFIED), in_sandbox(FALSE), ref_count(1) {}
 	unsigned ref_count;
	/** Private so that users don't accidentally use it instead of DecRef(). */
	~DocumentOrigin() {}
	/** Private and not implemented to avoid accidental copying. */
	DocumentOrigin(DocumentOrigin& other);
	/** Private and not implemented to avoid accidental copying. */
	DocumentOrigin& operator=(DocumentOrigin& other);
};

#endif /* _DOCUMENTORIGIN_H_ */
