/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*-
 *
 * Copyright (C) 1995-2008 Opera Software AS.  All rights reserved.
 *
 * This file is part of the Opera web browser.
 * It may not be distributed under any circumstances.
 */

#ifndef OP_SPELL_CHECKER_H
#define OP_SPELL_CHECKER_H

// Notice: when it's written scUTF-32 somewhere in the comments, this means characters in the dictionary-encoding
// converted to 32-bit values. If the dictionary is in UTF-8 this "format" happens to be UTF32, otherwise it's just
// 8-bit encoded characters converted to 32-bit.

#include "modules/util/simset.h"
#include "modules/util/opfile/opfile.h"
#include "modules/pi/OpSystemInfo.h"
#include "modules/encodings/encoders/outputconverter.h"
#include "modules/encodings/decoders/inputconverter.h"
#include "modules/unicode/unicode.h"

class OpSpellChecker;
class OpSpellCheckerAffix;
class OpSpellCheckerRadix;
class OpSpellCheckerLanguage;
class OpSpellCheckerWordIterator;
class OpSpellCheckerListener;
class OpSpellCheckerAllocator;
class HunspellAffixFileParseState;

//#define SPC_SORT_PARTIALLY_FOR_COMPRESSION	// In modules.tweaks
//#define SPC_SETUP_CHAR_COMBINATIONS // Enable for setting up the bit-field of existing index-combinations
#define SPC_MAX_COMB_BITS_BYTES 2000000 // Max size of the bit-field of existing index-combinations
//#define SPC_MAX_HASH_LOOKUP_BYTES 1000000 // In modules.tweaks

//#ifdef _DEBUG
//#define SPC_DEBUG
//#endif

#ifdef SPC_DEBUG
#define SPC_ERROR(x) sc_error(x)
#else
#define SPC_ERROR(x) x
#endif

#ifdef SPC_DEBUG
#define sc_printf(format,...)
#endif // SPC_DEBUG

#ifdef SIXTY_FOUR_BIT
#define SPC_UINT_PTR UINT64
#else // SIXTY_FOUR_BIT
#define SPC_UINT_PTR UINT32
#endif // !SIXTY_FOUR_BIT

/** Size in bytes of the temp-buffer obtained by OpSpellChecker::TempBuffer(). */
#define SPC_TEMP_BUFFER_BYTES 4096

/** Maximum length of words (nr of characters exluding '\0') that we are able to handle in dictionary, to spellcheck, etc. */
#define SPC_MAX_DIC_FILE_WORD_LEN 200

#ifndef SPC_USE_HUNSPELL_ENGINE
/** Max allowed nr of flags per word in .dic file or affix-rule in .aff file */
#define SPC_MAX_FLAGS_PER_LINE 100

/** Max allowed nr of includes "accumulated" compound rule flags (from affixes-rules) */
#define SPC_MAX_COMPOUND_RULE_FLAGS_PER_LINE 30

/** When the this number of words has been found under a certain radix-entry, a new radix will be allocated for that entry */
#define SPC_RADIX_ALLOCATION_COUNT 128

/** When more words will be added to a "word-unit" but we don't have the words available yet, approximate that they will occupy this many bytes. */
#define SPC_ALLOCATE_BYTES_PER_UNKNOWN_WORD 3

/** How many bits that will specify how many indexes from the previous word in a chain of word-units 
	that should be copied to the current word, that is, of the indexes after the first ones which are 
	implicitly specified by the position in the radix-tree. */
#define SPC_COPY_PREV_WORD_BITS 3

/** The capacity (in nr of words) of the TempDictionaryParseData class used during the dictionary "parsing passes" 1, 2 and 3. */
#define SPC_MAX_TEMP_WORDS 2000

/** The data in the dic. file will be read, allocated, and deallocated in chunks of this many bytes. */
#define SPC_DICTIONARY_CHUNK_SIZE (128*1024)

/** The characters which are not present in any affix and whos occurence in the .dic file is < "total number of characters" / SPC_MIN_USUAL_DIVIDE
	will be be mapped to the last index for "unusual characters". For an exception, see SPC_MIN_USUAL_FACTOR below */
#define SPC_MIN_USUAL_DIVIDE 2000

/** (This refers to the above comment for SPC_MIN_USUAL_DIVIDE) However, the frequency of the characters mapped into the "unusual mapping"
	are not allowed to be >= 1/SPC_MIN_USUAL_FACTOR */
#define SPC_MIN_USUAL_FACTOR 100

/** For beeing able to add new words to the dictionary during runtime which includes characters previously not 
	represented in the dictionary is the capacity of the mapping for "unusual characters" extended with this
	many "slots" for handling those new characters. For simplicity and performance is this taken into account during 
	the loading of the dictionary and not extended when needed, because the bit-number in the word-units specifying
	which particular character in a mapping an index refers to is dependent of this number. Otherwise the tree would more
	or less have to be "reconstructed" when adding such words.
	However YOU DON'T NEED TO CARE ABOUT THIS STUFF! Because there is currently no support for adding words to the 
	dictionary-tree dynamically. This is handled in a simplified way by using the OpSpellChecker::m_added_words collection. */
#define SPC_MIN_CAPACITY_FOR_NEW_CHARS 50

/** The maximum number of replacement-suggestions generated by the replacement-algorithm. */
#define SPC_MAX_REPLACEMENTS 5

/** The index-combination bit-field will at most include valid combinations of this many indexes, not important
	as long as SPC_SETUP_CHAR_COMBINATIONS is disabled.*/
#define SPC_MAX_COMBINATION_LEN 10

/** The maximum number of words + 1 that is allowed to be put in a chain of word-units accessibly by a single radix-entry. */
#define SPC_MAX_RADIX_COUNTER 4095

/** A "marker" created before the entry-array of an OpSpellCheckerRadix object to mark the beginning of the entry-array.
	Before this marker (which has sizeof(void*)) will a OpSpellCheckerRadix* pointer to the radix's parent be located.
	This is used for implementing RadixEntry::GetParent() and RadixEntry::GetIndex(). */
#define SPC_RADIX_START_MARKER 4095

/** The maximum allowed number of conditions for a compound-rule, conditions marked with '+' is translated to two
	conditions by converting like this: x+ -> xx*. For example, this counts as three conditions: "[abcde]+[fghijk]*" */
#define SPC_COMPUNDRULE_MAX_CONDITIONS 254

/** Maximum number of compounds in a word that are allowed, if COMPOUNDWORDMAX in the dictionary is > this value, we
	will anyway only use this as the maximum number. This number defines the maximum recursion-depth that could occur
	in the recursive "compound-matching" functions in dictionarylookup.cpp (like OpSpellChecker::HasCorrectChild). */
#define SPC_MAX_ALLOWED_COMPOUNDWORDMAX 10

/** Maxium number of characters that are allowed to be mapped together with a MAP statement in the .aff files. */
#define SPC_MAX_PER_MAPPING 1000

/** When matching words with affixes in order to get the matching affix-rules in OpSpellChecker::DictionaryParsePass1,
	for affixes with nr of matching rules <= SPC_MAX_RULE_PER_AFFIX_CACHE - will the matching rules be cached to that
	the matching won't be necessary for the following two passes. */
#define SPC_MAX_RULE_PER_AFFIX_CACHE 3

/** Bytes to allocate (at most) per word in .dic file for the affix-rule cache mentioned in the comment just above. */
#define SPC_MAX_FLAG_BUF_SIZE ((SPC_MAX_FLAGS_PER_LINE*(SPC_MAX_RULE_PER_AFFIX_CACHE+1) + 1) * sizeof(UINT16))

/** Maximum allowed affix-rules to match for a single word+affix combination. */
#define SPC_MAX_MATCHING_RULES_PER_WORD 8000

/** Maximum number of conditions for an affix-rule, example: [abcdee]^[sfwerw] is two conditions. */
#define SPC_MAX_AFFIX_RULE_CONDITION_LENGTH 100

/** Maximum number of allowed rules for a single affix. */
#define SPC_MAX_RULES_PER_AFFIX 20000

/** Number of UINT32 characters to allocate for the "temporary words" in the TempDictionaryParseData class. */
#define SPC_TEMP_WORD_CHAR_BUF_SZ (SPC_MAX_TEMP_WORDS*30)

/** Number of IndexAndMapping structs to allocate for the "temporary words" in the TempDictionaryParseData class.*/
#define SPC_TEMP_INDEX_AND_MAPPING_BUF_SZ (SPC_MAX_TEMP_WORDS*15)

/** Number of UINT16 compound flag indexes to allocate for the "temporary words" in the TempDictionaryParseData class.*/
#define SPC_TEMP_COMPOUND_RULE_FLAG_BUF_SZ (SPC_MAX_TEMP_WORDS*5)

/** How many unicode codepoints we should be able to handle. */
#define SPC_UNICODE_REAL_CODEPOINTS 0x110000

/** Maximum allowed words (including inflections) that we can have in the dictionary. */
#define SPC_MAX_TOTAL_WORD_COUNT 100000000

/** Maximum allowed number of indexes we can handle. */
#define SPC_MAX_INDEXES 40000

/** Any huffman-code are allowed to have at most this many bits. */
#define SPC_HUFFMAN_MAX_LEVELS 16

/** Huffman-codes for encoding the "bit-flags" (like AFFIX_TYPE_ONLYINCOMPOUND) stored in the "word-units" in the radix-tree
	are allowed to have at most this many bits. MUST be <= SPC_HUFFMAN_MAX_LEVELS! */
#define SPC_FLAGS_HUFFMAN_LEVELS MIN((int)(SPC_RADIX_FLAGS) + 4, SPC_HUFFMAN_MAX_LEVELS)

/** The maximum allowed number of words to have in single "word-unit", when more words then this should be under the same 
	"leaf" in an OpSpellCheckerRadix - a new "word-unit" must be allocated and linked to the previous one. */
#define SPC_MAX_WORDS_PER_COMPRESSED_UNIT 100

/** When allocating a "word-unit", allocate this many bytes, but later we'll probably "shrink" the allocation by using
	OpSpellCheckerAllocator::OnlyUsed to what we (at least think) is necessary. */
#define SPC_COMPRESSED_WORDS_ALLOCATE 8192

/** Number of indexes to cache in RadixEntryCache. */
#define SPC_RADIX_CACHE_INDEXES 20

/** Take out the first character of string str and put into ch, and increase str. */
#define SPC_NEXT_CHAR(ch,str) do { if (m_8bit_enc) ch = *((str)++); else {int bytes_used = 0; ch = utf8_decode(str,bytes_used); str += bytes_used;}} while (0)

/** Returns strlen of str. */
#define SPC_STRLEN(str) (m_8bit_enc ? op_strlen((const char*)str) : utf8_strlen(str))

/** Used for checking if the "next word-unit" pointer in a "word-unit" is a pointer to the text word-unit, or if this
	is the currently last word-unit in the word-unit chain and the "pointer" is instead an offset into the word-unit
	where info is stored about how to continue to append words to the unit. */
#define SPC_IS_NEXT_UNIT_POINTER(x) !!((SPC_UINT_PTR)(x) & ~4095)

/** Does a DictionaryEncoding enum represents an 8-bit encoding, a bit stupid as the DictionaryEncoding enum type 
	is sort of unnecessary now-a-days, see below. */
#define IS_8BIT_ENCODING(x) ((x) != UTF8_ENC)

/** Enumeration for different encodings, this has been reduced to a BOOL in practice (8-bit encoding or utf-8 encoding)
	as the encoding module handle, now, handles the different 8-bit encoding. FIXME: Remove this (if you like),
	as OpSpellChecker::m_8bit_enc tells the same thing. */
enum DictionaryEncoding
{
	BIT8_ENC = 0,	///< Everything we support except of utf-8
	UTF8_ENC,		///< UTF-8
	UNKNOWN_ENC		///< FIXME: doesn't have any real purpose anymore.
};

/** Different encodings of flags as defined by the FLAGS keyword in the .aff file. */
enum FlagEncoding
{
	FLAG_ASCII_ENC = 0, ///< The default encoding used if no FLAGS keyword is present, a flag is an ascii-char between 1-255.
	FLAG_LONG_ENC,		///< A flag is TWO ascii-chars
	FLAG_NUM_ENC,		///< Flags are ',' separated (ascii-coded) numbers between 0-65535. FIXME: gl_ES has numbers over 65535, which makes that dictionary fail!
	FLAG_UTF8_ENC,		///< A flag is one utf-8 character
	FLAG_UNKNOWN_ENC	///< Unknown flag-encoding, will result in aborted loading of dictionary.
};

/** Types of flags that we support, the type of the flag is defined in the .aff file. */
enum AffixFlagType // WordReplacement::bit_flags MUST have >= SPC_RADIX_FLAGS bits!
{
	AFFIX_TYPE_COMPOUNDBEGIN = 0,	///< The COMPOUNDBEGIN keyword
	AFFIX_TYPE_COMPOUNDMIDDLE,		///< The COMPOUNDMIDDLE keyword
	AFFIX_TYPE_COMPOUNDEND,			///< The COMPOUNDEND keyword
	AFFIX_TYPE_COMPOUNDFLAG,		///< The COMPOUNDFLAG keyword
	AFFIX_TYPE_ONLYINCOMPOUND,		///< The ONLYINCOMPOUND keyword
	AFFIX_TYPE_KEEPCASE,			///< The KEEPCASE keyword
	AFFIX_TYPE_NOSUGGEST,			///< The NOSUGGEST keyword
	AFFIX_TYPE_NEEDAFFIX,			///< The NEEDAFFIX keyword
	AFFIX_TYPE_COMPOUNDPERMITFLAG,	///< The COMPOUNDPERMITFLAG keyword
	AFFIX_TYPE_COMPOUNDFORBIDFLAG,	///< The COMPOUNDFORBIDFLAG keyword
	AFFIX_TYPE_CIRCUMFIX,			///< The CIRCUMFIX keyword
	AFFIX_TYPE_COMPOUNDRULEFLAG,	///< Represents a compound rule, that is, flag that occurs in one or more COMPOUNDRULE statements in the .aff file
	AFFIX_TYPE_AFFIX,				///< Represent an affix, that is, an PFX or SFX statement together with it's affix-rules in the .aff file
	AFFIX_TYPE_COUNT,				///< Number of types...

	SPC_RADIX_FLAGS = AFFIX_TYPE_COMPOUNDPERMITFLAG,  ///< the first affix type not stored as "ON/OFF" bits for each word in the "word-units" for the "radix-tree"
	SPC_RADIX_COMPOUND_FLAGS = ((1<<AFFIX_TYPE_COMPOUNDBEGIN) | (1<<AFFIX_TYPE_COMPOUNDMIDDLE) | (1<<AFFIX_TYPE_COMPOUNDEND) | (1<<AFFIX_TYPE_COMPOUNDFLAG)),
	AFFIX_BIT_TYPE_COUNT = AFFIX_TYPE_AFFIX, ///< the first AFFIX_BIT_TYPE_COUNT types abong the types above are stored as bits "ON/OFF" in TempWordInfo::info::bit_flags
	AFFIX_BIT_HAS_AFFIXES = AFFIX_BIT_TYPE_COUNT, ///< Used by OpSpellChecker::GetFlagPtr to signal presence of affixes
	AFFIX_BIT_HAS_COMPOUNDRULEFLAG = AFFIX_BIT_HAS_AFFIXES+1, ///< Used by OpSpellChecker::GetFlagPtr to signal presence of compoundrule flags
	AFFIX_BIT_BITS = AFFIX_BIT_HAS_COMPOUNDRULEFLAG + 1 ///< must be <= 16!!!
};

#endif // !SPC_USE_HUNSPELL_ENGINE


/** A simple UTF16->UTF32 converter which isn't writing any Byte-Order-Mark or such things.. */
class SCUTF16ToUTF32Converter : public CharConverter
{
public:
	virtual int Convert(const void *src, int len, void *dest, int maxlen, int *read);
	virtual const char *GetCharacterSet() { return NULL; };
	virtual const char *GetDestinationCharacterSet() { return NULL; };
};

/** A simple UTF32->UTF16 converter */
class SCUTF32ToUTF16Converter : public CharConverter
{
	virtual int Convert(const void *src, int len, void *dest, int maxlen, int *read);
	virtual const char *GetCharacterSet() { return NULL; };
	virtual const char *GetDestinationCharacterSet() { return NULL; };
};

/** A class which allocates fixed size chunks of memory. This is used by OpSpellCheckerAllocator and UTFMapping to allocate memory.
	The purpose of this class is somewhat unclear and it might be a nice idea to integrate this into OpSpellCheckerAllocator instead... */
class OpSpellCheckerFixedAllocator
{
public:
	
	/** Create an allocator that uses malloc/calloc to allocate "objects" of size unit_bytes.
		@param unit_bytes The size of each "object".
		@param units_per_alloc How many "objects" to allocate for each malloc/calloc
		@param mem_zero If TRUE will calloc be used.
	*/
	OpSpellCheckerFixedAllocator(int unit_bytes, int units_per_alloc, BOOL mem_zero);
	~OpSpellCheckerFixedAllocator() { Clean(); }
	
	/** Frees all allocated memory */
	void Clean();
	
	/** Frees all allocated memory except of the first malloc/calloc allocation. */
	void Reset();
	
	/** Allocates one "object" */
	void *Allocate();
	
	/** Allocates one "object" 
		@param malloc_index The index of the malloc allocated chunk where the allocation occured in.
		@parm byte_offset The byte offset into that chunk for the allocation.
	*/
	void *Allocate(int &malloc_index, int &byte_offset);
	
	/** Returns the address of the malloc allocated chunk with index malloc_index. */
	void *GetMallocAddr(int malloc_index) { OP_ASSERT(malloc_index <= m_current_global_index); return (void*)m_malloced[malloc_index]; }
	
	/** Returns the number of bytes allocated using malloc/calloc */
	UINT32 GetAllocatedBytes() { return (m_current_global_index + 1) * m_unit_bytes * m_units_per_alloc; }

#ifdef SPC_DEBUG
	void VerifyMemOk();
#endif // SPC_DEBUG

private:
	void Init(int unit_bytes, int units_per_alloc, BOOL mem_zero);
	BOOL MallocMore();

private:
	UINT8 **m_malloced;	///< Array of chunks of data where each chunk is allocated using malloc/calloc
	int m_malloced_count; ///< Number of such allocations
	int m_unit_bytes; ///< Size of each "object".
	int m_units_per_alloc; ///< Nr of "objects" allocated by each call to malloc/calloc
	int m_current_local_index; ///< "object" index into the last allocated chunk which the next allocation will return
	int m_current_global_index; ///< index into m_malloced to the malloc/calloc allocated chunk for which the next allocation will take place
	BOOL m_mem_zero; /// True if calloc should be used.
};

/** Class for allocating bytes which makes larger allocations using OpSpellCheckerFixedAllocator, and allocates 
	portions of this memory "little-by-little" - for avoiding too much use of the real memory allocator. */
class OpSpellCheckerAllocator
{
public:
	/** Each chunk that will be allocated by the real allocator will be 128kb */
	OpSpellCheckerAllocator();
	
	/** @param chunk_sz Number of bytes for each real allocation. */
	OpSpellCheckerAllocator(int chunk_sz);
	~OpSpellCheckerAllocator() { Clean(); }
	
	/** Frees all memory. */
	void Clean() { m_allocator.Clean(); Init(m_chunk_sz); }
	
	/** Same effect as Clean(), but the first allocation (by the real allocator) will not be freed. */
	void Reset();
	
	/** Allocates bytes bytes that will be 1-byte aligned */
	void *Allocate8(int bytes) { return AllocateAny(bytes,0); }
	
	/** Allocates bytes bytes that will be 2-byte aligned */
	void *Allocate16(int bytes) { return AllocateAny(bytes,1); }
	
	/** Allocates bytes bytes that will be 4-byte aligned */
	void *Allocate32(int bytes) { return AllocateAny(bytes,3); }
	
	/** Allocates bytes bytes that will be 8-byte aligned */
	void *Allocate64(int bytes) { return AllocateAny(bytes,7); }
#ifdef SIXTY_FOUR_BIT
	/** Allocates bytes bytes that will be 8-byte aligned */
	void *AllocatePtr(int bytes) { return Allocate64(bytes); }
#else // SIXTY_FOUR_BIT
	/** Allocates bytes bytes that will be 4-byte aligned */
	void *AllocatePtr(int bytes) { return Allocate32(bytes); }
#endif // !SIXTY_FOUR_BIT
	
	/** Tells how many bytes of the last allocation that really was used (the rest will be availible for for allocations that follows). */
	void OnlyUsed(int bytes, BOOL zero_out_unused = TRUE);
	
	/** Similar to OnlyUsed(), but allocations might have occured in between. This in-between allocations must also 
		have been "rewinded", or we will end up corrupting data.. */
	void RewindBytesIfPossible(int bytes);
	
	/** Returns number of bytes allocated by the real allocator. */
	UINT32 GetAllocatedBytes() { return m_allocator.GetAllocatedBytes(); }

#ifdef SPC_DEBUG
	void VerifyMemOk() { m_allocator.VerifyMemOk(); }
#endif // SPC_DEBUG
	
private:
	void Init(int chunk_sz);
	void *AllocateAny(int bytes, int align_pattern);

private:
	OpSpellCheckerFixedAllocator m_allocator; ///< We use OpSpellCheckerFixedAllocator to do the actual allocations.
	UINT8 *m_mem; ///< Address where the next allocation will take place.
	int m_bytes_left; ///< Bytes left before next real allocation must be made.
	int m_last_allocated; ///< How many byte where allocated last time, necessary of OnlyUsed()
	int m_chunk_sz;	///< Size (in bytes) of each real allocation.
};

#ifndef SPC_USE_HUNSPELL_ENGINE

/** This class encapsulates a stream of bits that could be read/written, for operations like: write 3 bits, then write 7 bits, etc. */
class OpSpellCheckerBitStream
{
public:

	/** Constructs a bit-stream that starts at mem. For using WriteVal32, ReadVal32, WriteVal16 and ReadVal16 must this
		be true: mem MUST have >= 4 bytes available (at the "end" of the stream) then what will be read/written */
	OpSpellCheckerBitStream(void *mem) { m_byte = (UINT8*)mem; m_start = m_byte; m_bit = 0; }
	
	/** Same as the constructor above, but specifies also which bit the stream starts at. */
	OpSpellCheckerBitStream(void *mem, int bit) { SetStart(mem, bit); }
	OpSpellCheckerBitStream() {};
	
	/** Sets the start of the stream to byte mem at bit bit. */
	void SetStart(void *mem, int bit) { m_byte = ((UINT8*)mem) + (bit>>3); m_start = m_byte; m_bit = bit & 7; }
	
	/** Writes all 32 bits of the value val to the stream, increases the stream-position with bits bits (bits must be <= 32) */
	void WriteVal32(UINT32 val, int bits);
	
	/** Writes all 16 bits of the value val to the stream, increases the stream-position with bits bits (bits must be <= 16) */
	void WriteVal16(UINT16 val, int bits);
	
	/** Writes the value val to the stream, but not over any unecessary byte-limit. For example, if the current bit
		position (m_bit) is 5 and bits is 7, then will 11 bits of val be written to the stream (but the bit-position
		will only be increased with 7)*/
	void WriteVal16NoOverByte(UINT16 val, int bits);
	
	/* Writes a single bit to the stream, 1 if set is TRUE, 0 if it's FALSE. */
	void WriteBit(BOOL set);
	
	/** Writes bits bits from data to the stream without overwriting any bits after the last. bits can have any positive value. */
	void WriteDataNoOverWrite(UINT8 *data, int bits);	
	
	/** Read bits bits where bits must be <= 32 */
	UINT32 ReadVal32(int bits);
	
	/** Read bits bits where bits must be <= 16 */
	UINT16 ReadVal16(int bits)
	{
		OP_ASSERT(bits <= 16);
		
		UINT32 res = 0;

		int end_bit_ofs = m_bit + bits;
		int bytes_needed = (end_bit_ofs + 7)/8;
		
		switch ( bytes_needed )
		{
			case 3:
				res |= m_byte[2] << 16;
			case 2:
				res |= m_byte[1] << 8;
			case 1:
				res = (res | m_byte[0]) >> m_bit;
				break;
			case 0:
				return 0;
			default:
				OP_ASSERT(!"Too many bytes");
				return 0;
		}

		res &= ~((UINT32)-1 << bits);

		m_byte += end_bit_ofs >> 3;
		m_bit = end_bit_ofs & 7;
		return res;	
	}

	/** Reads a single bit from the stream and returns TRUE if its set. */
	BOOL ReadBit()
	{
		BOOL res = !!(*m_byte & 1<<m_bit);
		m_byte += ++m_bit >> 3;
		m_bit &= 7;
		return res;
	}
	
	/** Reads a pointer from the stream (32 or 64 bits). */
	void* ReadPointer();
	
	/** Read 16 bits from the stream, but don't increase the stream-pointer (special construction for reading huffman codes) */
	UINT16 Read16BitNoInc();
	
	/** Increases the position in the stream with bits bits, if bits < 0 will we go backward in the stream instead. */
	void IncBits(int bits) { m_byte += (bits+m_bit)>>3; m_bit = (bits+m_bit)&7; }
	
	/** Returns how many full bytes that is the size of the stream. */
	int GetCurrentBytes() { return (int)(m_byte - m_start) + (m_bit ? 1 : 0); }
	
	/** Returns how many full bytes that is the size of the stream - if the stream would have started at mem */
	int GetByteCountFrom(void *mem) { return (int)(m_byte - (UINT8*)mem) + (m_bit ? 1 : 0); }
	
	/** Returns the number of bits there are between the current position in the stream (where next bit will be read/written) and mem */
	int GetBitCountFrom(void *mem) { return ((int)(m_byte - (UINT8*)mem) << 3) + m_bit; }
	
	/** Current byte position (where next bit will be read/written) */
	UINT8 *Byte() { return m_byte; }
	
	/** Current bit position (where next bit will be read/written) */
	int Bit() { return m_bit; }

private:
	UINT8 *m_byte; ///< See Byte()
	int m_bit; ///< See Bit()
	UINT8 *m_start; ///< The start byte of the stream.
};

/** Template for implementing a sparse representation of the unicode space, to say it in a fancy way. The idea is
	that there are several situations where we like to save some data for the unicode characters in an utf-8 dictionary,
	e.g. the frequency if each character that is present in the dictionary. As there are 0x110000 different characters
	in utf-8, we don't want to create an array with so many elements. So instead we create an array where each element
	represents 256 codepoints. And when we need to assign a value to any codepoint, an array of 256 elements is created
	which is pointed to by the "root array". When trying to read a value (by GetVal()) where no "leaf array" is allocated
	will on object of type T be returned with all bytes set to zero. So, objects with vtables should NOT be stored here. 
	This implementation is now a bit "messy" due to the slight over-use of OpSpellCheckerFixedAllocator, so if something
	seems unclear, re-implement this class! */
template <class T> class UTFMapping
{
#define SPC_UTF_CODEPOINT_BITS 21
#define SPC_UTF_LEAF_BITS 8
#define SPC_UTF_LEAFS_PER_MALLOC_BITS 4
#define SPC_UTF_MAX_CODEPOINTS (1<<SPC_UTF_CODEPOINT_BITS)
#define SPC_UTF_LEAFS_PER_MALLOC (1<<SPC_UTF_LEAFS_PER_MALLOC_BITS)
#define SPC_UTF_CODEPOINTS_PER_LEAF (1<<SPC_UTF_LEAF_BITS)
#define SPC_UTF_LEAF_ARRAY_SIZE (SPC_UTF_MAX_CODEPOINTS/SPC_UTF_CODEPOINTS_PER_LEAF)
#define SPC_UTF_LEAF_ADDR(idx) (&(((T*)(m_leaf_allocator->GetMallocAddr((idx)>>SPC_UTF_LEAFS_PER_MALLOC_BITS)))[(((idx)&(SPC_UTF_LEAFS_PER_MALLOC-1))<<SPC_UTF_LEAF_BITS)]))
#define SPC_UTF_VALUE_AT_ALLOC_IDX(idx,cp) (((T*)(m_leaf_allocator->GetMallocAddr((idx)>>SPC_UTF_LEAFS_PER_MALLOC_BITS)))[(((idx)&(SPC_UTF_LEAFS_PER_MALLOC-1))<<SPC_UTF_LEAF_BITS) + ((cp)&(SPC_UTF_CODEPOINTS_PER_LEAF-1))])

public:
	/** Static create function, as you can see... */
	static UTFMapping<T> *Create();
	~UTFMapping() { OP_DELETE(m_leaf_allocator); }
	
	/** Returns the value at codepoint cp into res - if not FALSE is returned, that means there are no "leaf array" for the codepoint. */
	BOOL GetVal(int cp, T &res);
	
	/** Returns the value at codepoint cp, a zeroed out object is returned if there are no "leaf-array" for the codepoint. */
	T GetVal(int cp);
	
	/** Returns the value at codpoint cp assuming we know there is a "leaf-array" for that codepoints - or we will crash! */
	T GetExistingVal(int cp);
	
	/** Sets the value at codepoint cp to val, ERR_NO_MEMORY will be returned if a "leaf-array" must be allocated - and allocation fails. */
	OP_STATUS SetVal(int cp, T val);

	/** Returns how many codepoints that have values that are not zeroed-out. */
	int CountNonZero();
	
	/** Writes out all (codepoint,value) pairs where value is not zeroed-out, like this (starting at memory location dst):
		(cp0,value0),(cp1,value1),(cp2,value2),...,(cpn,valuen). The number of pairs written out will be equal to CountNonZero(). */
	void WriteOutNonZeroIndexesAndElements(void *dst);
	
	/** Returns the number of bytes allocated by this UTFMapping, including sizeof(*this). */
	UINT32 GetAllocatedBytes() { return sizeof(*this) + sizeof(*m_leaf_allocator) + m_leaf_allocator ? m_leaf_allocator->GetAllocatedBytes() : 0; }

protected:
	UTFMapping();
	OP_STATUS Init();

private:
	OP_STATUS AllocateLeafAt(int leaf_idx);

private:
	OpSpellCheckerFixedAllocator *m_leaf_allocator; ///< The allocator for the "leaf-arrays".
	
	/** The array of "pointers" to "leaf-arrays", the "pointers" are indexes of the allocated "leaf-arrays", m_leaf_allocator can be used for
		obtaining the real addresses. An index of 0xFFFF implies that no "leaf-array" is allocated. */
	UINT16 m_leafs[SPC_UTF_LEAF_ARRAY_SIZE];	
	OP_STATUS m_status; ///< Init status, only used during construction.
	int m_leaf_count; ///< Number of allocated "leaf-arrays"
	T m_empty; /// A zeroed-out instance of T...
};

#endif // !SPC_USE_HUNSPELL_ENGINE

/** The class encapsulates an "expanding buffer", that is, an array of T objects which doubles it's size each
	time it become full. */
template <class T> class ExpandingBuffer
{
public:
	/**
		@param init_capacity The number of elements to allocate for the first allocation.
		@param allocator If allocator == NULL is memory allocated/deallocated using new[] and delete[]. Ohterwise
		its allocated by using allocator. This implies that when the size doubles and a new array is allocated - the
		old array will not be freed.
	*/
	ExpandingBuffer(int init_capacity, OpSpellCheckerAllocator *allocator) { Init(init_capacity, allocator); }
	
	/** The new[] and delete[] operators will be used for allocation/deallocation of the array. The number of elements allocated the first time will be init_capacity. */
	ExpandingBuffer(int init_capacity) { Init(init_capacity, NULL); }
	
	/** The new[] and delete[] operators will be used for allocation/deallocation of the array. The number of elements allocated the first time will be 128. */
	ExpandingBuffer() { Init(128, NULL); }
	
	/** The destructor will not free the array if an OpSpellCheckerAllocator is used. */
	~ExpandingBuffer() { if (!m_allocator) OP_DELETEA(m_data); }
	void Init(int init_capacity, OpSpellCheckerAllocator *allocator) { m_data = NULL; m_size = 0; m_capacity = 0; m_init_capacity = init_capacity; m_allocator = allocator; }
	T& operator[] (unsigned int i) { return m_data[i]; }
	
	/** Adds an element "last" in the buffer and returns the index in the buffer (array), or -1 if a memory allocation failed. */
	int AddElement(T elm);
	
	/** Returs the address of the start of the buffer (array). */
	T *Data() { return m_data; }
	
	/** Resizes the buffer with new_size elements. */
	OP_STATUS Resize(int new_size);
	
	/** Returns the element at index index... */
	T GetElement(int index) { OP_ASSERT((unsigned int)index < (unsigned int)m_size); return m_data[index]; }
	
	/** Returns a pointer to the element at index index. */
	T* GetElementPtr(int index) { OP_ASSERT((unsigned int)index < (unsigned int)m_size); return &m_data[index]; }
	
	/** Returns number of elements in the buffer. */
	int GetSize() { return m_size; }
	
	/** Empties the buffer, if free_data == TRUE and OpSpellCheckerAllocator is not used, then will the array be freed. */
	void Reset(BOOL free_data = TRUE) { if (!m_allocator && free_data) { OP_DELETEA(m_data); } m_data = NULL; m_size = 0; m_capacity = 0; }
	
	/** Sets the initial capacity to init_capacity elements, that is, how many elements space should be allocated for at the first allocation. */
	void SetInitCapacity(int init_capacity) { m_init_capacity = init_capacity; }
	
	/** If an OpSpellCheckerAllocator is not used will this return the size in bytes of the array. */
	UINT32 GetAllocatedBytes() { return m_allocator ? 0 : sizeof(T)*m_capacity; }

private:
	T *m_data; ///< The array (buffer)
	int m_size; ///< How many elements there are currently in the buffer.
	int m_capacity; ///< How many elements there are space for before a new allocation needs to take place.
	int m_init_capacity; ///< How many elements space should be allocated for at the first allocation.
	OpSpellCheckerAllocator *m_allocator; ///< The OpSpellCheckerAllocator used for allocation or NULL if new[] should be used.
};

// Define these in the header file to get rid of explicit
// specification of each template in the cpp file. Also, ADS seems to
// have problem with it if they reside in the cpp file.

template <class T> int ExpandingBuffer<T>::AddElement(T elm)
{
	if (m_size == m_capacity)
	{
		int new_capacity = m_data ? m_capacity*2 : MAX(m_init_capacity,1);
		T *new_data;
		if (m_allocator)
			new_data = (T*)m_allocator->AllocatePtr(sizeof(T)*new_capacity);
		else
			new_data = OP_NEWA(T, new_capacity);
		if (!new_data)
			return -1;
		if (m_data)
		{
			op_memcpy(new_data,m_data,m_capacity*sizeof(T));
			if (!m_allocator)
				OP_DELETEA(m_data);
		}
		m_data = new_data;
		m_capacity = new_capacity;
	}
	m_data[m_size] = elm;
	return m_size++;
}

template <class T> OP_STATUS ExpandingBuffer<T>::Resize(int new_size)
{
	if (new_size > m_capacity)
	{
		int new_capacity = m_data ? new_size*2 : MAX(new_size,m_init_capacity);
		T *new_data;
		if (m_allocator)
			new_data = (T*)m_allocator->AllocatePtr(sizeof(T)*new_capacity);
		else
			new_data = OP_NEWA(T, new_capacity);
		if (!new_data)
			return OpStatus::ERR_NO_MEMORY;
		if (m_data)
		{
			op_memcpy(new_data,m_data,m_capacity*sizeof(T));
			if (!m_allocator)
				OP_DELETEA(m_data);
		}
		m_data = new_data;
		m_capacity = new_capacity;
	}
	m_size = new_size;
	return OpStatus::OK;
}

/** This class holds an (usually) sorted array of uni_char strings. Binary search is used for finding a particular string. 
	Both pointers to the strings and the strings themselves are allocated and managed by this class. So AddString() for 
	example creates a copy of the argument-string. */
class SortedStringCollection
{
public:
	SortedStringCollection() { m_string_allocator = NULL; m_sorted = TRUE; m_ref = 0; }
	
	/** If string_allocator != NULL will this allocator be used for all allocations. This means that no data will be freed, e.g. upon calling RemoveString(). */
	SortedStringCollection(OpSpellCheckerAllocator *string_allocator) { m_string_allocator = string_allocator; m_sorted = TRUE; m_ref = 0; }
	~SortedStringCollection();
	
	/** Adds a copy of the string string to the collection, if sort == FALSE might the collection become unsorted
		so Sort() has to be used later before RemoveString() and HasString() can be used. */
	OP_STATUS AddString(const uni_char *string, BOOL sort = TRUE);
	
	/** Removes all copies of string from the collection, returns TRUE if there where any copies that where removed. */
	BOOL RemoveString(const uni_char *string);
	
	/** Returns TRUE if there is at least one copy of string in the collection. */
	BOOL HasString(const uni_char *string, const uni_char* string_end = NULL);
	
	/** Makes sure the collection is sorted, might result in ERR_NO_MEMORY. */
	OP_STATUS Sort();
	
	/** Reference counter, "hack" for inheriting the collection of ignored words in between OpSpellCheckerSession objects... */
	int IncRef() { return ++m_ref; }
	int DecRef() { return --m_ref; }
	int GetRef() { return m_ref; }

private:
	ExpandingBuffer<uni_char*> m_strings; ///< Pointers to the strings in the collection, usually sorted.
	OpSpellCheckerAllocator *m_string_allocator; ///< If this != NULL is it the allocator for all allocations (of the pointer array and copies of strings).
	BOOL m_sorted; ///< If this is FALSE is the collection un-sorted and needs sorting before HasString() and RemoveString() can be used.
	int m_ref; ///< The reference-counter mentioned previously.
};

/** "Shortcut" for opening an OpFile, calls Construct(), Open() and GetFileLength() - and Close() upon error. */
OP_STATUS SCOpenFile(uni_char *path, OpFile &file, int &file_length, int mode = OPFILE_READ);

/** Returns c == '\n' || c == '\r' */
BOOL is_ascii_new_line(UINT8 c);

/** Returns c == ' ' || c == '\t' */
BOOL is_word_separator(UINT8 c);

/** Returns TRUE if s1 and s2 are identical. */
BOOL str_eq(UINT8 *s1, UINT8 *s2);

/** Returns TRUE if i1 and i2 are identical, but only check at most len "characters". */
BOOL str_equal(UINT8 *i1, UINT8 *i2, int len);

/** Returns TRUE if str contains ch. */
BOOL uni_fragment_is_any_of(uni_char ch, const uni_char *str, const uni_char* str_end);

#ifndef SPC_USE_HUNSPELL_ENGINE

/** Returns TRUE if str contains at least one character not present in not. */
BOOL has_any_not(UINT8 *str, const char *_not_);

/** Removes all backslashes from word, and returns (without any real point actually) word again. */
UINT8 *bs_strip(UINT8 *word);

/** Trims ascii-spaces and '\t' from word where word_end points at the first character after word, and returns the trimmed string which is >= word. */
UINT8 *ws_strip(UINT8 *word, UINT8 *word_end);

/** Scanns through the bytes bytes starting at data and replaces all occurences of old_byte with new_byte. */
void replace_byte(UINT8 *data, int bytes, UINT8 old_byte, UINT8 new_byte);

/** Reads a positive decimal integer from str which are allowed to be <= max_num. bytes_used will be set to the 
	number of bytes "consumed" and the number read will be returned, or -1 if the number where invalid or too large. */
int pos_num_to_int(UINT8 *str, int max_num, int &bytes_used);

/** Returns TRUE if str only contains characters >= '0' && <= '9' */
BOOL is_number(const uni_char *str);

#endif //!SPC_USE_HUNSPELL_ENGINE

BOOL is_ambiguous_break(uni_char ch);

#ifndef SPC_USE_HUNSPELL_ENGINE

/** strlen for scUTF-32. */
int wide_str_len(UINT32 *str);

/** strcmp for scUTF-32 strings word and other with lengths word_len and other_len. */
int wide_strcmp(UINT32 *word, int word_len, UINT32 *other, int other_len);

/** Returns TRUE if the len first scUINT32 values of i1 and i2 is identical. */
BOOL indexes_equal(UINT32 *i1, UINT32 *i2, int len);

/** Returns the length of the UTF-8 string str in number of characters, NOT bytes. */
int utf8_strlen(UINT8 *str);

/** Decode a single UTF-8 character and return the codepoint of -1 if it's an invalid character, return the bytes read into bytes_used. */
int utf8_decode(UINT8 *str, int &bytes_used);

/** Converts the UTF-8 string str to UTF-32 and writes it to res.
	@param str The UTF-8 string to convert.
	@param res Where the resulting UTF-32 string should be written. 
	@param bytes_consumed Number of bytes consumed from str, this INCLUDES '\0'.
	@param How many UTF-32 characters where written but EXCLUDING the terminating 0.
	@return FALSE if str is wrongly encoded. */
BOOL utf8_decode_str(UINT8 *str, UINT32 *res, int &bytes_consumed, int &chars_written);

/** Reads a flag (from an .aff or .dic file) and returns the numerical representation of it (see the FlagEncoding enum). 
	@param str Where to read the flag from
	@param encoding The type of encoding used by the dictionary. 
	@param bytes_used Will contain the number of bytes "consumed".
	@return The numerical representation of the flag, or -1 if decoding failed. */
int decode_flag(UINT8 *str, FlagEncoding encoding, int &bytes_used);

/** Converts the 8-bit encoded or UTF-8 encoded string str to scUTF-32.
	@param str The 8-bit encoded or UTF8-encoded string
	@param result The result-buffer, if it's NULL, allocater must not be NULL.
	@param bit8_enc TRUE means 8-bit encoding, FALSE means UTF-8.
	@param allocator If allocator != NULL will it be used for allocating the result buffer, result will then be set to that buffer.	
	@return Error status code. */
OP_STATUS get_wide_str_ptr(UINT8 *str, UINT32 *&result, BOOL bit8_enc, OpSpellCheckerAllocator *allocator);

/** Converts the 8-bit encoded or UTF-8 encoded string str to an scUTF-32 string which will be added to buffer.
	@param str The string to convert and add.
	@param buffer The buffer of pointers to scUTF-32 strings.
	@param bit8_enc TRUE means 8-bit encoding, FALSE means UTF-8.
	@param allocator The allocator to use for allocating the resulting scUTF-32 string that is added to buffer.
	@return Error status code. */
OP_STATUS add_str_to_buffer(UINT8 *str, ExpandingBuffer<UINT32*> *buffer, BOOL bit8_enc, OpSpellCheckerAllocator *allocator);

/** Returns how many bits are necessary to represent count numbers, examples: 0->0, 1->0, 2->1, 3->2, 4->2, 5->3, etc. */
int bits_to_represent(int count);

/** Non-recursive quick-sort, uses buffer as stack instead. Notice: buffer must have room for >= 8*elements bytes! */
template <class T> void quick_sort(T *unsorted, int elements, void *buffer);

/** Add element elm to heap heap which contains elements element. This is for a "reversed" heap where: 
	heap[0] > "any other element in the heap", instead of the opposite. */
template <class T> void rev_heap_add(T elm, T *heap, int elements);

/** Sets heap[0] = elm, and then "bubbles" elm to it's right position in the heap. See description of rev_heap_add() above. */
template <class T> void rev_heap_replace_first(T elm, T *heap, int elements);

/** This struct represents a value (index) and a frequency (count) with some operation so that we can sort this
	structs according to their frequency. */
struct IndexCounterMapping
{
	IndexCounterMapping() {}
	void Set(int count, int index) { this->count = count; this->index = index; }
	BOOL operator==(const IndexCounterMapping &other) const { return this->count == other.count; }
	BOOL operator!=(const IndexCounterMapping &other) const { return this->count != other.count; }
	BOOL operator>=(const IndexCounterMapping &other) const { return this->count >= other.count; }
	BOOL operator<=(const IndexCounterMapping &other) const { return this->count <= other.count; }
	BOOL operator>(const IndexCounterMapping &other) const { return this->count > other.count; }
	BOOL operator<(const IndexCounterMapping &other) const { return this->count < other.count; }
	
	UINT32 index;
	UINT32 count;
};

/** Represents a mapping as defined by the MAP keyword in .aff files. This is used so that we can map several
	characters to the same index in the radix tree. We also makes own mappings so that we map upper-/lower- case
	forms of the same character to the same index. In addition do we make a special mapping for "unusual" characters
	in the dictionary for keeping the number of indexes down. This is because 1. we don't want to make the arrays
	in the radix tree very much larger due to some characters that occurs very seldom, and 2. because we will wast
	even more space for this "unusual" characters in the "index-combination" bit-field where we set bits for existing
	combination of n indexes that follows each other (for "n-gram suggestions", this is currently not used though..). */
struct MapMapping
{
	int char_count; ///< Number of characters in this mapping.
	int bit_representation; ///< How many bits is necessary to index into chars excluding the most frequent char, this is bit_representation(char_count-1).
	UINT32 *chars; ///< The characters in the mapping.
	UINT32 *frequencies; /// For each character in chars, how many times do they appear in the dictionary, including inflections?
	BOOL exists; ///< Does this mapping contain any characters that exists att all anywhere in the dictionary?
};

/** A "pointer" to a specific character in a specific mapping. */
struct MappingPointer
{
	MappingPointer() { mapping_idx = 0; mapping_pos = 0; }
	MappingPointer(UINT16 idx, UINT16 pos) { mapping_idx = idx; mapping_pos = pos; }
	BOOL IsPointer() { return !!mapping_idx; }
	BOOL operator!=(const MappingPointer &other) const { return this->mapping_idx != other.mapping_idx || this->mapping_pos != other.mapping_pos; }

	UINT16 mapping_idx; ///< The index into OpSpellChecker::m_mappings where the character is mapped.
	UINT16 mapping_pos; ///< The index into MapMapping::chars where the char is (including the most frequent character).
};

/** This is a struct for converting indexes stored in the radix-tree in memory to characters. After each word in the
	dictionary is there a list of characters in the word that is NOT the most frequent in it's mapping. This consists
	of positios into the "index-string" along with the indexes into the associated MapMapping's chars array excluding
	the most frequent character. */
struct IndexAndMapping
{
	IndexAndMapping() {}
	IndexAndMapping(UINT16 index, UINT16 mapping) { this->index = index; this->mapping = mapping; }
	UINT16 index; ///< Index into the "index string"
	UINT16 mapping; ///< Index into the mapping's (MapMapping's) chars array - excluding the most frequent character.
};

/** An entry in a huffman-table. */
struct HuffmanEntry
{
	HuffmanEntry() { Set(0,0); }
	void Set(UINT16 val, UINT16 code_len) { value = val; code_length = code_len; }
	UINT16 value; ///< The value that the huffman-code encodes.
	UINT16 code_length; ///< The number of bits in the huffman-code.
};

/** A "higher-level" variant of IndexAndMapping (see description of that struct). Here is also the position into
	an "index-string" stored, but it is so together with the real character that should be on that position. */
struct PosToChar
{
	PosToChar(UINT32 pos, UINT32 char_value) : pos(pos), char_value(char_value) {}
	UINT32 pos; ///< Position into an "index-string".
	UINT32 char_value; ///< The character (which is not the most frequent one in it's mapping).
};

/** A condition in an affix-rule as it is returned by OpSpellChecker::GetAffixConditionData. For example, the condition-string
	^[abc] will result in an AffixConditionData that has: chars == "abc", len == 3 and negative == TRUE. */
struct AffixConditionData
{
	UINT32 *chars; ///< The characters in the condition (see above)
	int len; ///< The length of chars.
	BOOL negative; ///< TRUE for a '^' condition.
};

/** A condition in an compoundrule as it is returned by OpSpellChecker::GetCompoundRuleConditionData. For example, the condition-string
	^[abc]+ will result in an AffixConditionData that has: chars == "abc", len == 3, negative == TRUE and multiple_type == '+'. */
struct CompoundRuleConditionData
{
	UINT16 *indexes; ///< The flag-indexes for the compoundrule flags in the condition (see above).
	int len; ///< The length of indexes.
	BOOL negative; ///< TRUE for a '^' condition.
	int multiple_type; /// 0 means no multiple type, otherwise will the value be '+' or '*'.
};

/** This class corresponds to the ALIAS statements in the .aff files. Aliases are numbers (starting at 1), that is
	a substitution for a specific set of flags. */
class FlagAlias
{
public:
	FlagAlias() { SetFlags(NULL,0); }
	FlagAlias(UINT16 *flags, int flag_count) { SetFlags(flags,flag_count); }
	~FlagAlias() {}
	void SetFlags(UINT16 *flags, int flag_count) { m_flags = flags; m_flag_count = flag_count; }
	UINT16 *GetFlags() { return m_flags; }
	int GetFlagCount() { return m_flag_count; }

private:	
	int m_flag_count; ///< Number of flags represented by this alias..
	UINT16 *m_flags; ///< ...and the flags themselves.
};

/** For each flag defined in the .aff file, we'll create a corresponding FlagClass object - or an object of a derived
	class, i.e. the OpSpellCheckerAffix class. The FlagClass objects are accessable by looking up OpSpellChecker::m_flag_mapping and
	OpSpellChecker::m_flag_pointers. */
class FlagClass
{
public:
	/** Creates a FlagClass of type type with index flag_index. For example, if the flags are ascii-encoded in the
		.aff and .dic files and we encounter the line "COMPOUNDBEGIN x" in the .aff file, then will a new flag be created
		by constructing i using FlagClass(AffixFlagType::AFFIX_TYPE_COMPOUNDBEGIN, 'x'), OpSpellChecker::m_flag_mapping and
		OpSpellChecker::m_flag_pointers must then also be updated accordingly. */
	FlagClass(AffixFlagType type, int flag_index) { m_type = type; m_flag_index = flag_index; m_flag_type_id = -1; m_flag_global_id = -1; }
	virtual ~FlagClass() {}
	AffixFlagType GetType() { return m_type; }
	
	/** Return the "flag index", that is, the numerical encoding of the flag, the index of flag 'A' is 65, and so on. */
	int GetFlagIndex() { return m_flag_index; }
	
	/** Sets the "type ID". All FlagClass'es of the same type have different type ids, starting from 0. */
	void SetFlagTypeId(int id) { m_flag_type_id = id; }
	
	/** Returns the "type ID". All FlagClass'es of the same type have different type ids, starting from 0. */
	int GetFlagTypeId() { return m_flag_type_id; }

	/** Sets the "global ID". All FlagClass'es different global ids, starting from 1. */
	void SetFlagGlobalId(int id) { m_flag_global_id = id; }
	
	/** Returns the "global ID". All FlagClass'es different global ids, starting from 1. */
	int GetFlagGlobalId() { return m_flag_global_id; }

private:
	AffixFlagType m_type;
	int m_flag_index;
	int m_flag_type_id;
	int m_flag_global_id;
};

#define AFFIX_LINE_STACK_WORDS 6
#define HS_INVALID_LINE 0

/** Class for representing a single line which containins a statement (like SFX, MAP, etc) in the .aff file. */
class HunspellAffixFileLine
{
public:
	HunspellAffixFileLine();
	~HunspellAffixFileLine();
	
	/** Sets up this object by giving it the '\0' terminated string line_data. This function will chop up the
		line in words and so on.. */
	OP_STATUS Tokenize(UINT8 *line_data, OpSpellChecker *spell_checker);
	
	/** Returns the "type" of the line, that is, the type of statement (like SFX, MAP, etc). The type is in fact
		an index into the OpSpellChecker::m_affix_parsers array (which contains function-pointers to the functions
		for parsing statements, etc). The special type HS_INVALID_LINE indicates an invalid, or not supported statement. */
	int Type() { return m_type; }
	
	/** Returns the word with index i in the line. The first word in the line which indicates the type of statement,
		"SFX" etc, is not available, so index 0 is the first word after that identifier. Notice: Tokenize() has inserted
		'\0' after each word, so each word is a separate string. */
	UINT8 *WordAt(int i) { return (unsigned int)i < (unsigned int)m_word_count ? m_words[i] : NULL; }
	
	/** Returns the number of words in the line, excluding the statment-identifier (like "SFX"). */
	int GetWordCount() { return m_word_count; }
	
	/** Sets the index of the line among the lines HunspellAffixFileData::m_lines. */
	void SetLineIndex(int i) { m_line_index = i; }
	
	/** Gets the index of the line among the lines HunspellAffixFileData::m_lines. */
	int GetLineIndex() { return m_line_index; }
	
	/** Should be used to set the line index in the original .aff file, perhaps for being able to implement some
		kind of error-reporting when parsing of the .aff file fails. */
	void SetOriginalLineNr(int nr) { m_org_line_nr = nr; }

	/** Should be used to retrieve the line index in the original .aff file, perhaps for being able to implement some
		kind of error-reporting when parsing of the .aff file fails. */
	int GetOriginalLineNr() { return m_org_line_nr; }
	
	/** Any kind of value kan be stored here, for storing miscellaneous information in-between affix-file parsing passes. */
	void SetParserData(void *data) { m_parser_data = data; }
	
	/** Get the data set by SetParserData() */
	void *GetParserData() { return m_parser_data; }

private:
	void ResetData();

private:
	UINT8 **m_words; ///< The words.
	UINT8 *m_tmp_words[AFFIX_LINE_STACK_WORDS]; ///< Buffer for not having to allocate m_words for lines with <= AFFIX_LINE_STACK_WORDS words.
	int m_word_count; ///< Number of elements in m_words.
	int m_type; ///< The type, see description of the Type() function.
	int m_line_index; ///< See description of GetLineIndex()
	int m_org_line_nr; ///< See GetOriginalLineNr()
	void *m_parser_data; ///< See SetParserData()
};

/** This class contains information about the affix-file (.aff), this data is needed during construction of the dictionary-tree
	in memory. It contains fox example the FlagClass objects which describes the meaning of flags. */
class HunspellAffixFileData
{
public:
	/**
		@param affix_data A pointer to the content of the .aff file. This data sould not be deleted or modified by
		code outside of this class after this, that will be handled by this class.
		@param affix_len The length of that data.
		@param spell_checker The OpSpellChecker instance using this data.
	*/
	HunspellAffixFileData(UINT8 *affix_data, unsigned int affix_len, OpSpellChecker *spell_checker);
	~HunspellAffixFileData();
	
	/** Creates HunspellAffixFileLine objects for all lines with valid statements in the affix-file, and
		calls HunspellAffixFileLine::Tokenize() on each. */
	OP_STATUS Tokenize();
	
	/** Frees all data that's not necessary during dictionary-parsing, such as the file-data obtained in the constructor,
		and the HunspellAffixFileLine objects. */
	void ClearFileData();
	
	/** Returns the HunspellAffixFileLine object at line i, this index usually doesn't correspond directly to the lines
		in the file, as only lines with valid/supported statements are taken into account. */
	HunspellAffixFileLine *LineAt(int i) { return (unsigned int)i < (unsigned int)m_line_count ? &m_lines[i] : NULL; }
	
	/** Returns the number of lines, see LineAt(). */
	int LineCount() { return m_line_count; }
	
	/** Sets the type of flag-encoding used. */
	OP_STATUS SetFlagEncoding(FlagEncoding encoding);
	
	/** Returns the type of flag-encoding used. */
	FlagEncoding GetFlagEncoding() { return m_flag_encoding; }
	
	/** Sets the dictionary encoding. */
	OP_STATUS SetDictionaryEncoding(DictionaryEncoding encoding);
	
	/** Returns the dictionary encoding. */
	DictionaryEncoding GetDicionaryEncoding() { return m_encoding; }
	
	/** Returns the FlagClass object for the flag with the numerical representaion of flag_index (see FlagClass). */
	FlagClass *GetFlagClass(int flag_index);
	
	/** Returns the FlagClass with the global ID flag_id, see FlagClass. */
	FlagClass *GetFlagClassByGlobalId(int flag_id) { return m_flag_pointers.GetElement(flag_id); }
	
	/** Adds a new FlagClass object. We're not supposed to be able to add FlagClass'es with the same numerial representation
		twice, since that means that the same flag is defined > 1 time in the .aff file. This happens for a couple
		of dictionaries however, so there is a "hack" in here for "overwriting" the old flag. */
	OP_STATUS AddFlagClass(FlagClass *flag_class, HunspellAffixFileParseState *state);
	
	/** Returns the OpSpellCheckerAllocator which is used by this class, data allocated with this allocator will be
		freed when this object is deleted, which should happen as soon as this loading of the dictionary finishes. */
	OpSpellCheckerAllocator *Allocator() { return &m_allocator; }
	
	/** Returns the number of flags of a cerain type, see AffixFlagType. */
	int GetFlagTypeCount(AffixFlagType type) { return m_flag_type_counters[type]; }
	
	/** Returns bits_to_represent(GetFlagTypeCount(type)), cached here as a little optimization, see bits_to_represent(). */
	int GetFlagTypeBits(AffixFlagType type) { return m_flag_type_bits[type]; }
	
	/** Returns the number of bits needed for representing all global IDs. */
	int GetGlobalIdBits() { return m_global_id_bits; }
	
	/** Setup the bit-represenation counts so that GetGlobalIdBits() nd GetFlagTypeBits() will work. */
	void SetTypeIdBits();
	
	/** Increases the type ID for flag-type type with to_inc steps, without adding any new flags. */
	void IncTypeIdFor(AffixFlagType type, int to_inc = 1) { m_flag_type_counters[type] += to_inc; }
	
	/** Returns a pointer to the affix-char-substitution number for character ch in affix, returns NULL upon OOM.  */
	UINT16 *GetCharIndexPtrForAffix(int ch, OpSpellCheckerAffix *affix);
	
	/** Returns the affix-char-subsitution number for ch in affix. */
	UINT16 GetCharIndexForAffix(int ch, OpSpellCheckerAffix *affix);
	
	/** Converts the scUTF-32 string str with length str_len into affix-char-subsitution numbers for affix and writes it to result. */
	void ConvertStrToAffixChars(UINT32 *str, int str_len, UINT32 *&result, OpSpellCheckerAffix *affix);
	
	/** Convert all aliases's (in m_aliases) flags from numerical representations to global ID's (unknown flags will simply dissapear). */
	OP_STATUS ConvertAliasIndexes();
	
	/** Adds a new alias */
	OP_STATUS AddAlias(FlagAlias alias) { return m_aliases.AddElement(alias) < 0 ? SPC_ERROR(OpStatus::ERR_NO_MEMORY) : OpStatus::OK; }
	
	/** Return the number of aliases. */
	int GetAliasCount() { return m_aliases.GetSize(); }
	
	/** Return the alias a index. */
	FlagAlias *GetAlias(int index) { return index >= 1 && index <= GetAliasCount() ? m_aliases.GetElementPtr(index-1) : NULL; }
	
	/** Store the numerical flag representation flag_index without creating any FlagClass object. This is because circumfixes are flags
		inteneded only to be added on affix-rules. Therefore it seems to be a common practice to mark the full affix as a circumfix
		by using CIRCUMFIX x in the .aff file where e.g. SFX x is defined at another place. So we postpone the creation of the FlagClass
		object unit we know that x is not in fact an affix, the we instead mark all rules in that affix with the AFFIX_TYPE_CIRCUMFIX flag. */
	OP_STATUS AddCircumfixFlagIndex(int flag_index) { return m_circumfixes.AddElement(flag_index) < 0 ? SPC_ERROR(OpStatus::ERR_NO_MEMORY) : OpStatus::OK; }	
	
	/** Should be called after both the first and second pass of the affix file parsing. The first time we'll create a FlagClass
		of the AFFIX_TYPE_CIRCUMFIX for those circumfixes that are not affixes, and after the next pass we set the bit-flag
		AFFIX_TYPE_CIRCUMFIX on all affix rules for affixes that is marked as CIRCUMFIX, see description of AddCircumfixFlagIndex(). */
	OP_STATUS PostProcessCircumfixes(HunspellAffixFileParseState *state);
	
	/** Add the contents of a WORDCHARS statement, str is encoded in the dictionary-encoding. */
	OP_STATUS AddWordChars(UINT8 *str) { return add_str_to_buffer(str, &m_word_chars, m_8bit_enc, &m_allocator); }
	
	/** Add the contents of a BREAK statement, str is encoded in the dictionary-encoding. */
	OP_STATUS AddBreakChars(UINT8 *str) { return add_str_to_buffer(str, &m_break_chars, m_8bit_enc, &m_allocator); }
	
	/** Return the contents of all WORDCHARS statements as scUTF-32 strings. */
	ExpandingBuffer<UINT32*> *GetWordChars() { return &m_word_chars; }
	
	/** Return the contents of all BREAK statements as scUTF-32 strings. */
	ExpandingBuffer<UINT32*> *GetBreakChars() { return &m_break_chars; }

private:
	HunspellAffixFileLine *m_lines; ///< Array of lines with valid/supported statement types.
	UINT8 *m_affix_data; ///< Pointer to the content of the .aff file given in the constructor, this will be tokenized etc.
	unsigned int m_data_len; ///< Length of the .aff file data as given in the constructor.
	int m_line_count; ///< Number of lines in m_lines.
	UINT8 *m_last_line_copy; ///< A copy of the last line in the .aff file-data, if that data wasn't '\0' terminated - because we need it '\0' terminated.
	OpSpellChecker *m_spell_checker; ///< The OpSpellChecker using this object.
	FlagEncoding m_flag_encoding; ///< The flag encoding used.
	DictionaryEncoding m_encoding; ///< The dictionary encoding used.
	BOOL m_8bit_enc; ///< TRUE for 8-bit encodings and FALSE for UTF-8, m_encoding and m_8bit_enc is now actually quite redundant.
	OpSpellCheckerAllocator m_allocator; ///< The allocator used for allocations that should be freed when dictionary loading has finished.
	int m_flag_type_counters[AFFIX_TYPE_COUNT]; ///< Counters for how many flags there are of different AffixFlagType types, or a little less if IncTypeIdFor() has been used to increase type IDs without creating flags.
	int m_flag_type_bits[AFFIX_TYPE_COUNT]; ///< The lookup-table used by GetFlagTypeBits().
	int m_global_id_bits; ///< The value returned by GetGlobalIdBits().
	ExpandingBuffer<FlagAlias> m_aliases; ///< The aliases defined in the .aff file with AF statements.
	ExpandingBuffer<int> m_circumfixes; ///< The numerical representations of the CIRCUMFIX flags encounted, see AddCircumfixFlagIndex().
	ExpandingBuffer<UINT32*> m_word_chars; ///< See GetWordChars()
	ExpandingBuffer<UINT32*> m_break_chars; ///< See GetBreakChars()

	union
	{
		UINT16 *linear;
		UTFMapping<UINT16> *utf;
		void *data;
	} m_flag_mapping; ///< A mapping for fast lookup of numerical flag representation to FlagClass pointers, here is indexes into m_flag_pointers stored, where the actual pointers are.
	ExpandingBuffer<FlagClass*> m_flag_pointers; ///< Pointers to all FlagClass objects created, e.g. to get the flag-class for 'x' with ascii-encoded flags, use m_flag_pointers[m_flag_mapping.linear['x']].
	
	/** Mapping to pointers to the affix-char-substitution tables, e.g. if there is a 8-bit dictionary encoding
		and we wants to know which number we should substitute 'x' for in the OpSpellCheckerAffix a, then we do 
		like this: int substitution = m_affix_char_mapping.linear['x'][a->GetFlagTypeId()]. */
	union
	{
		UINT16 **linear;
		UTFMapping<UINT16*> *utf;
		void *data;
	} m_affix_char_mapping;
};

/** This class describes the current state of the parsing of the affix-file (.aff). We keep this state in order to
	make the parsing interruptable, so we can go back to the message-loop and continue later. For a couple of different
	reasons it's necessary to iterate through the lines in the affix file two times (two "passes"), not all lines however. 
	Therefore HunspellAffixFileParseState keeps a list of lines which should be iterated in the current pass. */
class HunspellAffixFileParseState
{
private:
	/** @param fd The HunspellAffixFileData associated with the parsing. */
	HunspellAffixFileParseState(HunspellAffixFileData *fd);
public:
	/** @param fd The HunspellAffixFileData associated with the parsing. */
	static HunspellAffixFileParseState* Create(HunspellAffixFileData *fd);
	~HunspellAffixFileParseState();

	/** Returns the current line index into the lines kept by the HunspellAffixFileData, see HunspellAffixFileData::LineAt(). */
	int GetCurrentLineIndex();
	
	/** Returns the current line, or NULL if all lines have been parsed in the current pass. */
	HunspellAffixFileLine *GetCurrentLine();
	
	/** Get line index among the lines that are left in the current pass (index into m_current_indexes). */
	HunspellAffixFileLine *GetLineInPass(int index);
	
	/** Returns FALSE if all lines in the current pass has been parsed. */
	BOOL HasMoreLines() { return m_current < m_current_count; }
	
	/** Sets the current line to index line which indexes into all lines, not just the ones in the current pass. It's
		supposed that index line is among the lines to iterate in the current pass. The index into the indexes of lines
		that should be iterated in the current pass is returned. */
	int SetCurrentLine(int line);
	
	/** Goto next line among the lines to iterate in the current pass, returns FALSE if we've reached the last line. */
	BOOL NextLine();
	
	/** Goes to the next pass, the current line will be reset to the first line to iterate in the next pass. Returns
		FALSE if there are no lines to iterate in the next pass. */
	BOOL NextPass();
	
	/** Retuns TRUE if this pass is the last pass. */
	BOOL IsLastPass() { return !m_next_count;}
	
	/** Returns the current pass, starting from zero. */
	int GetPass() { return m_pass; }
	
	/** Adds line_index to the list of lines to iterate in the next pass, line_index is an index to all lines. */
	void IterateInNextPass(int line_index);
	
	/** Adds line to the list of lines to iterate in the next pass. */
	void IterateInNextPass(HunspellAffixFileLine *line) { OP_ASSERT(line); IterateInNextPass(line->GetLineIndex()); }

	/** Returns TRUE if flags has been decoded, if so, this implies that FLAGS statements that is encouted will result in an error
		as we're not supposed to use more then one flag encoding. */
	BOOL GetFlagsUsed() { return m_flags_used; }
	
	/** Set whether flags has been used (decoded), see GetFlagsUsed(). */
	void SetFlagsUsed(BOOL used = TRUE) { m_flags_used = used; }

	/** Returns TRUE if characters have been decoded useing the dictionary-encoding, if so, this implies that SET 
		statements that is encouted will result in an error	as we're not supposed to use more then one dictionary encoding. */
	BOOL GetDicCharsUsed() { return m_dic_chars_used; }
	
	/** Set whether dictionary character have been used (decoded), see GetDicCharsUsed(). */
	void SetDicCharsUsed(BOOL used = TRUE) { m_dic_chars_used = used; }
	
	/** Returns TRUE if at leas one AF statement have been found so far. */
	BOOL GetAliasesDetected() { return m_aliases_detected; }
	
	/** Sets whether at least one AF statement have been found so far. */
	void SetAliasesDetected(BOOL detected = TRUE) { m_aliases_detected = detected; }
	
	/** Returns TRUE if at leas one MAP statement have been found so far. */
	BOOL GetMapDetected() { return m_map_detected; }
	
	/** Sets whether at least one MAP statement have been found so far. */
	void SetMapDetected(BOOL detected = TRUE) { m_map_detected = detected; }

private:
	BOOL m_flags_used; ///< See GetFlagsUsed()
	BOOL m_dic_chars_used; ///< See GetDicFlagsUsed() 
	BOOL m_aliases_detected; ///< See GetAliasesDetected()
	BOOL m_map_detected; ///< See GetMapDetected()
	HunspellAffixFileData *m_file_data; ///< The associated HunspellAffixFileData
	int m_pass; ///< The current pass, starting from zero.
	int *m_current_indexes; ///< The array of indexes among all lines that should be iterated in the current pass.
	int m_current; ///< The current index among the indexes of all lines that should be iterated in the current pass.
	int m_current_count; ///< The number of indexes in m_current_indexes.
	int *m_next_indexes; ///< The array of indexes among all lines that should be iterated in the NEXT pass.
	int m_next_count; ///< The number of indexes in m_next_indexes.
};

/** This is a simple mapping between a keyword (statment) in the affix-file (.aff) and the member-function
	in OpSpellChecker that should parse it, i.e. "COMPOUNDRULE" and OpSpellChecker::ParseCOMPOUNDRULE(). */
struct AffixParser
{
	AffixParser() { str = NULL; func = NULL; }
	AffixParser(OP_STATUS (OpSpellChecker::*func)(HunspellAffixFileData*,HunspellAffixFileParseState*), UINT8 *str)
	{
		this->func = func;
		this->str = str;
	}
	UINT8 *str; // The keyword
	OP_STATUS (OpSpellChecker::*func)(HunspellAffixFileData*,HunspellAffixFileParseState*); // The parsing function.
};

/** This class represent a single affix-rule that belongs to an OpSpellCheckerAffix. */
class OpSpellCheckerAffixRule
{
public:
	/**
		@param strip_chars 0 terminated scUTF-32 string with the characters to strip.
		@param affix_chars 0 terminated scUTF-32 string with the affix to add.
		@param flags The flags (if any) for the affix-rule. Upon construction will this be set to the un-decoded
		flags as being found in the affix-file. Later will this change to point at the parsed flags as parsed by
		OpSpellChecker::GetFlagPtr().
		@parm cond_str The unparsed condition string as found in the affix-file.
	*/
	OpSpellCheckerAffixRule(UINT32 *strip_chars, UINT32 *affix_chars, void *flags, UINT8 *cond_str) 
	{ 
		m_strip_chars = strip_chars; 
		m_strip_char_len = wide_str_len(strip_chars);
		m_affix_chars = affix_chars; 
		m_affix_char_len = wide_str_len(affix_chars);
		m_flags.data = flags; 
		m_cond_str = cond_str; 
		m_cond_len = 0;
		m_index = -1;
	}
	OpSpellCheckerAffixRule()
	{
		m_strip_chars = NULL;
		m_strip_char_len = 0;
		m_affix_chars = NULL;
		m_affix_char_len = 0;
		m_flags.data = NULL; 
		m_cond_str = NULL; 
		m_cond_len = 0;
		m_index = -1;
	}
	~OpSpellCheckerAffixRule() {}
	
	/* A bunch of getters and setters... */
	UINT32* GetStripChars() { return m_strip_chars; }
	void SetStripChars(UINT32 *strip_chars) { m_strip_chars = strip_chars; m_strip_char_len = wide_str_len(strip_chars); }
	int GetStripCharLength() { return m_strip_char_len; }
	UINT32 *GetAffixChars() { return m_affix_chars; }
	void SetAffixChars(UINT32 *affix_chars) { m_affix_chars = affix_chars; m_affix_char_len = wide_str_len(affix_chars); }
	int GetAffixCharLength() { return m_affix_char_len; }
	UINT16 *GetFlags() { return m_flags.flags; }
	UINT8 *GetFlagsUnParsed() { return m_flags.un_parsed; }
	void SetFlags(void *flags) { m_flags.data = flags; }
	UINT8 *GetConditionString() { return m_cond_str; }
	void SetConditionString(UINT8 *cond_str) { m_cond_str = cond_str; }
	void SetConditionLength(int cond_len) { m_cond_len = cond_len; }
	int GetConditionLength() { return m_cond_len; }
	void SetIndex(int index) { m_index = index; }
	int GetIndex() { return m_index; }

private:
	UINT32 *m_strip_chars; ///< The strip chars.
	int m_strip_char_len; ///< The number of strip chars.
	UINT32 *m_affix_chars; ///< The affix-chars which should be added to the stem.
	int m_affix_char_len; ///< The number of affix-chars.
	union {
		UINT16 *flags; ///< The format of the parsed flags is: bit-flags (16bit) + affix_count (16bit) + affixes(16bit*) + compoundrule flag count(16bit) + compoundrule flags(16bit*)
		UINT8 *un_parsed; ///< The unparsed flags.
		void *data;
	} m_flags; /// The flags
	UINT8 *m_cond_str; ///< The unparsed condition string.
	int m_cond_len; ///< The number of conditions, e.g. [hejsan][hej] is 2 conditions.
	int m_index; ///< The index of this affix-rule among the OpSpellCheckerAffix's affix-rules.
};

/** This class represents an affix together with a list of affix-rules (OpSpellCheckerAffixRule), that is, it represents 
	a prefix specified by the PFX statement or a suffix, specified with the SFX statement, in the affix-file. */
class OpSpellCheckerAffix : public FlagClass
{
public:
	/**
		@param flag_index The numerical representation of the flag, e.g. 'x' for "SFX x".
		@param is_suffix TRUE if this is a suffix, FALSE if it's a prefix.
		@param can_combine TRUE if this affix is allowed to be combined with another affix on the other size of the word.
		@param rule_count The number of affix-rules - this doesn't need to be correct however.
	*/
	OpSpellCheckerAffix(int flag_index, BOOL is_suffix, BOOL can_combine, int rule_count);
	virtual ~OpSpellCheckerAffix();
	int GetRuleCount() { return m_rules.GetSize(); }
	OpSpellCheckerAffixRule GetRule(int index) { return m_rules.GetElement(index); }
	OpSpellCheckerAffixRule *GetRulePtr(int index) { return m_rules.GetElementPtr(index); }
	OP_STATUS AddRule(OpSpellCheckerAffixRule rule);
	BOOL IsSuffix() { return m_is_suffix; }
	BOOL CanCombine() { return m_can_combine; }
	
	/** This function should be called after all rules have been added using AddRule(), it will finalize the setup the 
		"match-bits" that are used for rapidly obtaining the set of affix-rules that matches a certain word. 
		@param max_cond_len Should be the maximum OpSpellCheckerAffixRule::GetConditionLength() among the affix-rules
		that where added to this affix.
		@param char_count The number of different characters that participates in the conditions in the affix-rules. */
	OP_STATUS SetConditionProperties(int max_cond_len, int char_count);
	
	/** Adds a single condition to rule which is supposed to belong to this affix. For example, if rule has a condition
		string that is "[hejsan]^[hej]" should this function be called twice:
		SetRuleConditions(rule, {h*,e*,j*,s*,a*,n*}, 6, 0, FALSE);
		SetRuleConditions(rule, {h*,e*,j*}, 3, 1, TRUE);
		The * means: The character converted so it's affix-char-subsitution number for this affix, obtained by HunspellAffixFileData::GetCharIndexForAffix.
		This will setup the some of the "match-bits" used for rapidly obtaining the set of affix-rules that matches a certain word.
		@param rule The affix rule.
		@param The condition chars as scUTF-32 (but here represented as int).
		@param count The number of condition characters.
		@param pos The position for this condition among all conditions in the rule, see example above.
		@param negative TRUE if the condition is a negative condition (preceded by '^'). */
	void SetRuleConditions(OpSpellCheckerAffixRule *rule, int *cond, int count, int pos, BOOL negative);
	
	/** Returns the maximum condition-length, as was given as argument to SetConditionProperties(). */
	int GetMaxConditionLength() { return m_max_cond_len; }
	
	/** Takes a string converted to affix-char-substitution numbers and outputs indexes of the rules that matches. 
		@param chars Affix-char-substitution numbers obtained from a string by using HunspellAffixFileData::ConvertStrToAffixChars().
		@param bit_buffer A buffer to use during the matching for the bit-mask used for matching rules.
		@param result Buffer for the resulting indexes, that is, indexes of the rules that matches.
		@param result_count The number of matching rules. */
	void GetMatchingRules(UINT32 *chars, UINT32 *bit_buffer, int *result, int &result_count);
	
	/** Returns a cached copy of bits_to_represent(GetRuleCount()). */
	int GetRuleCountBits() { return m_rule_count_bits; }

private:
	BOOL m_is_suffix; ///< TRUE if this is a suffix, FALSE if it's a prefix.
	BOOL m_can_combine; ///< TRUE if this affix could be combined with affixes on the ohter side of words.
	ExpandingBuffer<OpSpellCheckerAffixRule> m_rules; ///< The affix-rules.
	ExpandingBuffer<OpSpellCheckerAffixRule*> m_unconditional_rules; ///< Pointer to the unconditional rules marked with '.' in the affix file.
	int m_char_count; ///< The number of different chars that occurs in rule-conditions, as it was given as an argument to SetConditionProperties().
	int m_max_cond_len; ///< The maximum condition length for any rule in this affix, as it was given as an argument to SetConditionProperties().
	int m_bit_units; ///< Number of UINT32 that is necessary to keep each bit-mask.
	UINT32 *m_bits; ///< m_accept_mask and m_terminate_mask is allocated by a single allocation, this is that memory.
	UINT32 *m_accept_mask; ///< The accept masks for masking out which rules that still matches the string during each step of the matching process.
	UINT32 *m_terminate_mask; ///< The termination masks for determine when conditions for rules have been fully matched. 
	int m_rule_count_bits; ///< See GetRuleCountBits()
};

/** This class represents a single compound rule, as defined by a COMPOUNDRULE statement in the affix-file. */
class CompoundRule
{
public:
	CompoundRule() { Initialize(0,0,NULL); }
	~CompoundRule() { OP_DELETEA(m_multiple); };
	
	/** Initializes the compound-rule.
		@param cond_count The number of conditions that will be added to this compound rule, where '+' conditions are 
		split into two conditions, x+ -> xx*. Example: [abcd]+[werwsd]*[adfa] is 4 conditions.
		@param total_cond_flags The number of different compound rule flags totally (used by all compound rules), this number
		includes the "no compound rule flag" 0, used for words with no compound rule flags when matching compounds. */
	OP_STATUS Initialize(int cond_count, int total_cond_flags, OpSpellCheckerAllocator *allocator);
	
	/** Returns the number of conditions for this compound rule, as givent as argument cond_count to Initialize(). */
	int GetTotalConditionCount() { return m_cond_count; }
	
	/** Returns how many conditions that have been added to this compound rule so far, by AddCondition(). */
	int GetAddedConditionCount() { return m_added_conditions; }
	
	/** Adds a condition to the compound-rule, then m_total_cond_flags have been added will this function setup the
		"jump" and "retry" tables used for matching. Example, this adds the condition ^[abc]* to the compoundrule:
		AddCondition({a*,b*,c*}, 3, TRUE, TRUE); where a*, b* and c* is the type IDs for the compoundruleflags 'a',
		'b' and 'c'.
		@param cond_ids The type IDs for the compoundrule flags, see example above.
		@param cond_ids_count, the number of elements in cond_ids.
		@param is_negative TRUE if this is a '^' condition.
		@param multiple TRUE if this is a '*' condition. As mentioned above should '+' conditions be split like: x+ -> xx*.
		*/
	void AddCondition(UINT16 *cond_ids, int cond_ids_count, BOOL is_negative, BOOL multiple);
	
	/** Reuturns TRUE if the array of compoundrule flag type ID's indexes matches this compound rule. 
		@param indexes The compoundrule flag type ID's to match. For words in the compound that has no compoundrule flags should
		0 be put in this array to indicate "no compound rule flag".
		@param count The number of compoundrule flag type ID's (the number of compounds in the compound word to test).
		*/
	BOOL Matches(UINT16 *indexes, int count);

private:
	int m_cond_count; ///< The number of conditions in this compoundrule as given as argument to Initialize().
	
	/** The "jump" table, used for determine the next condition to match, example: The comopoundrule is a*b*c*d and we
		matches a compound word with compoundrule flags cde. Then will c match c* and the jump-table will tell us to "jump"
		2 steps forward in the condition (to perhaps match c again). And next we matches d in the jump-table and get a match
		and a jump of 2, that is, immediately after the last condition. When looking in the jump table for matching e we'll
		find SPC_NO_JUMP which is everyware in the jump-table at the last "column" after the condition. SPC_NO_JUMP indicates
		that the matching has failed. */
	UINT8 *m_jump_table; 
	
	/** Sometimes we need to try several different "jump-paths" for matching the conditions. For example, if the compound rule
		is: a*b*ac and we try to match the compound word with compoundrule flags ac. Then we'll match a* and the jump-table
		indicates 0 (as we can continue to match a*), when we'll then tries to match c however, that will fail. But if we in 
		the first place would have matched the second a instead we would have succeeded. Therefore, when a* is matched we will
		get a retry-position from the retry-table which will tell us to save our current position and continue at the second
		a if the matching will fail. */
	UINT8 *m_retry_table;
	BOOL *m_multiple; ///< A list of booleans corresponding to each of the conditions in this compoundrule to indicates if it's a '*' condition.
	int m_total_cond_flags; ///< Number of different compoundrule flags totally, as given as argument to Initialize().
	
	/** Last index into m_multiple that is FALSE (or -1 if there is only '*' conditions, which would by the way be a compoundrule
		making all compounds to be valid). If we've pass by this position during matching and reached the end of the compound,
		then we've found a match. */
	int m_last_single_pos; 
	int m_added_conditions; ///< The number of conditions added so far with AddCondition().
};

/** This struct contains flags that have been decoded, in a more suitable format. */
struct FlagPtrInfo
{
	/** Sets the content of this struct from the output from OpSpellChecker::GetFlagPtr() which is located in flag_ptr. */
	void SetValuesFrom(UINT16 *flag_ptr)
	{
		compound_rule_flags = NULL;
		affixes = NULL;

		if (!flag_ptr || !*flag_ptr)
		{
			bit_flags = 0;
			affix_count = 0;
			compound_rule_flag_count = 0;
			return;
		}
		bit_flags = *flag_ptr;
		int pos = 1;
		if (bit_flags & 1<<AFFIX_BIT_HAS_AFFIXES)
		{
			affix_count = flag_ptr[1];
			affixes = flag_ptr + 2;
			pos += 1 + affix_count;
		}
		else
			affix_count = 0;
		if (bit_flags & 1<<AFFIX_BIT_HAS_COMPOUNDRULEFLAG)
		{
			compound_rule_flag_count = flag_ptr[pos];
			compound_rule_flags = flag_ptr + pos + 1;
		}
		else
			compound_rule_flag_count = 0;
	}
	
	UINT16 bit_flags; ///< Flags that can be represented as "ON/OFF" values, such as AFFIX_TYPE_ONLYINCOMPOUND, see AffixFlagType.
	UINT16 *compound_rule_flags; ///< The type IDs for the compoundrules flags decoded.
	UINT16 compound_rule_flag_count; ///< Number of elements in compound_rule_flags.
	int affix_count; ///< Number of elements in affixes.
	UINT16 *affixes; ///< The global IDs for the affix flags decoded.
};

/** This struct represents a word (stem/inflection) that should be added to the dictionary. We use a buffer for up to SPC_MAX_TEMP_WORDS
	such words which we adds to the dictionary tree in "chunks", this is useful as it saves time and space, in particular when all words
	that should be under the same "leaf" in the radix-tree is added together - which quite often will be the case as most dictionarys
	are sorted somwhat alphabetically, and as prefixes are not so usual. */
struct TempWordInfo
{
	void Init() { op_memset(this,0,sizeof(*this)); }
	UINT32 *word; ///< The word in scUTF-32, will later (int OpSpellChecker::ProcessStemAndInflectionsPass[x]) be converted to indexes.
	int word_len; ///< length of word.
	int removed_chars; ///< So many chars where removed due to the strip-chars in an affix-rule, only used for words which are prefix0 + stem OR stem + suffix0.
	int inserted_chars; ///< So many chars where added due to the affix in an affix-rule, only used for words which are prefix0 + stem OR stem + suffix0.
	FlagPtrInfo info; ///< The flags for this word, but without the affix-flags as they are not necessary here.
	
	/** After the scUTF-32 chars have been converted to indexes does this array keep mappings of chars that are not the most 
		frequent in the mapping. See description of IndexAndMapping. */
	IndexAndMapping *mapped_chars;
	int mapped_char_count; ///< Number of elements in mapped_chars.
};

/** The sole purpose of this struct is to sort an array of TempWordInfo by using the quick_sort template function. */
struct TempWordPointer
{
	BOOL operator==(const TempWordPointer &other) const { return !HasDiff(other.word_info) == 0; }
	BOOL operator!=(const TempWordPointer &other) const { return HasDiff(other.word_info) != 0; }
	BOOL operator<=(const TempWordPointer &other) const { return FirstDiff(other.word_info) <= 0; }
	BOOL operator>=(const TempWordPointer &other) const { return FirstDiff(other.word_info) >= 0; }
	BOOL operator>(const TempWordPointer &other) const { return FirstDiff(other.word_info) > 0; }
	BOOL operator<(const TempWordPointer &other) const { return FirstDiff(other.word_info) < 0; }

	TempWordInfo *word_info;

private:
	int FirstDiff(TempWordInfo *other) const { return wide_strcmp(word_info->word, word_info->word_len, other->word, other->word_len); }
	BOOL HasDiff(TempWordInfo *other) const { return word_info->word_len != other->word_len || FirstDiff(other); }
};

/** This class is used for buffering up to SPC_MAX_TEMP_WORDS TempWordInfo structs before adding them to the dictionary in one
	"chunk", see description of TempWordInfo. The TempWordInfo structs themselves and their associated characters etc is all
	allocated here, the purpose is to not having to do any memory-allocations while generating words that should be added to
	the dictionary tree. Notice: prefix0 and suffix0 refers to the affix-flags defined in the .dic file, like hejsan/abc - 
	while prefix1 and suffix1 refers to affix-flags defined on individual affix-rules in the .aff file. */
class TempDictionaryParseData
{
public:
	/** Constructor
		@param encoding The dictionary encoding. 
		@param has_compound_rule_flags TRUE if COMPOUNDRULE statements are used in the affix-file. 
		@param fd The HunspellAffixFileData object with information about how to generate words, meaning of flags etc. */
	TempDictionaryParseData(DictionaryEncoding encoding, BOOL has_compound_rule_flags, HunspellAffixFileData *fd);
	~TempDictionaryParseData();
	OP_STATUS Init(); ///< Allocates the internal data-structures.
	
	/** Returns the number of bytes allocated by this object, including the object itself + allocated data-structures. */
	UINT32 GetAllocatedBytes() { return m_alloced; }
	
	/** Returns TRUE if the internal buffers seems to stand the pressure of another word, which might generate a lot of inflections,
		This might not be true anyway as some dictionaries (e.g. Hungarian) have words that generates > 100000 inflections. This
		will sadly result in that the loading of the dictionary will fail. */
	BOOL ReadyForAnotherDictionaryWord()
	{
		return tmp_word_count*2 < SPC_MAX_TEMP_WORDS 
			&& m_char_buf_ofs*2 < SPC_TEMP_WORD_CHAR_BUF_SZ 
			&& index_and_mapping_ofs*2 < SPC_TEMP_INDEX_AND_MAPPING_BUF_SZ
			&& m_compound_rule_flag_buf_ofs*2 < SPC_TEMP_COMPOUND_RULE_FLAG_BUF_SZ;
	}

	/** Only called during pass 1. Adds the stem and nothing more. 
		@param word The dictionary-encoded stem.
		@param info The parsed flags. */
	OP_STATUS SetStem(UINT8 *word, FlagPtrInfo *info) { return SetStemInternal(word, info->bit_flags, info->compound_rule_flags, info->compound_rule_flag_count, TRUE); }
	
	/** Only called during pass 2+. Adds the stem and all combinations of prefix0 + stem AND stem + suffix0. 
		@param word The dictionary-encoded stem. 
		@param flags The flags, endcoded with the flag-encoding. 
		@param reset_lists. If TRUE will all internal buffers be cleared and we will add the stem as the first word. During
		pass 3 we want the size of the chunks of words to add to be as large as possible, therefore we adds several sets of
		stem+inflections after each other before we adds them to the dictionary. In that case will reset_lists be FALSE, except
		of for the first stem in the chunk. */
	OP_STATUS SetStem(UINT8 *word, UINT8 *flags, BOOL reset_lists);
	
	/** Only called during pass 1. Adds one word that is stem + rule, rule belongs to affix and it's condition matches stem. */
	OP_STATUS AddStemInflection(OpSpellCheckerAffix *affix, OpSpellCheckerAffixRule *rule); // Only called during pass 0
	
	/** Called after stem and all combinations of prefix0 + stem AND stem + suffix0 has been added. Creates all valid combinations of:
	 *  prefix0 + stem + suffix0 
	 *  prefix1 + prefix0 + stem
	 *  prefix1 + prefix0 + stem + suffix0 
	 *  stem + suffix0 + suffix1
	 *  prefix0 + stem + suffix0 + suffix1
	 */
	OP_STATUS CreateCombines();
	
private:
	/** See description and "implementation" of SetStem(2). */
	OP_STATUS SetStemInternal(UINT8 *word, UINT16 bit_flags, UINT16 *compound_rule_flags, int compound_rule_flag_count, BOOL reset_lists);

	/** Adds one word that is a combination of prefix0 + stem (pfx_rule_stem) AND stem + suffix0 (sfx_rule_stem) */
	OP_STATUS AddStemInflectionCombineInternal(TempWordInfo *pfx_rule_stem, TempWordInfo *sfx_rule_stem) { return AllocateTempWordAndCombine(pfx_rule_stem,sfx_rule_stem) ? OpStatus::OK : SPC_ERROR(OpStatus::ERR); }
	
	/** Adds words for ALL matching rules of prefix to the word afx_rule_stem which is prefix0 + stem OR stem + suffix0 */
	OP_STATUS GenerateAffixInflectionsInternal(OpSpellCheckerAffix *affix, TempWordInfo *afx_rule_stem);
	
	/** Adds one word that is the combination of prefix1 + prefix0 + stem (pfx_pfx_stem) AND stem + suffix0 (sfx_stem) */
	OP_STATUS AddCombineSuffixInflectionCombineInternal(TempWordInfo *pfx_pfx_stem, TempWordInfo *sfx_stem) { return AllocateTempWordAndCombine(pfx_pfx_stem,sfx_stem) ? OpStatus::OK : SPC_ERROR(OpStatus::ERR); }
	
	/** Adds one word that is the combination of stem + suffix0 + suffix1 (sfx_sfx_stem) AND prefix0 + stem (pfx_stem) */
	OP_STATUS AddCombinePrefixInflectionCombineInternal(TempWordInfo *sfx_sfx_stem, TempWordInfo *pfx_stem)  { return AllocateTempWordAndCombine(pfx_stem,sfx_sfx_stem) ? OpStatus::OK : SPC_ERROR(OpStatus::ERR); }
	
	/** Combines the compoundrule flags from left and right into the compoundruleflags of dst, but don't add the same flag twice. */
	OP_STATUS CombineCompoundRuleFlags(TempWordInfo *left, TempWordInfo *right, TempWordInfo *dst);
	
	/** Create a new TempWordInfo and add to the internal buffer which is stem combined with affix-rule rule that belongs to affix. 
		@param stem The "stem" which doesn't need to be the stem but could be something already combined with an affix. 
		@param affx The affix that rule belongs to.
		@param rule An affix-rule that matches stem and and should be combined with stem in order to generate a new word.
		@param affix_at_opposite TRUE if there is an affix at the "oposite" side of the word, that is the side where the new
		affix will not be appended. */
	TempWordInfo* AllocateTempWordAndCombine(TempWordInfo *stem, OpSpellCheckerAffix *affix, OpSpellCheckerAffixRule *rule, BOOL affix_at_opposite);
	
	/** Creates a new word that is a combination of a prefixed word (pfx_stem) and a suffixed word (sfx_stem). */
	TempWordInfo* AllocateTempWordAndCombine(TempWordInfo *pfx_stem, TempWordInfo *sfx_stem);

public:
	int *rule_match_buf; ///< Buffer for array of rule-indexes that matches a word for the corresponding affix
	UINT32 *affix_char_buf; ///< Buffer for converting (the necessary part of) a word to affix-char-substitution numbers.
	UINT32 *affix_match_bit_buf; ///< Bit-buffer for OpSpellCheckerAffix::GetMatchingRules().
	TempWordInfo *stem_word_info; // TempWordInfo for the stem
	UINT32 *stem; ///< TempWordInfo::word for the stem.
	int stem_len; ///< Number of characters in stem.
	TempWordInfo *tmp_words; ///< The buffer of TempWordInfos.
#ifdef SPC_SORT_PARTIALLY_FOR_COMPRESSION
	UINT8 *sort_buf; ///< Sort-buffer as required by quick_sort, for sorting word_pointers.
#endif // SPC_SORT_PARTIALLY_FOR_COMPRESSION
	int tmp_word_count; ///< Number of words currently added in tmp_words.
	TempWordInfo **valid_words; ///< Array of pointers into tmp_words that should be inserted in the dictionary, this excludes e.g. stems with the NEEDAFFIX flag.
	int valid_word_count; ///< Number of elements in valid_words.
	IndexAndMapping *index_and_mapping_buf; ///< Buffer for creating IndexAndMapping structs when converting scUTF-32 words to indexes.
	int index_and_mapping_ofs; ///< First free index in index_and_mapping_ofs.
	UINT32 m_alloced; ///< Number of bytes allocated, see GetAllocatedBytes().

private:
	DictionaryEncoding m_encoding; ///< The dictionary-encoding.
	BOOL m_8bit_enc; ///< TRUE if the dictionary-encoding is a 8-bit encoding.
	BOOL m_has_compound_rule_flags; ///< If there are COMPOUNDRULE statements in the affix-file, as given as argument to the constructor.
	HunspellAffixFileData *m_fd; ///< The associated HunspellAffixFileData object.
	TempWordInfo *m_next_to_alloc;
	UINT32 *m_char_buf; ///< Buffer for allocating chars for the TempWordInfo structs.
	int m_char_buf_ofs; ///< First free index into m_char_buf.
	UINT16 *m_compound_rule_flag_buf; ///< Buffer for the compoundrule flag type ID's referenced from TempWordInfo::info.compound_rule_flags.
	int m_compound_rule_flag_buf_ofs; ///< First free element in m_compound_rule_flag_buf.
	int *m_compound_rule_lookup; ///< Buffer used when combining compoundrule flags for two words, to not duplicate any words.
	int m_compound_lookup_times; ///< Increased by one for each time compoundrule flags for two words are combined, to use together with m_compound_rule_lookup.
	
	int m_stem_prefix_combine_count; ///< Number of prefix0 + stem that could also have suffixes
	int m_stem_suffix_combine_count; ///< number of stem + suffix0 that could also have prefixes
	int m_generated_inflection_count; ///< Number of words generated by the last call to GeneratePrefixInflectionsInternal/GenerateSuffixInflectionsInternal
	int m_stem_prefixes_with_affixes_count; ///< Number of all prefix0 + stem that have affixes
	int m_stem_suffixes_with_affixes_count; ///< Number of all stem + suffix0 that have affixes
	TempWordInfo **m_stem_prefix_combines; ///< Pointers into tmp_words for prefix0 + stem that could also have suffixes
	TempWordInfo **m_stem_suffix_combines; ///< Pointers into tmp_words for stem + suffix0 that could also have prefixes
	TempWordInfo **m_generated_inflections; ///< Pointers into tmp_words for the words generated by the last call to GeneratePrefixInflectionsInternal/GenerateSuffixInflectionsInternal
	TempWordInfo **m_stem_prefixes_with_affixes; ///< Pointers into tmp_words for all prefix0 + stem that have affixes
	TempWordInfo **m_stem_suffixes_with_affixes; ///< Pointers into tmp_words for all stem + suffix0 that have affixes
};

/** An object of this class is created for each word (stem) in the .dic file. It contains offsets to the start of the word
	and to the start of the flags. The structure of this class is somewhat "compressed" for keeping it down to 4 bytes on
	a 32-bit system. Hence is the word saved as an offset from the previous word and the flags are saved as an offset from
	the word. In OpSpellChecker::DictionaryParsePass1() is the flags converted to an internal format however, where also 
	matching affix-rule indexes might be cached. It might not be enough space for simply overwriting the old encoded flags.
	Therefore we might replace the offset to the flags (m_flags) and the space for the encoded flags with a pointer to this
	data - in a way that is a bit tricky to explain, check the code... */
class HunspellDictionaryFileLine
{
public:
	HunspellDictionaryFileLine() { op_memset(this,0,sizeof(*this)); }
	
	/**
		@param prev_word The address to the previous word (not it's flags but the word itself).
		@param word	The word.
		@param flags The flags.
	*/
	HunspellDictionaryFileLine(UINT8 *prev_word, UINT8 *word, UINT8 *flags);
	~HunspellDictionaryFileLine() {}

	/** Returns the word, given the prvious word. */
	UINT8 *GetWord(UINT8 *prev_word) { return prev_word + m_word_ofs; }
	
	/** Returns the flags, given the word. */
	UINT8 *GetFlags(UINT8 *word);
	
	/** Transform the representation of the flags from an offset from word into a pointer (flags), 
		see description of this class. */
	void SetNewFlagsPtr(UINT8 *word, UINT8 *flags);

private:
	unsigned int m_word_ofs/*:10*/; ///< Offset to the word from the previous word (or zero, it it's the first word). Comments can span several lines and really throw this off!
	unsigned int m_is_pointer:1; ///< Set if the reference to the flags is stored as a pointer instead of an offset, see description of this class.
#ifdef SIXTY_FOUR_BIT
	unsigned int padding:21;
	UINT8 *m_flags; ///< For 64-bit systems we simply always store the reference to the flags as a pointer.
#else
	/** Offset to flags from the word, or higher 16 bits of flag pointer + possibly (if there is space) offset from 
		word to the space where the encoded flags where, where the lower 16 bits of the pointer will now resist. */
	unsigned int m_flags:21; 
#endif //SIXTY_FOUR_BIT
};

class DictionaryChunks;

/** This class holds a <= 128kb large portion of the .dic file. The idea behind "chunking up" the .dic file like this is
	that during dircionary pass 3 when we are adding words to the dictionary-tree in memory - we can deallocate chunks as
	we are finished using them - this decreases the peak memory usage. */
class SingleDictionaryChunk
{
public:
	SingleDictionaryChunk() { op_memset(this, 0, sizeof(*this)); }
	
	/** Set this chunk to include the portion data from the .dic file with length len. */
	SingleDictionaryChunk(UINT8 *data, int len) { op_memset(this, 0, sizeof(*this)); m_dic_data = data; m_dic_len = len; }
	~SingleDictionaryChunk() { }
	
	/** Frees the data in the chunk. */
	void FreeChunkData() { OP_DELETEA(m_dic_data); OP_DELETEA(m_last_word); op_memset(this, 0, sizeof(*this)); }
	
	/** Creates HunspellDictionaryFileLine objects of the lines in thechunk and adds them to line_dst. */
	OP_STATUS Tokenize(ExpandingBuffer<HunspellDictionaryFileLine> *line_dst);
	
	/** Returns the first line in the chunk, as was given as argument to SetFirstLineInChunk(). */
	HunspellDictionaryFileLine *GetFirstLineInChunk() { return m_lines; }
	
	/** Returns the number of lines in this chunk (number of HunspellDictionaryFileLine objects created). */
	int GetLinesInChunk() { return m_line_count; }
	
	/** If the chunk-data isn't '\0' terminated, then will a copy of that line be allocated separatly, and this is that copy. */
	UINT8 *PossibleLastWord() { return m_last_word; }
	
	/** Returns the address to the chunk-data as given as argument to the constructor. */
	UINT8 *FileData() { return m_dic_data; }

protected:
	/** Sets the first line in the chunk to line, this is handled by the class DictionaryChunks. */
	void SetFirstLineInChunk(HunspellDictionaryFileLine *line) { m_lines = line; }

private:
	OP_STATUS CopyLastDictionaryWord(UINT8 *&word_start, UINT8 *&word_end);

private:
	HunspellDictionaryFileLine *m_lines; ///< Pointer to the first HunspellDictionaryFileLine as given as argument to SetFirstLineInChunk.
	int m_line_count; ///< Number of HunspellDictionaryFileLines in this chunk.
	UINT8 *m_dic_data; ///< The chunk-data given as argument in the constructor.
	int m_dic_len; ///< Number of bytes in the chunk-data as given as argument in the constructor.
	UINT8 *m_last_word; ///< See PossibleLastWord().

	friend class DictionaryChunks;
};

/** This is a collection of all dictionary chunks as well as all HunspellDictionaryFileLine objects. This class is also
	responsible for opening+reading the .dic and .oow files. */
class DictionaryChunks
{
public:
	/** Constructor
		@param dictionary_path The absolute path to the .dic file.
		@param own_path The absolute path to the .oow file containing words that are added/removed by the user.	*/
	DictionaryChunks(uni_char *dictionary_path, uni_char *own_path);
	~DictionaryChunks();
	
	/** Read the .oow file and the dictionary into chunks - make the dictionary data ready for use. The added words in the 
		.oow file will get an own chunk, the removed words will be made available for reading directly.
		@param time_out Deadline when this function should return (to the message loop).
		@param finished Will be set to TRUE if the dictionary-data was fully loaded, or FALSE if the function timed out.
		@return Error status code. */
	OP_STATUS ReadDictionaryChunks(double time_out, BOOL &finished);
	
	/** Returns the number of dictionary-chunks. */
	int GetChunkCount() { return m_chunks.GetSize(); }
	
	/** Returns the chunk at index. */
	SingleDictionaryChunk *GetChunkAt(int index) { return m_chunks.GetElementPtr(index); }
	
	/** Returns the HunspellDictionaryFileLine at index, among all lines in the chunks. */
	HunspellDictionaryFileLine *LineAt(int index) { return m_lines.GetElementPtr(index); }
	
	/** Returns the total number of HusnpellDictionaryFileLines. */
	int GetLineCount() { return m_lines.GetSize(); }
	
	/** Returns a pointer to the data read from the .oow file that contains the list of removed words. */
	UINT8 *GetRemoveData() { return m_remove_data; } 
	
	/** Reuturns the number of bytes in the data returned by GetRemoveData(). */
	int GetRemoveDataLen() { return m_remove_data_len; }

private:
	OP_STATUS ReadOwnData();

private:
	ExpandingBuffer<SingleDictionaryChunk> m_chunks; ///< The chunks.
	ExpandingBuffer<HunspellDictionaryFileLine> m_lines; ///< The list of all lines (from all chunks).
	UINT8 *m_remove_data; ///< Pointer to the remove data (see GetRemoveData()).
	int m_remove_data_len; ///< Number of bytes in m_remove_data.
	UINT8 *m_insert_data; ///< Pointer to the data in the .oow file to the list of words added by the user.
	int m_insert_data_len; ///< Number of bytes in m_insert_data
	int m_approx_insert_count; ///< "Approximation" of how many words that should be added from the .oow file, for calculating the initial size of m_lines.
	uni_char *m_dictionary_path; ///< The absolute path to the .dic file.
	uni_char *m_own_path; ///< The absolute path to the .oow file.
	int m_dic_pos; ///< Byte offset into the .dic file from where the next chunk should be read from.
	BOOL m_own_data_finished; ///< TRUE after we've read the data in the .oow file.
};

/** The idea (I think) behind this class is that it should contain the data that is needed only during the parsing of the 
	dictionary - that isn't needed during the affix-file parsing, and that can be discared after the loading has finished. */
class HunspellDictionaryFileData
{
public:
	/** Constructor
		@param dic_chunks The dicionary chunks that has already been created.
		@param spell_checker The OpSpellChecker that uses this object. 
		@param affix_fd The HunspellAffixFileData for the affix-file. */
	HunspellDictionaryFileData(DictionaryChunks *dic_chunks, OpSpellChecker *spell_checker, HunspellAffixFileData *affix_fd);
	~HunspellDictionaryFileData() {}
	OP_STATUS Init() { return m_tmp_data.Init(); }
	
	/** Reuturns the TempDictionaryParseData object that contained in this class. */
	TempDictionaryParseData *GetTempData() { return &m_tmp_data; }
	
	/** Returns the allocator for allocations that is needed during dictionary-parsing, but can be discared when loading has finished. */
	OpSpellCheckerAllocator *Allocator() { return &m_allocator; }
	
	/** Returns the dictionary chunks given as argument to the constructor. */
	DictionaryChunks *GetDictionaryChunks() { return m_dic_chunks; }

private:
	OpSpellChecker *m_spell_checker; ///< The OpSpellChecker instance using this object.
	OpSpellCheckerAllocator m_allocator; ///< See Allocator().
	TempDictionaryParseData m_tmp_data; ///< The buffers for creating temporary chunks of TempWordInfo objects.
	DictionaryChunks *m_dic_chunks; ///< The dictionary chunks given as argument to the constructor.
};

/** This class is similar to HunspellAffixFileParseState, but this is the parsing state of the .dic file instead of the
	.aff file. A bit simpler however as each and every line in the .dic file must be iterated in each "dictionary parsing pass". 
	Currently there are 4 parsing passes necessary for building up the dictionary in memory. */
class HunspellDictionaryParseState
{
public:
	/** Constructor
		@param dic_file_data The HunspellDictionaryFileData associated with this parsing state.
		@param pass_count The number off passes for parsing the dictionary, currently will this be 4. */
	HunspellDictionaryParseState(HunspellDictionaryFileData *dic_file_data, int pass_count)
	{
		m_pass_count = pass_count;
		m_dic_file_data = dic_file_data;
		m_pass = -1;
		NextPass();
	}
	~HunspellDictionaryParseState() {}

	/** Returns the current line index, corresponding to the HunspellDictionaryFileLines in DictionaryChunks. */
	int GetCurrentLineIndex() { return m_current; }
	
	/** Set the flags for the current HunspellDictionaryFileLine to the pointer new_flags, see description of HunspellDictionaryFileLine. */
	void SetNewCurrentFlags(UINT8 *new_flags) { m_current_line->SetNewFlagsPtr(m_word, new_flags); }
	
	/** Returns FALSE if we've reached the last line. */
	BOOL HasMoreLines() { return m_current_in_chunk_idx < m_current_chunk->GetLinesInChunk() || m_current_chunk_idx < m_dic_file_data->GetDictionaryChunks()->GetChunkCount()-1; }
	
	/** Iterate to the next line, returns FALSE if there are no more lines. */
	BOOL NextLine();
	
	/** Start over with the next pass, and set the current line to the first. */
	void NextPass();
	
	/** Returns the current pass, starting from zero. */
	int GetPass() { return m_pass; }
	
	/** Returns the current dictionary word. */
	UINT8 *GetCurrentWord() { return m_word; }
	
	/** Returns the dictionary flags for the current word, or to the flags in the internal format if SetNewCurrentFlags() has been used. */
	UINT8 *GetCurrentFlags() { return m_current_line->GetFlags(m_word); }

private:
	int m_pass; ///< The current pass.
	int m_pass_count; ///< Total number of passes, as given as argument to the constructor.
	int m_current; ///< Current line index among all lines.
	int m_current_chunk_idx; ///< Index of current dictionary chunk.
	int m_current_in_chunk_idx; ///< index among the lines in the current dictionary chunk.
	SingleDictionaryChunk *m_current_chunk; ///< The current dictionary chunk.
	HunspellDictionaryFileData *m_dic_file_data; ///< The associated HunspellDictionaryFileData.
	UINT8 *m_word; ///< Pointer to the word in the current line.
	HunspellDictionaryFileLine *m_current_line; ///< The current line.
};

/** The radix-tree consists of OpSpellCheckerRadix objects which in turn consists of an array of RadixEntry objects.
	An entry in the radix (RadixEntry) can be either A pointer to a child OpSpellCheckerRadix, a pointer to a chain of
	"word-units" which contains words in the dictionary (the part of the word that is not implicity specified by the position
	in the radix-tree), a counter for counting how many words that should be placed somewhere under the entry, or a
	NULL entry - for "dead ends" in the radix-tree. */
class RadixEntry
{
public:
	RadixEntry(OpSpellCheckerRadix *radix) { SetRadix(radix); }

	/** Returns TRUE if this entry is a pointer to a child OpSpellCheckerRadix. */
	BOOL IsRadixPtr() { return (m_entry & ~SPC_MAX_RADIX_COUNTER) && m_entry & 1; }
	
	/** Returns TRUE if this entry is a pointer to a linked list of "word-units" for storing the words. */
	BOOL IsBitPtr() { return (m_entry & ~SPC_MAX_RADIX_COUNTER) && !(m_entry & 1); }
	
	/** Returns TRUE if this entry is a counter for how many words that should be placed under this entry - or a NULL entry. */
	BOOL IsCounter() { return !(m_entry & ~SPC_MAX_RADIX_COUNTER); }
	
	/** The same as IsCounter().  */
	BOOL IsNoPointer() { return !(m_entry & ~SPC_MAX_RADIX_COUNTER); }
	
	/** Returns a pointer to the child OpSpellCheckerRadix for this entry. (IsRadixPtr() must be TRUE) */
	OpSpellCheckerRadix *GetRadix() { OP_ASSERT(IsRadixPtr()); return reinterpret_cast<OpSpellCheckerRadix*>(m_entry & ~1); }
	
	/** Returns The a pointer to the first "word-unit" in the linked list of "word-units" under this entry. (IsBitPtr() must be TRUE) */
	UINT8 *GetBits() { OP_ASSERT(IsBitPtr()); return reinterpret_cast<UINT8*>(m_entry); }
	
	/** Returns the current count of how many words that should be under this entry. (IsCounter() must be TRUE) */
	int GetCount() { OP_ASSERT(IsCounter()); return static_cast<int>(m_entry); }
	
	/** Increases the counter. (IsCounter() must be TRUE) */
	void IncCount() { OP_ASSERT(IsCounter()); m_entry++; }
	
	/** Returns the entry directly as a SPC_UINT_PTR. */
	SPC_UINT_PTR Entry() { return m_entry; }

	/** Set this entry to point at a child OpSpellCheckerRadix. */
	void SetRadix(OpSpellCheckerRadix *radix) { m_entry = reinterpret_cast<SPC_UINT_PTR>(radix) | 1; }
	
	/** Set this entry to point at the first "word-unit" in a linked lists of "word-units" containing words. */
	void SetBits(UINT8 *bits) { m_entry = reinterpret_cast<SPC_UINT_PTR>(bits); }
	
	/** Set the counter to count. (IsCounter() should have been TRUE before)*/
	void SetCount(int count) { m_entry = static_cast<SPC_UINT_PTR>(count); }

	/** Returns the parent of this entry, that is - the OpSpellCheckerRadix which has this entry in it's entry-array. */
	OpSpellCheckerRadix *GetParent();
	
	/** Returns the index of this entry in the parents entry-array. */
	int GetIndex();
	
	/** Outputs an array of indexes into buf, describing how to reach this entry in the radix-tree. 
		@param buf The buffer where the resulting indexes will be written to.
		@param len The number of indexes. 
		@param self_entry If self_entry == TRUE, this means that the entry is a "self entry" which is the first entry in
		the entry's parent OpSpellCheckerRadix's entry array which points at the "word-unit" list for words with indexes that
		are equal to the indexes for the path heading up to the OpSpellCheckerRadix. In case of a self-entry will buf not
		contain the last index (0) and len will be equal to the length of all words located in the "word-unit" list of the
		self entry. */
	void ConvertToIndexes(UINT32 *buf, int &len, BOOL &self_entry);

private:
	/** The actual entry, which is just a pointer, the kind of entry depens of the value of this pointer:
		1. If m_entry <= SPC_MAX_RADIX_COUNTER -> this entry is a counter (or NULL entry if the counter is 0). 
		2. Else if (m_entry & 1) != 0 -> this entry is a pointer to a child OpSpellCheckerRadix.
		3. Else (m_entry is an odd number) -> this entry is a pointer to a linked list of word-units. */
	SPC_UINT_PTR m_entry;
};

/** This is the class for the entry-arrays in the radix-tree. This objects should not be created directly but instead by
	using OpSpellChecker::CreateRadix(). What's really created by the OpSpellChecker::CreateRadix() is an array of values
	with this structure: 
	|OpSpellCheckerRadix *parent|SPC_RADIX_START_MARKER|OpSpellCheckerRadix::m_self_entry|rest of the RadixEntry objects|. */
class OpSpellCheckerRadix
{
public:
	/** Returns a pointer to the RadixEntry at index i, where the self-entry is at index 0. */
	RadixEntry *AtIndex(int i) { return &m_self_entry + i; }
	
	/** Return the leaf radix-entry for the len indexes in idx_str, with leaf means that the returned entry will not be
		a pointer to an OpSpellCheckerRadix. 
		@param idx_str The array of indexes to find the leaf-entry for.
		@param len The number of indexes in idx_str.
		@param level The level in the radix tree where the returned RadixEntry is located is written here, starting from zero
		for entries in this radix itself. If len == level does this mean that the RadixEntry returned is a self-entry. 
		@param radix The parent radix of the entry will be written here. */
	RadixEntry *GetRadixEntry(UINT32 *idx_str, int len, int &level, OpSpellCheckerRadix *&radix);
	
	/** Resets all radix-entries that are counters in this OpSpellCheckerRadix and in all descending OpSpellCheckerRadix
		objects. of this.
		@param entries The number of entries in all OpSpellCheckerRadixes, including the self-entry. */
	OP_STATUS ClearCounters(int entries); // entries == total entries in radix INCLUDING "self entry"
	
	/** Returns the parent OpSpellCheckerRadix for this OpSpellCheckerRadix (NULL if this is the root-radix). */
	OpSpellCheckerRadix *Parent() { return *((OpSpellCheckerRadix**)(&((&m_self_entry)[-2]))); }
	
	/** Returns the first index of the child in the entry-array for this OpSpellCheckerRadix that has the same value of 
		RadixEntry::Entry as child. Notice: will crash if there is no such entry. */
	int GetIndexOf(RadixEntry *child);
	
	/** Returns the index into the entry-array for the entry which points at child. Notice: will crash if there is no such entry. */
	int GetIndexOf(OpSpellCheckerRadix *child);

private:
	/** Here is the self-entry, but OpSpellChecker::CreateRadix() has allocated as many RadixEntry objects as are needed to 
		represent all indexes which characters are mapped to, and before m_self_entry in memory will a o pointer to the 
		parent OpSpellCheckerRadix as well as SPC_RADIX_START_MARKER be located, see description of this class. */
	RadixEntry m_self_entry;
};

/** This struct is a construction for avoiding using OpSpellCheckerRadix::GetRadixEntry for obtaining a RadixEntry if this
	turns out to be the same entry that was obtained last time (and therefore is pointed out here). The idea is that when the
	radix-tree is processed in the dictionary passes, words will be handled in more or less alphabetical order. This implies
	that it's the most usual case that a word should be located under the same radix-entry as the previous word. This is 
	carried out by OpSpellChecker::GetRadixEntry. */
struct RadixEntryCache
{
	RadixEntry *entry; ///< The RadixEntry obtained last time.
	BOOL valid; ///< Is the cache valid (will be set to FALSE by OpSpellChecker::CreateRadix()).
	int level; ///< The number of valid indexes in indexes.
	OpSpellCheckerRadix *parent; ///< The parent to entry.
	UINT32 indexes[SPC_RADIX_CACHE_INDEXES]; ///< The cached indexes.
};

/** This struct keeps a position (byte+bit) in a "word-unit" in a linked list of word-unit as found under a "leaf" RadixEntry. This 
	struct is used during dictionary lookup and the OpSpellCheckerBitStream bit_stream points at the location for the index-mappings
	after the compressed indexes. It's assumed that the indexes will match the indexes that where subject to lookup, see
	OpSpellChecker::GetIndexEntry(). The usage of this struct is iterating though all word in the dictionary that have the same
	indexes. */
struct OpSpellCheckerDictionaryPosition
{
	/** Sets the content of this struct, see description of the individual members. */
	void Set(int same_len, UINT8 *next_unit, BOOL self_entry, OpSpellCheckerBitStream bit_stream)
	{
		this->same_len = same_len;
		this->next_unit = next_unit;
		this->self_entry = self_entry;
		this->bit_stream = bit_stream;
	}

	int same_len; ///< How many indexes (in the beginning of the words) that are shared among all words in the word-unit list.
	UINT8 *next_unit; ///< Pointer to the next word-unit, or NULL if we're already pointing into the last.
	BOOL self_entry; ///< TRUE if we're in the word-unit list for a "self entry".
	OpSpellCheckerBitStream bit_stream; ///< Pointer into the word-unit to the current position (byte+bit).
};

/** This is a struct for simply keeping the keeps the flags bit-flags ("ON/OFF", like COMPOUNDEND) and the compoundrule flags
	for a word. It's used during lookup of compound words (see DictionaryPositionTree). */
struct DictionaryWordData
{
	DictionaryWordData() {} // FIXME: Shouldn't be needed but we handle arrays of these
	DictionaryWordData(UINT16 bit_flags, UINT16 *compound_rule_flags, int compound_rule_flag_count)
	{
		this->bit_flags = bit_flags;
		this->compound_rule_flags = compound_rule_flags;
		this->compound_rule_flag_count = compound_rule_flag_count;
	}
	
	UINT16 bit_flags; ///< The bit-flags stored along with each word in the "word-units" in the radix-tree.
	UINT16 *compound_rule_flags; ///< Type IDs for the compoundrule flags for this word.
	int compound_rule_flag_count; ///< Number of elements in compound_rule_flags.
};

/** This struct represents a node in the tree that is created during the compound word lookup. When a word we're 
	looking up can't be found in the dictionary - we're breaking down the word in different combinations in order to
	try to find a valid combination - and this becomes a tree. For example, if the first 2 letters of the full word 
	exists in the dictionary, then we add a child A to the root-node with len==2. But, say that the first 4 letters
	of the full word is an existing word as well - then we'll add another child B to the root-node with len==4.
	Then, let's say that the remaining letter, for example 7, after the first 2 letters is a valid word - then we'll
	add a child Aa to A with len==7. But futhermore, the 3 first letters after the first 2 in the full word is also
	a valid word, then we add a child Ab to a with len==3. Then mighth the 2 first letter after the 5 first letters 
	be a valid word as well, then we add child to Ab...etc, etc, etc, etc... The point is - we're getting a tree. 
	Each path in the tree represents a way of splitting the indexes in the orignal word to get a list of sub-words
	that have the same indexes. However this isn't enough as the same index can be mapped to different characters.
	And also, even though the characters are the same, it might be an invalid compound (due to compoundrule flags,
	and flags like COMPOUNDEND, etc). 
	Therefore, we use DictionaryPositionTree::dic_pos for iterating through words with the same indexes and save those
	nodes in the tree which have a matching characters. We also needs to find a combination of words that is a valid
	compound, and since we might find several words with the same spelling - we store compoundrule flags, etc, for each
	such word in DictionaryPositionTree::word_data.	*/
struct DictionaryPositionTree : public Link, public Head
{
	DictionaryPositionTree() : Link(), Head() {}
	DictionaryPositionTree(int len, const OpSpellCheckerDictionaryPosition& dic_pos, OpSpellCheckerAllocator *allocator)
		: Link(), Head(), dic_pos(dic_pos), word_data(2, allocator)
	{
		this->len = len;
	}
	void Init(int len, const OpSpellCheckerDictionaryPosition& dic_pos, OpSpellCheckerAllocator *allocator)
	{
		this->len = len;
		this->dic_pos = dic_pos;
		word_data.Init(2, allocator);
	}

	int len; ///< The length of the indexes.
	OpSpellCheckerDictionaryPosition dic_pos; ///< See description of this class.
	ExpandingBuffer<DictionaryWordData> word_data; ///< See description of this class.
};

/** This struct is "pointer" to a word in the dictionary, it contains an RadixEntry which points at the linked 
	list of "word-units" where the word resists - and also an index into that list where the word is. The purpose
	if the struct is to use it as a replacement-suggestion, and because we want to be able to sort candidates for
	being replacement suggestion do we also keep an "edit distance" here. Furthermore, for generating replacement
	suggestions that are compounds (not implemented...yet...), we keep a pointer (byte+bit) to the compoundrule flag,
	and we also keep the "bit-flags", for checking COMPOUNDEND, etc. This is to be able to rapidly check a set of
	WordReplacement structs (using OpSpellChecker::IsValidCompoundWordReplacement) to see if it's a valid compound.
	This information could be found using the RadixEntry and index as well, but that would be slow for many checks. */
struct WordReplacement
{
	/** Operators for sorting algorithms... */
	BOOL operator==(const WordReplacement &other) const { return this->edit_dist == other.edit_dist; }
	BOOL operator!=(const WordReplacement &other) const { return this->edit_dist != other.edit_dist; }
	BOOL operator>=(const WordReplacement &other) const { return this->edit_dist >= other.edit_dist; }
	BOOL operator<=(const WordReplacement &other) const { return this->edit_dist <= other.edit_dist; }
	BOOL operator>(const WordReplacement &other) const { return this->edit_dist > other.edit_dist; }
	BOOL operator<(const WordReplacement &other) const { return this->edit_dist < other.edit_dist; }

	RadixEntry *bit_entry; ///< The radix-entry that points at the "word-unit" list where the word resists.
	unsigned int in_bits_index : 12; ///< The index into the list pointed out by bit_entry.
	unsigned int edit_dist : 4; ///< The "edit distance" from the misspelled word.
	unsigned int bit_flags : SPC_RADIX_FLAGS; ///< The "bit-flags" for the word. Notice: The number of bits for this field MUST be at least SPC_RADIX_FLAGS!!!
	unsigned int compressed_compoundrule_bit : 3; ///< The start-bit of the compoundrule flags, used together with compressed_compound_rule_byte.
	unsigned int unused_bits : 7; ///< Bits left to use in this struct...
	UINT8 *compressed_compound_rule_byte; ///< Address of the compressed compoundrule flags, used together with compressed_compoundrule_bit.
};

/** This struct is simply a set of WordReplacement structs which should represent a replacement-suggestion which
	is a compound of the individual words. */
struct CompoundWordReplacement
{
	CompoundWordReplacement() {}
	
	/** Constuctor for making a single WordReplacement to a CompoundWordReplacement with one word. */
	CompoundWordReplacement(WordReplacement *word) { Set(word, 1, word->edit_dist); }
	void Set(WordReplacement *words, int count, int edit_dist) { this->words = words; this->count = count; this->edit_dist = edit_dist; }

	/** Operators for sorting algorithms... */
	BOOL operator==(const CompoundWordReplacement &other) const { return this->edit_dist == other.edit_dist; }
	BOOL operator!=(const CompoundWordReplacement &other) const { return this->edit_dist != other.edit_dist; }
	BOOL operator>=(const CompoundWordReplacement &other) const { return this->edit_dist >= other.edit_dist; }
	BOOL operator<=(const CompoundWordReplacement &other) const { return this->edit_dist <= other.edit_dist; }
	BOOL operator>(const CompoundWordReplacement &other) const { return this->edit_dist > other.edit_dist; }
	BOOL operator<(const CompoundWordReplacement &other) const { return this->edit_dist < other.edit_dist; }

	WordReplacement *words; ///< Pointer to array of WordReplacement structs that this compound word consists of.
	int count; ///< Number of elements in words.
	int edit_dist; ///< "Edit distance" for the compound word.
};

/** This is an iterator for iterating through each and every word stored in memory. It's being used during the seach
	for replacement-suggestions. */
class ReplacementWordIterator
{
public:	
	ReplacementWordIterator(OpSpellChecker *spell_checker) { op_memset(this,0,sizeof(*this)); m_spell_checker = spell_checker; };
	~ReplacementWordIterator() { OP_DELETEA(m_radixes); OP_DELETEA(m_radix_indexes); OP_DELETEA(m_finished_indexes); }
	OP_STATUS Init();
	
	/** Sets the start-point in the radix-tree where the iteration will start from. The idea is that start_indexes
		should be the indexes for a misspelled word. Example: If the misspelled word is "blabla", then we might
		find a "word-unit" list in the radix-tree for all words starting with the indexes for "bla". We iterate though
		that list, and next - we iterate through all words in the dictionary starting with the indexes for "bl",
		except of the ones starting with "bla" which we alread have iterated. Then we iterate through all words
		starting with "b", except of the ones starting with "bl", etc. 
		This makes the replacement-algorithm at least iterate through all words that have the same beginning as the
		misspelled word - if the operation times out before the full dictionary has been iterated. 
		@param start_indexes The indexes to use as the "start-point" in the dictionary, see above. 
		@param len The number of elements in start_indexes. */
	void SetStart(UINT32 *start_indexes, int len);
	
	/** Iterate to the next word in the dictionary (see description of SetStart() for details of iteration-order). 
		@param result_pos The position in the dictionary will be written to result_pos (everything except of edit 
		distance will be filled in). 
		@param mapping_pos If != NULL will the byte+bit position of the mapping-information for the word be written here.
		@return FALSE if we've iterated all words so that Next() didn't result in any new word. */
	BOOL Next(WordReplacement &result_pos, OpSpellCheckerBitStream *mapping_pos = NULL);

	/** Returns the index-string for the last word obtained by Next(). */
	UINT32 *Indexes() { return m_radix_indexes; }
	
	/** Returns the number of elements (indexes) in Indexes(). */
	int Length() { return m_word_len; }

private:
	/** Goto next "word-unit" list, include the current position in the radix-tree if include_current==TRUE,
		returns FALSE when there are no more word-units lists to iterate. */
	BOOL GotoNextBits(BOOL include_current);

private:
	OpSpellChecker *m_spell_checker; ///< The OpSpellChecker instance using this iterator, as given as argument to the constructor.
	OpSpellCheckerRadix **m_radixes; ///< The path of radix-arrays to our current position in the radix-tree + extra indexes for the current word.
	UINT32 *m_radix_indexes; ///< The path of indexes to our current position in the radix-tree.
	UINT32 *m_finished_indexes; ///< Keeps track of where to not enter in the radix-tree as we've already been there (see description of SetStart()).
	OpSpellCheckerBitStream m_bit_pos; ///< Current bit position in the current word-unit list.
	int m_level; //< Current level in the radix-tree, starting from zero.
	UINT8 *m_next; ///< Next word-unit in the current linked list of word-units.
	BOOL m_finished; ///< TRUE when iteration has finished.
	int m_word_len; ///< Length of the current word.
	int m_in_bits_index; ///< Index of the current word in the current linked list of word-units.
	RadixEntry *m_bit_entry; ///< The radix-entry pointing at the current linked list of word-units.
};

/** This struct contains the state of the current replacement-suggestion search. The purpose with such state is that
	it should be possible to interrupt the search, go back to the message-loop, and then continue. */
struct OpSpellCheckerReplacementState
{
	enum ReplacementSearchState
	{
		NOT_STARTED = 0,
		TREE_ITERATE	///< The only implemented search state, iterates thorough the dictionary and compare Levenhstein-distances.
	};

	/** Constructor, replacements will allocate memory from allocator. */
	OpSpellCheckerReplacementState(OpSpellCheckerAllocator *allocator) : replacements(SPC_MAX_REPLACEMENTS, allocator)
	{ 
		rep_iterator = NULL;
		Reset();
	}
	~OpSpellCheckerReplacementState() { OP_DELETE(rep_iterator); }
	
	/** This function should be called after each finished replacement-search. */
	void Reset() 
	{
		replacements.Reset();
		replace_start_indexes = NULL;
		replace_start_indexes_len = 0; 
		state = NOT_STARTED;
		word_count = 0;
		word_heap = NULL;
	}
	
	/** Replacements found during the seach, should be available here after the replacement-search has finished. */
	ExpandingBuffer<CompoundWordReplacement> replacements; 
	
	UINT32 *replace_start_indexes; ///< The index-string for the misspelled word.
	int replace_start_indexes_len; ///< Number of elements (indexes) in replace_start_index.
	ReplacementWordIterator *rep_iterator; ///< The iterator used by OpSpellChecker::TreeIterateSearch().
	ReplacementSearchState state; ///< Which algorithm are we currently using, simple - as we only have one..
	UINT8 *lev_rows[3]; ///< Last three rows when doing the Levenhstein stuff.

	int word_count; ///< Number of elements in word_heap.
	WordReplacement *word_heap; ///< Heap with replace-suggesion candidates (word with LARGEST edit-distance first).
};

/** This struct keeps the state of the dictionary-lookup, and possibly also replacement-search, for the current word
	which is target for lookup. Because the replacement-search is interruptable (see OpSpellCheckerReplacementState),
	we use this state object. */
struct OpSpellCheckerLookupState
{
	OpSpellCheckerLookupState();
	void Reset(); ///< Must be called after each word-lookup (which might include replacement-suggestion search).

	/** Allocator to be used during word-lookup. Reset() will free all memory allocated except of 16kb, this is enough
		for practically all words, so in general doesn't a subsequent lookup need to allocate any memory. */
	OpSpellCheckerAllocator lookup_allocator;
	BOOL3 correct; ///< YES, NO - or MAYBE which means that we don't know yet.
	BOOL in_progress; ///< Is the lookup in still in progress, used for implementing interruptable replacement-suggestion search.
	ExpandingBuffer<uni_char*> replacements; ///< The resulting replacement suggestion after a replacement-suggestion search.
	OpSpellCheckerWordIterator *word_iterator; ///< The word-iterator containing the word to lookup / find replacement-suggestions for.
	OpSpellCheckerReplacementState rep_state; ///< The replacement-state, see description of OpSpellCheckerReplacementState.
};

/** This is the "main" class in this module. OpSpellChecker loads, keeps and manages a single dictionary, as well as
	handles lookup and replacement-suggestion search in that dictionary. */
class OpSpellChecker
{
public:
	/** Constructor. 
		@param language The OpSpellCheckerLanguage instance which uses this dictionary. */
	OpSpellChecker(OpSpellCheckerLanguage *language);
	~OpSpellChecker();
	
	/** Deletes the hash-bit-field for faster lookup, but only if the dictionary has finished loading. 
		@param toplevel_context Irrelevant... 
		@return TRUE if the memory where freed. */
	BOOL FreeCachedData(BOOL toplevel_context);

	/** Start the loading of a dictionary. This instance must not have any dictionary already loaded. 
		@param dic_path Absolute path to the .dic file.
		@param affix_path Absolute path to the .aff file.
		@param own_path Absolute path to the .oow file.
		@param time_out Maximum time until this function will return (but loading can continue later). If time_out==0.0
		does this means that this function shouldn't return until the loading has finished.
		@param finished Will be set to TRUE if the loading completed. FALSE means that we timed-out, for resuming the
		loading-process should ContinueLoadingDictionary() be used.
		@param error_str This pointer is intended for error-reporting upon failure, NOT implemented.
		@return Error status code. Will be OpStatus::ERR upon parsing-failure, or if we're not supporting the 
		dictionary for some other reason -> in general due to "inflection overflow".
		*/
	OP_STATUS LoadDictionary(uni_char *dic_path, uni_char *affix_path, uni_char *own_path, double time_out, BOOL &finished, uni_char **error_str);
	
	/** Continues loading of the dictionary. Loading must have been started by a previous call to LoadDictionary().
		@param time_out Maximum time until this function will return (but loading can continue later). If time_out==0.0
		does this means that this function shouldn't return until the loading has finished.
		@param finished Will be set to TRUE if the loading completed. FALSE means that we timed-out, for resuming the
		loading-process should ContinueLoadingDictionary() be called again.
		@param error_str This pointer is intended for error-reporting upon failure, NOT implemented.
		@return Error status code. Will be OpStatus::ERR upon parsing-failure, or if we're not supporting the 
		dictionary for some other reason -> in general due to "inflection overflow". */
	OP_STATUS ContinueLoadingDictionary(double time_out, BOOL &finished, uni_char **error_str);
	
	/** Clears all data in the dictionary - and everything else in this object. */
	void ClearDictionary();
	
	/** Returns TRUE if the dictionary is fully loaded. */
	BOOL HasDictionary() { return m_state == LOADING_FINISHED; }
	
	/** Returns FALSE for words that should be ignored totally from all "spellchecking activities", for 
		example - numbers and too long words. */
	BOOL CanSpellCheck(const uni_char *word, const uni_char* word_end);

	/** Returns FALSE if the word can't be added to the dictionary, e.g. if all character cant be represented using
		the dicionary's char-encoding. */
	BOOL CanAddWord(const uni_char *word) { return CanSpellCheck(word, word + uni_strlen(word)) && CanRepresentInDictionaryEncoding(word); }

	/** Reuturns TRUE if ch is considered to be a character that breaks words, either because it's a "default breaking
		character" like whitespaces, or because it's specified with BREAK in the affix-file. The default-breaking characters
		can also be overrided by WORDCHARS in the affix-file. Notice: CheckSpelling() has some special handling	of 
		"ambiguous" break characters like '-', which is sometimes part of a word - and sometimes used for breaking words. */
	BOOL IsWordSeparator(uni_char ch);
	
	/** The function used for check spelling of a word, and possibly - if find_replacements==TRUE - also seach for replacement
		suggestion words in case of a misspelled word.
		@param lookup_state The lookup state to use (see OpSpellCheckerLookupState and OpSpellCheckerRaplacementState). 
		@param time_out The deadline for when this function must return. This is only useful for when the word is misspelled
		and find_replacements==TRUE. If replacement_timeout==0.0 or if replacement_timeout > time_out, then might the 
		replacement search be paused and lookup_state->in_progress be set to TRUE - which implies that this function should
		be called again later to continue the replacement suggestion search. timeout_out == 0 implies deadline, 
		lookup_state->in_progress will be FALSE. However it's still possible that the replacement-search doesn't try all
		possible ways of finding replacements if replacement_timeout != 0.0. 
		@param find_replacements If TRUE and the word in lookup_state->word_iterator->GetCurrentWord() is misspelled,
		then will a replacement-suggestion search be performed.
		@param replacement_timeout The deadline when the replacement-suggestion search must finish and lookup_state->in_progress
		must be set to TRUE.
		@param override_word If override_word != NULL, then will this word be used instead of lookup_state->word_iterator->GetCurrentWord(),
		this is used internally for checking sub-words when the word contains "ambiguous" break characters (like '-'). 
		@return Error status code. */
	OP_STATUS CheckSpelling(OpSpellCheckerLookupState *lookup_state, double time_out, BOOL find_replacements, double replacement_timeout, const uni_char *override_word = NULL, const uni_char* override_word_end = NULL);
	
	/** Returns FALSE if the string str can't be represented using the dictionary's char-encoding. */
	BOOL CanRepresentInDictionaryEncoding(const uni_char *str);

	/** Adds word to the dictionary. If permanently == TRUE will the new word be written to the .oow file so that the
		change will be permanent. (But special handling if the word was in the orginal dictionary, but later removed..) */
	OP_STATUS AddWord(const uni_char *word, BOOL permanently);

	/** Removes a word from the dictionary. If permanently == TRUE will the new word be written to the .oow as a word that
		should be removed permanently. (But special handling if the word wasn't part of the original dictionary..). */
	OP_STATUS RemoveWord(const uni_char *word, BOOL permanently);

	/** Returns the number of bytes allocated by this OpSpellCheckerObject, including everything in the dictionary.
		This is the number of bytes allocated after loading has finished, the peak memory comsumption during the 
		dictionary parsing will be higher though.
		@param for_dictionary Bytes for storing the dictionary, etc, that must be in memory for the spellchecker to work.
		@param for_lookup Bytes not necessary for spellchecking to work, but is used for speed up dictionary lookup,
		i.e. The hash-bit-field and the existing-index-combination bit-field for "n-gram suggestions" (not used yet). */
	void GetAllocatedBytes(UINT32 &for_dictionary, UINT32 &for_lookup);


private:
	
	OP_STATUS CheckSharpSInternal(UINT32 *buf, int len, BOOL &is_correct);

    /** Returns pointer to the first word break in str or str_end if none. */
	const uni_char* FindWordBreak(const uni_char *str, const uni_char* str_end);
	
    /** Returns pointer to the first none word break character in str or str_end if only word breaks. */
	const uni_char* SkipWordBreaks(const uni_char *str, const uni_char* str_end);


protected:

	/** Returns the encoding of the dictionary, see description of DictionaryEncoding. */
	DictionaryEncoding GetDictionaryEncoding() { return m_encoding; }
	
	/** Returns the "type" (which is an index into m_affix_parsers) of the affix-file keyword in str (like "SFX", etc),
		if HS_INVALID_LINE is returned does this imply an unknown/unsupported keyword. */
	int GetAffixTypeFromString(UINT8 *str);
	
	/** Returns true if the affix-file contain COMPOUNDRULE statements. */
	BOOL GetHasCompoundRules() { return m_has_compound_rule_flags; }
	
	/** Returns the root OpSpellCheckerRadix of the radix-tree. */
	OpSpellCheckerRadix *GetRootRadix() { return m_root_radix; }
	
	/** Returns the number of radix-entries in the OpSpellCheckerRadixes entry-arrays. This corresponds to the number
		of indexes that words are converted to, but including index 0 which is used for "self-entries" in the radix-tree. */
	int GetRadixEntries() { return m_mappings.GetSize(); }

	/** Decodes huffman-coded indexes from bit_stream and puts the result in result_buf and the number of indexes
		decoded in result_len. */
	void ReadCompressedIndexes(OpSpellCheckerBitStream &bit_stream, UINT32 *result_buf, int &result_len);
	
	/** Reads the mapping info for a word from a OpSpellCheckerBitStream. 
		@param bit_stream The location to read the mappings in the "compressed" representation in the "word-unit" lists.
		@param The indexes for the word (necessary to interprete the data).
		@param len The number of elements (indexes) in indexes.
		@param pos_to_char The resulting "decompressed" mappings, see description of PosToChar.
		@param pos_to_char_len The number of elements written to pos_to_char. */
	void ReadMappingInfo(OpSpellCheckerBitStream &bit_stream, UINT32 *indexes, int len, PosToChar *pos_to_char, int &pos_to_char_len);
	
	/** Reads (decodes) the "bit-flags" from bit_stream. 
		@param bit_stream The OpSpellCheckerBitstream to read the huffman-coded bit-flags from.
		@return the bit-flags set bit 0 to SPC_RADIX_FLAGS-1 corresponds to this flags, see AffixFlagType. */
	UINT16 ReadBitFlags(OpSpellCheckerBitStream &bit_stream);
	
	/** Reads (decodes) the huffman-coded compoundrule flags (type ID's for those) from bit_stream. 
		@param bit_stream OpSpellCheckerBitStream to read the huffman-coded compoundrule flags from.
		@param result_buf The type ID's for compoundrule flags decoded. 
		@return The number of elements written to result_buf. */
	int ReadCompoundRuleFlags(OpSpellCheckerBitStream &bit_stream, UINT16 *result_buf);
	
	friend class HunspellDictionaryFileData;
	friend class ReplacementWordIterator;
	friend class HunspellAffixFileLine;

private:
	
	enum LoadState
	{
		INIT_STATE, ///< Loading not started.
		PARSING_AFFIXES, ///< Reading/parsing of the affix-file (.aff).
		READING_DICTIONARY, ///< Reading of the dictionary-file from disk (.dic, the .oow is also read here).
		PARSING_DICTIONARY, ///< Parsing of the dictionary, which includes build-up of the radix-tree in memory.
		LOADING_FINISHED ///< Loading has finished.
	};
	
	/** Reset all member variables to initial values. */
	void Init();
	
	/** Allocates memory structures necessary before the dictionary-loading can start at all. */
	OP_STATUS InitMemoryStructs();

	/** A "different" (non-interruptable) way of looking up a word in the dictionary - without using OpSpellCheckerLookupState. 
		This is used internally	on a couple of occations.
		@param word The word to lookup, if word_iterator == NULL. 
		@param has_word Will be set to TRUE if the word is correct.
		@find_replacements If TRUE will replacements be searched for in at most SPC_REPLACEMENT_SEARCH_MAX_DEFAULT_MS.
		@listener A listener to inform of the result if listener != NULL. 
		@param word_iterator An OpSpellCheckerWordIterator than will be used instead of word if it's not NULL. 
		@return Error status code. */
	OP_STATUS IsWordInDictionary(const uni_char *word, BOOL &has_word, BOOL find_replacements = FALSE, OpSpellCheckerListener *listener = NULL, OpSpellCheckerWordIterator *word_iterator = NULL);
	
	/** Checks if the words in str are all correct (then is_correct will be set to TRUE), they will be separated
		using IsWordSeparator(). */
	OP_STATUS IsManyWordsInDictionary(uni_char *str, BOOL &is_correct);

	/* Statistical stuff... */
	UINT32 GetRadixCounters() { return m_stat_counters; }
	UINT32 GetTotalCount() { return m_stat_count_total; }
	UINT32 GetRadixMaxCounter() { return m_stat_max_counter; }

	/** Runs a slice of affix-file parsing, see description of HunspellAffixFileData and HunspellAffixFileParseState. 
		@param fd The HunspellAffixFileData, see description of that class. 
		@param state The HunspellAffixFileParseState, see description of that class. 
		@param time_out The deadline for when we must return (to the message-loop), 0.0 means -> go on until finished.
		@param finished Will be set to TRUE if the affix-file parsing finished, otherwise we should call this function
		again to continue the parsing. */
	OP_STATUS ParseHunspellAffix(HunspellAffixFileData *fd, HunspellAffixFileParseState *state, double time_out, BOOL &finished);
	
	/** Runs a slice of the dictionary-parsing, including building up the dictionary in memory. 
		@param time_out The deadline for when we must return (to the message-loop), 0.0 means -> go on until finished.
		@param finished Will be set to TRUE if the dictionary parsing finished, otherwise we should call this function
		again to continue the parsing. */
	OP_STATUS ParseHunspellDictionary(double time_out, BOOL &finished);
	
	/** Used by RegisterAffixParsingFunctions to add a parsing-function for a certain type of affix-file statement,
		like "PFX", "MAP", etc. This functions will read the current HunspellAffixFileLine from the 
		HunspellAffixFileParseState and use it to setup the HunspellAffixFileData with more information.
		@param func The affix-line parsing function.
		@param str The keyword for which func should be used, e.g. "PFX" or "MAP", etc. */
	OP_STATUS RegisterParserFunc(OP_STATUS (OpSpellChecker::*func)(HunspellAffixFileData*,HunspellAffixFileParseState*), const char *str);
	
	/** Registers all the affix-line parsing functions using RegisterParserFunc(). */
	OP_STATUS RegisterAffixParsingFunctions();
	
	/** Convert the "replacement strings" for REP statements from characters to indexes. */
	void ConvertRepStrings();
	
	/** Returns the affix-line parser for statment-type type (type is just an index into m_affix_parsers). */
	AffixParser* GetAffixParser(int type);
	
	/** The function used by GetIndexFirstEntry() and GetIndexNextEntry(), see description of those functions below. */
	BOOL GetIndexEntry(UINT32 *indexes, int len, OpSpellCheckerDictionaryPosition &pos, BOOL continuing);
	
	/** Searches the radix-tree and word-unit lists for the first OpSpellCheckerDictionaryPosition matching indexes and len. 
		@param indexes The index-string we wants to find the first occurence of in the dictionary. 
		@param len Number of elements (indexes) in indexes. 
		@param pos The OpSpellCheckerDictionaryPosition for the first matching position, see description of
		OpSpellCheckerDictionaryPosition. Doesn't need to be initialized when making the call.
		@return FALSE if no word with the specified index-string existed. */
	BOOL GetIndexFirstEntry(UINT32 *indexes, int len, OpSpellCheckerDictionaryPosition &pos) { return GetIndexEntry(indexes, len, pos, FALSE); }

	/** Searches the radix-tree and word-unit lists for the next OpSpellCheckerDictionaryPosition after the current
		(at the time of the call) OpSpellCheckerDictionaryPosition - matching indexes and len. 
		@param indexes The index-string we wants to find the next occurence of in the dictionary. 
		@param len Number of elements (indexes) in indexes. 
		@param pos The OpSpellCheckerDictionaryPosition for the next matching position, see description of
		OpSpellCheckerDictionaryPosition. When this function is called should pos point at the bit after the last
		matching word in the dictionary, that is -> it should point at the "it-there-more-words-in-this-word-unit" bit.
		@return FALSE if no more word with the specified index-string existed. */
	BOOL GetIndexNextEntry(UINT32 *indexes, int len, OpSpellCheckerDictionaryPosition &pos) { return GetIndexEntry(indexes, len, pos, TRUE); }
	
	/** After a dictionary position tree has been created (see description of DictionaryPositionTree) is this function
		used for eliminating nodes which don't matches the mappings for the words we're looking up, or that have invalid
		compound "bit-flags" not matching their position, i.e. AFFIX_TYPE_COMPOUNDBEGIN, AFFIX_TYPE_COMPOUNDMIDDLE,
		AFFIX_TYPE_COMPOUNDEND and AFFIX_TYPE_COMPOUNDFLAG.	This function is recursive.
		@param tree The root of the tree to eliminate nodes in.
		@param indexes The indexes from which the tree is built from - excluding indexes in the beginning of the word
		we're checking that belongs to ancestors of tree. 
		@param len_left Number of elements (indexes) in indexes.
		@param real_pos The number of indexes before indexes which belongs to ancestor nodes for tree. 
		@param pos_to_char The mappings of the indexes, notice that PosToChar::pos refers to the position from the start
		of the full index-string (for the word we're looking up) - not from the start of indexes which points somewhere
		into the full index-string.
		@param pos_len Number of elements in pos_to_char. 
		@param result Returns FALSE if there tree is an invalid tree which doesn't match indexes in any path. */
	OP_STATUS HasCorrectChild(DictionaryPositionTree *tree, UINT32 *indexes, int len_left, int real_pos, PosToChar *pos_to_char, int pos_len, BOOL &result);
	
	/** Checks the compoundrule flags in dic_word_datas (of length len) and returns TRUE if it was possible to
		combine them after each other in a way that matches a COMPOUNDRULE statement in the affix-file. */
	BOOL MatchesCompoundRule(DictionaryWordData **dic_word_datas, int len);
	
	/** Used for replacement-suggestion search of compound words (not implemented yet). Returns TRUE if replacement
		represents a valid compound. */
	BOOL IsValidCompoundWordReplacement(CompoundWordReplacement *replacement);
	
	/** Checks the dictionary tree tree (see description of DictionaryPositionTree) in order to find a path in the
		tree representing a valid compound of words according to the compoundrule flags. This function is recursive.
		@param tree The DictionaryPositionTree to search for a valid compound in.
		@param dic_word_datas An array ("stack") for the current DictionaryWordDatas to check.
		@param level The current recursion-level level of the function, starts at zero. */
	BOOL HasValidCompoundRules(DictionaryPositionTree *tree, DictionaryWordData **dic_word_datas, int level);
	
	/** Checks if the scUTF-32 string str with length len is correct, that is if it's either in the dictionary
		as a single word (and don't have the ONLYINCOMPOUND set), or if it's a valid compound word. This function
		is used by CheckSpelling().
		@param str The scUTF-32 string to lookup.
		@param len the length of str.
		@param is_correct Will be set to TRUE if the word is correct. */
	OP_STATUS IsStringCorrectInternal(UINT32 *str, int len, BOOL &is_correct);
	
	/** This function is use for initially building up the DictionaryPositionTree, see description of DictionaryPositionTree.
		@param indexes The index-string that we should try to find a valid sub-string in that starts from the beginning
		of indexes, see description of DictionaryPositionTree. This function is recursive.
		@param len The number of elements (indexes) in indexes. 
		@param word_index The "index" of the words in the resulting compound that should be represented by the nodes
		created as children of result_root, starting from zero.
		@param result_root The root to which the created child-nodes should be appended. */
	OP_STATUS GetDictionaryPositionTree(UINT32 *indexes, int len, int word_index, DictionaryPositionTree *result_root);
	
	/** This function runs all the replacement-string algorithms (currently one...) in order to find replacement suggestions
		for the misspelled word in chars. The replacement-search could be interruptable, which means that this function can
		return (depending on the arguments) without being finished - and the search can continue later on a subsequent call.
		@param lookup_state The lookup-state which contains the OpSpellCheckerReplacementState defining the current state
		of the suggestion-replacement search progress, See OpSpellCheckerReplacementState for details. When the search
		finishes will this function (using WriteOutReplacementsUTF16) write the suggestions that where found to 
		lookup_state->replacements. If the search finishes (no subsequent calls should be made) will lookup_state->in_progress
		be set to FALSE.
		@param time_out The deadline when this function must return, 0.0 means that the search should continue unit it's
		finished, or unit replacement_timout has expired which implies that the search must finish even though it would be
		possible to continue the search.
		@param chars The misspelled word in scUTF-32. This argument is only used during the first call to this function
		when searching for replacements for a certain word.
		@param len The length of the string chars. 
		@param replacement_timeout Deadline for when the search must finish completely, that is, not just return to the
		message-loop waiting for another call, but write the replacements to lookup_state->replacements and set 
		lookup_state->in_progress to FALSE. A value of 0.0 means -> go on as long as you like. */
	OP_STATUS SearchReplacements(OpSpellCheckerLookupState *lookup_state, double time_out, UINT32 *chars, int len, double replacement_timeout);
	
	/** This function converts a WordReplacement struct to an scUTF-32 string.
		@param replacement The WordReplacement to convert.
		@param rep_iterator A ReplacementWordIterator that will be re-initialized, this is just for avoiding allocating
		a new iterator. 
		@param res The buffer where to write the scUTF-32 string to.
		@param len The number of chars written to res. */
	void WordReplacementToChars(WordReplacement *replacement, ReplacementWordIterator *rep_iterator, UINT32 *res, int &len);
	
	/** This function takes the information in lookup_state->replacement_state and converts that information to UTF-16
		strings which is added to lookup_state->replacements. */
	OP_STATUS WriteOutReplacementsUTF16(OpSpellCheckerLookupState *lookup_state);
	
	/** This function is used for generating replacement-suggestions. It calculates the DamerauLevenshtein distance 
		for as many single (not compound) words in the dictionary as there is time for, and when it's finished writes
		the resulting suggestions to lookup_state->replacement_state.replacements. This function is interruptable so
		that it can continue the iteration upon a subsequent call.
		@param lookup_state The lookup-state (which contains an OpSpellCheckerReplacementState, see description of that class).
		@param timeout Deadline
		@param time_out The deadline when this function must return, 0.0 means that the search should continue unit it's
		finished.
		@param must_finish If it's TRUE we must finish and generate the result during this call, we won't get any
		subsequent calls where we could continue the processing.
		@param finished Will be set to TRUE if the search finished. This will happen if this function finished iterating through 
		all words in the dictionary, or if must_finish was TRUE and the operation timed out. */
	OP_STATUS TreeIterateSearch(OpSpellCheckerLookupState *lookup_state, double timeout, BOOL must_finish, BOOL &finished);

	/** Creates a new OpSpellCheckerRadix which is a child to parent, parent should be NULL for the root radix. If NULL
		is returned does this imply OOM. */
	OpSpellCheckerRadix *CreateRadix(OpSpellCheckerRadix *parent);
	
	/** This function has the same result (and arguments) as m_root_radix->GetRadixEntry(...), but it speeds up the
		the operation a bit by using the RadixEntryCache. See description of OpSpellCheckerRadix::GetRadixEntry() and
		the description of the RadixEntryCache struct. */
	RadixEntry *GetRadixEntry(UINT32 *idx_str, int len, int &level, OpSpellCheckerRadix *&parent);
	
/********* The following functions are used for writing "compressed" words to the dictionary in memory *********/

	/** This function writes out the content of word_info->mapped_chars in the "compressed" form to bit_stream, this
		is used when writing out words in the "word-units" under the radix tree (where the word in the dictionary are
		stored)
		@param word_info The TempWordInfo for which we should write out the mapping info.
		@param bit_stream The destination bit-stream.
		@param end_write We must write the mapping info BEFORE this byte, if that fails, FALSE is returned.
		@return FALSE It there wasn't enough space, see end_write above. TRUE if the operation succeeded. */
	BOOL WriteMappingInfo(TempWordInfo *word_info, OpSpellCheckerBitStream &bit_stream, UINT8 *end_write);
	
	/** This function writes out the bit-flags in word_info->info.bit_flags as a huffman-code, and also the compoundrule
		flags stored in word_info->info.compound_rule_flags, which will be huffman-coded as well.
		@param word_info The TempWordInfo for which we should compress and write out the flags.
		@param bit_stream The destination bit-stream.
		@param end_write We must write the flags BEFORE this byte, if that fails, FALSE is returned.
		@return FALSE It there wasn't enough space, see end_write above. TRUE if the operation succeeded. */
	BOOL WriteExtraWordInfo(TempWordInfo *word_info, OpSpellCheckerBitStream &bit_stream, UINT8 *end_write);
	
	/** This function huffman-encodes and writes out indexes to the OpSpellCheckerBitStream bit_stream
		@param word_info The TempWordInfo for which we should compress and write out the flags.
		@param bit_stream The destination bit-stream.
		@param end_write We must write the indexes BEFORE this byte, if that fails, FALSE is returned.
		@return FALSE It there wasn't enough space, see end_write above. TRUE if the operation succeeded. */
	BOOL WriteCompressedIndexes(UINT32 *indexes, int len, OpSpellCheckerBitStream &bit_stream, UINT8 *end_write);

	/** This function uses the three functions above to write out a word in the dictionary to bit_stream. The indexes
		implicitly specified by the word-units location in the radix-tree is not written out. Also does this function
		check if the part of the word that should be written have the same "beginning" as prev_end (which is the previous
		word in the linked list of word-units). If so, does this function write 3 bits specifying how many (1-8) indexes
		we should copy from the previous word.
		@param word_info The TempWordInfo for the word we should write out.
		@param bit_stream The destination bit-stream.
		@param end_write We must write the mapping info BEFORE this byte, if that fails, FALSE is returned.
		@param self_entry TRUE if this word is written under a "self-entry" in the radix-tree, this implies that there are
		no indexes to write - and that the "copy-indexes-from-previous-word" bit should not be written.
		@param same_len The number of indexes in the beginning of the word that shouldn't be written out as they are
		implicated by the "word-unit" lists position in the radix-tree.
		@param prev_end The "end" (after the indexes implicated by the position in the radix-tree) of the previous word in
		this linked list of word-units.
		@param prev_end_len The number of elements (indexes) in prev_end.
		@return FALSE It there wasn't enough space, see end_write above. TRUE if the operation succeeded. */
	BOOL WriteUsingPrevWord(TempWordInfo *word_info, OpSpellCheckerBitStream &bit_stream, UINT8 *end_write, BOOL self_entry, int same_len, UINT32 *prev_end, int prev_end_len);
	
	/** This function has similar functionality as WriteUsingPrevWord(), but it doesn't use any previous word to
		compact the representation. This function is used for the first word in a linked list of word-units.
		@param word_info The TempWordInfo for the word we should write out.
		@param bit_stream The destination bit-stream.
		@param self_entry TRUE if this word is written under a "self-entry" in the radix-tree, this implies that there are
		no indexes to write - and that the "copy-indexes-from-previous-word" bit should not be written.
		@param ignore_len The number of indexes in the beginning of the word that shouldn't be written out as they are
		implicated by the "word-unit" lists position in the radix-tree.
		@param end_write We must write the mapping info BEFORE this byte, if that fails, FALSE is returned.		
		@return FALSE It there wasn't enough space, see end_write above. TRUE if the operation succeeded. */
	BOOL WriteAllOfWord(TempWordInfo *word_info, OpSpellCheckerBitStream &bit_stream, BOOL self_entry, int ignore_len, UINT8 *end_write);
	
/***************** End of functions for writing out the "compressed" word to the dictionary *****************/	
	
	/** When we're writing out a chunk of words to a linked list of word-units, it's a usual problem that not all words
		that should be in the word-units are available at the same time, especially when SPC_SORT_PARTIALLY_FOR_COMPRESSION
		isn't defined (which causes sorting of all pointers to the TempWordInfo structs in a TempDictionaryParseState). 
		So, we need to write some information at the end of the current word-unit in order to continue from there when
		more words that should be located in the same word-unit list. Because the end of the words (after the indexes
		implicated by the radix-tree position) are in general not written out completely but instead referes to the 
		previous word (with a number telling how many indexes to copy) - we need to write out the full "end" of the last
		word so that we don't need to iterate through all words to get the full last word. 
		The next-word-unit pointer in the last word-unit is set to point out where in the word-unit the info about where to
		continue is stored, this "pointer" is an offset into the word-unit divided by 2. The idea is that the value (offset/2)
		should be < 4096. This if for not confusing the offset with a pointer to a next word-unit, and to be 100% sure
		that it won't happen on any platform that memory really could be allocated so low that it might be confused with
		an offset, I've picked the ridiculously low "safe-limit" of 4096. The maximum size of a word-unit is 8096 bytes
		however, therefore is the offset divided by 2 before its stored. What is written at the place for the offset is
		the number of bits (0-15) which are free before the offset ("backward-offset" to the end of the last word written
		before the temporary full-copy of the last word). It's also written the number of words left to write in this linked
		word-unit list, and, as mentioned above, the last word previously written is written out completely (except
		of the "implicated" indexes) - by using huffman-codes as usual.
		Anyway, that's the task for this function to perform, to write out 1. the offset, 2. the "backward bit offset",
		3. number of words left, and 4. The "full" copy of the last word. BUT there is an exception, if only one word has been
		written to the linked list of word-units will the copy (4) NOT be written out, instead will the first word be
		used when we continue. Therfore we also write 5. A bit telling if we should use the first word, or if a temporary
		copy (4) has been written and should be used.
		@param data The start of the word-unit.
		@param bit_stream OpSpellCheckerBitStream pointing at the bit after the last written word.
		@param left_to_write Number of words left to write, this number comes originally from the counter in the radix
		entry, and is then decreased as new chunks of words are added.
		@param end_write We must write the mapping info BEFORE this byte, if that fails, FALSE is returned.	
		@param new_end_write If != NULL, the caller wants this function to decide the end of the current word-unit, this
		is possible if the word-unit has just been allocated using OpSpellCheckerAllocater so that the allocation can be
		"shrinked" using OpSpellCheckerAllocator::OnlyUsed(). What will be decided depends on how many words there are
		left to write - the idea is that we don't want to waste space thate will be unused, and we don't like to allocate
		more word-units either.
		@param prev_word_end The "end" of the previous word written, or NULL if we shouldn't write out the full "end" of the
		previous word, see above.
		@param prev_word_end_len The number of elements (indexes) in prev_word_end.
		@return FALSE It there wasn't enough space, see end_write above. TRUE if the operation succeeded. */
	BOOL WriteNextInfoBits(UINT8* data, OpSpellCheckerBitStream &bit_stream, int left_to_write, UINT8 *end_write, UINT8 **new_end_write, UINT32 *prev_word_end, int prev_word_end_len);
	
	/** This function creates a new word-unit and adds the words in related_words into that new word-unit. 
		@param related_word List of words, the first in_flow of this words should be in this linked list of word-units. 
		@param self_entry TRUE if this is we're in a "self-entry".
		@param in_flow See related_words.
		@param count The total number of words left that should be in this linked word-unit list, including in_flow.
		@param same_len Number of indexes implicated by the word-unit list's position in the radix-tree.
		@param data Here will the address of the new word-unit be written. 
		@param is_continuing FALSE it this should be the first word-unit in the linked list, TRUE otherwise.
		@param prev_end If self_entry==FALSE and is_continuing==TRUE should this be the "end" of the previous 
		written word.
		@param prev_end_len Number of elements (indexes) in prev_end. */
	OP_STATUS ConstructWordUnit(TempWordInfo **related_words, BOOL self_entry, int in_flow, int count, int same_len, UINT8 *&data, BOOL is_continuing = FALSE, UINT32 *prev_end = NULL, int prev_end_len = 0);
	
	/* Returns a pointer to a temporary buffer with size SPC_TEMP_BUFFER_BYTES bytes. */
	UINT8 *TempBuffer() { OP_ASSERT(!m_tmp_temp_buffer_users++); return m_tmp_buffer; }
	
	/** "Releases" the temporary buffer obtained by TempBuffer(). */
	void ReleaseTempBuffer() { m_tmp_temp_buffer_users--; }
	
	/** During affix-parsing is this function used for marking the len scUTF-32 characters chars as existing in the
		dictionary. There is special handling of characters on affixes as we don't wont to map those to the "unusual"
		characters (because they might be very usual even though they are unusual in the .dic file). */
	OP_STATUS AddExistingCharsAtAffix(UINT32 *chars, int len);
	
	/** Returns TRUE if the scUTF-32 character ch exists in the dictionar. */
	BOOL IsExistingChar(UINT32 ch);
	
	/** Setting up OpSpellChecker::m_approx_freq to hold "approximations" of how "usual" characters that exists
		in the dictionary are after dictionary parse pass 0. It's an approximation because inflections are not
		generated during pass 0, hence we only count the frequency in the .dic file. */
	OP_STATUS SetupApproxFreqenceOfExisting();
	
	/** Converts the scUTF-32 char ch to upper-case if it's a lower-case character, and to lower-case if it's an
		upper-case character and places the result in other_char. */
	OP_STATUS GetDifferentCaseChar(UINT32 ch, UINT32 &other_char);	
	
	/** Append a special mapping for "unusual" characters which have been "collected" in m_unusual_chars. */
	OP_STATUS AppendUnusualMapping();
	
	/** "Moves" the scUTF-32 character ch to m_unusual_chars and "re-maps" it to the "unusual mapping". */
	OP_STATUS MoveToUnusualMapping(UINT32 ch);
	
	/** Returns the "approximated" frequency of the character ch, the result is valid after all characters in the
		.dic file have been counted during dictionary parsing pass 0. */
	int GetApproxExistCountFor(UINT32 ch);

	/** Setup the character ch in m_char_mappings so that it points out it's mapping position, that is, which index
		(mapping_index) into m_mappings that points to the mapping where ch belongs, and which position pos in that
		mapping where ch is. */
	OP_STATUS AddToCharMappings(int ch, int mapping_index, int pos);
	
	/** Returns the MappingPointer for ch, which was previously setup by AddToCharMappings. See descriptions of
		MappingPointer and AddToCharMappings(). */
	MappingPointer GetCharMapping(int ch);
	
	/** Increase the frequencies in the associated mappings for the characters in str (with length str_len), used
		for finding out the most frequent character in each mapping later. */
	void CountMappedChars(UINT32 *str, int str_len);
	
	/** Make sure that the scUTF-32 characters ch1 and ch2 are in the same mapping, possibly by creating a new mapping. */
	OP_STATUS SetupPairMapping(int ch1, int ch2);

	OP_STATUS SetupSingleMapping(UINT32 ch);
	
	/** Convert the scUTF-32 string src to indexes and output it to dst.
		@param src The scUTF-32 string to convert.
		@param dst The buffer for writing the resulting indexes to.
		@param replace_unknown If TRUE -> Replace characters that are not mapped to any index to a (more or less)
		random index.
		@return FALSE if replace_unknown == FALSE and unknown (un-mapped) characters where encounted. */
	BOOL ConvertCharsToIndexes(UINT32 *src, UINT32 *dst, int len, BOOL replace_unknown = FALSE);
	
	/** Identical to ConvertCharsToIndexes except of that it can't handle unknown character in any way except of causing
		a crash... (to be used when it's known that all characters are mapped to indexes) */
	void ConvertCharsToIndexesFast(UINT32 *src, UINT32 *dst, int len);
	
	/** Converts the scUTF-32 string chars to indexes (by replacing the values). The mappings (for character that are not
		the most usual in their mappings) are also written to char_mapping.
		@param chars The scUTF-32 string that should be converted to indexes (by replacing the characters).
		@param len The length of chars.
		@param char_mapping Buffer where the mappings are written, see description of IndexAndMapping.
		@param char_mapping_count The number of elements in char_mapping will be written here. */
	void ConvertCharsToIndexesAndMappings(UINT32 *chars, int len, IndexAndMapping *char_mapping, int &char_mapping_count);
	
	/** Returns the PosToChar array for the scUTF-32 string chars.
		@param chars The scUTF-32 string to get the PosToChar array for.
		@param len The length of chars.
		@param pos_to_char The PosToChar array will be written here, see the description of PosToChar.
		@param pos_count Number of element written to pos_to_char. */
	void GetMappingPosToChar(UINT32 *chars, int len, PosToChar *pos_to_char, int &pos_count);
	
	/** For each "n-gram" (e.g. chars 0-3, chars 1-4, chars 2-5, etc) that exists in the index string indexes with 
		length len, set the corresponding bits in the existing-index-combination array - m_index_combinations, or
		m_short_combinations[len] for words with length less then the "combination lenght" (usually 4-5). */
	void SetIndexCombinations(UINT32 *indexes, int len);
	
	/** Allocate the m_index_combinations and m_short_combinations arrays for "n-gram" suggestions. */
	OP_STATUS SetupIndexCombinations();
	
	/** Count the combination of bit-flags and type IDs for compoundrule flags for word_info - in order to later setup
		huffman codes for this things. */
	void CountExtraInfo(TempWordInfo *word_info);

	/** Create the CharConverters for converting to and from UTF-16, also for 8-bit encoded dictionaries - setup lookup
		tables for converting between upper-case and lower-case. */
	OP_STATUS SetupCharConverters();
	
	/** Returns TRUE if the scUTF-32 string str with length len is all in upper-case. */
	BOOL IsAllUpperCase(UINT32 *str, int len);
	
	/** Returns TRUE if the scUTF-32 string str with length len is one upper-case character followed by only lower-case
		characters. */
	BOOL IsOneUpperFollowedByLower(UINT32 *str, int len);
	
	/** Converts the scUTF-32 string str with length len to lower-case. */
	void SCToLower(UINT32 *str, int len);
	
	/** Converts the scUTF-32 string str with length len to upper-case. */
	void SCToUpper(UINT32 *str, int len);
	
	/** Converts a set of strings in scUTF-32 to one string in UTF-16.
		@param buffer A set of scUTF-32 strings to convert to one UTF-16 string.
		@param dst Will contain a pointer to the UTF-16 string created.
		@param allocator The allocator to use when allocating the UTF-16 string. */
	OP_STATUS CreateUTF16FromBuffer(ExpandingBuffer<UINT32*> *buffer, uni_char *&dst, OpSpellCheckerAllocator *allocator);
	
	/** Converts a string in the dictionary-encoding to UTF-16
		@param str The string in the dictionary-encoding to convert.
		@param buf The buffer where the resulting UTF-16 string should be written.
		@param buf_bytes Number of bytes in the buffer. */
	OP_STATUS ConvertStringToUTF16(UINT8 *str, uni_char *buf, int buf_bytes);
	
	/** The reversed operation of ConvertStringToUTF16, see description of ConvertStringToUTF16. */
	OP_STATUS ConvertStringFromUTF16(const uni_char *word, UINT8 *buf, int buf_bytes);
	
	/** Allocates the "hash bit-field" for fast elimination of non-existing index-strings during dictionary lookup. */
	OP_STATUS SetupHashBitField();
	
	/** Sets a bit in the "hash bit-field" */
	void SetHashBit(int bit) { m_word_hash[bit>>3] |= 1 <<(bit&7); }
	
	/** The "SDBM" hashing function. */
	static UINT32 GetHashFor(UINT32 *indexes, int len)
	{
		int i;
		UINT32 hash = 0;
		for (i=0;i<len;i++)
			hash = indexes[i] + (hash << 6) + (hash << 16) - hash;
		return hash;
	}
	
	/** Returns the index into the "hash bit-field" corresponding to the index-string indexes with length len. */
	int GetHashBitFor(UINT32 *indexes, int len) { return GetHashFor(indexes, len) & m_word_hash_pattern; }
	
	/** Returns TRUE if the index-string indexes with length len has it's corresponding bit in the "hash bit-field"
		set. If it's not set (FALSE), then we know the word doesn't exist in the dictionary. */
	BOOL IsHashSetFor(UINT32 *indexes, int len) { int bit = GetHashBitFor(indexes,len); return !m_word_hash || m_word_hash[bit>>3] & 1<<(bit&7); }
	
	/** Here is hash a hash-value calculated from previous characters, now we "adds" the INDEX ch to the hash-value 
		and checks if the corresponding bit in the "hash bit-field" is set for the index-string ending with ch. This
		is an "optimization" used during compound word lookup. */
	BOOL IsHashSetFor(UINT32 ch, UINT32 &hash) { hash = ch + (hash << 6) + (hash << 16) - hash;	int bit = (int)(hash & m_word_hash_pattern); return !m_word_hash || m_word_hash[bit>>3] & 1<<(bit&7); }
	
	/** This function creates huffman-codes and setups a huffman table for decoding huffman-codes, as well as a value-
		to-huffman code table (for used when encoding).
		@param counters A list of frequencies for all values that should be encoded.
		@param len The number of elements in counters - the number of values to create huffman codes for.
		@param code_to_value The huffman-table created, the number of elements will be 1<<huffman_levels.
		@param value_to_code The value-to-code mapping, the number of elements will be len - the number of frequences
		that where zero.
		@param huffman-levels The maximum number of bits in each huffman-code. */
	OP_STATUS SetupHuffmanCodes(int *counters, int len, HuffmanEntry *&code_to_value, HuffmanEntry *&value_to_code, int huffman_levels);

	/** This function parses a condition-string for an affix and converts it to an array of AffixConditionData structs,
		see description of AffixConditionData. 
		@param str the affix-condition string in the dictionary encoding.
		@param buffer A buffer from where all the AffixConditionData structs and AffixConditionData::chars strings
		will be allocated. Must be 32-bit aligned.
		@param buf_len Number of bytes in buffer.
		@param cond_count Number of elements in the returned AffixConditionData array.
		@return The resulting AffixConditionData array, NULL indicates parsing error or that the buffer where too small. */
	AffixConditionData* GetAffixConditionData(UINT8 *str, UINT8 *buffer, int buf_len, int &cond_count);

	/** This function parses a compoundrule condition and converts it to an array of CompoundRuleConditionData structs,
		see description of CompoundRuleConditionData.
		@param str the compoundrule condition, the flags are encoded using the flag-encoding specified by flag_encoding.
		@param buffer A buffer from where all the CompoundRuleConditionData structs and CompoundRuleConditionData::indexes
		flag indexes will be allocated. Must be 32-bit aligned.
		@param buf_len Number of bytes in buffer.
		@param flag_encoding The flag-encoding used.
		@param cond_count Number of elements in the returned array.
		@return The resulting CompoundRuleConditionData array, NULL indicates parsing error or that the buffer where too small. */
	CompoundRuleConditionData* GetCompoundRuleConditionData(UINT8 *str, UINT8 *buffer, int buf_len, FlagEncoding flag_encoding, int &cond_count);

	/** This function is called after each affix-file parse pass (they are 2), end performs a couple of different tasks
		that should be done after those, this involves circumfix handling etc. See the source-code and documentation for 
		the functions called from there.
		@param fd The HunspellAffixFileData where the information for the affix-file parsing passes is stored.
		@param state The current state of the affix-file parsing (only used for checking which pass that has finished). */
	OP_STATUS PostProcessPass(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	
	/** This function takes a string str with flags encoded with the current flag-encoding, decodes it, and outputs
		it to result in a "special format", described here:
		1. bit-flags, one UINT16 with the "bit-flags". If AFFIX_BIT_HAS_AFFIXES is set does this mean that global IDs
		for affix-flags will follow (2 and 3). If AFFIX_BIT_HAS_COMPOUNDRULEFLAG is set does this means that type IDs
		for compoundrule flags will follow (4 and 5).
		2. affix-flag count, one UINT16 counting the number of affixes that follows.
		3. The global ID's for the affix flags.
		4. compoundrule-flag count, one UINT16 counting the number of compound rule flags that follows
		5. The type ID's for the compoundrule flags.
		@param str The '\0' terminated flags (in the current flag-encoding) that should be decoded.
		@param The buffer where the decoded flags (in the format described above) will be written. If result == NULL will
		null be set to a new buffer allocated using allocator.
		@param fd The HunspellAffixFileData object containing the necessary information for decoding the flags.
		@param allocator If result == NULL will this OpSpellCheckerAllocated allocate the buffer which result will be set
		to. At least one of result and allocator must be != NULL.
		*/
	OP_STATUS GetFlagPtr(UINT8 *str, UINT16 *&result, HunspellAffixFileData *fd, OpSpellCheckerAllocator *allocator);
	
	/** This function simply calls get_wide_str_ptr() with the same arguments + the argument bit8_enc == this->m_8bit_enc,
		see description of get_wide_str_ptr(). */
	OP_STATUS GetWideStrPtr(UINT8 *str, UINT32 *&result, OpSpellCheckerAllocator *allocator);
	
	/** This function is used for parsing the "bit-flags" like COMPOUNDEND, etc. It creates a new FlagClass object with
		the specified type and adds it to fd.
		@fd The HunspellAffixFileData to which we should add the new flag.
		@state The current HusnpellAffixFileParseState.
		@type The type of the "bit-flag". */
	OP_STATUS ParseFlagGeneral(HunspellAffixFileData *fd, HunspellAffixFileParseState *state, AffixFlagType type);
	
	/** This function parse simple value-statements in the affix-file such as COMPOUNDWORDMAX and set the in the iteger
		pointed out by result to the result. */
	OP_STATUS ParseIntValue(int *result, HunspellAffixFileData *fd, HunspellAffixFileParseState *state);

	/** Called during the first affix-file parsing pass for COMPOUNDRULE statements. FlagClass objects are created here
		for compoundrule flags that haven't appeared in any COMPOUNDRULE statement prior to the current line. */
	OP_STATUS ParseCompoundRulePass0(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	
	/** Called during the second affix-file parsing pass for COMPOUNDRULE statements. The reason for splitting up the
		compoundrule parsing to two passes is that the internal data structures in the CompoundRule objects depends on
		the total number of compoundrule-flags in all COMPOUNDRULE statements.
		During this pass is a CompoundRule object created and it's "jump" and "retry" tables are setup, see description
		of the CompoundRule class. */
	OP_STATUS ParseCompoundRulePass1(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	
	/** Called during the first affix-file parsing pass for PFX and SFX statements. An OpSpellCheckerAffix object is created,
		the affix-rules are parsed and OpSpellCheckerAffixRule objects are added to the OpSpellCheckerAffix. See description
		of OpSpellCheckerAffix and OpSpellCheckerAffixRule. */
	OP_STATUS ParseAffixPass0(HunspellAffixFileData *fd, HunspellAffixFileParseState *state, BOOL is_suffix);
	
	/** Called during the second affix-file parsing pass for PFX and SFX statements. The (most important) reason for splitting
		up the parsing in two passes is that allocation of the arrays pointed out by OpSpellChecker::m_affix_char_mapping
		should have the the total number of affixes+1 elements. 
		The OpSpellCheckerAffix created in the first affix-file pass is having their bit-masks arrays initialized and
		OpSpellChecker::m_affix_char_mapping is setup so that character can be converted to affix-char-substitution
		numbers - see description of OpSpellCheckerAffix. */
	OP_STATUS ParseAffixPass1(HunspellAffixFileData *fd, HunspellAffixFileParseState *state, BOOL is_suffix);
	
	/** This function more or less just calls ParseAffixPass0 and ParseAffixPass1. */
	OP_STATUS ParseAffix(HunspellAffixFileData *fd, HunspellAffixFileParseState *state, BOOL is_suffix);

	/*  Affix line parsers. This is the functions pointed out by the AffixParser structs in OpSpellChecker::m_affix_parsers,
		see description of AffixParser. One of this functions will be called for each line in the affix-file that should be
		parsed. */
	OP_STATUS ParseSFX(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParsePFX(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseREP(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParsePHONE(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseKEY(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseAF(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseTRY(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseSET(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseFLAG(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPLEXPREFIXES(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDFLAG(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDBEGIN(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDMIDDLE(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDEND(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDWORDMAX(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDROOT(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDPERMITFLAG(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDFORBIDFLAG(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCHECKCOMPOUNDDUP(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCHECKCOMPOUNDREP(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCHECKCOMPOUNDTRIPLE(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCHECKCOMPOUNDCASE(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseNOSUGGEST(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseFORBIDDENWORD(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseLEMMA_PRESENT(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCIRCUMFIX(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseONLYINCOMPOUND(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParsePSEUDOROOT(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseNEEDAFFIX(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDMIN(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDSYLLABLE(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseSYLLABLENUM(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCHECKNUM(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseWORDCHARS(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseIGNORE(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCHECKCOMPOUNDPATTERN(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCOMPOUNDRULE(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseMAP(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseBREAK(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseLANG(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseVERSION(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseMAXNGRAMSUGS(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseNOSPLITSUGS(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseSUGSWITHDOTS(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseKEEPCASE(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseSUBSTANDARD(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	OP_STATUS ParseCHECKSHARPS(HunspellAffixFileData *fd, HunspellAffixFileParseState *state);
	/* End of affix line parsers... */

	/** This function is called during dictionary parse pass 1 when the stem and inflections for one word in the .dic
		file have been generated and now resists in the TempDictionaryParseData object used.
		During pass 1 is the radix-tree created. The task for this function is to create new OpSpellCheckerRadix objects
		where it seems "necessary". When SPC_RADIX_ALLOCATION_COUNT words have been found that should be located under 
		a certain RadixEntry for which no child OpSpellCheckerRadix has yet been created - will such OpSpellCheckerRadix
		be created. Before that happens is the RadixEntry a counter for the words that belongs there. See descriptions
		of OpSpellCheckerRadix and RadixEntry. */
	OP_STATUS ProcessStemAndInflectionsPass1();

	/** This function is called during dictionary parse pass 2 when the stem and inflections for one word in the .dic
		file have been generated and now resists in the TempDictionaryParseData object used.
		During pass 2 is the number of words that should be added in each linked list of word-units counted. The index,
		bit-flags and compoundrule-flags frequenceis are also counted in order to create huffman-codes. */
	OP_STATUS ProcessStemAndInflectionsPass2();

	/** This function is called during dictionary parse pass 3 when the stem and inflections for "as many words possible"
		in the .dic	file have been generated and resists in the TempDictionaryParseData object used. Mor exactly, words
		are added to the TempDictionaryParseData object unit TempDicionaryParseData::ReadyForAnotherDictionaryWord()
		returns FALSE.
		During the third pass, which also is the last, are the words finally added to the dictionary - linked list of 
		word-units are created and the huffman-coded words are written. */
	OP_STATUS ProcessStemAndInflectionsPass3();

	/** Called for each stem in the .dic file during dictionary parse pass 0. Here is the frequency of existing characters
		counted in order to decide (in DictionaryPostProcessPass1()) which characters that should be mapped to the mapping
		for "unusual characters".*/
	OP_STATUS DictionaryParsePass0();
	
	/** Called for each stem in the .dic file during dictionary parse pass 1. The flags are decoded and stored so that they
		doesn't need to be decoded during the following passes, alse matching affix-rules for each affix when the number
		of matches is <= SPC_MAX_RULE_PER_AFFIX_CACHE is cached. All inflections are generated and ProcessStemAndInflectionsPass1()
		is called. */
	OP_STATUS DictionaryParsePass1();
		
	/** Called for each stem in the .dic file during dictionary parse pass 1. The flags are decoded and stored so that they
		doesn't need to be decoded during the following passes, alse matching affix-rules for each affix when the number
		of matches is <= SPC_MAX_RULE_PER_AFFIX_CACHE is cached. All inflections are generated and ProcessStemAndInflectionsPass1()
		is called. */
	OP_STATUS DictionaryParsePass2();

	/** Called for each stem in the .dic file during dictionary parse pass 1. All inflections are generated and 
		ProcessStemAndInflectionsPass1() is called. */
	OP_STATUS DictionaryParsePass3();
	
	/** Called during dictionary parse pass 1. As many stems "as possible" is processed and inflections are generated
		from those. Mor exactly, words are added to the TempDictionaryParseData object unit
		TempDicionaryParseData::ReadyForAnotherDictionaryWord()	returns FALSE.
		After this is ProcessStemAndInflectionsPass1() called. */
	OP_STATUS DictionaryPostProcessPass0();

	/** Here are mappings constructed (except for characters that have already been mapped due to MAP statements in the
		.aff file) - for existing chars that have upper-/lower- case versions. Also the mapping for "unusual characters"
		is created. Furthermore are the indexes setup, so that all characters that exists in the dictionary are mapped
		to certain indexes. */
	OP_STATUS DictionaryPostProcessPass1();
	
	/** This function is called after dictionary parse pass 1. The most common character in every mapping are marked as
		most frequent in their mappings. Huffman-codes are created for indexes, bit-flags and compoundrule-flags. */
	OP_STATUS DictionaryPostProcessPass2();
	
	/** Nothing is done.. */
	OP_STATUS DictionaryPostProcessPass3();

	/** Removes the word str from the .oow file, either among the added words (if in_added_words == TRUE) or among the
		removed words (if in_added_words == FALSE). removed Will be set to TRUE if a word was removed. */
	OP_STATUS RemoveStringsFromOwnFile(const uni_char *str, BOOL in_added_words, BOOL &removed);
	
	/** Creates the .oow file if it doesn't already exist. */
	OP_STATUS CreateOwnFileIfNotExists();
	
	/** Removes the new-line separated dicionary-encoding encoded words in data from the dictionary - which means that
		they're added to m_removed_words. The number of bytes in data should be len. */
	OP_STATUS RemoveWordList(UINT8 *data, int len);

#ifdef SPC_DEBUG
	void DebugPrintStat();
#endif

private:
	HunspellAffixFileData *m_affix_file_data; ///< The HunspellAffixFileData object used during affix-file and dictionary parsing.
	HunspellAffixFileParseState *m_affix_file_parse_state; ///< The HunspellAffixFileParseState object used during affix-file parsing.
	HunspellDictionaryFileData *m_dic_file_data; ///< The HunspellDictionaryFileData object used during dictionary parsing.
	HunspellDictionaryParseState *m_dic_parse_state; ///< The HunspellDictionaryParseState object used during dictionary parsing.
	DictionaryChunks *m_dic_file_chunks; ///< The chunks of data containing the .dic file.

	uni_char *m_dictionary_path; ///< Absolute path to the .dic file.
	uni_char *m_own_path; ///< Absolute path to the .aff file.
	uni_char **m_error_str; ///< Should be an error-string set e.g. when parsing fails, not used...

	LoadState m_state; ///< Current state of the dictionary.
	OpSpellCheckerLanguage *m_language; ///< The OpSpellCheckerLanguage that uses this OpSpellChecker.
	DictionaryEncoding m_encoding; ///< The dictionary encoding.
	BOOL m_8bit_enc; ///< TRUE if the dictionary is not UTF-8 encoded (we use a 8-bit encoding).
	char *m_enc_string; ///< String identifier for the dictionary-encoding, used when creating CharConverters.
	CharConverter *m_output_converter; ///< An OutputConverter used when translating UTF-16 to scUTF-32.
	CharConverter *m_input_converter; ///< An InputConverter used when translating scUTF-32 to UTF-16.
	UINT8 *m_8bit_to_upper; ///< When m_8bit_enc == TRUE is this a lookup table for converting character in the dictionary encoding to upper-case.
	UINT8 *m_8bit_to_lower; ///< When m_8bit_enc == TRUE is this a lookup table for converting character in the dictionary encoding to lower-case.

	AffixParser *m_affix_parsers; ///< The AffixParser structs.
	int m_affix_parser_count; ///< Number of elements in m_affix_parsers.
	UINT8 m_tmp_buffer[SPC_TEMP_BUFFER_BYTES]; ///< Temporary buffer used in several occations.
	int m_tmp_temp_buffer_users; ///< Used for debugging (checking that TempBuffer() isn't used twice without calling ReleaseTempBuffer()).
	ExpandingBuffer<CompoundRule*> m_compound_rules; ///< The compoundrules from the .aff file.
	
	/** The indexes-strings for the strings specified in REP statements in the .aff file. This array has the following
		format: 
		1. Number of indexes that should be replaced. 
		2. The indexes that should be replace.
		3. Number of indexes we should replace them with.
		4. The indexes we should replace with. */
	ExpandingBuffer<UINT32*> m_rep; 
	
	ExpandingBuffer<MapMapping> m_mappings; ///< The mappings, for all indexes that maps to more then one character.
	union
	{
		MappingPointer *linear;
		UTFMapping<MappingPointer> *utf;
		void *data;
	} m_char_mappings; ///< This union maps a scUTF-32 character to it's corresponding mapping.

	ExpandingBuffer<int> m_len_to_bit_representation; ///< Lookup array for bits_to_represent(n), 0,0,1,2,2,3,...

	uni_char *m_word_chars; ///< Characters in WORDCHARS statements in the .aff file.
	uni_char *m_break_chars; ///< Characters in BREAK statements in the .aff file.

	union
	{
		UINT32 *linear;
		UTFMapping<UINT32> *utf;
		void *data;
	} m_existing_chars; ///< The "approximated" frequency for each character that exists in the dictionary.

	int m_chars_in_dic_data; ///< Total number of characters in the .dic file.
	int m_approx_freq_len; ///< Number of elements in m_approx_freq.
	IndexCounterMapping *m_approx_freq; ///< (character, "approximated" frequency) pairs for all characters that exists in the dictionary.
	ExpandingBuffer<UINT32> m_unusual_chars; ///< All scUTF-32 characters that exists in the dictionary but is considered "unusual".
	int m_unusual_mapping_index; ///< Index into m_mappings for the MapMapping that should keep the "unusual" characters.
	
	OpSpellCheckerAllocator m_allocator; ///< OpSpellCheckerAllocator for data that should remain in the memory after the dictionary-loading is finished (e.g. the words).
	OpSpellCheckerAllocator *m_lookup_allocator; ///< Pointer to the allocator used for allocating temporary data during dictionary lookup.
	OpSpellCheckerRadix *m_root_radix; ///< The root in the radix-tree.
	RadixEntryCache m_radix_cache; ///< The radix-entry cache, see description of RadixEntryCache.
	UINT8 *m_index_combinations; ///< The existing-index-combinations for "n-gram suggestions" (where n is m_combination_len).
	UINT8 *m_short_combinations[SPC_MAX_COMBINATION_LEN]; ///< The existing-index-combinations for words shorter then m_combination_len.
	int m_combination_len; ///< The number of indexes used in "n-gram suggestions".
	int m_total_word_count; ///< Number of words (including inflections) in the dictionary.
	UINT8 *m_word_hash; ///< The hash-bit-field used for fast elimination of words that is not in the dictionary, used during dictionary lookup.
	UINT32 m_word_hash_pattern; ///< When obtaining a hash-value for an index-string, hash & m_word_hash_pattern gives the bit-index into m_word_hash.
	
	HuffmanEntry *m_huffman_to_index; ///< Huffman table for the indexes.
	HuffmanEntry *m_index_to_huffman; ///< The huffman-codes for each index.
	HuffmanEntry *m_huffman_to_flags; ///< Huffman table for the bit-flag combinations.
	HuffmanEntry *m_flags_to_huffman; ///< The huffman-codes for each bit-flag combination.
	HuffmanEntry *m_huffman_to_compound_rule_flag; ///< Huffman table for the compoundrule flags.
	HuffmanEntry *m_compound_rule_flag_to_huffman; ///< The huffman-codes for each compoundrule flag.

	int *m_index_counters; ///< Counters for the frequency of each index.
	int *m_flag_counters; ///< Counters for the frequency of each combination of bit-flags. 
	int *m_compound_rule_flag_counters; ///< Counters for the frequency of each compoundrule-flag.

	/** After reading a 16-bit value with the huffman-code for an index, value & m_huffman_index_mask gives the index 
		into m_huffman_to_index. */
	UINT32 m_huffman_to_index_mask;
	
	/** After reading a 16-bit value with the huffman-code for a compoundrule flag, value & m_huffman_to_compound_rule_flag_mask
		gives the index into m_huffman_to_compound_rule_flag. */
	UINT32 m_huffman_to_compound_rule_flag_mask; 

	BOOL m_has_compounds; ///< TRUE if compounds are allowed for some words in the dictionary.
	BOOL m_has_radix_flags; ///< TRUE if any bit-flags exists in the dictionary.
	BOOL m_has_compound_rule_flags; ///< TRUE if COMPOUNDRULE statements exists in the .aff file.

	BOOL m_checksharps; ///< TRUE if CHECKSHARPS is defined in the .aff file. TODO: implement
	UINT32 m_sharps_char;
	BOOL m_checkcompoundcase; ///< TRUE if CHECKCOMPOUNDCASE is defined in the. TODO: implement
	int m_compoundmin; ///< Smalleds allowed number of characters in a word that exists in a compound.
	int m_compoundwordmax; ///< Maximum allowed number of word in a compound.

	SortedStringCollection m_added_words; ///< Word that have been added to the dictionary after that the dictionary was loaded.
	SortedStringCollection m_removed_words; ///< Words that have been removed from the dictionary, including the "removed" words in the .oow file.
	SortedStringCollection m_added_from_file_and_later_removed; ///< Words that was added from the added words in the .oow file, but later removed.

	//Some statistical information...
	UINT32 m_stat_counters;
	UINT32 m_stat_count_total;
	UINT32 m_stat_max_counter;
	UINT32 m_stat_misc_dictionary_bytes;
	UINT32 m_stat_misc_lookup_bytes;
	UINT32 m_stat_lookup_bytes;
	double m_stat_start_time;
};

#else // !SPC_USE_HUNSPELL_ENGINE

#define HUNSPELL_FILE void
HUNSPELL_FILE* hunspell_fopen(const char *path, const char *mode);
int hunspell_fclose(HUNSPELL_FILE* fp);
size_t hunspell_fread(void *ptr, size_t size, size_t nmemb, HUNSPELL_FILE *stream);
char* hunspell_fgets(char *s, int size, HUNSPELL_FILE *stream);
char* hunspell_strncpy(char *dest, const char *src, size_t n);

#include "modules/spellchecker/hunspell_3p/src/hunspell/hunspell_impl.h" //FIXME: Shouldn't really use the internal API directly.

#define SPC_MAX_REPLACEMENTS MAXSUGGESTION

struct OpSpellCheckerLookupState
{
	OpSpellCheckerLookupState();
	~OpSpellCheckerLookupState() { Reset(); }
	void Reset(); ///< Must be called after each word-lookup (which might include replacement-suggestion search).
	BOOL3 correct; ///< YES, NO - or MAYBE which means that we don't kno yet.
	BOOL in_progress; ///< Is the lookup in still in progress, used for implementing interruptable replacement-suggestion search.
	ExpandingBuffer<uni_char*> replacements; ///< The resulting replacement suggestion after a replacement-suggestion search.
	OpSpellCheckerWordIterator *word_iterator; ///< The word-iterator containing the word to lookup / find replacement-suggestions for.
};


class OpSpellChecker
{
public:
	/** Constructor. 
		@param language The OpSpellCheckerLanguage instance which uses this dictionary. */
	OpSpellChecker(OpSpellCheckerLanguage *language);
	~OpSpellChecker();
	
	/** Deletes the hash-bit-field for faster lookup, but only if the dictionary has finished loading. 
		@param toplevel_context Irrelevant... 
		@return TRUE if the memory where freed. */
	BOOL FreeCachedData(BOOL toplevel_context);

	/** Start the loading of a dictionary. This instance must not have any dictionary already loaded. 
		@param dic_path Absolute path to the .dic file.
		@param affix_path Absolute path to the .aff file.
		@param own_path Absolute path to the .oow file.
		@param time_out Maximum time until this function will return (but loading can continue later). If time_out==0.0
		does this means that this function shouldn't return until the loading has finished.
		@param finished Will be set to TRUE if the loading completed. FALSE means that we timed-out, for resuming the
		loading-process should ContinueLoadingDictionary() be used.
		@param error_str This pointer is intended for error-reporting upon failure, NOT implemented.
		@return Error status code. Will be OpStatus::ERR upon parsing-failure, or if we're not supporting the 
		dictionary for some other reason -> in general due to "inflection overflow".
		*/
	OP_STATUS LoadDictionary(uni_char *dic_path, uni_char *affix_path, uni_char *own_path, double time_out, BOOL &finished, uni_char **error_str);
		
	/** Continues loading of the dictionary. Loading must have been started by a previous call to LoadDictionary().
		@param time_out Maximum time until this function will return (but loading can continue later). If time_out==0.0
		does this means that this function shouldn't return until the loading has finished.
		@param finished Will be set to TRUE if the loading completed. FALSE means that we timed-out, for resuming the
		loading-process should ContinueLoadingDictionary() be called again.
		@param error_str This pointer is intended for error-reporting upon failure, NOT implemented.
		@return Error status code. Will be OpStatus::ERR upon parsing-failure, or if we're not supporting the 
		dictionary for some other reason -> in general due to "inflection overflow". */
	OP_STATUS ContinueLoadingDictionary(double time_out, BOOL &finished, uni_char **error_str);

	/** Clears all data in the dictionary - and everything else in this object. */
	void ClearDictionary();
	
	/** Returns TRUE if the dictionary is fully loaded. */
	BOOL HasDictionary();
	
	/** Returns FALSE for words that should be ignored totally from all "spellchecking activities", for 
		example - numbers and too long words. */
	BOOL CanSpellCheck(const uni_char *word, const uni_char* word_end);

	/** Returns FALSE if the word can't be added to the dictionary */
	BOOL CanAddWord(const uni_char *word) { return CanSpellCheck(word, word + uni_strlen(word)); }

	/** Reuturns TRUE if ch is considered to be a character that breaks words, either because it's a "default breaking
		character" like whitespaces, or because it's specified with BREAK in the affix-file. The default-breaking characters
		can also be overrided by WORDCHARS in the affix-file. Notice: CheckSpelling() has some special handling	of 
		"ambiguous" break characters like '-', which is sometimes part of a word - and sometimes used for breaking words. */
	BOOL IsWordSeparator(uni_char ch);
	
	/** The function used for check spelling of a word, and possibly - if find_replacements==TRUE - also seach for replacement
		suggestion words in case of a misspelled word.
		@param lookup_state The lookup state to use (see OpSpellCheckerLookupState and OpSpellCheckerRaplacementState). 
		@param time_out The deadline for when this function must return. This is only useful for when the word is misspelled
		and find_replacements==TRUE. If replacement_timeout==0.0 or if replacement_timeout > time_out, then might the 
		replacement search be paused and lookup_state->in_progress be set to TRUE - which implies that this function should
		be called again later to continue the replacement suggestion search. timeout_out == 0 implies deadline, 
		lookup_state->in_progress will be FALSE. However it's still possible that the replacement-search doesn't try all
		possible ways of finding replacements if replacement_timeout != 0.0. 
		@param find_replacements If TRUE and the word in lookup_state->word_iterator->GetCurrentWord() is misspelled,
		then will a replacement-suggestion search be performed.
		@param replacement_timeout The deadline when the replacement-suggestion search must finish and lookup_state->in_progress
		must be set to TRUE.
		@param override_word If override_word != NULL, then will this word be used instead of lookup_state->word_iterator->GetCurrentWord(),
		this is used internally for checking sub-words when the word contains "ambiguous" break characters (like '-'). 
		@return Error status code. */
	OP_STATUS CheckSpelling(OpSpellCheckerLookupState *lookup_state, double time_out, BOOL find_replacements, double replacement_timeout, const uni_char *override_word = NULL, const uni_char* override_word_end = NULL);
	
	/** Returns FALSE if the string str can't be represented using the dictionary's char-encoding. */
	BOOL CanRepresentInDictionaryEncoding(const uni_char *str);

	/** Adds word to the dictionary. If permanently == TRUE will the new word be written to the .oow file so that the
		change will be permanent. (But special handling if the word was in the orginal dictionary, but later removed..) */
	OP_STATUS AddWord(const uni_char *word, BOOL permanently);

	/** Removes a word from the dictionary. If permanently == TRUE will the new word be written to the .oow as a word that
		should be removed permanently. (But special handling if the word wasn't part of the original dictionary..). */
	OP_STATUS RemoveWord(const uni_char *word, BOOL permanently);

	/** Returns the number of bytes allocated by this OpSpellCheckerObject, including everything in the dictionary.
		This is the number of bytes allocated after loading has finished, the peak memory comsumption during the 
		dictionary parsing will be higher though.
		@param for_dictionary Bytes for storing the dictionary, etc, that must be in memory for the spellchecker to work.
		@param for_lookup Bytes not necessary for spellchecking to work, but is used for speed up dictionary lookup,
		i.e. The hash-bit-field and the existing-index-combination bit-field for "n-gram suggestions" (not used yet). */
	void GetAllocatedBytes(UINT32 &for_dictionary, UINT32 &for_lookup);

	/** Removes the word str from the .oow file, either among the added words (if in_added_words == TRUE) or among the
		removed words (if in_added_words == FALSE). removed Will be set to TRUE if a word was removed. */
	OP_STATUS RemoveStringsFromOwnFile(const uni_char *str, BOOL in_added_words, BOOL &removed);
	
	/** Creates the .oow file if it doesn't already exist. */
	OP_STATUS CreateOwnFileIfNotExists();
	
	/** Converts a string in the dictionary-encoding to UTF-16
		@param str The string in the dictionary-encoding to convert.
		@param buf The buffer where the resulting UTF-16 string should be written.
		@param buf_bytes Number of bytes in the buffer. */
	OP_STATUS ConvertStringToUTF16(UINT8 *str, uni_char *buf, int buf_bytes);
	
	/** The reversed operation of ConvertStringToUTF16, see description of ConvertStringToUTF16. */
	OP_STATUS ConvertStringFromUTF16(const uni_char *word, UINT8 *buf, int buf_bytes);

	private:

	OP_STATUS AddContextToSuggestions(OpSpellCheckerLookupState *lookup_state, const uni_char* left_start, const uni_char* left_end, const uni_char* right_start, const uni_char* right_end);

	OP_STATUS CheckAmbiguousBreaks(OpSpellCheckerLookupState *lookup_state, double time_out, BOOL find_replacements, double replacement_timeout, const uni_char* str_start, const uni_char* str_end, BOOL split=TRUE);


	OP_STATUS AddToHunspell(const uni_char* word);
	OP_STATUS RemoveFromHunspell(const uni_char* word);
	OP_STATUS LoadOwnFile();
	

	/* Returns a pointer to a temporary buffer with size SPC_TEMP_BUFFER_BYTES bytes. */
	UINT8 *TempBuffer() { OP_ASSERT(!m_tmp_temp_buffer_users++); return m_tmp_buffer; }
	
	/** "Releases" the temporary buffer obtained by TempBuffer(). */
	void ReleaseTempBuffer() { m_tmp_temp_buffer_users--; }
	

	/** A "different" (non-interruptable) way of looking up a word in the dictionary - without using OpSpellCheckerLookupState. 
		This is used internally	on a couple of occations.
		@param word The word to lookup, if word_iterator == NULL. 
		@param has_word Will be set to TRUE if the word is correct.
		@find_replacements If TRUE will replacements be searched for in at most SPC_REPLACEMENT_SEARCH_MAX_DEFAULT_MS.
		@listener A listener to inform of the result if listener != NULL. 
		@param word_iterator An OpSpellCheckerWordIterator than will be used instead of word if it's not NULL. 
		@return Error status code. */
	OP_STATUS IsWordInDictionary(const uni_char *word, BOOL &has_word, BOOL find_replacements = FALSE, OpSpellCheckerListener *listener = NULL, OpSpellCheckerWordIterator *word_iterator = NULL);

	OP_STATUS AddWordChar(const uni_char b);
	OP_STATUS CreateWordChars(Hunspell* hunspell);

	OP_STATUS AddBreak(const uni_char* b);
	
	OP_STATUS CreateBreaks();

	OpSpellCheckerLanguage *m_language; ///< The OpSpellCheckerLanguage that uses this OpSpellChecker.

	class Hunspell* m_hunspell;

	uni_char *m_dictionary_path; ///< The absolute path to the .dic file.
	uni_char *m_own_path; ///< Absolute path to the .oow file.

	UINT8 m_tmp_buffer[SPC_TEMP_BUFFER_BYTES]; ///< Temporary buffer used in several occations.
	int m_tmp_temp_buffer_users; ///< Used for debugging (checking that TempBuffer() isn't used twice without calling ReleaseTempBuffer()).

	BOOL m_8bit_enc; ///< TRUE for 8-bit encodings and FALSE for UTF-8, m_encoding and m_8bit_enc is now actually quite redundant.
	CharConverter *m_output_converter; ///< An OutputConverter used when translating UTF-16 to scUTF-32.
	CharConverter *m_input_converter; ///< An InputConverter used when translating scUTF-32 to UTF-16.

	SortedStringCollection m_added_words; ///< Word that have been added to the dictionary after that the dictionary was loaded.
	SortedStringCollection m_removed_words; ///< Words that have been removed from the dictionary, including the "removed" words in the .oow file.

	OpString m_word_chars; ///< Characters that would normaly be treated as word separators, but are declared by the dictionary as being word chars.
	uni_char** m_breaks; ///< Strings that are word breaks according to the dictionary and we do not handle by our internal definitions.
	int m_num_breaks;
};

	
#endif // !SPC_USE_HUNSPELL_ENGINE





#endif // OP_SPELL_CHECKER_H
