/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*-
 *
 * Copyright (C) 1995-2011 Opera Software AS.  All rights reserved.
 *
 * This file is part of the Opera web browser.	It may not be distributed
 * under any circumstances.
 *
 * @author Arjan van Leeuwen
 */

#ifndef QUICK_BROWSER_VIEW_H
#define QUICK_BROWSER_VIEW_H

#include "adjunct/quick_toolkit/widgets/QuickOpWidgetWrapper.h"
#include "adjunct/quick_toolkit/widgets/OpBrowserView.h"
#include "adjunct/quick_toolkit/windows/DesktopWindowListener.h"

/**
 * A listener for events generated by a BrowserView.
 */
class BrowserViewListener
{
public:
	virtual ~BrowserViewListener() {}

	/**
	 * Called when the BrowserView is ready to load content.
	 *
	 * @param browser_view the BrowserView that is ready
	 */
	virtual void OnBrowserViewReady(class BrowserView& browser_view) = 0;
};


/**
 * An OpBrowserView that does OpPage-related clean-up earlier.
 *
 * Rather than in OnDeleted(), this widget destroys the OpWindow that displays
 * the document as soon as the widget's parent DesktopWindow says it's being
 * closed.  This way, we avoid a double-delete of the OpWindow that happens
 * when we let the platform destroy it before we even know it.
 */
class BrowserView : public OpBrowserView, public DesktopWindowListener
{
public:
	static OP_STATUS Construct(BrowserView** obj);

	virtual OP_STATUS Init();

	void SetListener(BrowserViewListener* listener) { m_listener = listener; }

	// DesktopWindowListener
	virtual void OnDesktopWindowClosing(DesktopWindow* desktop_window, BOOL user_initiated);

	// OpWidget
	virtual void OnAdded();
	virtual void OnRemoved();
	virtual void OnDeleted();

private:
	BrowserView() : m_listener(NULL) {}

	BrowserViewListener* m_listener;
};


class QuickBrowserView : public GenericQuickOpWidgetWrapper
{
	IMPLEMENT_TYPEDOBJECT(GenericQuickOpWidgetWrapper);
public:
	virtual OpBrowserView* GetOpBrowserView() const = 0;
	virtual void SetListener(BrowserViewListener* listener) = 0;
protected:
	// Overriding QuickWidget
	virtual unsigned GetDefaultMinimumWidth() { return 300; }
	virtual unsigned GetDefaultMinimumHeight(unsigned width) { return 300; }
	virtual unsigned GetDefaultNominalWidth() { return GetDefaultMinimumWidth(); }
	virtual unsigned GetDefaultNominalHeight(unsigned width) { return GetDefaultMinimumHeight(width); }
	virtual unsigned GetDefaultPreferredWidth() { return WidgetSizes::Infinity; }
	virtual unsigned GetDefaultPreferredHeight(unsigned width) { return WidgetSizes::Infinity; }
};

template <typename OpWidgetType>
class QuickBrowserViewWrapper : public QuickBrowserView
{
public:
	OP_STATUS Init()
	{ 
		OpWidgetType* browser_view = NULL;
		RETURN_IF_ERROR(OpWidgetType::Construct(&browser_view));
		SetOpWidget(browser_view);
		return OpStatus::OK;
	}
};

// this is needed when used inside a QuickWindow (or QuickDialog)
class QuickDialogBrowserView : public QuickBrowserViewWrapper<BrowserView>
{
	IMPLEMENT_TYPEDOBJECT(QuickBrowserViewWrapper<BrowserView>);
public:
	virtual BrowserView* GetOpBrowserView() const { return static_cast<BrowserView*>(GetOpWidget()); }
	virtual void SetListener(BrowserViewListener* listener) { GetOpBrowserView()->SetListener(listener); }
};

// this is needed when NOT used inside a QuickWindow (or QuickDialog)
class QuickOpBrowserView : public QuickBrowserViewWrapper<OpBrowserView>
{
	IMPLEMENT_TYPEDOBJECT(QuickBrowserViewWrapper<OpBrowserView>);
public:
	virtual OpBrowserView* GetOpBrowserView() const { return static_cast<OpBrowserView*>(GetOpWidget()); }
	virtual void SetListener(BrowserViewListener* listener) { OP_ASSERT(!"Not supported"); }
};

#endif // QUICK_BROWSER_VIEW_H
