/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4; c-file-style:"stroustrup" -*-
**
** Copyright (C) 2000-2008 Opera Software AS.  All rights reserved.
**
** This file is part of the Opera web browser.  It may not be distributed
** under any circumstances.
**
** Yngve Pettersen
**
*/

#ifndef _URL_LOP_API_H_
#define _URL_LOP_API_H_

#include "modules/formats/uri_escape.h"

#include "modules/util/opfile/opfolder.h"

class URL_LoadHandler;
class OpWindowCommander;

/** This class is the base class of objects used to register a 
 *	opera:name URL with the specified name or prefix.
 *
 *	These Generators are registered using g_url_api->RegisterOperaURL()
 *	The caller retains ownership of the object, and can unregister 
 *	by performing an Out operation on it, or simply by deleting the object.
 *
 *	The generator have three modes:
 *
 *		- Retrieve content from a specified path and file (only available for profiles with disk)
 *		- Generate in a oneshot
 *		- Generate in multiple steps by creating a URL_LoadHandler based object
 *
 *	The mode is indicated by the virtual Mode() value.
 */
class OperaURL_Generator : public ListElement<OperaURL_Generator>
{
private:
	OpString8 name;
	BOOL is_prefix;

public:

	/** An implementation will only implement one of these modes */
	enum GeneratorMode {
		/** The content is from a file on the local system.
		 *	Get the location from GetFileLocation()
		 *	Implementation SHOULD have a fallback if they 
		 *	are active in non-disk profiles
		 */
		KFileContent,

		/** The content is generated by this object
		 *	when calling the QuickGenerate() function,
		 *	which writes the generated content to the URL
		 */
		KQuickGenerate,

		/**	The content is generated by a URL_LoadHandler object
		 *	The loadhandler may be multistep, and can either write
		 *	the content directly to the URL, or use the native
		 *	ProcessDataReceived and ReadData calls to produce the data
		 *
		 *	The loadhandler is generated by RetrieveLoadHandler()
		 */
		KGenerateLoadHandler
	};

public:

	/** Constructor */
	OperaURL_Generator(){}

	/** Destructor */
	virtual ~OperaURL_Generator(){if(InList()) Out();}

	/** Construct the object 
	 *	@param p_name	The name of the opera: URL path to be matched. 
	 *					The name must contain any slashes that are part of the name
	 *	@param prefix	Is the name a prefix?
	 */
	OP_STATUS Construct(const OpStringC8 &p_name, BOOL prefix){OP_ASSERT(p_name.HasContent()); is_prefix = prefix; return name.Set(p_name);}

	/** What is the name of this method's path? */
	OpStringC8 GetName() const {return name;}

	/** Is this name a prefix */
	BOOL GetIsPrefix() const {return is_prefix;}

	/** Match path against name */
	BOOL MatchesPath(const OpStringC8 &path) const
	{
		if (is_prefix)
			return UriUnescape::isstrprefix(GetName().CStr(), path.CStr(), UriUnescape::All);
		else
			return UriUnescape::strcmp(GetName().CStr(), path.CStr(), UriUnescape::All) == 0;
	}

	/** Can this generator load resources from file.
	 *
	 *  Normally, loading resources from disk is disallowed because of security concerns. This method
	 *  must return TRUE if e.g. SVGs from "file:" URLs are used in the content.
	 */
	virtual BOOL IsAllowedFileURLAccess() const { return FALSE; }

	/** What is the mode of this generator ? */
	virtual GeneratorMode GetMode() const=0;

	/** Return the location of the file, possibly dependant on the url
	 *	The implementation MUST set the appropriate Content-Type and charset.
	 *
	 *	An empty name means that the path/prefix is not supported; next match will be tried instead
	 *
	 *	Content-Type and charset is either set using url.SetAttribute() calls with either 
	 *	URL::KMIME_ForceContentType or URL::KContentType in combination with URL::KMIME_CharSet 
	 *	as the attributes, with appropriate parameter values
	 *
	 *	Example 1:
	 *			op_err = url.SetAttribute(URL::KMIME_ForceContentType, "text/css; charset=utf-8"
	 *
	 *	Example 2:
	 *			op_err = url.SetAttribute(URL::KContentType, URL_CSS_CONTENT);
	 *			op_err = url.SetAttribute(URL::KMIME_CharSet, "utf-8");
	 *
	 *
	 *	@param	url		The URL to be created. Note that it does not contain fragment identfiers.
	 *	@param	fldr	Returns the folder for the file to be used
	 *	@param	op_err	Returns the OP_STATUS status of the operation. 
	 *					OpStatus::ERR_NOT_SUPPORTED trigger matching against next generator,
	 *					all other errors trigger a load failure.
	 *	@return			The filename relative to the specified path
	 */
	virtual OpStringC GetFileLocation(URL &url, OpFileFolder &fldr, OP_STATUS &op_err){fldr = OPFILE_ABSOLUTE_FOLDER; op_err = OpStatus::ERR_NOT_SUPPORTED; OpStringC res(NULL); return res;}


	/** Generate all the content in one step and write the content to the supplied URL 
	 *	The implementation MUST set the correct content type and other flags.
	 *	Message posting is supplied by the caller, and the implemention MUST NOT post messages
	 *	Note that the URL does not contain fragment identfiers.
	 *
	 *	It is recommended that this implementation use the OpGeneratedDocument API from the about module.
	 */
	virtual OP_STATUS QuickGenerate(URL &url, OpWindowCommander* commander) {return OpStatus::ERR_NOT_SUPPORTED;}

	/** Generate a URL_LoadHandler object that will produce the data.
	 *	The implementation must create object derived from URL_LoadHandler, 
	 *	initialized it with url.GetRep(), and return it. 
	 *	Loading will be started by caller
	 *	See documentation for URL_LoadHandler for more information
	 *
	 *	@param	url		URL to reference; use url.GetRep() to get URL_Rep object. Note that it does not contain fragment identfiers.
	 *	@param	lh		location to store the generated loadhandler.
	 */
	virtual OP_STATUS RetrieveLoadHandler(URL &url, URL_LoadHandler *&lh){lh = NULL; return OpStatus::ERR_NOT_SUPPORTED;}
};

#endif
