/* -*- Mode: c++; tab-width: 4; c-basic-offset: 4 -*-
 *
 * Copyright (C) 2002-2011 Opera Software ASA.  All rights reserved.
 *
 * This file is part of the Opera web browser.
 * It may not be distributed under any circumstances.
 */
#ifndef HAVE_OPTESTSUITE_H
#define HAVE_OPTESTSUITE_H
#ifdef SELFTEST

#define SELFTEST_CORE2_VERSION

#include "modules/hardcore/base/opstatus.h"
#include "modules/hardcore/mh/messobj.h"


// External API

/**
 * @defgroup testsuitemacros Test Suite Macros
 * Convenience macros for calling the externally used parts of of the global
 * TestSuite object.
 *
 * @{
 *	@def SELFTEST_MAIN(X, Y)
 *	@see TestSuite::_main()
 *
 *	@def SELFTEST_RUN(X)
 *	@see TestSuite::_run()
 *
 *	@def SELFTEST_STEP()
 *	@see TestSuite::_step()
 *
 *	@def SELFTEST_CLEANUP()
 *	@see TestSuite::_cleanup()
 * @}
 */

#define SELFTEST_MAIN(X,Y)	g_selftest.GetTestSuite()->_main( (X), (Y) )
#define SELFTEST_RUN(X)		g_selftest.GetTestSuite()->_run(X)
#define SELFTEST_STEP()		g_selftest.GetTestSuite()->_step()
#define SELFTEST_CLEANUP()	g_selftest.GetTestSuite()->_cleanup()

// Internal API

/**
 * Base class for test groups. Implementations are generated by the selftest parser.
 */
class TestGroup
{
public:
	/**
	 * Run the next test in the group.
	 *
	 * @return 0 Done with group, the caller should switch to the next group.
	 * @return 1 Done with test, the caller should call again.
	 * @return -1 Not done with test, the test is responsible for signaling when it
	 *			   should be called again.
	 */
	virtual int TS_step() = 0;

	/**
	 * Empty virtual constructor.
	 */
	virtual ~TestGroup() {};
};

class MessageHandler;

/**
 * The main external API in selftest. Used to init, run and cleanup selftest runs.
 *
 * External calls to the TestSuite is normally done through the following macros:
 * <ul>
 *	<li>SELFTEST_MAIN(ARGC, ARGV)</li>
 *	<li>SELFTEST_RUN(INITED)</li>
 *  <li>SELFTEST_STEP()</li>
 *  <li>SELFTEST_CLEANUP()</li>
 * </ul>
 */
class TestSuite :
	public MessageObject
{
	friend class TestUtils;
public:

	/**
	 * Simple constructor for the TestSuite class.
	 */
	TestSuite( );

	OP_STATUS Construct();

	virtual ~TestSuite();

	/**
	 * Runs the test suite.
	 *
	 * @param inited	1 if opera has been initialized, if 0 only tests that does not require
	 *					opera to be initialized will be run.
	 * @param force		Used when _main() want to run tests before opera is initied. Should be
	 *					0 otherwise.
	 */
	SELFTESTAPI void _run( int inited=1, int force=0 );

	/**
	 * Steps the test suite one step. This basically tells the test suite to run a test or parts
	 * of a test if the test is asynchronous or delayed.
	 *
	 * _step() is normally called by the the message handler when it receives an MSG_TESTSUITE_STEP message.
	 * This message is used throughout the selftest code to signal that _step() should be called.
	 *
	 * @param test_id	The index of the last test run. This is used to prevent detect when tests
	 *					are sending multiple messages and throwing selftest out of sync.
	 * 
	 * @return 1	The current test step is finished. The called should signal for _step()
	 *				to be called again.
	 * @return 0	The testsuite is done or has failed. _step() should not be called again.
	 * @return -1	The test step is not finished yet. The testsuite takes responsibility for
	 *				signaling when _step() should be called again.
	 */
	SELFTESTAPI int _step( int test_id );

	/**
	 * Parses a selftest specific command line.
	 *
	 * @param argc Number of command line options in argv
	 * @param argv An array of command line options
	 */
	SELFTESTAPI void _main( int *argc, char *argv[] );

	/**
	 * Resets and cleans the test suite.
	 */
	SELFTESTAPI void _cleanup( );

	/**
	 * Resets the test suite.
	 */
	void Reset( );

	/**
	  * @return Whether we are doing a selftest run
	  */
	BOOL DoRun() const { return do_run; }

	/**
	 * Retrieves information about a specific testgroup.
	 *
	 * @param n The Index of the group, -1 means the current group.
	 * @param name The string where the name of the group is filled in.
	 * @param module The string where the name of the module the test belong to is filled in.
	 * @return TRUE		Successful, a test group with index n existed.
	 * @return FALSE	Failed, most probably because no testgroup with that index existed.
	 *					name and module is not filled with any meaningful data.
	 */
	static BOOL GroupInfo( int n, const char*&name, const char *&module );

	/**
	 * Checks whether the test suite is run in initialized mode.
	 */
	inline int IsOperaInitialized() { return opera_is_initialized; }

	// From MessageObject
	virtual void HandleCallback(OpMessage msg, MH_PARAM_1 par1, MH_PARAM_2 par2);	
	
	/* Compatibility API */
	SELFTESTAPI static int  step( int test_id );
	SELFTESTAPI static void Cleanup( );
	SELFTESTAPI static void run( int inited=1, int force=0 );
	SELFTESTAPI static void main( int *argc, char *argv[] );

private:
	void PostStepMessage(int delay = 0);
	
	int opera_is_initialized;		///< The testsuite is in initialized mode
	BOOL looped;
	BOOL do_run;
	TestGroup *group;

	MessageHandler* message_handler;
};
#endif // SELFTEST
#endif // HAVE_OPTESTSUITE_H
