/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4; c-file-style:"stroustrup" -*-
**
** Copyright (C) 1995-2011 Opera Software ASA.  All rights reserved.
**
** This file is part of the Opera web browser.  It may not be distributed
** under any circumstances.
**
** Espen Sand
*/

#include "core/pch.h"
#include "iconsource.h"

#include "adjunct/desktop_util/string/stringutils.h"
#include "adjunct/desktop_util/datastructures/StreamBuffer.h"
#include "adjunct/desktop_util/datastructures/ZlibStream.h"
#include "adjunct/desktop_util/resources/pi/opdesktopproduct.h"

/*
All hardcoded data in this file are valid ico files if written to disk.

The UINT8 arrays are made by using the provided program. Verify that there is
only one image inside the source file and that the background is transparent.

The string buffers are base64 encoded gzipped icon sets. They should contain
images in 16x16, 32x32 and 48x48 They are parsed with the IcoPackage class
Adding quotes: cat <base64-file> | sed -e 's/^/"/;s/$/"/' > <base64-file-with-quotes>
*/

/*
#include <stdio.h>
#include <sys/stat.h>
int main(int argc, char** argv)
{
	FILE* fp = fopen( argv[1], "r" );
	if (!fp)
	{
		perror("Could not open file");
		return -1;
	}

	int i=0;
	int newline;

	struct stat buf;
	int rc = stat(argv[1], &buf);
	if (rc == -1)
	{
	     perror("Could not stat file");
	     return -1;
	}

	printf("static UINT8 buffer[%ld] =\n{\n\t", buf.st_size);
	while (!feof(fp))
	{
		newline = i>0 && i%32 == 0;

		int ch = fgetc(fp);
		if (ch == EOF)
			break;

		if (newline)
			printf(",\n\t");
		else if (i > 0 && !newline)
			printf(", ");
		printf("0x%02x", ch);

		i++;
	}
	printf("\n};\n");
	fclose(fp);
	return 0;
}
*/

// ICO Global header
struct GlobalHeader
{
	UINT16 reserved;
	UINT16 type;
	UINT16 num_images;
};

// ICO Image header
struct ImageHeader
{
	UINT8  width;
	UINT8  height;
	UINT8  num_color;
	UINT8  reserved;
	UINT16 color_planes;
	UINT16 bits_per_pixel;
	UINT32 size;
	UINT32 offset;
};

/**
 * This class sets up a list of single raw ico format images by reading a base64 encoded
 * gzipped buffer that contains one ore more ico format images. Each element in the list
 * is a valid ico image as it would have been stored on disk. It is assumed that the source
 * images are quadratic as only the width is used to look a parsed image up.
 */
class IcoPackage
{
public:
	/**
	 * Initializes the object
	 */
	IcoPackage(): m_init(FALSE) {}

	/**
	 * Parse and set up internal list of images
	 *
	 * @param src Raw image data. Must be valid ico format.
	 *
	 * @return OpStatus::OK on success, OpStatus::ERR if there are no valid image
	 *         in the buffer, OpStatus::ERR_NO_MEMORY on insufficient memory
	 */
	OP_STATUS Init(const OpStringC8& src);

	/**
	 * Return the first image in the list that matches the width.
	 *
	 * @param width The width to mtach
	 * @param data On successful return, the image data as a raw ico image
	 * @param size_in_bytes On successful return, size in bytes of image data
	 *
	 * @return TRUE on a match, otherwise FALSE.
	 */
	BOOL Get(UINT32 width, UINT8*& data, UINT32& size_in_bytes);

private:
	struct IcoData
	{
		StreamBuffer<char> buffer;
		UINT32 width;
	};

private:
	OP_STATUS Load(const OpStringC8& src, StreamBuffer<char>& buffer);
	OP_STATUS Split(StreamBuffer<char>& src);

private:
	BOOL m_init;
	OpAutoVector<IcoData> m_list;
};


OP_STATUS IcoPackage::Init(const OpStringC8& src)
{
	if (m_init)
		return OpStatus::OK;

	m_init = TRUE;

	StreamBuffer<char> buffer;
	RETURN_IF_ERROR(Load(src, buffer));
	RETURN_IF_ERROR(Split(buffer));
	return OpStatus::OK;
}


BOOL IcoPackage::Get(UINT32 width, UINT8*& data, UINT32& size_in_bytes)
{
	for (UINT32 i=0; i<m_list.GetCount(); i++)
	{
		IcoData* item = m_list.Get(i);
		if (item->width == width)
		{
			data = (UINT8*)item->buffer.GetData();
			size_in_bytes = item->buffer.GetFilled();
			return TRUE;
		}
	}
	return FALSE;
}


OP_STATUS IcoPackage::Load(const OpStringC8& src, StreamBuffer<char>& buffer)
{
	unsigned char* data;
	unsigned long length;

	OP_STATUS rc = DecodeBase64(src, data, length);
	RETURN_IF_ERROR(rc);

	DecompressZlibStream stream(TRUE);
	rc = stream.AddData((char*)data, length);
	OP_DELETEA(data);
	if (OpStatus::IsError(rc))
		return rc;

	return stream.Flush(buffer);
}


OP_STATUS IcoPackage::Split(StreamBuffer<char>& src)
{
	if (src.GetFilled() <= sizeof(GlobalHeader))
		return OpStatus::ERR; // No image present

	const char* p = (char*)src.GetData();

	GlobalHeader global_header;
	memcpy(&global_header, p, sizeof(global_header));
	p += sizeof(global_header);

	if (src.GetFilled() < (sizeof(GlobalHeader) + global_header.num_images * sizeof(ImageHeader)))
		return OpStatus::ERR; // Too little data in file to hold specified number of image headers

	GlobalHeader gh;
	gh = global_header;
	gh.num_images = 1;

	for (UINT16 i=0; i<global_header.num_images; i++)
	{
		ImageHeader image_header;
		memcpy(&image_header, p, sizeof(image_header));
		p += sizeof(image_header);

		ImageHeader ih;
		ih = image_header;
		ih.offset = sizeof(global_header) + sizeof(image_header);

		IcoData* item = OP_NEW(IcoData,());
		if (!item || OpStatus::IsError(item->buffer.Reserve(ih.offset+ih.size)))
		{
			// Mem error. Discard image but try next
			OP_DELETE(item);
			continue;
		}

		if (src.GetFilled() < image_header.offset + image_header.size)
		{
			// Currupt header or file. Discard image but try next
			OP_DELETE(item);
			continue;
		}

		item->width = image_header.width;

		OP_STATUS rc = item->buffer.Append((char*)&gh, sizeof(gh));
		if (OpStatus::IsSuccess(rc))
			rc = item->buffer.Append((char*)&ih, sizeof(ih));
		if (OpStatus::IsSuccess(rc))
			rc = item->buffer.Append((char*)&src.GetData()[image_header.offset], image_header.size);
		if (OpStatus::IsSuccess(rc))
			rc = m_list.Add(item);
		if (OpStatus::IsError(rc))
		{
			// Mem error. Discard image but try next
			OP_DELETE(item);
			continue;
		}
	}

	// Return an error if we failed to load any image
	if (global_header.num_images > 0 && m_list.GetCount() == 0)
		return OpStatus::ERR;
	else
		return OpStatus::OK;
}

// Set of opera application icon in 16, 32 and 48 sizes
static const char* opera_application_icon_set = 
"H4sICEcYhE8AA29wZXJhX25vcm1hbDMuaWNvAOxaB1RU19q9M8P0QhlpKsUCotKkChYsiL2jFHvU"
"WBKDikEUxV4Be0PAAvbeu8EXrLGisURsGBVFiWheUCPMft+de0fKEwXj+tf61//PWtszM87cs/dX"
"9jmXOQwjYESMuztDoz2zyYFhfBiGsbfnX5swzD16z8SEez3UiGFm0gec6DP0FaY/w73/iYfg/wCE"
"BBGBosNICDKCkmBM0BKsCXYEiiRTnw+dJ8GbhxehAcGFUIdQg1CNYE6gyDMqgpy/thE/l/Ar8Rbz"
"12bnqcrP70voSoggxDMC432MoOYtRuD+nBH4FTCCgPeMIAgcWhECCY3ek5Q3JC+PYWyzGEZzlL67"
"lBBNCCM05bXb8DFR8nr+iRZD3NnrKPh4W+rjJ6wRxIgD1zKSrs8YcTCEsjCIlX0I/SBR9YVEPYBH"
"P0jVfSBThxJCCN1gJOsIgZDV1OwZaVrHMFbt+Nyx8THluUv5XHyNPHyAUNrdRCANnUsoEMgGgFEM"
"g0AxAgL5ODDS6TymgZFNJkwiTCTEgJGPJ4yjz0ZCrB4ChUkYFMbBMJJ2YnNEOQtMoLyZfE2upXkH"
"GQllvSKFst55AtkQMMqRxCcWjCSekMCPcRBI42FRPQ313I+gaWAmAlplwtnjMKzt1sBISdpksbym"
"8aQ5mvLzLeTGoRCJO5OO1vmkJZIReBl9Te5G8lC1SNH/oEjOxjuKQPNLFoARLyYs4rEQphZp6Njt"
"EVq0yUKLtncREPQbmre5jbZdHqB1xzvoEf4CVWtsKM6RdIo+RwLKjUT9DdVZOBhRBzDCthmkw/pr"
"cBcrB9iJlQOvihRDKeYTaM55xHUFj+WEZYSlxH0j2nV9hHqel/9d1+PC1AZ+F/yGjjinGB7xi8K/"
"Raaff4vrk5u2uvXv4LDnMK+WRvrnEubQ9WYRZuo1CeVREKv6QyDuQTo6PWJE7Z3+CXeJapCDRDUk"
"R6SMIO5TaY4lxDWZkMRjJSEREuUatOvyGNUdz+TWaZDhlJ6ebvQxNG550cm7yfWc7qRBplnO110c"
"r8WgJxZC5WAae9Hr4FxG3PWLNEhV32ql6u+yxKpRxJ29diJxXcUjhUeyHh6+F1Cz/jmYVD3Svjzu"
"Bjg1OBvg3eRXePuf/1B3jJhqUTKvuI/Y+lIOp17qDYGkZxYj6WZRWf4yTcQ+mWYkBCq6rnR5Cc6r"
"SkGsSEPLtvchUO859DnuBlStffJUp+CnOrlmlb72GPGSEr3E65HNIA0RpCGMEHqkMtwVJqMHEcjn"
"4iFSLitRLyXB6bGpeRiOLmcgMd4RXlH+ptWPDnTzvYy6rj+X6KPiXuL0EORsb4+itSWcXV+GVIS7"
"0vQHtdJ0bK7cZBLkVejakpV8nX9MQxI8/C5CbroTNetus6gofwe3Axba6j8hsN29En3E9RIHgybS"
"oBwLkWwwRPLeeUJZiPZz/NXamClq7XiozBMhNS557ZLguAskyWjc8gblet2tinI3gFHuzmrf9TFx"
"W1Oql0p7A0FG/aCIIf79WEz/FHeNNlJhbD4pV20+E+b2aRDIErn4Sz6mYSU0VTbDzecchIo1aZXl"
"L1TsSPNvfkOntdr1Xz1V2h9Ii3IyjOTDYaQYkG8kD1GUx9/UcuoAU8tp0FZfRV6SStoTOUgNWFms"
"h1DVfh9qOB2FkSopprL8jVSbYpw9zursHY+ViH3KxyGbT/uOKbS3GkQYOKg8/lrr2fu01rNgX287"
"xCYUBxVdU0n6FSs5yEvoka1Erfo/wdxmOySaZRXuXQMkmnXh9o7HaZ9x6iN9lVxaExsv5RzaFw5h"
"cfBj3KtYxyosqscXmNssRR33vTAyWQuR8WoIyeOE6tXkoymcHlXSB031GpyE2jyV+mRRq8rylxqv"
"bmVefTcaNDxfTo+VgXI+rc1RtMcYXihRffNf+zxru/lBVrYLYOe0EbVc9kGu3QCpdj2kZusgMU2j"
"fKTymtaQJtKlToEr1b7EeDlMLRe6Vpa/qWWKq8J0PXybZlaIv1C5lPZ/M2jdHkH4Lqgs/+o1l4yr"
"VmMJnL0PUv0cgMZqG9RWW6G03Ayl+Sby0o2QVSE92nWkKY00pcLd7zytD/NhYbOgZmX5W9gk1RRT"
"Tv2bXePXdgM+zl+kWAGxZgHtVUcRIiaV5W/rkLjatvYK+LXMoBwchtZ2H/XxHphW20W9vBPG1tv1"
"mlSWpMliCxTmG9GA+AsUc+HktkhbWf5ObklagXwp57/SEh4hSSyjh4OQ+k5utozuG8YQIteX5e/o"
"nHaidr21aNPlEvFPR9Vax2BFa6tljUOwsDuAKnb7obXZCzNek3HVHXBvyHrnLLh7LzSpLH9375Um"
"jHwhGja/WtobSvldsR4BPddYpUBlFg2l2djTZfm7ee+86uK5Dd3Dr9Ne7DTs656EXZ2fYVvnX6ju"
"8BOq1j4G65pHSdMRWNofhDlpcvU+S3U5A5Xl/mENk8+Hb0Am7Z1SyB9YrzP4XSIHeWk9VWxTodZO"
"IMRkleXv1+xYtm+TwwgfcAd13S/Byf0iHF3Po7bzOfLJM6hRjzQ5Zeg12TimoxppcvE+Q/ynfjF/"
"gSIePk2vkDeshRH5QrHfrSr2O2XSB01Va28Cra+E2Jyy/APbn8tu3uY0Bg7LhrvvTVpXrxO/a3D2"
"vIp6HpdJzyXUcbsABxdW01m9JmfPU9S/k7+Yv5D25Z7+F6Agb+D8jrxO73epH9GUol+X2PWV1tn8"
"svw797yZ1TH4V3w/OgcNm96HT+N78GqURde/DY+Gt/SaXH1+hYvXNdT3yERd0lTfIwMi1ZfzFyln"
"w4PuAzTWO6i2t0JtuaW037FepyWvM0slTWvh6L4H7PqqtZ6ZXZZ/r29+vxzWLxujol7Q/esTNGn5"
"GI1bPIJ/82z4BTxAwyakqcldTpPfb7Tu3KK182dae6fS/Ut85fvXZ56J2Dgebr7nqJf26/3OjNYz"
"g99prLfpNalYTRakyWITXH0P0b1nHGHurbL8h/zwct/g715gzLh8dA7OQ6t2LxDYJhctWj9Ds6Ac"
"BASymh6Rpt/Js0kT5aiu2wnITGeQjtmV9s+6bnFamUkCecAZVKt9XO8Pem8o5Xd7OE1Vd8HCfhfc"
"/Y/AynY+LG3mHS/LP3L828WjogsQFfMX+n7zJzp0fYX2nfPRttNLtO6Qh6D2L0jTc7QkTc1JU0Cr"
"HDi5pENVZQ6s7adUev2ytJ1WU1VlMdXjKd7vTpFvG/yO/IH1u1rFflfL+Rg8Gh1FtRqLad+4KLks"
"/0kzioZMnKbD+El/Y+SYt+geUoCuPQrQJfgvdO7O6nmNDl1eoR1patOR01TX9QTMrBeiSrXYSu8f"
"tNaTXU2tV1IPnaJ91OUPfufg8gtx5f2O9XAn1u9OwLPxGbpXOgSb2sthU2vZyLL8Z8T97TUzHpgy"
"U4epM98jrPc7hPR6h57hb9Ej7A2CQ9+gW09WE+kJ/hOdSJMz7d+s7JOh1I6r9P5NoY1pYWW/HvU9"
"T1Mv3YS7zw2qJfI7r6uk6QrqNuD8Tu/hLmfRulMmeeIe1Kq3GjXqJPmX5T9iVLrR/KXIS1gELF0B"
"DBhcSHVUiD79/0bvfn8jvO87hPVhNb39oMnT9zzsHTew/Cu9f5abxoTb19kJV69zaBhwH75N7sG7"
"8R14+WdRnG8Ve7gXqykTof1ug9ZYOHtsLXCsv+Kj9zArVmH9smRg+Upg3AQdBg/VYdCQInzzbRHp"
"KUL/gcWaevX9G40CrqGO824ozMZW+v5FajIuuo7LYZ2XXyb5wxM0DXys94dGzR/Cj/xB7+Hkd96N"
"7pAf3kFo39v03lHy9UO7y7t/WbO+sPOa9UDyWiBlDfD9KGD4D4QROgz9Tochw0nTsCK9poGkqW2H"
"B1RDh3Uy07EpleUv0USnuHhmFDZufod66Q/O79rloiV5Q/PWT8kfiv1uwJAn6NLzOlq2PYuAoIwe"
"5fGfOiPdaMtOZG/aBuzaA0yeBkRGA6OjgJGRwA+jgREjge8idBhGmnr1zYe7V7pOZhJ18Qv4/+Lm"
"dVrXpsNT6qXXer/r0CWf/OEl+cMfpInzu/ZdnuvXpK4ht9Cpx/WcZkH7pJ+6h997sChyzwFgB/Ff"
"twmYOBmIiaV6mgiMjQF+HAeMGQuM+pH0ROrg4nYUctOx75oFxasryj0gMF6tMJvwrj6tHz3C3/D+"
"UFDsD+QNHbtxfjc84jUhD30GPUZY/wcTPvf3k207nyiOpePR0ePAvzKARUuB6bOBqTPIm6YDsVOB"
"CVN4TROAJs3OkqctgcZybEhF+astxnWyc1wD30YXuV7i/SG0N+cPej2E3v0LMG32G4yIfI1hEfk5"
"fQfdVn+OP4uTZwo7nzwDZJziMG8hMHceMCcBmDUXmDkHmDaLNM0knxr0CF4N9+sUZj8erSh/qrdt"
"DXyO6zp1e4RBQw3+UIh+rD8MIL/Te947zIp7j8kz3iM69j3GxBSEVoS7AZeuYPGFS8AvF4ADh4DF"
"y4CFlIv5izk98fOBONI0K04H/8Y/QWU+DqbWUQGf425qFe2nMY8t9PT5mfqoSN9LBn/4dhjreZw3"
"xE7RIWGhDtMpXlPnYHFluLM4fCRLev0GLl/7Fbh6DdhJ/bCS9dYkYBn565LlvKYl1M8//A5Pn+3k"
"o2OyazjF2pXH3dFlsrXcdMw9D5+9VCvZiGJ7KZrrJdYfRoxivQGYNFWHpFXAArp+wmJcHjsx65M9"
"Wx5u3nrrcOcusn+7Ddy4AezbD6xJJW8lf00mf01KARJJ0wrSFBqeCRePtVBox+QZW0WNtnGY4GDg"
"zWrSWEZFyM3G5NZzS0WnLle5XiJ/m0i9NJ56KZr1h/GUU8rr5q2ch9N6lLNoxVuHL+FuwO2sv6wf"
"PsTV+w+A21nU0yc5X0rbAKSuIz1pwOpUVgvV8eDraNbiAOq7J0FtHkNaolg9VFvRtEdNQqMmBxHe"
"+wZmxxVhDu1VZscBM8gbplMvTSF/WEnx2EsxYv17/WbkrEp7949+f/mQh5vP1TlPcfDJE+DefdJx"
"h3JxkOahOG3cDGwgPes2kh7SlDD/FSJG3UVY+CWEhF0kXNJj2Pd3iecrLKG9yeLlXN0t4HuJrcfD"
"R4FDR+i61Gu79+P05u0F1b8GdwMuXMiS5uYWRT5/jvynT4Hsh8CVTJprL7B1B7BlO5f3TVs4Tet5"
"TWye1vJ5WsWv68lU24kU6+TVHOfT5zifO5GB/GPpRZFr076s3iuCh9kvTV6+RMIff+BdTg7w8HeA"
"7Y+Tp7ncb99F2Als4zWxtbCRNG3gNbH6Dh4Gzp0HLl8BLl4GyOvekdclpP/8+rN/3/9aeJ77xu71"
"a0x59QrZ+fnAo8eclrv3gFu/cdzIv/DrdY4j62M3bnH/d+MmQL6ArDt6b5iSefVtzf8p3h9D3ot3"
"9V/lF4348zUSX79Cxqt8XKUcgXKEvDzgxQsgNxdXnz1DBvVR4qNHRcPvP/g6vfm/HdA/GKZ45KFj"
"mEns+JZh7Ngxn2GM2fEEw4jYcbLohIAdGeN8+ofRMXZvmUkMU8iOAewYUMiOb9nRjmHy2dG45DhJ"
"x44P2FFa/niCHUX/fPzcPGX5GXgbxmJd3KhjdU9i9UsfMPp4sMEsER9DvAzxM8QTD6Ql480WoT2h"
"mT4h7LP/f1TwwZ7rYc8VaRjuLJYroQMj0MQwQod9jLDhfUbY7BWhgBG04c9CNaHnfq8Zxv0Bw9ge"
"Yhg5+ztAN4Y762XLcGeH2LNh7JklwWfmF/Ac2LMucqE40FcgDd4lkIZAIBtEZfE9GNkYPQSyKEIk"
"BPSeUP4thNI+EEm6sed+ChlB892MwJM9z8Xek4uZ4vNKFX6I5H1jCIUC+Uial6pMHKeHWDkPNZ12"
"wd07HS6eR2FbeyN35kU6kTCeuAyFSBbCnnchHq2jKzMn+xArB8nFyoFb9WdbFFP4MwTcWYLqNXbS"
"PXs+OtA9QuceOege+gw9e79ASJ88ncpsKRjJDMJ0wmR9TATi7mBEHTcxwvbyis4vUQ3bJFZG0tzx"
"JX77SYSJ+Ub07JXPnqV57uh+Pri+1yUr94ZXqnk3uRYc2P7eizDiJVYs4uKkP3cyG4w+dr3peY/1"
"FZlbZhwRIdNEQqReUOJ3Gu73y2ZBv8HR7TLUVsdblf1eLedzjQPb/w4v/3P8eQv2+/MJCaRjFNVJ"
"ONVJyLBPza00jbJSmIwpUJovpc8azhJxkGtS0TH4KQSqPTfK+37V2qcuBof/AYFkRekzExLioRwN"
"oazXO6EsvFp531drYxJMreOgqpJcam4WNrUOwKfxNar7LQnlfV+o2T/9P81be1TUxxVeXFh2l8fy"
"RsCAkEREkHcMqEh5Bx9gFCQhBwXxkHBCsSYuojaIiBIRAwFU8FmxyEtEIo3WgmKJGk8rpQpKRCsg"
"kYeiNCgPeXy981uWU8Ct9vSfDmc4wOydOzP3zsx37zd4LW6FkWnF1DcPwp2YRv5E/pz7KlktwySR"
"lkFSr6llAfiMd2LzF4xV6sPW+RpmWl6kfvLCFS6gsDTE2rEWtk7XJthNVnOhpJYAZXFkr7J47RRf"
"1DX+OtTYPBNmViVjXDLjxkm3iKowBy6L6qBnUko/H3JXrP/kPGOLKiz0aJiyftx4xKlQUY+lGhU0"
"WdTwrW+y5jgUwejt0nGehC85immaMg5rkc8diiMZ559lqVj/UXMtw9PwXnz/FfpzoCTKhrJGAlQ1"
"Ps+YLGpinl2zwOsiplt8x/EDjO8Q6xdCyHHVJ+Dp38Rxujxhmr5i/ft1BBpH4R/YOm63CfrJpkLt"
"vRBJflMzWdTc8nBbQHAtzf8CDGaeg57pHzhOU8vkDMfDMP3c2yXhLoli/ekSZbV98F9O+kXM5w7I"
"3lgJxir9rDk9B2ra0rbJonOdS3tDIxphYU1+ZlUD01mXOS7P2KKS4x7cfRvH3n5uFyjWv5vPE6Zy"
"+uV2k/F0cl86AAPz44xn7J0sutCrGlExrbBxuoU5jnWYbVeLWbYs5/4jl3N3875N9qPz/XVFmAzf"
"ZQ8g0js5zscrS46BP8a1mVkVQcsgcUo/y4Lqe7+M7+Lysyzf7OR6F/YujbB9rwHWNKb5HvVQEm99"
"vX7RNnj534OWMbNbKTQ4Xp18icbDeDMrx3LGk02Z/5qo9rbN2xgv8xjuPh1w83qEBR5tcHVvwTy3"
"B3Bxv0nnxxZWFdqf1kfCeEtmK8aTyzkjXdOzHF+kb1YOO5cKGMxIm2L/DfF9VxN29GN15HMsDvgn"
"/Jb2wHdJN7w+eAIPvy68z/hYja20B+IV6p8m3ipR1kjEAs8GjktldjObXcPxJIzLsrSvhtP8s4zn"
"meL/SV8jKyUNiP/tEFaGDGDFqn4EruzDsg97sWT5L9x7DqE2nV/qm0wV6lfbbC7WTeHWitnNxvkW"
"rMY4HMbvunn/FQ4upwkv5EzZ/+n7EPpNlowjCAsfwierGVfzku7bQQR/NED2b4CGwU5agzgnRfqV"
"1ePnaRtnkP6GcbuxPL8j86X3byNkdSPs55XByr7ww8myR/IgOXgMvSwvII0fRdSno4iMGkXEuhGs"
"WTtCPnUfejMyIJDEBSjSr6KxKcTI/DD5zr1xu7l7t3Ocqw/dzaERTWSbS70uiy68EosUlWIvVS6m"
"/1IKrGe5t/VA9K+B5as68bZ1EVS1pAmK9As041Is7SrIbzqwbIXMbv7Ml5Y85XLVIWF3sXTlrRRF"
"8ucrYVJxHoMXKoGsfcA2eY59CxvDC9qP1Sx3dUWRvKok7oaDy1WyWT/HsTG7MT8KX8f4z16sje7q"
"W73u0XRF8qzUXIWU5dJ+uAJkZACpe4AUllfeCcz3vAFNwwSIdDZO8QGhttRV1ySN1vfGBLuFRw5h"
"V+oQNieyXPFAzH/SLS+1dShnOeIfrwMHc4HsbCAzk+VTO8mfT0NNT9oq1t0YINbZOJ2qCY0niNal"
"w8b5Aj77/MW43WLWy3Jhe2geKXtR+Ca6Wbl3H+o/3cU5eb4kn+USjwKHD40iOLQJnn6Vo7PmppM/"
"fgVdqu9YfwvXX1Vj3WcdSEqW2S1huyy/dYRkc46gZN9BvDH+ZeXnn8FvfYiU5hZZHupiFVBAeyP/"
"92x/dGFNZDPWRrXSfB8hLKIFiUnPkUVzzfgW+N1x4Oz3QNlZDBeXYft/o3dy6eqCfUcHKh61A01N"
"QO0N4I/ngTNlQHGBbEzFtF9O076prJLl665d5ziWkqpqzP1fdP97efYM07u7Efb4MUq6HuM2jekJ"
"y5s+bKPIuwWd/3iA2413UdhwByE366EYo/yfFcgLSx8kKnEphGEez32AX202oNps1qPaLGlWpS9J"
"D1erJT3811X5Z5kk+876Yf2xfrl0BdPTI+HUsjwFe7Q4k/fmeQol1Y8FSqoffaEkWFXFEwT18JQD"
"KdZc0kOxZhXFvlIez0vhHckXR87liyJuThN9SnhmE2GKxFGeiP0fTTz9vp4w2mpQf93U3weTZQXq"
"sZ4q6jGDdM+CvZEyt6ocfcfmL4RP/k736i1Y2l0a4f73RRjD3rcP0/g85bJirS0CkSS+ja+2m3Tt"
"h5HFOTrP/0bY7DsOi1NMF2A77ycYUXzLEyXTWOIoXgpro5iJw3sSg52R6rq7CdflUXxwFA6uTdAy"
"pDhHmK/DKRCW6mgZXaa/E04UUdwqSqU4JpZqZCRrNnwro0LHJJ9ix3KI9E9hoVcnBJqHJ+ApVc1T"
"dH62kyzFbaJMwjrJFCvEVLA2ho/NZjNM+mfCNRexyPcJtadNkJ8mzoW7XzfN6Qj3dk9JnAWhprSF"
"tdm9VzjM3hZZO9cTDqmTfU607eWEBSbsusj3MYR6xVDRpjhM8zjUdRP7WJOH7/fDnn7tHC5zobvd"
"zaeT4cGRifKJWOjdSTjrT9CecY7wH8N6ezj54NArbStCnnKcNOOiF1A/As2JOFRFI4n7+7u2tZg5"
"5zpmvFuFGRb7ufFHx9aXREV3070yjE/WDNL6tUFNN3lcnnAZX6yzC64erRzucCDMYeP8A2bb5pWw"
"9q+SWsO2JLRj81Zgw0bAZ2kHnd+ZDE/oyHTH6euYZBNGegi/ZU+590ALPS/Bxa0sjLWnZz8T7Ml4"
"0pKZNYj0DPb+4AXYXU/3XghnO4k0xMK6HIFBPYSbBhD0cQcCgy83LV5exZeP8dCxfvfcIy8GT+QP"
"4QTdC6HhzaM2jkXQN91LGK2QsEHbqHQTELuhD9Gxd/rXRde5Tt4DJwpHPI/nD3WXlA7jzJlhlNE5"
"XFhIdw2d+fsPDCEt/TmSUto7aL4K49STxdChKqVaVXBqpLugeBh5+S+7aHwXDhx+/kVWzi/qk2UG"
"qLdm2tXVqjzejh2yyt/+6ipvZ59lMkz2X46ERX7uOgAA";



// Set of opera next_application icon in 16, 32 and 48 sizes
static const char* opera_next_application_icon_set = 
"H4sICIwehE8AA29wZXJhX25leHQuaWNvAOybCXSVRZbHn42CjJxJy8jBpUdcR4+juIAIjQLNQUEb"
"ZJEtkU3UNrShQWxaFmnADQGRRREVxQXDKkgABVkkgNJKWAJIEgKEQMJLAknIShbyUvP/VV7B88kz"
"ScvMOXNmHufm26rq/u9at+r78Hgu8tTx3HWXR8frPItu9nhaeDye667zX//W40nRvd/+tuo68mKP"
"5zU1uFVt1MUzyFN1/xd+F/0foN+I6oikHU9d0aWiy0Rhon8TXSVqIpImPf/pV10z0b1+ai66W3SH"
"6BbR9aJrRI1E0ryngai+f+yL/bx+c4FwX+IfGz5X+/nfJ+ouGiZ645JLLlnVoEGD/ZdffnlWo0aN"
"Chs3blx69dVX+wJJ98r0rEhtTqjtAfVZp76zRaNE4aI2ftn/3a+Ty/zy/BpZnN4Z51/8+m6M/oSh"
"w1VXXTXn2muvTb7xxhvze/XqVTJu3LiKmTNn+mbMmOH74IMPKqE5c+ZUzp49u3L69Om+qVOn+l58"
"8cWKPn36lNx888356ntQY7x/2WWXdfTbDv1c7sdez2+LC2GHs3TDDTeE3XTTTeNEKQMGDDj18ccf"
"l61Zu9a3ceNGs2LFCrNgwQLz2WefGWE306ZNM++++66l9957z8yaNcvMmzfPtomOjva9PXt2We/e"
"vU8xlmgCY19IrIEkXV186623RooSwiMicr744ouKFTEx5ttvvzWbN282y5YtM6tXrzbIsW/fPpOQ"
"kGD+8Y9/mO3bt5vdu3dbilH7t956y8qFHJ9++qnkWGDenzvX16VLlxyNnQQPeF1I7Bqzwe233x59"
"zz33ZEQvWFi6a9du8+OPP1pMmzZtMsuXLzc7duwwe/bsMUeOHCmbP39+pnwlY9KkSV5oypQpGbJJ"
"5rFjx8q3bdtmVq5caT755BMj25l33nnHLF682KxctUo2+7CsRYsWmeK14rbbbmt8IbA3bdr0d3fd"
"ddfG9u3bZ3237fuKo0ePmtTUVHP48GETHx9vwPPdd9+ZDRs2GK/XWyKsma+88sprr776amvp+WpI"
"MtzPvcmTJ2fqV/LNN99Y/PiYxS55sN3nny8zX61Z6+vy6KMnxXPXnXfeedOvwS59X9+8efP48PCI"
"k3v37fd5vRkmKysLnObgwYMmLi7OpKenW3nS0tLOyDeyX3rppT6yycXnIz0bOHfu3Ozk5OQKZMD3"
"Fi5caH1u7dq1ltav32Bit2z1KcZPive+Zs2a/VMy3HvvvZe3bNlyW8+evbKSkw/7MjIyLG7wgxl/"
"OX78uJE+7T3JcEr5Z34o7I6Uf2Ik66n9+/ebnTt3Wvr888+tDbdu3Wq2bNmi47fmu20/+Lp165YF"
"BmG5orb477///k/btm3rTUhM9qWnH7eYIfD+8MMPBj/iHJmys7N9yie5Y8eO/UN1+NXm4UWLFuVI"
"5ooDBw6YpKQkqwtssGvXLhtD33//vYnXcfuOXT4wCMui2mBXn4h27dp5v1r7dWlautf6h+LSpKSk"
"WL+BH/iRR7o0ubm5RWPGjEn+6KOP6lWHX/HQYPTo0akFBQXF9FdM2/HRPzkLuyQmJtr8kHgg2ayI"
"WVUKFmHqXxPsbdq0adChQ4d9TzzxRE7igUNmm3Ig44EfzOTKQ4cOWXKxLN/KGT58+NLqsDtS29X5"
"+fl5+CQy4IeMg97RB/fgh30OHzlm+vcfkC1MCZLj8urwd+zYcWSnTp28cTt3n4nfs9fExsbavE1+"
"IJ+TZ8CuGLSETZSHsocOHTqxpvjVdqb8Jkt2s/534sQJ64vEAvbFh4gDbJKYdEB+tPsMmIRt1C9h"
"l4z1O3fuvO/ZqKhs5MbP0Qm4GWvNmjVWDuThnPvoSnNZ3p///Ocna4pfbaM0Tk5eXh62s/oAO/mI"
"OQU/YlyekRuyc/PNM5GRWcKWBMZQ+Lt27dpXMe9ds3ZdadyOnYYcR04gxzOPLlmyxMoBH+yCLNK9"
"7+WXX84ZMmRI55rij4yM7K46KUe+YvMoYxPH+BA6wyb4FEfo5MlsE7t5aynYhDEiFP6ePXt+KvLu"
"T0zyrVu3zkDr16+3WCHwY1f0zjVySE9l8ucs4W9eC/x/ULyf1K/U+SBxTDyhK3yfOIDAj2+dyM7x"
"gU0UfT7sjz76aP2+ffumjBw5Micp+aDNZ1999ZUlV5sxV+L/xDD4kU2+WzJ48GCvfOLGWuC/Oyoq"
"KkMxXILe8ReO+BH6Z15HBmSCsEW2YmX0mDHZwpjWo0ePn9V5jz/+eNt+/fp5FyxcWLw9bof1b3CD"
"1+GXn9t6B7/58ssvrWzK/cWDBg3yygaNa4r/2WefvYE+wl9MnoRc7gQ/diX3QMhBrZJ+3GsWLV5c"
"BEawBuMfOHDgUMbc+u135Zs2xZ7Fvkp1FXrW/GT9CbmoVzjyTPFVpDra+9e//rVhTfE/99xzV9BH"
"+AvxfWLX6Z1r+JKDoL1791q5kGH7jh3lYBTW54PxP/XUU9NF3oSERFvLE1f4CTUJ59S66J5zsH/9"
"9df2XL6ZL314//a3v4XVAn9D+ih/FjAm2MkL1FOcoxfyEHJxj3p8j+Q4eOiwD4yi2cH45ZPLRN7D"
"R4748G104HINWFUTW5/HZ5xc3Fd85YWHh3trit0/fzWkj2TPgw/2dfaGp+oLmyPIFU4m5NHc4AOj"
"aFUwfvnkRpH3WFpVbUbeJPfTFx1RJzI2OmdcZEEG+U+BasVa4VdtXZ8+6pvHGOgEXUD46NKlS608"
"6MnZnONBxTcYRduC8Q8bNixOdUlmVtYJ63dufUJNxZEx8U30wT13lP7zIyIi0muDH1Ie8Sp3FaJz"
"ciZYwYxe0BX6R3fuGcfU1KPmhRdeyBTW+GD8I0aMiFMOzzygfEx+IUaZZ13exCdZt2JbZHHrXOWI"
"As0rtcbfvXt3r3JLIeti1mAcP/zwQ/P222/bo9b89qia0D7n+IP0pzydKaxJwfgVf9tE3uzsHJuz"
"qEMgl5uJWc7JB9gHH4M0fxWApTbYtY6pTx/1zXfrFnTGEf9hXck1PF2ew4cShQOMorhg/PKd9SJv"
"Tm6uj3nPzR/MjcyRxBh1KLmMayeH2hZoTq8VfsVfQ/qIxyl8Hh/Cvx1e1pbMLy4PEh/wTzpwwAdG"
"0dZg/FoXfSryshahHoSon1yNiw86udx6gPlF9wpVVxFTNc6ff/rTnxrSR2Pkg5tc4Y7kCPBjCzeH"
"urynPMr+kVe0JBj/+PHjXxV5pfMzWluYU6dOqW46aYkaEF+hFnHrSFf748OPPPIIObnG85fmrsb0"
"kW0LXI4jXsnVHFlPcu7yHbJxvnPnrnIwiqYF458wYUJ/+aVXuea0fqaoqMgUFhYazZGWXE2bk5Nj"
"ZXL2kRxF1ObKQVfXFH/v3r1voI/iqxA/dPtD5DTw4zvkG1cncr5HaxH5WREYhfXpYPy6f6fysjcm"
"ZuWpM2fOmNLSUkslJSUGeYgF/AfbOPtAwn8aLPLn/6gpftWKd6N/4S9xOYEjMrg5k1xNPYF9wJ+S"
"cgS/ygGjsDYPxv/0009fPGnSpITJkyd7y8rKKisqKgxUXl5udG31js87eYqLi62NhL9M2DP/+Mc/"
"1rh+FvY/SIZM5YEy8oPLd+Q54gzs5AnkQQaOaWnpPv9+WMqoUaPOu4YR9tng17x+xufzmUBCBvIO"
"tnHywFvzXEWvXr1Oam33cE3xP/TQQ901f51U/qrAP/Abchv2JYbd2pSaDbk4KleUgU30caj1y5Qp"
"Uzq+8cYbXuXgvFD48XnmZnRGDFCba0450b59+xqvH9VW5X9UluK/kv7kMeZycgS+w/jkB0fw0dop"
"F2zC2CUUfuXAi6dPnx43Y8aMDPmJDz9h7cO6AntyBD82cHJhB82ZKe3atZtUU/xq+4Z4pJDHWAOT"
"H4glx4d5BZ8nLyGLMLAv7xW2eNUPdX9pDa92kbNmzfLKrsVuDUq8Btog0C7EgPL0sQceeGB5TfGr"
"bYzy+zFwu1wHgRs5kIm9CXQPBo1fBCZhG17d/gm1oWqPXe+++26GxqwI9iPyEHzdNTGuHJHXunXr"
"RM1LDarD/tprr9VT2wPSbS52DI4n8oPLDeRvyVExZ86cDGGKl+83qA4/pLYd1ccrHeUo//gC8TM+"
"8RR4T7mvVDrNaNOmTcfq8KtNW/lPhuK1PFAH+A62cPnOn799YACLMHWtCXZHqgdfFXmFtSTYBv49"
"w7PXxNfrr79+7L777qt2/5Y28oWjzOmuP2MxZiAPZGJ+AANYaoMdmjhxYt25c+eunzdvXoZs+xM/"
"QkfkNY5c4wPK2UXNmjXztmrVqnso7Dyjjc6L8fPzjRUg0xl4gwEstcUPKQauVw0ep7Uvex0/kQEf"
"xY/QEzHOHLps2bK0e+65x9u8efNXhLVVhw4dmijX/45z3ZuoZ2laTxxB1+QB8gJjMFbg2LpfER0d"
"nSne8WD4Z7A7ku81lh42IoPsUA5ex4c6iHkfXWIDfEI5JFM5LgM933333ZY4HzlyZIbqxwzyictl"
"zFnkfnw9IB+Xg108499///1f9f7Fkfy1gdY/0ao/iLvTxBX5gn0BajrsT1z7dVe1Z5mdfbZG5Zx7"
"Lm+5HECOQQfYQuc+xXAxPMRrlXz+qguB3ZHyXl2NG6m1UdLy5cuZO8/AH91hf3BC3HO1E7IE5trA"
"OhD9++O0UnNiOWMyNjzgdSGxB5LWomHiM0F6Oqo4zJXvlzFXOznAht7d+oaazL3rwAbOV8jLyltl"
"W7ZsyWUsxpSvV7u/f6Hogw8++J34jlRcxMlfM5R/8qTbUvnDGekceSqJC3Ij55pT8bkzkrdUdU7B"
"woULM+jLGIrTJv9TuM9HsvktwjF4/vz5U0QrRBtF3iDa6H82RW0Hqc8Fic3/7WTsz+M5d/RTpccz"
"nmOFx9OEY57HE8Yx1uOpw3FCndiLOHrC8vTHU8lxvMdT4WlS6ml77ljKscm5Y56nbQVfgbhjKsd6"
"oY+xHOv8+mN1fM7hEs6wn+M+J5fkbHtOXuOpE+ux+kCZAfpx+ir16+8spdYL1DffaV0namcNwtn/"
"/2r447se8vy/eqq+xWoq6nzppZeObdSo0eomTZoc4RuhW2655fTtt99umjZtajjXvQI9S73iiivW"
"1q1bVxb09PBUfet1rafq2yG+DeObpYuq4X+RHwPfutTXuPeJxwrVOEbrAPPyyy/b735Uf5jPPou2"
"e6Vac5k333zTjBs3zgwePNiobYX6xNDXU/V91CWec98r1fin2mrsfS1bVkya9Lr50v++iz3gVatW"
"2zUptbnbywCH1nH2+5158z4yo0ePMfTVGKNqw5Nfy5Yt64uW9uzZ03wTu9ns3h1v17/svcFT81sl"
"3wv179/fPP744/bbIc2DlWBSbWT3dNlHXvr5ctOtW3ejsRYxZk35811CVNRQszt+r51r2edgbGob"
"1s1PPPFElniHa119zZAhQ67Veb9BgwadooZgbcs+G9898V5tw8ZN5plnIo3GjK4Jb62BhoWHR5jE"
"AymqA6veATIm8z212UsvvWQGDBjQRUFeJ5DA8+KLL9o27DWxN6w6wu6vxu3cY3r17mM09pBf4v3g"
"gw9e2bFjx9N8n5CQkGjrQMZgrU4tSL0XHh5+KJi3Iz07Bn/qV3TGnjXvmuLj95i9PyYZjV0mHteE"
"4v/www9Pe3P6LPP99p12X5W9AvZtWH9T64AlIiJiZij+evY2fuL2Ntg7wT8ZY8euPWby1GlGPN47"
"H+8uXbrU79q1a2FqWobhPSN7vMTYkiVLrT2xrdY8pl+/fk+H4s8z9uipfRkD4h0Ha5jk5IPmaHqG"
"gQe8gvl37949YpRiJuFAsrUd8vNuExz+91EG+8r2HULx59mECRNsbYXtsBv7YPiOrRfTjxt4iFfP"
"YP69evV6a9kXK8wPcXH2/QwYeDfJOe9s2PuQv5snn3zytlD8lXvuGDZsmK2n2SskZjZsqPr+Bx2k"
"SodLly034jUjmL98Z+t++Xvs5s32HQe5zb0rghhTbdhTuzIUf54NHDjQrlmRm/d5+APjkaP2ohP5"
"JryC+St+0rNzT9kch98uWrTY6h470J91wWOPPWYkX8NQ/Hkm2ewaw9mNsfAf994362Q2eSs9mL9y"
"SmFefoHVG1hXrlxlseO79GNtJbuZqVOn1g/Fn2e0gb97Dwv/xYuXnH3nkp2TS/4qDOb/1FNPmVN5"
"+RY3++z4XkzMSvuebsWKGLt+ku+aULwd0QZdMQ98+GEVkZ/dOfzhFcw/MjKyML+g0OZb8gf2w2eI"
"X/yH9Vrnzp2r5U8bfMXZDT+eP/8z60sQ/OEVzF++nZ6m+GAf1OVxt08MJtYsyh3K5c+EtD/PaANW"
"9E3uxQb4UtX3It+YQ4dTiKOf2X/o0KHbsHuBdECeRW72n9gDYe134sRJ06lTJ3QXkj/PunXrZtuS"
"OwPf+2B7cjLfEIrXz/x/+PDhb5G7iour3iGA4bCwIjc5nW808S3NydeH4t+jR4+b+vTpY/iu0NnN"
"fasEDtau5BTx+ln8jxgxIoL8VlRUbPdMiGH0797VoAfme80h94bir2ctlZ/Ovg+kP7FDHsSXchXf"
"8BCv7sH8n3/++TBRIX3cz+1zkDuR6bnnnjPt27fvFoq/noW/8MIo+06HOZhYhty3OuCAh+i8tYju"
"T6OGcj7InEHe5YdPMR+1adNmQij+DzzwwOvUXsx//ndY8uPthu+aOZ8yZQr8J52Pt5//NaIydFBZ"
"WWn5goMfdiAmf//7328Lxb9Vq1a7tmzZaveXae/f/7HxSExr7NOiK0Px92MYOXbsWCsvv5KSUjv3"
"8cOfiAHJ+TMfUH3V+pFHHuFdpsXOHIjf4kvocPTo0fCP+iXeARhi0DVy8KM/xDcA8+dHU1+zJ9xN"
"sX6NeF6r8966l7lgwSLrY649P+Tn2wONuagmvP38G4jW8H0CcjgM2OLIkVT7LbNiyMaj5nz8WXl+"
"vfVV2jje+AC5V2MtDeVzv4ChDr7CmgK9s2fm/IFY2rRps2Jri50jYmO3yLf3n/UVbEcf2bFCY0ys"
"Dd/z4LhLtHrmzJl2LquKxfSzMjp5uccz2tDWL/Mdv4Z3EI4rRf0Zd9asWQnikU2sQjrP4h42FvUR"
"NbpQfP+7f2eTHdsHEy4y/q2ZtqV1YpuU1kttklcvNSy1nv6F5VmKDcurUx25tvTjjHEYj3Ht9g98"
"8sIsW/Yp+A9I13lqvk/RokWLuqIRoo2iPJHxH7keKQoL1bd169Z3iPZSg/AtkNZolcyPzEusrTSH"
"Gz3PEXUK7qv1U3vln7KPP/7EvTetpM6ZOHGi/aZI9a6PeuO9994nT1XQ3vV96KGH6orSv163wX1f"
"Y+sD1aK2FtfxUfryro46gHUj7enHc9U1T7762utaRyXYdRBrjL///e/0b+jv35Brch5xkHY800x9"
"Yxo105M819y8Om5XvPlGfamDyG1au/6kDuOa2GXtkKL5MyEp2dCPZ717907PPpVn48vVw9Tagf25"
"Ju+7bwVyVU+q31E/vooSPXPfjVGj6l55YH/qe2oP9ED9Wny6hDanXf/SsnJbizGP0q5v376+wP66"
"trmA/lBR8enA/umnNR76YX7l22Wt136Cn+vMzCyr39jYzbat+jn8S8kxzCfUI+Qdcq/rKz2z/rI1"
"E/Z13+/Sz9+/P/WVm1fJ41FRUaZDhw7Wfjo24pp5klqB3M0cRD9//7pgAXtVbs9Tbn+bubAPzzm+"
"+eZ0i5/5Fqxqf1BUJ0C/bf/yl7+UURvxk79WTpgwkTW1GT9+PLgrwUfdKSwlat8qOAZ0r70ohzUZ"
"+gn8cc3aYdCgQZnwCu4bMAbfeY4UbdQ8ljNmzBjmsxO6XicaIWoQ3KdUo6UqqmPreTz/teZmCGZu"
"xI5h8iC1ID0gvQA+X5Wa7joAAA==";


// Set of mail overlay icons in 16, 32 and 48 sizes
static const char* mail_overlay_icon_set = 
"H4sICFxq700AA21haWwyLmljbwDtmHdQ02ccxh9UBDdOFFHiwD1w7zrjwr0XKm40xIIgoAkBhDAl"
"wA+QQkBAhhaUigpqHSjaCBQHRlkFtIISBFFImKJ9+QGeF2u83mmvPfPcPffy5nnu4Pjj+/28AVTQ"
"HBoaIKc69qgC4wEwGA33k+pAEPlMT6/x3h6YSzyIdDTqe2j4/HPiWBoTs8GxIqfVfnAPmYDHMYcd"
"zwp8ex6cHA7D0cEaLnwOBK62OOrGh8dRe3h5HgXlJcAxXwpC4U8IFAYgJCSYOARhYScQHh6OiIgI"
"REVF4caNG3iaeQVZ2bnIycmF5NldvC7OQ6W0GDLpG1RUVKCmpgbv378lriN+/9m/VymllPp6Umup"
"2qK5CpphQIM6aPcm0tJqundiMBhtdHppNWu8d9FldO/ZTqdXU95Nt3sfXe3WOk13zR59dfv111Zv"
"6mtqdu3cUaN921YD/psCj/wTPuP6MfQl189ZBvEMKJ6zxsYsHDQ3g6WlJbiHTcHjWsDa2hoOtubg"
"25F5a2cDBzJv7e3t4ezsDBcXF3gInCFwdwfl4QiBQAAfL0d4eHjAi8xdb28KPj4+8PX1RUiAM4ID"
"3BDoTyEiyB7+/v4ICAggc1mIoKAgBAeTuUwcE+lEz+fQ0FCEnQim53NkZATOng7A+TOB9Ky+EuuO"
"czHh+DU+CnFxcUi65otLF8/j5pVIPLjlh4TrV/FQJIRYFETO43ggikTy7TikiOKQnhKO9N8jkCy6"
"iozUSOJTyLofjT/SziJHfA654vPIFl9GpjgRGeJkeidkpt9DRnoa8nNu4XmeCHk5aXjxJAmFT5Pp"
"PfGyIA3Fz8UoKXyMV0WZKC3KQmFhISQSCb1DSouf4E1JHspf50NWJoGsXIKysjJIy1/T+6W8vByV"
"shJIpVJUVb5GdWXDvqmuKkNNtZRYhurqanr/vK2tIq7Gu7q3qKurw7t3yl2k1NfR/i+pmZzkc9XR"
"a1avWLZk8SJ9otkjVeXz1vNaTZzPnPnDpDGjRvVpObu1fN5mx5R2ExbM6tBx6jgttSH6beVzDaMN"
"czCsxwBGd83mI0bpd5LPuxrvNFiorsPe2r7F+Mlj9bvJ59ps9qb+fbez2bu7DJw+bXEv+bwvmz24"
"9x4228hoVedBM5b2k891WcN7kphtZGi4UUVvue4n+cpum0m8z2inocHcoWs/zdkNMt67a9uW9es+"
"zeUknyul1DcU2SPvbcD7R/6aqucU8kTEVijmFAODTdi3dzfYrD0wNWHDxMQEpqamMDczhoWFBQ5b"
"mdAMc8jKHBwOh7wTD4DL5eKIjRnhGS5seBzYE6bh8XiwseHC1tYWR47YErYhfOPgAD6fDydHOzg6"
"8okdyc+EbVyt4erqCh8BB25ubvD3tqY5h/Jyg6enJ475uNOsI/Rzo3nHz88PoUInCP29G1inkXci"
"gl1w/PhxRIe50swTe8qFfov+HCH88B69cNobv0SHIjo6GhfP+uNSrD9iYmKQeNEL1+KEiI2NReIl"
"X6Rc98Htq0GIj4+nuefBrQDCO4E09yQkXMXtm/EQ3wnCo6RgPE4ORVrSSSQmJjYyUHgj/5xE5t1T"
"EIlEuJ9yGVn3ooijkZSUhOz7p4nPIDU1lbDRecJGF5D3KA55j+NpLvoz6xrxdTzLTkBWxl3CSQ9R"
"QBgpP+cO4aTfkJ2dTTjpDs1KT3Mf0rz0JDcLBU/FKCLcVJj/GC+fP4SkIAP5+fmQPM9BQUEBSiTp"
"NDOVlz4jzFSA0tJSSN+8gOxNIWGnIlRIXxI3cdMrVFW8gkwmo3mpsrISNVXlNDfV1lQQV35gptra"
"WsJM9X5L85OSnf6f8v7WUmtUq4+kqN9mwUGWnNop6nfZ323Snt3bthpsXLl4HpPJHNOe1UVRv7uT"
"8VDdNUY7DLesW67PZPZp+wOrp6J+b1feoYUa4/bt3r5544qZHXVmMVkMRf2+Hnzbw2b9egzs2nXg"
"lF6j5jKZrP6K+gMogfMR7o/Ntdw42pi6dtl8JmuQov4wivJy3dVZi6IoB0316RtWLGKPUNTXI73V"
"HQY7kD7lOLzlxCWrTEYr6o+nzJeqTqYaNbbZyNkHxivqT6BmtJjWVKfc57QeYjlBYd+i0ybqI61X"
"MVXct9tbXyN7wfOoC9/exnqnleL+30hRXymllFJKqe9SDa+l+u92G04bqP2rp/zv/97OvwCUeiy9"
"7hwAAA==";

// Set of chat overlay icons in 16, 32 and 48 sizes
static const char* chat_overlay_icon_set = 
"H4sICJ5r8E0AA29wZXJhX2NoYXRzZXQuaWNvAO1bB3hWRdY+33e/ni8JadRVWAXWVWDBVXAFaeIu"
"S+hVQATFVVxBwa4oogKC9CICCTV0QgkldAiEbuhFelM6CAgiKjD7npm5yVdT0P39/+f5z/O83Pvd"
"3Jk575kzc2fOGYgsZFDFioRrKZpehqgyEZUqpX8XIjqKZ4UKqd+dbER98MIDeAdFqAOp5/8v+ZVK"
"QAWgLtAYaATUAMoDZX5HvcJJRbJSe7LTAHJTKnkonSIsa8lr2YTrBvJY1T3DY1mKv6eRk3qjTEOU"
"jfuddI4GGkDnERRBy8gLHT32ReR2zCSXc4rGZB+kaEySvz32ueQ1VoLfMnJQH/h3zf9B3dtB7zGw"
"5xry2haSxzWe3K5kIEljDDBaY5QPvtQYqfGFRIR9BkWiLgcNRt1P/Bf17gE79YO9l1OEkUYuxyjY"
"cgQwXGOYxlCNIRqDNQZpDAQGaPTX+Bx1zkLdizFT9f4v6D4Q9Y4iL60huzWJbMZgjUHAQI0BGv3J"
"sM4hq3U1WSzrJazWZXiWir99BvQBemv00vhUwm4dAg5L0dZAtNn9N9J9GPxlOsbdSth/OH5zPw/S"
"4HYGaPQHpgF7gO3QezsZhgLf8zOi3UAK0EvjU+ATjY8lLLB/BNpy0CT87vkrdR8FW0zQuo/C7xEa"
"wyUvhaEaqcBO6LuNypbdS0IIP/Azq3Wb5rYG174an2n0AXpLsJ96ZD8M1zzvRsZhjktBPatQ3zj8"
"HgOM1hil8SUwEvrMxXUH2e1bg/QOBL9DtA1lFlBw/32u0Q9tDpQcLPKdu5FJUncrTcH9BGC85KQw"
"ViNZ/s1i2QG7f5Wn7ias1q9QZjfAdZj9N0Qjxz+tsJGbFmmeBZEZ8L+FZMO3hqT+jMkaKZKbwkTo"
"sgnXzfnWXfnSTlnGMLLI7EOiLzRM/1Q+6oAudppKylfzJxaaC96rcDdLY6bkpDBdYxrsNwv6byeb"
"bVOB9GcYxiY5rq1Wrj+JlH8G+uiXEi7YUnHMjyzA9305+o7LzNNg/56jMVsjFTqsw3UD7Lm9wPrH"
"xW2RZW02vk6g0D6aLGHA/nZpw+R82H4J9F+Ju3RgkcZCDR5z8zXSyLBthQ3XFVh3E1zWZt9F/j6a"
"QoE+ytxcUp/xuerO3xyHZTUZFsyXluUay/CXpRpLNBbL5zYbz5eZd62/1Zopx71hLKZgH52mMVVy"
"s6HvDTlHhxebbR25sRYzDMYajQzJy2pdpcHcVmAe3Czt53Dcvf52eyZaxdrHsZmCfTRVI2cMOqUv"
"zAirv9O+idzGetS3CXVvBDZI2GzrJTeFTMnP6dyBEhnkdq+9a/25LNfhcu3y8U/TR9M0csagkxZr"
"fqHFA5u6nVmobxv02wpkAV9JOBxbpJ0UwNPN64RVueln5D2PbpZ1OJxbA3x0sYb/GLThmZpXgiUi"
"4muKgL4ezx5gt4TbvQvYqbEDvLZLbpKfi9cCK/PSPxT83uM6Irxfh/TRwDFoAWySY7BERh4iL+zg"
"9R4EDgD7wWm/4hWxT8Lj2ZvNLyKC12LL70Z/Px5ch+rTjdn+afpoqDHowNzC/AIlOvoERUG3qKjj"
"wDHgqERk5BHgsOQnOWp+TucmaZdfr/8yWZfXu8fHR3n8hR6DDnBhfoESE3OGomHzmJhTVKgQ41vg"
"G4no6JOSX3T08Wx+TtdG6L/kV+vPdXBdUVEH4Je+Pro15Bh0MQc5X/lLXNxFioFecXEXgPMUG8s4"
"B5wFp7OSX0zMaYnY2DOofwPaXpwv/VevXm1r27ZtTIsWLTp16dLF6cuB6+D9TaGYI/DJvX4+6j8G"
"dyhAd+YYKAkJ31McdExIuApcofh4xmXgO+ASOF2SHJlf4SKXUBfvrdLzpf8777wT+cwzz4i+ffuK"
"hg0bfsJccvRXc0xs3EntowdzGYN7KQJcmF+gFC16k4oU+VHjBhUu/IPGdeCa5KdwlYoVu0puzzq0"
"vTDf9m/WrNnUadOmiaFDh4rGjRtPadOmTTz/jevguTEh4bSPjx7zGYNHQo7BQCle/A5wG7oxbgG/"
"gNPPGj9pfopjsWI3YINMuQfJr/83atToqTfffFMcOHBADB8+PJsD12EY6aj3go+P+o/B6OjAMXgs"
"SP977xVUsqSQ13vuEfSHPwgqUUJBcfPnFxGxFvqn5Vv/jz76yAH/P7Fx40Y/Dk7nFFlXseKXtX8q"
"H1XjL/wYDJQyZfh7KKh0aUH33y/ovvsU/vhHQaVK5XAz+UV418h9Y0HmH+jbd+TIkeLgwYOSw4gR"
"I0SrVq1EYv2OVK9eV6pevRdVq/Y5PfroaHrwwYWw12E9/nLGYHz8RckrlJQvL+ihhwTKCvrznwU9"
"8ICgP/1JgbkxR5NfQmH+/s4pkP6w/wMvvfSS1P/w4cPi0KFDEhs2bBAZGRli5syZYsqUKWL06NHi"
"k08+ER07dhS1a3+I9ldjDF6Fj6kxGE4efliAu6CKFRX+8hdBFSoIyatcOQWTX9Gi/P1NLZD+HTp0"
"cDVv3nxvZmamOHbsmMTRo0fFkSNHJB9fTsyRkZaWJl599VVRo0Yf8MiSc0s4+dvfBPpQUJUqgipX"
"VmA+jzwi6K9/FZIfo1Il9EHpk8Tr2rJlw34DgnRv2rRpSq9evcTJkyfFN998I68nTpyQOH78eK6c"
"Zs2aJV588UVRqxbHirCMCyGs+1NPCapVS9ATTyhUqyaoalVBjz8uJL/HHlP8yle4RLwWd7vD9oHf"
"2IXuE9jfz507J86ePSvOnDkjcfr0aXHq1Cnx7bff5slpx44dsi9QV0/irWIIqVtXUOPGgurUEfTk"
"k4Jq11ZgTjVrCqpRQ/XRE9VvYx0yG5iRp/5orzc4iEuXLonvvvtOXi9evCguXLggzp8/XyBOzKF9"
"+/YiMTGxUij969cX1Lq1oIYNBeYEQf/8pwLz+sc/BP3976qPmF9cPK8Lp+WqP8ZgJL63Uo9r166J"
"77//Xly9elV8+OGH4q233hLvv/++yMrKEmPGjJH+jm+1fL5u3TqRlJQk7wM5TZ8+XTRp0mRcuD5o"
"2VIQhj41aaL6gtGokeLUoIGQHBMTMY7LHSLem+amP+bMFwYNGiRu3rwpfvzxR4kbN24I/g7wOF6/"
"fr3Uie3K/bB27VqxZs0a6TPbtm0T+HYH9RP71bPPPouxUKtQKP3Z/v/+t6DnnhOE6VmCObVoIah5"
"c0HNmglq2hT9Ufcm/Gc61oMhOZi+M2nJkiXi1q1b4pdffhE///yz+Omnn8SECROkHjxPst3Hjh0r"
"+2Lq1Kliy5YtkkNKSorst3fffVdcvnzZz/fYH7GWahBuLmL7d++uOIAqtWsnCEswgitQmzbKx55+"
"Gt/pe3gdMSms/lj3LOD5/c6dOxK3b9+WYJ3Zj9jGbHvWbf/+/VK/rVu3SjvzN459a+fOnX6+xxg4"
"cCDr/0I4/VneeEMQzEP45NALLyifev55xalDB4XGTW5ifzEV/TAxnP2T09PTRaAsXLhQ2pTXc7yW"
"YN/gZ2xb9je2+7x58+Q3ety4cX6+98MPP4jevXuz/s1y0/+VVwShmwhTtrxnn3r5ZUGdOilODEzJ"
"9HjV48SxJbd7YpD+GGcdhwwZEqQ/j0H2GZ5btm/fLu3Nc/yKFSukv7POV65cEStXrgzyPe6L52HI"
"OnXqFMlNf5Zu3QSheRowQBCWj9S1q6DXXlPAVExduoAPOD1S+QBxzM8wxmGdMjlbf17rt4PzcZ/7"
"Cs8rbFsenzznsD/16NFD3nM/8BqCebGd2f99fY/XGuhXDhKFnH8CBVVKDsnJgjDt0dtvK0AFyYn9"
"rCt4Jja4iHURr+eTMSZMJGEd0o1Y199C2Pa8hqpfv36t/OjuK6NGCUK307BhiscHH6gxjmmD3ntP"
"EMxEr6F/nm7zPdV68hg9VD5Lombtw9SkaT9KTU39Vbqz//O8A9sPzK/tA2XwYEGzZytg2Mnx/fHH"
"gnr2VGOFATcI4vfGG+cxjw2CDUZJPQoqPJ7fg5EwnjhBYNyN7r4yaZIgTOsEVyVM2bJPPvtMUJ8+"
"guCycsx/+qk/vx49fsE8NhpzwMti2bJl+eLB317+VvA3HN/CtyjM2u1uhf1p+XJBq1YJwrAiTJU0"
"d64gbHUlmBsDUyEtWMCc71BS0lb617/exDz8nNzTz5kzR847e/bskd8uvp88ebLojs7jPRv2nj3K"
"lSvn/S31zk2wDaFNm/zBz3gO8xELvv/xDRo0aAG7DoZfTMG3bgN8ewnuJ0Pnnhij1ek3tvf/NlFe"
"9n/6ulFfM/X1hL4eN68fqesVMq811HM+hoJrBs8GuPbMvlr8rhTmGvieWc9xfb1CTnm9SSXl9Q59"
"pPT+2Ed/PqdVCqhJd39Oy+12S7hcrmw4nU6sicpif/AEUC0bdnth+Tffd83yBZUiRYpgP59A8fHx"
"+ObHUUzMPaivNkU5ulOsfRIVdcyn4s4VVNyVAayWKOpIowRHCsU6u5PXU0OWY3AdXBfXWaxYsTzb"
"vvfee6lkyZJUqlQpCZfreYq2JVEJ53Iq7JpP0a6UsOduIl1jKc41g0q4VlCcYwxFR7bJrofB9eZX"
"LJZe5LIOA6/FFOuYQU77YNi4P/A5MAR9kEY2Y7mCbS6eDQR6aXxKUfYk2GgZ6uiHtVO3AvbAAHJY"
"R6LtpWSzcA7fPE/BZyk4h3wY+h1GvQp8T8R7Ls7DwhmpB/AhWS0fU7w9laIMfv5BPtv+Am0no+3l"
"KM+fVN9zLbw13A+fOhS0r7DbOfb5Nan8Np/3yDnHEmufiToHkzrPkrsYlrFU2LEMbXPu3PcMAOvC"
"eYj9YWNDNtt+4jMcgWc22H6xsINhGZBH6xOpkG0hdJ1Gwfnu2bBz8PmXQBDtJZXf9D97YbUMoygb"
"1xv+zIXdkor2OU+ncrv+Zys417I7z/atVo6frSLfczDmGQqXdSL8aYz+HSjz0PYK8s/T+p6R2E5e"
"b97tc16CzwvlnBkw7Zck4TW43jEh2k+HfVjvUPlWtsm2PNtmlC27j5QPzCH/3L/K+XuM2eiLlKDW"
"vba1ZJdny1YE5PeXaHtmhYwhhfaBLM0j8JyJyt97jDR9r4RzUdHOLdn5wpy8GucMOZ7O53o2h2s/"
"SA9+l/0lx37+50U8RrruYyWch4nz7NT5pV0+eV2VL7PZdqDOjXm175N720ic//fPO6eTmXf2YM5U"
"/auE8w9xkQd1biUnp2LmihyObfK8Xz7a17knPhtong1Y65NrVTlkt7Fa9rEpnFsoEn1c5xVy8glm"
"noTPAVgsmWHbx5rY4fub37XZNmfnHM1co5lD9eo+NkXmtmQ+6wefnNbV7Dydx/M12s8I2T63jfUs"
"xyxeyWk/A+2YOe+cPFt2jhv9yzlRU1Re5w6VKKFyOcWL39K5KpWj8noPof1VYflz27yfT0xMHIM1"
"9sP8LttM5Zs412vmmHxzZ4ez2+f8BcPMzwTmZjhHxeMyl/a78X6F91OsQ2wsxrlnn853XtY5FjN/"
"dC67j03hPATnJMx8RGCeJS7udGBO3a997JeKtmzZUuzbt0/qUKvW01S9eieqUmUwVao0EfUthX1P"
"6fzldVyv4eqfL+FYfKi8AudLSpe+JMdQbv4P3nM5DsCxOY4x8l5t4sSJYhg2sBx/ad26tXjssWGo"
"6zB0uR00/3HsmePqZhzdNz9QocJNOW7Dtd+qVasHuX2IjOeHAucrOO7J+92qVYO/PxwL5/hrqHg4"
"68XnEdzuRSHbR9tfMleOF+cF7iPOX9WrV+/lQB04nsoxYTO+zbFtM65dtOg2ueYLbJ99v3///jK+"
"mF9w/Avt7wpsn+OfHHczY7kcxzVj1I8+epZ88nq+3Hczp+vXr0u8/fbbEuZ94DMTr7/+OutQLVAH"
"jgtyXI3jsr4x2fbtBeaBVTK34ts+6viJY6ccgywIOBaLsh2DHAHCsRWODXXu7B9fbdHiOsnzgcZM"
"fOfT43T7RzlGagrzNK8mfH+bwnkSlK0bqn2OHXLcp18//1gi26ZFi6uYv/icDp8TnYY54i1i3yuI"
"cM6C52y0n2u8avRoQUlJKt7EcTPfmGDbttcoMfE01at3iOCDMkaTH+E4se77d3Nr2xRun2MsHBfj"
"uF5g/Ivjep07r8Z4aSzzxOwL4WTXrl2iMzoVbRf48DTrwPGr+fMFzZihgOlE6sV/S04+i/4ZQjy/"
"jB8/XsafOT/KOQyOY3GMEO1mAK0K2nZBBPUXZdsCU4AsYCXzBWqHer9AjhNSVgPbhQyXSGSUBJxC"
"9MT47Gnh5VAO+Dc/zwCuRAtxqwbHKThZVop+j/9PVhN4GeuxJAm+J3o8H+WqkMsYT1GOLCrk2EUR"
"tuUS/JvhMnjfcT+/GCIP2I7cmMe99kzsA0aRDXsEB9aiDBvGtNUyEn9bQx4bn42sGlTaZSxAed4n"
"cdnpKDcl+2+qDt67DcV784GZAaVnQE8+F89lUvD+/KD6HXK9rs6Je+3rcc2JHdit68hp8Np6EfqK"
"dea9BUX6l1+Av/E+czL2u7Pwfs7ZYKfBuRzeG31FVsrEu9Ozy9fgD78ub80+mz0f/bIuu7zbuIh/"
"zwGn8M7XZnkpD2LRVAaLOFWe9VZ7ELt1a47tnLex7r2Db8htslnP+5WviIUOr1kqYxFkolKlSuI+"
"LAhjYmIK8zsJCUIiLk5QVOTF7PJVqlSRZUMJP+e/83u+68ySJX+WXM3ypvA9zyEMU1gXU09zbcZn"
"PaLca9DfXf3KhxLf8mxm89xChXJnZH+b5X2vJgLLs/CaxFyfPFXnRp78A8uzcJ7YzBW3bn2VfO0e"
"CH5fxVzDgOOfHHu1EN2hYPDz/wA6yc2c7joAAA==";

// Set of unite overlay icons in 16, 32 and 48 sizes
static const char* unite_overlay_icon_set = 
"H4sICC1y8E0AA29wZXJhX3VuaXRlc2V0LmljbwDtmmtIFFEUx0/UpyBayjKidEXUTGN9YImhib2x"
"sjIrKUUoLNMSrPBDD+lhilGBpmVU9PD1wdxApQjDNCv7EKtmRaayFpJuH9QeRvg4nbk7SVqbu6LM"
"QOcHP+7M7D2zZ3bhXvgzAJNgMnh5AY1aKHIBWAwAWq18rgFopWsajfl87xSAMzRhAc2hEogB83WG"
"YRiGYRiGYRiGYRiGYRiGYRiGYZj/FX3c+tpMP3v8l2nedtlK92mJnIC5WLbKASvXaof8VKXHvs9d"
"2HAsEktXOmA2zVG6T0tcCZqPjzc4D/MX3YZKcX45aJ5q+7+2XIu14W7DbEqNwa/NddhT/0icXw1x"
"VG3/N9c448uEYOx+Vip+8w/Xj+OLSA9sStmG7UUZ4vjGamfV9p+/zhV/dLaJ3r801mDjHn9siNZh"
"S+oOMUrmhbqqtv+ije743dgo+u/IO4ktKZvx9W5ffLs/EE0lF7AtPRoLw9xV23/xFk9sTQrG96e2"
"4ru4JWg8Eoq9b57iwLce8UzSWBzuqdr+70bq8FW8PxoTlw7ZkRmLvXUVov+umjuo365Tbf8SpTt9"
"sCzK969Kn5VHeZ81r6gAPFo/9sP4jINgfk9LSwYDv6fFMAzDMAzDMBON0jmf0jmd0jmb0jmZ0jmX"
"0jmV0jmT0jmRGnIe63OaiaAfRndQ5BQaMGcV45BTzCanWT9dR0aQ6bLSsYcVdS7kebKafE6WyFbL"
"niPtLdSuIG+R98g0UtoCbpN55CUylSwnCyz0Is3Vy7W5ZNaIz3LIFPl+uSNqL5IP5e/MAh+PB2Ay"
"maTMaMjEfXWQEF8tTEpohk1h4h5TzfVSzxVkIZkPyYcMNu5z9eQTsoq8D0eTDTbuUx1kO2kkDaLe"
"tn1mgJQOpYTso6i3bZ/A3+wU9bas8zOpzo6cQU6HPvqt621ep53mIDjZIzjOQjh4oMnmdXYhPcoi"
"N7OnT/SMaZ30o2UrwA8hdheOeZ1bRpdCAhEyIg5buU5J2eafSpnnaEr/9k/Zw/zp7joAAA==";


// From ico file. For the system tray button
static UINT8 opera_ico_1616[1150] =
{
	0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x10, 0x10, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x68, 0x04, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 
	0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 
	0x00, 0x1a, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x0d, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x40, 0x46, 0x00, 0x00, 0x69, 0x80, 0x09, 0x03, 0x9b, 0x99, 0x09, 0x03, 
	0x9b, 0x99, 0x00, 0x00, 0x69, 0x80, 0x00, 0x00, 0x3a, 0x4d, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x16, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x37, 0x29, 0x08, 0x03, 0x9f, 0xb3, 0x0e, 0x06, 0xcc, 0xff, 0x10, 0x06, 0xd2, 0xff, 0x0b, 0x04, 0xbb, 0xff, 0x0b, 0x04, 
	0xbb, 0xff, 0x0e, 0x06, 0xcc, 0xff, 0x0e, 0x06, 0xcc, 0xff, 0x08, 0x03, 0x9e, 0xb3, 0x00, 0x00, 0x23, 0x40, 0x00, 0x00, 0x00, 0x20, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x5d, 0x1a, 0x09, 0x04, 0xb1, 0xd9, 0x0c, 0x05, 0xc3, 0xff, 0x02, 0x01, 0x98, 0xff, 0x04, 0x00, 0xa5, 0xff, 0x06, 0x00, 0xb3, 0xcf, 0x06, 0x00, 
	0xb3, 0xbf, 0x04, 0x00, 0xa7, 0xff, 0x02, 0x01, 0x96, 0xff, 0x0c, 0x05, 0xc1, 0xff, 0x0a, 0x04, 0xb5, 0xf2, 0x00, 0x00, 0x3b, 0x4d, 0x00, 0x00, 0x00, 0x13, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x08, 0x03, 0xb3, 0xb3, 0x09, 0x04, 0xbc, 0xff, 0x00, 0x00, 0x98, 0xff, 0x04, 0x00, 0xa8, 0xff, 0x06, 0x00, 0xa5, 0x79, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0x06, 0x00, 0xb3, 0x70, 0x04, 0x00, 0xa6, 0xff, 0x00, 0x00, 0x91, 0xff, 0x08, 0x03, 0xb2, 0xff, 0x08, 0x03, 0xaa, 0xd9, 0x00, 0x00, 0x00, 0x2d, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0x00, 0x00, 0xa9, 0x30, 0x0b, 0x04, 0xc8, 0xff, 0x00, 0x00, 0xa5, 0xff, 0x00, 0x00, 0xa3, 0xff, 0x06, 0x00, 0xb2, 0xff, 0x00, 0x00, 0x00, 0x10, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x06, 0x00, 0xb1, 0xff, 0x00, 0x00, 0x99, 0xff, 0x00, 0x00, 0x98, 0xff, 0x0c, 0x05, 0xc5, 0xff, 0x00, 0x00, 0x71, 0x80, 0x00, 0x00, 0x00, 0x0a, 0xff, 0xff, 
	0xff, 0x00, 0x05, 0x02, 0xc4, 0x80, 0x07, 0x03, 0xc7, 0xff, 0x00, 0x00, 0xb5, 0xff, 0x01, 0x00, 0xb3, 0xff, 0x06, 0x00, 0xab, 0xd9, 0x00, 0x00, 0x00, 0x0d, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x06, 0x00, 0xb3, 0xcf, 0x01, 0x00, 0xa7, 0xff, 0x00, 0x00, 0xa4, 0xff, 0x03, 0x01, 0xab, 0xff, 0x07, 0x03, 0xac, 0xbf, 0x00, 0x00, 0x00, 0x16, 0xff, 0xff, 
	0xff, 0x00, 0x07, 0x03, 0xd9, 0xbf, 0x00, 0x00, 0xce, 0xff, 0x00, 0x00, 0xcb, 0xff, 0x02, 0x00, 0xc3, 0xff, 0x06, 0x00, 0xa8, 0xcc, 0x00, 0x00, 0x00, 0x0d, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x06, 0x00, 0xb3, 0xbf, 0x02, 0x00, 0xb5, 0xff, 0x00, 0x00, 0xb4, 0xff, 0x00, 0x00, 0xb2, 0xff, 0x09, 0x03, 0xc4, 0xe6, 0x00, 0x00, 0x00, 0x1a, 0xff, 0xff, 
	0xff, 0x00, 0x09, 0x03, 0xec, 0xbf, 0x00, 0x00, 0xe9, 0xff, 0x00, 0x00, 0xe5, 0xff, 0x02, 0x00, 0xd7, 0xff, 0x06, 0x00, 0xa8, 0xcc, 0x00, 0x00, 0x00, 0x0d, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x06, 0x00, 0xb3, 0xbf, 0x02, 0x00, 0xc5, 0xff, 0x00, 0x00, 0xc9, 0xff, 0x00, 0x00, 0xc6, 0xff, 0x05, 0x02, 0xcd, 0xff, 0x00, 0x00, 0x00, 0x1a, 0xff, 0xff, 
	0xff, 0x00, 0x08, 0x07, 0xfd, 0xbf, 0x05, 0x05, 0xfe, 0xff, 0x04, 0x04, 0xfd, 0xff, 0x04, 0x02, 0xea, 0xff, 0x06, 0x00, 0xa8, 0xcc, 0x00, 0x00, 0x00, 0x0d, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x06, 0x00, 0xb3, 0xbf, 0x02, 0x00, 0xd9, 0xff, 0x00, 0x00, 0xe3, 0xff, 0x00, 0x00, 0xe0, 0xff, 0x01, 0x01, 0xde, 0xff, 0x00, 0x00, 0x00, 0x13, 0xff, 0xff, 
	0xff, 0x00, 0x15, 0x15, 0xff, 0x9f, 0x15, 0x15, 0xff, 0xff, 0x13, 0x13, 0xff, 0xff, 0x0e, 0x0d, 0xec, 0xff, 0x06, 0x00, 0xa8, 0xcc, 0x00, 0x00, 0x00, 0x13, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x06, 0x00, 0xb3, 0xcf, 0x05, 0x03, 0xeb, 0xff, 0x03, 0x03, 0xfd, 0xff, 0x02, 0x02, 0xfb, 0xff, 0x01, 0x01, 0xe9, 0xcc, 0x00, 0x00, 0x00, 0x06, 0xff, 0xff, 
	0xff, 0x00, 0x25, 0x25, 0xff, 0x50, 0x25, 0x25, 0xff, 0xff, 0x23, 0x23, 0xff, 0xff, 0x1f, 0x1f, 0xfa, 0xff, 0x06, 0x00, 0xb3, 0xff, 0x00, 0x00, 0x00, 0x23, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x06, 0x00, 0xb3, 0xff, 0x14, 0x14, 0xff, 0xff, 0x12, 0x12, 0xff, 0xff, 0x10, 0x10, 0xff, 0xff, 0x0c, 0x0c, 0xeb, 0x79, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x34, 0x34, 0xff, 0xdf, 0x33, 0x33, 0xff, 0xff, 0x31, 0x31, 0xff, 0xff, 0x0e, 0x09, 0xc1, 0xff, 0x04, 0x00, 0x70, 0x66, 0x00, 0x00, 0x00, 0x13, 0xff, 0xff, 
	0xff, 0x00, 0x06, 0x00, 0xb3, 0x40, 0x0c, 0x07, 0xc1, 0xff, 0x23, 0x23, 0xff, 0xff, 0x22, 0x22, 0xff, 0xff, 0x1f, 0x1f, 0xff, 0xef, 0x18, 0x18, 0xff, 0x10, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x3f, 0x3f, 0xff, 0x30, 0x42, 0x42, 0xff, 0xef, 0x41, 0x41, 0xff, 0xff, 0x2e, 0x2c, 0xe7, 0xff, 0x06, 0x00, 0xb1, 0xf2, 0x05, 0x00, 0x9b, 0xa6, 0x06, 0x00, 
	0xa4, 0x9c, 0x06, 0x00, 0xb3, 0xef, 0x2a, 0x28, 0xec, 0xff, 0x33, 0x33, 0xff, 0xff, 0x31, 0x31, 0xff, 0xff, 0x2d, 0x2d, 0xff, 0x60, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x4d, 0x4d, 0xff, 0x30, 0x50, 0x50, 0xff, 0xdf, 0x4f, 0x4f, 0xff, 0xff, 0x45, 0x44, 0xf6, 0xff, 0x2a, 0x27, 0xd9, 0xff, 0x29, 0x26, 
	0xd9, 0xff, 0x40, 0x3f, 0xf6, 0xff, 0x45, 0x45, 0xff, 0xff, 0x43, 0x43, 0xff, 0xef, 0x3d, 0x3d, 0xff, 0x50, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x59, 0x59, 0xff, 0x10, 0x59, 0x59, 0xff, 0x60, 0x58, 0x58, 0xff, 0xaf, 0x57, 0x57, 0xff, 0xbf, 0x56, 0x56, 
	0xff, 0xbf, 0x54, 0x54, 0xff, 0xaf, 0x51, 0x51, 0xff, 0x70, 0x4e, 0x4e, 0xff, 0x10, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 
	0x00, 0x00, 0xfc, 0x3f, 0x00, 0x00, 0xf0, 0x0f, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00, 0xc3, 0xc3, 0x00, 0x00, 0xc3, 0xc1, 0x00, 0x00, 0x83, 0xc1, 0x00, 0x00, 0x83, 0xc1, 0x00, 0x00, 0x83, 0xc1, 
	0x00, 0x00, 0x83, 0xc1, 0x00, 0x00, 0x83, 0xc1, 0x00, 0x00, 0xc3, 0xc3, 0x00, 0x00, 0xc3, 0xc3, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00, 0xf0, 0x0f, 0x00, 0x00, 0xfc, 0x3f, 0x00, 0x00
};

// General drag-n-drop icon
static UINT8 opera_dnd_2427[2762] =
{
	0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x18, 0x1b, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0xb4, 0x0a, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x36, 0x00, 
	0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2b, 0x2a, 
	0x7b, 0xff, 0x15, 0x0f, 0x95, 0xff, 0x07, 0x00, 0x9d, 0xff, 0x0a, 0x00, 0xad, 0xff, 0x0a, 0x00, 0xad, 0xff, 0x07, 0x00, 0x97, 0xff, 0x1b, 0x17, 0x8a, 0xff, 0x2f, 0x2f, 0x6c, 0xff, 0xff, 0xff, 
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x20, 0x20, 0x86, 0xff, 0x08, 0x00, 0xa5, 0xff, 0x0a, 0x00, 
	0xaf, 0xff, 0x05, 0x00, 0x96, 0xff, 0x02, 0x00, 0x88, 0xff, 0x00, 0x00, 0x78, 0xff, 0x00, 0x00, 0x77, 0xff, 0x03, 0x00, 0x8a, 0xff, 0x06, 0x00, 0x99, 0xff, 0x0a, 0x00, 0xb2, 0xff, 0x07, 0x00, 
	0x9c, 0xff, 0x31, 0x30, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcb, 0xcb, 0xe5, 0xff, 0x13, 0x0e, 0xa1, 0xff, 0x09, 0x00, 0xb0, 0xff, 0x03, 0x00, 0x95, 0xff, 0x00, 0x00, 
	0x85, 0xff, 0x00, 0x00, 0x84, 0xff, 0x00, 0x00, 0xad, 0xff, 0x00, 0x00, 0xcc, 0xff, 0x00, 0x00, 0xc5, 0xff, 0x00, 0x00, 0xa5, 0xff, 0x00, 0x00, 0x7e, 0xff, 0x00, 0x00, 0x7d, 0xff, 0x03, 0x00, 
	0x90, 0xff, 0x0b, 0x00, 0xb5, 0xff, 0x20, 0x1d, 0x96, 0xff, 0xcd, 0xcd, 0xe4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xcf, 0xeb, 0xff, 0x15, 0x10, 0xa9, 0xff, 0x08, 0x00, 0xb0, 0xff, 0x01, 0x00, 0x93, 0xff, 0x00, 0x00, 0x8f, 0xff, 0x00, 0x00, 
	0x8e, 0xff, 0x00, 0x00, 0xd3, 0xff, 0x90, 0x90, 0xf3, 0xff, 0xef, 0xef, 0xfd, 0xff, 0xdf, 0xdf, 0xfc, 0xff, 0x80, 0x80, 0xf2, 0xff, 0x00, 0x00, 0xc0, 0xff, 0x00, 0x00, 0x86, 0xff, 0x00, 0x00, 
	0x85, 0xff, 0x01, 0x00, 0x89, 0xff, 0x0b, 0x00, 0xb8, 0xff, 0x23, 0x20, 0x9f, 0xff, 0xef, 0xef, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x22, 0x20, 0xad, 0xff, 0x09, 0x00, 0xb4, 0xff, 0x01, 0x00, 0x9d, 0xff, 0x00, 0x00, 0x9a, 0xff, 0x00, 0x00, 0x99, 0xff, 0x00, 0x00, 
	0xb3, 0xff, 0x80, 0x80, 0xf0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x80, 0xf0, 0xff, 0x00, 0x00, 0xae, 0xff, 0x00, 0x00, 
	0x8f, 0xff, 0x00, 0x00, 0x8d, 0xff, 0x01, 0x00, 0x90, 0xff, 0x0a, 0x00, 0xb7, 0xff, 0x41, 0x40, 0xac, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9f, 0x9f, 0xde, 0xff, 0x07, 0x00, 0xb2, 0xff, 0x02, 0x00, 0xa8, 0xff, 0x00, 0x00, 0xa4, 0xff, 0x00, 0x00, 0xa3, 0xff, 0x00, 0x00, 0xa2, 0xff, 0x00, 0x00, 
	0xd2, 0xff, 0xef, 0xef, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xdf, 0xfb, 0xff, 0x00, 0x00, 0xc8, 0xff, 0x00, 0x00, 
	0x98, 0xff, 0x00, 0x00, 0x97, 0xff, 0x00, 0x00, 0x96, 0xff, 0x02, 0x00, 0x9c, 0xff, 0x07, 0x00, 0xae, 0xff, 0xbf, 0xbf, 0xe4, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x22, 0x20, 0xbc, 0xff, 0x06, 0x00, 0xb6, 0xff, 0x00, 0x00, 0xaf, 0xff, 0x00, 0x00, 0xad, 0xff, 0x00, 0x00, 0xac, 0xff, 0x00, 0x00, 0xae, 0xff, 0x30, 0x30, 
	0xe2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x1f, 0xdf, 0xff, 0x00, 0x00, 
	0xa1, 0xff, 0x00, 0x00, 0xa0, 0xff, 0x00, 0x00, 0x9f, 0xff, 0x00, 0x00, 0x9e, 0xff, 0x07, 0x00, 0xb4, 0xff, 0x50, 0x50, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xcf, 0xcf, 0xf3, 0xff, 0x06, 0x00, 0xba, 0xff, 0x02, 0x01, 0xba, 0xff, 0x00, 0x00, 0xb8, 0xff, 0x00, 0x00, 0xb7, 0xff, 0x00, 0x00, 0xb6, 0xff, 0x00, 0x00, 0xbe, 0xff, 0x40, 0x40, 
	0xe2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x40, 0xe2, 0xff, 0x00, 0x00, 
	0xb3, 0xff, 0x00, 0x00, 0xa9, 0xff, 0x00, 0x00, 0xa8, 0xff, 0x00, 0x00, 0xa7, 0xff, 0x02, 0x00, 0xab, 0xff, 0x04, 0x00, 0xb1, 0xff, 0xef, 0xef, 0xfa, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x92, 0x92, 0xe7, 0xff, 0x08, 0x03, 0xc2, 0xff, 0x05, 0x05, 0xc6, 0xff, 0x04, 0x04, 0xc4, 0xff, 0x04, 0x04, 0xc3, 0xff, 0x03, 0x03, 0xc1, 0xff, 0x01, 0x01, 0xc7, 0xff, 0x40, 0x40, 
	0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x40, 0xdf, 0xff, 0x00, 0x00, 
	0xbc, 0xff, 0x00, 0x00, 0xb3, 0xff, 0x00, 0x00, 0xb2, 0xff, 0x00, 0x00, 0xb1, 0xff, 0x00, 0x00, 0xaf, 0xff, 0x04, 0x00, 0xb8, 0xff, 0xaf, 0xaf, 0xe6, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x84, 0x84, 0xea, 0xff, 0x0a, 0x07, 0xcd, 0xff, 0x09, 0x09, 0xd2, 0xff, 0x09, 0x09, 0xd0, 0xff, 0x08, 0x08, 0xcf, 0xff, 0x08, 0x08, 0xcd, 0xff, 0x04, 0x04, 0xcf, 0xff, 0x40, 0x40, 
	0xdd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x40, 0xdd, 0xff, 0x02, 0x02, 
	0xc3, 0xff, 0x02, 0x02, 0xbd, 0xff, 0x01, 0x01, 0xbc, 0xff, 0x01, 0x01, 0xba, 0xff, 0x00, 0x00, 0xb9, 0xff, 0x05, 0x00, 0xc0, 0xff, 0x7f, 0x7f, 0xdb, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x5a, 0x5a, 0xea, 0xff, 0x0e, 0x0b, 0xd6, 0xff, 0x0e, 0x0e, 0xde, 0xff, 0x0d, 0x0d, 0xdd, 0xff, 0x0d, 0x0d, 0xdb, 0xff, 0x0c, 0x0c, 0xda, 0xff, 0x06, 0x06, 0xd4, 0xff, 0x40, 0x40, 
	0xdb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x40, 0xdb, 0xff, 0x05, 0x05, 
	0xcc, 0xff, 0x06, 0x06, 0xc9, 0xff, 0x06, 0x06, 0xc8, 0xff, 0x05, 0x05, 0xc6, 0xff, 0x04, 0x04, 0xc5, 0xff, 0x06, 0x03, 0xc6, 0xff, 0x81, 0x81, 0xe1, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x4f, 0x4f, 0xf2, 0xff, 0x12, 0x11, 0xe6, 0xff, 0x13, 0x13, 0xeb, 0xff, 0x12, 0x12, 0xe9, 0xff, 0x12, 0x12, 0xe8, 0xff, 0x11, 0x11, 0xe6, 0xff, 0x08, 0x08, 0xd9, 0xff, 0x40, 0x40, 
	0xd9, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x40, 0xd9, 0xff, 0x09, 0x09, 
	0xd4, 0xff, 0x0b, 0x0b, 0xd6, 0xff, 0x0a, 0x0a, 0xd4, 0xff, 0x0a, 0x0a, 0xd3, 0xff, 0x09, 0x09, 0xd2, 0xff, 0x09, 0x08, 0xd0, 0xff, 0x83, 0x83, 0xe7, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x7d, 0x7d, 0xfc, 0xff, 0x18, 0x18, 0xf9, 0xff, 0x17, 0x17, 0xf7, 0xff, 0x17, 0x17, 0xf6, 0xff, 0x16, 0x16, 0xf4, 0xff, 0x16, 0x16, 0xf3, 0xff, 0x0b, 0x0b, 0xdd, 0xff, 0x40, 0x40, 
	0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x40, 0xd6, 0xff, 0x0c, 0x0c, 
	0xdd, 0xff, 0x0f, 0x0f, 0xe2, 0xff, 0x0f, 0x0f, 0xe1, 0xff, 0x0e, 0x0e, 0xe0, 0xff, 0x0e, 0x0e, 0xde, 0xff, 0x0d, 0x0d, 0xdd, 0xff, 0x86, 0x86, 0xed, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x90, 0x90, 0xff, 0xff, 0x21, 0x21, 0xff, 0xff, 0x1f, 0x1f, 0xff, 0xff, 0x1e, 0x1e, 0xff, 0xff, 0x1d, 0x1d, 0xfe, 0xff, 0x1b, 0x1b, 0xfe, 0xff, 0x0d, 0x0d, 0xe2, 0xff, 0x40, 0x40, 
	0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x40, 0xd4, 0xff, 0x0f, 0x0f, 
	0xe6, 0xff, 0x14, 0x14, 0xef, 0xff, 0x14, 0x14, 0xed, 0xff, 0x13, 0x13, 0xec, 0xff, 0x12, 0x12, 0xea, 0xff, 0x12, 0x12, 0xe9, 0xff, 0x96, 0x96, 0xf4, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xd8, 0xd8, 0xff, 0xff, 0x30, 0x30, 0xff, 0xff, 0x2e, 0x2e, 0xff, 0xff, 0x2c, 0x2c, 0xff, 0xff, 0x2a, 0x2a, 0xff, 0xff, 0x29, 0x29, 0xff, 0xff, 0x1b, 0x1b, 0xed, 0xff, 0x40, 0x40, 
	0xd3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x40, 0xd2, 0xff, 0x13, 0x13, 
	0xee, 0xff, 0x19, 0x19, 0xfb, 0xff, 0x18, 0x18, 0xfa, 0xff, 0x18, 0x18, 0xf8, 0xff, 0x17, 0x17, 0xf7, 0xff, 0x16, 0x16, 0xf5, 0xff, 0xd3, 0xd3, 0xfd, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x56, 0x56, 0xff, 0xff, 0x3d, 0x3d, 0xff, 0xff, 0x3b, 0x3b, 0xff, 0xff, 0x39, 0x39, 0xff, 0xff, 0x37, 0x37, 0xff, 0xff, 0x28, 0x28, 0xef, 0xff, 0x30, 0x30, 
	0xcc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x30, 0x30, 0xcc, 0xff, 0x21, 0x21, 
	0xf7, 0xff, 0x24, 0x24, 0xff, 0xff, 0x22, 0x22, 0xff, 0xff, 0x20, 0x20, 0xff, 0xff, 0x1f, 0x1f, 0xff, 0xff, 0x39, 0x39, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbc, 0xbc, 0xff, 0xff, 0x4c, 0x4c, 0xff, 0xff, 0x4a, 0x4a, 0xff, 0xff, 0x48, 0x48, 0xff, 0xff, 0x46, 0x46, 0xff, 0xff, 0x45, 0x45, 0xff, 0xff, 0x00, 0x00, 
	0xbe, 0xff, 0xef, 0xef, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xef, 0xfb, 0xff, 0x07, 0x07, 0xc6, 0xff, 0x34, 0x34, 
	0xff, 0xff, 0x33, 0x33, 0xff, 0xff, 0x31, 0x31, 0xff, 0xff, 0x2f, 0x2f, 0xff, 0xff, 0x2d, 0x2d, 0xff, 0xff, 0xaf, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x79, 0x79, 0xff, 0xff, 0x59, 0x59, 0xff, 0xff, 0x58, 0x58, 0xff, 0xff, 0x56, 0x56, 0xff, 0xff, 0x54, 0x54, 0xff, 0xff, 0x20, 0x20, 
	0xd4, 0xff, 0x90, 0x90, 0xe1, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x80, 0xdd, 0xff, 0x1e, 0x1e, 0xd8, 0xff, 0x43, 0x43, 
	0xff, 0xff, 0x42, 0x42, 0xff, 0xff, 0x40, 0x40, 0xff, 0xff, 0x3e, 0x3e, 0xff, 0xff, 0x6c, 0x6c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe2, 0xe2, 0xff, 0xff, 0x72, 0x72, 0xff, 0xff, 0x67, 0x67, 0xff, 0xff, 0x65, 0x65, 0xff, 0xff, 0x63, 0x63, 0xff, 0xff, 0x56, 0x56, 
	0xf6, 0xff, 0x06, 0x06, 0xbc, 0xff, 0x80, 0x80, 0xdb, 0xff, 0xbf, 0xbf, 0xed, 0xff, 0xbf, 0xbf, 0xed, 0xff, 0x80, 0x80, 0xdb, 0xff, 0x06, 0x06, 0xbc, 0xff, 0x50, 0x50, 0xfb, 0xff, 0x53, 0x53, 
	0xff, 0xff, 0x51, 0x51, 0xff, 0xff, 0x4f, 0x4f, 0xff, 0xff, 0x58, 0x58, 0xff, 0xff, 0xe8, 0xe8, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe5, 0xe5, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0x75, 0x75, 0xff, 0xff, 0x73, 0x73, 0xff, 0xff, 0x71, 0x71, 
	0xff, 0xff, 0x62, 0x62, 0xf6, 0xff, 0x2b, 0x2b, 0xd1, 0xff, 0x1c, 0x1c, 0xc8, 0xff, 0x1c, 0x1c, 0xc8, 0xff, 0x29, 0x29, 0xd1, 0xff, 0x5a, 0x5a, 0xf6, 0xff, 0x64, 0x64, 0xff, 0xff, 0x62, 0x62, 
	0xff, 0xff, 0x60, 0x60, 0xff, 0xff, 0x72, 0x72, 0xff, 0xff, 0xe0, 0xe0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xb1, 0xb1, 0xff, 0xff, 0x83, 0x83, 0xff, 0xff, 0x81, 0x81, 
	0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0x7d, 0x7d, 0xff, 0xff, 0x7b, 0x7b, 0xff, 0xff, 0x79, 0x79, 0xff, 0xff, 0x77, 0x77, 0xff, 0xff, 0x76, 0x76, 0xff, 0xff, 0x74, 0x74, 0xff, 0xff, 0x72, 0x72, 
	0xff, 0xff, 0xad, 0xad, 0xff, 0xff, 0xf6, 0xf6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf1, 0xf1, 0xff, 0xff, 0xbe, 0xbe, 
	0xff, 0xff, 0xa8, 0xa8, 0xff, 0xff, 0x8a, 0x8a, 0xff, 0xff, 0x8a, 0x8a, 0xff, 0xff, 0x88, 0x88, 0xff, 0xff, 0x8e, 0x8e, 0xff, 0xff, 0xa2, 0xa2, 0xff, 0xff, 0xc7, 0xc7, 0xff, 0xff, 0xef, 0xef, 
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};


static UINT8 up_arrow[414] =
{
	0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a, 0x00, 0x00, 0x00, 0x0d, 0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x17, 0x08, 0x06, 0x00, 0x00, 0x00, 0x0f, 0xe8, 0xbf, 
	0x9e, 0x00, 0x00, 0x00, 0x01, 0x73, 0x52, 0x47, 0x42, 0x00, 0xae, 0xce, 0x1c, 0xe9, 0x00, 0x00, 0x00, 0x06, 0x62, 0x4b, 0x47, 0x44, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0xa0, 0xbd, 0xa7, 0x93, 
	0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x0b, 0x12, 0x00, 0x00, 0x0b, 0x12, 0x01, 0xd2, 0xdd, 0x7e, 0xfc, 0x00, 0x00, 0x00, 0x07, 0x74, 0x49, 0x4d, 0x45, 0x07, 0xda, 0x09, 
	0x08, 0x07, 0x21, 0x1f, 0xb3, 0x0d, 0xd1, 0xe4, 0x00, 0x00, 0x01, 0x1e, 0x49, 0x44, 0x41, 0x54, 0x38, 0xcb, 0xed, 0x95, 0xb1, 0x4a, 0xc4, 0x40, 0x14, 0x45, 0xcf, 0x04, 0x95, 0x8d, 0x85, 0xc5, 
	0xda, 0x09, 0x6a, 0x61, 0xa7, 0x85, 0x5f, 0x60, 0x21, 0x76, 0x81, 0x60, 0x15, 0x09, 0x96, 0x29, 0x16, 0xb5, 0x0c, 0xa4, 0xb6, 0xb3, 0xb2, 0xb0, 0x37, 0xa4, 0x12, 0xec, 0xf3, 0x01, 0xf9, 0x08, 
	0x8b, 0x80, 0x08, 0x41, 0xb0, 0x13, 0x92, 0x14, 0x8a, 0x30, 0xb2, 0xfa, 0x6c, 0xa2, 0x84, 0x25, 0xd9, 0xcd, 0x16, 0x8b, 0x2c, 0xec, 0x83, 0x57, 0x3d, 0xb8, 0x73, 0xb9, 0x67, 0xe6, 0x8d, 0x12, 
	0x11, 0x66, 0x51, 0x06, 0x33, 0xaa, 0xf9, 0x13, 0x46, 0x44, 0xfe, 0xba, 0x65, 0x6e, 0x96, 0x65, 0x79, 0x09, 0xdc, 0x02, 0xbd, 0x49, 0x1a, 0xbf, 0x6d, 0x4c, 0x38, 0xd4, 0x04, 0xce, 0x83, 0x20, 
	0x58, 0x07, 0xfa, 0xc0, 0x4d, 0x9b, 0xf8, 0xb4, 0x51, 0x1c, 0x47, 0x51, 0xb4, 0x1d, 0x86, 0xe1, 0x17, 0xf0, 0x04, 0x6c, 0x02, 0x67, 0x5d, 0x84, 0x55, 0x3d, 0x02, 0xa5, 0x54, 0xdd, 0xed, 0x41, 
	0x92, 0x24, 0xa7, 0x96, 0x65, 0x19, 0x5a, 0xeb, 0x0f, 0x40, 0x03, 0xdf, 0xc0, 0x3e, 0x70, 0x0f, 0xdc, 0xd5, 0xa3, 0xe8, 0xe4, 0x58, 0x44, 0x76, 0xb2, 0x2c, 0x73, 0x1d, 0xc7, 0x59, 0xd1, 0x5a, 
	0x0f, 0x6b, 0xa3, 0x21, 0xf0, 0x00, 0x9c, 0x00, 0x87, 0x53, 0x45, 0x21, 0x22, 0x1b, 0x79, 0x9e, 0x0f, 0x6c, 0xdb, 0xee, 0x15, 0x45, 0xd1, 0x44, 0xf4, 0x1d, 0x78, 0x04, 0x2e, 0x80, 0xad, 0x4e, 
	0xc2, 0x15, 0x2c, 0xd7, 0xf3, 0xbc, 0xb5, 0x34, 0x4d, 0x97, 0xc7, 0x18, 0x7a, 0x05, 0x5e, 0x80, 0xab, 0x36, 0x98, 0xa3, 0x8e, 0x5d, 0xdf, 0xf7, 0x77, 0xe3, 0x38, 0xee, 0x77, 0xe0, 0xf3, 0x0c, 
	0xbc, 0x55, 0xe2, 0xe3, 0xe1, 0x35, 0x00, 0xbc, 0x56, 0x4a, 0x99, 0xc0, 0x67, 0x05, 0x6f, 0x15, 0xd8, 0x03, 0x8e, 0x46, 0xef, 0xf1, 0x62, 0x57, 0xb4, 0xd6, 0x52, 0xd3, 0xbb, 0xef, 0xb2, 0x5f, 
	0xfe, 0xcd, 0xb1, 0x5a, 0xfc, 0x20, 0xf3, 0x2b, 0xfc, 0x03, 0xf4, 0xa5, 0x81, 0x7b, 0xd6, 0xb4, 0xca, 0xb6, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45, 0x4e, 0x44, 0xae, 0x42, 0x60, 0x82
};


static UINT8 left_arrow[413] =
{
	0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a, 0x00, 0x00, 0x00, 0x0d, 0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x16, 0x08, 0x06, 0x00, 0x00, 0x00, 0x2b, 0x76, 0x07, 
	0x05, 0x00, 0x00, 0x00, 0x01, 0x73, 0x52, 0x47, 0x42, 0x00, 0xae, 0xce, 0x1c, 0xe9, 0x00, 0x00, 0x00, 0x06, 0x62, 0x4b, 0x47, 0x44, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0xa0, 0xbd, 0xa7, 0x93, 
	0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x0b, 0x12, 0x00, 0x00, 0x0b, 0x12, 0x01, 0xd2, 0xdd, 0x7e, 0xfc, 0x00, 0x00, 0x00, 0x07, 0x74, 0x49, 0x4d, 0x45, 0x07, 0xda, 0x09, 
	0x08, 0x07, 0x21, 0x0b, 0xa9, 0xd7, 0x05, 0x99, 0x00, 0x00, 0x01, 0x1d, 0x49, 0x44, 0x41, 0x54, 0x38, 0xcb, 0xed, 0x95, 0xb1, 0x4a, 0xc3, 0x50, 0x14, 0x86, 0xbf, 0x5b, 0x4a, 0xd3, 0x74, 0x0a, 
	0x8e, 0x82, 0x2e, 0x6e, 0x79, 0x07, 0x29, 0x2a, 0xae, 0x71, 0x8a, 0x93, 0x8b, 0x90, 0x41, 0xdc, 0x22, 0xbe, 0x84, 0x0e, 0x52, 0x10, 0x71, 0x48, 0xb3, 0xb8, 0xf9, 0x12, 0x79, 0x07, 0x15, 0x07, 
	0x87, 0x40, 0x41, 0x5d, 0x24, 0x24, 0x0e, 0xe2, 0x24, 0x1c, 0x97, 0x2b, 0xc4, 0xd0, 0x36, 0x37, 0x9a, 0x6e, 0xfd, 0xe1, 0x9f, 0xee, 0xb9, 0x1f, 0x3f, 0xe7, 0xdc, 0xc3, 0x55, 0x22, 0xc2, 0xa2, 
	0xd4, 0x61, 0x81, 0x5a, 0xc2, 0xa7, 0x4b, 0x44, 0x7e, 0x79, 0x8e, 0xfa, 0xc0, 0x45, 0xdd, 0xfd, 0xb2, 0x4d, 0x93, 0xaf, 0x03, 0x63, 0xc0, 0x6e, 0x12, 0xbc, 0x6b, 0x50, 0xb3, 0x05, 0x1c, 0x03, 
	0xcf, 0xc0, 0xa4, 0x4d, 0xf8, 0x01, 0xb0, 0x0f, 0x3c, 0x01, 0x6f, 0x80, 0xd5, 0x04, 0xae, 0xaa, 0x7d, 0x56, 0x4a, 0xfd, 0xf4, 0xf7, 0x08, 0xd8, 0x05, 0xee, 0x81, 0x8f, 0x52, 0xc9, 0xf6, 0x8c, 
	0xd9, 0xed, 0x98, 0x24, 0xef, 0x03, 0x23, 0x60, 0x0d, 0xb8, 0x03, 0xbe, 0x2a, 0x89, 0x1f, 0xff, 0xd3, 0x96, 0x4b, 0x60, 0x05, 0x78, 0xd0, 0x4f, 0xb5, 0x57, 0x39, 0xff, 0xfc, 0xf3, 0x3b, 0x2f, 
	0x8a, 0xe2, 0x25, 0x08, 0x82, 0x57, 0x9d, 0xb6, 0x89, 0xeb, 0xe1, 0x8e, 0xe3, 0x9c, 0x47, 0x51, 0x34, 0x89, 0xe3, 0xb8, 0x63, 0x59, 0xd6, 0x40, 0x27, 0x37, 0xb1, 0xd9, 0x40, 0x45, 0xc4, 0x06, 
	0xf6, 0x92, 0x24, 0x19, 0xfa, 0xbe, 0xdf, 0xcb, 0xf3, 0x5c, 0x4a, 0x83, 0x7b, 0x9f, 0xd1, 0x85, 0xd3, 0x46, 0x1b, 0x2a, 0x22, 0x9b, 0x69, 0x9a, 0x5e, 0xb9, 0xae, 0x1b, 0x03, 0x37, 0xda, 0xc6, 
	0x1b, 0x5a, 0xbb, 0xfe, 0x22, 0xb2, 0x91, 0x65, 0xd9, 0x99, 0xe7, 0x79, 0xd7, 0xc0, 0x6d, 0xab, 0x70, 0x0d, 0x58, 0x15, 0x91, 0x93, 0x30, 0x0c, 0xc7, 0xad, 0xc3, 0x35, 0xc4, 0x16, 0x91, 0xc3, 
	0x79, 0xb0, 0xaa, 0xd5, 0xf2, 0x9b, 0x9b, 0xa6, 0x6f, 0x8d, 0xaa, 0x09, 0x7a, 0xf2, 0xf5, 0x56, 0xb5, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45, 0x4e, 0x44, 0xae, 0x42, 0x60, 0x82
};

static UINT8 cross[318] =
{
        0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x10, 0x10, 0x10, 0x00, 0x01, 0x00, 0x04, 0x00, 0x28, 0x01, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 
        0x00, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x39, 0x34, 
        0x33, 0x00, 0x58, 0x53, 0x51, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 
        0x20, 0x00, 0x00, 0x02, 0x22, 0x22, 0x22, 0x22, 0x20, 0x11, 0x11, 0x02, 0x22, 0x22, 0x22, 0x22, 0x20, 0x11, 0x11, 0x02, 0x22, 0x22, 0x22, 0x22, 0x20, 0x11, 0x11, 0x02, 0x22, 0x22, 0x22, 0x22, 
        0x20, 0x11, 0x11, 0x02, 0x22, 0x22, 0x00, 0x00, 0x00, 0x11, 0x11, 0x00, 0x00, 0x00, 0x01, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x10, 0x01, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x10, 0x01, 0x11, 
        0x11, 0x11, 0x11, 0x11, 0x11, 0x10, 0x01, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x10, 0x00, 0x00, 0x00, 0x11, 0x11, 0x00, 0x00, 0x00, 0x22, 0x22, 0x20, 0x11, 0x11, 0x02, 0x22, 0x22, 0x22, 0x22, 
        0x20, 0x11, 0x11, 0x02, 0x22, 0x22, 0x22, 0x22, 0x20, 0x11, 0x11, 0x02, 0x22, 0x22, 0x22, 0x22, 0x20, 0x11, 0x11, 0x02, 0x22, 0x22, 0x22, 0x22, 0x20, 0x00, 0x00, 0x02, 0x22, 0x22, 0xf8, 0x1f, 
        0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 
};


static UINT8 menu_minimize[158] =
{
        0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x0b, 0x0b, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x88, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x16, 0x00, 
        0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x52, 
        0x4a, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xc0, 0x60, 0x00, 0x00, 0xc0, 0x60, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 
        0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xc0, 0x60, 0x00, 0x00, 0xc0, 0x60, 
        0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 
};

static UINT8 menu_restore[158] =
{
        0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x0b, 0x0b, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x88, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x16, 0x00, 
        0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x52, 
        0x4a, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x7d, 0xe0, 0x00, 0x00, 0x7d, 0xe0, 0x00, 0x00, 0x7d, 0xe0, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xc0, 
        0x00, 0x00, 0xff, 0xc0, 0x00, 0x00, 0xf7, 0xc0, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0x01, 0xe0, 0x00, 0x00, 0x7d, 0xe0, 0x00, 0x00, 0x7d, 0xe0, 
        0x00, 0x00, 0x7d, 0xe0, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x00, 0x00, 0xff, 0xc0, 0x00, 0x00, 0xf7, 0xc0, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00
};



static UINT8 menu_close[158] =
{
        0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x0b, 0x0b, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x88, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x16, 0x00, 
        0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x52, 
        0x4a, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0x8e, 0x20, 0x00, 0x00, 0xc4, 0x60, 0x00, 0x00, 0xe0, 0xe0, 0x00, 0x00, 0xf1, 0xe0, 0x00, 0x00, 0xf1, 0xe0, 0x00, 0x00, 0xe0, 0xe0, 
        0x00, 0x00, 0xc4, 0x60, 0x00, 0x00, 0x8e, 0x20, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0x8e, 0x20, 0x00, 0x00, 0xc4, 0x60, 0x00, 0x00, 0xe0, 0xe0, 
        0x00, 0x00, 0xf1, 0xe0, 0x00, 0x00, 0xf1, 0xe0, 0x00, 0x00, 0xe0, 0xe0, 0x00, 0x00, 0xc4, 0x60, 0x00, 0x00, 0x8e, 0x20, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 0xff, 0xe0, 0x00, 0x00, 
};


namespace IconSource
{
	BOOL GetApplicationIcon(UINT32& width, UINT8*& data, UINT32& size_in_bytes)
	{
		switch (g_desktop_product->GetProductType())
		{
			case PRODUCT_TYPE_OPERA_NEXT:
			{
				static IcoPackage package;
				if (OpStatus::IsSuccess(package.Init(OpStringC8(opera_next_application_icon_set))))
					if (package.Get(width, data, size_in_bytes))
						return TRUE;
			}
			// Labs and normal opera share the same icon
			case PRODUCT_TYPE_OPERA_LABS:
			default:
			{
				static IcoPackage package;
				if (OpStatus::IsSuccess(package.Init(OpStringC8(opera_application_icon_set))))
					if (package.Get(width, data, size_in_bytes))
						return TRUE;
			}
		}

		// Simple fallback for the most important icon
		width = 16;
		data = opera_ico_1616;
		size_in_bytes = sizeof(opera_ico_1616);
		return TRUE;
	}

	BOOL GetSystemTrayUniteIcon(UINT32& width, UINT8*& data, UINT32& size_in_bytes)
	{
		static IcoPackage package;
		RETURN_VALUE_IF_ERROR(package.Init(OpStringC8(unite_overlay_icon_set)), FALSE);
		return package.Get(width, data, size_in_bytes);
	}

	BOOL GetSystemTrayOperaIcon(UINT32& width, UINT8*& data, UINT32& size_in_bytes)
	{
		return GetApplicationIcon(width, data, size_in_bytes);
	}

	BOOL GetSystemTrayMailIcon(UINT32& width, UINT8*& data, UINT32& size_in_bytes)
	{
		static IcoPackage package;
		RETURN_VALUE_IF_ERROR(package.Init(OpStringC8(mail_overlay_icon_set)), FALSE);
		return package.Get(width, data, size_in_bytes);
	}

	BOOL GetSystemTrayChatIcon(UINT32& width, UINT8*& data, UINT32& size_in_bytes)
	{
		static IcoPackage package;
		RETURN_VALUE_IF_ERROR(package.Init(OpStringC8(chat_overlay_icon_set)), FALSE);
		return package.Get(width, data, size_in_bytes);
	}

	BOOL GetDragAndDropIcon(UINT8*& data, UINT32& size_in_bytes)
	{
		data = opera_dnd_2427;
		size_in_bytes = sizeof(opera_dnd_2427);
		return TRUE;
	}
	
	BOOL GetUpArrow(UINT8*& data, UINT32& size_in_bytes)
	{
		data = up_arrow;
		size_in_bytes = sizeof(up_arrow);
		return TRUE;
	}

	BOOL GetLeftArrow(UINT8*& data, UINT32& size_in_bytes)
	{
		data = left_arrow;
		size_in_bytes = sizeof(left_arrow);
		return TRUE;
	}

	BOOL GetCross(UINT8*& data, UINT32& size_in_bytes)
	{
		data = cross;
		size_in_bytes = sizeof(cross);
		return TRUE;
	}

	BOOL GetMenuMinimize(UINT8*& data, UINT32& size_in_bytes)
	{
		data = menu_minimize;
		size_in_bytes = sizeof(menu_minimize);
		return TRUE;
	}

	BOOL GetMenuRestore(UINT8*& data, UINT32& size_in_bytes)
	{
		data = menu_restore;
		size_in_bytes = sizeof(menu_restore);
		return TRUE;
	}

	BOOL GetMenuClose(UINT8*& data, UINT32& size_in_bytes)
	{
		data = menu_close;
		size_in_bytes = sizeof(menu_close);
		return TRUE;
	}
};



