#!/usr/bin/env python
import sys
import re
import array
import struct

# Script filter; if 'use_filter' == True, only scripts in 'script_filter' will be included.
use_filter = True
script_filter = ['Latin', 'Cyrillic', 'Greek', 'Hebrew', 'Arabic', 'Devanagari', 'Bengali', 'Gurmukhi',
				 'Gujarati', 'Oriya', 'Tamil', 'Telugu', 'Kannada', 'Malayalam', 'Sinhala', 'Lao',
				 'Tibetan', 'Myanmar', 'Thai', 'Han', 'Hiragana', 'Katakana', 'Hangul', 'Bopomofo', 'Cherokee']

def error(message):
	print >>sys.stderr, message
	sys.exit(1)

def get_script_code(base):
	if not use_filter or base in script_filter:
		return 'SC_' + base
	else:
		return 'SC_Unknown'

pattern = re.compile('([\dA-Fa-f]+)(\.\.([\dA-Fa-f]+))?[ ]*;[ ]*(\S*)')
table = []
reference = ['SC_Unknown'] * 65536

# Open and parse the input data
try:
	file = open('../data/Scripts.txt', 'r')
except:
	error("Error opening Scripts.txt")

for line in file.readlines():
	m = pattern.match(line)
	if not m:
		continue

	# Get range and script type from the parsed line
	c1_s, c2_s, script_s = m.group(1,3,4)

	script = get_script_code(script_s)
	if script == 'SC_Unknown':
		continue

	c1 = int(c1_s, 16)
	c2 = int(c2_s, 16) if c2_s else c1

	for c in range(c1, c2+1):
		# Only handle the BMP
		if c > 0xFFFF:
			continue

		# Update range table
		if len(table) and script == table[-1][1] and table[-1][0]+table[-1][2] == c:
			table[-1][2]+=1;
		elif script:
			table.append([c, script, 1])

		# Update reference table
		reference[c] = script

# Sort range table
table.sort()

# Generate the sorted enumeration
enum_data = set()
last_end = ent = 0
data = tbl = ""
for t in table:
	if last_end != t[0]:
		if ent:
			data += ", "
			tbl += ", "
		if not ent & 7:
			data += "\n\t"
			tbl += "\n\t"
		ent+=1

		tbl += ('0x%04x'%last_end);
		data += get_script_code("Unknown");

	if ent:
		data += ", "
		tbl += ", "
	if not ent & 7:
		data += "\n\t"
		tbl += "\n\t"
	ent+=1

	tbl += ('0x%04x'%t[0])
	last_end = t[0]+t[2];
	data += t[1];
	enum_data.add(t[1]);

enum_list = list(enum_data);
enum_list.sort()

# Add sentinel
if table[-1][1] != 0:
	tbl += (', 0x%04x'%(table[-1][0]+table[-1][2]))
	data += ', 0';

# Write the requested table
if len(sys.argv) > 1:
	if sys.argv[1] == "--enum":
		for i, item in enumerate(enum_list):
			print item+","
	else:
		print "Usage: make_script.py [--enum]"
		print "  --enum\tOutput enum-values to stdout"
else:
	# Write scripts range table
	print "/** @file scripts.inl"
	print " * This file is auto-generated by modules/unicode/scripts/make_scripts.pike."
	print " * DO NOT EDIT THIS FILE MANUALLY."
	print " */\n"
	print "#ifdef USE_UNICODE_INC_DATA"

	print "static const uni_char script_chars[] = {"+tbl+"\n};\n"
	print "static const char script_data[] = {"+data+"\n};\n"

	# Write the reference table for selftests
	# Create a backmap from enum to value
	backmap = {'SC_Unknown' : 0}
	enumidx = 1
	for enummap in enum_list:
		backmap[enummap] = enumidx
		enumidx = enumidx + 1

	out = open("script.dat", "wb")
	datfile = array.array("B", map(lambda x:backmap[x], reference))
	datfile.tofile(out)
	out.close()

	print "#endif // USE_UNICODE_INC_DATA"
