#!/usr/bin/pike
/* -*- Mode: pike; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */

#define SUPPORT_TEXT_DIRECTION
// #define UNICODE_BUILD_TABLES
#include "../unicode.h"


// Index 4: BIDI class

// Map UnicodeData.txt bidi class to enum identifier
string get_bidi_class(string cls) 
{
	return "BIDI_"+cls;
}

// Map unicode point to plane
int get_plane(int codepoint)
{
	return (codepoint & 0xff0000) >> 16;
}

// Map enum identifier to value (as read from unicode.h)
int get_bidi_class_i(string cls)
{
	return this[cls];
}

void main()
{
	array table = ({});
	string bidicls = "\1"*131072;
	int oc, oclsi;
	int last_plane = -1;

	// Read UnicodeData.txt.
	// foreach(Stdio.read_file("../data/UnicodeData.txt") / "\n"; int lno; string line) 
	foreach (Stdio.stdin.read()/"\n", string line)
	{
		// Ignore any comments and split into semi-colon separated fields
		sscanf(line, "%s#", line);
		array data = line / ";";
		if (sizeof(data) != 15)
			continue;

		// Fetch the code point value
		int c;
		sscanf(data[0], "%x", c);

		int plane = get_plane(c);

		// Only operate on planes 0-E
		if (plane <= 0xe) 
		{
			// Every plane that has characters needs to have x0000 defined
			if (plane != last_plane && (c & 0xffff) != 0)
			{
				table += ({ ({ c & 0xff0000, 0, 1 }) });
			}
			last_plane = plane;

			// Read the Bidi_Class field
			string cls = get_bidi_class(data[4]);
			int clsi=get_bidi_class_i(cls);

			if (sizeof(table) && table[-1][1] == cls && get_plane(table[-1][0]) == plane)
			{
				// Run of same class:
				// Store diff value so that we can check for gaps in the
				// table later
				table[-1][2] = (c-table[-1][0])+1;

				// Store data for selftests
				if (plane < 2)
					for(int i=oc+1; i<=c; i++)
						bidicls[i]=clsi;
				oc = c;
			}
			else
			{
				// New class:
				// Store data for selftests
				if (plane < 2)
					for(int i=oc+1; i<c; i++)
						bidicls[i]=oclsi;

				// Remember class and add an entry to the table
				oclsi=clsi;
				oc = c;
				if (plane < 2)
					bidicls[c]=clsi;
				table += ({ ({ c, cls, 1 }) });
			}
		}
	}
	
	// Add plane sentinels
	table += ({
		({ 0xffff, 0, 1 }), 
		({ 0x1ffff, 0, 1 }), 
		({ 0x2ffff, 0, 1 }), 
		({ 0xeffff, 0, 1 })
	});
	
	table = sort(table);
	
	// Find plane start points
	array planes = ({({"0", "0"})});
	for(int i = 0; i < 14; i++)
		planes += ({({"-1", "-1"})});
	int p = 0, prevp = 0, mark = 0;
	foreach (table, array t)
	{
		if (get_plane((int)t[0]) > p)
		{
			p = get_plane(t[0]);
			planes[p][0] = (string)mark;
			planes[prevp][1] = (string)(mark-1);
			prevp = p;
		}
		mark++;
	}
	planes[-1][1] = (string)(sizeof(table)-1);

	bidicls[0xfffe]=0x13;
	bidicls[0xffff]=0x13;
	Stdio.write_file("bidi.dat", bidicls);

	write(#"/** @file bidiclass.inl
 * This file is auto-generated by modules/unicode/scripts/make_bidi.pike.
 * DO NOT EDIT THIS FILE MANUALLY.
 */\n
#ifdef USE_UNICODE_INC_DATA\n");
	write( "static const short bidi_planes[][2] = {\n\t");
	foreach (planes, array p) 
		write( "{" + p*"," + "},");
	write("\n};\n");
	write("static const uni_char bidi_chars[] = {\n");
	int sz, last_255;
	int old_e;
	int entries;
	foreach (table, array row)
	{
// Unassigned code-points cause this:
//		if (row[0] != old_e)
//		{
//			 werror("Warning: Gap %x vs. %x\n", row[0], old_e);
//		}
		if (entries ++)
		{
			write(",\n");
		}
			
		write("\t0x%x", ((int)row[0])&0xffff);
		sz++;
		old_e = row[0]+row[2];

		// Remember the index to the last character in the range
		// U+0000-U+00FF.
		if (row[0] < 256)
			last_255 = sz;
	}
	write("\n#define BIDI_LAST_255 %d\n",last_255+1);
	write("};\n");

	write("static const char bidi_data[] = {\n");
	entries = 0;
	foreach (table, array row)
	{
		if (entries ++)
		{
			write(",\n");
		}
		write("\t%s", (string)row[1]);
	}
	write("\n};\n#endif // USE_UNICODE_INC_DATA\n");
}
