/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*-
 *
 * Copyright (C) 1995-2012 Opera Software ASA.  All rights reserved.
 *
 * This file is part of the Opera web browser.
 * It may not be distributed under any circumstances.
 */
#ifndef PI_DEVICE_API_OPADDRESSBOOK_H
#define PI_DEVICE_API_OPADDRESSBOOK_H

#ifdef PI_ADDRESSBOOK

typedef OP_STATUS OP_ADDRESSBOOKSTATUS;

/** @short Additional error status codes useful for reporting address book related errors.
 */
class OpAddressBookStatus : public OpStatus
{
public:

	enum
	{
		/** Item hasn't been found */
		ERR_ITEM_NOT_FOUND = USER_ERROR + 1,

		/** Field at this index isn't supported */
		ERR_FIELD_INDEX_OUT_OF_RANGE,

		/** Value at this index isn't supported */
		ERR_VALUE_INDEX_OUT_OF_RANGE,

		/** Item has not yet been registered with underlying store */
		ERR_NOT_YET_REGISTERED,

		/** No space left in the underlying store */
		ERR_NO_SPACE_LEFT
	};
};

class OpAddressBookItem;

/** @short Callback class interface for FindAddressBookItem and GetAddressBookItem.
 */
class OpAddressBookIterationCallback
{
public:
	/** This method is called when find operation has a result ready.
	 *
	 * @param result - result of a search is stored here. This value is
	 * allocated by platform(using OP_NEW) and ownership of it is handed over to the
	 * callback, so it needs to be freed (with OP_DELETE).
	 *
	 * @return TRUE if the callback wants more results or FALSE if it wants to
	 * stop the search. In case it returns FALSE platform code MUST NOT call
	 * this (or OnFindFinished/Failed) callback again!
	 */
	virtual BOOL OnItem(OpAddressBookItem* result) = 0;

	/** Called when find operation finishes successfully.
	 */
	virtual void OnFinished() = 0;

	/** Called when the find operation fails.
	 *
	 * @param error - error code.
	 */
	virtual void OnError(OP_ADDRESSBOOKSTATUS error) = 0;
};

/** Callback class interface for modifying address book operations(add/delete/update).
 */
class OpAddressBookItemModificationCallback
{
public:
	virtual ~OpAddressBookItemModificationCallback(){}
	/** Called when modification operation finished successfully
	 *
	 * @param id ID of the item being modified. In case the callback is
	 * invoked for the AddItem operation, it is the ID of the newly added
	 * item, as generated by the platform.
	 */
	virtual void OnSuccess(const uni_char* id) = 0;

	/** Called when modification operation fails. If this callback is called
	 * then this means that no changes should have ocurred in platform address book
	 *
	 * @param id id of the modified event. For add operation this is always NULL.
	 * @param error error code
	 *        - ERR_NO_MEMORY - in case of oom
	 *        - ERR_NO_ACCESS - if we dont have acccess to address book
	 *        - ERR_NO_SUCH_RESOURCE - if the event with a given id does not evist in the address book.
	 *        - ERR - any other error
	 */
	virtual void OnFailed(const uni_char* id, OP_STATUS error) = 0;
};

/** Callback class interface for OpAddressBook::GetItemCount.
 */
class OpGetAddressBookItemCountCallback
{
public:
	virtual ~OpGetAddressBookItemCountCallback(){}
	/** Called when count operation finished successfully
	 *
	 * @param count number of events in address book
	 */
	virtual void OnSuccess(unsigned int count) = 0;

	/** Called when count operation fails.
	 *
	 * @param error error code
	 *        - ERR_NO_MEMORY - in case of oom
	 *        - ERR_NO_ACCESS - if we dont have acccess to address book
	 *        - ERR - any other error
	 */
	virtual void OnFailed(OP_STATUS error) = 0;
};

/** @short Represents an address book.
 */
class OpAddressBook
{
public:
	virtual ~OpAddressBook() {}

	enum AddressBookType
	{
		ADDRESSBOOK_SIM,		//< Contacts from SIM card
		ADDRESSBOOK_DEVICE		//< Contacts from device's address book
	};

	/** Constructs OpAddressBook.
	 *
	 * @param[out] new_address_book Set to the newly created AddressBook object.
	 * The caller is responsible for deleting it with OP_DELETE
	 * @param type Type of address book to create.
	 *
	 * @return
	 *	- OpStatus::OK
	 *	- OpStatus::ERR_NO_MEMORY
	 *	- OpStatus::ERR_NOT_SUPPORTED
	 */
	static OP_STATUS Create(OpAddressBook** new_address_book, AddressBookType type);

	/** Retrieves AddressBookItem with a given id.
	 *
	 * Find result will be passed through callback. If the item is found then
	 * OnItem shall be called one time and if it returns TRUE then it will be
	 * followed by OnFinished. If the item is not found then only OnFinished
	 * will be called.
	 * If any errors occur during search then OnError shall be called with
	 * relevant error code.
	 *
	 * @param callback Callback object which is called when operation finishes
	 * or fails. Can be called synchronously (during execution of this function)
	 * or asynchronously. MUST NOT be NULL.
	 * @param item_id ID of the item to be retrieved.
	 */
	virtual void GetItem(OpAddressBookIterationCallback* callback, const uni_char* item_id) = 0;

	/** Calls callback for each addressbook item in the addressbook.
	 *
	 * @param callback Callback object which is called when operation finishes
	 * or fails. Can be called synchronously (during execution of this function)
	 * or asynchronously. MUST NOT be NULL.
	 */
	virtual void Iterate(OpAddressBookIterationCallback* callback) = 0;

	/** Adds an AddressBookItem to the AddressBook.
	 *
	 * The item being added must be new (i.e. not yet registered with the
	 * address book). Once the item is added, the platform generates a unique
	 * ID for it. The item argument is not modified but the ID is provided as
	 * an argument to the callback's OnSuccess method.
	 * The ID may then be used as an argument to the GetItem method to retrieve
	 * data for that item.
	 *
	 * @param callback Callback object which is called when operation finishes
	 * or fails. Can be called synchronously (during execution of this function)
	 * or asynchronously. MUST NOT be NULL.
	 * @param item Item to add, cannot be NULL. Item's ID must be NULL (see
	 * OpAddressBookItem::GetID). Must not be modified.
	 */
	virtual void AddItem(OpAddressBookItemModificationCallback* callback, OpAddressBookItem* item) = 0;

	/** Removes an AddressBookItem from the AddressBook
	 *
	 * @param callback Callback object which is called when operation finishes
	 * or fails. Can be called synchronously (during execution of this function)
	 * or asynchronously. MUST NOT be NULL.
	 * @param item_id ID of the item to remove, cannot be null
	 */
	virtual void RemoveItem(OpAddressBookItemModificationCallback* callback, const uni_char* item_id) = 0;

	/** Gets number of items in this AddressBook
	 *
	 * @param callback Callback object which is called when operation finishes
	 * or fails. Can be called synchronously (during execution of this function)
	 * or asynchronously. MUST NOT be NULL.
	 */
	virtual void GetItemCount(OpGetAddressBookItemCountCallback* callback) = 0;

	/** Commits changes to an OpAddressBookItem to the AddressBook.
	 *
	 * @param item Item to commit. Cannot be NULL. The item must be item already
	 * added to the addressbook (i.e. has non-empty id)
	 * @param callback Callback object which is called when operation finishes
	 * or fails. Can be called synchronously (during execution of this function)
	 * or asynchronously. MUST NOT be NULL.
	 */
	virtual void CommitItem(OpAddressBookItemModificationCallback* callback, OpAddressBookItem* item) = 0;

	/** Structure describing addressbook field
	 */
	struct OpAddressBookFieldInfo
	{
		/** Predefined common meanings of a field
		 */
		enum Meaning
		{
			MEANING_DISPLAY_NAME,

			MEANING_FULL_NAME,
			MEANING_FIRST_NAME,
			MEANING_MIDDLE_NAME,
			MEANING_LAST_NAME,

			MEANING_TITLE,	///< Mr, Mrs, Dr etc.

			MEANING_FULL_ADDRESS,
			MEANING_ADDRESS_COUNTRY,
			MEANING_ADDRESS_STATE,
			MEANING_ADDRESS_CITY,
			MEANING_ADDRESS_STREET,
			MEANING_ADDRESS_NUMBER,
			MEANING_ADDRESS_POSTAL_CODE,

			MEANING_COMPANY,
			MEANING_EMAIL,

			MEANING_PRIVATE_LANDLINE_PHONE,
			MEANING_WORK_LANDLINE_PHONE,
			MEANING_LANDLINE_PHONE,
			MEANING_PRIVATE_MOBILE_PHONE,
			MEANING_WORK_MOBILE_PHONE,
			MEANING_MOBILE_PHONE,
			MEANING_WORK_PHONE,
			MEANING_PRIVATE_PHONE,

			MEANING_PRIVATE_FAX,
			MEANING_WORK_FAX,
			MEANING_FAX,

			// custom properties
			MEANING_CUSTOM_PHONE, ///< Any phone number other than those above
			MEANING_CUSTOM        ///< Any other field than those above
		};

		/** Most fitting predefinced meaning or MEANING_CUSTOM if nothing fits. */
		Meaning meaning;

		/** How many values can be assigned to this field.
		 * Ie. contact can have 3 workphones.
		 * 0 means infinite.
		 */
		unsigned int multiplicity;

		/** Platform name of the field.
		 *
		 * May contan only alphanumeric characters, spaces and hyphens('-').
		 * It should be as human readable as posible (eg. 'Work Phone' and not
		 * 'workPhone') and if possible, the same as in the built-in address
		 * book application.
		 */
		const uni_char* name;
	};

	/** Gets information about the fields supported by the address book.
	 *
	 * As this method cannot fail, the implementation should prepare the field
	 * info array during the construction.
	 *
	 * @param[out] field_infos Set to an array of OpAddressBookFieldInfo structs
	 * describing fields of the address book. This array is owned by OpAddressBook
	 * and SHOULD not be modified in any way. MUST NOT be NULL.
	 * @param[out] count Set to length of field info array. MUST NOT be NULL.
	 */
	virtual void GetFieldInfos(const OpAddressBookFieldInfo** field_infos, unsigned int* count) = 0;
};

/** Object containg contact data
 *
 * It is intended only for communication with OpAddressBook as a container for
 * fields data.
 * It is NOT implemented by platform.
 *
 * The OpAddressBookItem stores all the values of fields of a given contact.
 * Fields are indexed the same as in the array provided by
 * OpAddressBook::GetFieldInfos.
 * Since in many cases fields may have many values, they too are being accessed
 * by value index (0 based). The number of possible values is determined by
 * OpAddressBook::OpAddressBookFieldInfo::multiplicity. This implies that there
 * always is a value at index 0.
 *
 * Note: NULL values (distinct from empty string) may also be stored in the
 * OpAddressBookItem. Since values are retrieved as OpStrings, OpString::CStr()
 * may be used to distinguish between NULL and empty string.
 */
class OpAddressBookItem
{
public:
	~OpAddressBookItem();

	/** Constructs a new OpAddressBookItem object
	 *
	 * @param[out] new_addressbook_item Set to newly constructed
	 * OpAddressBookItem. Must not be NULL.
	 * @param address_book An address book to which the address book will be
	 * added. Must not be NULL.
	 */
	static OP_STATUS Make(OpAddressBookItem** new_addressbook_item, OpAddressBook* address_book);

	/** Constructs a new OpAddressBookItem object copying data from another OpAddressBookItem.
	 *
	 * The constructed object is an exact copy, i.e. not only has the copy of all
	 * the field data but also the same ID and is associated with the same
	 * address book.
	 *
	 * @param[out] new_addressbook_item Set to newly constructed
	 * OpAddressBookItem. Must not be NULL.
	 * @param item Address book item to copy. Must not be NULL.
	 */
	static OP_STATUS Make(OpAddressBookItem** new_addressbook_item, OpAddressBookItem* item);

	/** Imports field data (not ID or address_book) from another OpAddressBookItem.
	 *
	 * @param item Address book from which we will import data. Must belong to
	 * the same address book. Must not be NULL.
	 */
	OP_STATUS ImportFields(OpAddressBookItem* item);

	/** Gets address book to which this item belongs.
	 *
	 * Should always be non-NULL even for items not actually added to address book.
	 */
	OpAddressBook* GetAddressBook() const { OP_ASSERT(m_addressbook); return m_addressbook; }

	/** Gets unique ID of the item
	 *
	 * @param[out] id Set to the ID of this item.
	 */
	OP_STATUS GetId(OpString* id) const { return id->Set(m_id); }

	/** @overload */
	const uni_char* GetId() const { return m_id.CStr(); }

	/** Sets unique ID of the item
	 *
	 * As this is the value that binds this object to actual record in device's address book
	 * modyfing it should be done with caution.
	 *
	 * @param id ID to set.
	 */
	OP_STATUS SetId(const uni_char* id) { return m_id.Set(id); }

	/** Is this item already saved in an address book.
	 */
	BOOL WasAdded() const { return !m_id.IsEmpty();}

	/** Appends a value to a field.
	 *
	 * @param field_info_index Index of the field which is accessed.
	 * @param value Value to add. May be NULL.
	 *
	 * @return
	 *  - OpStatus::OK - success
	 *  - OpAddressBookStatus::ERR_FIELD_INDEX_OUT_OF_RANGE - if field_info_index is invalid
	 *  - OpAddressBookStatus::ERR_VALUE_INDEX_OUT_OF_RANGE - if there is no space for a new value
	 *  - OpStatus::ERR_NO_MEMORY - in case of an oom
	 */
	OP_ADDRESSBOOKSTATUS AppendValue(unsigned int field_info_index, const uni_char* value);

	/** Removes a value from a field.
	 *
	 * Values after the removed value are being moved by one index backwards.
	 *
	 * @param field_info_index Index of the field which is accessed.
	 * @param value_index Index of the value.
	 *
	 * @return
	 *  - OpStatus::OK - success
	 *  - OpAddressBookStatus::ERR_FIELD_INDEX_OUT_OF_RANGE - if field_info_index is invalid
	 *  - OpAddressBookStatus::ERR_VALUE_INDEX_OUT_OF_RANGE - if value_index is invalid
	 *  - OpStatus::ERR_NO_MEMORY - in case of an oom
	 */
	OP_ADDRESSBOOKSTATUS RemoveValue(unsigned int field_info_index, unsigned int value_index);

	/** Gets the value of a field.
	 *
	 * @param field_info_index Index of the field which is accessed.
	 * @param value_index Index of the value.
	 * @param[out] value Set to the value of the field. Must not be NULL.
	 *
	 * @return
	 *  - OpStatus::OK - success
	 *  - OpAddressBookStatus::ERR_FIELD_INDEX_OUT_OF_RANGE - if field_info_index is invalid
	 *  - OpAddressBookStatus::ERR_VALUE_INDEX_OUT_OF_RANGE - if value_index is invalid
	 *  - OpStatus::ERR_NO_MEMORY - in case of an oom
	 */
	OP_ADDRESSBOOKSTATUS GetValue(unsigned int field_info_index, unsigned int value_index, OpString* value) const;

	/** Provides pointer to value of a field.
	 *
	 * Returns value of a field without the necessity to copy it. The returned
	 * pointer may only be used right after the call and should not be stored.
	 *
	 * GetValue should be the preferred method of getting the attribute value.
	 *
	 * @param field_info_index Index of the field which is accessed.
	 * @param value_index Index of the value.
	 * @param[out] value Set to point to the internal value of the field.
	 * Must not be NULL.
	 *
	 * @return
	 *  - OpStatus::OK - success
	 *  - OpAddressBookStatus::ERR_FIELD_INDEX_OUT_OF_RANGE - if field_info_index is invalid
	 *  - OpAddressBookStatus::ERR_VALUE_INDEX_OUT_OF_RANGE - if value_index is invalid
	 *  - OpStatus::ERR_NO_MEMORY - in case of an oom
	 */
	OP_ADDRESSBOOKSTATUS PeekValue(unsigned int field_info_index, unsigned int value_index, const uni_char** value) const;

	/** Sets a value of a field.
	 *
	 * @param field_info_index Index of the field which is accessed.
	 * @param value_index Index of the value to be modified.
	 * @param value Value to set the field to. May be NULL.
	 *
	 * @return
	 *  - OpStatus::OK - success
	 *  - OpAddressBookStatus::ERR_FIELD_INDEX_OUT_OF_RANGE - if field_info_index is invalid
	 *  - OpAddressBookStatus::ERR_VALUE_INDEX_OUT_OF_RANGE - if value_index is invalid
	 *  - OpStatus::ERR_NO_MEMORY - in case of an oom
	 */
	OP_ADDRESSBOOKSTATUS SetValue(unsigned int field_info_index, unsigned int value_index, const uni_char* value);

	/** Gets the number of values in the field
	 *
	 * @param field_info_index - index of the field which is accessed.
	 * @param count - Set to the number of the values. Must not be NULL.
	 *
	 * @return
	 *  - OpStatus::OK - success
	 *  - OpAddressBookStatus::ERR_FIELD_INDEX_OUT_OF_RANGE - if field_info_index is invalid
	 */
	OP_ADDRESSBOOKSTATUS GetValueCount(unsigned int field_info_index, unsigned int* count) const;

private:
	OpAddressBookItem(OpAddressBook* address_book) : m_addressbook(address_book) {};
	OpAddressBookItem(const OpAddressBookItem& );
	OpAddressBookItem& operator=(const OpAddressBookItem& );

	/** Helper for checking if field_info_index and value_index are valid.
	 *
	 * It is valid if the address book to which the item belongs has such field
	 * and the field is capable of storing a value at value_index.
	 *
	 * @param field_info_index Index of the field which is accessed.
	 * @param value_index Index of the value to check.
	 *
	 * @return
	 *  - OpStatus::OK - success
	 *  - OpAddressBookStatus::ERR_FIELD_INDEX_OUT_OF_RANGE - if field_info_index is invalid
	 *  - OpAddressBookStatus::ERR_VALUE_INDEX_OUT_OF_RANGE - if value_index is invalid
	 */
	OP_ADDRESSBOOKSTATUS EnsureValue(unsigned int field_info_index, unsigned int value_index) const;

	OpAddressBook* const m_addressbook;
	OpString m_id;
	OpAutoVector<OpString>* m_fields;
};
#endif // PI_ADDRESSBOOK

#endif // PI_DEVICE_API_OPADDRESSBOOKITEM_H
