/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; c-file-style: "stroustrup" -*-
 *
 * Copyright (C) Opera Software ASA  2002 - 2011
 *
 * Bytecode compiler for ECMAScript -- overview, common data and functions
 *
 * @author Jens Lindstrom
 */
#include "core/pch.h"

#include "modules/ecmascript/carakan/src/es_pch.h"

#include "modules/ecmascript/carakan/src/compiler/es_native.h"
#include "modules/ecmascript/carakan/src/compiler/es_instruction_data.h"
#include "modules/ecmascript/carakan/src/util/es_instruction_string.h"
#include "modules/ecmascript/carakan/src/compiler/es_analyzer.h"

#ifdef ES_NATIVE_SUPPORT
#ifdef ARCHITECTURE_ARM

#define SCRATCHI1 ES_CodeGenerator::REG_R0
#define SCRATCHI2 ES_CodeGenerator::REG_R1
#define SCRATCHD1 ES_CodeGenerator::VFPREG_D0
#define SCRATCHD2 ES_CodeGenerator::VFPREG_D1
#define SCRATCHS1 ES_CodeGenerator::VFPREG_S0
#define SCRATCHS2 ES_CodeGenerator::VFPREG_S1
#define SCRATCHS3 ES_CodeGenerator::VFPREG_S2
#define SCRATCHS4 ES_CodeGenerator::VFPREG_S3

#define CONTEXT_POINTER ES_CodeGenerator::REG_R9
#define REGISTER_FRAME_POINTER ES_CodeGenerator::REG_R10
#define PARAMETERS_COUNT ES_CodeGenerator::REG_R1
#define VALUE_TRANSFER_POINTER ES_CodeGenerator::REG_R8

/* These registers are used for the output from the code generated by
   EmitAllocateObject(). */
#define ALLOCATED_OBJECT_REGISTER ES_CodeGenerator::REG_R8
#define ALLOCATED_OBJECT_NAMED_PROPERTIES_REGISTER ES_CodeGenerator::REG_R4
#define ALLOCATED_OBJECT_INDEXED_PROPERTIES_REGISTER ES_CodeGenerator::REG_R5

/* Until decided when/if, leave in previous code to handle inline
   allocations + constructor_final_classes */
#ifdef SUPPORT_INLINE_ALLOCATION
#error "Not currently supported"
#endif // SUPPORT_INLINE_ALLOCATION

#ifndef ES_VALUES_AS_NANS
#  error "Unsupported configuration."
#endif // ES_VALUES_AS_NANS

#if IEEE_BITS_HI
#  define SCRATCHI_TYPE SCRATCHI2
#  define SCRATCHI_IVALUE SCRATCHI1
#  define SCRATCHI_DVALUEH SCRATCHI2
#  define SCRATCHI_DVALUEL SCRATCHI1
#else // IEEE_BITS_HI
#  define SCRATCHI_TYPE SCRATCHI1
#  define SCRATCHI_IVALUE SCRATCHI2
#  define SCRATCHI_DVALUEH SCRATCHI1
#  define SCRATCHI_DVALUEL SCRATCHI2
#endif // IEEE_BITS_HI

#define DECLARE_NOTHING() ES_DECLARE_NOTHING()
#define ES_OFFSETOF_SUBOBJECT(basecls, subcls) (static_cast<int>(reinterpret_cast<UINTPTR>(static_cast<basecls *>(reinterpret_cast<subcls *>(static_cast<UINTPTR>(0x100))))) - 0x100)
#define OBJECT_MEMBER(reg, cls, member) reg, ES_OFFSETOF(cls, member)

/* Byte offset from REG_SP where stack-bound register values are located.
   The scratch space is used when saving/restoring condition state (VFP only.) */
#ifdef ARCHITECTURE_ARM_VFP
#define STACK_VALUE_OFFSET 8
#else
#define STACK_VALUE_OFFSET 0
#endif // ARCHITECTURE_ARM_VFP

static int
VALUE_OFFSET(unsigned index)
{
    return index * sizeof(ES_Value_Internal);
}

static int
VALUE_OFFSET(ES_Native::VirtualRegister *vr)
{
    if (vr->stack_frame_offset == INT_MAX)
        return VALUE_OFFSET(vr->index);
    else
    {
        int size = vr->stack_frame_type == ESTYPE_DOUBLE ? 8 : 4;
        return -(size + STACK_VALUE_OFFSET + vr->stack_frame_offset);
    }
}

static int
DVALUE_OFFSET(ES_Native::VirtualRegister *vr)
{
    return VALUE_OFFSET(vr);
}

static int
TYPE_OFFSET(unsigned index)
{
    return index * sizeof(ES_Value_Internal) + VALUE_TYPE_OFFSET;
}

static int
TYPE_OFFSET(ES_Native::VirtualRegister *vr)
{
    OP_ASSERT(vr->stack_frame_offset == INT_MAX);
    return TYPE_OFFSET(vr->index);
}

static int
IVALUE_OFFSET(unsigned index)
{
    return index * sizeof(ES_Value_Internal) + VALUE_IVALUE_OFFSET;
}

static int
IVALUE_OFFSET(ES_Native::VirtualRegister *vr)
{
    if (vr->stack_frame_offset == INT_MAX)
        return IVALUE_OFFSET(vr->index);
    else
    {
        OP_ASSERT(vr->stack_frame_type == ESTYPE_INT32 || vr->stack_frame_type == ESTYPE_BOOLEAN);
        return VALUE_OFFSET(vr);
    }
}

static ES_CodeGenerator::Register
BASE_REGISTER(ES_Native::VirtualRegister *vr)
{
    return vr->stack_frame_offset == INT_MAX ? REGISTER_FRAME_POINTER : ES_CodeGenerator::REG_SP;
}

static ES_CodeGenerator::Register
BASE_REGISTER(const ES_Native::Operand &op)
{
    return BASE_REGISTER(op.virtual_register);
}

static ES_CodeGenerator::Register
INTEGER_REGISTER(ES_Native::NativeRegister *nr)
{
    return static_cast<ES_CodeGenerator::Register>(nr->register_code);
}

static ES_CodeGenerator::Register
INTEGER_REGISTER(const ES_Native::Operand &op)
{
    return INTEGER_REGISTER(op.native_register);
}

#ifdef ARCHITECTURE_ARM_VFP

static ES_CodeGenerator::VFPDoubleRegister
DOUBLE_REGISTER(ES_Native::NativeRegister *nr)
{
    return static_cast<ES_CodeGenerator::VFPDoubleRegister>(nr->register_code);
}

static ES_CodeGenerator::VFPDoubleRegister
DOUBLE_REGISTER(const ES_Native::Operand &op)
{
    return DOUBLE_REGISTER(op.native_register);
}

#endif // ARCHITECTURE_ARM_VFP

static ES_Native::NativeRegister *
INTEGER_NR(ES_Native *native, ES_CodeGenerator::Register reg)
{
    OP_ASSERT(reg >= ES_CodeGenerator::REG_R2 || reg <= ES_CodeGenerator::REG_R8);
    return native->NR(reg - ES_CodeGenerator::REG_R2);
}

static ES_CodeGenerator::Register
TYPE_REGISTER(ES_CodeGenerator::Register reg1, ES_CodeGenerator::Register reg2)
{
    OP_ASSERT((reg1 & 1) == 0);
    OP_ASSERT(reg1 + 1 == reg2);

#if IEEE_BITS_HI
    return reg2;
#else // IEEE_BITS_HI
    return reg1;
#endif // IEEE_BITS_HI
}

static ES_CodeGenerator::Register
IVALUE_REGISTER(ES_CodeGenerator::Register reg1, ES_CodeGenerator::Register reg2)
{
    OP_ASSERT((reg1 & 1) == 0);
    OP_ASSERT(reg1 + 1 == reg2);

#if IEEE_BITS_HI
    return reg1;
#else // IEEE_BITS_HI
    return reg2;
#endif // IEEE_BITS_HI
}

static ES_CodeGenerator::Register
DVALUEH_REGISTER(ES_CodeGenerator::Register reg1, ES_CodeGenerator::Register reg2)
{
    return TYPE_REGISTER(reg1, reg2);
}

static ES_CodeGenerator::Register
DVALUEL_REGISTER(ES_CodeGenerator::Register reg1, ES_CodeGenerator::Register reg2)
{
    return IVALUE_REGISTER(reg1, reg2);
}

static void
AddImmediateToRegister(ES_CodeGenerator &cg, ES_CodeGenerator::Register source, int value, ES_CodeGenerator::Register target, ES_NativeJumpCondition condition = ES_NATIVE_CONDITION_ALWAYS)
{
    if (ES_CodeGenerator::Operand::EncodeImmediate(value))
        cg.ADD(source, value, target, ES_CodeGenerator::UNSET_CONDITION_CODES, condition);
    else if (ES_CodeGenerator::Operand::EncodeImmediate(-value))
        cg.SUB(source, -value, target, ES_CodeGenerator::UNSET_CONDITION_CODES, condition);
    else
    {
        ES_CodeGenerator::Register scratch;

        if (source != target)
            scratch = target;
        else if (source == SCRATCHI1)
            scratch = SCRATCHI2;
        else
            scratch = SCRATCHI1;

        BOOL subtract = FALSE;

        if (ES_CodeGenerator::NotOperand::EncodeImmediate(value))
            cg.MOV(value, scratch, ES_CodeGenerator::UNSET_CONDITION_CODES, condition);
        else if (ES_CodeGenerator::NotOperand::EncodeImmediate(-value))
        {
            cg.MOV(value, scratch, ES_CodeGenerator::UNSET_CONDITION_CODES, condition);
            subtract = TRUE;
        }
        else
        {
            ES_CodeGenerator::Constant *constant = cg.NewConstant(value);
            cg.LDR(constant, scratch, condition);
        }

        if (subtract)
            cg.SUB(source, scratch, target, ES_CodeGenerator::UNSET_CONDITION_CODES, condition);
        else
            cg.ADD(source, scratch, target, ES_CodeGenerator::UNSET_CONDITION_CODES, condition);
    }
}

static void
AddImmediateToRegister(ES_CodeGenerator &cg, int value, ES_CodeGenerator::Register target, ES_NativeJumpCondition condition = ES_NATIVE_CONDITION_ALWAYS)
{
    AddImmediateToRegister(cg, target, value, target, condition);
}

static void
MoveImmediateToRegister(ES_CodeGenerator &cg, int value, ES_CodeGenerator::Register target, ES_NativeJumpCondition condition = ES_NATIVE_CONDITION_ALWAYS)
{
    if (ES_CodeGenerator::NotOperand::EncodeImmediate(value))
        cg.MOV(value, target, ES_CodeGenerator::UNSET_CONDITION_CODES, condition);
    else
    {
        ES_CodeGenerator::Constant *constant = cg.NewConstant(value);
        cg.LDR(constant, target, condition);
    }
}

static void
MovePointerToRegister(ES_CodeGenerator &cg, void *value, ES_CodeGenerator::Register target, ES_NativeJumpCondition condition = ES_NATIVE_CONDITION_ALWAYS)
{
    ES_CodeGenerator::Constant *constant = cg.NewConstant(value);
    cg.LDR(constant, target, condition);
}

static void
CompareRegisterToImmediate(ES_CodeGenerator &cg, ES_CodeGenerator::Register lhs, int rhs, ES_NativeJumpCondition condition = ES_NATIVE_CONDITION_ALWAYS)
{
    if (ES_CodeGenerator::NegOperand::EncodeImmediate(rhs))
        cg.CMP(lhs, rhs, condition);
    else
    {
        OP_ASSERT(lhs != SCRATCHI1);
        MoveImmediateToRegister(cg, rhs, SCRATCHI1, condition);
        cg.CMP(lhs, SCRATCHI1, condition);
    }
}

static void
EqCompareRegisterToImmediate(ES_CodeGenerator &cg, ES_CodeGenerator::Register lhs, int rhs, ES_NativeJumpCondition condition = ES_NATIVE_CONDITION_ALWAYS)
{
    if (ES_CodeGenerator::Operand::EncodeImmediate(rhs))
        cg.TEQ(lhs, rhs, condition);
    else
    {
        OP_ASSERT(lhs != SCRATCHI1);
        MoveImmediateToRegister(cg, rhs, SCRATCHI1, condition);
        cg.TEQ(lhs, SCRATCHI1, condition);
    }
}

static void
LoadValue(ES_CodeGenerator &cg, ES_CodeGenerator::Register base, unsigned offset, ES_CodeGenerator::Register dst1, ES_CodeGenerator::Register dst2)
{
    if ((dst1 & 1) == 0 && dst1 + 1 == dst2 && ES_CodeGenerator::SupportedOffset(offset, ES_CodeGenerator::LOAD_STORE_DOUBLE_WORD))
        cg.LDRD(base, offset, dst1);
    else if (ES_CodeGenerator::SupportedOffset(offset + 4))
    {
        cg.LDR(base, offset, dst1);
        cg.LDR(base, offset + 4, dst2);
    }
    else
    {
        AddImmediateToRegister(cg, base, offset, dst2);
        cg.LDR(dst2, 4, dst1, TRUE, FALSE);
        cg.LDR(dst2, 0, dst2);
    }
}

static void
LoadValue(ES_CodeGenerator &cg, ES_Native::VirtualRegister *source, ES_CodeGenerator::Register dst1, ES_CodeGenerator::Register dst2)
{
    LoadValue(cg, BASE_REGISTER(source), VALUE_OFFSET(source), dst1, dst2);
}

static void
LoadOffset(ES_CodeGenerator &cg, ES_CodeGenerator::Register source, unsigned offset, ES_CodeGenerator::Register target)
{
    if (ES_CodeGenerator::SupportedOffset(offset, ES_CodeGenerator::LOAD_STORE_WORD))
        cg.LDR(source, offset, target);
    else
    {
        AddImmediateToRegister(cg, source, offset, target);
        cg.LDR(target, 0, target);
    }
}

static void
StoreOffset(ES_CodeGenerator &cg, ES_CodeGenerator::Register source, ES_CodeGenerator::Register base, unsigned offset, ES_CodeGenerator::Register scratch)
{
    OP_ASSERT((scratch != SCRATCHI1 || source != SCRATCHI2) && (scratch != SCRATCHI2 || source != SCRATCHI1));

    if (ES_CodeGenerator::SupportedOffset(offset, ES_CodeGenerator::LOAD_STORE_WORD))
        cg.STR(source, base, offset);
    else
    {
        AddImmediateToRegister(cg, base, offset, scratch);
        cg.STR(source, scratch, 0);
    }
}

static void
StoreValue(ES_CodeGenerator &cg, ES_CodeGenerator::Register src1, ES_CodeGenerator::Register src2, ES_CodeGenerator::Register base, unsigned offset)
{
    if ((src1 & 1) == 0 && src1 + 1 == src2 && ES_CodeGenerator::SupportedOffset(offset, ES_CodeGenerator::LOAD_STORE_DOUBLE_WORD))
        cg.STRD(src1, base, offset);
    else if (ES_CodeGenerator::SupportedOffset(offset + 4))
    {
        cg.STR(src1, base, offset);
        cg.STR(src2, base, offset + 4);
    }
    else
    {
        ES_CodeGenerator::Register dst;

        if (src1 == SCRATCHI1)
        {
            cg.PUSH(ES_CodeGenerator::REG_R8);
            dst = ES_CodeGenerator::REG_R8;
        }
        else
            dst = SCRATCHI1;

        AddImmediateToRegister(cg, base, offset, dst);
        cg.STR(src1, dst, 4, TRUE, FALSE);
        cg.STR(src2, dst, 0);

        if (src1 == SCRATCHI1)
            cg.POP(ES_CodeGenerator::REG_R8);
    }
}

static void
StoreValue(ES_CodeGenerator &cg, ES_CodeGenerator::Register src1, ES_CodeGenerator::Register src2, ES_Native::VirtualRegister *target)
{
    StoreValue(cg, src1, src2, BASE_REGISTER(target), VALUE_OFFSET(target));
}

static void
CopyValue(ES_CodeGenerator &cg, ES_CodeGenerator::Register src_base, unsigned src_offset, ES_CodeGenerator::Register dst_base, unsigned dst_offset)
{
    OP_ASSERT(src_base != SCRATCHI1 && src_base != SCRATCHI2);
    OP_ASSERT(dst_base != SCRATCHI1 && dst_base != SCRATCHI2);

    LoadValue(cg, src_base, src_offset, SCRATCHI1, SCRATCHI2);
    StoreValue(cg, SCRATCHI1, SCRATCHI2, dst_base, dst_offset);
}

static void
CopyValue(ES_CodeGenerator &cg, ES_CodeGenerator::Register src_base, unsigned src_offset, ES_Native::VirtualRegister *dst_vr)
{
    CopyValue(cg, src_base, src_offset, BASE_REGISTER(dst_vr), VALUE_OFFSET(dst_vr->index));
}

static void
CopyValue(ES_CodeGenerator &cg, ES_Native::VirtualRegister *src_vr, ES_CodeGenerator::Register dst_base, unsigned dst_offset)
{
    CopyValue(cg, BASE_REGISTER(src_vr), VALUE_OFFSET(src_vr->index), dst_base, dst_offset);
}

static void
CopyValue(ES_CodeGenerator &cg, ES_Native::VirtualRegister *src_vr, ES_Native::VirtualRegister *dst_vr)
{
    CopyValue(cg, BASE_REGISTER(src_vr), VALUE_OFFSET(src_vr->index), BASE_REGISTER(dst_vr), VALUE_OFFSET(dst_vr->index));
}

static void
CopyDataToValue(ES_CodeGenerator &cg, ES_CodeGenerator::Register source, ES_CodeGenerator::Register source_type, ES_Native::VirtualRegister *target_vr)
{
    ES_CodeGenerator::OutOfOrderBlock *handle_untyped = cg.StartOutOfOrderBlock();
    CopyValue(cg, source, 0, target_vr);
    cg.EndOutOfOrderBlock();

    unsigned handled = (~(ES_STORAGE_BITS_DOUBLE | ES_STORAGE_BITS_WHATEVER) << ES_STORAGE_TYPE_SHIFT) & ES_STORAGE_TYPE_MASK;

    MoveImmediateToRegister(cg, handled, SCRATCHI1);
    cg.TST(source_type, SCRATCHI1);
    cg.Jump(handle_untyped->GetJumpTarget(), ES_NATIVE_CONDITION_EQUAL);
    ES_CodeGenerator::JumpTarget *non_null_target = cg.ForwardJump();

    ES_CodeGenerator::OutOfOrderBlock *handle_null = cg.StartOutOfOrderBlock();
    cg.TEQ(SCRATCHI_IVALUE, 0);
    cg.Jump(non_null_target, ES_NATIVE_CONDITION_NOT_EQUAL);
    MoveImmediateToRegister(cg, ESTYPE_NULL, SCRATCHI_TYPE);
    StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);
    cg.EndOutOfOrderBlock();

    cg.LDR(source, 0, SCRATCHI_IVALUE);
    cg.TST(source_type, ES_STORAGE_NULL_MASK);
    cg.Jump(handle_null->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.SetJumpTarget(non_null_target);

    MoveImmediateToRegister(cg, ES_VALUE_TYPE_INIT_MASK, SCRATCHI_TYPE);
    cg.ORR(source_type, SCRATCHI_TYPE, SCRATCHI_TYPE);
    StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);

    cg.SetOutOfOrderContinuationPoint(handle_untyped);
    cg.SetOutOfOrderContinuationPoint(handle_null);
}

static void
CopyTypedDataToValue(ES_CodeGenerator &cg, ES_CodeGenerator::Register source, unsigned source_offset, ES_StorageType source_type, ES_Native::VirtualRegister *target_vr)
{
    if (source_type == ES_STORAGE_WHATEVER || source_type == ES_STORAGE_DOUBLE)
        CopyValue(cg, source, source_offset, target_vr);
    else
    {
        LoadOffset(cg, source, source_offset, SCRATCHI_IVALUE);

        ES_CodeGenerator::OutOfOrderBlock *null_block = NULL;

        if (ES_Layout_Info::IsNullable(source_type))
        {
            null_block = cg.StartOutOfOrderBlock();
            MoveImmediateToRegister(cg, ESTYPE_NULL, SCRATCHI2);
            cg.STR(SCRATCHI2, BASE_REGISTER(target_vr), TYPE_OFFSET(target_vr->index));
            cg.EndOutOfOrderBlock();

            cg.TEQ(SCRATCHI_IVALUE, 0);
            cg.Jump(null_block->GetJumpTarget(), ES_NATIVE_CONDITION_EQUAL);
        }
        MoveImmediateToRegister(cg, ES_Value_Internal::ConvertToValueType(source_type), SCRATCHI_TYPE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);

        if (null_block)
            cg.SetOutOfOrderContinuationPoint(null_block);
    }
}

static void
CopyValueToData2(ES_CodeGenerator &cg, ES_CodeGenerator::Register source, ES_CodeGenerator::Register target, ES_CodeGenerator::JumpTarget *size_4_target, ES_CodeGenerator::JumpTarget *size_8_target)
{
    unsigned target_offset = 0;
    OP_ASSERT(size_4_target || size_8_target);

    ES_CodeGenerator::JumpTarget *done_target = NULL;
    if (size_8_target)
    {
        cg.SetJumpTarget(size_8_target, FALSE);
        if (!size_4_target)
        {
            CopyValue(cg, source, 0, target, target_offset);
            size_8_target = NULL;
        }
        else
        {
            cg.LDR(source, 4, SCRATCHI1);
            cg.STR(SCRATCHI1, target, target_offset + 4);
        }
    }
    if (size_4_target || size_8_target)
    {
        if (size_4_target)
            cg.SetJumpTarget(size_4_target, FALSE);

        cg.LDR(source, 0, SCRATCHI1);
        cg.STR(SCRATCHI1, target, target_offset);
    }

    if (done_target)
        cg.SetJumpTarget(done_target);
}

static void
CopyValueToData(ES_CodeGenerator &cg, ES_Native::VirtualRegister *source_vr, ES_CodeGenerator::Register target, ES_CodeGenerator::Register target_type, ES_CodeGenerator::JumpTarget *slow_case_target)
{
    unsigned target_offset = 0;
    ES_CodeGenerator::OutOfOrderBlock *handle_untyped = cg.StartOutOfOrderBlock();
    CopyValue(cg, source_vr, target, target_offset);
    cg.EndOutOfOrderBlock();

    unsigned handled = (~(ES_STORAGE_BITS_DOUBLE | ES_STORAGE_BITS_WHATEVER) << ES_STORAGE_TYPE_SHIFT) & ES_STORAGE_TYPE_MASK;

    MoveImmediateToRegister(cg, handled, SCRATCHI1);
    cg.TST(target_type, SCRATCHI1);
    cg.Jump(handle_untyped->GetJumpTarget(), ES_NATIVE_CONDITION_EQUAL);

    cg.LDR(BASE_REGISTER(source_vr), TYPE_OFFSET(source_vr->index), SCRATCHI2);
    ES_CodeGenerator::JumpTarget *non_null = cg.ForwardJump(), *done_target = cg.ForwardJump();
    cg.TST(target_type, ES_STORAGE_NULL_MASK);
    cg.Jump(non_null, ES_NATIVE_CONDITION_EQUAL);
    cg.CMP(SCRATCHI2, ESTYPE_NULL);
    cg.Jump(non_null, ES_NATIVE_CONDITION_NOT_EQUAL);
    cg.MOV(0, SCRATCHI1);
    cg.STR(SCRATCHI1, target, target_offset);
    cg.Jump(done_target);
    cg.SetJumpTarget(non_null);

    MoveImmediateToRegister(cg, ES_VALUE_TYPE_INIT_MASK, SCRATCHI1);
    cg.ORR(target_type, SCRATCHI1, SCRATCHI1);
    cg.CMP(SCRATCHI1, SCRATCHI2);
    cg.Jump(slow_case_target, ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.LDR(BASE_REGISTER(source_vr), VALUE_OFFSET(source_vr->index), SCRATCHI1);
    cg.STR(SCRATCHI1, target, target_offset);
    cg.SetJumpTarget(done_target);
    cg.SetOutOfOrderContinuationPoint(handle_untyped);
}

static void
CopyValueToTypedData(ES_CodeGenerator &cg, ES_Native::VirtualRegister *source_vr, ES_CodeGenerator::Register target, unsigned target_offset, ES_StorageType target_type, ES_CodeGenerator::JumpTarget *type_check_fail, ES_CodeGenerator::Register scratch1, ES_CodeGenerator::Register scratch2)
{
    /* Assert that scratch1 and scratch2 isn't a combination of
       SCRATCHI1 and SCRATCHI2 since StoreOffset will clobber those.
     */
    OP_ASSERT((scratch1 != SCRATCHI1 || scratch2 != SCRATCHI2) && (scratch1 != SCRATCHI2 || scratch2 != SCRATCHI1));

    ES_CodeGenerator::Register type = TYPE_REGISTER(scratch1, scratch2);
    ES_CodeGenerator::Register value = IVALUE_REGISTER(scratch1, scratch2);

    if (target_type == ES_STORAGE_WHATEVER || target_type == ES_STORAGE_DOUBLE)
    {
        LoadValue(cg, source_vr, scratch1, scratch2);
        if (target_type == ES_STORAGE_DOUBLE && type_check_fail)
        {
            cg.CMP(type, ESTYPE_DOUBLE);
            cg.Jump(type_check_fail, ES_NATIVE_CONDITION_GREATER);
        }
        StoreValue(cg, scratch1, scratch2, target, target_offset);
    }
    else if (type_check_fail)
    {
        ES_ValueType value_type = ES_Value_Internal::ConvertToValueType(target_type);
        LoadValue(cg, source_vr, scratch1, scratch2);
        if (ES_Layout_Info::IsNullable(target_type))
        {
            cg.CMP(type, ESTYPE_NULL);
            cg.MOV(0, value, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_EQUAL);
            cg.CMP(type, value_type, ES_NATIVE_CONDITION_NOT_EQUAL);
            cg.Jump(type_check_fail, ES_NATIVE_CONDITION_NOT_EQUAL);
            StoreOffset(cg, value, target, target_offset, type);
        }
        else
        {
            cg.CMP(type, value_type);
            cg.Jump(type_check_fail, ES_NATIVE_CONDITION_NOT_EQUAL);
            StoreOffset(cg, value, target, target_offset, type);
        }
    }
    else if (target_type == ES_STORAGE_NULL)
    {
        cg.MOV(0, value);
        StoreOffset(cg, value, target, target_offset, type);
    }
    else if (target_type != ES_STORAGE_UNDEFINED)
    {
        OP_ASSERT(ES_Layout_Info::SizeOf(target_type) == 4);
        cg.LDR(BASE_REGISTER(source_vr), VALUE_OFFSET(source_vr->index), value);
        StoreOffset(cg, value, target, target_offset, type);
    }
}

static void
JumpToSize(ES_CodeGenerator &cg, unsigned size, ES_CodeGenerator::JumpTarget *&size_4_target, ES_CodeGenerator::JumpTarget *&size_8_target)
{
    if (size == 4)
    {
        if (!size_4_target)
            size_4_target = cg.ForwardJump();
        cg.Jump(size_4_target);
    }
    else
    {
        OP_ASSERT(size == 8);
        if (!size_8_target)
            size_8_target = cg.ForwardJump();
        cg.Jump(size_8_target);
    }
}

void
ES_Native::EmitTypeConversionHandlers(VirtualRegister *source_vr, ES_CodeGenerator::Register properties, unsigned offset, ES_CodeGenerator::JumpTarget *null_target, ES_CodeGenerator::JumpTarget *int_to_double_target)
{
    ES_CodeGenerator::Register source(ES_CodeGenerator::REG_R2);
    OP_ASSERT(source != properties);

    ES_CodeGenerator::OutOfOrderBlock *int_to_double_block = NULL;
    if (int_to_double_target)
    {
        int_to_double_block = cg.StartOutOfOrderBlock();
        cg.SetJumpTarget(int_to_double_target);

        AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(source_vr), source);

#ifdef ARCHITECTURE_ARM_VFP
        if (ARCHITECTURE_HAS_FPU())
        {
            cg.LDR(source, VALUE_IVALUE_OFFSET, SCRATCHI1);
            cg.FMSR(SCRATCHI1, SCRATCHS1);
            cg.FSITOD(SCRATCHS1, SCRATCHD1);
            if (ES_CodeGenerator::SupportedOffset(offset, ES_CodeGenerator::LOAD_STORE_DOUBLE))
                cg.FSTD(SCRATCHD1, properties, offset);
            else
            {
                AddImmediateToRegister(cg, properties, offset, SCRATCHI1);
                cg.FSTD(SCRATCHD1, SCRATCHI1, 0);
            }
        }
        else
#endif // ARCHITECTURE_ARM_VFP
        {
            cg.LDR(source, VALUE_IVALUE_OFFSET, ES_CodeGenerator::REG_R0);
            AddImmediateToRegister(cg, properties, offset, ES_CodeGenerator::REG_R1);
            cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::StoreIntAsDouble)), ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);
        }
        cg.EndOutOfOrderBlock();
    }

    ES_CodeGenerator::OutOfOrderBlock *null_block = NULL;
    if (null_target)
    {
        null_block = cg.StartOutOfOrderBlock();
        cg.SetJumpTarget(null_target);

        cg.MOV(0, source);
        StoreOffset(cg, source, properties, offset, SCRATCHI2);
        cg.EndOutOfOrderBlock();
    }

    if (int_to_double_block)
        cg.SetOutOfOrderContinuationPoint(int_to_double_block);

    if (null_block)
        cg.SetOutOfOrderContinuationPoint(null_block);
}

static ES_CodeGenerator::OutOfOrderBlock *
RecoverFromFailedPropertyValueTransfer(ES_Native *native, ES_Native::VirtualRegister *target_vr, ES_CodeGenerator::Register transfer_register)
{
    if (!native->is_light_weight)
    {
        ES_CodeGenerator &cg = native->cg;
        ES_CodeGenerator::OutOfOrderBlock *recover = cg.StartOutOfOrderBlock();

        cg.SetOutOfOrderContinuationPoint(native->current_slow_case);
        native->current_slow_case = NULL;

        native->property_value_fail = cg.ForwardJump();
        native->EmitRegisterTypeCheck(target_vr, ESTYPE_OBJECT, native->property_value_fail);

        int register_offset = static_cast<int>(VALUE_OFFSET(target_vr->index)) - static_cast<int>(native->property_value_offset);

        if (register_offset)
            AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, register_offset, transfer_register);
        else
            cg.MOV(REGISTER_FRAME_POINTER, transfer_register);

        cg.EndOutOfOrderBlock();

        return recover;
    }
    else
        return NULL;
}

unsigned
ES_ArchitecureMixin::StackSpaceAllocated()
{
    ES_Native *self = static_cast<ES_Native *>(this);

    unsigned stack_space_allocated = 3 * sizeof(void *);

    if (self->code->type == ES_Code::TYPE_FUNCTION)
    {
        stack_space_allocated += sizeof(void *);

        if (self->code->CanHaveVariableObject())
            stack_space_allocated += sizeof(void *);

        stack_space_allocated += sizeof(void *);
    }

    if ((stack_space_allocated + sizeof(void *)) & sizeof(void *))
        stack_frame_padding = sizeof(void *);
    else
        stack_frame_padding = 0;

    stack_space_allocated += stack_frame_padding;

    return stack_space_allocated;
}

ES_CodeGenerator::Constant *
ES_ArchitecureMixin::GetInstructionHandler(ES_Instruction instruction)
{
    ES_Native *self = static_cast<ES_Native *>(this);

    if (!c_ih)
    {
        c_ih = OP_NEWGROA_L(ES_CodeGenerator::Constant *, ESI_LAST_INSTRUCTION, self->arena);
        op_memset(c_ih, 0, sizeof (ES_CodeGenerator::Constant *) * ESI_LAST_INSTRUCTION);
    }

    if (!c_ih[instruction])
        c_ih[instruction] = self->cg.NewConstant(g_ecma_instruction_handlers[instruction]);

    return c_ih[instruction];
}

void
ES_ArchitecureMixin::LoadObjectOperand(unsigned source_index, ES_CodeGenerator::Register target)
{
    ES_Native *self = static_cast<ES_Native *>(this);
    ES_Native::VirtualRegister *source_vr = self->VR(source_index);

    if (self->property_value_read_vr && self->property_value_nr)
    {
        OP_ASSERT(self->property_value_read_vr == source_vr);
        LoadOffset(self->cg, INTEGER_REGISTER(self->property_value_nr), self->property_value_offset, target);
    }
    else
        self->cg.LDR(BASE_REGISTER(source_vr), IVALUE_OFFSET(source_vr), target);
}

ES_Boxed *
AllocateSimple(ES_Context *context, ES_Heap *heap, unsigned nbytes)
{
    return heap->AllocateSimple(context, nbytes);
}

void
ES_ArchitecureMixin::EmitAllocateObject(ES_Class *actual_klass, ES_Class *final_klass, unsigned property_count, unsigned *nindexed, ES_Compact_Indexed_Properties *representation, ES_CodeGenerator::JumpTarget *slow_case)
{
    OP_ASSERT(!actual_klass->HasHashTableProperties());
    DECLARE_NOTHING();

    ES_Native *self = static_cast<ES_Native *>(this);
    ES_CodeGenerator &cg = self->cg;

    ES_Native::ObjectAllocationData data;

    self->GetObjectAllocationData(data, actual_klass, final_klass, property_count, nindexed, representation);
    OP_ASSERT(data.named_bytes >= sizeof(ES_Box) + (final_klass ? final_klass : actual_klass)->SizeOf(property_count));

    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R8);

    /* Carve a chunk off the heap's current free block, check that it was big
       enough and if so, update the heap's current_top pointer. */

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("       EmitAllocateObject(): allocate memory\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    ES_CodeGenerator::Register heap(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register current_top(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register current_limit(ES_CodeGenerator::REG_R4);

    unsigned total_bytes = data.main_bytes + data.named_bytes + data.indexed_bytes;

    ES_CodeGenerator::OutOfOrderBlock *simple_allocation = cg.StartOutOfOrderBlock();

    MoveImmediateToRegister(cg, total_bytes, ES_CodeGenerator::REG_R2);
    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, heap), ES_CodeGenerator::REG_R1);
    cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&AllocateSimple)), ES_CodeGenerator::REG_LR);
    cg.BLX(ES_CodeGenerator::REG_LR);

    cg.TEQ(ES_CodeGenerator::REG_R0, 0);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
    cg.MOV(ES_CodeGenerator::REG_R0, object);
    cg.EndOutOfOrderBlock();

    slow_case = simple_allocation->GetJumpTarget();

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, heap), heap);
    cg.LDR(OBJECT_MEMBER(heap, ES_Heap, current_top), object);
    cg.LDR(OBJECT_MEMBER(heap, ES_Heap, current_limit), current_limit);

    AddImmediateToRegister(cg, object, total_bytes, current_top);

    cg.CMP(current_top, current_limit);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER);
    cg.STR(current_top, OBJECT_MEMBER(heap, ES_Heap, current_top));

    ES_CodeGenerator::Register bytes_live(ES_CodeGenerator::REG_R3);

    cg.LDR(OBJECT_MEMBER(heap, ES_Heap, bytes_live), bytes_live);
    AddImmediateToRegister(cg, bytes_live, total_bytes, bytes_live);
    cg.STR(bytes_live, OBJECT_MEMBER(heap, ES_Heap, bytes_live));

    cg.SetOutOfOrderContinuationPoint(simple_allocation);

    /* Initialize object. */

    /* R4 == ALLOCATED_OBJECT_NAMED_PROPERTIES_REGISTER */
    ES_CodeGenerator::Register properties(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register indexed_properties(ES_CodeGenerator::REG_R5);

    AddImmediateToRegister(cg, object, data.main_bytes + sizeof(ES_Box), properties);

    if (nindexed)
        if (representation)
            MovePointerToRegister(cg, representation, indexed_properties);
        else
            AddImmediateToRegister(cg, object, data.main_bytes + data.named_bytes, indexed_properties);
    else
        cg.MOV(0, indexed_properties);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("       EmitAllocateObject(): initialize ES_Object\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    MoveImmediateToRegister(cg, data.gctag + (data.main_bytes << 16), ES_CodeGenerator::REG_R0);
    MoveImmediateToRegister(cg, data.object_bits, ES_CodeGenerator::REG_R1);
    MoveImmediateToRegister(cg, property_count, ES_CodeGenerator::REG_R2);
    MovePointerToRegister(cg, actual_klass, ES_CodeGenerator::REG_R3);

    /* Store R0-R5. */
    cg.STM(object, 0x003f);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("       EmitAllocateObject(): initialize ES_Properties\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    MoveImmediateToRegister(cg, GCTAG_ES_Box + (data.named_bytes << 16), SCRATCHI1);
    cg.STR(SCRATCHI1, object, data.main_bytes);

    if (nindexed)
    {
#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
        cg.Annotate(UNI_L("       EmitAllocateObject(): initialize 'length' property\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

        /* Set length property. */
        MoveImmediateToRegister(cg, ESTYPE_INT32, SCRATCHI_TYPE);
        MoveImmediateToRegister(cg, data.length, SCRATCHI_IVALUE);
        cg.STRD(SCRATCHI1, properties, 0);

        if (!representation)
        {
#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
            cg.Annotate(UNI_L("       EmitAllocateObject(): initialize ES_Compact_Indexed_Properties\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

            MoveImmediateToRegister(cg, GCTAG_ES_Compact_Indexed_Properties + (data.indexed_bytes << 16), ES_CodeGenerator::REG_R0);
            MoveImmediateToRegister(cg, *nindexed, ES_CodeGenerator::REG_R1);
            MoveImmediateToRegister(cg, data.length, ES_CodeGenerator::REG_R2);

            /* Store R0-R3. */
            cg.STM(indexed_properties, 0x0007);
        }
    }
}

void
ES_ArchitecureMixin::EmitInitProperty(ES_CodeWord *word, const ES_CodeGenerator::Register properties, unsigned index)
{
    ES_Native *self = static_cast<ES_Native *>(this);
    ES_CodeGenerator &cg = self->cg;
    ES_Code *code = self->code;

    ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_CodeGenerator::Register ivalue(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_CodeGenerator::Register dvalueh(DVALUEH_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_CodeGenerator::Register dvaluel(DVALUEL_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

    switch (word ? word->instruction : ESI_LOAD_UNDEFINED)
    {
    case ESI_LOAD_STRING:
        {
            JString *value = code->GetString(word[2].index);

            MoveImmediateToRegister(cg, ESTYPE_STRING, type);
            MovePointerToRegister(cg, value, ivalue);
        }
        break;

    case ESI_LOAD_DOUBLE:
        {
            double &value = code->data->doubles[word[2].index];

            if (op_isnan(value))
                MoveImmediateToRegister(cg, ESTYPE_DOUBLE, type);
            else
            {
                UINT32 high, low;

                op_explode_double(value, high, low);

                MoveImmediateToRegister(cg, high, dvalueh);
                MoveImmediateToRegister(cg, low, dvaluel);
            }
        }
        break;

    case ESI_LOAD_INT32:
        MoveImmediateToRegister(cg, ESTYPE_INT32, type);
        MoveImmediateToRegister(cg, word[2].immediate, ivalue);
        break;

    case ESI_LOAD_NULL:
        MoveImmediateToRegister(cg, ESTYPE_NULL, type);
        cg.MOV(0, ivalue);
        break;

    case ESI_LOAD_UNDEFINED:
        MoveImmediateToRegister(cg, ESTYPE_UNDEFINED, type);
        cg.MOV(1, ivalue); // "hidden boolean"
        break;

    case ESI_LOAD_TRUE:
        MoveImmediateToRegister(cg, ESTYPE_BOOLEAN, type);
        cg.MOV(1, ivalue);
        break;

    case ESI_LOAD_FALSE:
        MoveImmediateToRegister(cg, ESTYPE_BOOLEAN, type);
        cg.MOV(0, ivalue);
        break;

    case ESI_PUTI_IMM:
    case ESI_PUTN_IMM:
    case ESI_INIT_PROPERTY:
        {
            ES_Native::VirtualRegister *source_vr = self->VR(word[3].index);

            ES_ValueType value_type;
            BOOL type_known = self->GetType(source_vr, value_type);

            if (!type_known || (value_type != ESTYPE_UNDEFINED && value_type != ESTYPE_NULL))
                LoadValue(cg, source_vr, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);
            else
                cg.MOV(value_type, type);

            if (word->instruction == ESI_PUTI_IMM)
                if (!type_known)
                {
                    cg.CMP(type, ESTYPE_UNDEFINED);
                    cg.MOV(1, ivalue, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_EQUAL);
                }
                else if (value_type == ESTYPE_UNDEFINED)
                    cg.MOV(1, ivalue);
        }
        break;

    default:
        OP_ASSERT(FALSE);
    }

    StoreValue(cg, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3, properties, VALUE_OFFSET(index));
}

void
ES_ArchitecureMixin::EmitFunctionCall(ES_CodeGenerator::Constant *function)
{
    ES_Native *self = static_cast<ES_Native *>(this);

    if (self->is_light_weight)
        self->cg.MOV(ES_CodeGenerator::REG_LR, ES_CodeGenerator::REG_R8);

    self->cg.LDR(function, ES_CodeGenerator::REG_LR);
    self->cg.BLX(ES_CodeGenerator::REG_LR);

    if (self->is_light_weight)
        self->cg.MOV(ES_CodeGenerator::REG_R8, ES_CodeGenerator::REG_LR);
}

//#define DUMP_TRAMPOLINE_CODE_VECTORS
#ifndef DUMP_TRAMPOLINE_CODE_VECTORS

const unsigned cv_BytecodeToNativeTrampoline_vfp[] =
{
    0xe92d5ff0,    // push	{r4, r5, r6, r7, r8, r9, sl, fp, ip, lr}
    0xed2d8b10,    // vpush	{d8 - d15}
    0xe5909000,    // ldr	r9, [r0]
    0xe590800c,    // ldr	r8, [r0, #12]
    0xe5983000,    // ldr	r3, [r8]
    0xe52d3004,    // push	{r3}		; (str r3, [sp, #-4]!)
    0xe588d000,    // str	sp, [r8]
    0xe5907008,    // ldr	r7, [r0, #8]
    0xe3a04000,    // mov	r4, #0
    0xe52d4004,    // push	{r4}		; (str r4, [sp, #-4]!)
    0xe587d000,    // str	sp, [r7]
    0xe5906004,    // ldr	r6, [r0, #4]
    0xe3a05000,    // mov	r5, #0
    0xe92d0060,    // push	{r5, r6}
    0xe590a010,    // ldr	sl, [r0, #16]
    0xe52d0004,    // push	{r0}		; (str r0, [sp, #-4]!)
    0xe1a0e00f,    // mov	lr, pc
    0xe590f014,    // ldr	pc, [r0, #20]
    0xe49d0004,    // pop	{r0}		; (ldr r0, [sp], #4)
    0xe5907008,    // ldr	r7, [r0, #8]
    0xe3a02000,    // mov	r2, #0
    0xe5872000,    // str	r2, [r7]
    0xe28dd00c,    // add	sp, sp, #12
    0xe590800c,    // ldr	r8, [r0, #12]
    0xe49d2004,    // pop	{r2}		; (ldr r2, [sp], #4)
    0xe5882000,    // str	r2, [r8]
    0xe3a00001,    // mov	r0, #1
    0xecbd8b10,    // vpop	{d8 - d15}
    0xe8bd9ff0     // pop	{r4, r5, r6, r7, r8, r9, sl, fp, ip, pc}
};

const unsigned cv_BytecodeToNativeTrampoline[] =
{
    0xe92d5ff0,    // push	{r4, r5, r6, r7, r8, r9, sl, fp, ip, lr}
    0xe5909000,    // ldr	r9, [r0]
    0xe590800c,    // ldr	r8, [r0, #12]
    0xe5983000,    // ldr	r3, [r8]
    0xe52d3004,    // push	{r3}		; (str r3, [sp, #-4]!)
    0xe588d000,    // str	sp, [r8]
    0xe5907008,    // ldr	r7, [r0, #8]
    0xe3a04000,    // mov	r4, #0
    0xe52d4004,    // push	{r4}		; (str r4, [sp, #-4]!)
    0xe587d000,    // str	sp, [r7]
    0xe5906004,    // ldr	r6, [r0, #4]
    0xe3a05000,    // mov	r5, #0
    0xe92d0060,    // push	{r5, r6}
    0xe590a010,    // ldr	sl, [r0, #16]
    0xe52d0004,    // push	{r0}		; (str r0, [sp, #-4]!)
    0xe1a0e00f,    // mov	lr, pc
    0xe590f014,    // ldr	pc, [r0, #20]
    0xe49d0004,    // pop	{r0}		; (ldr r0, [sp], #4)
    0xe5907008,    // ldr	r7, [r0, #8]
    0xe3a02000,    // mov	r2, #0
    0xe5872000,    // str	r2, [r7]
    0xe28dd00c,    // add	sp, sp, #12
    0xe590800c,    // ldr	r8, [r0, #12]
    0xe49d2004,    // pop	{r2}		; (ldr r2, [sp], #4)
    0xe5882000,    // str	r2, [r8]
    0xe3a00001,    // mov	r0, #1
    0xe8bd9ff0     // pop	{r4, r5, r6, r7, r8, r9, sl, fp, ip, pc}
};

const unsigned cv_ReconstructNativeStackTrampoline1[] =
{
    0xe52de004,    // push	{lr}		; (str lr, [sp, #-4]!)
    0xe1a04000,    // mov	r4, r0
    0xe1a05001,    // mov	r5, r1
    0xe1a00009,    // mov	r0, r9
    0xe1a0100d,    // mov	r1, sp
    0xe594d020,    // ldr	sp, [r4, #32]
    0xe1a0e00f,    // mov	lr, pc
    0xe594f01c,    // ldr	pc, [r4, #28]
    0xe1a01005,    // mov	r1, r5
    0xe1a0d000,    // mov	sp, r0
    0xe49de004,    // pop	{lr}		; (ldr lr, [sp], #4)
    0xe594f018     // ldr	pc, [r4, #24]
};

const unsigned cv_ReconstructNativeStackTrampoline2[] =
{
    0xe52de004,    // push	{lr}		; (str lr, [sp, #-4]!)
    0xe1a04000,    // mov	r4, r0
    0xe1a05001,    // mov	r5, r1
    0xe1a00009,    // mov	r0, r9
    0xe1a0100d,    // mov	r1, sp
    0xe594d020,    // ldr	sp, [r4, #32]
    0xe1a0e00f,    // mov	lr, pc
    0xe594f01c,    // ldr	pc, [r4, #28]
    0xe1a01005,    // mov	r1, r5
    0xe1a0d000,    // mov	sp, r0
    0xe594f018     // ldr	pc, [r4, #24]
};

const unsigned cv_ThrowFromMachineCode_vfp[] =
{
    0xe5901004,    // ldr	r1, [r0, #4]
    0xe3a02000,    // mov	r2, #0
    0xe5812000,    // str	r2, [r1]
    0xe5901008,    // ldr	r1, [r0, #8]
    0xe591d000,    // ldr	sp, [r1]
    0xe49d2004,    // pop	{r2}		; (ldr r2, [sp], #4)
    0xe5812000,    // str	r2, [r1]
    0xe3a00000,    // mov	r0, #0
    0xecbd8b10,    // vpop	{d8 - d15}
    0xe8bd9ff0     // pop	{r4, r5, r6, r7, r8, r9, sl, fp, ip, pc}
};

const unsigned cv_ThrowFromMachineCode[] =
{
    0xe5901004,    // ldr	r1, [r0, #4]
    0xe3a02000,    // mov	r2, #0
    0xe5812000,    // str	r2, [r1]
    0xe5901008,    // ldr	r1, [r0, #8]
    0xe591d000,    // ldr	sp, [r1]
    0xe49d2004,    // pop	{r2}		; (ldr r2, [sp], #4)
    0xe5812000,    // str	r2, [r1]
    0xe3a00000,    // mov	r0, #0
    0xe8bd9ff0     // pop	{r4, r5, r6, r7, r8, r9, sl, fp, ip, pc}
};

#else // DUMP_TRAMPOLINE_CODE_VECTORS

static const OpExecMemory *
DumpCodeVector(const char *key, ES_CodeGenerator &cg)
{
    extern FILE *g_native_disassembler_file;

    FILE *stored_native_disassembler_file = g_native_disassembler_file;
    g_native_disassembler_file = stdout;

    fprintf(g_native_disassembler_file, "const unsigned %s[] =\n{\n", key);

    cg.DumpCodeVector();

    const OpExecMemory *block = cg.GetCode(g_executableMemory);

    fprintf(g_native_disassembler_file, "};\n");
    fflush(g_native_disassembler_file);

    cg.Finalize(g_executableMemory, block);

    g_native_disassembler_file = stored_native_disassembler_file;
    return block;
}

#endif // DUMP_TRAMPOLINE_CODE_VECTORS

/* static */ BOOL
(*ES_Native::GetBytecodeToNativeTrampoline())(void **, unsigned)
{
#ifndef DUMP_TRAMPOLINE_CODE_VECTORS
#  ifdef CONSTANT_DATA_IS_EXECUTABLE
    if (ES_CodeGenerator::SupportsVFP())
        return (BOOL (*)(void **, unsigned))(cv_BytecodeToNativeTrampoline_vfp);
    else
        return (BOOL (*)(void **, unsigned))(cv_BytecodeToNativeTrampoline);
#  else // CONSTANT_DATA_IS_EXECUTABLE
    if (!g_ecma_bytecode_to_native_block)
    {
        if (ES_CodeGenerator::SupportsVFP())
        {
            g_ecma_bytecode_to_native_block = g_executableMemory->AllocateL(sizeof cv_BytecodeToNativeTrampoline_vfp);
            op_memcpy(g_ecma_bytecode_to_native_block->address, cv_BytecodeToNativeTrampoline_vfp, sizeof cv_BytecodeToNativeTrampoline_vfp);
        }
        else
        {
            g_ecma_bytecode_to_native_block = g_executableMemory->AllocateL(sizeof cv_BytecodeToNativeTrampoline);
            op_memcpy(g_ecma_bytecode_to_native_block->address, cv_BytecodeToNativeTrampoline, sizeof cv_BytecodeToNativeTrampoline);
        }
        OpExecMemoryManager::FinalizeL(g_ecma_bytecode_to_native_block);
    }

    return reinterpret_cast<BOOL (*)(void **, unsigned)>(g_ecma_bytecode_to_native_block->address);
#  endif // CONSTANT_DATA_IS_EXECUTABLE
#else // DUMP_TRAMPOLINE_CODE_VECTORS
    if (!g_ecma_bytecode_to_native_block)
    {
        ES_CodeGenerator cg;

        ES_CodeGenerator::Register pointers(ES_CodeGenerator::REG_R0);
        //ES_CodeGenerator::Register argc(ES_CodeGenerator::REG_R1);
        ES_CodeGenerator::Register scratch1(ES_CodeGenerator::REG_R2);
        ES_CodeGenerator::Register native_stack_frame(ES_CodeGenerator::REG_R7);
        ES_CodeGenerator::Register stack_ptr(ES_CodeGenerator::REG_R8);

#define POINTER(name) pointers, TRAMPOLINE_POINTER_##name * sizeof(void *)

        cg.PUSH(0x5ff0); // R4-R12 + LR

        /* context => R10 */
        cg.LDR(POINTER(CONTEXT), CONTEXT_POINTER);

        /* Save previous value of context->stack_ptr. */
        cg.LDR(POINTER(CONTEXT_STACK_PTR), stack_ptr);
        cg.LDR(stack_ptr, 0, ES_CodeGenerator::REG_R3);
        cg.PUSH(ES_CodeGenerator::REG_R3);

        /* SP => context->stack_ptr */
        cg.STR(ES_CodeGenerator::REG_SP, stack_ptr, 0);

        /* Set-up a terminating native stack frame, with a register frame pointer
           corresponding to the calling code's register frame. */
        cg.LDR(POINTER(CONTEXT_NATIVE_STACK_FRAME), native_stack_frame);
        cg.MOV(0, ES_CodeGenerator::REG_R4);
        cg.PUSH(ES_CodeGenerator::REG_R4);
        cg.STR(ES_CodeGenerator::REG_SP, native_stack_frame, 0);

        cg.LDR(POINTER(CONTEXT_REG), ES_CodeGenerator::REG_R6);
        cg.MOV(0, ES_CodeGenerator::REG_R5);
        cg.PUSH(1 << ES_CodeGenerator::REG_R5 | 1 << ES_CodeGenerator::REG_R6);

        /* reg => R11 */
        cg.LDR(POINTER(REGISTER_FRAME), REGISTER_FRAME_POINTER);

        cg.PUSH(pointers);

        cg.MOV(ES_CodeGenerator::REG_PC, ES_CodeGenerator::REG_LR);
        cg.LDR(POINTER(TRAMPOLINE_DISPATCHER), ES_CodeGenerator::REG_PC);

        cg.POP(pointers);

        /* NULL => context->native_stack_frame */
        cg.LDR(POINTER(CONTEXT_NATIVE_STACK_FRAME), native_stack_frame);
        cg.MOV(0, scratch1);
        cg.STR(scratch1, native_stack_frame, 0);

        /* Drop terminating stack frame. */
        cg.ADD(ES_CodeGenerator::REG_SP, 12, ES_CodeGenerator::REG_SP);

        /* Restore previous context->stack_ptr. */
        cg.LDR(POINTER(CONTEXT_STACK_PTR), stack_ptr);
        cg.POP(scratch1);
        cg.STR(scratch1, stack_ptr, 0);

        cg.MOV(1, ES_CodeGenerator::REG_R0);
        cg.POP(0x9ff0); // R4-R12 + PC

#undef POINTER

        g_ecma_bytecode_to_native_block = DumpCodeVector("cv_BytecodeToNativeTrampoline", cg);

        GetReconstructNativeStackTrampoline(TRUE);
        GetReconstructNativeStackTrampoline(FALSE);
        GetThrowFromMachineCode();
    }

    return reinterpret_cast<BOOL (*)(void **, unsigned)>(g_ecma_bytecode_to_native_block->address);
#endif // DUMP_TRAMPOLINE_CODE_VECTORS
}

/* static */ void *
ES_Native::GetReconstructNativeStackTrampoline(BOOL prologue_entry_point)
{
#ifndef DUMP_TRAMPOLINE_CODE_VECTORS
#  ifdef CONSTANT_DATA_IS_EXECUTABLE
    if (prologue_entry_point)
        return const_cast<unsigned *>(cv_ReconstructNativeStackTrampoline1);
    else
        return const_cast<unsigned *>(cv_ReconstructNativeStackTrampoline2);
#  else // CONSTANT_DATA_IS_EXECUTABLE
    const OpExecMemory **block;
    const unsigned *cv;
    unsigned cv_length;

    if (prologue_entry_point)
        block = &g_ecma_reconstruct_native_stack1_block, cv = cv_ReconstructNativeStackTrampoline1, cv_length = sizeof cv_ReconstructNativeStackTrampoline1;
    else
        block = &g_ecma_reconstruct_native_stack2_block, cv = cv_ReconstructNativeStackTrampoline2, cv_length = sizeof cv_ReconstructNativeStackTrampoline2;

    if (!*block)
    {
        *block = g_executableMemory->AllocateL(cv_length);
        op_memcpy((*block)->address, cv, cv_length);
        OpExecMemoryManager::FinalizeL(*block);
    }

    return (*block)->address;
#  endif // CONSTANT_DATA_IS_EXECUTABLE
#else // DUMP_TRAMPOLINE_CODE_VECTORS
    const OpExecMemory **block;
    const char *cv;

    if (prologue_entry_point)
        block = &g_ecma_reconstruct_native_stack1_block, cv = "cv_ReconstructNativeStackTrampoline1";
    else
        block = &g_ecma_reconstruct_native_stack2_block, cv = "cv_ReconstructNativeStackTrampoline2";

    if (!*block)
    {
        ES_CodeGenerator cg;

        ES_CodeGenerator::Register pointers(ES_CodeGenerator::REG_R4);
        ES_CodeGenerator::Register argc(ES_CodeGenerator::REG_R5);

        cg.PUSH(ES_CodeGenerator::REG_LR);

        cg.MOV(ES_CodeGenerator::REG_R0, pointers);
        cg.MOV(ES_CodeGenerator::REG_R1, argc);

#define POINTER(name) pointers, TRAMPOLINE_POINTER_##name * sizeof(void *)

        cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
        cg.MOV(ES_CodeGenerator::REG_SP, ES_CodeGenerator::REG_R1);

        cg.LDR(POINTER(STACK_LIMIT), ES_CodeGenerator::REG_SP);
        cg.MOV(ES_CodeGenerator::REG_PC, ES_CodeGenerator::REG_LR);
        cg.LDR(POINTER(RECONSTRUCT_NATIVE_STACK), ES_CodeGenerator::REG_PC);

        cg.MOV(argc, PARAMETERS_COUNT);
        cg.MOV(ES_CodeGenerator::REG_R0, ES_CodeGenerator::REG_SP);

        if (prologue_entry_point)
            cg.POP(ES_CodeGenerator::REG_LR);

        cg.LDR(POINTER(NATIVE_DISPATCHER), ES_CodeGenerator::REG_PC);

#undef POINTER

        *block = DumpCodeVector(cv, cg);
    }

    return (*block)->address;
#endif // DUMP_TRAMPOLINE_CODE_VECTORS
}

/* static */ void
(*ES_Native::GetThrowFromMachineCode())(void **)
{
#ifndef DUMP_TRAMPOLINE_CODE_VECTORS
#  ifdef CONSTANT_DATA_IS_EXECUTABLE
    if (ES_CodeGenerator::SupportsVFP())
        return (void (*)(void **))(cv_ThrowFromMachineCode_vfp);
    else
        return (void (*)(void **))(cv_ThrowFromMachineCode);
#  else // CONSTANT_DATA_IS_EXECUTABLE
    if (!g_ecma_throw_from_machine_code_block)
    {
        if (ES_CodeGenerator::SupportsVFP())
        {
            g_ecma_throw_from_machine_code_block = g_executableMemory->AllocateL(sizeof cv_ThrowFromMachineCode_vfp);
            op_memcpy(g_ecma_throw_from_machine_code_block->address, cv_ThrowFromMachineCode_vfp, sizeof cv_ThrowFromMachineCode_vfp);
        }
        else
        {
            g_ecma_throw_from_machine_code_block = g_executableMemory->AllocateL(sizeof cv_ThrowFromMachineCode);
            op_memcpy(g_ecma_throw_from_machine_code_block->address, cv_ThrowFromMachineCode, sizeof cv_ThrowFromMachineCode);
        }
        OpExecMemoryManager::FinalizeL(g_ecma_throw_from_machine_code_block);
    }

    return reinterpret_cast<void (*)(void **)>(g_ecma_throw_from_machine_code_block->address);
#  endif // CONSTANT_DATA_IS_EXECUTABLE
#else // DUMP_TRAMPOLINE_CODE_VECTORS
    if (!g_ecma_throw_from_machine_code_block)
    {
        ES_CodeGenerator cg;

        ES_CodeGenerator::Register pointers(ES_CodeGenerator::REG_R0);
        ES_CodeGenerator::Register scratch1(ES_CodeGenerator::REG_R1);
        ES_CodeGenerator::Register scratch2(ES_CodeGenerator::REG_R2);

#define POINTER(name) pointers, THROW_POINTER_##name * sizeof(void *)

        /* NULL => context->native_stack_frame */
        cg.LDR(POINTER(CONTEXT_NATIVE_STACK_FRAME), scratch1);
        cg.MOV(0, scratch2);
        cg.STR(scratch2, scratch1, 0);

        /* context->stack_ptr => SP, NULL => context->stack_ptr */
        cg.LDR(POINTER(CONTEXT_STACK_PTR), scratch1);
        cg.LDR(scratch1, 0, ES_CodeGenerator::REG_SP);
        cg.POP(scratch2);
        cg.STR(scratch2, scratch1, 0);

        cg.MOV(0, ES_CodeGenerator::REG_R0);
        cg.POP(0x9ff0); // R4-R12 + PC

#undef POINTER

        g_ecma_throw_from_machine_code_block = DumpCodeVector("cv_ThrowFromMachineCode", cg);
    }

    return reinterpret_cast<void (*)(void **)>(g_ecma_throw_from_machine_code_block->address);
#endif // DUMP_TRAMPOLINE_CODE_VECTORS
}

#ifdef DEBUG_ENABLE_OPASSERT

/* static */ BOOL
ES_Native::IsAddressInBytecodeToNativeTrampoline(void *address)
{
#if !defined DUMP_TRAMPOLINE_CODE_VECTORS && defined CONSTANT_DATA_IS_EXECUTABLE
    if (ES_CodeGenerator::SupportsVFP())
        return reinterpret_cast<unsigned *>(address) >= cv_BytecodeToNativeTrampoline_vfp && reinterpret_cast<unsigned *>(address) < cv_BytecodeToNativeTrampoline_vfp + ARRAY_SIZE(cv_BytecodeToNativeTrampoline);
    else
        return reinterpret_cast<unsigned *>(address) >= cv_BytecodeToNativeTrampoline && reinterpret_cast<unsigned *>(address) < cv_BytecodeToNativeTrampoline + ARRAY_SIZE(cv_BytecodeToNativeTrampoline);
#else // !DUMP_TRAMPOLINE_CODE_VECTORS && CONSTANT_DATA_IS_EXECUTABLE
    return TRUE;
#endif // !DUMP_TRAMPOLINE_CODE_VECTORS && CONSTANT_DATA_IS_EXECUTABLE
}

#endif // DEBUG_ENABLE_OPASSERT

void
ES_Native::UpdateCode(ES_Code *new_code)
{
    code = new_code;
    OP_ASSERT(c_code == NULL);
}

/*
  The limitation checked for here is the LDR/STR instruction, used to load/store words from memory,
  which has an immediate offset range of +/- 4096.  With a pointer to register zero in a native
  register, we can easily access the first 512 virtual registers, but beyond that we need
  to calculate an offset into a second native register to use as an offset each and every
  time we access such a virtual register. Not worth the effort.

  (Of course, we could extend the range to 1024 virtual registers by adding 4096 to the register
  frame pointer and using negative offsets for the first 512 registers, but the number of
  functions with 513-1024 registers, that is interesting to JIT in the first place, is
  probably quite close to zero.)
*/
BOOL
ES_Native::CheckLimits()
{
    if (!ES_CodeGenerator::SupportedOffset(code->data->register_frame_size * sizeof(ES_Value_Internal)))
        return FALSE;

    return TRUE;
}

void
ES_Native::InitializeNativeRegisters()
{
    integer_registers_count = ARM_INTEGER_REGISTER_COUNT;
    double_registers_count = ARCHITECTURE_HAS_FPU() ? ARM_DOUBLE_REGISTER_COUNT : 0;

    native_registers_count = integer_registers_count + double_registers_count;
    native_registers = OP_NEWGROA_L(NativeRegister, native_registers_count, Arena());

    for (unsigned index = 0; index < native_registers_count; ++index)
    {
        NativeRegister &native_register = native_registers[index];
        native_register.first_allocation = native_register.last_allocation = native_register.current_allocation = native_register.previous_allocation = NULL;
        native_register.type = index < integer_registers_count ? NativeRegister::TYPE_INTEGER : NativeRegister::TYPE_DOUBLE;
    }

    native_registers[ARM_REGISTER_INDEX_R2].register_code = ARM_REGISTER_CODE_R2;
    native_registers[ARM_REGISTER_INDEX_R3].register_code = ARM_REGISTER_CODE_R3;
    native_registers[ARM_REGISTER_INDEX_R4].register_code = ARM_REGISTER_CODE_R4;
    native_registers[ARM_REGISTER_INDEX_R5].register_code = ARM_REGISTER_CODE_R5;
    native_registers[ARM_REGISTER_INDEX_R6].register_code = ARM_REGISTER_CODE_R6;
    native_registers[ARM_REGISTER_INDEX_R7].register_code = ARM_REGISTER_CODE_R7;
    native_registers[ARM_REGISTER_INDEX_R8].register_code = ARM_REGISTER_CODE_R8;

#ifdef ARCHITECTURE_ARM_VFP
    if (ARCHITECTURE_HAS_FPU())
    {
        native_registers[ARM_REGISTER_INDEX_D2].register_code = ARM_REGISTER_CODE_D2;
        native_registers[ARM_REGISTER_INDEX_D3].register_code = ARM_REGISTER_CODE_D3;
        native_registers[ARM_REGISTER_INDEX_D4].register_code = ARM_REGISTER_CODE_D4;
        native_registers[ARM_REGISTER_INDEX_D5].register_code = ARM_REGISTER_CODE_D5;
        native_registers[ARM_REGISTER_INDEX_D6].register_code = ARM_REGISTER_CODE_D6;
        native_registers[ARM_REGISTER_INDEX_D7].register_code = ARM_REGISTER_CODE_D7;
        native_registers[ARM_REGISTER_INDEX_D8].register_code = ARM_REGISTER_CODE_D8;
        native_registers[ARM_REGISTER_INDEX_D9].register_code = ARM_REGISTER_CODE_D9;
        native_registers[ARM_REGISTER_INDEX_D10].register_code = ARM_REGISTER_CODE_D10;
        native_registers[ARM_REGISTER_INDEX_D11].register_code = ARM_REGISTER_CODE_D11;
        native_registers[ARM_REGISTER_INDEX_D12].register_code = ARM_REGISTER_CODE_D12;
        native_registers[ARM_REGISTER_INDEX_D13].register_code = ARM_REGISTER_CODE_D13;
        native_registers[ARM_REGISTER_INDEX_D14].register_code = ARM_REGISTER_CODE_D14;
        native_registers[ARM_REGISTER_INDEX_D15].register_code = ARM_REGISTER_CODE_D15;
    }
#endif // ARCHITECTURE_ARM_VFP
}

static void
SetupNativeStackFrame(ES_Native *nc, ES_Code *code, unsigned &stack_space_allocated, unsigned &stack_frame_padding, BOOL &has_variable_object, BOOL entry_point = FALSE)
{
    ES_CodeGenerator &cg = nc->cg;
    unsigned local_stack_space_allocated = 0;

    DECLARE_NOTHING();

    /* Push previous frame pointer and record new one. */
    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, native_stack_frame), ES_CodeGenerator::REG_R4);
    cg.PUSH(1 << ES_CodeGenerator::REG_R4 | 1 << ES_CodeGenerator::REG_LR);
    cg.STR(ES_CodeGenerator::REG_SP, OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, native_stack_frame));
    local_stack_space_allocated += sizeof(void *);

    if (!nc->c_code) nc->c_code = cg.NewConstant(code);

    cg.MOV(REGISTER_FRAME_POINTER, ES_CodeGenerator::REG_R8);
    cg.LDR(nc->c_code, ES_CodeGenerator::REG_R7);

    unsigned registers(1 << ES_CodeGenerator::REG_R8 | 1 << ES_CodeGenerator::REG_R7);
    local_stack_space_allocated += 2 * sizeof(void *);

    if (code->type == ES_Code::TYPE_FUNCTION)
    {
        if (entry_point)
        {
            cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, arguments_object), ES_CodeGenerator::REG_R6);
            cg.MOV(0, SCRATCHI1);
            cg.STR(SCRATCHI1, OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, arguments_object));
        }
        else
            cg.MOV(0, ES_CodeGenerator::REG_R6);

        registers |= 1 << ES_CodeGenerator::REG_R6;
        local_stack_space_allocated += sizeof(void *);

        has_variable_object = code->CanHaveVariableObject();

        if (has_variable_object)
        {
            if (entry_point)
            {
                cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, variable_object), ES_CodeGenerator::REG_R5);
                cg.MOV(0, SCRATCHI1);
                cg.STR(SCRATCHI1, OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, variable_object));
            }
            else
                cg.MOV(0, ES_CodeGenerator::REG_R5);

            registers |= 1 << ES_CodeGenerator::REG_R5;
            local_stack_space_allocated += sizeof(void *);
        }

        cg.MOV(PARAMETERS_COUNT, ES_CodeGenerator::REG_R4);
        registers |= 1 << ES_CodeGenerator::REG_R4;
        local_stack_space_allocated += sizeof(void *);
    }
    else
        has_variable_object = FALSE;

    cg.PUSH(registers);

    if (((local_stack_space_allocated + sizeof(void *)) & sizeof(void *)) != 0)
    {
        cg.SUB(ES_CodeGenerator::REG_SP, 4, ES_CodeGenerator::REG_SP);
        local_stack_space_allocated += stack_frame_padding = sizeof(void *);
    }
    else
        stack_frame_padding = 0;

    if (!entry_point)
        stack_space_allocated = local_stack_space_allocated;
}

void
ES_Native::EmitRegisterTypeCheck(VirtualRegister *source, ES_ValueType value_type, ES_CodeGenerator::JumpTarget *slow_case, BOOL invert_result)
{
    OP_ASSERT((property_value_fail && property_value_fail == slow_case) || property_value_needs_type_check || property_value_read_vr == NULL || property_value_read_vr != source);
    OP_ASSERT(source->stack_frame_offset == INT_MAX);

    if (!slow_case)
    {
        if (!current_slow_case)
            EmitSlowCaseCall();

        slow_case = current_slow_case->GetJumpTarget();
    }

    if (property_value_read_vr == source && property_value_nr && !property_value_fail)
        LoadOffset(cg, INTEGER_REGISTER(property_value_nr), property_value_offset + VALUE_TYPE_OFFSET, SCRATCHI1);
    else
        cg.LDR(BASE_REGISTER(source), TYPE_OFFSET(source), SCRATCHI1);

    if (value_type == ESTYPE_INT32_OR_DOUBLE)
    {
        cg.CMP(SCRATCHI1, ESTYPE_INT32);
        cg.Jump(slow_case, invert_result ? ES_NATIVE_CONDITION_LESS_OR_EQUAL : ES_NATIVE_CONDITION_GREATER);
    }
    else if (value_type == ESTYPE_DOUBLE)
    {
        cg.CMP(SCRATCHI1, ESTYPE_DOUBLE);
        cg.Jump(slow_case, invert_result ? ES_NATIVE_CONDITION_LESS_OR_EQUAL : ES_NATIVE_CONDITION_GREATER);
    }
    else
    {
        cg.CMP(SCRATCHI1, value_type);
        cg.Jump(slow_case, invert_result ? ES_NATIVE_CONDITION_EQUAL : ES_NATIVE_CONDITION_NOT_EQUAL);
    }
}

void
ES_Native::EmitLoadRegisterValue(NativeRegister *target, VirtualRegister *source, ES_CodeGenerator::JumpTarget *type_check_fail)
{
    if (source->stack_frame_offset != INT_MAX)
    {
        OP_ASSERT((target->type == NativeRegister::TYPE_INTEGER && (source->stack_frame_type == ESTYPE_INT32 || source->stack_frame_type == ESTYPE_BOOLEAN)) || source->stack_frame_type == ESTYPE_DOUBLE);

        type_check_fail = NULL;
    }

    if (target->type == NativeRegister::TYPE_INTEGER)
    {
        if (type_check_fail)
            EmitRegisterTypeCheck(source, ESTYPE_INT32, type_check_fail);

        cg.LDR(BASE_REGISTER(source), IVALUE_OFFSET(source), INTEGER_REGISTER(target));
    }

#ifdef ARCHITECTURE_ARM_VFP
    if (target->type == NativeRegister::TYPE_DOUBLE)
    {
        if (type_check_fail)
        {
            LoadValue(cg, source, SCRATCHI1, SCRATCHI2);

            cg.CMP(SCRATCHI_TYPE, ESTYPE_DOUBLE);
            cg.Jump(type_check_fail, ES_NATIVE_CONDITION_GREATER);
            cg.FMDRR(SCRATCHI_TYPE, SCRATCHI_IVALUE, DOUBLE_REGISTER(target));
        }
        else
            cg.FLDD(BASE_REGISTER(source), DVALUE_OFFSET(source), DOUBLE_REGISTER(target));
    }
#endif // ARCHITECTURE_ARM_VFP
}

BOOL
ES_Native::IsComplexStore(VirtualRegister *target, NativeRegister *source)
{
#ifdef ARCHITECTURE_ARM_VFP
    return (source->type == NativeRegister::TYPE_DOUBLE && target->stack_frame_offset == INT_MAX);
#else
    return FALSE;
#endif // ARCHITECTURE_ARM_VFP
}

void
ES_Native::EmitSaveCondition()
{
#ifdef ARCHITECTURE_ARM_VFP
    if (is_light_weight)
        cg.STR(ES_CodeGenerator::REG_LR, ES_CodeGenerator::REG_SP, -4);

    cg.MRS(ES_CodeGenerator::REG_LR);
#endif // ARCHITECTURE_ARM_VFP
}

void
ES_Native::EmitRestoreCondition()
{
#ifdef ARCHITECTURE_ARM_VFP
    cg.MSR(ES_CodeGenerator::REG_LR, FALSE, FALSE, FALSE, TRUE);
    if (is_light_weight)
        cg.LDR(ES_CodeGenerator::REG_SP, -4, ES_CodeGenerator::REG_LR);
#endif // ARCHITECTURE_ARM_VFP
}

void
ES_Native::EmitStoreRegisterValue(VirtualRegister *target, NativeRegister *source, BOOL write_type, BOOL saved_condition)
{
    if (target->stack_frame_offset != INT_MAX)
    {
        OP_ASSERT(target->stack_frame_type == (source->type == NativeRegister::TYPE_INTEGER ? ESTYPE_INT32 : ESTYPE_DOUBLE));

        write_type = FALSE;
    }

    if (source->type == NativeRegister::TYPE_INTEGER)
    {
        cg.STR(INTEGER_REGISTER(source), BASE_REGISTER(target), IVALUE_OFFSET(target));

        if (write_type)
        {
            cg.MOV(ESTYPE_INT32, SCRATCHI1, ES_CodeGenerator::UNSET_CONDITION_CODES);
            cg.STR(SCRATCHI1, REGISTER_FRAME_POINTER, TYPE_OFFSET(target));
        }
    }

#ifdef ARCHITECTURE_ARM_VFP
    if (source->type == NativeRegister::TYPE_DOUBLE)
    {
        if (target->stack_frame_offset == INT_MAX)
        {
            ES_CodeGenerator::Register type(SCRATCHI_TYPE), value(SCRATCHI_IVALUE);

            cg.FMRRD(DOUBLE_REGISTER(source), type, value);
            cg.CMP(type, ESTYPE_DOUBLE);
            cg.MOV(ESTYPE_DOUBLE, type, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_GREATER);

            StoreValue(cg, SCRATCHI1, SCRATCHI2, target);
        }
        else
            cg.FSTD(DOUBLE_REGISTER(source), BASE_REGISTER(target), DVALUE_OFFSET(target));
    }
#endif // ARCHITECTURE_ARM_VFP
}

void
ES_Native::EmitStoreGlobalObject(VirtualRegister *target_vr)
{
    if (!c_global_object)
        c_global_object = cg.NewConstant(code->global_object);

    cg.LDR(c_global_object, SCRATCHI_IVALUE);
    cg.MOV(ESTYPE_OBJECT, SCRATCHI_TYPE);

    StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);
}

void
ES_Native::EmitStoreConstantBoolean(VirtualRegister *target_vr, BOOL value)
{
    cg.MOV(ESTYPE_BOOLEAN, SCRATCHI_TYPE);
    cg.MOV(value ? 1 : 0, SCRATCHI_IVALUE);

    StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);
}

void
ES_Native::EmitConvertRegisterValue(NativeRegister *target, VirtualRegister *source, unsigned handled, unsigned possible, ES_CodeGenerator::JumpTarget *slow_case)
{
    if (target->type == NativeRegister::TYPE_INTEGER)
    {
        ES_CodeGenerator::JumpTarget *is_integer = NULL, *is_finished = NULL;

        if (handled == ESTYPE_BITS_DOUBLE)
        {
            if (handled != possible)
                EmitRegisterTypeCheck(source, ESTYPE_DOUBLE, slow_case);
        }
        else
        {
            cg.LDR(BASE_REGISTER(source), TYPE_OFFSET(source), SCRATCHI1);

            if ((handled & ESTYPE_BITS_INT32) != 0)
            {
                is_integer = cg.ForwardJump();

                cg.CMP(SCRATCHI1, ESTYPE_INT32);
                cg.Jump(is_integer, ES_NATIVE_CONDITION_EQUAL);
            }

            if ((handled & ESTYPE_BITS_NULL) != 0)
            {
                if (!is_finished)
                    is_finished = cg.ForwardJump();

                cg.CMP(SCRATCHI1, ESTYPE_NULL);
                cg.MOV(0, INTEGER_REGISTER(target), ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_EQUAL);
                cg.Jump(is_finished, ES_NATIVE_CONDITION_EQUAL);
            }

            if ((handled & ESTYPE_BITS_UNDEFINED) != 0)
            {
                if (!is_finished)
                    is_finished = cg.ForwardJump();

                cg.CMP(SCRATCHI1, ESTYPE_UNDEFINED);
                cg.MOV(0, INTEGER_REGISTER(target), ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_EQUAL);
                cg.Jump(is_finished, ES_NATIVE_CONDITION_EQUAL);
            }

            if ((handled & ESTYPE_BITS_DOUBLE) != 0)
            {
                if (handled != possible)
                {
                    cg.CMP(SCRATCHI1, ESTYPE_DOUBLE);
                    cg.Jump(slow_case, ES_NATIVE_CONDITION_GREATER);
                }
            }
            else if (handled != possible)
                cg.Jump(slow_case);
        }

        if ((handled & ESTYPE_BITS_DOUBLE) != 0)
        {
            cg.FLDD(BASE_REGISTER(source), DVALUE_OFFSET(source), SCRATCHD1);
            cg.FTOSIZD(SCRATCHD1, SCRATCHS1);
            cg.FMRS(SCRATCHS1, INTEGER_REGISTER(target));

            cg.CMP(INTEGER_REGISTER(target), INT_MAX);
            cg.CMP(INTEGER_REGISTER(target), INT_MIN, ES_NATIVE_CONDITION_NOT_EQUAL);
            cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

            if (is_integer)
            {
                if (!is_finished)
                    is_finished = cg.ForwardJump();
                cg.Jump(is_finished);
            }
        }

        if (is_integer)
        {
            cg.SetJumpTarget(is_integer);
            cg.LDR(BASE_REGISTER(source), IVALUE_OFFSET(source), INTEGER_REGISTER(target));
        }

        if (is_finished)
            cg.SetJumpTarget(is_finished);
    }

#ifdef ARCHITECTURE_ARM_VFP
    if (target->type == NativeRegister::TYPE_DOUBLE)
    {
        LoadValue(cg, source, SCRATCHI1, SCRATCHI2);

        if (handled == ESTYPE_BITS_INT32)
        {
            if (handled != possible)
            {
                cg.CMP(SCRATCHI_TYPE, ESTYPE_INT32);
                cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
            }

            cg.FMSR(SCRATCHI_IVALUE, SCRATCHS1);
            cg.FSITOD(SCRATCHS1, DOUBLE_REGISTER(target));
        }
        else
        {
            if (handled == (ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE) && handled == possible)
            {
                cg.CMP(SCRATCHI_TYPE, ESTYPE_INT32);
                cg.FMSR(SCRATCHI_IVALUE, SCRATCHS1, ES_NATIVE_CONDITION_EQUAL);
                cg.FSITOD(SCRATCHS1, DOUBLE_REGISTER(target), ES_NATIVE_CONDITION_EQUAL);
                cg.FMDRR(SCRATCHI_TYPE, SCRATCHI_IVALUE, DOUBLE_REGISTER(target), ES_NATIVE_CONDITION_NOT_EQUAL);
            }
            else
            {
                ES_CodeGenerator::JumpTarget *is_finished = NULL, *load_existing = NULL;

                if ((handled & ESTYPE_BITS_INT32) != 0)
                {
                    is_finished = cg.ForwardJump();

                    cg.CMP(SCRATCHI_TYPE, ESTYPE_INT32);
                    cg.FMSR(SCRATCHI_IVALUE, SCRATCHS1, ES_NATIVE_CONDITION_EQUAL);
                    cg.FSITOD(SCRATCHS1, DOUBLE_REGISTER(target), ES_NATIVE_CONDITION_EQUAL);
                    cg.Jump(is_finished, ES_NATIVE_CONDITION_EQUAL);
                }

                if ((handled & ESTYPE_BITS_UNDEFINED) != 0)
                {
                    if (!load_existing)
                        load_existing = cg.ForwardJump();

                    cg.CMP(SCRATCHI_TYPE, ESTYPE_UNDEFINED);
                    cg.Jump(load_existing, ES_NATIVE_CONDITION_EQUAL);
                }

                if ((handled & ESTYPE_BITS_NULL) != 0)
                {
                    if (!load_existing)
                        load_existing = cg.ForwardJump();

                    cg.CMP(SCRATCHI1, ESTYPE_NULL);
                    cg.Jump(load_existing, ES_NATIVE_CONDITION_EQUAL);
                }

                if (handled != possible)
                {
                    cg.CMP(SCRATCHI_TYPE, ESTYPE_DOUBLE);
                    cg.Jump(slow_case, ES_NATIVE_CONDITION_GREATER);
                }

                if (load_existing)
                    cg.SetJumpTarget(load_existing);

                cg.FMDRR(SCRATCHI_TYPE, SCRATCHI_IVALUE, DOUBLE_REGISTER(target));

                if (is_finished)
                    cg.SetJumpTarget(is_finished);
            }
        }
    }
#endif // ARCHITECTURE_ARM_VFP
}

void
ES_Native::EmitSetRegisterType(VirtualRegister *target, ES_ValueType type)
{
    if (type != ESTYPE_DOUBLE)
    {
        cg.MOV(type, SCRATCHI1);
        cg.STR(SCRATCHI1, BASE_REGISTER(target), TYPE_OFFSET(target));
    }
}

void
ES_Native::EmitSetRegisterValue(VirtualRegister *target, const ES_Value_Internal &value, BOOL write_type, BOOL saved_condition)
{
    if (target->stack_frame_offset != INT_MAX)
        write_type = FALSE;

    if (value.IsDouble())
    {
        unsigned high, low;

        op_explode_double(value.GetDouble(), high, low);

        MoveImmediateToRegister(cg, high, SCRATCHI_TYPE);
        MoveImmediateToRegister(cg, low, SCRATCHI_IVALUE);

        StoreValue(cg, SCRATCHI1, SCRATCHI2, target);
    }
    else if (value.IsUndefined() || value.IsNull())
        EmitSetRegisterType(target, value.Type());
    else
    {
        if (write_type)
            cg.MOV(value.Type(), SCRATCHI_TYPE);

        if (value.IsInt32())
            MoveImmediateToRegister(cg, value.GetInt32(), SCRATCHI_IVALUE);
        else if (value.IsBoolean())
            cg.MOV(value.GetBoolean(), SCRATCHI_IVALUE);
        else if (value.IsString() || value.IsObject())
        {
            ES_CodeGenerator::Constant *constant = cg.NewConstant(value.GetDecodedBoxed());
            cg.LDR(constant, SCRATCHI_IVALUE);
        }

        if (write_type)
            StoreValue(cg, SCRATCHI1, SCRATCHI2, target);
        else
            cg.STR(SCRATCHI_IVALUE, BASE_REGISTER(target), IVALUE_OFFSET(target));
    }
}

void
ES_Native::EmitSetRegisterValueFromStackFrameStorage(VirtualRegister *target)
{
    if (target->stack_frame_type == ESTYPE_DOUBLE)
    {
        LoadValue(cg, target, SCRATCHI1, SCRATCHI2);

        cg.CMP(SCRATCHI_TYPE, ESTYPE_DOUBLE);
        cg.MOV(ESTYPE_DOUBLE, SCRATCHI_TYPE, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_GREATER);
    }
    else
    {
        cg.MOV(target->stack_frame_type, SCRATCHI_TYPE);
        cg.LDR(BASE_REGISTER(target), IVALUE_OFFSET(target), SCRATCHI_IVALUE);
    }

    target->stack_frame_offset = INT_MAX;

    StoreValue(cg, SCRATCHI1, SCRATCHI2, target);
}

void
ES_Native::EmitRegisterCopy(const Operand &source, const Operand &target, ES_CodeGenerator::JumpTarget *slow_case)
{
    if (source.native_register && target.virtual_register)
        EmitStoreRegisterValue(target.virtual_register, source.native_register, TRUE);
    else if (source.virtual_register && target.native_register)
        if (source.codeword)
        {
            OP_ASSERT(target.native_register->type == NativeRegister::TYPE_INTEGER);
            MoveImmediateToRegister(cg, source.codeword->immediate, INTEGER_REGISTER(target));
        }
        else
            EmitLoadRegisterValue(target.native_register, source.virtual_register, NULL);
    else if (source.codeword)
        EmitRegisterInt32Assign(source, target);
    else if (source.native_register && target.native_register)
        if (target.native_register->type == NativeRegister::TYPE_DOUBLE)
            if (source.native_register->type == NativeRegister::TYPE_INTEGER)
            {
                cg.FMSR(INTEGER_REGISTER(source.native_register), SCRATCHS1);
                cg.FSITOD(SCRATCHS1, DOUBLE_REGISTER(target.native_register));
            }
            else
                cg.FCPYD(DOUBLE_REGISTER(source.native_register), DOUBLE_REGISTER(target));
        else
            if (source.native_register->type == NativeRegister::TYPE_INTEGER)
                cg.MOV(INTEGER_REGISTER(source.native_register), INTEGER_REGISTER(target));
            else
            {
                OP_ASSERT(slow_case);

                cg.FTOSIZD(DOUBLE_REGISTER(source.native_register), SCRATCHS1);
                cg.FMRS(SCRATCHS1, INTEGER_REGISTER(target));

                cg.CMP(INTEGER_REGISTER(target), INT_MAX);
                cg.CMP(INTEGER_REGISTER(target), INT_MIN, ES_NATIVE_CONDITION_NOT_EQUAL);
                cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
            }
    else
        OP_ASSERT(FALSE);
}

void
ES_Native::EmitRegisterCopy(VirtualRegister *source, VirtualRegister *target)
{
    OP_ASSERT(source->stack_frame_offset == INT_MAX);
    OP_ASSERT(target->stack_frame_offset == INT_MAX);

    CopyValue(cg, source, target);
}

void
ES_Native::EmitRegisterInt32Copy(VirtualRegister *source, VirtualRegister *target)
{
    cg.MOV(ESTYPE_INT32, SCRATCHI1);
    cg.STR(SCRATCHI1, BASE_REGISTER(target), TYPE_OFFSET(target));

    cg.LDR(BASE_REGISTER(source), IVALUE_OFFSET(source), SCRATCHI2);
    cg.STR(SCRATCHI2, BASE_REGISTER(target), IVALUE_OFFSET(target));
}

void
ES_Native::EmitRegisterInt32Assign(const Operand &source, const Operand &target)
{
    OP_ASSERT(source.codeword);

    ES_CodeGenerator::Register itarget;

    if (target.native_register && target.native_register->type == NativeRegister::TYPE_INTEGER)
        itarget = INTEGER_REGISTER(target);
    else
        itarget = SCRATCHI1;

    MoveImmediateToRegister(cg, source.codeword->immediate, itarget);

    if (target.native_register)
        if (target.native_register->type == NativeRegister::TYPE_INTEGER)
            return;
        else
        {
            cg.FMSR(SCRATCHI1, SCRATCHS1);
            cg.FSITOD(SCRATCHS1, DOUBLE_REGISTER(target));
        }
    else
    {
        cg.STR(SCRATCHI1, BASE_REGISTER(target), IVALUE_OFFSET(target.virtual_register));
        EmitSetRegisterType(target.virtual_register, ESTYPE_INT32);
    }
}

void
ES_Native::EmitRegisterInt32Assign(int source, NativeRegister *target)
{
    MoveImmediateToRegister(cg, source, INTEGER_REGISTER(target));
}

void
ES_Native::EmitRegisterDoubleAssign(const double *value, const Operand &target)
{
    if (target.native_register)
    {
        ES_CodeGenerator::Constant *constant = cg.NewConstant(*value);
        cg.FLDD(constant, DOUBLE_REGISTER(target));
    }
    else if (op_isnan(*value))
    {
        cg.MOV(ESTYPE_DOUBLE, SCRATCHI1);
        cg.STR(SCRATCHI1, BASE_REGISTER(target), TYPE_OFFSET(target.virtual_register));
    }
    else
    {
        unsigned high, low;

        op_explode_double(*value, high, low);

        MoveImmediateToRegister(cg, high, SCRATCHI_TYPE);
        MoveImmediateToRegister(cg, low, SCRATCHI_IVALUE);

        StoreValue(cg, SCRATCHI1, SCRATCHI2, target.virtual_register);
    }
}

void
ES_Native::EmitRegisterStringAssign(JString *value, const Operand &target)
{
    ES_CodeGenerator::Constant *constant = cg.NewConstant(value);

    if (target.native_register)
        cg.LDR(constant, INTEGER_REGISTER(target));
    else
    {
        cg.MOV(ESTYPE_STRING, SCRATCHI_TYPE);
        cg.LDR(constant, SCRATCHI_IVALUE);

        StoreValue(cg, SCRATCHI1, SCRATCHI2, target.virtual_register);
    }
}

void
ES_Native::EmitToPrimitive(VirtualRegister *source)
{
    if (!current_slow_case)
        EmitSlowCaseCall();

    cg.LDR(BASE_REGISTER(source), TYPE_OFFSET(source), SCRATCHI1);
    cg.CMP(SCRATCHI1, ESTYPE_OBJECT);
    cg.Jump(current_slow_case->GetJumpTarget(), ES_NATIVE_CONDITION_EQUAL);
}

void
ES_Native::EmitInstructionHandlerCall()
{
    if (is_light_weight)
        LEAVE(OpStatus::ERR);

    ES_CodeWord *word = CurrentCodeWord();
    ES_Instruction instruction = word->instruction;

    if (instruction == ESI_TOPRIMITIVE)
        instruction = ESI_TOPRIMITIVE1;

    if (word == entry_point_cw && !entry_point_jump_target)
        entry_point_jump_target = EmitEntryPoint();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    if (current_slow_case)
    {
        OpString annotation;

        annotation.AppendL("  from EmitInstructionHandlerCall() for ");
        annotation.AppendL(annotator.GetInstructionName(word->instruction));
        annotation.AppendFormat(UNI_L(" (cw_index=%u):\n"), cw_index);

        cg.Annotate(annotation.CStr());
    }
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    switch (instruction)
    {
    case ESI_EVAL:
    case ESI_CALL:
    case ESI_CONSTRUCT:
        {
            ES_CodeGenerator::Constant *constant;

            if (word->instruction == ESI_EVAL || word->instruction == ESI_CALL)
                constant = c_CallFromMachineCode ? c_CallFromMachineCode : c_CallFromMachineCode = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::CallFromMachineCode));
            else
                constant = c_ConstructFromMachineCode ? c_ConstructFromMachineCode : c_ConstructFromMachineCode = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::ConstructFromMachineCode));

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);

            MoveImmediateToRegister(cg, cw_index + (constructor ? code->data->codewords_count : 0), ES_CodeGenerator::REG_R1);

            cg.LDR(constant, ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);
            cg.TEQ(ES_CodeGenerator::REG_R0, 0);
            cg.BX(ES_CodeGenerator::REG_R0, ES_NATIVE_CONDITION_NOT_EQUAL);
        }
        break;

    case ESI_GETN_IMM:
    case ESI_GET_LENGTH:
    case ESI_PUTN_IMM:
    case ESI_INIT_PROPERTY:
        {
            ES_CodeGenerator::Constant *constant;

            if (instruction == ESI_GETN_IMM || instruction == ESI_GET_LENGTH)
            {
                if (!c_GetNamedImmediate) c_GetNamedImmediate = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::GetNamedImmediate));
                constant = c_GetNamedImmediate;
            }
            else
            {
                if (!c_PutNamedImmediate) c_PutNamedImmediate = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::PutNamedImmediate));
                constant = c_PutNamedImmediate;
            }

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);

            MoveImmediateToRegister(cg, cw_index + (constructor ? code->data->codewords_count : 0), ES_CodeGenerator::REG_R1);

            cg.LDR(constant, ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);

            cg.TEQ(ES_CodeGenerator::REG_R0, 0);
            cg.BX(ES_CodeGenerator::REG_R0, ES_NATIVE_CONDITION_NOT_EQUAL);
        }
        break;

    case ESI_GET_GLOBAL:
    case ESI_PUT_GLOBAL:
        {
            ES_CodeGenerator::Constant *constant;

            if (instruction == ESI_GET_GLOBAL)
                if (code->global_caches[word[3].index].class_id == ES_Class::GLOBAL_IMMEDIATE_CLASS_ID)
                {
                    if (!c_GetGlobalImmediate) c_GetGlobalImmediate = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::GetGlobalImmediate));
                    constant = c_GetGlobalImmediate;
                }
                else
                {
                    if (!c_GetGlobal) c_GetGlobal = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::GetGlobal));
                    constant = c_GetGlobal;
                }
            else
            {
                if (!c_PutGlobal) c_PutGlobal = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::PutGlobal));
                constant = c_PutGlobal;
            }

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);

            MoveImmediateToRegister(cg, cw_index + (constructor ? code->data->codewords_count : 0), ES_CodeGenerator::REG_R1);

            cg.LDR(constant, ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);

            cg.TEQ(ES_CodeGenerator::REG_R0, 0);
            cg.BX(ES_CodeGenerator::REG_R0, ES_NATIVE_CONDITION_NOT_EQUAL);
        }
        break;

    case ESI_REDIRECTED_CALL:
        {
            DECLARE_NOTHING();

            cg.MOV(constructor ? 1 : 0, SCRATCHI1);
            cg.STR(SCRATCHI1, OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, in_constructor));
        }

    default:
        {
            DECLARE_NOTHING();

            if (!c_codewords) c_codewords = cg.NewConstant(code->data->codewords);

            cg.LDR(c_codewords, ES_CodeGenerator::REG_R1);

            AddImmediateToRegister(cg, (cw_index + 1) * sizeof(ES_CodeWord), ES_CodeGenerator::REG_R1);

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
            cg.LDR(GetInstructionHandler(word->instruction), ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);

            /* Clear the in_constructor flag set prior to call */
            if (instruction == ESI_REDIRECTED_CALL && constructor)
            {
                cg.MOV(0, SCRATCHI2);
                cg.STR(SCRATCHI2, OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, in_constructor));
            }

            switch (word->instruction)
            {
            case ESI_GET:
            case ESI_GETI_IMM:
            case ESI_PUT:
            case ESI_PUTI_IMM:
                {
                    if (!c_UpdateNativeDispatcher) c_UpdateNativeDispatcher = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::UpdateNativeDispatcher));

                    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);

                    MoveImmediateToRegister(cg, code->data->instruction_offsets[instruction_index + 1] + (constructor ? code->data->codewords_count : 0), ES_CodeGenerator::REG_R1);

                    cg.LDR(c_UpdateNativeDispatcher, ES_CodeGenerator::REG_LR);
                    cg.BLX(ES_CodeGenerator::REG_LR);

                    cg.TEQ(ES_CodeGenerator::REG_R0, 0);
                    cg.BX(ES_CodeGenerator::REG_R0, ES_NATIVE_CONDITION_NOT_EQUAL);
                }
            }
        }
        break;
    }
}

void
ES_Native::EmitSlowCaseCall(BOOL failed_inlined_function)
{
    if (is_light_weight)
    {
        if (!light_weight_failure)
            EmitLightWeightFailure();

        current_slow_case = light_weight_failure;
    }
    else
    {
        current_slow_case = cg.StartOutOfOrderBlock();

#ifdef ES_SLOW_CASE_PROFILING
        extern BOOL g_slow_case_summary;
        if (g_slow_case_summary)
        {
            MovePointerToRegister(cg, context->rt_data->slow_case_calls + CurrentCodeWord()->instruction, SCRATCHI1);
            cg.LDR(SCRATCHI1, 0, SCRATCHI2);
            cg.ADD(SCRATCHI2, 1, SCRATCHI2);
            cg.STR(SCRATCHI2, SCRATCHI1, 0);
        }
#endif // ES_SLOW_CASE_PROFILING

        if (property_value_read_vr && property_value_nr)
        {
            CopyTypedDataToValue(cg, INTEGER_REGISTER(property_value_nr), property_value_offset, property_value_needs_type_check ? ES_STORAGE_WHATEVER : ES_STORAGE_OBJECT, property_value_read_vr);

            current_slow_case_main = cg.Here();
        }
        else
            current_slow_case_main = NULL;

        if (failed_inlined_function)
        {
            DECLARE_NOTHING();

            if (!c_code) c_code = cg.NewConstant(code);

            cg.MOV(1, SCRATCHI1);
            cg.LDR(c_code, SCRATCHI2);
            cg.STR(SCRATCHI1, OBJECT_MEMBER(SCRATCHI2, ES_Code, has_failed_inlined_function));
        }

        EmitInstructionHandlerCall();

        cg.EndOutOfOrderBlock();
    }
    if (property_value_fail)
    {
        cg.StartOutOfOrderBlock();
        cg.SetJumpTarget(property_value_fail);

        if (is_light_weight || current_slow_case_main == NULL)
            cg.Jump(current_slow_case->GetJumpTarget());
        else
            cg.Jump(current_slow_case_main);

        cg.EndOutOfOrderBlock(FALSE);
        property_value_fail = NULL;
    }
}

void
ES_Native::EmitLightWeightFailure()
{
    if (!light_weight_failure)
    {
        light_weight_failure = cg.StartOutOfOrderBlock();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
        cg.Annotate(UNI_L("  from EmitLightWeightFailure():\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

        cg.MOV(fncode->GetData()->formals_count, PARAMETERS_COUNT);

        light_weight_wrong_argc = cg.Here();

        BOOL has_variable_object;

        SetupNativeStackFrame(this, code, stack_space_allocated, stack_frame_padding, has_variable_object);

        ES_CodeGenerator::Constant *constant = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::LightWeightDispatcherFailure));

        cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
        cg.LDR(constant, ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);
        cg.CMP(ES_CodeGenerator::REG_R0, 0);
        cg.BX(ES_CodeGenerator::REG_R0, ES_NATIVE_CONDITION_NOT_EQUAL);
        cg.ADD(ES_CodeGenerator::REG_SP, stack_space_allocated, ES_CodeGenerator::REG_SP);
        cg.POP(ES_CodeGenerator::REG_PC);
        cg.EndOutOfOrderBlock(FALSE);
    }
}

void
ES_Native::EmitExecuteBytecode(unsigned start_instruction_index, unsigned end_instruction_index, BOOL last_in_slow_case)
{
    if (!c_ExecuteBytecode) c_ExecuteBytecode = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::ExecuteBytecode));

    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);

    MoveImmediateToRegister(cg, start_instruction_index, ES_CodeGenerator::REG_R1);

    if (constructor)
        MoveImmediateToRegister(cg, ~end_instruction_index, ES_CodeGenerator::REG_R2);
    else
        AddImmediateToRegister(cg, ES_CodeGenerator::REG_R1, end_instruction_index - start_instruction_index, ES_CodeGenerator::REG_R2);

    cg.LDR(c_ExecuteBytecode, ES_CodeGenerator::REG_LR);
    cg.BLX(ES_CodeGenerator::REG_LR);

    cg.TEQ(ES_CodeGenerator::REG_R0, 0);
    cg.BX(ES_CodeGenerator::REG_R0, ES_NATIVE_CONDITION_NOT_EQUAL);
}

static BOOL
IsMultipleOfTwo(int value)
{
    return (value & (value - 1)) == 0;
}

static unsigned
Log2(int value)
{
    for (unsigned result = 1;; ++result)
        if (1 << result == value)
            return result;
}

void
ES_Native::EmitInt32Arithmetic(Int32ArithmeticType type, const Operand &target, const Operand &lhs, const Operand &rhs, BOOL as_condition, ES_CodeGenerator::JumpTarget *overflow_target)
{
    OP_ASSERT(target.native_register);
    OP_ASSERT(lhs.native_register || lhs.codeword);
    OP_ASSERT(rhs.native_register || rhs.codeword);
    OP_ASSERT(lhs.native_register || rhs.native_register);

    ES_CodeGenerator::SetConditionCodes set_condition_codes = (as_condition || overflow_target) ? ES_CodeGenerator::SET_CONDITION_CODES : ES_CodeGenerator::UNSET_CONDITION_CODES;
    ES_CodeGenerator::Register rtarget = INTEGER_REGISTER(target);
    ES_CodeGenerator::Register rlhs, rrhs;
    const ES_CodeWord *lhs_codeword = lhs.codeword, *rhs_codeword = rhs.codeword;
    BOOL is_shift = type >= INT32ARITHMETIC_LSHIFT && type <= INT32ARITHMETIC_RSHIFT_UNSIGNED;

    if (lhs.native_register)
        rlhs = INTEGER_REGISTER(lhs);
    else if (!lhs_codeword)
    {
        UseInPlace(lhs.virtual_register);
        cg.LDR(BASE_REGISTER(lhs.virtual_register), IVALUE_OFFSET(lhs.virtual_register), SCRATCHI1);
        rlhs = SCRATCHI1;
    }
    else if (is_shift || !ES_CodeGenerator::Operand::EncodeImmediate(lhs_codeword->immediate))
    {
        MoveImmediateToRegister(cg, lhs_codeword->immediate, SCRATCHI1);
        rlhs = SCRATCHI1;
        lhs_codeword = NULL;
    }

    if (rhs.native_register)
        rrhs = INTEGER_REGISTER(rhs);
    else if (!rhs_codeword)
    {
        UseInPlace(rhs.virtual_register);
        cg.LDR(BASE_REGISTER(rhs.virtual_register), IVALUE_OFFSET(rhs.virtual_register), SCRATCHI2);
        rrhs = SCRATCHI2;
    }
    else if (!ES_CodeGenerator::Operand::EncodeImmediate(rhs_codeword->immediate))
    {
        MoveImmediateToRegister(cg, rhs_codeword->immediate, SCRATCHI2);
        rrhs = SCRATCHI2;
        rhs_codeword = NULL;
    }

    switch (type)
    {
    case INT32ARITHMETIC_LSHIFT:
        if (!rhs_codeword)
        {
            cg.AND(rrhs, 31, SCRATCHI2);
            cg.MOV(ES_CodeGenerator::NotOperand(rlhs, SCRATCHI2, ES_CodeGenerator::SHIFT_LOGICAL_LEFT), rtarget, set_condition_codes);
        }
        else if (rhs_codeword->immediate == 0)
            cg.MOV(rlhs, rtarget, set_condition_codes);
        else
            cg.MOV(ES_CodeGenerator::NotOperand(rlhs, rhs_codeword->immediate & 31u, ES_CodeGenerator::SHIFT_LOGICAL_LEFT), rtarget, set_condition_codes);
        break;

    case INT32ARITHMETIC_RSHIFT_SIGNED:
        if (!rhs_codeword)
        {
            cg.AND(rrhs, 31, SCRATCHI2);
            cg.MOV(ES_CodeGenerator::NotOperand(rlhs, SCRATCHI2, ES_CodeGenerator::SHIFT_ARITHMETIC_RIGHT), rtarget, set_condition_codes);
        }
        else if (rhs_codeword->immediate == 0)
            cg.MOV(rlhs, rtarget, set_condition_codes);
        else
            cg.MOV(ES_CodeGenerator::NotOperand(rlhs, rhs_codeword->immediate & 31u, ES_CodeGenerator::SHIFT_ARITHMETIC_RIGHT), rtarget, set_condition_codes);
        break;

    case INT32ARITHMETIC_RSHIFT_UNSIGNED:
        if (!rhs_codeword)
        {
            cg.AND(rrhs, 31, SCRATCHI2);
            cg.MOV(ES_CodeGenerator::NotOperand(rlhs, SCRATCHI2, ES_CodeGenerator::SHIFT_LOGICAL_RIGHT), rtarget, set_condition_codes);

            if (overflow_target)
                cg.Jump(overflow_target, ES_NATIVE_CONDITION_NEGATIVE);
        }
        else if (rhs_codeword->immediate == 0)
        {
            cg.MOV(rlhs, rtarget, set_condition_codes);

            if (overflow_target)
                cg.Jump(overflow_target, ES_NATIVE_CONDITION_NEGATIVE);
        }
        else
            cg.MOV(ES_CodeGenerator::NotOperand(rlhs, rhs_codeword->immediate & 31u, ES_CodeGenerator::SHIFT_LOGICAL_RIGHT), rtarget, set_condition_codes);
        break;

    case INT32ARITHMETIC_AND:
        if (!lhs_codeword && !rhs_codeword)
            cg.AND(rlhs, rrhs, rtarget, set_condition_codes);
        else if (rhs_codeword)
            cg.AND(rlhs, rhs_codeword->immediate, rtarget, set_condition_codes);
        else
            cg.AND(rrhs, lhs_codeword->immediate, rtarget, set_condition_codes);
        break;

    case INT32ARITHMETIC_OR:
        if (!lhs_codeword && !rhs_codeword)
            cg.ORR(rlhs, rrhs, rtarget, set_condition_codes);
        else if (rhs_codeword)
            cg.ORR(rlhs, rhs_codeword->immediate, rtarget, set_condition_codes);
        else
            cg.ORR(rrhs, lhs_codeword->immediate, rtarget, set_condition_codes);
        break;

    case INT32ARITHMETIC_XOR:
        if (!lhs_codeword && !rhs_codeword)
            cg.EOR(rlhs, rrhs, rtarget, set_condition_codes);
        else if (rhs_codeword)
            cg.EOR(rlhs, rhs_codeword->immediate, rtarget, set_condition_codes);
        else
            cg.EOR(rrhs, lhs_codeword->immediate, rtarget, set_condition_codes);
        break;

    case INT32ARITHMETIC_ADD:
        if (!lhs_codeword && !rhs_codeword)
            cg.ADD(rlhs, rrhs, rtarget, set_condition_codes);
        else if (rhs_codeword)
            cg.ADD(rlhs, rhs_codeword->immediate, rtarget, set_condition_codes);
        else
            cg.ADD(rrhs, lhs_codeword->immediate, rtarget, set_condition_codes);

        if (overflow_target)
            cg.Jump(overflow_target, ES_NATIVE_CONDITION_OVERFLOW);
        break;

    case INT32ARITHMETIC_SUB:
        if (!lhs_codeword && !rhs_codeword)
            cg.SUB(rlhs, rrhs, rtarget, set_condition_codes);
        else if (!lhs_codeword)
            cg.SUB(rlhs, rhs_codeword->immediate, rtarget, set_condition_codes);
        else
            cg.RSB(rrhs, lhs_codeword->immediate, rtarget, set_condition_codes);

        if (overflow_target)
            cg.Jump(overflow_target, ES_NATIVE_CONDITION_OVERFLOW);
        break;

    case INT32ARITHMETIC_MUL:
        if (lhs_codeword || rhs_codeword)
        {
            ES_CodeGenerator::Register src, dst = rtarget;
            int imm;

            if (!lhs_codeword)
            {
                src = rlhs;
                imm = rhs_codeword->immediate;
            }
            else
            {
                src = rrhs;
                imm = lhs_codeword->immediate;
            }

            if (imm == 0)
            {
                if (overflow_target)
                {
                    cg.TST(src, src);
                    cg.Jump(overflow_target, ES_NATIVE_CONDITION_NEGATIVE);
                }

                cg.MOV(0, dst, set_condition_codes);
                return;
            }
            else if (imm < 0)
            {
                if (overflow_target)
                {
                    cg.TEQ(src, 0);
                    cg.Jump(overflow_target, ES_NATIVE_CONDITION_EQUAL);
                }
            }
            else if (IsMultipleOfTwo(imm) || IsMultipleOfTwo(imm + 1))
            {
                if (overflow_target)
                {
                    CompareRegisterToImmediate(cg, src, INT_MAX / imm);
                    cg.Jump(overflow_target, ES_NATIVE_CONDITION_GREATER);
                    CompareRegisterToImmediate(cg, src, INT_MIN / imm);
                    cg.Jump(overflow_target, ES_NATIVE_CONDITION_LESS);
                }

                if (IsMultipleOfTwo(imm))
                    cg.MOV(ES_CodeGenerator::Operand(src, Log2(imm)), dst, set_condition_codes);
                else
                    cg.RSB(src, ES_CodeGenerator::Operand(src, Log2(imm + 1)), dst, set_condition_codes);

                return;
            }

            MoveImmediateToRegister(cg, imm, SCRATCHI1);
            cg.SMULL(src, SCRATCHI1, SCRATCHI1, dst);
        }
        else
        {
            if (overflow_target)
            {
                ES_CodeGenerator::JumpTarget *safe = cg.ForwardJump();

                cg.CMP(rlhs, 0);
                cg.CMP(rrhs, 0, ES_NATIVE_CONDITION_LESS_OR_EQUAL);
                cg.Jump(safe, ES_NATIVE_CONDITION_GREATER);

                /* Both are either negative or zero.  This check will catch the
                   case of at least one being zero.  It will "falsely" catch the
                   case of both being zero too, but I'm fine with that. */
                cg.TST(rlhs, rrhs);
                cg.Jump(overflow_target, ES_NATIVE_CONDITION_EQUAL);
                cg.SetJumpTarget(safe);
            }

            cg.SMULL(rlhs, rrhs, SCRATCHI1, rtarget);
        }

        if (overflow_target)
        {
            cg.CMP(SCRATCHI1, 0);
            cg.CMP(SCRATCHI1, -1, ES_NATIVE_CONDITION_NOT_EQUAL);
            cg.Jump(overflow_target, ES_NATIVE_CONDITION_NOT_EQUAL);
            cg.TEQ(SCRATCHI1, rtarget);
            cg.Jump(overflow_target, ES_NATIVE_CONDITION_NEGATIVE);
        }

        if (as_condition)
            cg.TST(rtarget, rtarget);
        break;

    case INT32ARITHMETIC_DIV:
    case INT32ARITHMETIC_REM:
        OP_ASSERT(rhs.virtual_register && rhs_codeword && IsMultipleOfTwo(rhs_codeword->immediate));
        OP_ASSERT(overflow_target);

        ES_CodeGenerator::Register src = rlhs, dst = rtarget;
        int imm = rhs_codeword->immediate;

        if (type == INT32ARITHMETIC_DIV)
        {
            if (imm == 1)
            {
                if (src != dst || set_condition_codes == ES_CodeGenerator::SET_CONDITION_CODES)
                    cg.MOV(src, dst, set_condition_codes);
            }
            else
            {
                /* Check if the result will be an integer. */
                cg.TST(src, imm - 1);
                cg.Jump(overflow_target, ES_NATIVE_CONDITION_NOT_EQUAL);

                cg.MOV(ES_CodeGenerator::Operand(src, Log2(imm), ES_CodeGenerator::SHIFT_ARITHMETIC_RIGHT), dst, set_condition_codes);
            }
        }
        else
        {
            if (imm == 1)
                cg.MOV(0, dst, set_condition_codes);
            else
                cg.AND(src, imm - 1, dst, set_condition_codes);
        }
    }
}

void
ES_Native::EmitInt32Complement(const Operand &target, const Operand &source, BOOL as_condition)
{
    cg.MVN(INTEGER_REGISTER(source), INTEGER_REGISTER(target), as_condition ? ES_CodeGenerator::SET_CONDITION_CODES : ES_CodeGenerator::UNSET_CONDITION_CODES);
}

void
ES_Native::EmitInt32Negate(const Operand &target, const Operand &source, BOOL as_condition, ES_CodeGenerator::JumpTarget *overflow_target)
{
    ES_CodeGenerator::Register src(INTEGER_REGISTER(source));
    ES_CodeGenerator::Register dst(INTEGER_REGISTER(target));
    ES_CodeGenerator::Register use_dst;

    if (src == dst)
        use_dst = SCRATCHI1;
    else
        use_dst = dst;

    cg.RSB(src, 0, use_dst);

    /* Detect zero and INT_MIN cases: check if src and dst have the same sign;
       if they do, our negation obviously failed. */
    cg.TEQ(src, use_dst);
    cg.Jump(overflow_target, ES_NATIVE_CONDITION_POSITIVE_OR_ZERO);

    if (dst != use_dst)
        cg.MOV(use_dst, dst);
}

void
ES_Native::EmitInt32IncOrDec(BOOL inc, const Operand &target, ES_CodeGenerator::JumpTarget *overflow_target)
{
    if (inc)
        cg.ADD(INTEGER_REGISTER(target), 1, INTEGER_REGISTER(target), overflow_target != NULL ? ES_CodeGenerator::SET_CONDITION_CODES : ES_CodeGenerator::UNSET_CONDITION_CODES);
    else
        cg.SUB(INTEGER_REGISTER(target), 1, INTEGER_REGISTER(target), overflow_target != NULL ? ES_CodeGenerator::SET_CONDITION_CODES : ES_CodeGenerator::UNSET_CONDITION_CODES);

    if (overflow_target)
        cg.Jump(overflow_target, ES_NATIVE_CONDITION_OVERFLOW);
}

void
ES_Native::EmitInt32Relational(RelationalType relational_type, const Operand &lhs, const Operand &rhs, const Operand *value_target, ES_CodeGenerator::JumpTarget *true_target, ES_CodeGenerator::JumpTarget *false_target, ArithmeticBlock *arithmetic_block)
{
    OP_ASSERT(!true_target || !false_target);
    OP_ASSERT(lhs.native_register || rhs.native_register);

    if (lhs.native_register && rhs.native_register)
        cg.CMP(INTEGER_REGISTER(lhs), INTEGER_REGISTER(rhs));
    else if (lhs.native_register)
        CompareRegisterToImmediate(cg, INTEGER_REGISTER(lhs), rhs.codeword->immediate);
    else
    {
        CompareRegisterToImmediate(cg, INTEGER_REGISTER(rhs), lhs.codeword->immediate);

        switch (relational_type)
        {
        case RELATIONAL_LT:  relational_type = RELATIONAL_GT; break;
        case RELATIONAL_LTE: relational_type = RELATIONAL_GTE; break;
        case RELATIONAL_GT:  relational_type = RELATIONAL_LT; break;
        case RELATIONAL_GTE: relational_type = RELATIONAL_LTE; break;
        }
    }

    if (value_target && value_target->virtual_register->stack_frame_offset == INT_MAX)
    {
        cg.MOV(ESTYPE_BOOLEAN, SCRATCHI1);

        if (value_target->virtual_register->stack_frame_offset == INT_MAX)
            cg.STR(SCRATCHI1, BASE_REGISTER(value_target->virtual_register), TYPE_OFFSET(value_target->virtual_register));
    }

    ES_NativeJumpCondition true_condition, false_condition;

    switch (relational_type)
    {
    case RELATIONAL_EQ:  true_condition = ES_NATIVE_CONDITION_EQUAL, false_condition = ES_NATIVE_CONDITION_NOT_EQUAL; break;
    case RELATIONAL_NEQ: true_condition = ES_NATIVE_CONDITION_NOT_EQUAL, false_condition = ES_NATIVE_CONDITION_EQUAL; break;
    case RELATIONAL_LT:  true_condition = ES_NATIVE_CONDITION_LESS, false_condition = ES_NATIVE_CONDITION_GREATER_OR_EQUAL; break;
    case RELATIONAL_LTE: true_condition = ES_NATIVE_CONDITION_LESS_OR_EQUAL, false_condition = ES_NATIVE_CONDITION_GREATER; break;
    case RELATIONAL_GT:  true_condition = ES_NATIVE_CONDITION_GREATER, false_condition = ES_NATIVE_CONDITION_LESS_OR_EQUAL; break;
    default: true_condition = ES_NATIVE_CONDITION_GREATER_OR_EQUAL, false_condition = ES_NATIVE_CONDITION_LESS;
    }

    ES_CodeGenerator::OutOfOrderBlock *trampoline = NULL;

    if ((true_target || false_target) && arithmetic_block && FlushToVirtualRegisters(arithmetic_block, TRUE))
    {
        trampoline = cg.StartOutOfOrderBlock();

        FlushToVirtualRegisters(arithmetic_block, FALSE, TRUE);

        if (true_target)
        {
            cg.Jump(true_target);
            true_target = trampoline->GetJumpTarget();
        }
        else
        {
            cg.Jump(false_target);
            false_target = trampoline->GetJumpTarget();
        }

        cg.EndOutOfOrderBlock(FALSE);
    }

    if (value_target)
    {
        cg.MOV(0, SCRATCHI1, ES_CodeGenerator::UNSET_CONDITION_CODES, false_condition);
        cg.MOV(1, SCRATCHI1, ES_CodeGenerator::UNSET_CONDITION_CODES, true_condition);
        cg.STR(SCRATCHI1, BASE_REGISTER(value_target->virtual_register), IVALUE_OFFSET(value_target->virtual_register));
    }

    if (true_target)
        cg.Jump(true_target, true_condition);
    else if (false_target)
        cg.Jump(false_target, false_condition);

    if (trampoline)
        FlushToVirtualRegisters(arithmetic_block, FALSE, FALSE);
}

void
ES_Native::EmitDoubleArithmetic(DoubleArithmeticType type, const Operand &target, const Operand &lhs, const Operand &rhs, BOOL as_condition)
{
    OP_ASSERT(target.native_register);
    OP_ASSERT(lhs.native_register);
    OP_ASSERT(rhs.native_register);

    switch (type)
    {
    case DOUBLEARITHMETIC_ADD:
        cg.FADDD(DOUBLE_REGISTER(lhs), DOUBLE_REGISTER(rhs), DOUBLE_REGISTER(target));
        break;

    case DOUBLEARITHMETIC_SUB:
        cg.FSUBD(DOUBLE_REGISTER(lhs), DOUBLE_REGISTER(rhs), DOUBLE_REGISTER(target));
        break;

    case DOUBLEARITHMETIC_MUL:
        cg.FMULD(DOUBLE_REGISTER(lhs), DOUBLE_REGISTER(rhs), DOUBLE_REGISTER(target));
        break;

    case DOUBLEARITHMETIC_DIV:
        cg.FDIVD(DOUBLE_REGISTER(lhs), DOUBLE_REGISTER(rhs), DOUBLE_REGISTER(target));
    }

    if (as_condition)
    {
        cg.FCMPZD(DOUBLE_REGISTER(target));
        cg.FMSTAT();
    }
}

void
ES_Native::EmitDoubleRelational(RelationalType relational_type, const Operand &lhs, const Operand &rhs, const Operand *value_target, ES_CodeGenerator::JumpTarget *true_target, ES_CodeGenerator::JumpTarget *false_target, ArithmeticBlock *arithmetic_block)
{
    OP_ASSERT(!true_target || !false_target);
    OP_ASSERT(lhs.native_register);
    OP_ASSERT(rhs.native_register);

    cg.FCMPD(DOUBLE_REGISTER(lhs), DOUBLE_REGISTER(rhs));
    cg.FMSTAT();

    if (value_target && value_target->virtual_register->stack_frame_offset == INT_MAX)
    {
        cg.MOV(ESTYPE_BOOLEAN, SCRATCHI1);

        if (value_target->virtual_register->stack_frame_offset == INT_MAX)
            cg.STR(SCRATCHI1, BASE_REGISTER(value_target->virtual_register), TYPE_OFFSET(value_target->virtual_register));
    }

    ES_NativeJumpCondition true_condition, false_condition;

    switch (relational_type)
    {
    case RELATIONAL_EQ:  true_condition = ES_NATIVE_CONDITION_EQUAL, false_condition = ES_NATIVE_CONDITION_NOT_EQUAL; break;
    case RELATIONAL_NEQ: true_condition = ES_NATIVE_CONDITION_NOT_EQUAL, false_condition = ES_NATIVE_CONDITION_EQUAL; break;
    case RELATIONAL_LT:  true_condition = ES_NATIVE_CONDITION_LESS, false_condition = ES_NATIVE_CONDITION_GREATER_OR_EQUAL; break;
    case RELATIONAL_LTE: true_condition = ES_NATIVE_CONDITION_LESS_OR_EQUAL, false_condition = ES_NATIVE_CONDITION_GREATER; break;
    case RELATIONAL_GT:  true_condition = ES_NATIVE_CONDITION_GREATER, false_condition = ES_NATIVE_CONDITION_LESS_OR_EQUAL; break;
    default: true_condition = ES_NATIVE_CONDITION_GREATER_OR_EQUAL, false_condition = ES_NATIVE_CONDITION_LESS;
    }

    ES_CodeGenerator::OutOfOrderBlock *trampoline = NULL;

    if ((true_target || false_target) && arithmetic_block && FlushToVirtualRegisters(arithmetic_block, TRUE))
    {
        trampoline = cg.StartOutOfOrderBlock();

        FlushToVirtualRegisters(arithmetic_block, FALSE, TRUE);

        if (true_target)
        {
            cg.Jump(true_target);
            true_target = trampoline->GetJumpTarget();
        }
        else
        {
            cg.Jump(false_target);
            false_target = trampoline->GetJumpTarget();
        }

        cg.EndOutOfOrderBlock(FALSE);
    }

    if (value_target)
    {
        cg.MOV(1, SCRATCHI1, ES_CodeGenerator::UNSET_CONDITION_CODES, true_condition);
        cg.MOV(0, SCRATCHI1, ES_CodeGenerator::UNSET_CONDITION_CODES, false_condition);

        if (true_condition == ES_NATIVE_CONDITION_LESS || true_condition == ES_NATIVE_CONDITION_LESS_OR_EQUAL)
            cg.MOV(0, SCRATCHI1, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_OVERFLOW);

        cg.STR(SCRATCHI1, BASE_REGISTER(value_target->virtual_register), IVALUE_OFFSET(value_target->virtual_register));
    }

    ES_CodeGenerator::JumpTarget *skip_target = NULL;

    if (true_condition == ES_NATIVE_CONDITION_LESS || true_condition == ES_NATIVE_CONDITION_LESS_OR_EQUAL)
        if (true_target)
            cg.Jump(skip_target = cg.ForwardJump(), ES_NATIVE_CONDITION_OVERFLOW);
        else if (false_target)
            cg.Jump(false_target, ES_NATIVE_CONDITION_OVERFLOW);

    if (true_target)
        cg.Jump(true_target, true_condition);
    else if (false_target)
        cg.Jump(false_target, false_condition);

    if (skip_target)
        cg.SetJumpTarget(skip_target);

    if (trampoline)
        FlushToVirtualRegisters(arithmetic_block, FALSE, FALSE);
}

void
ES_Native::EmitDoubleNegate(const Operand &target, const Operand &source, BOOL as_condition)
{
    OP_ASSERT(target.native_register);
    OP_ASSERT(source.native_register);

    cg.FNEGD(DOUBLE_REGISTER(source), DOUBLE_REGISTER(target));

    if (as_condition)
    {
        cg.FCMPZD(DOUBLE_REGISTER(target));
        cg.FMSTAT();
    }
}

void
ES_Native::EmitDoubleIncOrDec(BOOL inc, const Operand &target)
{
    OP_ASSERT(target.native_register);

    cg.MOV(inc ? 1 : -1, SCRATCHI1);
    cg.FMSR(SCRATCHI1, SCRATCHS1);
    cg.FSITOD(SCRATCHS1, SCRATCHD1);
    cg.FADDD(DOUBLE_REGISTER(target), SCRATCHD1, DOUBLE_REGISTER(target));
}

void
ES_Native::EmitNullOrUndefinedComparison(BOOL eq, VirtualRegister *vr, const Operand *value_target, ES_CodeGenerator::JumpTarget *true_target, ES_CodeGenerator::JumpTarget *false_target)
{
    ES_CodeGenerator::Register type(ES_CodeGenerator::REG_R2);

    cg.LDR(BASE_REGISTER(vr), TYPE_OFFSET(vr), type);

    cg.CMP(type, ESTYPE_NULL);
    cg.CMP(type, ESTYPE_UNDEFINED, ES_NATIVE_CONDITION_NOT_EQUAL);

    if (value_target)
    {
        cg.MOV(ESTYPE_BOOLEAN, SCRATCHI_TYPE);
        cg.MOV(eq ? 0 : 1, SCRATCHI_IVALUE, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_NOT_EQUAL);
        cg.MOV(eq ? 1 : 0, SCRATCHI_IVALUE, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_EQUAL);

        StoreValue(cg, SCRATCHI1, SCRATCHI2, value_target->virtual_register);
    }

    if (true_target)
        cg.Jump(true_target, eq ? ES_NATIVE_CONDITION_EQUAL : ES_NATIVE_CONDITION_NOT_EQUAL);
    else if (false_target)
        cg.Jump(false_target, eq ? ES_NATIVE_CONDITION_NOT_EQUAL : ES_NATIVE_CONDITION_EQUAL);
}

void
ES_Native::EmitStrictNullOrUndefinedComparison(BOOL eq, VirtualRegister *vr, ES_ValueType type, ES_CodeGenerator::JumpTarget *true_target, ES_CodeGenerator::JumpTarget *false_target)
{
    cg.LDR(BASE_REGISTER(vr), TYPE_OFFSET(vr), SCRATCHI1);
    cg.CMP(SCRATCHI1, type);

    if (true_target)
        cg.Jump(true_target, eq ? ES_NATIVE_CONDITION_EQUAL : ES_NATIVE_CONDITION_NOT_EQUAL);
    else if (false_target)
        cg.Jump(false_target, eq ? ES_NATIVE_CONDITION_NOT_EQUAL : ES_NATIVE_CONDITION_EQUAL);
}

void
ES_Native::EmitComparison(BOOL eq, BOOL strict, VirtualRegister *lhs_vr, VirtualRegister *rhs_vr, unsigned handled_types, const Operand *value_target, ES_CodeGenerator::JumpTarget *true_target, ES_CodeGenerator::JumpTarget *false_target)
{
    DECLARE_NOTHING();

    if (!eq)
    {
        ES_CodeGenerator::JumpTarget *temporary = false_target;
        false_target = true_target;
        true_target = temporary;
    }

    ES_CodeGenerator::JumpTarget *use_true_target = true_target;
    ES_CodeGenerator::JumpTarget *use_false_target = false_target;

    if (value_target || !use_true_target)
        use_true_target = cg.ForwardJump();
    if (value_target || !use_false_target)
        use_false_target = cg.ForwardJump();

    ES_CodeGenerator::Register lhs_type(TYPE_REGISTER(ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5));
    ES_CodeGenerator::Register lhs_value(IVALUE_REGISTER(ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5));
    ES_CodeGenerator::Register rhs_type(TYPE_REGISTER(ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7));
    ES_CodeGenerator::Register rhs_value(IVALUE_REGISTER(ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7));

    LoadValue(cg, lhs_vr, ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5);
    LoadValue(cg, rhs_vr, ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7);

    ES_CodeGenerator::JumpTarget *jt_slow_case = cg.ForwardJump();

    cg.CMP(lhs_type, rhs_type);
    cg.Jump(jt_slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    if (handled_types)
    {
        if (handled_types & (ESTYPE_BITS_NULL | ESTYPE_BITS_UNDEFINED))
        {
            cg.CMP(lhs_type, ESTYPE_NULL);
            cg.CMP(lhs_type, ESTYPE_UNDEFINED, ES_NATIVE_CONDITION_NOT_EQUAL);
            cg.Jump(use_true_target, ES_NATIVE_CONDITION_EQUAL);
        }

        ES_CodeGenerator::JumpTarget *jt_next = NULL;

        if (handled_types & (ESTYPE_BITS_BOOLEAN | ESTYPE_BITS_INT32))
        {
            jt_next = cg.ForwardJump();

            if ((handled_types & (ESTYPE_BITS_BOOLEAN | ESTYPE_BITS_INT32)) == (ESTYPE_BITS_BOOLEAN | ESTYPE_BITS_INT32))
            {
#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                cg.Annotate(UNI_L("       EmitComparison(): comparing ESTYPE_INT32+ESTYPE_BOOLEAN\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                cg.CMP(lhs_type, ESTYPE_BOOLEAN);
                cg.CMP(lhs_type, ESTYPE_INT32, ES_NATIVE_CONDITION_NOT_EQUAL);
                cg.Jump(jt_next, ES_NATIVE_CONDITION_NOT_EQUAL);
            }
            else
            {
#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                if (handled_types & ESTYPE_BOOLEAN)
                    cg.Annotate(UNI_L("       EmitComparison(): comparing ESTYPE_BOOLEAN\n"));
                else
                    cg.Annotate(UNI_L("       EmitComparison(): comparing ESTYPE_INT32\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                cg.CMP(lhs_type, (handled_types & ESTYPE_BOOLEAN) ? ESTYPE_BOOLEAN : ESTYPE_INT32);
                cg.Jump(jt_next, ES_NATIVE_CONDITION_NOT_EQUAL);
            }

            cg.TEQ(lhs_value, rhs_value);
            cg.Jump(use_true_target, ES_NATIVE_CONDITION_EQUAL);
            cg.Jump(use_false_target);
        }

        if (handled_types & ESTYPE_BITS_OBJECT)
        {
            if (jt_next)
                cg.SetJumpTarget(jt_next);
            jt_next = cg.ForwardJump();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
            cg.Annotate(UNI_L("       EmitComparison(): comparing ESTYPE_OBJECT\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

            cg.CMP(lhs_type, ESTYPE_OBJECT);
            cg.Jump(jt_next, ES_NATIVE_CONDITION_NOT_EQUAL);

            cg.TEQ(lhs_value, rhs_value);
            cg.Jump(use_true_target, ES_NATIVE_CONDITION_EQUAL);

            ES_CodeGenerator::OutOfOrderBlock *compare_shifty_objects = cg.StartOutOfOrderBlock();

            if (!c_CompareShiftyObjects) c_CompareShiftyObjects = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::CompareShiftyObjects));

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
            cg.MOV(lhs_value, ES_CodeGenerator::REG_R1);
            cg.MOV(rhs_value, ES_CodeGenerator::REG_R2);

            cg.LDR(c_CompareShiftyObjects, ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);

            cg.TEQ(ES_CodeGenerator::REG_R0, 0);
            cg.Jump(use_true_target, ES_NATIVE_CONDITION_NOT_EQUAL);
            cg.Jump(use_false_target);

            cg.EndOutOfOrderBlock(FALSE);

            cg.LDR(OBJECT_MEMBER(lhs_value, ES_Object, object_bits), SCRATCHI1);
            cg.TST(SCRATCHI1, ES_Object::MASK_MULTIPLE_IDENTITIES);
            cg.LDR(OBJECT_MEMBER(rhs_value, ES_Object, object_bits), SCRATCHI1, ES_NATIVE_CONDITION_EQUAL);
            cg.TST(SCRATCHI1, ES_Object::MASK_MULTIPLE_IDENTITIES, ES_NATIVE_CONDITION_EQUAL);
            cg.Jump(compare_shifty_objects->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);

            cg.Jump(use_false_target);
        }

        if (handled_types & ESTYPE_BITS_STRING)
        {
            if (jt_next)
                cg.SetJumpTarget(jt_next);
            jt_next = cg.ForwardJump();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
            cg.Annotate(UNI_L("       EmitComparison(): comparing ESTYPE_STRING\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

            cg.CMP(lhs_type, ESTYPE_STRING);
            cg.Jump(jt_next, ES_NATIVE_CONDITION_NOT_EQUAL);

            cg.TEQ(lhs_value, rhs_value);
            cg.Jump(use_true_target, ES_NATIVE_CONDITION_EQUAL);

            cg.LDR(OBJECT_MEMBER(lhs_value, JString, length), SCRATCHI1);
            cg.LDR(OBJECT_MEMBER(rhs_value, JString, length), SCRATCHI2);
            cg.TEQ(SCRATCHI1, SCRATCHI2);
            cg.Jump(use_false_target, ES_NATIVE_CONDITION_NOT_EQUAL);

            if (!c_Equals) c_Equals = cg.NewFunction(reinterpret_cast<void (*)()>(&Equals));

            cg.MOV(lhs_value, ES_CodeGenerator::REG_R0);
            cg.MOV(rhs_value, ES_CodeGenerator::REG_R1);

            cg.LDR(c_Equals, ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);

            cg.TEQ(ES_CodeGenerator::REG_R0, 0);
            cg.Jump(use_true_target, ES_NATIVE_CONDITION_NOT_EQUAL);
            cg.Jump(use_false_target);
        }

        ES_CodeGenerator::OutOfOrderBlock *update;

        if (jt_next)
        {
            update = cg.StartOutOfOrderBlock();
            cg.SetJumpTarget(jt_next);
        }
        else
            update = NULL;

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
        cg.Annotate(UNI_L("       EmitComparison(): calling ES_Execution_Context::UpdateComparison\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

        if (!c_UpdateComparison) c_UpdateComparison = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::UpdateComparison));

        cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
        MoveImmediateToRegister(cg, cw_index, ES_CodeGenerator::REG_R1);

        cg.LDR(c_UpdateComparison, ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);

        cg.TEQ(ES_CodeGenerator::REG_R0, 0);
        cg.BX(ES_CodeGenerator::REG_R0, ES_NATIVE_CONDITION_NOT_EQUAL);

        if (update)
        {
            cg.EndOutOfOrderBlock();
            cg.SetOutOfOrderContinuationPoint(update);
        }
    }

    cg.SetJumpTarget(jt_slow_case);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    if (strict)
        cg.Annotate(UNI_L("       EmitComparison(): calling ES_Execution_Context::EqStrict\n"));
    else
        cg.Annotate(UNI_L("       EmitComparison(): calling ES_Execution_Context::EqFromMachineCode\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(rhs_vr), ES_CodeGenerator::REG_R2);
    AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(lhs_vr), ES_CodeGenerator::REG_R1);

    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);

    if (strict)
    {
        if (!c_EqStrict) c_EqStrict = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::EqStrict));

        cg.LDR(c_EqStrict, ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);
    }
    else
    {
        MoveImmediateToRegister(cg, cw_index, ES_CodeGenerator::REG_R3);

        if (!c_EqFromMachineCode) c_EqFromMachineCode = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::EqFromMachineCode));

        cg.LDR(c_EqFromMachineCode, ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);
    }

    cg.TEQ(ES_CodeGenerator::REG_R0, 0);
    cg.Jump(use_true_target, ES_NATIVE_CONDITION_NOT_EQUAL);
    cg.Jump(use_false_target);

    if (value_target)
    {
        ES_CodeGenerator::JumpTarget *jt_when_true = true_target ? true_target : cg.ForwardJump();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
        cg.Annotate(UNI_L("       EmitComparison(): storing result in value target\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

        cg.SetJumpTarget(use_true_target);
        cg.MOV(ESTYPE_BOOLEAN, SCRATCHI_TYPE);
        cg.MOV(eq ? 1 : 0, SCRATCHI_IVALUE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, value_target->virtual_register);
        cg.Jump(jt_when_true);

        cg.SetJumpTarget(use_false_target);
        cg.MOV(ESTYPE_BOOLEAN, SCRATCHI_TYPE);
        cg.MOV(eq ? 0 : 1, SCRATCHI_IVALUE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, value_target->virtual_register);
        if (false_target)
            cg.Jump(false_target);

        if (!true_target)
            cg.SetJumpTarget(jt_when_true);
    }
    else
    {
        if (!true_target)
            cg.SetJumpTarget(use_true_target);
        if (!false_target)
            cg.SetJumpTarget(use_false_target);
    }
}

void
ES_Native::EmitInstanceOf(VirtualRegister *object_vr, VirtualRegister *constructor_vr, const Operand *value_target, ES_CodeGenerator::JumpTarget *true_target, ES_CodeGenerator::JumpTarget *false_target)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::JumpTarget *use_true_target = true_target;
    ES_CodeGenerator::JumpTarget *use_false_target = false_target;

    if (value_target || !true_target)
        use_true_target = cg.ForwardJump();
    if (value_target || !false_target)
        use_false_target = cg.ForwardJump();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register constructor_type(TYPE_REGISTER(ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5));
    ES_CodeGenerator::Register constructor(IVALUE_REGISTER(ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5));

    /* right-hand side not an object => exception (because it must be a
       function, and ToObject() never produces a function) */
    if (!IsType(constructor_vr, ESTYPE_OBJECT))
    {
        LoadValue(cg, constructor_vr, ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5);

        cg.CMP(constructor_type, ESTYPE_OBJECT);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
    }
    else
        cg.LDR(BASE_REGISTER(constructor_vr), IVALUE_OFFSET(constructor_vr), constructor);

    ES_CodeGenerator::Register constructor_bits(constructor_type);

    cg.LDR(constructor, 0, constructor_bits);
    cg.AND(constructor_bits, ES_Header::MASK_GCTAG, constructor_bits);
    cg.TEQ(constructor_bits, GCTAG_ES_Object_Function);
    cg.TEQ(constructor_bits, GCTAG_ES_Object_RegExp_CTOR, ES_NATIVE_CONDITION_NOT_EQUAL);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    ES_CodeGenerator::Register object_type(TYPE_REGISTER(ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7));
    ES_CodeGenerator::Register object(IVALUE_REGISTER(ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7));

    if (!IsType(object_vr, ESTYPE_OBJECT))
    {
        LoadValue(cg, object_vr, ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7);

        cg.CMP(object_type, ESTYPE_OBJECT);
        cg.Jump(use_false_target, ES_NATIVE_CONDITION_NOT_EQUAL);
    }
    else
        cg.LDR(BASE_REGISTER(object_vr), IVALUE_OFFSET(object_vr), object);

    ES_CodeGenerator::JumpTarget *simple_case = cg.ForwardJump();
    ES_CodeGenerator::Register constructor_class(object_type);
    ES_CodeGenerator::Register constructor_class_id(ES_CodeGenerator::REG_R8);

    cg.LDR(OBJECT_MEMBER(constructor, ES_Object, klass), constructor_class);
    cg.LDR(OBJECT_MEMBER(constructor_class, ES_Class, class_id), constructor_class_id);

    CompareRegisterToImmediate(cg, constructor_class_id, code->global_object->GetNativeFunctionWithPrototypeClass()->GetId(context));
    CompareRegisterToImmediate(cg, constructor_class_id, code->global_object->GetBuiltInConstructorClass()->GetId(context), ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.Jump(simple_case, ES_NATIVE_CONDITION_EQUAL);

    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
    cg.MOV(object, ES_CodeGenerator::REG_R1);
    cg.MOV(constructor, ES_CodeGenerator::REG_R2);

    EmitFunctionCall(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::InstanceOf)));

    cg.TEQ(ES_CodeGenerator::REG_R0, 0);
    cg.Jump(use_true_target, ES_NATIVE_CONDITION_NOT_EQUAL);
    cg.Jump(use_false_target);

    cg.SetJumpTarget(simple_case);
    cg.LDR(OBJECT_MEMBER(constructor, ES_Object, properties), SCRATCHI1);

    ES_CodeGenerator::Register prototype_type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_CodeGenerator::Register prototype(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

    LoadValue(cg, SCRATCHI1, ES_Function::GetPropertyOffset(ES_Function::PROTOTYPE), ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

    cg.CMP(prototype_type, ESTYPE_OBJECT);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    ES_CodeGenerator::JumpTarget *loop = cg.Here();

    cg.LDR(OBJECT_MEMBER(object, ES_Object, klass), SCRATCHI1);
    cg.LDR(OBJECT_MEMBER(SCRATCHI1, ES_Class, static_data), SCRATCHI1);
    cg.LDR(OBJECT_MEMBER(SCRATCHI1, ES_Class_Data, prototype), object);

    cg.TEQ(object, prototype);
    cg.Jump(use_true_target, ES_NATIVE_CONDITION_EQUAL);

    cg.TEQ(object, 0);
    cg.Jump(loop, ES_NATIVE_CONDITION_NOT_EQUAL);

    if (use_false_target != false_target)
    {
        cg.SetJumpTarget(use_false_target);

        if (value_target)
        {
            cg.MOV(ESTYPE_BOOLEAN, SCRATCHI_TYPE);
            cg.MOV(0, SCRATCHI_IVALUE);

            StoreValue(cg, SCRATCHI1, SCRATCHI2, value_target->virtual_register);
        }
    }

    ES_CodeGenerator::JumpTarget *finished = NULL;

    if (false_target)
        cg.Jump(false_target);
    else
        cg.Jump(finished = cg.ForwardJump());

    if (use_true_target != true_target)
    {
        cg.SetJumpTarget(use_true_target);

        if (value_target)
        {
            cg.MOV(ESTYPE_BOOLEAN, SCRATCHI_TYPE);
            cg.MOV(1, SCRATCHI_IVALUE);

            StoreValue(cg, SCRATCHI1, SCRATCHI2, value_target->virtual_register);
        }

        if (true_target)
            cg.Jump(true_target);
    }

    if (finished)
        cg.SetJumpTarget(finished);
}

void
ES_Native::EmitConditionalJump(ES_CodeGenerator::JumpTarget *true_target, ES_CodeGenerator::JumpTarget *false_target, BOOL check_implicit_boolean, ArithmeticBlock *arithmetic_block)
{
    BOOL more_to_flush = FALSE;

    if (check_implicit_boolean)
    {
        DECLARE_NOTHING();

        cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, implicit_bool), SCRATCHI1);
        cg.TEQ(SCRATCHI1, 0);
    }
    else if (arithmetic_block)
    {
        more_to_flush = FlushToVirtualRegisters(arithmetic_block, TRUE);

        if (more_to_flush)
        {
            ES_CodeGenerator::OutOfOrderBlock *trampoline = cg.StartOutOfOrderBlock();

            FlushToVirtualRegisters(arithmetic_block, FALSE, TRUE);

            if (true_target)
            {
                cg.Jump(true_target);
                true_target = trampoline->GetJumpTarget();
            }
            else
            {
                cg.Jump(false_target);
                false_target = trampoline->GetJumpTarget();
            }

            cg.EndOutOfOrderBlock(FALSE);
        }
    }

    if (true_target)
        cg.Jump(true_target, ES_NATIVE_CONDITION_NOT_EQUAL);
    else if (false_target)
        cg.Jump(false_target, ES_NATIVE_CONDITION_EQUAL);

    if (more_to_flush)
        FlushToVirtualRegisters(arithmetic_block, FALSE, FALSE);
}

void
ES_Native::EmitConditionalJumpOnValue(VirtualRegister *value_vr, ES_CodeGenerator::JumpTarget *true_target, ES_CodeGenerator::JumpTarget *false_target)
{
    ES_CodeGenerator::OutOfOrderBlock *slow_case = cg.StartOutOfOrderBlock();

    if (!c_ReturnAsBoolean) c_ReturnAsBoolean = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Value_Internal::ReturnAsBoolean));

    AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(value_vr), ES_CodeGenerator::REG_R0);
    EmitFunctionCall(c_ReturnAsBoolean);

    cg.TEQ(ES_CodeGenerator::REG_R0, 0);
    cg.EndOutOfOrderBlock();

    ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_CodeGenerator::Register value(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

    LoadValue(cg, value_vr, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

    cg.CMP(type, ESTYPE_BOOLEAN);
    cg.CMP(type, ESTYPE_INT32, ES_NATIVE_CONDITION_NOT_EQUAL);
    cg.Jump(slow_case->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);
    cg.TEQ(value, 0);

    cg.SetOutOfOrderContinuationPoint(slow_case);

    if (true_target)
        cg.Jump(true_target, ES_NATIVE_CONDITION_NOT_EQUAL);
    else
        cg.Jump(false_target, ES_NATIVE_CONDITION_EQUAL);
}

void
ES_Native::EmitNewObject(VirtualRegister *target_vr, ES_Class *klass, ES_CodeWord *first_property, unsigned nproperties)
{
    OP_ASSERT(nproperties == klass->Count());

    EmitSlowCaseCall();
    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    EmitAllocateObject(klass, klass, 0, NULL, NULL, slow_case);

    ES_CodeGenerator::Register object(ALLOCATED_OBJECT_REGISTER);
    ES_CodeGenerator::Register properties(ALLOCATED_OBJECT_NAMED_PROPERTIES_REGISTER);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("       EmitNewObject(): update target virtual register\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    cg.MOV(ESTYPE_OBJECT, SCRATCHI_TYPE);
    cg.MOV(object, SCRATCHI_IVALUE);

    StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("       EmitNewObject(): copy property values\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    for (unsigned index = 0; index < nproperties; ++index)
    {
        ES_Layout_Info layout = klass->GetLayoutInfoAtIndex(ES_LayoutIndex(index));
        ES_StorageType type = layout.GetStorageType();

        ES_CodeGenerator::JumpTarget *null_target = NULL, *int_to_double_target = NULL;

        ES_ValueType value_type;
        VirtualRegister *source_vr = VR(first_property[index].index);
        BOOL known_type = GetType(source_vr, value_type);
        ES_StorageType source_type = ES_STORAGE_WHATEVER;
        if (known_type)
            source_type = ES_Value_Internal::ConvertToStorageType(value_type);

        BOOL skip_write = FALSE;
        if (type != ES_STORAGE_WHATEVER && source_type != type)
            skip_write = EmitTypeConversionCheck(type, source_type, source_vr, current_slow_case->GetJumpTarget(), null_target, int_to_double_target);

        if (!skip_write)
            CopyValueToTypedData(cg, source_vr, properties, layout.GetOffset(), layout.GetStorageType(), NULL, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

        EmitTypeConversionHandlers(source_vr, properties, layout.GetOffset(), null_target, int_to_double_target);
    }

    EmitNewObjectSetPropertyCount(nproperties);
}

#ifdef SUPPORT_INLINE_ALLOCATION
void
ES_Native::EmitNewObjectValue(ES_CodeWord *word, unsigned index)
{
    ES_CodeGenerator::Register properties(ALLOCATED_OBJECT_NAMED_PROPERTIES_REGISTER);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    OpString annotation;
    JString *name = constructor_final_class->GetNameAtIndex(index);

    annotation.AppendFormat(UNI_L("  from EmitNewObjectValue(), for property %u: '"), index);
    annotation.Append(Storage(context, name), Length(name));
    annotation.AppendL("'\n");

    cg.Annotate(annotation.CStr());
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    EmitInitProperty(word, properties, index);
}
#endif // SUPPORT_INLINE_ALLOCATION

void
ES_Native::EmitNewObjectSetPropertyCount(unsigned nproperties)
{
    DECLARE_NOTHING();
    MoveImmediateToRegister(cg, nproperties, SCRATCHI1);
    cg.STR(SCRATCHI1, OBJECT_MEMBER(ALLOCATED_OBJECT_REGISTER, ES_Object, property_count));
}

void
ES_Native::EmitNewArray(VirtualRegister *target_vr, unsigned length, unsigned &capacity, ES_Compact_Indexed_Properties *representation)
{
    DECLARE_NOTHING();

    ES_Class *klass = code->global_object->GetArrayClass();

    ES_CodeGenerator::Register object(ALLOCATED_OBJECT_REGISTER);
    ES_CodeGenerator::Register indexed_properties(ALLOCATED_OBJECT_INDEXED_PROPERTIES_REGISTER);

    if (CanAllocateObject(klass, length))
    {
        ES_CodeGenerator::OutOfOrderBlock *slow_case = cg.StartOutOfOrderBlock();

        EmitInstructionHandlerCall();

        if (!representation)
        {
            cg.LDR(BASE_REGISTER(target_vr), IVALUE_OFFSET(target_vr), object);
            cg.LDR(OBJECT_MEMBER(object, ES_Object, indexed_properties), indexed_properties);
        }

        cg.EndOutOfOrderBlock();

        EmitAllocateObject(klass, klass, 1, &length, representation, slow_case->GetJumpTarget());

        capacity = length;

        ES_CodeGenerator::Register type(SCRATCHI_TYPE), value(SCRATCHI_IVALUE);

        cg.MOV(ESTYPE_OBJECT, type);
        cg.MOV(object, value);

        StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);

        cg.SetOutOfOrderContinuationPoint(slow_case);
    }
    else
    {
        capacity = length;

        EmitInstructionHandlerCall();

        if (!representation)
        {
            cg.LDR(BASE_REGISTER(target_vr), IVALUE_OFFSET(target_vr), object);
            cg.LDR(OBJECT_MEMBER(object, ES_Object, indexed_properties), indexed_properties);
        }
    }

    cg.ADD(indexed_properties, ES_OFFSETOF(ES_Compact_Indexed_Properties, values), indexed_properties);
}

void
ES_Native::EmitNewArrayValue(VirtualRegister *target_vr, ES_CodeWord *word, unsigned index)
{
    ES_CodeGenerator::Register indexed_properties(ALLOCATED_OBJECT_INDEXED_PROPERTIES_REGISTER);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    OpString annotation;
    annotation.AppendFormat(UNI_L("  from EmitNewArrayValue(), for index %u:\n"), index);
    cg.Annotate(annotation.CStr());
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    EmitInitProperty(word, indexed_properties, index);
}

void
ES_Native::EmitNewArrayReset(VirtualRegister *target_vr, unsigned start_index, unsigned end_index)
{
    ES_CodeGenerator::Register indexed_properties(ALLOCATED_OBJECT_INDEXED_PROPERTIES_REGISTER);

    cg.MOV(ESTYPE_UNDEFINED, SCRATCHI_TYPE);
    cg.MOV(0, SCRATCHI_IVALUE);

    for (unsigned index = start_index; index < end_index; ++index)
        StoreValue(cg, SCRATCHI1, SCRATCHI2, indexed_properties, VALUE_OFFSET(index));
}

static void
ES_CheckIsBuiltin(ES_CodeGenerator &cg, ES_Native::VirtualRegister *function_vr, ES_BuiltInFunction builtin, ES_CodeGenerator::JumpTarget *slow_case)
{
    DECLARE_NOTHING();

    cg.LDR(BASE_REGISTER(function_vr), IVALUE_OFFSET(function_vr), SCRATCHI1);
    cg.LDR(OBJECT_MEMBER(SCRATCHI1, ES_Object, object_bits), SCRATCHI1);
    cg.MOV(ES_CodeGenerator::Operand(SCRATCHI1, ES_Object::FUNCTION_ID_SHIFT, ES_CodeGenerator::SHIFT_LOGICAL_RIGHT), SCRATCHI1);
    cg.TEQ(SCRATCHI1, builtin);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
}

void
ES_Native::EmitCall(VirtualRegister *this_vr, VirtualRegister *function_vr, unsigned argc, ES_BuiltInFunction builtin)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::JumpTarget *slow_case;

    if (!current_slow_case)
        EmitSlowCaseCall();

    slow_case = current_slow_case->GetJumpTarget();

    BOOL is_call = CurrentCodeWord()->instruction != ESI_CONSTRUCT;

    if (builtin != ES_BUILTIN_FN_NONE && builtin != ES_BUILTIN_FN_DISABLE)
    {
        if (builtin == ES_BUILTIN_FN_sqrt || builtin == ES_BUILTIN_FN_abs || builtin == ES_BUILTIN_FN_sin || builtin == ES_BUILTIN_FN_cos)
        {
            if (argc != 1)
                goto normal_builtin_call;

            VirtualRegister *arg_vr = this_vr + 2;
            ES_ValueType argtype;

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
            cg.Annotate(UNI_L("    EmitCall(), inlining Math.sqrt() or Math.abs():\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

            if (!GetType(arg_vr, argtype) || argtype == ESTYPE_INT32_OR_DOUBLE)
            {
                unsigned argbits = code->data->profile_data[cw_index + 2];

                if (argbits == ESTYPE_BITS_INT32)
                    argtype = ESTYPE_INT32;
                else if (ARCHITECTURE_HAS_FPU() && ((argbits & ESTYPE_BITS_DOUBLE) != 0 || argtype == ESTYPE_INT32_OR_DOUBLE))
                    argtype = ESTYPE_DOUBLE;
                else
                    goto normal_builtin_call;

                ES_CheckIsBuiltin(cg, function_vr, builtin, slow_case);
                EmitRegisterTypeCheck(arg_vr, argtype, slow_case);
            }
            else if (!ARCHITECTURE_HAS_FPU() && argtype == ESTYPE_DOUBLE)
                goto normal_call;
            else
                ES_CheckIsBuiltin(cg, function_vr, builtin, slow_case);

            if (builtin == ES_BUILTIN_FN_sqrt)
                if (ARCHITECTURE_HAS_FPU())
                {
                    if (argtype == ESTYPE_INT32)
                    {
                        cg.LDR(BASE_REGISTER(arg_vr), IVALUE_OFFSET(arg_vr), SCRATCHI1);
                        cg.FMSR(SCRATCHI1, SCRATCHS1);
                        cg.FSITOD(SCRATCHS1, SCRATCHD1);
                    }
                    else
                        cg.FLDD(BASE_REGISTER(arg_vr), DVALUE_OFFSET(arg_vr), SCRATCHD1);

                    cg.FSQRTD(SCRATCHD1, SCRATCHD1);
                    cg.FSTD(SCRATCHD1, BASE_REGISTER(this_vr), DVALUE_OFFSET(this_vr));
                }
                else
                    goto normal_builtin_call;
            else if (builtin == ES_BUILTIN_FN_abs)
                if (argtype == ESTYPE_INT32)
                {
                    LoadValue(cg, arg_vr, SCRATCHI1, SCRATCHI2);

                    cg.TST(SCRATCHI_IVALUE, 0x80000000u);
                    cg.RSB(SCRATCHI_IVALUE, 0, SCRATCHI_IVALUE, ES_CodeGenerator::SET_CONDITION_CODES, ES_NATIVE_CONDITION_NEGATIVE);
                    cg.Jump(slow_case, ES_NATIVE_CONDITION_NEGATIVE);

                    StoreValue(cg, SCRATCHI1, SCRATCHI2, this_vr);
                }
                else if (ARCHITECTURE_HAS_FPU())
                {
                    cg.FLDD(BASE_REGISTER(arg_vr), DVALUE_OFFSET(arg_vr), SCRATCHD1);
                    cg.FABSD(SCRATCHD1, SCRATCHD1);
                    cg.FSTD(SCRATCHD1, BASE_REGISTER(this_vr), DVALUE_OFFSET(this_vr));
                }
                else
                    goto normal_builtin_call;
            else
            {
                ES_CodeGenerator::Constant *constant;

                if (builtin == ES_BUILTIN_FN_sin)
                {
                    if (!c_MathSin) c_MathSin = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::MathSin));
                    constant = c_MathSin;
                }
                else
                {
                    if (!c_MathCos) c_MathCos = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::MathCos));
                    constant = c_MathCos;
                }

                AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(this_vr), ES_CodeGenerator::REG_R0);
                EmitFunctionCall(constant);
            }
        }
        else if (builtin == ES_BUILTIN_FN_push && argc != 1)
            goto normal_builtin_call;
        else if (builtin == ES_BUILTIN_FN_pow)
        {
            if (argc != 2)
                goto normal_builtin_call;

            if ((code->data->profile_data[cw_index + 2] & ~(ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE)) != 0)
                goto normal_builtin_call;

            ES_CheckIsBuiltin(cg, function_vr, builtin, slow_case);

            VirtualRegister *number_vr = this_vr + 2, *exp_vr = this_vr + 3;
            ES_Value_Internal exp_value;

            if (IsImmediate(exp_vr, exp_value, TRUE))
                if (exp_value.IsNumber())
                {
                    double exp = exp_value.GetNumAsDouble();

                    if (op_isfinite(exp))
                        if (exp == 2)
                        {
                            if (!IsType(number_vr, ESTYPE_DOUBLE))
                            {
                                ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
                                ES_CodeGenerator::Register value(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

                                LoadValue(cg, number_vr, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

                                if (!IsType(number_vr, ESTYPE_INT32))
                                {
                                    cg.CMP(type, ESTYPE_INT32);
                                    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
                                }

                                cg.SMULL(value, value, SCRATCHI1, value);
                                cg.TEQ(value, 0);
                                cg.Jump(slow_case, ES_NATIVE_CONDITION_NEGATIVE);
                                cg.TEQ(SCRATCHI1, 0);
                                cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

                                StoreValue(cg, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3, this_vr);
                                return;
                            }
                        }
                        else if (ARCHITECTURE_HAS_FPU() && (exp == 0.5 || exp == -0.5))
                        {
                            ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
                            ES_CodeGenerator::Register value(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

                            LoadValue(cg, number_vr, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

                            BOOL is_int32 = IsType(number_vr, ESTYPE_INT32);
                            BOOL is_double = IsType(number_vr, ESTYPE_DOUBLE);

                            if (!is_int32 && !is_double)
                            {
                                cg.CMP(type, ESTYPE_INT32);
                                cg.FMSR(value, SCRATCHS1, ES_NATIVE_CONDITION_EQUAL);
                                cg.FSITOD(SCRATCHS1, SCRATCHD1, ES_NATIVE_CONDITION_EQUAL);
                                cg.FMDRR(type, value, SCRATCHD1, ES_NATIVE_CONDITION_LESS);
                                cg.Jump(slow_case, ES_NATIVE_CONDITION_GREATER);
                            }

                            cg.FSQRTD(SCRATCHD1, SCRATCHD1);

                            if (exp == -0.5)
                            {
                                cg.FLDD(cg.NewConstant(1.0), SCRATCHD2);
                                cg.FDIVD(SCRATCHD2, SCRATCHD1, SCRATCHD1);
                            }

                            cg.FMRRD(SCRATCHD1, type, value);
                            cg.CMP(type, ESTYPE_DOUBLE);
                            cg.MOV(ESTYPE_DOUBLE, type, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_GREATER);

                            StoreValue(cg, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3, this_vr);
                            return;
                        }
                }

            if (!c_MathPow) c_MathPow = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::MathPow));

            AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(this_vr), ES_CodeGenerator::REG_R0);
            EmitFunctionCall(c_MathPow);

            cg.TEQ(ES_CodeGenerator::REG_R0, 0);
            cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
        }
        else if (builtin == ES_BUILTIN_FN_charCodeAt || builtin == ES_BUILTIN_FN_charAt)
        {
            ES_CheckIsBuiltin(cg, function_vr, builtin, slow_case);

            EmitRegisterTypeCheck(this_vr, ESTYPE_STRING, slow_case);
            if (argc > 0)
                EmitRegisterTypeCheck(this_vr + 2, ESTYPE_INT32, slow_case);

            EmitInt32StringIndexedGet(this_vr, this_vr, argc > 0 ? this_vr + 2 : NULL, 0, builtin == ES_BUILTIN_FN_charCodeAt);
        }
        else if (builtin == ES_BUILTIN_FN_fromCharCode && argc == 1)
        {
            VirtualRegister *arg_vr = this_vr + 2;
            ES_Value_Internal arg_value;

            if (IsImmediate(arg_vr, arg_value, TRUE))
            {
                if (arg_value.IsInt32())
                {
                    int arg = arg_value.GetNumAsInt32();
                    if (arg >= 0 && arg < STRING_NUMCHARS)
                    {
                        ES_CheckIsBuiltin(cg, function_vr, builtin, slow_case);

                        ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
                        ES_CodeGenerator::Register value(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

                        cg.MOV(ESTYPE_STRING, type);
                        MovePointerToRegister(cg, context->rt_data->strings[STRING_nul + arg], value);
                        StoreValue(cg, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3, this_vr);
                        return;
                    }
                }
                goto normal_builtin_call;
            }
            else
            {
                ES_CheckIsBuiltin(cg, function_vr, builtin, slow_case);

                EmitRegisterTypeCheck(arg_vr, ESTYPE_INT32, slow_case);
                cg.LDR(BASE_REGISTER(arg_vr), IVALUE_OFFSET(arg_vr), SCRATCHI2);
                CompareRegisterToImmediate(cg, SCRATCHI2, STRING_NUMCHARS);
                cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

                ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
                ES_CodeGenerator::Register value(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

                cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, rt_data), value);
                AddImmediateToRegister(cg, value, ES_OFFSETOF(ESRT_Data, strings) + STRING_nul * sizeof(void *), value);
                cg.ADD(value, ES_CodeGenerator::Operand(SCRATCHI2, 2), value);
                cg.LDR(value, 0, value);

                cg.MOV(ESTYPE_STRING, type);
                StoreValue(cg, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3, this_vr);
            }
        }
        else if (builtin == ES_BUILTIN_FN_push)
        {
#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
            cg.Annotate(UNI_L("       EmitCall(): inline Array.prototype.push\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

            EmitRegisterTypeCheck(this_vr, ESTYPE_OBJECT, slow_case);

            ES_Value_Internal value;
            EmitInt32IndexedPut(this_vr, NULL, 0, this_vr + 2, FALSE, FALSE, value, TRUE);
        }
        else if (builtin == ES_BUILTIN_FN_OTHER)
        {
        normal_builtin_call:
            ES_CodeGenerator::Register header(ES_CodeGenerator::REG_R1);
            ES_CodeGenerator::Register function(ES_CodeGenerator::REG_R4);

            cg.LDR(BASE_REGISTER(function_vr), IVALUE_OFFSET(function_vr), function);
            cg.LDR(function, 0, header);
            cg.AND(header, ES_Header::MASK_GCTAG, header);
            cg.TEQ(header, GCTAG_ES_Object_Function);
            cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

            ES_CodeGenerator::Register fncode(ES_CodeGenerator::REG_R3);
            cg.LDR(OBJECT_MEMBER(function, ES_Function, function_code), fncode);
            cg.TEQ(fncode, 0);
            cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
            cg.MOV(argc, ES_CodeGenerator::REG_R1);
            AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(this_vr->index + 2), ES_CodeGenerator::REG_R2);
            AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(this_vr), ES_CodeGenerator::REG_R3);

            cg.LDR(OBJECT_MEMBER(function, ES_Function, data.builtin.call), ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);

            ES_CodeGenerator::OutOfOrderBlock *exception_thrown = cg.StartOutOfOrderBlock();
            {
                /* This call doesn't return. */

                if (!c_ThrowFromMachineCode) c_ThrowFromMachineCode = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::ThrowFromMachineCode));

                cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
                EmitFunctionCall(c_ThrowFromMachineCode);
            }
            cg.EndOutOfOrderBlock(FALSE);

            cg.TEQ(ES_CodeGenerator::REG_R0, 0);
            cg.Jump(exception_thrown->GetJumpTarget(), ES_NATIVE_CONDITION_EQUAL);
        }
        else
            goto normal_call;
    }
    else
    {
    normal_call:
        ES_CodeGenerator::Register object_bits(ES_CodeGenerator::REG_R1);
        ES_CodeGenerator::Register function(ES_CodeGenerator::REG_R2);

        cg.LDR(BASE_REGISTER(function_vr), IVALUE_OFFSET(function_vr), function);
        cg.LDR(OBJECT_MEMBER(function, ES_Object, object_bits), object_bits);
        cg.TST(object_bits, is_call ? ES_Object::MASK_IS_NATIVE_FN : ES_Object::MASK_IS_DISPATCHED_CTOR);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

        if (is_call)
        {
            cg.LDR(OBJECT_MEMBER(function, ES_Function, function_code), function);
            cg.LDR(OBJECT_MEMBER(function, ES_FunctionCode, native_dispatcher), function);
            cg.TEQ(function, 0);
            cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
        }
        else
            cg.LDR(OBJECT_MEMBER(function, ES_Function, data.native.native_ctor_dispatcher), function);

        cg.MOV(argc, PARAMETERS_COUNT);
        AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(this_vr->index), REGISTER_FRAME_POINTER);

        cg.BLX(function);

        SetFunctionCallReturnTarget(cg.Here());

        AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, -VALUE_OFFSET(this_vr->index), REGISTER_FRAME_POINTER);
    }
}

void
ES_Native::EmitRedirectedCall(VirtualRegister *function_vr, VirtualRegister *apply_vr, BOOL check_constructor_object_result)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register apply(ES_CodeGenerator::REG_R2);

    cg.LDR(BASE_REGISTER(apply_vr), IVALUE_OFFSET(apply_vr), apply);
    cg.LDR(OBJECT_MEMBER(apply, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_IS_APPLY_FN);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    /* Sets stack_frame_padding as a side-effect. */
    StackSpaceAllocated();

    cg.LDR(ES_CodeGenerator::REG_SP, stack_frame_padding, PARAMETERS_COUNT);
    cg.CMP(PARAMETERS_COUNT, ES_REDIRECTED_CALL_MAXIMUM_ARGUMENTS);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER);

    ES_CodeGenerator::Register function(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register function_code(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register dispatcher(ES_CodeGenerator::REG_R5);

    cg.LDR(BASE_REGISTER(function_vr), IVALUE_OFFSET(function_vr), function);
    cg.LDR(OBJECT_MEMBER(function, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_IS_NATIVE_FN);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
    cg.LDR(OBJECT_MEMBER(function, ES_Function, function_code), function_code);
    cg.LDR(OBJECT_MEMBER(function_code, ES_FunctionCode, native_dispatcher), dispatcher);
    cg.TEQ(dispatcher, 0);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    if (constructor)
    {
        LoadValue(cg, VR(0), ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7);
        cg.PUSH((1 << 6) | (1 << 7));
    }

    cg.STR(function, BASE_REGISTER(VR(1)), IVALUE_OFFSET(VR(1)));
    cg.BLX(dispatcher);

    SetFunctionCallReturnTarget(cg.Here());

    if (constructor)
    {
        cg.POP((1 << 6) | (1 << 7));

        ES_CodeGenerator::JumpTarget *keep_object_result = NULL;
        if (check_constructor_object_result)
        {
            /* Restore reg[0] if return value isn't an object. */
            cg.LDR(BASE_REGISTER(VR(0)), TYPE_OFFSET(VR(0)), SCRATCHI1);
            cg.CMP(SCRATCHI1, ESTYPE_OBJECT);

            keep_object_result = cg.ForwardJump();
            cg.Jump(keep_object_result, ES_NATIVE_CONDITION_EQUAL);
        }
        StoreValue(cg, ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7, VR(0));
        if (keep_object_result)
            cg.SetJumpTarget(keep_object_result);
    }
}

void
ES_Native::EmitApply(VirtualRegister *apply_vr, VirtualRegister *this_vr, VirtualRegister *function_vr, unsigned argc)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register apply(ES_CodeGenerator::REG_R2);

    cg.LDR(BASE_REGISTER(apply_vr), IVALUE_OFFSET(apply_vr), apply);
    cg.LDR(OBJECT_MEMBER(apply, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_IS_APPLY_FN);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    ES_CodeGenerator::Register function(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register function_code(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register dispatcher(ES_CodeGenerator::REG_R5);

    cg.LDR(BASE_REGISTER(function_vr), IVALUE_OFFSET(function_vr), function);
    cg.LDR(OBJECT_MEMBER(function, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_IS_NATIVE_FN);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
    cg.LDR(OBJECT_MEMBER(function, ES_Function, function_code), function_code);
    cg.LDR(OBJECT_MEMBER(function_code, ES_FunctionCode, native_dispatcher), dispatcher);
    cg.TEQ(dispatcher, 0);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    AddImmediateToRegister(cg, VALUE_OFFSET(this_vr), REGISTER_FRAME_POINTER);
    MoveImmediateToRegister(cg, argc, PARAMETERS_COUNT);

    cg.BLX(dispatcher);

    SetFunctionCallReturnTarget(cg.Here());

    AddImmediateToRegister(cg, -VALUE_OFFSET(this_vr), REGISTER_FRAME_POINTER);
}

void
ES_Native::EmitEval(VirtualRegister *this_vr, VirtualRegister *function_vr, unsigned argc)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register function(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register object_bits(ES_CodeGenerator::REG_R3);

    cg.LDR(BASE_REGISTER(function_vr), IVALUE_OFFSET(function_vr), function);
    cg.LDR(OBJECT_MEMBER(function, ES_Object, object_bits), object_bits);
    cg.TST(object_bits, ES_Object::MASK_IS_EVAL_FN);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    if (!c_EvalFromMachineCode) c_EvalFromMachineCode = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::EvalFromMachineCode));

    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
    MoveImmediateToRegister(cg, cw_index, ES_CodeGenerator::REG_R1);
    cg.LDR(c_EvalFromMachineCode, ES_CodeGenerator::REG_LR);
    cg.BLX(ES_CodeGenerator::REG_LR);
}

ES_CodeGenerator::OutOfOrderBlock *
ES_Native::EmitInlinedCallPrologue(VirtualRegister *this_vr, VirtualRegister *function_vr, ES_Function *function, unsigned char *mark_failure, unsigned argc, BOOL function_fetched)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::OutOfOrderBlock *failure = cg.StartOutOfOrderBlock();

    cg.SUB(REGISTER_FRAME_POINTER, VALUE_OFFSET(this_vr->index), REGISTER_FRAME_POINTER);

    if (!function_fetched)
    {
        MoveImmediateToRegister(cg, ESTYPE_OBJECT, SCRATCHI_TYPE);
        MovePointerToRegister(cg, function, SCRATCHI_IVALUE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, function_vr);
    }

    if (!c_code) c_code = cg.NewConstant(code);

    cg.LDR(c_code, SCRATCHI1);
    cg.MOV(1, SCRATCHI2);
    cg.STR(SCRATCHI2, OBJECT_MEMBER(SCRATCHI1, ES_Code, has_failed_inlined_function));

    MovePointerToRegister(cg, mark_failure, SCRATCHI1);
    cg.LDRB(SCRATCHI1, 0, SCRATCHI2);
    cg.ORR(SCRATCHI2, ES_CodeStatic::GET_FAILED_INLINE, SCRATCHI2);
    cg.STRB(SCRATCHI2, SCRATCHI1, 0);

    if (!c_ForceUpdateNativeDispatcher) c_ForceUpdateNativeDispatcher = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::ForceUpdateNativeDispatcher));

    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
    MoveImmediateToRegister(cg, cw_index + (constructor ? code->data->codewords_count : 0), ES_CodeGenerator::REG_R1);
    cg.LDR(c_ForceUpdateNativeDispatcher, ES_CodeGenerator::REG_LR);
    cg.BLX(ES_CodeGenerator::REG_LR);
    cg.BX(ES_CodeGenerator::REG_R0);

    cg.EndOutOfOrderBlock(FALSE);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("    EmitInlinedCallPrologue():\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    AddImmediateToRegister(cg, VALUE_OFFSET(this_vr->index), REGISTER_FRAME_POINTER);

    return failure;
}

void
ES_Native::EmitInlinedCallEpilogue(VirtualRegister *this_vr, VirtualRegister *function_vr, unsigned argc)
{
    AddImmediateToRegister(cg, -VALUE_OFFSET(this_vr->index), REGISTER_FRAME_POINTER);
}

void
ES_Native::EmitReturn()
{
    if (cw_index != code->data->instruction_offsets[code->data->instruction_count - 1])
        cg.Jump(epilogue_jump_target);
}

void
ES_Native::EmitNormalizeValue(VirtualRegister *vr, ES_CodeGenerator::JumpTarget *slow_case)
{
    ES_CodeGenerator::JumpTarget *is_int = NULL;

    if (!ARCHITECTURE_HAS_FPU())
    {
        EmitRegisterTypeCheck(vr, ESTYPE_INT32, slow_case);
        return;
    }
    else
    {
        is_int = cg.ForwardJump();
        EmitRegisterTypeCheck(vr, ESTYPE_INT32, is_int, TRUE);
    }

    EmitRegisterTypeCheck(vr, ESTYPE_DOUBLE, slow_case);

    if (!slow_case)
    {
        OP_ASSERT(current_slow_case != NULL);
        slow_case = current_slow_case->GetJumpTarget();
    }

    cg.FLDD(BASE_REGISTER(vr), DVALUE_OFFSET(vr), ES_CodeGenerator::VFPREG_D0);
    cg.FTOSID(SCRATCHD1, ES_CodeGenerator::VFPREG_S4);
    cg.FSITOD(ES_CodeGenerator::VFPREG_S4, ES_CodeGenerator::VFPREG_D1);
    cg.FCMPD(ES_CodeGenerator::VFPREG_D0, ES_CodeGenerator::VFPREG_D1);
    cg.FMSTAT();

    // checks both in-equality and un-ordered
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.FMRS(ES_CodeGenerator::VFPREG_S4, SCRATCHI_IVALUE);
    MoveImmediateToRegister(cg, ESTYPE_INT32, SCRATCHI_TYPE);

    StoreValue(cg, SCRATCHI1, SCRATCHI2, vr);

    cg.SetJumpTarget(is_int);
}

void
ES_Native::EmitInt32IndexedGet(VirtualRegister *target_vr, VirtualRegister *object_vr, VirtualRegister *index_vr, unsigned constant_index)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register properties(ES_CodeGenerator::REG_R3);

    LoadObjectOperand(object_vr->index, object);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_SIMPLE_COMPACT_INDEXED);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, indexed_properties), properties);

    ES_CodeGenerator::Register index(ES_CodeGenerator::REG_R4);

    /* Check that (constant) index is within array's capacity: */
    if (!index_vr)
    {
        cg.LDR(OBJECT_MEMBER(properties, ES_Compact_Indexed_Properties, capacity), SCRATCHI2);
        CompareRegisterToImmediate(cg, SCRATCHI2, constant_index);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_LOWER_OR_SAME);

        AddImmediateToRegister(cg, properties, VALUE_OFFSET(constant_index) + ES_OFFSETOF(ES_Compact_Indexed_Properties, values[0]), index);
    }
    else
    {
        cg.LDR(BASE_REGISTER(index_vr), IVALUE_OFFSET(index_vr), index);
        cg.LDR(OBJECT_MEMBER(properties, ES_Compact_Indexed_Properties, capacity), SCRATCHI1);
        cg.CMP(index, SCRATCHI1);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

        cg.ADD(properties, ES_CodeGenerator::Operand(index, 3), index);
        cg.ADD(index, ES_OFFSETOF(ES_Compact_Indexed_Properties, values[0]), index);
    }

    if (property_value_write_vr)
    {
        OP_ASSERT(property_value_write_vr == target_vr);

        cg.LDR(index, VALUE_TYPE_OFFSET, SCRATCHI1);
        cg.CMP(SCRATCHI1, ESTYPE_UNDEFINED);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

        SetPropertyValueTransferRegister(INTEGER_NR(this, VALUE_TRANSFER_POINTER), 0, TRUE);
        cg.MOV(index, VALUE_TRANSFER_POINTER);

        if (current_slow_case)
        {
            ES_CodeGenerator::OutOfOrderBlock *recover = cg.StartOutOfOrderBlock();

            cg.SetOutOfOrderContinuationPoint(current_slow_case);
            current_slow_case = NULL;

            AddImmediateToRegister(cg, REGISTER_FRAME_POINTER, VALUE_OFFSET(target_vr->index), VALUE_TRANSFER_POINTER);

            cg.EndOutOfOrderBlock();
            cg.SetOutOfOrderContinuationPoint(recover);
        }
    }
    else
    {
        LoadValue(cg, index, 0, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

        cg.CMP(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3), ESTYPE_UNDEFINED);
        cg.TEQ(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3), 0, ES_NATIVE_CONDITION_EQUAL);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

        StoreValue(cg, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3, target_vr);
    }
}

void
ES_Native::EmitInt32IndexedPut(VirtualRegister *object_vr, VirtualRegister *index_vr, unsigned constant_index, VirtualRegister *source_vr, BOOL known_type, BOOL known_value, const ES_Value_Internal &the_value, BOOL is_push)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register indexed_properties(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register klass(ES_CodeGenerator::REG_R4);

    unsigned length_offset = code->global_object->GetArrayClass()->GetLayoutInfoAtIndex(ES_LayoutIndex(0)).GetOffset();

    LoadObjectOperand(object_vr->index, object);

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case;

    if (!is_light_weight && property_value_read_vr && property_value_nr)
    {
        ES_CodeGenerator::OutOfOrderBlock *flush_object_vr = cg.StartOutOfOrderBlock();

        MoveImmediateToRegister(cg, ESTYPE_OBJECT, SCRATCHI1);
        cg.STR(SCRATCHI1, BASE_REGISTER(object_vr), TYPE_OFFSET(object_vr));
        cg.STR(object, BASE_REGISTER(object_vr), IVALUE_OFFSET(object_vr));

        cg.Jump(current_slow_case_main);
        cg.EndOutOfOrderBlock(FALSE);

        slow_case = flush_object_vr->GetJumpTarget();
    }
    else
        slow_case = current_slow_case->GetJumpTarget();

    cg.LDR(OBJECT_MEMBER(object, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_MUTABLE_COMPACT_INDEXED);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, indexed_properties), indexed_properties);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, klass), klass);

    ES_CodeGenerator::Register index(ES_CodeGenerator::REG_R5);
    ES_CodeGenerator::Register value(ES_CodeGenerator::REG_R6);

    if (is_push)
    {
        /* Theory: When inlining Array.prototype.push, do the class check
           early since we'll need to get the 'length' property from the
           array object. If the class id check holds, it is not just
           guaranteed that the object is an array, but also that the type
           of length is UINT32. */
        cg.LDR(OBJECT_MEMBER(klass, ES_Class, class_id), SCRATCHI2);
        CompareRegisterToImmediate(cg, SCRATCHI2, code->global_object->GetArrayClass()->GetId(context));
        cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
        cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), SCRATCHI1);
        cg.LDR(SCRATCHI1, length_offset, index);
    }
    else if (index_vr)
        cg.LDR(BASE_REGISTER(index_vr), IVALUE_OFFSET(index_vr), index);
    else
        MoveImmediateToRegister(cg, constant_index, index);

    ES_CodeGenerator::OutOfOrderBlock *check_capacity_and_length = !is_push ? cg.StartOutOfOrderBlock() : NULL;
    ES_CodeGenerator::JumpTarget *no_length_update = !is_push ? check_capacity_and_length->GetContinuationTarget() : NULL;

    /* If the index is equal to or higher than capacity, the put can't be
       performed, so jump to the slow case.  This is an unsigned comparison, so
       if the index is negative, it will appear higher than capacity (which
       can be assumed to be at most INT_MAX.) */
    cg.LDR(OBJECT_MEMBER(indexed_properties, ES_Compact_Indexed_Properties, capacity), SCRATCHI1);
    cg.CMP(index, SCRATCHI1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

    /* Check if object requires a length update + if it is an instance of Array */
    if (!is_push)
    {
        cg.LDR(object, 0, SCRATCHI2);
        cg.AND(SCRATCHI2, ES_Header::MASK_GCTAG, SCRATCHI2);
        cg.TEQ(SCRATCHI2, GCTAG_ES_Object_Array);
        cg.Jump(no_length_update, ES_NATIVE_CONDITION_NOT_EQUAL);
        cg.LDR(OBJECT_MEMBER(klass, ES_Class, class_id), SCRATCHI2);
        CompareRegisterToImmediate(cg, SCRATCHI2, code->global_object->GetArrayClass()->GetId(context));
        cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
    }

    /* We get to this slow case because index>=top.  We get this far in the slow
       case if index<capacity.  Since top is defined to be the same as capacity
       on non-arrays, index>=top&&index<capacity => instanceof Array.  Thus we
       can assume that the first named property on the object is 'length'. */

    cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), SCRATCHI1);

    /* Check if 'length' is higher than the assigned index, in which case we
       shouldn't update it. */
    if (!is_push)
    {
        cg.LDR(SCRATCHI1, length_offset, SCRATCHI2);
        cg.CMP(index, SCRATCHI2);
        cg.Jump(no_length_update, ES_NATIVE_CONDITION_LOWER);
    }

    /* Update 'length' to index+1.  We're ignoring overflow here, see comment in
       EmitInt32IndexedPut() in es_native_ia32.cpp for an explanation.  Also
       update 'top' which must have been equal to 'length', since 'length' is
       obviously less 'capacity'. */
    cg.ADD(index, 1, SCRATCHI2);
    cg.STR(SCRATCHI2, SCRATCHI1, length_offset);
    cg.STR(SCRATCHI2, OBJECT_MEMBER(indexed_properties, ES_Compact_Indexed_Properties, top));

    if (check_capacity_and_length)
    {
        cg.EndOutOfOrderBlock();
        cg.LDR(OBJECT_MEMBER(indexed_properties, ES_Compact_Indexed_Properties, top), SCRATCHI1);
        cg.CMP(index, SCRATCHI1);
        cg.Jump(check_capacity_and_length->GetJumpTarget(), ES_NATIVE_CONDITION_HIGHER_OR_SAME);
        cg.SetOutOfOrderContinuationPoint(check_capacity_and_length);
    }

    cg.ADD(indexed_properties, ES_CodeGenerator::Operand(index, 3), value);
    cg.ADD(value, ES_OFFSETOF(ES_Compact_Indexed_Properties, values), value);

    ES_CodeGenerator::Register value_type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_CodeGenerator::Register value_value(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

    if (known_type && the_value.Type() != ESTYPE_DOUBLE && the_value.Type() != ESTYPE_STRING && the_value.Type() != ESTYPE_OBJECT)
    {
        cg.MOV(the_value.Type(), value_type);

        if (known_value || the_value.IsNullOrUndefined())
            switch (the_value.Type())
            {
            case ESTYPE_NULL:
                break;

            case ESTYPE_UNDEFINED:
                cg.MOV(1, value_value);
                break;

            case ESTYPE_BOOLEAN:
                cg.MOV(the_value.GetBoolean() ? 1 : 0, value_value);
                break;

            case ESTYPE_INT32:
                MoveImmediateToRegister(cg, the_value.GetInt32(), value_value);
            }
        else
            cg.LDR(BASE_REGISTER(source_vr), IVALUE_OFFSET(source_vr), value_value);
    }
    else
    {
        LoadValue(cg, source_vr, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

        if (!known_type)
        {
            cg.CMP(value_type, ESTYPE_UNDEFINED);
            cg.MOV(1, value_value, ES_CodeGenerator::UNSET_CONDITION_CODES, ES_NATIVE_CONDITION_EQUAL);
        }
    }

    cg.STRD(ES_CodeGenerator::REG_R2, value, 0);
    if (is_push)
    {
        /* The return value of Array.prototype.push is the new value of the
           'length' property. We know that 'object_vr' is the receiver of
           the call to Array.prototype.push and that that register doubles as
           return register, so write the new length here instead of doing a
           EmitLengthGet which will check types that we've already checked. */
        cg.MOV(ESTYPE_INT32, SCRATCHI_TYPE);
        cg.ADD(index, 1, SCRATCHI_IVALUE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, object_vr);
    }
}

void
ES_Native::EmitInt32ByteArrayGet(VirtualRegister *target_vr, VirtualRegister *object_vr, VirtualRegister *index_vr, unsigned constant_index)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register bytearray(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register index(ES_CodeGenerator::REG_R4);

    LoadObjectOperand(object_vr->index, object);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_BYTE_ARRAY_INDEXED);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, indexed_properties), bytearray);

    if (index_vr)
        cg.LDR(BASE_REGISTER(index_vr), IVALUE_OFFSET(index_vr), index);
    else
        MoveImmediateToRegister(cg, constant_index, index);

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Byte_Array_Indexed, capacity), SCRATCHI1);
    cg.CMP(index, SCRATCHI1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Byte_Array_Indexed, byte_array), bytearray);

    cg.LDRB(bytearray, index, ES_CodeGenerator::SHIFT_LOGICAL_LEFT, 0, TRUE, SCRATCHI_IVALUE);
    cg.MOV(ESTYPE_INT32, SCRATCHI_TYPE);

    StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);
}

void
ES_Native::EmitInt32ByteArrayPut(VirtualRegister *object_vr, VirtualRegister *index_vr, unsigned constant_index, VirtualRegister *source_vr, int *known_value)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register bytearray(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register index(ES_CodeGenerator::REG_R4);

    LoadObjectOperand(object_vr->index, object);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_BYTE_ARRAY_INDEXED);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, indexed_properties), bytearray);

    if (index_vr)
        cg.LDR(BASE_REGISTER(index_vr), IVALUE_OFFSET(index_vr), index);
    else
        MoveImmediateToRegister(cg, constant_index, index);

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Byte_Array_Indexed, capacity), SCRATCHI1);
    cg.CMP(index, SCRATCHI1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

    if (known_value)
        MoveImmediateToRegister(cg, static_cast<signed char>(*known_value), SCRATCHI1);
    else
        cg.LDR(BASE_REGISTER(source_vr), IVALUE_OFFSET(source_vr), SCRATCHI1);

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Byte_Array_Indexed, byte_array), bytearray);
    cg.STRB(SCRATCHI1, bytearray, index, ES_CodeGenerator::SHIFT_LOGICAL_LEFT, 0, TRUE);
}

static void
EmitTypedArrayPrecondition(ES_Native &native, ES_Native::VirtualRegister *object_vr, ES_Native::VirtualRegister *index_vr, unsigned constant_index)
{
    DECLARE_NOTHING();

    ES_CodeGenerator &cg = native.cg;

    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register bytearray(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register index(ES_CodeGenerator::REG_R5);

    if (!native.current_slow_case)
        native.EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = native.current_slow_case->GetJumpTarget();

    native.LoadObjectOperand(object_vr->index, object);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, object_bits), SCRATCHI2);
    cg.TST(SCRATCHI2, ES_Object::MASK_TYPE_ARRAY_INDEXED);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, indexed_properties), bytearray);

    if (index_vr)
        cg.LDR(BASE_REGISTER(index_vr), IVALUE_OFFSET(index_vr), index);
    else
        MoveImmediateToRegister(cg, constant_index, index);
}

void
ES_Native::EmitInt32TypedArrayGet(VirtualRegister *target_vr, VirtualRegister *object_vr, VirtualRegister *index_vr, unsigned constant_index, unsigned type_array_bits)
{
    DECLARE_NOTHING();

    ANNOTATE("EmitInt32TypedArrayGet");

    ES_CodeGenerator::Register bytearray(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register index(ES_CodeGenerator::REG_R5);
    ES_CodeGenerator::Register kind(ES_CodeGenerator::REG_R6);

    EmitTypedArrayPrecondition(*this, object_vr, index_vr, constant_index);

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Type_Array_Indexed, capacity), SCRATCHI1);
    cg.CMP(index, SCRATCHI1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Type_Array_Indexed, kind), kind);

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Type_Array_Indexed, byte_offset), SCRATCHI1);
    cg.LDR(OBJECT_MEMBER(bytearray, ES_Type_Array_Indexed, byte_array), bytearray);
    cg.LDR(OBJECT_MEMBER(bytearray, ES_Byte_Array_Indexed, byte_array), bytearray);
    cg.ADD(SCRATCHI1, bytearray, bytearray);

    ES_CodeGenerator::JumpTarget *done_int_target = NULL;
    ES_CodeGenerator::JumpTarget *done_target = NULL;

    unsigned array_kind = ES_Type_Array_Indexed::Int8Array;
    unsigned handled = type_array_bits;

    if (type_array_bits & ES_Type_Array_Indexed::AnyIntArray)
        done_int_target = cg.ForwardJump();

    for(; handled; array_kind++)
    {
        ES_CodeGenerator::JumpTarget *next;

        if (!(handled & 1))
        {
            handled = handled >> 1;
            continue;
        }

        handled = handled >> 1;

        if (handled)
            next = cg.ForwardJump();
        else
            next = slow_case;

        CompareRegisterToImmediate(cg, kind, array_kind);
        cg.Jump(next, ES_NATIVE_CONDITION_NOT_EQUAL);

        switch (array_kind)
        {
        case ES_Type_Array_Indexed::Int8Array:
            ANNOTATE("EmitInt32TypedArrayGet, Int8Array");
            cg.LDRSB(bytearray, index, TRUE, SCRATCHI_IVALUE);
            cg.Jump(done_int_target, ES_NATIVE_UNCONDITIONAL);
            break;

        case ES_Type_Array_Indexed::Uint8Array:
        case ES_Type_Array_Indexed::Uint8ClampedArray:
            ANNOTATE("EmitInt32TypedArrayGet, Uint8Array");
            cg.LDRB(bytearray, index, ES_CodeGenerator::SHIFT_LOGICAL_LEFT, 0, TRUE, SCRATCHI_IVALUE);
            cg.Jump(done_int_target, ES_NATIVE_UNCONDITIONAL);
            break;

        case ES_Type_Array_Indexed::Int16Array:
            ANNOTATE("EmitInt32TypedArrayGet, Int16Array");
            cg.ADD(index, index, index);
            cg.LDRSH(bytearray, index, TRUE, SCRATCHI_IVALUE);
            cg.Jump(done_int_target, ES_NATIVE_UNCONDITIONAL);
            break;

        case ES_Type_Array_Indexed::Uint16Array:
            ANNOTATE("EmitInt32TypedArrayGet, Uint16Array");
            cg.ADD(index, index, index);
            cg.LDRH(bytearray, index, TRUE, SCRATCHI_IVALUE);
            cg.Jump(done_int_target, ES_NATIVE_UNCONDITIONAL);
            break;

        case ES_Type_Array_Indexed::Int32Array:
            ANNOTATE("EmitInt32TypedArrayGet, Int32Array");
            cg.LDR(bytearray, index, ES_CodeGenerator::SHIFT_LOGICAL_LEFT, 2, TRUE, SCRATCHI_IVALUE);
            cg.Jump(done_int_target, ES_NATIVE_UNCONDITIONAL);
            break;

        case ES_Type_Array_Indexed::Uint32Array:
            ANNOTATE("EmitInt32TypedArrayGet, Uint32Array");
            cg.LDR(bytearray, index, ES_CodeGenerator::SHIFT_LOGICAL_LEFT, 2, TRUE, SCRATCHI_IVALUE);

            cg.TST(SCRATCHI_IVALUE, 0x80000000u);
            cg.Jump(done_int_target, ES_NATIVE_CONDITION_EQUAL);

            if (ARCHITECTURE_HAS_FPU())
            {
                cg.FMSR(SCRATCHI1, SCRATCHS1);
                cg.FUITOD(SCRATCHS1, SCRATCHD1);
                cg.FSTD(SCRATCHD1, BASE_REGISTER(target_vr), DVALUE_OFFSET(target_vr));
            }
            else
            {
                AddImmediateToRegister(cg, BASE_REGISTER(target_vr), DVALUE_OFFSET(target_vr), ES_CodeGenerator::REG_R1);

                if (SCRATCHI1 != ES_CodeGenerator::REG_R0)
                    cg.MOV(SCRATCHI_IVALUE, ES_CodeGenerator::REG_R0);

                cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::StoreUIntAsDouble)), ES_CodeGenerator::REG_LR);
                cg.BLX(ES_CodeGenerator::REG_LR);
            }

            done_target = cg.ForwardJump();
            cg.Jump(done_target);

            break;

        case ES_Type_Array_Indexed::Float32Array:
            ANNOTATE("EmitInt32TypedArrayGet, Float32Array");

            cg.ADD(bytearray, ES_CodeGenerator::Operand(index, 2, ES_CodeGenerator::SHIFT_LOGICAL_LEFT), SCRATCHI1);

            if (ARCHITECTURE_HAS_FPU())
            {
                cg.FLDS(SCRATCHI1, 0, SCRATCHS1);
                cg.FCVTDS(SCRATCHS1, SCRATCHD1);
                cg.FSTD(SCRATCHD1, BASE_REGISTER(target_vr), DVALUE_OFFSET(target_vr));
            }
            else
            {
                AddImmediateToRegister(cg, BASE_REGISTER(target_vr), DVALUE_OFFSET(target_vr), ES_CodeGenerator::REG_R1);
                cg.MOV(bytearray, ES_CodeGenerator::REG_R0);
                cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::StoreFloatAsDouble)), ES_CodeGenerator::REG_LR);
                cg.BLX(ES_CodeGenerator::REG_LR);
            }

            if (handled || done_int_target)
            {
                if (!done_target)
                    done_target = cg.ForwardJump();

                cg.Jump(done_target, ES_NATIVE_UNCONDITIONAL);
            }
            else
                return;

            break;

        case ES_Type_Array_Indexed::Float64Array:
            ANNOTATE("EmitInt32TypedArrayGet, Float64Array");
            cg.ADD(bytearray, ES_CodeGenerator::Operand(index, 3, ES_CodeGenerator::SHIFT_LOGICAL_LEFT), bytearray);
            CopyValue(cg, bytearray, 0, target_vr);

            if (done_int_target)
            {
                if (!done_target)
                    done_target = cg.ForwardJump();
                cg.Jump(done_target, ES_NATIVE_UNCONDITIONAL);
            }
            else if (!done_target)
                return;
        }

        if (next != slow_case)
            cg.SetJumpTarget(next);
    }

    if (done_int_target)
    {
        cg.SetJumpTarget(done_int_target);
        cg.MOV(ESTYPE_INT32, SCRATCHI_TYPE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);
    }

    if (done_target)
        cg.SetJumpTarget(done_target);
}

static void
LoadDoubleAsInt(ES_Native &n, ES_Native::VirtualRegister *source_vr, ES_CodeGenerator::Register dst, BOOL has_fpu, BOOL do_truncate, ES_CodeGenerator::JumpTarget *slow_case)
{
    ES_CodeGenerator &cg = n.cg;

    if (has_fpu)
    {
        cg.FLDD(BASE_REGISTER(source_vr), DVALUE_OFFSET(source_vr), SCRATCHD1);
        if (do_truncate)
            cg.FTOSIZD(SCRATCHD1, SCRATCHS1);
        else
            cg.FTOSID(SCRATCHD1, SCRATCHS1);
        cg.FMRS(SCRATCHS1, dst);

        cg.CMP(dst, INT_MAX);
        cg.CMP(dst, INT_MIN, ES_NATIVE_CONDITION_NOT_EQUAL);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
    }
    else
    {
        AddImmediateToRegister(cg, BASE_REGISTER(source_vr), VALUE_OFFSET(source_vr->index), ES_CodeGenerator::REG_R0);
        if (do_truncate)
            cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::GetDoubleAsInt)), ES_CodeGenerator::REG_LR);
        else
            cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::GetDoubleAsIntRound)), ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);

        if (dst != ES_CodeGenerator::REG_R0)
            cg.MOV(ES_CodeGenerator::REG_R0, dst);
    }
}

static void
StoreDoubleAsFloat2(ES_Native &n, ES_Native::VirtualRegister *source_vr, ES_CodeGenerator::Register dst, ES_CodeGenerator::JumpTarget *slow_case, BOOL has_fpu)
{
    ES_CodeGenerator &cg = n.cg;

    if (has_fpu)
    {
        cg.FLDD(BASE_REGISTER(source_vr), DVALUE_OFFSET(source_vr), SCRATCHD1);
        cg.FCVTSD(SCRATCHD1, SCRATCHS1);
        cg.FSTS(SCRATCHS1, dst, 0);
    }
    else
    {
        AddImmediateToRegister(cg, BASE_REGISTER(source_vr), DVALUE_OFFSET(source_vr), ES_CodeGenerator::REG_R0);
        if (dst != ES_CodeGenerator::REG_R1)
            cg.MOV(dst, ES_CodeGenerator::REG_R1);

        cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::StoreDoubleAsFloat)), ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);
    }
}

static void
StoreIntAsFloat2(ES_Native &n, ES_Native::VirtualRegister *source_vr, ES_CodeGenerator::Register dst, ES_CodeGenerator::JumpTarget *slow_case, BOOL has_fpu)
{
    ES_CodeGenerator &cg = n.cg;

    if (has_fpu)
    {
        cg.LDR(BASE_REGISTER(source_vr), IVALUE_OFFSET(source_vr), SCRATCHI1);
        cg.FMSR(SCRATCHI1, SCRATCHS1);
        cg.FSITOS(SCRATCHS1, SCRATCHS1);
        cg.FSTS(SCRATCHS1, dst, 0);
    }
    else
    {
        cg.LDR(BASE_REGISTER(source_vr), IVALUE_OFFSET(source_vr), ES_CodeGenerator::REG_R0);
        if (dst != ES_CodeGenerator::REG_R1)
            cg.MOV(dst, ES_CodeGenerator::REG_R1);

        cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::StoreIntAsFloat)), ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);
    }
}

static void
StoreIntAsDouble2(ES_Native &n, ES_Native::VirtualRegister *source_vr, ES_CodeGenerator::Register dst, ES_CodeGenerator::JumpTarget *slow_case, BOOL has_fpu)
{
    ES_CodeGenerator &cg = n.cg;

    if (has_fpu)
    {
        cg.LDR(BASE_REGISTER(source_vr), IVALUE_OFFSET(source_vr), SCRATCHI1);
        cg.FMSR(SCRATCHI1, SCRATCHS1);
        cg.FSITOD(SCRATCHS1, SCRATCHD1);
        cg.FSTD(SCRATCHD1, dst, 0);
    }
    else
    {
        cg.LDR(BASE_REGISTER(source_vr), IVALUE_OFFSET(source_vr), ES_CodeGenerator::REG_R0);
        if (dst != ES_CodeGenerator::REG_R1)
            cg.MOV(dst, ES_CodeGenerator::REG_R1);

        cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::StoreIntAsDouble)), ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);
    }
}

static BOOL
EmitInt32TypedArrayPutLoadSource(ES_Native &native, ES_Native::VirtualRegister *source_vr, ES_CodeGenerator::Register value, unsigned char source_type_bits, ES_ValueType *known_type, BOOL has_fpu, BOOL do_truncate, ES_CodeGenerator::JumpTarget *slow_case)
{
    ES_CodeGenerator &cg = native.cg;
    if (known_type)
    {
        if (*known_type == ESTYPE_DOUBLE)
        {
            LoadDoubleAsInt(native, source_vr, value, has_fpu, do_truncate, slow_case);
            return TRUE;
        }
        else if (*known_type == ESTYPE_INT32)
            cg.LDR(BASE_REGISTER(source_vr), VALUE_OFFSET(source_vr->index), value);
    }
    else
    {
        if ((source_type_bits & (ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE)) == (ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE))
        {
            ES_CodeGenerator::OutOfOrderBlock *double_to_int = cg.StartOutOfOrderBlock();
            ANNOTATE("EmitInt32TypedArrayPut, convert double to int");
            LoadDoubleAsInt(native, source_vr, value, has_fpu, do_truncate, slow_case);

            cg.EndOutOfOrderBlock();

            cg.LDR(BASE_REGISTER(source_vr), TYPE_OFFSET(source_vr->index), SCRATCHI2);
            CompareRegisterToImmediate(cg, SCRATCHI2, ESTYPE_INT32);

            cg.Jump(double_to_int->GetJumpTarget(), ES_NATIVE_CONDITION_LESS);
            cg.Jump(slow_case, ES_NATIVE_CONDITION_GREATER);

            cg.LDR(BASE_REGISTER(source_vr), VALUE_OFFSET(source_vr->index), value);
            cg.SetOutOfOrderContinuationPoint(double_to_int);
            return TRUE;
        }
        else if (source_type_bits & ESTYPE_BITS_INT32)
        {
            native.EmitRegisterTypeCheck(source_vr, ESTYPE_INT32, slow_case);
            cg.LDR(BASE_REGISTER(source_vr), VALUE_OFFSET(source_vr->index), value);
        }
        else if (source_type_bits & ESTYPE_BITS_DOUBLE)
        {
            native.EmitRegisterTypeCheck(source_vr, ESTYPE_DOUBLE, slow_case);
            LoadDoubleAsInt(native, source_vr, value, has_fpu, do_truncate, slow_case);
            return TRUE;
        }
    }
    return FALSE;
}

void
ES_Native::EmitInt32TypedArrayPut(VirtualRegister *object_vr, VirtualRegister *index_vr, unsigned constant_index, unsigned type_array_bits, VirtualRegister *source_vr, unsigned char source_type_bits, ES_Value_Internal *known_value, ES_ValueType *known_type)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ANNOTATE("EmitInt32TypedArrayPut");

    OP_ASSERT(!known_type || *known_type == ESTYPE_DOUBLE || *known_type == ESTYPE_INT32);
    OP_ASSERT((source_type_bits & ~(ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE)) == 0);

    ES_CodeGenerator::Register bytearray(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register index(ES_CodeGenerator::REG_R5);
    ES_CodeGenerator::Register kind(ES_CodeGenerator::REG_R6);

    EmitTypedArrayPrecondition(*this, object_vr, index_vr, constant_index);

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Type_Array_Indexed, capacity), SCRATCHI1);
    cg.CMP(index, SCRATCHI1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Type_Array_Indexed, kind), kind);

    cg.LDR(OBJECT_MEMBER(bytearray, ES_Type_Array_Indexed, byte_offset), SCRATCHI1);
    cg.LDR(OBJECT_MEMBER(bytearray, ES_Type_Array_Indexed, byte_array), bytearray);
    cg.LDR(OBJECT_MEMBER(bytearray, ES_Byte_Array_Indexed, byte_array), bytearray);
    cg.ADD(SCRATCHI1, bytearray, bytearray);

    ES_CodeGenerator::JumpTarget *next = NULL;
    ES_CodeGenerator::JumpTarget *done_target = cg.ForwardJump();

    if (type_array_bits & ES_Type_Array_Indexed::AnyIntArray)
    {
        ES_CodeGenerator::Register value(ES_CodeGenerator::REG_R7);

        if (type_array_bits & ES_Type_Array_Indexed::AnyFloatArray)
            next = cg.ForwardJump();
        else
            next = slow_case;

        CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Uint32Array);
        cg.Jump(next, ES_NATIVE_CONDITION_GREATER);

        BOOL needed_double_conversion = FALSE;
        if (known_value)
        {
            ES_CodeGenerator::JumpTarget *done = NULL;
            BOOL loaded_known_value = FALSE;
            if (type_array_bits & (0x1 << ES_Type_Array_Indexed::Uint8ClampedArray))
            {
                ES_CodeGenerator::JumpTarget *not_clamped = NULL;
                if ((type_array_bits & ES_Type_Array_Indexed::AnyIntArray) != (0x1 << ES_Type_Array_Indexed::Uint8ClampedArray))
                {
                    not_clamped = cg.ForwardJump();
                    CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Uint8ClampedArray);
                    cg.Jump(not_clamped, ES_NATIVE_CONDITION_NOT_EQUAL);
                }
                MoveImmediateToRegister(cg, known_value->AsNumber(context).GetNumAsUint8Clamped(), value);
                if (not_clamped)
                {
                    done = cg.ForwardJump();
                    cg.Jump(done);
                    cg.SetJumpTarget(not_clamped);
                }
                else
                    loaded_known_value = TRUE;
            }
            if (!loaded_known_value)
                MoveImmediateToRegister(cg, known_value->AsNumber(context).GetNumAsUInt32(), value);
            if (done)
                cg.SetJumpTarget(done);
        }
        else
        {
            BOOL needs_truncation = (type_array_bits & ES_Type_Array_Indexed::AnyIntArray) != (0x1 << ES_Type_Array_Indexed::Uint8ClampedArray);
            needed_double_conversion = EmitInt32TypedArrayPutLoadSource(*this, source_vr, value, source_type_bits, known_type, ARCHITECTURE_HAS_FPU(), needs_truncation, slow_case);
        }

        ES_CodeGenerator::JumpTarget *next_int = NULL;

        if (type_array_bits & ES_Type_Array_Indexed::AnyInt8Array)
        {
            ANNOTATE("EmitInt32TypedArrayPut, (U)Int8Array");

            if (type_array_bits & (ES_Type_Array_Indexed::AnyInt16Array | ES_Type_Array_Indexed::AnyInt32Array))
                next_int = cg.ForwardJump();
            else
                next_int = next;

            if (type_array_bits & (0x1 << ES_Type_Array_Indexed::Uint8ClampedArray))
                CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Uint8ClampedArray);
            else
                CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Uint8Array);
            cg.Jump(next_int, ES_NATIVE_CONDITION_GREATER);

            if (!known_value && (type_array_bits & (0x1 << ES_Type_Array_Indexed::Uint8ClampedArray)))
            {
                ES_CodeGenerator::JumpTarget *write_value = cg.ForwardJump();
                cg.Jump(write_value, ES_NATIVE_CONDITION_NOT_EQUAL);

                /* If source_vr has been converted from a double _and_ it wasn't only
                   at a clamped type, repeat the load, but with different rounding. */
                if (needed_double_conversion && ((type_array_bits & ES_Type_Array_Indexed::AnyIntArray) != (0x1 << ES_Type_Array_Indexed::Uint8ClampedArray)))
                    EmitInt32TypedArrayPutLoadSource(*this, source_vr, value, source_type_bits, known_type, ARCHITECTURE_HAS_FPU(), FALSE, slow_case);
                CompareRegisterToImmediate(cg, value, 0);
                MoveImmediateToRegister(cg, 0, value, ES_NATIVE_CONDITION_LESS);
                CompareRegisterToImmediate(cg, value, 0xff);
                MoveImmediateToRegister(cg, 0xff, value, ES_NATIVE_CONDITION_GREATER);
                cg.SetJumpTarget(write_value);
            }
            cg.STRB(value, bytearray, index, ES_CodeGenerator::SHIFT_LOGICAL_LEFT, 0, TRUE);
            cg.Jump(done_target, ES_NATIVE_UNCONDITIONAL);
        }

        if (type_array_bits & ES_Type_Array_Indexed::AnyInt16Array)
        {
            ANNOTATE("EmitInt32TypedArrayPut, (U)Int16Array");

            if (next_int)
                cg.SetJumpTarget(next_int);

            if (type_array_bits & ES_Type_Array_Indexed::AnyInt32Array)
                next_int = cg.ForwardJump();
            else
                next_int = next;

            if ((type_array_bits & ES_Type_Array_Indexed::AnyInt8Array) == 0)
            {
                CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Int16Array);
                /* If TRUE, an 8-bit array. Subsequent cases cannot apply, so slow case. */
                cg.Jump(slow_case, ES_NATIVE_CONDITION_LESS);
            }

            CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Uint16Array);
            cg.Jump(next_int, ES_NATIVE_CONDITION_GREATER);

            cg.ADD(index, index, index);
            cg.STRH(value, bytearray, index, TRUE);

            cg.Jump(done_target, ES_NATIVE_UNCONDITIONAL);
        }

        if (type_array_bits & ES_Type_Array_Indexed::AnyInt32Array)
        {
            ANNOTATE("EmitInt32TypedArrayPut, (U)Int32Array");

            if (next_int)
                cg.SetJumpTarget(next_int);

            if ((type_array_bits & ES_Type_Array_Indexed::AnyInt16Array) == 0)
            {
                CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Int32Array);
                /* If TRUE, has an integral kind. Subsequent float cases cannot apply, so slow case. */
                cg.Jump(slow_case, ES_NATIVE_CONDITION_LESS);
            }

            CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Uint32Array);
            cg.Jump(next, ES_NATIVE_CONDITION_GREATER);

            cg.STR(value, bytearray, index, ES_CodeGenerator::SHIFT_LOGICAL_LEFT, 2, TRUE);
            cg.Jump(done_target, ES_NATIVE_UNCONDITIONAL);
        }
    }

    if (type_array_bits & ES_Type_Array_Indexed::Float32ArrayBit)
    {
        ANNOTATE("EmitInt32TypedArrayPut, Float32Array");

        if (next)
            cg.SetJumpTarget(next);

        if (type_array_bits & ES_Type_Array_Indexed::Float64ArrayBit)
            next = cg.ForwardJump();
        else
            next = slow_case;

        CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Float32Array);
        cg.Jump(next, ES_NATIVE_CONDITION_NOT_EQUAL);

        cg.ADD(bytearray, ES_CodeGenerator::Operand(index, 2, ES_CodeGenerator::SHIFT_LOGICAL_LEFT), bytearray);

        if (known_type)
        {
            if (*known_type == ESTYPE_DOUBLE)
                StoreDoubleAsFloat2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
            else if (*known_type == ESTYPE_INT32)
                StoreIntAsFloat2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
        }
        else
        {
            if ((source_type_bits & (ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE)) == (ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE))
            {
                ES_CodeGenerator::OutOfOrderBlock *double_to_float = cg.StartOutOfOrderBlock();
                ANNOTATE("EmitInt32TypedArrayPut, convert double to float");
                StoreDoubleAsFloat2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
                cg.EndOutOfOrderBlock();

                cg.LDR(BASE_REGISTER(source_vr), TYPE_OFFSET(source_vr->index), SCRATCHI2);
                CompareRegisterToImmediate(cg, SCRATCHI2, ESTYPE_INT32);

                cg.Jump(double_to_float->GetJumpTarget(), ES_NATIVE_CONDITION_LESS);
                cg.Jump(slow_case, ES_NATIVE_CONDITION_GREATER);

                StoreIntAsFloat2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
                cg.SetOutOfOrderContinuationPoint(double_to_float);
            }
            else if (source_type_bits & ESTYPE_BITS_INT32)
            {
                EmitRegisterTypeCheck(source_vr, ESTYPE_INT32, slow_case);
                StoreIntAsFloat2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
            }
            else if (source_type_bits & ESTYPE_BITS_DOUBLE)
            {
                EmitRegisterTypeCheck(source_vr, ESTYPE_DOUBLE, slow_case);
                StoreDoubleAsFloat2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
            }
        }

        cg.Jump(done_target, ES_NATIVE_UNCONDITIONAL);
    }

    if (type_array_bits & ES_Type_Array_Indexed::Float64ArrayBit)
    {
        ANNOTATE("EmitInt32TypedArrayPut, Float64Array");

        if (next)
            cg.SetJumpTarget(next);

        CompareRegisterToImmediate(cg, kind, ES_Type_Array_Indexed::Float64Array);

        cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

        cg.ADD(bytearray, ES_CodeGenerator::Operand(index, 3, ES_CodeGenerator::SHIFT_LOGICAL_LEFT), bytearray);

        if (known_type)
        {
            if (*known_type == ESTYPE_DOUBLE)
                CopyValue(cg, source_vr, bytearray, 0);
            else if (*known_type == ESTYPE_INT32)
                StoreIntAsDouble2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
        }
        else
        {
            if ((source_type_bits & (ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE)) == (ESTYPE_BITS_INT32 | ESTYPE_BITS_DOUBLE))
            {
                ES_CodeGenerator::OutOfOrderBlock *int_to_double = cg.StartOutOfOrderBlock();
                ANNOTATE("EmitInt32TypedArrayPut, convert int to double");
                StoreIntAsDouble2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
                cg.EndOutOfOrderBlock();

                cg.LDR(BASE_REGISTER(source_vr), TYPE_OFFSET(source_vr->index), SCRATCHI2);
                CompareRegisterToImmediate(cg, SCRATCHI2, ESTYPE_INT32);

                cg.Jump(int_to_double->GetJumpTarget(), ES_NATIVE_CONDITION_EQUAL);
                cg.Jump(slow_case, ES_NATIVE_CONDITION_GREATER);

                CopyValue(cg, source_vr, bytearray, 0);
                cg.SetOutOfOrderContinuationPoint(int_to_double);
            }
            else if (source_type_bits & ESTYPE_BITS_INT32)
            {
                EmitRegisterTypeCheck(source_vr, ESTYPE_INT32, slow_case);
                StoreIntAsDouble2(*this, source_vr, bytearray, slow_case, ARCHITECTURE_HAS_FPU());
            }
            else if (source_type_bits & ESTYPE_BITS_DOUBLE)
            {
                EmitRegisterTypeCheck(source_vr, ESTYPE_DOUBLE, slow_case);
                CopyValue(cg, source_vr, bytearray, 0);
            }
        }
    }

    cg.SetJumpTarget(done_target);
}

void
ES_Native::EmitGetEnumerated(VirtualRegister *target_vr, VirtualRegister *object_vr, VirtualRegister *name_vr)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register name(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register enumerated_class_id(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register object_class_id(ES_CodeGenerator::REG_R5);
    ES_CodeGenerator::Register type(ES_CodeGenerator::REG_R6);

    cg.LDR(BASE_REGISTER(name_vr), IVALUE_OFFSET(name_vr), name);

    LoadObjectOperand(object_vr->index, object);

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_string), SCRATCHI1);
    cg.TEQ(name, SCRATCHI1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_class_id), enumerated_class_id);

    ES_CodeGenerator::Register property_value(object_class_id);
    ES_CodeGenerator::OutOfOrderBlock *indexed = cg.StartOutOfOrderBlock();

    cg.TEQ(enumerated_class_id, ES_Class::NOT_CACHED_CLASS_ID);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, object_bits), SCRATCHI1);
    cg.TST(SCRATCHI1, ES_Object::MASK_SIMPLE_COMPACT_INDEXED);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    ES_CodeGenerator::Register index(name);
    ES_CodeGenerator::Register indexed_properties(enumerated_class_id);

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_index), index);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, indexed_properties), indexed_properties);
    cg.LDR(OBJECT_MEMBER(indexed_properties, ES_Compact_Indexed_Properties, capacity), SCRATCHI1);
    cg.CMP(index, SCRATCHI1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

    cg.ADD(indexed_properties, ES_OFFSETOF(ES_Compact_Indexed_Properties, values), property_value);
    cg.ADD(property_value, ES_CodeGenerator::Operand(index, 3), property_value);

    LoadValue(cg, property_value, 0, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

    cg.CMP(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3), ESTYPE_UNDEFINED);
    cg.CMP(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3), 0, ES_NATIVE_CONDITION_EQUAL);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    StoreValue(cg, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3, target_vr);
    cg.EndOutOfOrderBlock();

    cg.LDR(OBJECT_MEMBER(object, ES_Object, klass), object_class_id);
    cg.LDR(OBJECT_MEMBER(object_class_id, ES_Class, class_id), object_class_id);
    cg.TEQ(enumerated_class_id, object_class_id);
    cg.Jump(indexed->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);

    ES_CodeGenerator::Register enumerated_limit(enumerated_class_id);
    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_limit), enumerated_limit);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI1);
    cg.CMP(SCRATCHI1, enumerated_limit);
    cg.Jump(current_slow_case->GetJumpTarget(), ES_NATIVE_CONDITION_LESS_OR_EQUAL);

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_cached_type), type);

    ES_CodeGenerator::Register enumerated_cached_offset(enumerated_class_id);
    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_cached_offset), enumerated_cached_offset);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), property_value);
    cg.ADD(property_value, enumerated_cached_offset, property_value);
    CopyDataToValue(cg, property_value, type, target_vr);

    cg.SetOutOfOrderContinuationPoint(indexed);
}

void
ES_Native::EmitPutEnumerated(VirtualRegister *object_vr, VirtualRegister *name_vr, VirtualRegister *source_vr)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register name(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register enumerated_class_id(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register object_class_id(ES_CodeGenerator::REG_R5);

    cg.LDR(BASE_REGISTER(name_vr), IVALUE_OFFSET(name_vr), name);

    LoadObjectOperand(object_vr->index, object);

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_string), SCRATCHI1);
    cg.TEQ(name, SCRATCHI1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_class_id), enumerated_class_id);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, klass), object_class_id);
    cg.LDR(OBJECT_MEMBER(object_class_id, ES_Class, class_id), object_class_id);
    cg.TEQ(enumerated_class_id, object_class_id);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    ES_CodeGenerator::Register enumerated_limit(object_class_id);
    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_limit), enumerated_limit);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI1);
    cg.CMP(SCRATCHI1, enumerated_limit);
    cg.Jump(current_slow_case->GetJumpTarget(), ES_NATIVE_CONDITION_LESS_OR_EQUAL);

    ES_CodeGenerator::Register enumerated_cached_offset(enumerated_class_id);
    ES_CodeGenerator::Register property_value(enumerated_limit);

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_cached_offset), enumerated_cached_offset);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), property_value);
    cg.ADD(property_value, enumerated_cached_offset, property_value);

    ES_CodeGenerator::Register type(ES_CodeGenerator::REG_R6);
    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, enumerated_cached_type), type);

    CopyValueToData(cg, source_vr, property_value, type, current_slow_case->GetJumpTarget());
}

void
ES_Native::EmitInt32StringIndexedGet(VirtualRegister *target_vr, VirtualRegister *string_vr, VirtualRegister *index_vr, unsigned constant_index, BOOL returnCharCode)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register string(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register value(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register offset(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register length(ES_CodeGenerator::REG_R5);

    LoadObjectOperand(string_vr->index, string);

    cg.LDR(OBJECT_MEMBER(string, JString, value), value);

    cg.TST(value, 1);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.ADD(value, ES_OFFSETOF(JStringStorage, storage), value);

    cg.LDR(OBJECT_MEMBER(string, JString, offset), offset);
    cg.ADD(value, ES_CodeGenerator::Operand(offset, 1), value);

    cg.LDR(OBJECT_MEMBER(string, JString, length), length);

    if (index_vr || !ES_CodeGenerator::Operand::EncodeImmediate(constant_index))
    {
        if (index_vr)
            cg.LDR(BASE_REGISTER(index_vr), IVALUE_OFFSET(index_vr), offset);
        else
        {
            MoveImmediateToRegister(cg, constant_index, offset);
            constant_index = 0;
        }

        cg.CMP(offset, length);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

        cg.ADD(value, ES_CodeGenerator::Operand(offset, 1), value);
    }
    else
    {
        cg.CMP(length, constant_index);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

        if (!ES_CodeGenerator::SupportedOffset(constant_index * 2, ES_CodeGenerator::LOAD_STORE_HALF_WORD))
        {
            AddImmediateToRegister(cg, value, constant_index * 2, value);
            constant_index = 0;
        }
    }

    ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7));
    ES_CodeGenerator::Register character(IVALUE_REGISTER(ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7));

    cg.LDRH(value, constant_index * 2, character);

    if (returnCharCode)
        cg.MOV(ESTYPE_INT32, type);
    else
    {
        cg.CMP(character, STRING_NUMCHARS);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER_OR_SAME);

        cg.MOV(ESTYPE_STRING, type);

        cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, rt_data), string);
        AddImmediateToRegister(cg, string, ES_OFFSETOF(ESRT_Data, strings) + STRING_nul * sizeof(void *), string);
        cg.ADD(string, ES_CodeGenerator::Operand(character, 2), string);
        cg.LDR(string, 0, character);
    }

    StoreValue(cg, ES_CodeGenerator::REG_R6, ES_CodeGenerator::REG_R7, target_vr);
}

void
ES_Native::EmitPrimitiveNamedGet(VirtualRegister *target_vr, ES_Object *check_object, ES_Object *property_object, unsigned class_id, unsigned property_offset, ES_StorageType storage_type, ES_Object *function)
{
    DECLARE_NOTHING();
    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register object_class(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register klass_id(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register value(ES_CodeGenerator::REG_R7);

    MovePointerToRegister(cg, check_object, object);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, klass), object_class);
    cg.LDR(OBJECT_MEMBER(object_class, ES_Class, class_id), klass_id);
    EqCompareRegisterToImmediate(cg, klass_id, class_id);
    cg.Jump(current_slow_case->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);

    if (function)
    {
        MovePointerToRegister(cg, function, SCRATCHI_IVALUE);
        MoveImmediateToRegister(cg, ESTYPE_OBJECT, SCRATCHI_TYPE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);
    }
    else if (property_object)
    {
        if (check_object != property_object)
            MovePointerToRegister(cg, property_object, object);

        cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), value);
        if (property_value_write_vr == target_vr && (storage_type == ES_STORAGE_OBJECT || storage_type == ES_STORAGE_WHATEVER))
        {
            SetPropertyValueTransferRegister(INTEGER_NR(this, ES_CodeGenerator::REG_R7), property_offset, storage_type != ES_STORAGE_OBJECT);
            if (current_slow_case)
                if (ES_CodeGenerator::OutOfOrderBlock *recover_from_failure = RecoverFromFailedPropertyValueTransfer(this, target_vr, ES_CodeGenerator::REG_R7))
                    cg.SetOutOfOrderContinuationPoint(recover_from_failure);
        }
        else
            CopyTypedDataToValue(cg, value, property_offset, storage_type, target_vr);
    }
    else
        EmitSetRegisterType(target_vr, ESTYPE_UNDEFINED);
}

void
ES_Native::EmitFetchFunction(VirtualRegister *target_vr, ES_Object *function, VirtualRegister *object_vr, unsigned class_id, BOOL fetch_value)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register object_class(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register klass_id(ES_CodeGenerator::REG_R4);

    LoadObjectOperand(object_vr->index, object);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, klass), object_class);
    cg.LDR(OBJECT_MEMBER(object_class, ES_Class, class_id), klass_id);
    EqCompareRegisterToImmediate(cg, klass_id, class_id);
    cg.Jump(current_slow_case->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);

    if (fetch_value)
    {
        MovePointerToRegister(cg, function, SCRATCHI_IVALUE);
        MoveImmediateToRegister(cg, ESTYPE_OBJECT, SCRATCHI_TYPE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);
    }
}

static void
ES_EmitGetCachePositiveBranch(ES_CodeGenerator &cg, const ES_Native::GetCacheGroup &group, const ES_Native::GetCacheGroup::Entry &entry, ES_CodeGenerator::JumpTarget **copy_value, ES_CodeGenerator::Register object, ES_CodeGenerator::Register value, BOOL properties_loaded, BOOL jump_when_finished)
{
    DECLARE_NOTHING();

    OP_ASSERT(entry.positive);

    if (!properties_loaded)
        cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), value);

    if (!group.single_offset && entry.positive_offset)
        AddImmediateToRegister(cg, value, entry.positive_offset, value);

    if (jump_when_finished)
        cg.Jump(copy_value[group.storage_type]);
}

static const ES_Native::GetCacheGroup *
ES_FindGroupByStorageType(const ES_Native::GetCacheGroup *groups, ES_StorageType storage_type)
{
    while (groups->storage_type != storage_type)
        ++groups;
    return groups;
}

static void
ES_EmitGetCacheNegativeBranch(ES_CodeGenerator &cg, const ES_Native::GetCacheGroup *groups, const ES_Native::GetCacheGroup &group, const ES_Native::GetCacheGroup::Entry &entry, ES_CodeGenerator::JumpTarget **copy_value, ES_CodeGenerator::Register object, ES_CodeGenerator::Register value, BOOL jump_when_finished)
{
    DECLARE_NOTHING();

    OP_ASSERT(entry.prototype);

    const ES_Native::GetCacheGroup *actual_group;
    if (entry.prototype_storage_type == group.storage_type)
        actual_group = &group;
    else
    {
        actual_group = ES_FindGroupByStorageType(groups, entry.prototype_storage_type);
        jump_when_finished = TRUE;
    }

    MovePointerToRegister(cg, entry.prototype, object);
    cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), value);

    if (!actual_group->single_offset && entry.prototype_offset)
        AddImmediateToRegister(cg, value, entry.prototype_offset, value);

    if (jump_when_finished)
        cg.Jump(copy_value[actual_group->storage_type]);
}

void
ES_Native::EmitNamedGet(VirtualRegister *target_vr, VirtualRegister *object_vr, const ES_Native::GetCacheAnalysis &analysis, BOOL for_inlined_function_call, BOOL fetch_value)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register object_class(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register klass_id(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register property_count(ES_CodeGenerator::REG_R5);
    ES_CodeGenerator::Register value;

    const ES_Native::GetCacheGroup *groups = analysis.groups;
    unsigned groups_count = analysis.groups_count;
    const ES_Native::GetCacheNegative *negatives = analysis.negatives;
    unsigned negatives_count = analysis.negatives_count;

    /* Storage type indexed array of jump targets for jumping to the code that
       actually copies the value from a certain storage type.  Only set for
       those storage types we actually handle.  */
    ES_CodeGenerator::JumpTarget *copy_value[FIRST_SPECIAL_TYPE] = { NULL };

    BOOL load_properties = FALSE;
    unsigned positive_count = 0;

    for (unsigned group_index = 0; group_index < groups_count; ++group_index)
    {
        const GetCacheGroup &group = groups[group_index];

        if (!load_properties)
            for (unsigned entry_index = 0; !load_properties && entry_index < group.entries_count; ++entry_index)
                if (group.entries[entry_index].positive && ++positive_count > 1)
                    load_properties = TRUE;

        if (!copy_value[group.storage_type])
            copy_value[group.storage_type] = cg.ForwardJump();
    }

    if (property_value_read_vr)
        value = ES_CodeGenerator::REG_R6;
    else
        value = ES_CodeGenerator::REG_R7;

    ES_CodeGenerator::JumpTarget *jt_not_found = negatives_count ? cg.ForwardJump() : NULL;
    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();

    LoadObjectOperand(object_vr->index, object);

    property_value_nr = NULL;

    cg.LDR(OBJECT_MEMBER(object, ES_Object, klass), object_class);

    ES_CodeGenerator::JumpTarget *jt_finished = cg.ForwardJump();

    cg.LDR(OBJECT_MEMBER(object_class, ES_Class, class_id), klass_id);

    if (load_properties)
        cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), value);

    ES_CodeGenerator::OutOfOrderBlock *recover_from_failure = NULL;
    BOOL property_value_transfer = property_value_write_vr == target_vr;

    if (property_value_transfer)
    {
        SetPropertyValueTransferRegister(INTEGER_NR(this, ES_CodeGenerator::REG_R7), property_value_write_offset, FALSE);

        recover_from_failure = RecoverFromFailedPropertyValueTransfer(this, target_vr, ES_CodeGenerator::REG_R7);
    }

    /* Somewhat silly, but makes it easier to read the code below.  :-) */
    BOOL properties_loaded = load_properties;

    ES_CodeGenerator::JumpTarget *jt_next_group = NULL;

    for (unsigned group_index = 0; group_index < groups_count; ++group_index)
    {
        const GetCacheGroup &group = groups[group_index];
        BOOL last_group = group_index == groups_count - 1;
        BOOL last_group_with_entries = last_group || groups[group_index + 1].only_secondary_entries;

        if (!jt_next_group)
            jt_next_group = last_group_with_entries && !negatives_count ? slow_case : cg.ForwardJump();

        unsigned entry_index = 0;

        while (entry_index < group.entries_count)
        {
            const GetCacheGroup::Entry &entry = group.entries[entry_index];
            if (entry.prototype && entry.prototype_secondary_entry)
            {
                OP_ASSERT(!entry.positive && !entry.negative);
                ++entry_index;
            }
            else
                break;
        }

        for (unsigned index = entry_index; index < group.entries_count; ++index)
            if (group.entries[index].limit != UINT_MAX)
            {
                cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), property_count);
                break;
            }

        for (; entry_index < group.entries_count; ++entry_index)
        {
            const GetCacheGroup::Entry &entry = group.entries[entry_index];

            BOOL last_entry = entry_index == group.entries_count - 1;
            ES_CodeGenerator::JumpTarget *jt_next_entry = NULL;

            CompareRegisterToImmediate(cg, klass_id, entry.class_id);

            BOOL need_positive_cache_hit_code = FALSE;
            BOOL need_negative_cache_hit_code = FALSE;
            BOOL need_limit_check = FALSE;

            if (entry.positive)
            {
                if (!properties_loaded)
                    /* Need to load properties pointer. */
                    need_positive_cache_hit_code = TRUE;

                if (!group.single_offset && entry.positive_offset)
                    /* Need to add offset to 'properties'. */
                    need_positive_cache_hit_code = TRUE;
            }

            if (entry.limit != UINT_MAX)
                /* Need limit check. */
                need_limit_check = TRUE;

            if (entry.prototype)
            {
                /* All secondary entries should precede any regular entries, so
                   the loop above should have skipped them all. */
                OP_ASSERT(!entry.prototype_secondary_entry);

                /* Need to fetch properties pointer from prototype object. */
                need_negative_cache_hit_code = TRUE;
            }

            if (!need_positive_cache_hit_code && !need_negative_cache_hit_code && !need_limit_check)
                /* The class ID check is all we need. */
                if (last_entry)
                    /* This is the last entry; jump to next group, or slow-case,
                       if class ID check failed, and fall through to the value
                       copying code if it succeeded. */
                    cg.Jump(jt_next_group, ES_NATIVE_CONDITION_NOT_EQUAL);
                else
                    /* Otherwise jump to the value copying code if the class ID
                       check succeeded, and fall through to the next entry if it
                       failed. */
                    cg.Jump(copy_value[group.storage_type], ES_NATIVE_CONDITION_EQUAL);
            else
            {
                /* We need to execute one or more instructions after a
                   successful class ID check, so first jump to the next entry,
                   or next group (or slow-case,) if it failed. */
                if (last_entry)
                    cg.Jump(jt_next_group, ES_NATIVE_CONDITION_NOT_EQUAL);
                else
                    cg.Jump(jt_next_entry = cg.ForwardJump(), ES_NATIVE_CONDITION_NOT_EQUAL);

                if (need_limit_check)
                {
                    /* Perform limit check. */
                    CompareRegisterToImmediate(cg, property_count, entry.limit);

                    if (!need_negative_cache_hit_code)
                    {
                        /* This cache either has no negative side at all, or
                           doesn't need to execute any instructions if the limit
                           check fails. */
                        ES_CodeGenerator::JumpTarget *jt_negative;
                        if (entry.negative)
                        {
                            /* Failed limit check => property doesn't exist. */
                            if (!jt_not_found)
                                jt_not_found = cg.ForwardJump();
                            jt_negative = jt_not_found;
                        }
                        else
                            /* Failed limit check => cache entry not valid, and
                               since we've done a class ID check, we know no
                               other cache entry is valid either. */
                            jt_negative = slow_case;

                        if (!need_positive_cache_hit_code && !last_entry)
                        {
                            cg.Jump(copy_value[group.storage_type], ES_NATIVE_CONDITION_HIGHER);
                            cg.Jump(jt_negative);
                        }
                        else
                        {
                            cg.Jump(jt_negative, ES_NATIVE_CONDITION_LOWER_OR_SAME);
                            ES_EmitGetCachePositiveBranch(cg, group, entry, copy_value, object, value, properties_loaded, !last_entry);
                        }
                    }
                    else if (!need_positive_cache_hit_code)
                    {
                        /* This cache either has no positive side at all, or
                           doesn't need to execute any additional instructions
                           if the limit check succeeded, but do if the limit
                           check failed. */
                        ES_CodeGenerator::JumpTarget *jt_positive;
                        if (entry.positive)
                            /* Successful limit check => copy value (at single
                               offset, or zero offset.) */
                            jt_positive = copy_value[group.storage_type];
                        else
                            /* "Successful" limit check => cache entry not
                               valid, and since we've done a class ID check, we
                               know no other cache entry is valid either. */
                            jt_positive = slow_case;

                        cg.Jump(jt_positive, ES_NATIVE_CONDITION_HIGHER);

                        ES_EmitGetCacheNegativeBranch(cg, groups, group, entry, copy_value, object, value, !last_entry);
                    }
                    else
                    {
                        /* We need to execute some additional instruction both
                           on successful and failed limit check.  Generate jump
                           that skips the "negative branch" if the limit check
                           fails. */
                        ES_CodeGenerator::JumpTarget *jt_positive = cg.ForwardJump();
                        cg.Jump(jt_positive, ES_NATIVE_CONDITION_HIGHER);

                        ES_EmitGetCacheNegativeBranch(cg, groups, group, entry, copy_value, object, value, TRUE);

                        cg.SetJumpTarget(jt_positive);

                        ES_EmitGetCachePositiveBranch(cg, group, entry, copy_value, object, value, properties_loaded, !last_entry);
                    }
                }
                else
                {
                    /* No limit check needed, but some additional instructions
                       need to be executed. */

                    /* Since we'll have no limit check, we must have exactly one
                       branch of additional instructions. */
                    OP_ASSERT(need_positive_cache_hit_code != need_negative_cache_hit_code);

                    if (need_positive_cache_hit_code)
                        ES_EmitGetCachePositiveBranch(cg, group, entry, copy_value, object, value, properties_loaded, !last_entry);
                    else
                        ES_EmitGetCacheNegativeBranch(cg, groups, group, entry, copy_value, object, value, !last_entry);
                }

                if (jt_next_entry)
                    cg.SetJumpTarget(jt_next_entry);
            }
        }

        unsigned offset_immediate = group.single_offset ? group.common_offset : 0;

        cg.SetJumpTarget(copy_value[group.storage_type]);

        if (property_value_transfer)
        {
            if (group.storage_type == ES_STORAGE_WHATEVER)
            {
                LoadOffset(cg, value, offset_immediate + VALUE_TYPE_OFFSET, SCRATCHI2);
                EqCompareRegisterToImmediate(cg, SCRATCHI2, ESTYPE_OBJECT);
                cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
            }
            else if (group.storage_type == ES_STORAGE_OBJECT_OR_NULL)
            {
                LoadOffset(cg, value, offset_immediate, SCRATCHI2);
                EqCompareRegisterToImmediate(cg, SCRATCHI2, 0);
                cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);
            }

            if (offset_immediate != property_value_offset)
                AddImmediateToRegister(cg, value, offset_immediate, ES_CodeGenerator::REG_R7);
            else if (value != ES_CodeGenerator::REG_R7)
                cg.MOV(value, ES_CodeGenerator::REG_R7);
        }
        else if (fetch_value)
            CopyTypedDataToValue(cg, value, offset_immediate, group.storage_type, target_vr);

        if (!last_group || jt_not_found)
            cg.Jump(jt_finished);

        if (!last_group_with_entries)
        {
            cg.SetJumpTarget(jt_next_group);
            jt_next_group = NULL;
        }
    }

    if (negatives_count)
    {
        if (jt_next_group)
            cg.SetJumpTarget(jt_next_group);

        for (unsigned index = 0; index < negatives_count; ++index)
            if (negatives[index].limit != UINT_MAX)
            {
                cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), property_count);
                break;
            }

        for (unsigned index = 0; index < negatives_count; ++index)
        {
            BOOL last_missing = index == negatives_count - 1;

            CompareRegisterToImmediate(cg, klass_id, negatives[index].class_id);
            if (negatives[index].limit == UINT_MAX)
                if (last_missing)
                    cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
                else
                    cg.Jump(jt_not_found, ES_NATIVE_CONDITION_EQUAL);
            else
            {
                ES_CodeGenerator::JumpTarget *jt_next = NULL;

                if (last_missing)
                    jt_next = slow_case;
                else
                    jt_next = cg.ForwardJump();

                cg.Jump(jt_next, ES_NATIVE_CONDITION_NOT_EQUAL);

                CompareRegisterToImmediate(cg, property_count, negatives[index].limit);

                if (last_missing)
                    cg.Jump(slow_case, ES_NATIVE_CONDITION_HIGHER);
                else
                {
                    cg.Jump(jt_not_found, ES_NATIVE_CONDITION_LOWER_OR_SAME);
                    cg.Jump(slow_case);

                    cg.SetJumpTarget(jt_next);
                }
            }
        }
    }

    if (jt_not_found)
    {
        cg.SetJumpTarget(jt_not_found);
        EmitSetRegisterType(target_vr, ESTYPE_UNDEFINED);
    }

    cg.SetJumpTarget(jt_finished);

    if (!property_value_nr)
        property_value_write_vr = NULL;

    if (recover_from_failure)
        cg.SetOutOfOrderContinuationPoint(recover_from_failure);
}

static void
ES_EmitPropertyArraySizeCheck(ES_CodeGenerator &cg, ES_CodeGenerator::Register properties, unsigned size, ES_CodeGenerator::Register scratch, ES_CodeGenerator::JumpTarget *failure)
{
    ES_DECLARE_NOTHING();

    /* Offset from 'properties' to the size part of ES_Header::header. */
    int objectsize_offset = -static_cast<int>(sizeof(ES_Header)) + 2;

    cg.LDRH(properties, objectsize_offset, scratch);

    if (size >= LARGE_OBJECT_LIMIT)
    {
        /* If the object's size is not 0xffff, then it's not allocated as a
           large object, meaning its size is smaller than LARGE_OBJECT_LIMIT and
           thus smaller than 'size'. */
        CompareRegisterToImmediate(cg, scratch, 0xffff);
        cg.Jump(failure, ES_NATIVE_CONDITION_NOT_EQUAL);

        /* Offset from 'properties' to ES_PageHeader::limit. */
        int pagelimit_offset = -static_cast<int>(sizeof(ES_Header)) - ES_PageHeader::HeaderSize() + ES_OFFSETOF(ES_PageHeader, limit);

        /* The object must be allocated as a large object, so calculate the
           page's actual size and use that as the size of the object. */
        cg.LDR(properties, pagelimit_offset, scratch);
        cg.SUB(scratch, properties, scratch);
        cg.ADD(scratch, static_cast<int>(sizeof(ES_Header)), scratch);
    }

    /* Check that the object's size is at least 'size'. */
    CompareRegisterToImmediate(cg, scratch, size);
    cg.Jump(failure, ES_NATIVE_CONDITION_LESS);
}

void
ES_Native::EmitNamedPut(VirtualRegister *object_vr, VirtualRegister *source_vr, JString *name, BOOL has_complex_cache, ES_StorageType source_type)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register object_class(ES_CodeGenerator::REG_R3);
    ES_CodeGenerator::Register class_id(ES_CodeGenerator::REG_R4);
    ES_CodeGenerator::Register properties(ES_CodeGenerator::REG_R5);
    ES_CodeGenerator::Register source(ES_CodeGenerator::REG_R6);

    unsigned constant_cached_offset = 0;
    LoadObjectOperand(object_vr->index, object);

    ES_Code::PropertyCache *cache = &code->property_put_caches[CurrentCodeWord()[4].index];

    OP_ASSERT(cache->class_id != ES_Class::NOT_CACHED_CLASS_ID);

#ifdef SUPPORT_INLINE_ALLOCATION
    if (constructor_final_class && object_vr->index == 0)
    {
        CopyValue(cg, source_vr, properties, cache->cached_index, object, object_class);
        return;
    }
#endif // SUPPORT_INLINE_ALLOCATION

    if (!current_slow_case)
        EmitSlowCaseCall();

    cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), properties);

    ES_CodeGenerator::JumpTarget *slow_case;

    if (!is_light_weight && property_value_read_vr && property_value_nr)
    {
        ES_CodeGenerator::OutOfOrderBlock *flush_object_vr = cg.StartOutOfOrderBlock();

        cg.MOV(object, SCRATCHI_IVALUE);
        cg.MOV(ESTYPE_OBJECT, SCRATCHI_TYPE);
        StoreValue(cg, SCRATCHI1, SCRATCHI2, object_vr);

        cg.Jump(current_slow_case_main);
        cg.EndOutOfOrderBlock(FALSE);

        slow_case = flush_object_vr->GetJumpTarget();
    }
    else
        slow_case = current_slow_case->GetJumpTarget();

    cg.LDR(OBJECT_MEMBER(object, ES_Object, klass), object_class);
    cg.LDR(OBJECT_MEMBER(object_class, ES_Class, class_id), class_id);

    while (!ES_Code::IsSimplePropertyCache(cache, FALSE))
        cache = cache->next;

    ES_CodeGenerator::JumpTarget *size_4_target = NULL, *size_8_target = NULL, *null_target = NULL, *int_to_double_target = NULL;

    AddImmediateToRegister(cg, BASE_REGISTER(source_vr), VALUE_OFFSET(source_vr->index), source);

    if (has_complex_cache)
    {
        unsigned cache_size = 0;
        ES_CodeGenerator::JumpTarget *new_class_slow_case = NULL;

        while (TRUE)
        {
            ES_Code::PropertyCache *next_cache = cache_size++ < MAX_PROPERTY_CACHE_SIZE ? cache->next : NULL;

            while (next_cache && (!ES_Code::IsSimplePropertyCache(next_cache, FALSE) || cache->class_id == next_cache->class_id))
                next_cache = next_cache->next;

            unsigned current_id = cache->class_id;

            ES_CodeGenerator::JumpTarget *jt_next_unpaired = next_cache ? cg.ForwardJump() : slow_case;

            EqCompareRegisterToImmediate(cg, class_id, current_id);
            cg.Jump(jt_next_unpaired, ES_NATIVE_CONDITION_NOT_EQUAL);

            ES_CodeGenerator::JumpTarget *jt_next = NULL;

            while (cache->class_id == current_id)
            {
                ES_StorageType type = cache->GetStorageType();
                ES_CodeGenerator::JumpTarget *current_slow_case = slow_case;
                jt_next = cache->next != next_cache && next_cache ? cg.ForwardJump() : NULL;

                BOOL needs_limit_check = cache->object_class->NeedLimitCheck(cache->GetLimit(), cache->data.new_class != NULL);
                unsigned size = ES_Layout_Info::SizeOf(type);

                if (needs_limit_check)
                {
                    cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI2);
                    CompareRegisterToImmediate(cg, SCRATCHI2, cache->GetLimit());

                    /* The cache limit checks are as follows:

                    In both cases the stored cache limit is the index that we wrote the property to originally.

                    For the case where cache->data.new_class is not NULL, we are appending a new property
                    which means that the stored index must at this point be equal to the property count, and
                    if it isn't the cache is invalid.

                    If cache->data.new_class is NULL we are writing into an object, and thus for the cache to
                    be valid, the cache limit must be less than the property count, i.e. if the property count
                    is less than or equal to the cache limit the cache is invalid.
                    */

                    if (cache->data.new_class)
                        cg.Jump(jt_next ? jt_next : slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
                    else
                        cg.Jump(jt_next ? jt_next : slow_case, ES_NATIVE_CONDITION_LESS_OR_EQUAL);
                }

                if (cache->data.new_class)
                {
                    OP_ASSERT(!cache->data.new_class->HasHashTableProperties());

                    ES_EmitPropertyArraySizeCheck(cg, properties, cache->GetOffset() + size + sizeof(ES_Header), SCRATCHI2, slow_case);

                    if (cache->data.new_class != cache->object_class)
                    {
                        MovePointerToRegister(cg, cache->data.new_class, SCRATCHI1);
                        cg.STR(SCRATCHI1, OBJECT_MEMBER(object, ES_Object, klass));
                    }
                    cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI1);
                    cg.ADD(SCRATCHI1, 1, SCRATCHI1);
                    cg.STR(SCRATCHI1, OBJECT_MEMBER(object, ES_Object, property_count));

                    if (!new_class_slow_case)
                    {
                        ES_CodeGenerator::OutOfOrderBlock *revert_put_cached_new = cg.StartOutOfOrderBlock();
                        cg.STR(object_class, OBJECT_MEMBER(object, ES_Object, klass));
                        cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI1);
                        cg.SUB(SCRATCHI1, 1, SCRATCHI1);
                        cg.STR(SCRATCHI1, OBJECT_MEMBER(object, ES_Object, property_count));
                        cg.Jump(slow_case);
                        cg.EndOutOfOrderBlock(FALSE);
                        new_class_slow_case = revert_put_cached_new->GetJumpTarget();
                    }

                    current_slow_case = new_class_slow_case;

                }
                if (cache->GetOffset() != 0)
                    AddImmediateToRegister(cg, properties, cache->GetOffset(), properties);

                BOOL skip_write = FALSE;
                if (type != ES_STORAGE_WHATEVER && source_type != type)
                    skip_write = EmitTypeConversionCheck(type, source_type, source_vr, current_slow_case, null_target, int_to_double_target);

                if (type == ES_STORAGE_NULL)
                {
                    if (!null_target)
                        null_target = cg.ForwardJump();

                    cg.Jump(null_target);
                }
                else if (!skip_write)
                    JumpToSize(cg, size, size_4_target, size_8_target);

                if (jt_next)
                    cg.SetJumpTarget(jt_next);

                if (cache->next)
                    cache = cache->next;
                else
                {
                    OP_ASSERT(next_cache == NULL);
                    break;
                }
            }

            if (next_cache)
            {
                cg.SetJumpTarget(jt_next_unpaired);
                cache = next_cache;
            }
            else
                break;
        }

        if (size_4_target || size_8_target)
            CopyValueToData2(cg, source, properties, size_4_target, size_8_target);
    }
    else
    {
        EqCompareRegisterToImmediate(cg, class_id, cache->class_id);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

        constant_cached_offset = cache->GetOffset();
        ES_StorageType constant_cached_type = cache->GetStorageType();

        if (cache->data.new_class)
        {
            OP_ASSERT(!cache->data.new_class->HasHashTableProperties());
            if (cache->object_class->NeedLimitCheck(cache->GetLimit(), TRUE))
            {
                cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI2);
                EqCompareRegisterToImmediate(cg, SCRATCHI2, cache->GetLimit());
                cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
            }

            ES_EmitPropertyArraySizeCheck(cg, properties, constant_cached_offset + ES_Layout_Info::SizeOf(constant_cached_type) + sizeof(ES_Header), SCRATCHI2, slow_case);

            if (cache->data.new_class != cache->object_class)
            {
                MovePointerToRegister(cg, cache->data.new_class, SCRATCHI1);
                cg.STR(SCRATCHI1, OBJECT_MEMBER(object, ES_Object, klass));
            }
            cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI1);
            cg.ADD(SCRATCHI1, 1, SCRATCHI1);
            cg.STR(SCRATCHI1, OBJECT_MEMBER(object, ES_Object, property_count));

            ES_CodeGenerator::OutOfOrderBlock *revert_put_cached_new = cg.StartOutOfOrderBlock();
            if (cache->data.new_class != cache->object_class)
                cg.STR(object_class, OBJECT_MEMBER(object, ES_Object, klass));
            cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI1);
            cg.SUB(SCRATCHI1, 1, SCRATCHI1);
            cg.STR(SCRATCHI1, OBJECT_MEMBER(object, ES_Object, property_count));
            cg.Jump(slow_case);
            cg.EndOutOfOrderBlock(FALSE);
            slow_case = revert_put_cached_new->GetJumpTarget();
        }
        else if (cache->object_class->NeedLimitCheck(cache->GetLimit(), FALSE))
        {
            cg.LDR(OBJECT_MEMBER(object, ES_Object, property_count), SCRATCHI2);
            CompareRegisterToImmediate(cg, SCRATCHI2, cache->GetLimit());
            cg.Jump(slow_case, ES_NATIVE_CONDITION_LESS_OR_EQUAL);
        }

        BOOL skip_write = FALSE;

        if (constant_cached_type != ES_STORAGE_WHATEVER && source_type != constant_cached_type)
            skip_write = EmitTypeConversionCheck(constant_cached_type, source_type, source_vr, slow_case, null_target, int_to_double_target);

        if (!skip_write)
            CopyValueToTypedData(cg, source_vr, properties, constant_cached_offset, constant_cached_type, NULL, object, object_class);
    }

    EmitTypeConversionHandlers(source_vr, properties, constant_cached_offset, null_target, int_to_double_target);
}

void
ES_Native::EmitLengthGet(VirtualRegister *target_vr, VirtualRegister *object_vr, unsigned handled, unsigned possible, ES_CodeGenerator::JumpTarget *slow_case)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::OutOfOrderBlock *handle_object = NULL;

    if (handled & ESTYPE_BITS_OBJECT)
    {
        if (!slow_case)
        {
            if (!current_slow_case)
                EmitSlowCaseCall();
            slow_case = current_slow_case->GetJumpTarget();
        }

        if (handled & ESTYPE_BITS_STRING)
            handle_object = cg.StartOutOfOrderBlock();

        if (handled != possible)
            EmitRegisterTypeCheck(object_vr, ESTYPE_OBJECT, slow_case);

        ES_CodeGenerator::Register array(ES_CodeGenerator::REG_R2);
        ES_CodeGenerator::Register klass(ES_CodeGenerator::REG_R3);
        ES_CodeGenerator::Register properties(ES_CodeGenerator::REG_R4);
        ES_CodeGenerator::Register class_id(ES_CodeGenerator::REG_R5);

        LoadObjectOperand(object_vr->index, array);

        cg.LDR(OBJECT_MEMBER(array, ES_Object, klass), klass);
        cg.LDR(OBJECT_MEMBER(klass, ES_Class, class_id), class_id);
        EqCompareRegisterToImmediate(cg, class_id, code->global_object->GetArrayClass()->GetId(context));
        cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);
        cg.LDR(OBJECT_MEMBER(array, ES_Object, properties), properties);

        CopyTypedDataToValue(cg, properties, 0, ES_STORAGE_INT32, target_vr);

        if (!handle_object)
            return;

        cg.EndOutOfOrderBlock();
        slow_case = handle_object->GetJumpTarget();
    }

    if (possible != ESTYPE_BITS_STRING)
    {
        if (!slow_case)
        {
            if (!current_slow_case)
                EmitSlowCaseCall();
            slow_case = current_slow_case->GetJumpTarget();
        }

        EmitRegisterTypeCheck(object_vr, ESTYPE_STRING, slow_case);
    }

    ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_CodeGenerator::Register value(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));

    LoadObjectOperand(object_vr->index, value);

    cg.MOV(ESTYPE_INT32, type);
    cg.LDR(OBJECT_MEMBER(value, JString, length), value);
    StoreValue(cg, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3, target_vr);

    if (handle_object)
        cg.SetOutOfOrderContinuationPoint(handle_object);
}

void
ES_Native::EmitFetchPropertyValue(VirtualRegister *target_vr, VirtualRegister *object_vr, ES_Object *static_object, ES_Class *klass, unsigned property_index)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::Register object(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register properties(VALUE_TRANSFER_POINTER);

    if (object_vr)
        cg.LDR(BASE_REGISTER(object_vr), IVALUE_OFFSET(object_vr), object);
    else
        MovePointerToRegister(cg, static_object, object);

    cg.LDR(OBJECT_MEMBER(object, ES_Object, properties), properties);

    ES_Layout_Info layout = klass->GetLayoutInfoAtInfoIndex(ES_PropertyIndex(property_index));

    ES_StorageType storage_type = layout.GetStorageType();
    if (property_value_write_vr && (storage_type == ES_STORAGE_OBJECT || storage_type == ES_STORAGE_WHATEVER))
    {
        OP_ASSERT(property_value_write_vr == target_vr);

        SetPropertyValueTransferRegister(INTEGER_NR(this, VALUE_TRANSFER_POINTER), layout.GetOffset(), storage_type != ES_STORAGE_OBJECT);
    }
    else
        CopyTypedDataToValue(cg, properties, layout.GetOffset(), storage_type, target_vr);
}

void
ES_Native::EmitGlobalGet(VirtualRegister *target_vr, unsigned class_id, unsigned cached_offset, unsigned cached_type)
{
    if (class_id != ES_Class::NOT_CACHED_CLASS_ID)
    {
        DECLARE_NOTHING();

        if (!current_slow_case)
            EmitSlowCaseCall();

        ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();

        if (!c_global_object) c_global_object = cg.NewConstant(code->global_object);

        ES_CodeGenerator::Register global_object(ES_CodeGenerator::REG_R2);
        ES_StorageType type = ES_Value_Internal::StorageTypeFromCachedTypeBits(cached_type);

        cg.LDR(c_global_object, global_object);

        ES_CodeGenerator::Register klass(ES_CodeGenerator::REG_R3), id(klass);
        cg.LDR(OBJECT_MEMBER(global_object, ES_Object, klass), klass);
        cg.LDR(OBJECT_MEMBER(klass, ES_Class, class_id), id);
        EqCompareRegisterToImmediate(cg, id, class_id);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

        ES_CodeGenerator::Register properties(ES_CodeGenerator::REG_R7);
        cg.LDR(OBJECT_MEMBER(global_object, ES_Object, properties), properties);

        if (property_value_write_vr && (type == ES_STORAGE_OBJECT || type == ES_STORAGE_WHATEVER))
        {
            OP_ASSERT(property_value_write_vr == target_vr);

            SetPropertyValueTransferRegister(INTEGER_NR(this, ES_CodeGenerator::REG_R7), cached_offset, type != ES_STORAGE_OBJECT);

            if (current_slow_case)
            {
                ES_CodeGenerator::OutOfOrderBlock *recover_from_failure = RecoverFromFailedPropertyValueTransfer(this, target_vr, ES_CodeGenerator::REG_R7);
                if (recover_from_failure)
                    cg.SetOutOfOrderContinuationPoint(recover_from_failure);
            }
        }
        else
            CopyTypedDataToValue(cg, properties, cached_offset, type, target_vr);
    }
    else
        EmitInstructionHandlerCall();

    if (!property_value_nr)
        property_value_write_vr = NULL;
}

void
ES_Native::EmitGlobalPut(VirtualRegister *source_vr, unsigned class_id, unsigned cached_offset, unsigned cached_type, ES_StorageType source_type)
{
    if (class_id != ES_Class::NOT_CACHED_CLASS_ID)
    {
        DECLARE_NOTHING();

        if (!current_slow_case)
            EmitSlowCaseCall();

        ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();

        if (!c_global_object) c_global_object = cg.NewConstant(code->global_object);

        ES_CodeGenerator::Register global_object(ES_CodeGenerator::REG_R2);

        cg.LDR(c_global_object, global_object);

        ES_CodeGenerator::Register klass(ES_CodeGenerator::REG_R3), id(klass);
        cg.LDR(OBJECT_MEMBER(global_object, ES_Object, klass), klass);
        cg.LDR(OBJECT_MEMBER(klass, ES_Class, class_id), id);
        EqCompareRegisterToImmediate(cg, id, class_id);
        cg.Jump(slow_case, ES_NATIVE_CONDITION_NOT_EQUAL);

        ES_CodeGenerator::Register properties(ES_CodeGenerator::REG_R3);
        cg.LDR(OBJECT_MEMBER(global_object, ES_Object, properties), properties);

        ES_CodeGenerator::JumpTarget *null_target = NULL, *int_to_double_target = NULL;
        ES_StorageType type = ES_Value_Internal::StorageTypeFromCachedTypeBits(cached_type);
        BOOL skip_write = FALSE;

        if (type != ES_STORAGE_WHATEVER && source_type != type)
            skip_write = EmitTypeConversionCheck(type, source_type, source_vr, slow_case, null_target, int_to_double_target);

        if (!skip_write)
            CopyValueToTypedData(cg, source_vr, properties, cached_offset, type, slow_case, ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5);

        EmitTypeConversionHandlers(source_vr, properties, cached_offset, null_target, int_to_double_target);
    }
    else
        EmitInstructionHandlerCall();
}

void
ES_Native::EmitGlobalImmediateGet(VirtualRegister *target_vr, unsigned index, BOOL for_inlined_function_call, BOOL fetch_value)
{
    DECLARE_NOTHING();

    if (for_inlined_function_call)
    {
        if (!c_code) c_code = cg.NewConstant(code);

        ES_CodeGenerator::Register code_pointer(ES_CodeGenerator::REG_R2);

        cg.LDR(c_code, code_pointer);

        ES_CodeGenerator::OutOfOrderBlock *failed_inlined_function = cg.StartOutOfOrderBlock();

        cg.MOV(1, SCRATCHI1);
        cg.STR(SCRATCHI1, OBJECT_MEMBER(code_pointer, ES_Code, has_failed_inlined_function));

        EmitInstructionHandlerCall();

        cg.EndOutOfOrderBlock(FALSE);

        cg.LDR(OBJECT_MEMBER(code_pointer, ES_Code, global_object), SCRATCHI2);
        cg.LDR(OBJECT_MEMBER(SCRATCHI2, ES_Global_Object, immediate_function_serial_nr), SCRATCHI2);

        EqCompareRegisterToImmediate(cg, SCRATCHI2, code->global_object->immediate_function_serial_nr);
        cg.Jump(failed_inlined_function->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);
    }

    if (fetch_value)
    {
        if (!c_global_object) c_global_object = cg.NewConstant(code->global_object);

        cg.LDR(c_global_object, SCRATCHI1);
        cg.LDR(OBJECT_MEMBER(SCRATCHI1, ES_Global_Object, variable_values), ES_CodeGenerator::REG_R2);
        CopyValue(cg, ES_CodeGenerator::REG_R2, VALUE_OFFSET(index), target_vr);
    }
}

void
ES_Native::EmitGlobalImmediatePut(unsigned index, VirtualRegister *source_vr)
{
    DECLARE_NOTHING();

    if (!c_global_object) c_global_object = cg.NewConstant(code->global_object);

    cg.LDR(c_global_object, SCRATCHI1);
    cg.LDR(OBJECT_MEMBER(SCRATCHI1, ES_Global_Object, variable_values), ES_CodeGenerator::REG_R2);

    CopyValue(cg, source_vr, ES_CodeGenerator::REG_R2, VALUE_OFFSET(index));
}

void
ES_Native::EmitLexicalGet(VirtualRegister *target_vr, VirtualRegister *function_vr, unsigned scope_index, unsigned variable_index)
{
    DECLARE_NOTHING();

    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::JumpTarget *slow_case = current_slow_case->GetJumpTarget();
    ES_CodeGenerator::Register function(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register variables(ES_CodeGenerator::REG_R3);

    cg.LDR(BASE_REGISTER(function_vr), IVALUE_OFFSET(function_vr), function);

    unsigned offset = ES_OFFSETOF(ES_Function, scope_chain) + scope_index * sizeof(ES_Object *);

    if (ES_CodeGenerator::SupportedOffset(offset, ES_CodeGenerator::LOAD_STORE_WORD))
        cg.LDR(function, offset, variables);
    else
    {
        AddImmediateToRegister(cg, function, offset, variables);
        cg.LDR(variables, 0, variables);
    }

    cg.LDR(OBJECT_MEMBER(variables, ES_Object, properties), variables);
    AddImmediateToRegister(cg, variables, 4, variables);

    LoadValue(cg, variables, VALUE_OFFSET(variable_index), ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5);

    cg.CMP(TYPE_REGISTER(ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5), ESTYPE_BOXED);
    cg.Jump(slow_case, ES_NATIVE_CONDITION_EQUAL);

    StoreValue(cg, ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5, target_vr);
}

void
ES_Native::EmitFormatString(VirtualRegister *target_vr, VirtualRegister *source_vr, unsigned cache_index)
{
    if (!current_slow_case)
        EmitSlowCaseCall();

    ES_CodeGenerator::Register string(ES_CodeGenerator::REG_R2);
    ES_CodeGenerator::Register from(ES_CodeGenerator::REG_R3);

    cg.LDR(BASE_REGISTER(source_vr), VALUE_OFFSET(source_vr), string);
    cg.LDR(cg.NewConstant(&code->format_string_caches[cache_index].from), from);

    cg.TEQ(string, from);
    cg.Jump(current_slow_case->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.MOV(ESTYPE_STRING, SCRATCHI_TYPE);
    cg.LDR(cg.NewConstant(&code->format_string_caches[cache_index].to), SCRATCHI_IVALUE);

    StoreValue(cg, SCRATCHI1, SCRATCHI2, target_vr);
}

void
ES_Native::EmitTableSwitch(VirtualRegister *value_vr, int minimum, int maximum, ES_CodeGenerator::Constant *table, ES_CodeGenerator::JumpTarget *default_target)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::Register type(TYPE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_CodeGenerator::Register value(IVALUE_REGISTER(ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3));
    ES_ValueType known_type;

    if (!GetType(value_vr, known_type))
        known_type = ESTYPE_UNDEFINED;
    else if (known_type != ESTYPE_INT32 && known_type != ESTYPE_DOUBLE && known_type != ESTYPE_INT32_OR_DOUBLE)
    {
        cg.Jump(default_target);
        return;
    }

    if (known_type != ESTYPE_INT32)
    {
        ES_CodeGenerator::OutOfOrderBlock *convert_double = NULL;
        if (known_type != ESTYPE_DOUBLE)
            convert_double = cg.StartOutOfOrderBlock();

        cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
        cg.ADD(REGISTER_FRAME_POINTER, VALUE_OFFSET(value_vr), ES_CodeGenerator::REG_R1);
        cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::GetDoubleAsInt)), ES_CodeGenerator::REG_LR);
        cg.BLX(ES_CodeGenerator::REG_LR);
        cg.MOV(ES_CodeGenerator::REG_R0, value);

        cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, implicit_bool), SCRATCHI1);
        cg.TEQ(SCRATCHI1, 0);
        cg.Jump(default_target, ES_NATIVE_CONDITION_NOT_EQUAL);

        if (convert_double)
        {
            cg.EndOutOfOrderBlock();

            LoadValue(cg, value_vr, ES_CodeGenerator::REG_R2, ES_CodeGenerator::REG_R3);

            cg.CMP(type, ESTYPE_DOUBLE);
            cg.Jump(convert_double->GetJumpTarget(), ES_NATIVE_CONDITION_LESS_OR_EQUAL);

            if (known_type != ESTYPE_INT32_OR_DOUBLE)
            {
                cg.CMP(type, ESTYPE_INT32);
                cg.Jump(default_target, ES_NATIVE_CONDITION_NOT_EQUAL);
            }

            cg.SetOutOfOrderContinuationPoint(convert_double);
        }
    }
    else
        cg.LDR(BASE_REGISTER(value_vr), IVALUE_OFFSET(value_vr), value);

    if (minimum != 0)
        AddImmediateToRegister(cg, -minimum, value);

    CompareRegisterToImmediate(cg, value, maximum - minimum);
    cg.Jump(default_target, ES_NATIVE_CONDITION_HIGHER);

    ES_CodeGenerator::Register jump_table(ES_CodeGenerator::REG_R3);

    cg.LDR(cg.AddressOf(table), jump_table);
    cg.LDR(jump_table, value, ES_CodeGenerator::SHIFT_LOGICAL_LEFT, 2, TRUE, ES_CodeGenerator::REG_PC);
}

ES_CodeGenerator::JumpTarget *
ES_Native::EmitPreConditionsStart(unsigned cw_index)
{
    DECLARE_NOTHING();

    ES_CodeGenerator::OutOfOrderBlock *slow_case = cg.StartOutOfOrderBlock();

    if (!c_code) c_code = cg.NewConstant(code);

    cg.MOV(1, ES_CodeGenerator::REG_R0);
    cg.LDR(c_code, ES_CodeGenerator::REG_R1);
    cg.STR(ES_CodeGenerator::REG_R0, OBJECT_MEMBER(ES_CodeGenerator::REG_R1, ES_Code, has_failed_preconditions));

    if (!c_ForceUpdateNativeDispatcher) c_ForceUpdateNativeDispatcher = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::ForceUpdateNativeDispatcher));

    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
    MoveImmediateToRegister(cg, cw_index + (constructor ? code->data->codewords_count : 0), ES_CodeGenerator::REG_R1);
    cg.LDR(c_ForceUpdateNativeDispatcher, ES_CodeGenerator::REG_LR);
    cg.BLX(ES_CodeGenerator::REG_LR);
    cg.BX(ES_CodeGenerator::REG_R0);

    cg.EndOutOfOrderBlock(FALSE);

    return slow_case->GetJumpTarget();
}

void
ES_Native::EmitArithmeticBlockSlowPath(ArithmeticBlock *arithmetic_block)
{
    DECLARE_NOTHING();

    unsigned end_instruction_index = arithmetic_block->end_instruction_index;
    unsigned last_instruction_index = end_instruction_index - 1;
    unsigned last_cw_index = code->data->instruction_offsets[last_instruction_index];

    switch (code->data->codewords[last_cw_index].instruction)
    {
    case ESI_RETURN_VALUE:
    case ESI_JUMP_IF_TRUE:
    case ESI_JUMP_IF_FALSE:
        end_instruction_index = last_instruction_index;
    }

    unsigned end_cw_index = code->data->instruction_offsets[end_instruction_index];

    arithmetic_block->slow_case = current_slow_case = cg.StartOutOfOrderBlock();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    OpString annotation;

    annotation.AppendFormat(UNI_L("  from EmitArithmeticBlockSlowPath(), for block [%u, %u)"), arithmetic_block->start_cw_index, arithmetic_block->end_cw_index);

    if (end_cw_index == last_cw_index)
    {
        annotation.AppendL(" with trailing ");
        annotation.AppendL(g_instruction_strings[code->data->codewords[last_cw_index].instruction]);
    }

    annotation.AppendL(":\n");
    cg.Annotate(annotation.CStr());
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    if (!is_light_weight)
        EmitExecuteBytecode(arithmetic_block->start_instruction_index, end_instruction_index, end_cw_index != last_cw_index);

    if (end_cw_index == last_cw_index)
    {
        if (end_cw_index == entry_point_cw_index)
            entry_point_jump_target = EmitEntryPoint();

        switch (code->data->codewords[last_cw_index].instruction)
        {
        case ESI_RETURN_VALUE:
            cw_index = last_cw_index;

            EmitRegisterCopy(VR(code->data->codewords[last_cw_index + 1].index), VR(0));
            cg.Jump(epilogue_jump_target);

            cg.EndOutOfOrderBlock(FALSE);
            break;

        case ESI_JUMP_IF_TRUE:
        case ESI_JUMP_IF_FALSE:
            ES_CodeGenerator::JumpTarget *true_target = NULL, *false_target = NULL;
            Operand value_target;

            GetConditionalTargets(last_cw_index, value_target, true_target, false_target);

            cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, implicit_bool), SCRATCHI1);
            cg.TEQ(SCRATCHI1, 0);

            if (true_target)
                cg.Jump(true_target, ES_NATIVE_CONDITION_NOT_EQUAL);
            else
                cg.Jump(false_target, ES_NATIVE_CONDITION_EQUAL);

            cg.EndOutOfOrderBlock();
        }
    }
    else
        cg.EndOutOfOrderBlock();
}

ES_CodeGenerator::JumpTarget *
ES_Native::EmitEntryPoint(BOOL custom_light_weight_entry)
{
    if (is_light_weight)
    {
        DECLARE_NOTHING();

        if (entry_point_cw_index == 0 && prologue_entry_point)
            return NULL;

        ES_CodeGenerator::JumpTarget *real_entry_point = NULL;
        ES_CodeGenerator::OutOfOrderBlock *the_lightweight_entry_point = NULL;

        if (!custom_light_weight_entry)
        {
            real_entry_point = cg.Here();
            the_lightweight_entry_point = cg.StartOutOfOrderBlock();
        }

        cg.ADD(ES_CodeGenerator::REG_SP, StackSpaceAllocated() - sizeof(void *), ES_CodeGenerator::REG_SP);
        cg.POP(SCRATCHI1);
        cg.STR(SCRATCHI1, OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, native_stack_frame));
        cg.POP(ES_CodeGenerator::REG_LR);

        if (!custom_light_weight_entry)
        {
            cg.Jump(real_entry_point);
            cg.EndOutOfOrderBlock(FALSE);

            return the_lightweight_entry_point->GetJumpTarget();
        }
        else
            return NULL;
    }
#ifndef OPPSEUDOTHREAD_STACK_SWAPPING
    else if (prologue_entry_point)
    {
        ES_CodeGenerator::JumpTarget *real_entry_point = cg.Here();
        ES_CodeGenerator::OutOfOrderBlock *the_prologue_entry_point = cg.StartOutOfOrderBlock();

        BOOL has_variable_object;

        SetupNativeStackFrame(this, code, stack_space_allocated, stack_frame_padding, has_variable_object, TRUE);

        cg.Jump(real_entry_point);
        cg.EndOutOfOrderBlock(FALSE);

        return the_prologue_entry_point->GetJumpTarget();
    }
#endif // OPPSEUDOTHREAD_STACK_SWAPPING
    else
        return cg.Here();
}

ES_CodeGenerator::JumpTarget *
ES_Native::EmitInlineResumption(unsigned class_id, VirtualRegister *object_vr, ES_CodeGenerator::JumpTarget *failure, unsigned char *mark_failure)
{
    DECLARE_NOTHING();

    entry_point_jump_target = EmitEntryPoint(FALSE);
    ES_CodeGenerator::OutOfOrderBlock *extra_check = cg.StartOutOfOrderBlock();

    LoadObjectOperand(object_vr->index, SCRATCHI2);

    cg.LDR(OBJECT_MEMBER(SCRATCHI2, ES_Object, klass), SCRATCHI2);
    cg.LDR(OBJECT_MEMBER(SCRATCHI2, ES_Class, class_id), SCRATCHI2);
    EqCompareRegisterToImmediate(cg, SCRATCHI2, class_id);
    cg.Jump(entry_point_jump_target, ES_NATIVE_CONDITION_EQUAL);

    /* Update profile data to indicate that the instruction failed inlining. */
    MovePointerToRegister(cg, mark_failure, SCRATCHI1);
    cg.LDRB(SCRATCHI1, 0, SCRATCHI2);
    cg.ORR(SCRATCHI2, ES_CodeStatic::GET_FAILED_INLINE, SCRATCHI2);
    cg.STRB(SCRATCHI2, SCRATCHI1, 0);
    cg.Jump(failure);

    cg.EndOutOfOrderBlock(FALSE);
    return extra_check->GetJumpTarget();
}

void
ES_Native::EmitTick()
{
    DECLARE_NOTHING();

    if (!check_out_of_time)
    {
        check_out_of_time = cg.StartOutOfOrderBlock();

        cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
        cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::CheckOutOfTimeFromMachineCode)), ES_CodeGenerator::REG_PC);

        cg.EndOutOfOrderBlock(FALSE);
    }

    ES_CodeGenerator::Register time_until_check(ES_CodeGenerator::REG_R2);

    cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, time_until_check), time_until_check);
    cg.SUB(time_until_check, 1, time_until_check, ES_CodeGenerator::SET_CONDITION_CODES);
    cg.STR(time_until_check, OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, time_until_check), ES_NATIVE_CONDITION_NOT_EQUAL);

    cg.Call(check_out_of_time->GetJumpTarget(), ES_NATIVE_CONDITION_EQUAL);
}

static ES_CodeGenerator::OutOfOrderBlock *
InitializeFormals(ES_Native *native, unsigned formals_count)
{
    ES_CodeGenerator &cg = native->cg;

    ES_CodeGenerator::OutOfOrderBlock *initialize_formals = cg.StartOutOfOrderBlock();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("  from GeneratePrologue(), initializing formals to undefined:\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    cg.MOV(ESTYPE_UNDEFINED, ES_CodeGenerator::REG_R0);

    unsigned vr_index = 2 + formals_count - 1;

    if (formals_count > 1)
    {
        /* Let PARAMETERS_COUNT be the number of formals to set
           to undefined minus one.  This will be at least zero,
           because this is a slow case that is only called the
           function is called with too few arguments. */
        cg.RSB(PARAMETERS_COUNT, formals_count - 1, PARAMETERS_COUNT);

        ES_NativeJumpCondition condition(ES_NATIVE_CONDITION_ALWAYS);

        while (TRUE)
        {
            cg.STR(ES_CodeGenerator::REG_R0, BASE_REGISTER(native->VR(vr_index)), TYPE_OFFSET(native->VR(vr_index)), condition);

            if (--vr_index < 2)
                break;

            cg.SUB(PARAMETERS_COUNT, 1, PARAMETERS_COUNT, ES_CodeGenerator::SET_CONDITION_CODES);
            condition = ES_NATIVE_CONDITION_POSITIVE_OR_ZERO;
        }
    }
    else
        cg.STR(ES_CodeGenerator::REG_R0, BASE_REGISTER(native->VR(vr_index)), TYPE_OFFSET(native->VR(vr_index)));

    cg.EndOutOfOrderBlock();

    return initialize_formals;
}

void
ES_Native::GeneratePrologue()
{
    DECLARE_NOTHING();

    if (!constructor && code->data->codewords_count == 1 && code->data->codewords[0].instruction == ESI_RETURN_NO_VALUE)
        return;

    cg.StartPrologue();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("  from GeneratePrologue():\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    if (entry_point_cw == code->data->codewords && prologue_entry_point && !entry_point_jump_target)
        entry_point_jump_target = cg.Here();

    BOOL has_variable_object = FALSE;

    if (is_trivial)
    {
        if (code->type == ES_Code::TYPE_FUNCTION)
        {
            ES_FunctionCodeStatic *data = fncode->GetData();
            if (is_inlined_function_call)
            {
                for (unsigned index = inlined_call_argc; index < data->formals_count; index++)
                    EmitSetRegisterType(VR(2 + index), ESTYPE_UNDEFINED);
            }
            else if (data->formals_count > 0)
            {
                ES_CodeGenerator::OutOfOrderBlock *initialize_formals = InitializeFormals(this, data->formals_count);

                cg.CMP(PARAMETERS_COUNT, data->formals_count);
                cg.Jump(initialize_formals->GetJumpTarget(), ES_NATIVE_CONDITION_LESS);

                cg.SetOutOfOrderContinuationPoint(initialize_formals);
            }
        }
    }
    else
    {
        if (is_light_weight)
        {
            if (!light_weight_failure)
                EmitLightWeightFailure();

            if (fncode)
                if (is_inlined_function_call)
                {
                    ES_FunctionCode *fncode = static_cast<ES_FunctionCode *>(code);

                    for (unsigned index = inlined_call_argc; index < fncode->GetData()->formals_count; ++index)
                        EmitSetRegisterType(VR(2 + index), ESTYPE_UNDEFINED);
                }
                else
                {
                    cg.CMP(PARAMETERS_COUNT, fncode->GetData()->formals_count);
                    cg.Jump(light_weight_wrong_argc, ES_NATIVE_CONDITION_NOT_EQUAL);
                }

            if (optimization_data->uses_this && !code->data->is_strict_mode)
            {
                ES_CodeGenerator::OutOfOrderBlock *prepare_this(cg.StartOutOfOrderBlock());
                ES_CodeGenerator::Constant *c_PrepareThisObjectLightWeight(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::PrepareThisObjectLightWeight)));

                cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
                cg.MOV(REGISTER_FRAME_POINTER, ES_CodeGenerator::REG_R1);

                EmitFunctionCall(c_PrepareThisObjectLightWeight);

                cg.EndOutOfOrderBlock();

                EmitRegisterTypeCheck(VR(0), ESTYPE_OBJECT, prepare_this->GetJumpTarget());

                cg.SetOutOfOrderContinuationPoint(prepare_this);
            }
        }
        else
        {
#ifdef OPPSEUDOTHREAD_STACK_SWAPPING
            /* A loop dispatcher will have its stack frame created by
               ES_Execution_Context::ReconstructNativeStack(). */
            if (!loop_dispatcher)
#endif // OPPSEUDOTHREAD_STACK_SWAPPING
            {
                SetupNativeStackFrame(this, code, stack_space_allocated, stack_frame_padding, has_variable_object);
                OP_ASSERT(StackSpaceAllocated() == stack_space_allocated);
            }
            else
                stack_space_allocated = StackSpaceAllocated();

            ES_CodeGenerator::OutOfOrderBlock *stack_or_register_limit_exceeded = cg.StartOutOfOrderBlock();
            ES_CodeGenerator::Constant *constant = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::StackOrRegisterLimitExceeded));

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
            cg.Annotate(UNI_L("  from GeneratePrologue(), calling StackOrRegisterLimitExceeded():\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
            cg.MOV(constructor ? 1 : 0, ES_CodeGenerator::REG_R1);
            cg.LDR(constant, ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);
            cg.ADD(ES_CodeGenerator::REG_SP, stack_space_allocated, ES_CodeGenerator::REG_SP);
            cg.POP(ES_CodeGenerator::REG_PC);
            cg.EndOutOfOrderBlock(FALSE);

            if (entry_point_cw_index == 0 && !prologue_entry_point)
            {
                ES_CodeGenerator::OutOfOrderBlock *entry_point = cg.StartOutOfOrderBlock();

                entry_point_jump_target = cg.Here();

                cg.LDR(ES_CodeGenerator::REG_SP, stack_frame_padding, PARAMETERS_COUNT);
                cg.EndOutOfOrderBlock();
                cg.SetOutOfOrderContinuationPoint(entry_point);
            }

            cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, stack_limit), SCRATCHI1);
            cg.CMP(ES_CodeGenerator::REG_SP, SCRATCHI1);
            cg.Jump(stack_or_register_limit_exceeded->GetJumpTarget(), ES_NATIVE_CONDITION_LOWER);

            cg.MOV(PARAMETERS_COUNT, ES_CodeGenerator::REG_R4);
            cg.LDR(OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, reg_limit), SCRATCHI1);
            AddImmediateToRegister(cg, -static_cast<int>(code->data->register_frame_size * sizeof(ES_Value_Internal)), SCRATCHI1);
            cg.MOV(ES_CodeGenerator::REG_R4, PARAMETERS_COUNT);

            cg.CMP(SCRATCHI1, REGISTER_FRAME_POINTER);
            cg.Jump(stack_or_register_limit_exceeded->GetJumpTarget(), ES_NATIVE_CONDITION_LOWER);

            if (fncode)
            {
                if (!constructor && optimization_data->uses_this && !code->data->is_strict_mode)
                {
                    ES_CodeGenerator::OutOfOrderBlock *prepare_this(cg.StartOutOfOrderBlock());
                    ES_CodeGenerator::Constant *c_PrepareThisObject(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Native::PrepareThisObject)));

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                    cg.Annotate(UNI_L("  from GeneratePrologue(), calling PrepareThis():\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                    cg.MOV(PARAMETERS_COUNT, ES_CodeGenerator::REG_R4);
                    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
                    cg.MOV(REGISTER_FRAME_POINTER, ES_CodeGenerator::REG_R1);
                    cg.LDR(c_PrepareThisObject, ES_CodeGenerator::REG_LR);
                    cg.BLX(ES_CodeGenerator::REG_LR);
                    cg.MOV(ES_CodeGenerator::REG_R4, PARAMETERS_COUNT);

                    cg.EndOutOfOrderBlock();

                    EmitRegisterTypeCheck(VR(0), ESTYPE_OBJECT, prepare_this->GetJumpTarget());

                    cg.SetOutOfOrderContinuationPoint(prepare_this);
                }

                ES_FunctionCodeStatic *data = fncode->GetData();
                ES_CodeGenerator::OutOfOrderBlock *create_arguments_object;

                /* Create arguments array, if necessary. */
                BOOL may_use_arguments = !data->is_strict_mode || data->has_redirected_call;
                if (data->arguments_index == ES_FunctionCodeStatic::ARGUMENTS_NOT_USED && may_use_arguments)
                    create_arguments_object = cg.StartOutOfOrderBlock();
                else
                    create_arguments_object = NULL;

                if (data->arguments_index != ES_FunctionCodeStatic::ARGUMENTS_NOT_USED || may_use_arguments)
                {
#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                    cg.Annotate(UNI_L("    GeneratePrologue(): create arguments object\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                    ES_CodeGenerator::Constant *c_CreateArgumentsObject(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::CreateArgumentsObject)));

                    if (data->formals_count != 0)
                        cg.MOV(PARAMETERS_COUNT, ES_CodeGenerator::REG_R5);

                    cg.MOV(PARAMETERS_COUNT, ES_CodeGenerator::REG_R3);
                    cg.MOV(REGISTER_FRAME_POINTER, ES_CodeGenerator::REG_R2);
                    cg.LDR(BASE_REGISTER(VR(1)), IVALUE_OFFSET(VR(1)), ES_CodeGenerator::REG_R1);
                    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);

                    cg.LDR(c_CreateArgumentsObject, ES_CodeGenerator::REG_LR);
                    cg.BLX(ES_CodeGenerator::REG_LR);

                    if (data->formals_count != 0)
                        cg.MOV(ES_CodeGenerator::REG_R5, PARAMETERS_COUNT);

                    if (create_arguments_object)
                        cg.EndOutOfOrderBlock();
                }

                ES_CodeGenerator::OutOfOrderBlock *initialize_formals;

                if (data->formals_count != 0)
                    initialize_formals = InitializeFormals(this, data->formals_count);
                else
                    initialize_formals = NULL;

                if (create_arguments_object || initialize_formals)
                {
                    cg.CMP(PARAMETERS_COUNT, data->formals_count);

                    if (initialize_formals)
                        cg.Jump(initialize_formals->GetJumpTarget(), ES_NATIVE_CONDITION_LOWER);
                    if (create_arguments_object)
                        cg.Jump(create_arguments_object->GetJumpTarget(), ES_NATIVE_CONDITION_HIGHER);

                    if (initialize_formals)
                        cg.SetOutOfOrderContinuationPoint(initialize_formals);
                    if (create_arguments_object)
                        cg.SetOutOfOrderContinuationPoint(create_arguments_object);
                }

                EmitTick();

                if (constructor)
                {
                    ES_CodeGenerator::Register object(ALLOCATED_OBJECT_REGISTER);
                    BOOL inline_allocation = constructor_final_class && CanAllocateObject(constructor_final_class, 0);
                    ES_CodeGenerator::OutOfOrderBlock *complex_case = cg.StartOutOfOrderBlock();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                    cg.Annotate(UNI_L("    GeneratePrologue(): calling ES_Execution_Context::AllocateObjectFromMachineCodeComplex()\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
                    cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::AllocateObjectFromMachineCodeComplex)), ES_CodeGenerator::REG_LR);
                    cg.BLX(ES_CodeGenerator::REG_LR);
                    cg.MOV(ES_CodeGenerator::REG_R0, object);
                    cg.EndOutOfOrderBlock();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                    cg.Annotate(UNI_L("    GeneratePrologue(): verifying ctor.prototype hasn't changed\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                    ES_CodeGenerator::Register function(ES_CodeGenerator::REG_R1);
                    ES_CodeGenerator::Register function_properties(ES_CodeGenerator::REG_R2);

                    cg.LDR(BASE_REGISTER(VR(1)), IVALUE_OFFSET(VR(1)), function);
                    cg.LDR(OBJECT_MEMBER(function, ES_Object, properties), function_properties);

                    unsigned prototype_offset = ES_Function::GetPropertyOffset(ES_Function::PROTOTYPE);
                    if (constructor_prototype)
                    {
                        LoadValue(cg, function_properties, prototype_offset, ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5);
                        MovePointerToRegister(cg, constructor_prototype, ES_CodeGenerator::REG_R0);

                        cg.CMP(TYPE_REGISTER(ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5), ESTYPE_OBJECT);
                        cg.CMP(IVALUE_REGISTER(ES_CodeGenerator::REG_R4, ES_CodeGenerator::REG_R5), ES_CodeGenerator::REG_R0, ES_NATIVE_CONDITION_EQUAL);
                        cg.Jump(complex_case->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);
                    }
                    else
                    {
                        cg.LDR(function_properties, prototype_offset + VALUE_TYPE_OFFSET, ES_CodeGenerator::REG_R0);
                        cg.CMP(ES_CodeGenerator::REG_R0, ESTYPE_OBJECT);
                        cg.Jump(complex_case->GetJumpTarget(), ES_NATIVE_CONDITION_EQUAL);
                    }

                    ES_CodeGenerator::OutOfOrderBlock *non_trivial;

                    if (inline_allocation)
                        non_trivial = cg.StartOutOfOrderBlock();
                    else
                        non_trivial = NULL;

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                    cg.Annotate(UNI_L("    GeneratePrologue(): calling ES_Execution_Context::AllocateObjectFromMachineCodeSimple()\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                    cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
                    MovePointerToRegister(cg, constructor_class, ES_CodeGenerator::REG_R1);
                    cg.LDR(cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::AllocateObjectFromMachineCodeSimple)), ES_CodeGenerator::REG_LR);
                    cg.BLX(ES_CodeGenerator::REG_LR);
                    cg.MOV(ES_CodeGenerator::REG_R0, object);

                    if (non_trivial)
                    {
                        cg.EndOutOfOrderBlock();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                        cg.Annotate(UNI_L("    GeneratePrologue(): inlined object allocation \n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                        EmitAllocateObject(constructor_class, constructor_final_class, 0, 0, NULL, non_trivial->GetJumpTarget());

                        cg.SetOutOfOrderContinuationPoint(non_trivial);
                    }

                    cg.SetOutOfOrderContinuationPoint(complex_case);

                    cg.MOV(ESTYPE_OBJECT, TYPE_REGISTER(ES_CodeGenerator::REG_R0, ES_CodeGenerator::REG_R1));
                    cg.MOV(object, IVALUE_REGISTER(ES_CodeGenerator::REG_R0, ES_CodeGenerator::REG_R1));

                    StoreValue(cg, ES_CodeGenerator::REG_R0, ES_CodeGenerator::REG_R1, VR(0));
                }
            }
        }

        if (loop_dispatcher && first_loop_io)
        {
            if (!c_ReadLoopIO) c_ReadLoopIO = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::ReadLoopIO));

            if (!c_code) c_code = cg.NewConstant(code);

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
            cg.LDR(c_code, ES_CodeGenerator::REG_R1);
            cg.MOV(REGISTER_FRAME_POINTER, ES_CodeGenerator::REG_R2);
            cg.LDR(c_ReadLoopIO, ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);
        }
    }

    cg.EndPrologue();
}

void
ES_Native::GenerateEpilogue()
{
    cg.SetJumpTarget(epilogue_jump_target);

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
    cg.Annotate(UNI_L("  from GenerateEpilogue():\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

    if (!constructor && code->data->codewords_count == 1 && code->data->codewords[0].instruction == ESI_RETURN_NO_VALUE)
    {
        if (!is_inlined_function_call)
            cg.BX(ES_CodeGenerator::REG_LR);
        return;
    }

    if (loop_dispatcher && first_loop_io)
    {
         if (!c_WriteLoopIO) c_WriteLoopIO = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::WriteLoopIO));

         if (!c_code) c_code = cg.NewConstant(code);

         cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
         cg.LDR(c_code, ES_CodeGenerator::REG_R1);
         cg.MOV(REGISTER_FRAME_POINTER, ES_CodeGenerator::REG_R2);
         cg.LDR(c_WriteLoopIO, ES_CodeGenerator::REG_LR);
         cg.BLX(ES_CodeGenerator::REG_LR);
    }

    if (!is_trivial && !is_light_weight)
    {
        DECLARE_NOTHING();

        if (code->type == ES_Code::TYPE_FUNCTION)
        {
            ES_FunctionCode *fncode = static_cast<ES_FunctionCode *>(code);
            ES_FunctionCodeStatic *data = fncode->GetData();

            ES_CodeGenerator::OutOfOrderBlock *detach_arguments;

            if (data->arguments_index == ES_FunctionCodeStatic::ARGUMENTS_NOT_USED)
                detach_arguments = cg.StartOutOfOrderBlock();
            else
                detach_arguments = NULL;

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
            cg.Annotate(UNI_L("    GenerateEpilogue(): detach arguments object\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

            ES_CodeGenerator::Constant *c_DetachArgumentsObject = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::DetachArgumentsObject));

            cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
            if (!detach_arguments)
                cg.LDR(ES_CodeGenerator::REG_SP, stack_space_allocated - 4 * sizeof(void *), ES_CodeGenerator::REG_R1);
            cg.LDR(c_DetachArgumentsObject, ES_CodeGenerator::REG_LR);
            cg.BLX(ES_CodeGenerator::REG_LR);

            if (detach_arguments)
            {
                cg.EndOutOfOrderBlock();

                cg.LDR(ES_CodeGenerator::REG_SP, stack_space_allocated - 4 * sizeof(void *), ES_CodeGenerator::REG_R1);
                cg.TEQ(ES_CodeGenerator::REG_R1, 0);
                cg.Jump(detach_arguments->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);

                cg.SetOutOfOrderContinuationPoint(detach_arguments);
            }

            if (data->CanHaveVariableObject())
            {
                ES_CodeGenerator::OutOfOrderBlock *detach_variables = cg.StartOutOfOrderBlock();

#ifdef NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT
                cg.Annotate(UNI_L("    GenerateEpilogue(): detach variables object\n"));
#endif // NATIVE_DISASSEMBLER_ANNOTATION_SUPPORT

                ES_CodeGenerator::Constant *c_DetachArgumentsObject = cg.NewFunction(reinterpret_cast<void (*)()>(&ES_Execution_Context::DetachVariableObject));

                cg.MOV(CONTEXT_POINTER, ES_CodeGenerator::REG_R0);
                cg.LDR(c_DetachArgumentsObject, ES_CodeGenerator::REG_LR);
                cg.BLX(ES_CodeGenerator::REG_LR);
                cg.EndOutOfOrderBlock();

                cg.LDR(ES_CodeGenerator::REG_SP, stack_space_allocated - 5 * sizeof(void *), ES_CodeGenerator::REG_R1);
                cg.TEQ(ES_CodeGenerator::REG_R1, 0);
                cg.Jump(detach_variables->GetJumpTarget(), ES_NATIVE_CONDITION_NOT_EQUAL);

                cg.SetOutOfOrderContinuationPoint(detach_variables);
            }
        }

        cg.ADD(ES_CodeGenerator::REG_SP, StackSpaceAllocated() - sizeof (void *), ES_CodeGenerator::REG_SP);
        cg.POP(SCRATCHI1);
        cg.STR(SCRATCHI1, OBJECT_MEMBER(CONTEXT_POINTER, ES_Execution_Context, native_stack_frame));
        cg.POP(ES_CodeGenerator::REG_PC);
    }
    else if (!is_inlined_function_call)
        cg.BX(ES_CodeGenerator::REG_LR);
}

/* static */ unsigned
ES_NativeStackFrame::GetFrameSize(ES_Code *code, BOOL include_return_address)
{
    /* Previous stack frame, register frame pointer and code pointer: */
    unsigned frame_size = 3 * sizeof(void *);

    if (code->type == ES_Code::TYPE_FUNCTION)
    {
        /* Arguments object and arguments count: */
        frame_size += 2 * sizeof(void *);

        if (code->CanHaveVariableObject())
            /* Variable object: */
            frame_size += sizeof(void *);
    }

    if ((frame_size + sizeof(void *)) & sizeof(void *))
        frame_size += sizeof(void *);

    if (include_return_address)
        frame_size += sizeof(void *);

    return frame_size;
}

#endif // ARCHITECTURE_ARM
#endif // ES_NATIVE_SUPPORT
