/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*-
**
** Copyright (C) 2003-2008 Opera Software AS.  All rights reserved.
**
** This file is part of the Opera web browser.  It may not be distributed
** under any circumstances.
**
** Yngve Pettersen
**
*/

#ifndef _FL_REC_H_
#define _FL_REC_H_

#ifdef DATASTREAM_SEQUENCE_ENABLED

/** DataStream_BaseRecord
 *	Contains the record's id and payload length, and can optionally load arbitrary payload items.
 *	Extra payload structures may, after the constructor call, be prepended before the tag and payload. 
 *	Subclasses may also change the order of the tag and recordlength fields, or disable either or both.
 *
 *	This implementation is only intended to handle records on the form
 *
 *		tag
 *		[payload length]
 *		[payload]
 *
 *	payload length and payload are optional. Payload have to be defined by the subclasses
 * 
 *  Presently limited to 32 bit tag-id and record payload length.
 *  Implementations may however override tag-id and length binary representations
 *
 *	Payload loading is handled through a 
 */
class DataStream_BaseRecord : public DataStream_SequenceBase
{
public:
	enum {
		RECORD_TAG    = DATASTREAM_START_VALUE + 0,
		RECORD_LENGTH = DATASTREAM_START_VALUE + 1,
		BASERECORD_START_VALUE  // start new normal IDs in subclasses from this value
	};

private:
	/** Have the tag field been read */
	UINT	read_tag_field:1;

	/** Have the length field been read */
	UINT	read_length_field:1;

	/** The default tagfield, used if the subcall does not define a special record */
	DataStream_UIntVarLength default_tag;

	/** The default tagfield, used if the subcall does not define a special record */
	DataStream_UIntVarLength default_length;

	/** 
	 *	The pipe used to read the payload of the content 
	 *	The maximum length to be read can be configured.
	 */
	DataStream_LengthLimitedPipe payload_source;

private:
	/** Generated list of records embedded in the payload. Is generated by the IndexRecord function */
	AutoDeleteHead payload_records_index;

protected:
	DataRecord_Spec	spec;

protected:
	/** Reference to the tag field of the record, actual object is either default_tag 
	 *	or a subclass owned object */
	DataStream_UIntReference tag;

	/** Reference to the length field of the record, actual object is either default_length
	 *	or a subclass owned object */
	DataStream_UIntReference length;

protected:

	//virtual DataStream *CreateRecordL();

	virtual DataStream *GetInputSource(DataStream *src);

#ifdef DATASTREAM_USE_SWAP_TAG_LEN_FUNC
	/** Swap the tag and length sequence in the record, write the length first */
	void SwapTagAndLength();
#endif

#ifdef DATASTREAM_USE_ADV_TAG_LEN_FUNC
	/** Register the argument as the real tag field of the record. Used when the tag has special binary representations. e.g. PGP and ANS.1 */
	void RegisterTag(DataStream_UIntBase *app_tag);

	/** Register the argument as the real length field of the record. Used when the tag has special binary representations. e.g. PGP and ANS.1 */
	void RegisterLength(DataStream_UIntBase *app_length);
#endif // DATASTREAM_USE_ADV_TAG_LEN_FUNC


public:

	/** Default constructor */
	DataStream_BaseRecord();

	/**
	 *	Returns the records tag number 
	 *
	 *	@return	uint32	The tag number
	 */
	uint32 GetTag() const
			{return tag->GetValue();}

	/** 
	 *	Sets this record's tag number 
	 *
	 *	@param	tag_number	The new tagnumber
	 */

	virtual void SetTag(uint32 tag_number);

	/** 
	 *	Calculates the value that is assigned to the Length field of the record 
	 *
	 *	By default the payload length is the amount of data that are registered 
	 *	*after* the length field, even if this is a tagfield (if the tag and 
	 *	length fields have been switched around).
	 */
	virtual uint32 CalculatePayloadLength();


	/** Get The record specification used for this record */
	virtual const DataRecord_Spec *GetRecordSpec();

	/** Set The record specification used for this record */
	void SetRecordSpec(const DataRecord_Spec &new_spec);
	void SetRecordSpec(const DataRecord_Spec *new_spec){SetRecordSpec(*new_spec);};

	DataStream_BaseRecord *First(){return (DataStream_BaseRecord *) Head::First();}
	DataStream_BaseRecord *Last(){return (DataStream_BaseRecord *) Head::Last();}

	/**
	 *	Read all the records in the payload, and insert them in a list that may then be searched
	 *	The payload MUST be fully read
	 *
	 *	The function will LEAVE if allocation fails
	 */
	void IndexRecordsL(); // Gets all records into a list;

	/**
	 *	Read all the records in the payload, and insert them in a list that may then be searched
	 *	The payload MUST be fully read
	 *
	 *	@return OpStatus::ERR_NO_MEMORY if allocation fails, else OpStatus::OK.
	 *
	 *	@see IndexRecordsL() - Same function but using LEAVE instead of OP_STATUS.
	 */
	OP_STATUS IndexRecords() { TRAPD(status, IndexRecordsL()); return status; }

	/**
	 *	Get the first record from the indexed list that has the give tag id
	 *
	 *	@param	tag		The tag id of the desired record 
	 *
	 *	@return	DataStream_GenericRecord *	Pointer to the requested record. NULL is returned if it was not available
	 */
	DataStream_BaseRecord *GetIndexedRecord(uint32 tag);

	/**
	 *	Get the first record after the record ptag from the indexed list 
	 *	that has the give tag id (or if the tag is 0, the same tag id as the prec record)
	 *
	 *	@param	p_rec	The record preceding the part of the indexed list 
	 *					that will be searched for the desired record
	 *					If this pointer is NULL, the search will start 
	 *					at the beginning of the list, and tag must be specified
	 *					If the record is not in the list 
	 *	@param	tag		The tag id of the desired record 
	 *					If tag is 0 (default) tag assumed to be the 
	 *					same as prec->GetTag() if prec is not NULL
	 *
	 *	@return	DataStream_GenericRecord *	Pointer to the requested record. NULL is returne if it was not 
	 *								available, or prec was not in the list of indexed records
	 */
	DataStream_BaseRecord *GetIndexedRecord(DataStream_BaseRecord *p_rec, uint32 tag = 0);

	/** Read the target record from the indexed record with the given tag **/
	OP_STATUS ReadRecordFromIndexedRecordL(DataStream *target, uint32 tag, BOOL content_only=FALSE);
	OP_STATUS ReadRecordFromIndexedRecordL(DataStream &target, uint32 tag, BOOL content_only=FALSE)
		{return ReadRecordFromIndexedRecordL(&target, tag);}
	OP_STATUS ReadContentFromIndexedRecordL(DataStream *target, uint32 tag)
		{return ReadRecordFromIndexedRecordL(target, tag, TRUE);}
	OP_STATUS ReadContentFromIndexedRecordL(DataStream &target, uint32 tag)
		{return ReadRecordFromIndexedRecordL(&target, tag, TRUE);}

	virtual uint32 GetAttribute(DataStream::DatastreamUIntAttribute attr);
	virtual void	PerformActionL(DataStream::DatastreamAction action, uint32 arg1=0, uint32 arg2=0);
#if defined _SSL_SUPPORT_
	virtual OP_STATUS PerformStreamActionL(DataStream::DatastreamStreamAction action, DataStream *src_target);
#endif

private:
	DataStream *GetLastHeaderElement();

#ifdef _DATASTREAM_DEBUG_
protected:
	virtual const char *Debug_ClassName(){return "DataStream_BaseRecord";}
#endif
};

#endif // DATASTREAM_SEQUENCE_ENABLED

#endif // _FL_REC_H_
