#!/usr/bin/perl -w
# -*- Mode: perl; tab-width: 4; indent-tabs-mode: t; -*-
#
# File name:          shaping.pl
# Contents:           Extracts the Arabic shaping information from the
#                     files from the Unicode data base and creates the
#                     code needed for the text shaping code.
#
# Copyright 2006-2011 Opera Software ASA. All rights reserved.
# This file is part of the Opera web browser.  It may not be distributed
# under any circumstances.
#
# Peter Krefting <peter@opera.com>

require v5.6.0;

# Read DerivedJoiningType.txt to extract the joining class for the
# characters.
# http://www.unicode.org/Public/UNIDATA/extracted/DerivedJoiningType.txt
open SHAPING, '<', '../../unicode/data/DerivedJoiningType.txt'
    or die "Could not find DerivedJoiningType.txt";
while (<SHAPING>)
{
    my ($start, $stop, $jointype) = (-1, -1, undef);
    if (/^([0-9A-F]{4})\s*; ([RLDCUT])/)
    {
        ($start, $stop, $jointype) = (hex($1), hex($1), $2);
    }
    elsif (/^([0-9A-F]{4})\.\.([0-9A-F]{4})\s*; ([RLDCUT])/)
    {
        ($start, $stop, $jointype) = (hex($1), hex($2), $3);
    }
    foreach $codepoint ($start..$stop)
    {
        if ($codepoint >= 0x620 && $codepoint <= 0x074a)
        {
            $joinclass[$codepoint - 0x0620] = $jointype;
        }
    }
}

# Read NamesList.txt to extract the entry point into the Arabic
# presentation forms blocks for the shaping characters.
# http://www.unicode.org/Public/UNIDATA/NamesList.txt
my (@prestablestart, @prestablecount);
open NAMES, '<', '../../unicode/data/NamesList.txt'
    or die "Cannot read NamesList.txt";
my $ignore = 1;
my $namescodepoint;
my $lastmapfrom;
while (<NAMES>)
{
    # Speed up by ignoring the uninteresting blocks.
    $ignore = 1 if /^@@/;
    $ignore = 0 if /^@@.*Arabic Presentation Forms/;
    next if $ignore;

	# ITVSDK-515: U+0649 has conflicting shaping forms. Ignore the
	# ones for Turkic languages.
	$namescodepoint = 0, next
		if /ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA/;

    if (/^([0-9A-F]{4})/)
    {
        # The current code point
        $namescodepoint = hex($1);
    }
    elsif (/# <.*> ([0-9A-F]{4})$/ && $namescodepoint)
    {
        # The shaping information for a single Arabic character that
        # we are looking for
        my $mapfrom = hex($1);
        if ($mapfrom >= 0x0620 && $mapfrom <= 0x06D3)
        {
            if (!defined $prestablestart[$mapfrom - 0x0620])
            {
                # Add the first code point found only, the other presentation
                # forms follow.

                $prestablestart[$mapfrom - 0x0620] = $namescodepoint;
                $prestablecount[$mapfrom - 0x0620] = 1;
                $lastmapfrom = $mapfrom;
            }
            else
            {
                # Count the number of presentation forms for this character.
                $prestablecount[$mapfrom - 0x0620] ++
                    if $lastmapfrom == $mapfrom;
				if ($lastmapfrom != $mapfrom)
				{
					printf "Conflicting shaping for U+%04X\n", $mapfrom;
				}
            }
        }
    }
}



# Output data
open OUTPUT, '>', '../text_shaper.inl'
    or die "Could not create text_shaper.inl";

print OUTPUT "/** \@file text_shaper.inl\n";
print OUTPUT "  * This file is auto-generated by modules/display/scripts/shaping.pl.\n";
print OUTPUT "  * DO NOT EDIT THIS FILE MANUALLY.\n";
print OUTPUT "  */\n\n";

# prestable_start
# Table mapping from the Arabic character to the first presentation form
# in the presentation form block.
print OUTPUT "/** Mapping from Arabic character to the first presentation form. */\n";
print OUTPUT "static const uni_char prestable_start[] = {\n";
for (my $cp = 0x0620; $cp <= 0x06D3; ++ $cp)
{
    if (defined $prestablestart[$cp - 0x0620])
    {
        printf OUTPUT "\t%#06x", $prestablestart[$cp - 0x0620];
    }
    else
    {
        print OUTPUT "\t0";
    }

    print OUTPUT ',' if $cp < 0x06D3;
    printf OUTPUT " // %#06x\n", $cp;
}
print OUTPUT "};\n\n";

# prestable_count
# Table mapping the Arabic character to the number of presentation forms
# available in the presentation form block.
print OUTPUT "/** Count of presentation forms for each Arabic character. */\n";
print OUTPUT "static const UINT8 prestable_count[] = {\n";
for (my $cp = 0x0620; $cp <= 0x06D3; ++ $cp)
{
    my $count = 0;
    $count = $prestablecount[$cp - 0x0620]
        if defined $prestablecount[$cp - 0x0620];

    print OUTPUT "\t$count";
    print OUTPUT ',' if $cp < 0x06D3;
    printf OUTPUT " // %#06x\n", $cp;
}
print OUTPUT "};\n\n";

# joinclasstable
# Table mapping the Arabic character to the joining class.
my %joinclasstablemapping =
(
    'R' => 'RIGHT_JOINING',
    'L' => 'LEFT_JOINING',
    'D' => 'DUAL_JOINING',
    'C' => 'JOIN_CAUSING',
    'U' => 'NON_JOINING',
    'T' => 'TRANSPARENT_JOINING',
);
print OUTPUT "/** Joining class for each Arabic and Syriac character. */\n";
print OUTPUT "static const char joinclasstable[] = {\n";
for (my $cp = 0x0620; $cp <= 0x074a; ++ $cp)
{
    my $class = 'U';
    if (defined $joinclass[$cp - 0x0620])
    {
        $class = $joinclass[$cp - 0x0620];
    }
    die "Undefined join class: $class for $cp"
        if !defined $joinclasstablemapping{$class};
    print OUTPUT "\tTextShaper::", $joinclasstablemapping{$class};
    print OUTPUT ',' if $cp < 0x074a;
    printf OUTPUT " // %#06x\n", $cp;
}
print OUTPUT "};\n";

close OUTPUT;
