
import os
import multiprocessing

# Configuration and logging

helpDescription = 'Build an Opera product.'

@option(('-c', '--conf'), 'Load the specified configuration file.', metavar='FILE')
def confFiles():
    return ['/etc/opera.conf.py',
            '/usr/local/etc/opera.conf.py',
            os.environ.get('FLOWER_USER_CONF', os.path.expanduser('~/.opera.conf.py')),
            '*.conf.py']

@option(('-j', '--jobs'), 'Maximum number of processes to run in parallel.')
def processQuota():
    return multiprocessing.cpu_count()

@option(('-k', '--keep-going'), 'Continue as much as possible after an error.', value=False)
def failEarly():
    return True

@option('--color', 'Colorize console output.')
def colorizeOutput():
    return True

@option(('-q', '--quiet'), 'Quiet (suppress all output except warnings and errors).', value=True)
def quiet():
    return False

@option(('-Q', '--very-quiet'), 'Very quiet (suppress all output except critical errors).', value=True)
def veryQuiet():
    return False

@option(('-v', '--verbose'), 'Verbose (print commands before executing them).', value=True)
def verbose():
    return False

def echoStdout():
    return not config.quiet and not config.veryQuiet

def echoStderr():
    return not config.veryQuiet

def echoCommands():
    return config.verbose

def echoMessage():
    return not config.quiet and not config.veryQuiet

@option(('-l', '--log'), 'Build log file name.', metavar='FILE')
def logName():
    return 'build.log'

@option('--no-log', 'Do not write the build log into a file.', value=False)
def logToFile():
    return True

@option('--log-keep', 'Number of older build log files to keep.')
def logKeep():
    return 8

goal = None

# Buildroot directories

@option(('-B', '--buildroot'), 'Directory for all files generated by the build (cannot be the source directory).', metavar='PATH')
def buildroot():
    return 'build'

def buildrootSources():
    return config.buildroot + '/src'

def buildrootObjects(binary, pgo=None):
    if pgo == 'generate':
        root = config.buildrootPGO
    else:
        root = config.buildroot
    return "{root}/obj/{b}".format(root=root, b=binary)

def buildrootBinaries(pgo=None):
    if pgo == 'generate':
        root = config.buildrootPGO
    else:
        root = config.buildroot
    return root + '/bin'

def buildrootExtra():
    return config.buildroot + '/etc'

def buildrootPGO():
    return config.buildroot + '/pgo'

def cleanPatterns():
    return [config.buildroot]

# Basic preprocessor and linker flags

libraries = []

@option(('-D', '--define'), 'Pass a macro definition to the preprocessor.', metavar='SYMBOL[=VALUE]')
def defines():
    defines = {}
    for lib in config.libraries:
        defines.update(lib.defines)
    return defines

@option(('-I', '--include-path'), 'Pass an include path to the preprocessor.', metavar='PATH')
def includePaths(binary=None):
    includePaths = ['.']
    if binary != 'opera_autoupdatechecker':
        includePaths += [config.buildrootObjects] if binary else []
        includePaths += [config.buildrootSources]
    for lib in config.libraries:
        includePaths += lib.includePaths
    return includePaths

def preprocessorFlags():
    flags = []
    for sym, value in config.defines.iteritems():
        if value:
            flags.append("-D{0}={1}".format(sym, value))
        else:
            flags.append("-D{0}".format(sym))
    flags += ["-I" + path for path in config.includePaths]
    return flags

def linkerFlags():
    flags = []
    for lib in config.libraries:
        flags += ['-Wl,' + prefix for prefix in lib.linkingPrefix]
        flags += ['-L' + path for path in lib.libraryPaths]
        flags += ['-l' + name for name in lib.libraryNames]
        flags += ['-Wl,' + postfix for postfix in lib.linkingPostfix]
    return flags
