#!/usr/bin/pike
// Extract canonical combining classes from the UnicodeData.txt file.
array(int) make_surrogates(int cc)
{
	if (cc < 0x10000)
		error("%x: Invalid call to make_surrogates", cc);
	int high = 0xD800 | ((cc - 0x10000) >> 10);
	int low  = 0xDC00 | ((cc - 0x10000) & 0x3FF);
	return ({ high, low });
}

int main()
{
	mapping classes = ([]), len=([]);
	int last = 0, last_canonical_combining_class_index;
	int last_canonical_combining_class;

	string comb = "";

	// Read UnicodeData.txt
	foreach (Stdio.stdin.read()/"\n", string line)
	{
		// Strip comments and split into semi-colon separated fields
		sscanf(line, "%s#", line);
		if (!sizeof(line))
			continue;
		array data = line / ";";

		if (sizeof(data) != 15)
			continue;

		// Get the codepoint for the current character and its
		// canonical combining class
		int codepoint, canonical_combining_class;
		sscanf(data[0], "%x", codepoint);
		sscanf(data[3], "%d", canonical_combining_class);

		if (canonical_combining_class)
		{
			// Remember for selftest
			string entry = "\0"*4;
			entry[0] = (codepoint >> 16) & 0xFF;
			entry[1] = (codepoint >> 8) & 0xFF;
			entry[2] = codepoint & 0xFF;
			entry[3] = canonical_combining_class;
			comb += entry;

			// Add to table		
			if (canonical_combining_class == last_canonical_combining_class &&
			    last+1 == codepoint)
			{
				// Run-length compression of runs of the same class
				len[last_canonical_combining_class_index]++;
			}
			else 
			{
				// Start a new run
				classes[codepoint] = canonical_combining_class;
				last_canonical_combining_class_index = codepoint;
				last_canonical_combining_class = canonical_combining_class;
			}
			last = codepoint;
		}
	}

	// Write selftest data
	Stdio.write_file("comb.dat", comb);

	// Write output
	write(#"/** @file canonicals.inl
 * This file is auto-generated by modules/unicode/scripts/make_canonicals.pike.
 * DO NOT EDIT THIS FILE MANUALLY.
 */\n
#ifdef USE_UNICODE_INC_DATA\n");
	write("static const Canonical canonicals[] = {\n\t");
	mapping top=([]);

	int hasnonbmp = 0;
	int entries = 0;
	foreach (sort(indices(classes)), int codepoint)
	{
		int utf16codeunits = codepoint;
		if (codepoint > 65535)
		{
			if (!hasnonbmp)
			{
				entries = 0;
				hasnonbmp = 1;
				write("\n};\n");
				write("/* Non-BMP portion of table */\n");
				write("static const CanonicalNonBMP canonicalsnonbmp[] = {\n\t");
			}
			array(int) surrogates = make_surrogates(codepoint);
			utf16codeunits = surrogates[0] << 16 | surrogates[1];
		}
		if (entries ++)
		{
			write(",\n\t");
		}
		write("{0x%04XU,%3d,%2d}", utf16codeunits, classes[codepoint], len[codepoint]+1);
		if (utf16codeunits != codepoint)
			write(" /* U+%X */", codepoint);
	}
	write("\n};\n#endif // USE_UNICODE_INC_DATA\n");
	return 0;
}
