#!/usr/bin/env python

import operator

# The input to this script should be a text file with entity name as first
# string (including ending semicolon if it requires it), and unicode number
# in the U+number format.  Anything after the first space after the number
# will be ignored.
# It should work to just copy the table from the spec into a text file and
# run the script on that textfile.

entities_file = 'html5_entities.txt'
c_file = 'html5entity_nodes_init.h'
nodes = []

class Entity:
    def __init__(self, name, uni_char):
        self.name = name
        self.uni_char = uni_char

class Node:
    def __init__(self, char):
        self.char = char
        self.uni_char = None
        self.children = {}
        self.nodeid = len(nodes)
        nodes.append(self)

    def insert(self, name, final_uni_char):
        if not name:
            self.uni_char = final_uni_char
        else:
            next = name[0]
            if self.children.has_key(next):
                n = self.children[next]
            else:
                n = Node(next)
                self.children[next] = n
            n.insert(name[1:], final_uni_char)

    def __str__(self):
        chlds = [str(n) for n in self.children.values()]
        return '%d %s %s [%s]' % (self.nodeid, self.char, self.uni_char or '', ', '.join(chlds))

    def count(self):
        return 1 + reduce(operator.add, [c.count() for c in self.children.values()], 0)

    def write_c_code(self, f):
        nchildren = len(self.children)
        id = self.nodeid
        f.write("\tentity_tree_nodes[%d].chr = '%s';\n" % (id, self.char or '\\0'))
        if self.uni_char:
            for i in range(2):
                char = self.uni_char[i]
                f.write("\tHTML5EntityMatcher::GetUnicodeSurrogates(%s, entity_tree_nodes[%d].replacement_chars[%d], entity_tree_nodes[%d].replacement_chars[%d]);\n" % (char, id, i * 2, id, i * 2 + 1))
        else:
            f.write('\t')
            for i in range(4):
                f.write("entity_tree_nodes[%d].replacement_chars[%d] = " % (id, i))
            f.write('0;\n')

        f.write("\tentity_tree_nodes[%d].nchildren = %d;\n" % (id, nchildren))
        children = self.children.values()
        if nchildren <= 2:
            for i in range(nchildren):
                f.write("\tentity_tree_nodes[%d].children[%d] = %d;\n" % (id, i, children[i].nodeid))
        else:
            f.write("\tentity_tree_nodes[%d].multiple_children = OP_NEWA_L(UINT16, %d);\n" % (id, nchildren))
            for i in range(nchildren):
                f.write("\tentity_tree_nodes[%d].multiple_children[%d] = %d;\n" % (id, i, children[i].nodeid))
        f.write('\n')


def read_entities_file():
    entities = []

    for line in file(entities_file):
        if not line.strip():
            continue
        tokens = line.split()
        [name,code] = tokens[:2]
        code = code.replace('U+', '0x')
        if len(tokens) > 2 and tokens[2].startswith('U+'):
            code2 = tokens[2].replace('U+', '0x')
            assert(len(name) >= 4)  # we will need space to replace at least 4 characters
        else:
            code2 = 0
        entities.append(Entity(name, (code, code2)))

    return entities

def build_tree(entities):
    root = Node(None)
    for ent in entities:
        root.insert(ent.name, ent.uni_char)
    return root

def write_c_code(filename):
    f = file(filename, 'w')

    f.write("// This file is autogenerated from the generate_entities_init.py script.  Don't edit manually!\n\n")
    f.write('\tnum_state_nodes = %d;\n\n' % len(nodes))
    f.write('\tentity_tree_nodes = OP_NEWA_L(HTML5EntityNode, num_state_nodes);\n\n')

    for n in nodes:
         n.write_c_code(f)
    

entities = read_entities_file()
root = build_tree(entities)
write_c_code(c_file)

