﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/os.h>
#include <nv/nv_MemoryManagement.h>
#include <nnt/gfx/testGfx_GraphicsFramework.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt.h>

static const int ExpectedVsyncCount = 300;
static const int ExpectedEventCount = 1;

static const int VsyncCountTolerance = 5;

static const int VsyncReceiveThreadCount = 2;
static const int VsyncReceiveThreadCoreNumberList[] = {1, 2, 0};
nn::os::ThreadType g_VsyncReceiveThreadList[VsyncReceiveThreadCount];
NN_ALIGNAS(4096) char g_VsyncReceiveThreadStackList[VsyncReceiveThreadCount][16 * 1024];
bool g_IsVsyncEventGot[VsyncReceiveThreadCount] = {};
int g_VsyncCountList[VsyncReceiveThreadCount] = {};
std::atomic_bool g_StopVsyncThreads;


namespace {
    const size_t GraphicsDonateSize = 8 * 1024 * 1024;

    void* GraphicsAllocate(size_t size, size_t alignment, void*) NN_NOEXCEPT
    {
        if(size > 1024 * 1024)
        {
            NN_LOG("Large alloc %d bytes\n", static_cast<int>(size));
        }
        return aligned_alloc(alignment, size);
    }

    void GraphicsFree(void* p, void*) NN_NOEXCEPT
    {
        free(p);
    }

    void* GraphicsReallocate(void* p, size_t size, void*) NN_NOEXCEPT
    {
        return realloc(p, size);
    }

    struct VsyncReceiverThreadFunctionParameter
    {
        nnt::gfx::GraphicsFramework* pGfw;
        nn::os::BarrierType* pBarrier;
        int threadIndex;
    };
    void VsyncReceiverThreadFunction(void* pArg) NN_NOEXCEPT
    {
        auto pParam = reinterpret_cast<const VsyncReceiverThreadFunctionParameter*>(pArg);

        nn::os::SystemEventType event;
        auto result = nn::vi::GetDisplayVsyncEvent(&event, pParam->pGfw->GetDisplay());

        nn::os::AwaitBarrier(pParam->pBarrier);

        if(result.IsSuccess())
        {
            g_IsVsyncEventGot[pParam->threadIndex] = true;

            while(!g_StopVsyncThreads)
            {
                nn::os::WaitSystemEvent(&event);
                nn::os::ClearSystemEvent(&event);
                g_VsyncCountList[pParam->threadIndex]++;
            }

            nn::os::DestroySystemEvent(&event);
        }
    }
}

TEST(Vi, VsyncEvent)
{
    nnt::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsDonateSize, GraphicsAllocate, GraphicsFree, GraphicsReallocate, nullptr);

    nnt::gfx::GraphicsFramework::FrameworkInfo info;
    info.SetDefault();
    nnt::gfx::GraphicsFramework gfw;
    gfw.Initialize(info);
    NN_UTIL_SCOPE_EXIT {
        gfw.Finalize();
    };

    g_StopVsyncThreads = false;

    nn::os::BarrierType barrier;
    nn::os::InitializeBarrier(&barrier, 1 + VsyncReceiveThreadCount);
    NN_UTIL_SCOPE_EXIT {
        nn::os::FinalizeBarrier(&barrier);
    };

    VsyncReceiverThreadFunctionParameter paramList[VsyncReceiveThreadCount] = {};
    for(int i = 0; i < VsyncReceiveThreadCount; i++)
    {
        auto& param = paramList[i];
        param.pGfw = &gfw;
        param.pBarrier = &barrier;
        param.threadIndex = i;
        nn::os::CreateThread(
            &g_VsyncReceiveThreadList[i],
            VsyncReceiverThreadFunction,
            &param,
            g_VsyncReceiveThreadStackList[i],
            sizeof(g_VsyncReceiveThreadStackList[i]),
            nn::os::DefaultThreadPriority,
            VsyncReceiveThreadCoreNumberList[i % (sizeof(VsyncReceiveThreadCoreNumberList) / sizeof(VsyncReceiveThreadCoreNumberList[0]))]
        );
        nn::os::StartThread(&g_VsyncReceiveThreadList[i]);
    }

    int frameCount = 0;

    nn::os::AwaitBarrier(&barrier);
    for(int frame = 0; frame < ExpectedVsyncCount; frame++)
    {
        gfw.AcquireTexture(0);
        gfw.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));
        gfw.BeginFrame(0);
        gfw.EndFrame(0);
        gfw.ExecuteCommand(0);
        gfw.QueuePresentTexture(1);
        gfw.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));
        frameCount++;
    }

    g_StopVsyncThreads = true;
    for(int i = 0; i < VsyncReceiveThreadCount; i++)
    {
        nn::os::WaitThread(&g_VsyncReceiveThreadList[i]);
        nn::os::DestroyThread(&g_VsyncReceiveThreadList[i]);
    }

    NN_LOG("FrameCount : %d\n", frameCount);
    int eventCount = 0;
    for(int i = 0; i < VsyncReceiveThreadCount; i++)
    {
        NN_LOG("Vsync%d     : %d\n", i, g_VsyncCountList[i]);
        if(g_IsVsyncEventGot[i])
        {
            eventCount++;
            EXPECT_NEAR(frameCount, g_VsyncCountList[i], VsyncCountTolerance);
        }
        else
        {
            EXPECT_EQ(0, g_VsyncCountList[i]);
        }
    }
    EXPECT_EQ(ExpectedEventCount, eventCount);

}
