﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ServiceImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/vi/vi_Result.h>

#include "Log.h"
#include "ManagerObject.h"
#include "IndirectLayerSlotList.h"

namespace nnt{ namespace vi{

    ServiceImpl::ServiceImpl() NN_NOEXCEPT
    {
    }

    ServiceImpl::~ServiceImpl() NN_NOEXCEPT
    {
    }

    // Indirect Display
    nn::Result ServiceImpl::CreateIndirectLayer(std::int32_t slot) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(slot >= 0 && slot < IndirectLayerSlotCount, nn::vi::ResultInvalidRange());

        auto& layer = g_IndirectLayerSlotList[slot];
        NN_RESULT_THROW_UNLESS(layer.hLayer == 0, nn::vi::ResultAlreadyOpened());

        NN_RESULT_DO(g_ManagerObject.GetManagerDisplayService()->CreateIndirectLayer(&layer.hLayer));
        NNT_VI_LOG("CreateIndirectLayer -> %lld\n", layer.hLayer);
        NN_RESULT_SUCCESS;
    }

    nn::Result ServiceImpl::DestroyIndirectLayer(std::int32_t slot) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(slot >= 0 && slot < IndirectLayerSlotCount, nn::vi::ResultInvalidRange());

        auto& layer = g_IndirectLayerSlotList[slot];
        NN_RESULT_THROW_UNLESS(layer.hLayer != 0, nn::vi::ResultNotFound());

        NNT_VI_LOG("DestroyIndirectLayer %lld\n", layer.hLayer);
        NN_RESULT_DO(g_ManagerObject.GetManagerDisplayService()->DestroyIndirectLayer(layer.hLayer));
        layer.hLayer = 0;
        layer.hProducerEndPoint = 0;
        layer.hConsumerEndPoint = 0;

        NN_RESULT_SUCCESS;
    }

    nn::Result ServiceImpl::CreateIndirectLayerProducerEndPoint(
        nn::sf::Out<nn::vi::IndirectProducerHandleType> outHandle,
        std::int32_t slot,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(slot >= 0 && slot < IndirectLayerSlotCount, nn::vi::ResultInvalidRange());

        auto& layer = g_IndirectLayerSlotList[slot];
        NN_RESULT_THROW_UNLESS(layer.hLayer != 0, nn::vi::ResultNotFound());
        NN_RESULT_THROW_UNLESS(layer.hProducerEndPoint == 0, nn::vi::ResultAlreadyOpened());

        NN_RESULT_DO(g_ManagerObject.GetManagerDisplayService()->CreateIndirectProducerEndPoint(&layer.hProducerEndPoint, layer.hLayer, aruid));
        NNT_VI_LOG("CreateIndirectLayerProducerEndPoint -> %lld\n", layer.hProducerEndPoint);
        outHandle.Set(layer.hProducerEndPoint);
        NN_RESULT_SUCCESS;
    }

    nn::Result ServiceImpl::DestroyIndirectLayerProducerEndPoint(std::int32_t slot) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(slot >= 0 && slot < IndirectLayerSlotCount, nn::vi::ResultInvalidRange());

        auto& layer = g_IndirectLayerSlotList[slot];
        NN_RESULT_THROW_UNLESS(layer.hLayer != 0, nn::vi::ResultNotFound());
        NN_RESULT_THROW_UNLESS(layer.hProducerEndPoint != 0, nn::vi::ResultNotFound());

        NNT_VI_LOG("DestroyIndirectLayerProducerEndPoint %lld\n", layer.hProducerEndPoint);
        NN_RESULT_DO(g_ManagerObject.GetManagerDisplayService()->DestroyIndirectProducerEndPoint(layer.hLayer));
        layer.hProducerEndPoint = 0;
        NN_RESULT_SUCCESS;
    }

    nn::Result ServiceImpl::CreateIndirectLayerConsumerEndPoint(
        nn::sf::Out<nn::vi::IndirectConsumerHandleType> outHandle,
        std::int32_t slot,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(slot >= 0 && slot < IndirectLayerSlotCount, nn::vi::ResultInvalidRange());

        auto& layer = g_IndirectLayerSlotList[slot];
        NN_RESULT_THROW_UNLESS(layer.hLayer != 0, nn::vi::ResultNotFound());
        NN_RESULT_THROW_UNLESS(layer.hConsumerEndPoint == 0, nn::vi::ResultAlreadyOpened());

        NN_RESULT_DO(g_ManagerObject.GetManagerDisplayService()->CreateIndirectConsumerEndPoint(&layer.hConsumerEndPoint, layer.hLayer, aruid));
        NNT_VI_LOG("CreateIndirectLayerConsumerEndPoint -> %lld\n", layer.hConsumerEndPoint);
        outHandle.Set(layer.hConsumerEndPoint);
        NN_RESULT_SUCCESS;
    }

    nn::Result ServiceImpl::DestroyIndirectLayerConsumerEndPoint(std::int32_t slot) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(slot >= 0 && slot < IndirectLayerSlotCount, nn::vi::ResultInvalidRange());

        auto& layer = g_IndirectLayerSlotList[slot];
        NN_RESULT_THROW_UNLESS(layer.hLayer != 0, nn::vi::ResultNotFound());
        NN_RESULT_THROW_UNLESS(layer.hConsumerEndPoint != 0, nn::vi::ResultNotFound());

        NNT_VI_LOG("DestroyIndirectLayerConsumerEndPoint %lld\n", layer.hConsumerEndPoint);
        NN_RESULT_DO(g_ManagerObject.GetManagerDisplayService()->DestroyIndirectConsumerEndPoint(layer.hLayer));
        layer.hConsumerEndPoint = 0;
        NN_RESULT_SUCCESS;
    }

    nn::Result ServiceImpl::SetIndirectLayerProducerFlipOffset(std::int32_t slot, nn::TimeSpan offset) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(slot >= 0 && slot < IndirectLayerSlotCount, nn::vi::ResultInvalidRange());

        auto& layer = g_IndirectLayerSlotList[slot];
        NN_RESULT_THROW_UNLESS(layer.hLayer != 0, nn::vi::ResultNotFound());
        NN_RESULT_THROW_UNLESS(layer.hProducerEndPoint != 0, nn::vi::ResultNotFound());

        NNT_VI_LOG("SetIndirectLayerProducerFlipOffset %lld, %lldus\n", layer.hProducerEndPoint, offset.GetMicroSeconds());
        NN_RESULT_DO(g_ManagerObject.GetManagerDisplayService()->SetIndirectProducerFlipOffset(layer.hLayer, layer.hProducerEndPoint, offset));
        NN_RESULT_SUCCESS;
    }


}}
