﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include "SettingsFixture.h"
#include "SettingsStub.h"
#include "settings/visrv_Settings.h"
#include "settings/visrv_PlatformConfig.h"
#include "settings/visrv_Strategy.h"
#include "settings/visrv_InterfaceType.h"

typedef SettingsFixture Settings;

TEST_F(Settings, Initialize_AllDefault)
{
    nn::visrv::settings::Initialize();

    const nn::visrv::settings::PlatformConfig* config = nn::visrv::settings::GetPlatformConfig();
    ASSERT_NE(nullptr, config);
    EXPECT_EQ(0, config->physicalDisplayCount);
    EXPECT_EQ(nn::visrv::settings::Strategy::HighestPriority, config->defaultDisplayStrategy);

    for( int i = 0; i < sizeof(config->dcList) / sizeof(config->dcList[0]); ++i )
    {
        const nn::visrv::settings::DisplayController& dc = config->dcList[i];

        EXPECT_EQ(-1, dc.priority);
        EXPECT_EQ(nn::visrv::settings::InterfaceType::Null, dc.interface);
        EXPECT_EQ(0, dc.modeCount);

        for( int j = 0; j < sizeof(dc.modes) / sizeof(dc.modes[0]); ++j )
        {
            EXPECT_EQ(0, dc.modes[j].clientInfo.width);
            EXPECT_EQ(0, dc.modes[j].clientInfo.height);
            EXPECT_EQ(0.f, dc.modes[j].clientInfo.refreshRate);
            EXPECT_EQ(nn::vi::StereoMode_None, dc.modes[j].clientInfo.mode);
            EXPECT_EQ(0, dc.modes[j].ratio);
        }
    }

    for( int i = 0; i < sizeof(config->displays) / sizeof(config->displays[0]); ++i )
    {
        for( int j = 0; j < sizeof(config->displays[i].name); ++j )
        {
            EXPECT_EQ(0, config->displays[i].name[j]);
        }

        for( int j = 0; j < sizeof(config->displays[i].edidDisplayName); ++j )
        {
            EXPECT_EQ(0, config->displays[i].edidDisplayName[j]);
        }

        EXPECT_EQ(-1, config->displays[i].dc);
    }

    nn::visrv::settings::Finalize();
}

TEST_F(Settings, PhysicalDisplay)
{
    nnt::settings::WriteValue(1, "physical_display_count");
    nnt::settings::WriteBuffer("External", "physical_display_0_name");
    nnt::settings::WriteBuffer("Edid", "physical_display_0_edid_display_name");
    nnt::settings::WriteValue(5, "physical_display_0_dc");

    nn::visrv::settings::Initialize();

    const nn::visrv::settings::PlatformConfig* config = nn::visrv::settings::GetPlatformConfig();

    ASSERT_EQ(1, config->physicalDisplayCount);
    EXPECT_STREQ("External", config->displays[0].name);
    EXPECT_STREQ("Edid", config->displays[0].edidDisplayName);
    EXPECT_EQ(5, config->displays[0].dc);

    for( int i = 1; i < sizeof(config->displays) / sizeof(config->displays[0]); ++i )
    {
        for( int j = 0; j < sizeof(config->displays[i].name); ++j )
        {
            EXPECT_EQ(0, config->displays[i].name[j]);
        }

        for( int j = 0; j < sizeof(config->displays[i].edidDisplayName); ++j )
        {
            EXPECT_EQ(0, config->displays[i].edidDisplayName[j]);
        }

        EXPECT_LT(config->displays[i].dc, 0);
    }

    nn::visrv::settings::Finalize();
}

TEST_F(Settings, PhysicalDisplay_InvalidName)
{
    nnt::settings::WriteValue(1, "physical_display_count");
    nnt::settings::WriteBuffer("Edid", "physical_display_0_edid_display_name");
    nnt::settings::WriteValue(5, "physical_display_0_dc");

    nn::visrv::settings::Initialize();

    const nn::visrv::settings::PlatformConfig* config = nn::visrv::settings::GetPlatformConfig();

    ASSERT_EQ(0, config->physicalDisplayCount);

    for( int i = 0; i < sizeof(config->displays) / sizeof(config->displays[0]); ++i )
    {
        for( int j = 0; j < sizeof(config->displays[i].name); ++j )
        {
            EXPECT_EQ(0, config->displays[i].name[j]);
        }

        for( int j = 0; j < sizeof(config->displays[i].edidDisplayName); ++j )
        {
            EXPECT_EQ(0, config->displays[i].edidDisplayName[j]);
        }

        EXPECT_LT(config->displays[i].dc, 0);
    }

    nn::visrv::settings::Finalize();
}

TEST_F(Settings, PhysicalDisplay_InvalidDc)
{
    nnt::settings::WriteValue(1, "physical_display_count");
    nnt::settings::WriteBuffer("External", "physical_display_0_name");
    nnt::settings::WriteValue(-5, "physical_display_0_dc");

    nn::visrv::settings::Initialize();

    const nn::visrv::settings::PlatformConfig* config = nn::visrv::settings::GetPlatformConfig();

    ASSERT_EQ(0, config->physicalDisplayCount);

    for( int i = 0; i < sizeof(config->displays) / sizeof(config->displays[0]); ++i )
    {
        for( int j = 0; j < sizeof(config->displays[i].name); ++j )
        {
            EXPECT_EQ(0, config->displays[i].name[j]);
        }

        for( int j = 0; j < sizeof(config->displays[i].edidDisplayName); ++j )
        {
            EXPECT_EQ(0, config->displays[i].edidDisplayName[j]);
        }

        EXPECT_LT(config->displays[i].dc, 0);
    }

    nn::visrv::settings::Finalize();
}
