﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.private.h>

typedef nnt::vi::util::Fixture SetLayerSize;

TEST_F(SetLayerSize, FullscreenEnabled_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    nn::vi::LayerSettings settings;
    settings.Reset();
    settings.Set<nn::vi::LayerFlags::Fullscreen>();
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay, settings));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, width, height));

    // layer is automatically scaled to fullscreen
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerSize(pLayer, 200, 200));

    nn::vi::DestroyLayer(pLayer);
    nn::vi::CloseDisplay(pDisplay);
}

// all applets will use fullscreen...
TEST_F(SetLayerSize, DISABLED_FullscreenDisabled_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    nn::vi::LayerSettings settings;
    settings.Reset();
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay, settings));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, width, height));
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, 0, 0));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidDimensions, nn::vi::SetLayerSize(pLayer, -1, 0));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidDimensions, nn::vi::SetLayerSize(pLayer, 0, -1));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerSize(pLayer, width + 1, 0));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerSize(pLayer, 0, height + 1));

    const int X = 100;
    const int Y = 150;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetLayerPosition(pLayer, X, Y));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, width - X, 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, 0, height - Y));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerSize(pLayer, width - X, height - Y));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerSize(pLayer, width - X + 1, 0));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerSize(pLayer, 0, height - Y + 1));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetLayerSize(pLayer, width - X + 1, height - Y + 1));

    nn::vi::DestroyLayer(pLayer);
    nn::vi::CloseDisplay(pDisplay);
}

TEST(SetLayerSizePreconditions, ViNotInitialized_Any)
{
    nn::vi::Layer* pLayer = reinterpret_cast<nn::vi::Layer*>(0x1234);
    ASSERT_DEATH_IF_SUPPORTED(nn::vi::SetLayerSize(pLayer, 200, 200), "");
}

TEST(SetLayerSizePreconditions, pLayerNull_Any)
{
    nn::vi::Initialize();

    EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetLayerSize(nullptr, 200, 200), "");

    nn::vi::Finalize();
}

TEST(SetLayerSizePreconditions, pLayerInvalidHandle_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::Layer* pLayer = nullptr;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay));

        if( pLayer != nullptr )
        {
            nn::vi::DestroyLayer(pLayer);
            EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetLayerSize(pLayer, 200, 200), "");
        }

        nn::vi::CloseDisplay(pDisplay);
    }

    nn::vi::Finalize();
}
