﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.private.h>
#include <nn/ae.h>

typedef nnt::vi::util::Fixture SetDisplayUnderscan;

TEST_F(SetDisplayUnderscan, Default_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    int underscan = 0;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayUnderscan(pDisplay, underscan));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayUnderscan, External_IcosaCopper)
{
    if( nn::ae::GetOperationMode() == nn::ae::OperationMode_Console )
    {
        nn::vi::Initialize();

        nn::vi::Display* pDisplay;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

        int underscan;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayUnderscan(&underscan, pDisplay));

        nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
        int modeCount = nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay);
        ASSERT_GT(modeCount, 0);

        nn::vi::DisplayModeInfo current;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&current, pDisplay));

        for( int i = 0; i < modeCount; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayMode(pDisplay, modes + i));

            const int UnderscanMax = 20;
            for( int j = 0; j <= UnderscanMax; ++j )
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayUnderscan(pDisplay, j));
            }
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayUnderscan(pDisplay, underscan));
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayMode(pDisplay, &current));

        nn::vi::CloseDisplay(pDisplay);
    }
}

TEST_F(SetDisplayUnderscan, OutOfRange_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayUnderscan(pDisplay, -1));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayUnderscan(pDisplay, 21));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayUnderscan, Edid_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    int underscan = 0;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayUnderscan(pDisplay, underscan));

    nn::vi::CloseDisplay(pDisplay);
}

TEST(SetDisplayUnderscanPreconditions, ViNotInitialized_Any)
{
    nn::vi::Display* pDisplay = reinterpret_cast<nn::vi::Display*>(0x1234);
    int underscan = 0;
    ASSERT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayUnderscan(pDisplay, underscan), "");
}

TEST(SetDisplayUnderscanPreconditions, pDisplayNull_Any)
{
    nn::vi::Initialize();

    int underscan = 0;
    EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayUnderscan(nullptr, underscan), "");

    nn::vi::Finalize();
}

TEST(SetDisplayUnderscanPreconditions, pDisplayInvalidHandle_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    if( pDisplay != nullptr )
    {
        nn::vi::CloseDisplay(pDisplay);
        int underscan = 0;
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayUnderscan(pDisplay, underscan), "");
    }

    nn::vi::Finalize();
}
