﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/os.h>
#include <nn/vi.h>

typedef nnt::vi::util::Fixture GetDisplayVsyncEvent;

TEST_F(GetDisplayVsyncEvent, Default_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEventType vsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay));

    const int Samples = 1024;
    for( int i = 0; i < Samples; ++i )
    {
        bool isSignaled = nn::os::TimedWaitSystemEvent(&vsync, nn::TimeSpan::FromSeconds(1));
        EXPECT_TRUE(isSignaled);
    }

    nn::os::DestroySystemEvent(&vsync);
    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(GetDisplayVsyncEvent, Default_DoubleOpen_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEventType vsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay));

    nn::os::SystemEventType dupe;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultDenied, nn::vi::GetDisplayVsyncEvent(&dupe, pDisplay));

    nn::os::DestroySystemEvent(&vsync);
    nn::vi::CloseDisplay(pDisplay);
}

TEST(GetDisplayVsyncEventPreconditions, ViNotInitialized_Any)
{
    nn::os::SystemEventType vsync;
    // just need some non-null value to avoid tripping other preconditions
    nn::vi::Display* pDisplay = reinterpret_cast<nn::vi::Display*>(0x1234);

    ASSERT_DEATH_IF_SUPPORTED(nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay), "");
}

TEST(GetDisplayVsyncEventPreconditions, pOutEventNull_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::GetDisplayVsyncEvent(nullptr, pDisplay), "");

        nn::vi::CloseDisplay(pDisplay);
    }

    nn::vi::Finalize();
}

TEST(GetDisplayVsyncEventPreconditions, pDisplayNull_Any)
{
    nn::vi::Initialize();

    nn::os::SystemEventType vsync;
    EXPECT_DEATH_IF_SUPPORTED(nn::vi::GetDisplayVsyncEvent(&vsync, nullptr), "");

    nn::vi::Finalize();
}

TEST(GetDisplayVsyncEventPreconditions, pDisplayInvalidHandle_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::CloseDisplay(pDisplay);

        nn::os::SystemEventType vsync;
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay), "");
    }

    nn::vi::Finalize();
}
