﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testVi_Scene.h"

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include "../Common/framework/Framework.h"
#include "../Common/panel/PanelRenderer.h"
#include "../Common/framework/ImageTransferBuffer.h"
#include "../Common/framework/detail/DisplayModule.h"
#include "../Common/framework/Hid.h"

#include <nnt/vi/testVi_TestServerProxy.h>
#include <nn/vi/vi_IndirectConsumer.h>

//#define ENABLE_TRANSFER_MODE

namespace nnt{ namespace vi{

    const int CopyPosX   = 0;
    const int CopyPosY   = 360;
    const int CopyWidth  = 1280;
    const int CopyHeight = 360;

    const int32_t TestLayerSlot = 0;
    nnt::vi::TestServiceProxy g_TestProxy;
    uint64_t g_hConsumerEp = 0;

    nn::os::Tick g_PrevTick;
    std::list<float> g_FpsHistory;

    nn::util::Color4f g_Colors[8] = {
        {0, 0, 0, 1},
        {1, 0, 0, 1},
        {0, 1, 0, 1},
        {0, 0, 1, 1},
        {1, 1, 0, 1},
        {0, 1, 1, 1},
        {1, 0, 1, 1},
        {1, 1, 1, 1},
    };
    int g_ColorIndex = 0;

#ifdef ENABLE_TRANSFER_MODE
    void* g_pTransferMemory = nullptr;
    nn::os::SystemEventType g_ImageReadyEvent;
#endif

    Scene::Scene() NN_NOEXCEPT
        : m_FrameCount(0)
    {
        m_pPanelBackGround = std::make_shared<PanelText>();
        m_pPanelBackGround->SetTransparent(false);
        m_pPanelBackGround->SetColor(g_Colors[0]);
        m_pPanelBackGround->SetPosition(0, 0);
        m_pPanelBackGround->SetSize(1280, 720);

        m_pPanelFrameCount = std::make_shared<PanelText>();
        m_pPanelFrameCount->SetTransparent(true);
        m_pPanelFrameCount->SetColor(nn::util::Color4f(0, 0, 0, 0));
        m_pPanelFrameCount->SetPosition(nn::util::Vector3f(100, 100, 0));
        m_pPanelFrameCount->SetSize(nn::util::MakeFloat2(600, 200));
        m_pPanelFrameCount->SetTextColor(nn::util::Color4f(.5f, .5f, .5f, 1));
        m_pPanelFrameCount->SetTextPosition(0, 0);
        m_pPanelFrameCount->SetTextSize(30);

        m_pPanelIndirectImage = std::make_shared<PanelImage>();
        m_pPanelIndirectImage->SetTransparent(true);
        m_pPanelIndirectImage->SetPosition(CopyPosX, CopyPosY);
        m_pPanelIndirectImage->SetSize(CopyWidth, CopyHeight);
        m_pPanelIndirectImage->SetColor(nn::util::Color4f(0, 0, 0, 0));

        m_PreGetImageWaitMicroSeconds = 0;

        m_Random = std::mt19937(static_cast<std::int_fast32_t>(nn::os::GetSystemTick().GetInt64Value()));

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_TestProxy.Initialize());
        g_TestProxy.GetService()->CreateIndirectLayer(TestLayerSlot);

        g_TestProxy.GetService()->DestroyIndirectLayerConsumerEndPoint(TestLayerSlot);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_TestProxy.GetService()->CreateIndirectLayerConsumerEndPoint(&g_hConsumerEp, TestLayerSlot, nn::applet::GetAppletResourceUserId()));
        NN_LOG("ConsumerEp #%lld\n", g_hConsumerEp);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::GetIndirectImageMemoryRequirement(&m_RequiredSize, &m_RequiredAlignment, CopyWidth, CopyHeight));
#ifdef ENABLE_TRANSFER_MODE
        g_pTransferMemory = Framework::Allocate(m_RequiredSize, m_RequiredAlignment);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenIndirectImage(&g_ImageReadyEvent, g_pTransferMemory, m_RequiredSize, g_hConsumerEp, Width, Height));
#endif

        m_pDelayChecker = std::make_shared<DelayChecker>();
    }

    static std::shared_ptr<ImageTransferBuffer> GetIndirectImage(
        int x,
        int y,
        int width,
        int height,
        size_t requiredSize,
        size_t requiredAlignment,
        bool enablePrintValue
    ) NN_NOEXCEPT
    {
        auto pTrans = std::make_shared<ImageTransferBuffer>(
            width,
            height,
            nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb,
            requiredSize,
            requiredAlignment
        );

        size_t outSize = 0;
        size_t outStride = 0;
        {

#ifdef ENABLE_TRANSFER_MODE
            auto pNormalMemory = g_pTransferMemory;
            auto tick0 = nn::os::GetSystemTick();
            auto result = nn::vi::UpdateIndirectImage(&outSize, &outStride, g_hConsumerEp, 0, 0, 1, 1);
            nn::os::WaitSystemEvent(&g_ImageReadyEvent);
            auto tick1 = nn::os::GetSystemTick();
#else
            auto pNormalMemory = Framework::Allocate(requiredSize, requiredAlignment);
            NN_UTIL_SCOPE_EXIT { Framework::Free(pNormalMemory); };
            auto tick0 = nn::os::GetSystemTick();
            auto result = nn::vi::GetIndirectImage(
                &outSize,
                &outStride,
                pNormalMemory,
                requiredSize,
                g_hConsumerEp,
                width,
                height,
                static_cast<float>(x) / static_cast<float>(Scene::ImageWidth),
                static_cast<float>(y) / static_cast<float>(Scene::ImageHeight),
                static_cast<float>(width) / static_cast<float>(Scene::ImageWidth),
                static_cast<float>(height) / static_cast<float>(Scene::ImageHeight)
            );
            auto tick1 = nn::os::GetSystemTick();
#endif
            auto timeUs = (tick1 - tick0).ToTimeSpan().GetMicroSeconds();
            if(result.IsFailure())
            {
                NN_LOG("[%lldus]GetImage -> %d.%d\n", timeUs, result.GetModule(), result.GetDescription());
                return nullptr;
            }
            else
            {
                NN_LOG("[%lldus]GetImage ... ok\n", timeUs);
                if(enablePrintValue)
                {
                    // 色を確認
                    int positions[12][2] = {
                        {  50, 410 },
                        { 150, 410 },
                        { 250, 410 },
                        { 350, 410 },
                        {  50, 510 },
                        { 150, 510 },
                        { 250, 510 },
                        { 350, 510 },
                        {  50, 610 },
                        { 150, 610 },
                        { 250, 610 },
                        { 350, 610 },
                    };
                    nn::util::Color4u8Type colors[12];
                    auto data = reinterpret_cast<uint8_t*>(pNormalMemory);
                    for(int i = 0; i < 12; i++)
                    {
                        auto p = data + 4 * (positions[i][1] * width + positions[i][0]);
                        colors[i].v[0] = p[0];
                        colors[i].v[1] = p[1];
                        colors[i].v[2] = p[2];
                        colors[i].v[3] = p[3];
                    }
                    NN_LOG("   (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d)\n",
                        static_cast<int>(colors[0 + 0].v[0]), static_cast<int>(colors[0 + 0].v[1]), static_cast<int>(colors[0 + 0].v[2]), static_cast<int>(colors[0 + 0].v[3]),
                        static_cast<int>(colors[0 + 1].v[0]), static_cast<int>(colors[0 + 1].v[1]), static_cast<int>(colors[0 + 1].v[2]), static_cast<int>(colors[0 + 1].v[3]),
                        static_cast<int>(colors[0 + 2].v[0]), static_cast<int>(colors[0 + 2].v[1]), static_cast<int>(colors[0 + 2].v[2]), static_cast<int>(colors[0 + 2].v[3]),
                        static_cast<int>(colors[0 + 3].v[0]), static_cast<int>(colors[0 + 3].v[1]), static_cast<int>(colors[0 + 3].v[2]), static_cast<int>(colors[0 + 3].v[3])
                        );
                    NN_LOG("   (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d)\n",
                        static_cast<int>(colors[4 + 0].v[0]), static_cast<int>(colors[4 + 0].v[1]), static_cast<int>(colors[4 + 0].v[2]), static_cast<int>(colors[4 + 0].v[3]),
                        static_cast<int>(colors[4 + 1].v[0]), static_cast<int>(colors[4 + 1].v[1]), static_cast<int>(colors[4 + 1].v[2]), static_cast<int>(colors[4 + 1].v[3]),
                        static_cast<int>(colors[4 + 2].v[0]), static_cast<int>(colors[4 + 2].v[1]), static_cast<int>(colors[4 + 2].v[2]), static_cast<int>(colors[4 + 2].v[3]),
                        static_cast<int>(colors[4 + 3].v[0]), static_cast<int>(colors[4 + 3].v[1]), static_cast<int>(colors[4 + 3].v[2]), static_cast<int>(colors[4 + 3].v[3])
                        );
                    NN_LOG("   (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d), (% 3d,% 3d,% 3d;% 3d)\n",
                        static_cast<int>(colors[8 + 0].v[0]), static_cast<int>(colors[8 + 0].v[1]), static_cast<int>(colors[8 + 0].v[2]), static_cast<int>(colors[8 + 0].v[3]),
                        static_cast<int>(colors[8 + 1].v[0]), static_cast<int>(colors[8 + 1].v[1]), static_cast<int>(colors[8 + 1].v[2]), static_cast<int>(colors[8 + 1].v[3]),
                        static_cast<int>(colors[8 + 2].v[0]), static_cast<int>(colors[8 + 2].v[1]), static_cast<int>(colors[8 + 2].v[2]), static_cast<int>(colors[8 + 2].v[3]),
                        static_cast<int>(colors[8 + 3].v[0]), static_cast<int>(colors[8 + 3].v[1]), static_cast<int>(colors[8 + 3].v[2]), static_cast<int>(colors[8 + 3].v[3])
                        );
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }

            }
            {
                auto pMemory = pTrans->AcquireMemory();
                auto size = pTrans->GetMemorySize();
                std::memcpy(pMemory, pNormalMemory, std::min(size, requiredSize));
                pTrans->ReleaseMemory();
            }
        }

        return pTrans;
    }

    void Scene::Update()
    {

        float fps = 0;
        float fpsMin = 0;
        float fpsMax = 0;
        auto tick = nn::os::GetSystemTick();
        if(m_FrameCount > 0)
        {
            auto time = (tick - g_PrevTick).ToTimeSpan();
            fps = static_cast<float>(1000ll * 1000ll / time.GetMicroSeconds());
            g_FpsHistory.push_back(fps);
            if(g_FpsHistory.size() > 100)
            {
                g_FpsHistory.pop_front();
            }
            fpsMin = 9999;
            fpsMax = 0;
            for(auto v : g_FpsHistory)
            {
                fpsMin = std::min(fpsMin, v);
                fpsMax = std::max(fpsMax, v);
            }
        }
        g_PrevTick = tick;

        m_FrameCount++;
        bool enablePrintValue = false;

        auto buttons = Hid::GetButtonState();
        if(buttons.isDown.A)
        {
            m_pDelayChecker->Activate();
        }
        if(buttons.isPressed.X && buttons.isTriggered.R)
        {
            int delta = buttons.isPressed.Y ? 100 : 10;
            m_PreGetImageWaitMicroSeconds += delta;
            m_PreGetImageWaitMicroSeconds = std::min(m_PreGetImageWaitMicroSeconds, 10 * 1000 * 1000);
        }
        if(buttons.isPressed.X && buttons.isTriggered.L)
        {
            int delta = buttons.isPressed.Y ? 100 : 10;
            m_PreGetImageWaitMicroSeconds -= delta;
            m_PreGetImageWaitMicroSeconds = std::max(m_PreGetImageWaitMicroSeconds, 0);
        }
        if(buttons.isDown.Y)
        {
            g_ColorIndex++;
            g_ColorIndex %= 8;
            m_pPanelBackGround->SetColor(g_Colors[g_ColorIndex]);
        }
        if(buttons.isPressed.B)
        {
            enablePrintValue = true;
        }

        char buffer[256] = {};
        nn::util::SNPrintf(buffer, sizeof(buffer),
            "IndirectConsumer\n"
            "Frame: %d\n"
            "%.1ffps(%.1ffps-%.1ffps)\n"
            "Wait %dus\n",
            m_FrameCount,
            fps, fpsMin, fpsMax,
            m_PreGetImageWaitMicroSeconds
        );
        m_pPanelFrameCount->SetText(buffer);
        m_pPanelFrameCount->Update();

        UpdateAllSquareObjects();
        if(m_FrameCount % 1 == 0)
        {
            AddSquareObject();
        }

        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(m_PreGetImageWaitMicroSeconds));
        auto pTrans = GetIndirectImage(CopyPosX, CopyPosY, CopyWidth, CopyHeight, m_RequiredSize, m_RequiredAlignment, enablePrintValue);
        if(pTrans)
        {
            m_pPanelIndirectImage->SetTransferBuffer(pTrans);
            m_pPanelIndirectImage->Update();
        }

        m_pDelayChecker->Update();
    }

    void Scene::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int resourceLockIndex) NN_NOEXCEPT
    {
        PanelRenderer::Render(pCommandBuffer, m_pPanelBackGround, Framework::GetScreenRectangle(), resourceLockIndex);
        for(auto& e : m_SquareObjectList)
        {
            PanelRenderer::Render(pCommandBuffer, e.pPanel, Framework::GetScreenRectangle(), resourceLockIndex);
        }
        PanelRenderer::Render(pCommandBuffer, m_pPanelFrameCount, Framework::GetScreenRectangle(), resourceLockIndex);

        PanelRenderer::Render(pCommandBuffer, m_pPanelIndirectImage, Framework::GetScreenRectangle(), resourceLockIndex);
        PanelRenderer::Render(pCommandBuffer, m_pDelayChecker->GetPanel(), Framework::GetScreenRectangle(), resourceLockIndex);
    }

    void Scene::AddSquareObject() NN_NOEXCEPT
    {
        auto sizeDist = std::uniform_real_distribution<float>(10, 50);
        float w = sizeDist(m_Random);
        float h = sizeDist(m_Random);
        float rX = ImageWidth + w;
        float rY = ImageHeight + h;

        auto spdDist = std::uniform_real_distribution<float>(1, 10);
        auto radDist = std::uniform_real_distribution<float>(0, 2 * M_PI);
        auto startDist = std::uniform_real_distribution<float>(0, 2 * (rX + rY));
        auto colorDist = std::uniform_real_distribution<float>(0.5f, 1);



        float spd = spdDist(m_Random);
        float dir = radDist(m_Random);
        float start = startDist(m_Random);


        float x = 0;
        float y = 0;
        float vx = spd * std::cos(dir);
        float vy = spd * std::sin(dir);
        if(start < rX) // 上辺
        {
            x = start;
            y = 0;
            if(vy < 0)
            {
                vy = -vy;
            }
        }
        else if(start < rX + rY) // 右辺
        {
            x = rX;
            y = start - rX;
            if(vx > 0)
            {
                vx = -vx;
            }
        }
        else if(start < 2 * rX + rY) // 下辺
        {
            x = start - (rX + rY);
            y = rY;
            if(vy > 0)
            {
                vy = -vy;
            }
        }
        else // 左辺
        {
            x = 0;
            y = start - (2 * rX + rY);
            if(vx < 0)
            {
                vx = -vx;
            }
        }

        nn::util::Vector3f radi(w / 2, h / 2, 0);
        nn::util::Vector3f pos(x - w / 2, y - h / 2, 0);
        nn::util::Vector3f vel(vx, vy, 0);
        nn::util::Color4f color(colorDist(m_Random), colorDist(m_Random), colorDist(m_Random));

        SquareObject obj;
        obj.radius   = radi;
        obj.position = pos;
        obj.velocity = vel;
        obj.pPanel = std::make_shared<PanelText>();
        obj.pPanel->SetColor(color);
        obj.pPanel->SetPosition(pos - radi);
        obj.pPanel->SetSize(nn::util::MakeFloat2(w, h));
        obj.pPanel->SetVisible(true);
        m_SquareObjectList.push_back(std::move(obj));
    }

    void Scene::UpdateAllSquareObjects() NN_NOEXCEPT
    {
        for(auto& obj : m_SquareObjectList)
        {
            obj.position += obj.velocity;
            obj.pPanel->SetPosition(obj.position - obj.radius);
        }

        m_SquareObjectList.remove_if(
            [&](const SquareObject& obj)->bool
            {
                float x = obj.position.GetX();
                float y = obj.position.GetY();
                if(x < - obj.radius.GetX())
                {
                    return true;
                }
                if(x > ImageWidth + obj.radius.GetX())
                {
                    return true;
                }
                if(y < -obj.radius.GetY())
                {
                    return true;
                }
                if(y > ImageHeight + obj.radius.GetY())
                {
                    return true;
                }
                return false;
            }
        );

    }

}}
