﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>
#include <nv/nv_ServiceName.h>

#include "testVi_MemoryManagement.h"
#include "../Common/framework/Framework.h"
#include "../Common/panel/PanelRenderer.h"
#include "../Common/panel/PanelPreparationTaskProcessor.h"
#include "../Common/framework/Hid.h"
#include "testVi_Scene.h"

namespace {
    NN_ALIGNAS(4096) char g_FileSystemCache[16 * 1024];

    framework::FrameworkParameter GetFrameworkParameter() NN_NOEXCEPT
    {
        framework::FrameworkParameter param = {};

        param.width  = 1280;
        param.height = 720;
        param.rootCommandBufferCount = 2;
        param.frameBufferCount       = 2;

        param.firmwareMemorySize     = 3 * 256 * 1024;
        param.rootCommandMemorySize  =      32 * 1024;
        param.rootControlMemorySize  =       2 * 1024;

        param.memoryPool.texturePoolSize        = 16 * 1024 * 1024;
        param.memoryPool.commandBufferPoolSize  = param.rootCommandBufferCount * param.rootCommandMemorySize;
        param.memoryPool.constantBufferPoolSize =       512 * 1024;
        param.memoryPool.shaderPoolSize         =        16 * 1024;
        param.memoryPool.dataPoolSize           = 16 * 1024 * 1024;

        param.poolTextureWidth  = 1280;
        param.poolTextureHeight = 720;
        param.poolTextureCount  = 2;

        param.poolTextWriterCharCountMax = 128;
        param.poolTextWriterCount        =  10;

        return param;
    }
}


extern "C" void nninitStartup() NN_NOEXCEPT
{
    InitializeAllocator();
}

void ProgramMain(void*) NN_NOEXCEPT
{
    nv::SetGraphicsServiceName("nvdrv:t");

    // Rom をマウント
    nn::fs::MountRom("rom", g_FileSystemCache, sizeof(g_FileSystemCache));
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount("rom"); };

    Framework::Initialize(GetFrameworkParameter());
    NN_UTIL_SCOPE_EXIT { Framework::Finalize(); };
    PanelRenderer::Initialize();
    NN_UTIL_SCOPE_EXIT { PanelRenderer::Finalize(); };

    {
        HidButtonHandleParameter param = {};
        param.repeatTriggerFrameBeginCount = 20;
        param.repeatTriggerFrameInterval   = 3;
        Hid::SetParameter(param);
    }

    {
        Framework::BeginFrameCommand(0);
        auto pCommandBuffer = Framework::GetCommandBuffer(0);
        Framework::MakeInitializeCommand(pCommandBuffer);
        Framework::EndFrameCommand(0);
        Framework::QueueSubmitCommand(pCommandBuffer);
        Framework::QueueFlush();
        Framework::WaitQueueFinish();
    }

    nn::gfx::Fence windowFence;
    nn::gfx::Fence commandFence[2];
    {
        nn::gfx::FenceInfo info;
        info.SetDefault();
        windowFence.Initialize(Framework::GetDevice(), info);
        commandFence[0].Initialize(Framework::GetDevice(), info);
        commandFence[1].Initialize(Framework::GetDevice(), info);
    }
    NN_UTIL_SCOPE_EXIT {
        windowFence.Finalize(Framework::GetDevice());
        commandFence[0].Finalize(Framework::GetDevice());
        commandFence[1].Finalize(Framework::GetDevice());
    };

    // シーンを準備
    std::shared_ptr<nnt::vi::Scene> pScene = std::make_shared<nnt::vi::Scene>();

    int commandIndex = 0;
    uint64_t frameCount = 0;
    for(;;)
    {
        frameCount++;
        commandIndex = 1 - commandIndex; // 0, 1, 0, 1, ...

        nn::util::Color4f clearColor(0, 0, 0, 0);

        Hid::Update();
        pScene->Update();

        Framework::WindowAcquireTexture(&windowFence);
        Framework::QueueWaitFence(&windowFence);

        Framework::WaitFence(&commandFence[commandIndex]);      // 前回のコマンドが終わるのを待つ
        Framework::ReleaseFrameCommandResource(commandIndex);   // 終わったコマンドのリソースを解放

        Framework::BeginFrameCommand(commandIndex);
        auto pCommandBuffer = Framework::GetCommandBuffer(commandIndex);

        // 描画リソースの作成コマンドを処理
        panel::PanelPreparationTaskProcessor::Process(Framework::GetPanelPreparationTaskQueue(), pCommandBuffer, commandIndex);

        // RenderTarget をフレームバッファにして描画
        Framework::MakeSetFrameBufferRenderTargetCommand(pCommandBuffer);
        Framework::MakeSetViewportScissorCommand(pCommandBuffer, Framework::GetScreenRectangle());
        Framework::MakeClearCommand(pCommandBuffer, clearColor);
        Framework::DiscardPreviousRendererdRegion();

        pScene->Draw(pCommandBuffer, commandIndex);

        Framework::EndFrameCommand(commandIndex);
        Framework::QueueSubmitCommand(pCommandBuffer);
        Framework::QueueSignalFence(&commandFence[commandIndex]);
        Framework::QueuePresentTexture();
        Framework::QueueFlush();

        Framework::WaitFence(&windowFence);
    }

}
