﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "PanelRenderer.h"
#include "PanelRendererImpl.h"

#include <nn/nn_Log.h>
#include <nn/gfx/util/gfx_ObjectHolder.h>
#include "../framework/Framework.h"
#include "../framework/ResShader.h"
#include "../framework/TexturePool.h"

namespace panel{
    PanelRendererImpl g_RendererImpl;
}


void PanelRenderer::Initialize() NN_NOEXCEPT
{
    auto& impl = panel::g_RendererImpl;
    impl.panelShader.Initialize("rom:/PanelShader.bnsh");
    {
        nn::gfx::RasterizerStateInfo info;
        info.SetDefault();
        info.SetCullMode(nn::gfx::CullMode_None);
        info.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
        info.SetScissorEnabled(true);
        info.SetDepthClipEnabled(false);
        impl.rasterizerState.Initialize(Framework::GetDevice(), info);
    }
    {
        nn::gfx::BlendStateInfo info;
        info.SetDefault();
        nn::gfx::BlendTargetStateInfo targetInfo;
        targetInfo.SetDefault();
        targetInfo.SetBlendEnabled(true);
        targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
        targetInfo.SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);
        targetInfo.SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_One);
        targetInfo.SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
        info.SetBlendTargetStateInfoArray(&targetInfo, 1);
        impl.blendState.Initialize(Framework::GetDevice(), info);
    }
    {
        nn::gfx::DepthStencilStateInfo info;
        info.SetDefault();
        info.SetDepthTestEnabled(false);
        info.SetDepthWriteEnabled(false);
        impl.depthStencilState.Initialize(Framework::GetDevice(), info);
    }
}

void PanelRenderer::Finalize() NN_NOEXCEPT
{
    auto& impl = panel::g_RendererImpl;
    impl.panelShader.Finalize();
    impl.depthStencilState.Finalize(Framework::GetDevice());
    impl.blendState.Finalize(Framework::GetDevice());
    impl.rasterizerState.Finalize(Framework::GetDevice());
}

//----------------------------

void PanelRenderer::Render(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<IPanel>& panel,
    const framework::Rectangle& cropRect,
    int resourceLockIndex
) NN_NOEXCEPT
{
    RenderContext ctxt;
    ctxt.cropRect = cropRect;
    ctxt.isRegisterRegionRequired = true;
    ctxt.resourceLockIndex = resourceLockIndex;
    RenderImpl(pCommandBuffer, panel, ctxt);
}

void PanelRenderer::RenderImpl(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<IPanel>& panel,
    const RenderContext& ctxt
) NN_NOEXCEPT
{
    if(auto pContainer = PanelContainer::IsContainer(panel))
    {
        RenderContainer(pCommandBuffer, pContainer, ctxt);
    }
    else if(auto pImage = PanelImage::IsImage(panel))
    {
        RenderImage(pCommandBuffer, pImage, ctxt);
    }
    else if(auto pText = PanelText::IsText(panel))
    {
        RenderText(pCommandBuffer, pText, ctxt);
    }
    else
    {
        RenderMisc(pCommandBuffer, panel, ctxt);
    }
}

//----------------------------

void PanelRenderer::RenderMisc(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<IPanel>& panel,
    const RenderContext& ctxt
) NN_NOEXCEPT
{
    if(!panel->IsVisible())
    {
        return;
    }
    DrawSolid(pCommandBuffer, panel, ctxt);
}

void PanelRenderer::RenderContainer(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<PanelContainer>& panel,
    const RenderContext& ctxt
) NN_NOEXCEPT
{
    if(!panel->IsVisible())
    {
        return;
    }
    bool isRendererd = DrawSolid(pCommandBuffer, panel, ctxt);

    RenderContext childCtxt = ctxt;
    if(panel->IsCroppingEnabled())
    {
        childCtxt.cropRect = ctxt.cropRect.Product(panel->GetGlobalRectangle());
        if(isRendererd)
        {
            childCtxt.isRegisterRegionRequired = false;
        }
    }

    int n = panel->GetChildCount();
    for(int i = 0; i < n; i++)
    {
        RenderImpl(pCommandBuffer, panel->GetChild(i), childCtxt);
    }
}

void PanelRenderer::RenderImage(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<PanelImage>& panel,
    const RenderContext& ctxt
) NN_NOEXCEPT
{
    if(!panel->IsVisible())
    {
        return;
    }

    if(auto texture = panel->GetTexture())
    {
        DrawImplShader(pCommandBuffer, panel, texture, ctxt);
    }
    else
    {
        DrawSolid(pCommandBuffer, panel, ctxt);
    }

}

void PanelRenderer::RenderText(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<PanelText>& panel,
    const RenderContext& ctxt
    ) NN_NOEXCEPT
{
    if(!panel->IsVisible())
    {
        return;
    }

    bool isRendererd = DrawSolid(pCommandBuffer, panel, ctxt);

    RenderContext textCtxt = ctxt;
    if(isRendererd)
    {
        textCtxt.isRegisterRegionRequired = false;
    }

    if(auto pTextWriter = panel->GetTextWriter())
    {
        DrawImplTextWriter(pCommandBuffer, panel, pTextWriter, textCtxt);
    }
}

//----------------------------

bool PanelRenderer::DrawSolid(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<IPanel>& panel,
    const RenderContext& ctxt
    ) NN_NOEXCEPT
{
    if(panel->IsTransparent())
    {
        return DrawImplShader(pCommandBuffer, panel, nullptr, ctxt);
    }
    else
    {
        return DrawImplClearColor(pCommandBuffer, panel, ctxt);
    }
}
