﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "IPanel.h"

template<typename TBaseType>
class PanelBase
    : public TBaseType
{
    NN_DISALLOW_COPY(PanelBase);
public:
    explicit PanelBase(PanelTypeType type)
        : m_Type(type)
        , m_IsVisible(true)
        , m_IsTransparent(false)
        , m_IsFlipYEnabled(false)
    {
        m_Position = nn::util::Vector3f(0, 0, 0);
        m_Size = nn::util::MakeFloat2(0, 0);
        m_Color = nn::util::Color4f(0, 0, 0, 0);
    }

    virtual PanelTypeType GetType() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Type;
    }

    virtual nn::util::Vector3f GetPosition() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Position;
    }

    virtual nn::util::Float2 GetSize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Size;
    }

    virtual nn::util::Color4f GetColor() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Color;
    }

    virtual bool IsVisible() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsVisible;
    }

    virtual bool IsTransparent() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsTransparent;
    }

    virtual bool GetFlipYEnabled() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsFlipYEnabled;
    }

    virtual nn::util::Vector3f GetGlobalPosition() const NN_NOEXCEPT NN_OVERRIDE
    {
        if(auto p = m_Parent.lock())
        {
            return p->GetGlobalPosition() + m_Position;
        }
        else
        {
            return m_Position;
        }
    }

    virtual framework::Rectangle GetGlobalRectangle() const NN_NOEXCEPT NN_OVERRIDE
    {
        auto pos = GetGlobalPosition();
        return framework::Rectangle(
            static_cast<int>(pos.GetX()),
            static_cast<int>(pos.GetY()),
            static_cast<int>(m_Size.x),
            static_cast<int>(m_Size.y)
        );
    }


    void SetPosition(const nn::util::Vector3f& value) NN_NOEXCEPT
    {
        m_Position = value;
    }

    void SetPosition(float x, float y, float z = 0) NN_NOEXCEPT
    {
        SetPosition(nn::util::Vector3f(x, y, z));
    }

    void SetPosition(int x, int y, int z = 0) NN_NOEXCEPT
    {
        SetPosition(nn::util::Vector3f(static_cast<float>(x), static_cast<float>(y), static_cast<float>(z)));
    }

    void SetSize(const nn::util::Float2& value) NN_NOEXCEPT
    {
        m_Size = value;
    }

    void SetSize(float w, float h) NN_NOEXCEPT
    {
        SetSize(nn::util::MakeFloat2(w, h));
    }

    void SetSize(int w, int h) NN_NOEXCEPT
    {
        SetSize(nn::util::MakeFloat2(static_cast<float>(w), static_cast<float>(h)));
    }

    void SetColor(const nn::util::Color4f& value) NN_NOEXCEPT
    {
        m_Color = value;
    }

    void SetVisible(bool value) NN_NOEXCEPT
    {
        m_IsVisible = value;
    }

    void SetTransparent(bool value) NN_NOEXCEPT
    {
        m_IsTransparent = value;
    }

    void SetFlipYEnabled(bool value) NN_NOEXCEPT
    {
        m_IsFlipYEnabled = value;
    }

protected:
    virtual std::shared_ptr<IPanel> GetParent() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Parent.lock();
    }

    virtual void SetParent(const std::shared_ptr<IPanel>& value) NN_NOEXCEPT NN_OVERRIDE
    {
        m_Parent = value;
    }

private:
    PanelTypeType         m_Type;
    nn::util::Vector3f    m_Position;
    nn::util::Float2      m_Size;
    nn::util::Color4f     m_Color;
    bool                  m_IsVisible;
    bool                  m_IsTransparent;
    bool                  m_IsFlipYEnabled;
    std::weak_ptr<IPanel> m_Parent;
};

