﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfx_GraphicsFramework.h"
#include <nn/nn_SdkAssert.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
    #if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
        #include <nvnTool/nvnTool_GlslcInterface.h>
    #endif
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
#include <nv/nv_MemoryManagement.h>
#include <cstdlib>
#endif

#include "detail/DisplayModule.h"


namespace nns {
namespace gfx {


GraphicsFramework::GraphicsFramework() :
m_pDisplay(nullptr),
m_pLayer(nullptr),
m_pCommandBuffer(nullptr),
m_pShaderScratchMemory(nullptr),
m_pColorTargetView(nullptr),
//m_pDepthStencilBuffer(nullptr),
//m_pDepthStencilView(nullptr),
m_AllocateFunction(nullptr),
m_FreeFunction(nullptr),
m_pAllocateFunctionUserData(nullptr),
m_IsInitialized(false)
{}

//------------------------------------------------------------------------------
// 初期化・終了
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeGraphicsSystem(
    size_t graphicsSystemMemorySize,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    nn::FreeFunctionWithUserData pFreeFunction,
    ReallocateFunctionWithUserData pReallocateFunction,
    void* pAllocateFunctionUserData
) NN_NOEXCEPT
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        if ((pAllocateFunction != nullptr) && (pFreeFunction != nullptr) && (pReallocateFunction != nullptr)) {
            void* memory = pAllocateFunction(graphicsSystemMemorySize, 1, pAllocateFunctionUserData);
            nv::SetGraphicsAllocator(pAllocateFunction, pFreeFunction, pReallocateFunction, pAllocateFunctionUserData);
            nv::SetGraphicsDevtoolsAllocator(pAllocateFunction, pFreeFunction, pReallocateFunction, pAllocateFunctionUserData);
            nv::InitializeGraphics(memory, graphicsSystemMemorySize);
        }
        else {
            void* memory = DefaultAllocateFunction(graphicsSystemMemorySize, 1, nullptr);
            nv::SetGraphicsAllocator(DefaultAllocateFunction, DefaultFreeFunction, DefaultReallocateFunction, nullptr);
            nv::SetGraphicsDevtoolsAllocator(DefaultAllocateFunction, DefaultFreeFunction, DefaultReallocateFunction, nullptr);
            nv::InitializeGraphics(memory, graphicsSystemMemorySize);
        }
    }
#else
    NN_UNUSED(graphicsSystemMemorySize);
    NN_UNUSED(pAllocateFunction);
    NN_UNUSED(pFreeFunction);
    NN_UNUSED(pReallocateFunction);
    NN_UNUSED(pAllocateFunctionUserData);
#endif

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
    // GLSLC のためのメモリアロケータを設定します。
    if ((pAllocateFunction != nullptr) && (pFreeFunction != nullptr) && (pReallocateFunction != nullptr)) {
        glslcSetAllocator(pAllocateFunction, pFreeFunction, pReallocateFunction, pAllocateFunctionUserData);
    }
    else {
        glslcSetAllocator(DefaultAllocateFunction, DefaultFreeFunction, DefaultReallocateFunction, nullptr);
    }
#endif
}

void GraphicsFramework::InitializeGraphicsSystem(
    size_t graphicsSystemMemorySize
) NN_NOEXCEPT
{
    return InitializeGraphicsSystem(
        graphicsSystemMemorySize,
        DefaultAllocateFunction,
        DefaultFreeFunction,
        DefaultReallocateFunction,
        nullptr
    );
}

void GraphicsFramework::Initialize(
    const FrameworkInfo& info,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    nn::FreeFunctionWithUserData pFreeFunction,
    void* pAllocateFunctionUserData
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());

    if ((pAllocateFunction != nullptr) && (pFreeFunction != nullptr)) {
        m_AllocateFunction = pAllocateFunction;
        m_FreeFunction = pFreeFunction;
        m_pAllocateFunctionUserData = pAllocateFunctionUserData;
    }
    else {
        m_AllocateFunction = DefaultAllocateFunction;
        m_FreeFunction = DefaultFreeFunction;
        m_pAllocateFunctionUserData = nullptr;
    }

    m_DisplayWidth = info.GetDisplayWidth();
    m_DisplayHeight = info.GetDisplayHeight();
    m_BufferCount = info.GetBufferCount();

    // VI
    framework::detail::g_DisplayModule.Initialize();
    m_pLayer = framework::detail::g_DisplayModule.GetLayer();

    // GFX
    {
        // GFX 初期化
        nn::gfx::Initialize();

        // Device
        {
            nn::gfx::Device::InfoType deviceInfo;
            deviceInfo.SetDefault();
            deviceInfo.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
            deviceInfo.SetDebugMode(info.GetDebugMode());
            m_Device.Initialize(deviceInfo);
            nvnDeviceSetWindowOriginMode(m_Device.ToData()->pNvnDevice, NVNwindowOriginMode::NVN_WINDOW_ORIGIN_MODE_UPPER_LEFT);
        }

        // Queue
        {
            nn::gfx::Queue::InfoType queueInfo;
            queueInfo.SetDefault();
            queueInfo.SetCapability(nn::gfx::QueueCapability_Graphics
                | nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy);
            m_Queue.Initialize(&m_Device, queueInfo);
        }

        // Fence
        {
            nn::gfx::Fence::InfoType fenceInfo;
            fenceInfo.SetDefault();
            m_pFence.Initialize(&m_Device, fenceInfo);
        }

        // MemoryPool
        {
            nn::gfx::MemoryPool::InfoType memorypoolInfo;
            int memoryPoolProperty[MemoryPoolType_End] = {
                /* CommandBuffer */     nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached,
                /* RenderTarget */      nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible,
                /* ConstantBuffer */    nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
                /* Shader */            nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_ShaderCode,
                /* Data */              nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached,
                /* Others */            nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible | nn::gfx::MemoryPoolProperty_ShaderCode,
            };

            for (int i = 0; i < MemoryPoolType_End; i++)
            {
                memorypoolInfo.SetDefault();
                memorypoolInfo.SetMemoryPoolProperty(memoryPoolProperty[i]);

                size_t size = info.GetMemoryPoolSize(static_cast<MemoryPoolType>(i));
                if (size == 0) {
                    m_MemoryPool[i].size = 0;
                    continue;
                }
                size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, memorypoolInfo);
                void* pAlignedMemory = m_AllocateFunction(size, alignment, m_pAllocateFunctionUserData);

                memorypoolInfo.SetPoolMemory(pAlignedMemory, nn::util::align_down(size,
                    nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, memorypoolInfo)));
                m_MemoryPool[i].object.Initialize(&m_Device, memorypoolInfo);

                m_MemoryPool[i].allocator.Initialize(
                    MemoryPoolAllocatorMalloc,
                    reinterpret_cast<void*>(this),
                    MemoryPoolAllocatorFree,
                    reinterpret_cast<void*>(this),
                    &m_MemoryPool[i].object,
                    0,
                    size,
                    nn::gfx::util::MemoryPoolAllocator::AlignmentMax,
                    true
                );
                NN_SDK_ASSERT(m_MemoryPool[i].allocator.IsInitialized());

                m_MemoryPool[i].pMemory = pAlignedMemory;
                m_MemoryPool[i].size = size;
            }
        }

        // DescriptorPool
        {
            int descriptorBaseIndex[nn::gfx::DescriptorPoolType_End];
            descriptorBaseIndex[nn::gfx::DescriptorPoolType_BufferView] = 0;
            descriptorBaseIndex[nn::gfx::DescriptorPoolType_TextureView] = 0;
            descriptorBaseIndex[nn::gfx::DescriptorPoolType_Sampler] = 0;
#if NN_GFX_IS_TARGET_NVN
            nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(m_Device);
            nvnDeviceGetInteger(deviceData.pNvnDevice,
                NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &descriptorBaseIndex[nn::gfx::DescriptorPoolType_TextureView]);
            nvnDeviceGetInteger(deviceData.pNvnDevice,
                NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &descriptorBaseIndex[nn::gfx::DescriptorPoolType_Sampler]);
#endif
            nn::gfx::DescriptorPool::InfoType descriptorpoolInfo;

            for (int i = 0; i < nn::gfx::DescriptorPoolType_End; i++)
            {
                nn::gfx::DescriptorPoolType descriptorpoolType = static_cast<nn::gfx::DescriptorPoolType>(i);
                m_DescriptorPoolSlotCount[i] = info.GetDescriptorPoolSlotCount(descriptorpoolType);
                if (m_DescriptorPoolSlotCount[i] > 0) {
                    NN_SDK_ASSERT_GREATER(m_DescriptorPoolSlotCount[i], descriptorBaseIndex[i]);

                    descriptorpoolInfo.SetDefault();
                    descriptorpoolInfo.SetDescriptorPoolType(descriptorpoolType);
                    descriptorpoolInfo.SetSlotCount(m_DescriptorPoolSlotCount[i]);

                    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, descriptorpoolInfo);
                    MemoryPoolType memorypoolType = MemoryPoolType_ConstantBuffer;
                    size_t alignment = nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, descriptorpoolInfo);
                    ptrdiff_t offset = m_MemoryPool[memorypoolType].allocator.Allocate(size, alignment);
                    NN_SDK_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);

                    m_DescriptorPool[i].Initialize(&m_Device, descriptorpoolInfo, &m_MemoryPool[memorypoolType].object, offset, size);

                    m_DescriptorPoolAllocator[i].Initialize(
                        DescriptorPoolAllocatorMalloc,
                        reinterpret_cast<void*>(this),
                        DescriptorPoolAllocatorFree,
                        reinterpret_cast<void*>(this),
                        &m_DescriptorPool[i],
                        descriptorBaseIndex[i],
                        m_DescriptorPoolSlotCount[i] - descriptorBaseIndex[i],
                        true
                    );
                    NN_SDK_ASSERT(m_DescriptorPoolAllocator[i].IsInitialized());
                }
            }
        }

        // RootCommandBuffer
        {
            int count = m_BufferCount;
            m_pCommandBuffer = reinterpret_cast<GraphicsFramework::CommandBuffer*>(m_AllocateFunction(sizeof(GraphicsFramework::CommandBuffer) * count, NN_ALIGNOF(GraphicsFramework::CommandBuffer), m_pAllocateFunctionUserData));
            for (int i = 0; i < count; i++) {
                new(&m_pCommandBuffer[i])GraphicsFramework::CommandBuffer;
                m_pCommandBuffer[i].commandMemorySize = info.GetRootCommandBufferCommandMemorySize();
                m_pCommandBuffer[i].controlMemorySize = info.GetRootCommandBufferControlMemorySize();
            }
            m_ShaderScratchMemorySize = 0;

            // CommandBuffer
            for (int i = 0; i < count; i++) {
                {
                    nn::gfx::CommandBuffer::InfoType commandbufferInfo;
                    commandbufferInfo.SetDefault();
                    commandbufferInfo.SetQueueCapability(nn::gfx::QueueCapability_Graphics
                        | nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy);
                    commandbufferInfo.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);

                    m_pCommandBuffer[i].object.Initialize(&m_Device, commandbufferInfo);
                }

                // CommandMemory
                {
                    MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
                    size_t alignment = nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device);
                    size_t size = m_pCommandBuffer[i].commandMemorySize;
                    m_pCommandBuffer[i].commandMemoryOffset = m_MemoryPool[memorypoolType].allocator.Allocate(size, alignment);
                    NN_SDK_ASSERT(m_pCommandBuffer[i].commandMemoryOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                }

                // ControlMemory
                {
                    size_t alignment = nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device);
                    size_t size = m_pCommandBuffer[i].controlMemorySize;
                    m_pCommandBuffer[i].pControlMemory = m_AllocateFunction(size, alignment, m_pAllocateFunctionUserData);
                }
            }
        }

        // ShaderScratchMemory
        {
            int count = m_BufferCount;
            m_pShaderScratchMemory = reinterpret_cast<ShaderScratchMemoryInfo*>(m_AllocateFunction(sizeof(ShaderScratchMemoryInfo) * count, NN_ALIGNOF(ShaderScratchMemoryInfo), m_pAllocateFunctionUserData));
            for (int i = 0; i < count; i++) {
                new(&m_pShaderScratchMemory[i])ShaderScratchMemoryInfo;
                m_pShaderScratchMemory[i].offset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
                m_pShaderScratchMemory[i].size = 0;
            }
        }

        // SwapChain
        {
            nn::gfx::SwapChain::InfoType swapchainInfo;
            swapchainInfo.SetDefault();
            swapchainInfo.SetLayer(m_pLayer);
            swapchainInfo.SetWidth(m_DisplayWidth);
            swapchainInfo.SetHeight(m_DisplayHeight);
            swapchainInfo.SetFormat(info.GetSwapChainFormat());
            swapchainInfo.SetBufferCount(info.GetSwapChainBufferCount());
            if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
            {
                MemoryPoolType type = MemoryPoolType_RenderTarget;
                size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, swapchainInfo);
                size_t alignment = nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, swapchainInfo);
                ptrdiff_t offset = m_MemoryPool[type].allocator.Allocate(size, alignment);
                NN_SDK_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                m_SwapChain.Initialize(&m_Device, swapchainInfo, &m_MemoryPool[type].object, offset, size);
            }
            else
            {
                m_SwapChain.Initialize(&m_Device, swapchainInfo, nullptr, 0, 0);
            }
        }

        // NOTE: DevOverlayDisp ではメモリ節約のため直接 SwapChain のテクスチャに書き込む

        // ColorTargetView
        {
            nn::gfx::ColorTargetView::InfoType colorTargetViewInfo;
            colorTargetViewInfo.SetDefault();
            colorTargetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
            colorTargetViewInfo.SetImageFormat(info.GetSwapChainFormat());

            int count = m_BufferCount;
            m_pColorTargetView = reinterpret_cast<nn::gfx::ColorTargetView**>(
                m_AllocateFunction(sizeof(nn::gfx::ColorTargetView*) * count, NN_ALIGNOF(nn::gfx::ColorTargetView*), m_pAllocateFunctionUserData));
            m_SwapChain.GetScanBufferViews(m_pColorTargetView, count);
        }

        // NOTE: 現状、 DevOverlayDisp ではデプスバッファを使わない
#if 0
        // DepthStencilBuffer
        {
            nn::gfx::Texture::InfoType textureInfo;
            textureInfo.SetDefault();
            textureInfo.SetWidth(m_DisplayWidth);
            textureInfo.SetHeight(m_DisplayHeight);
            textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
            textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
            textureInfo.SetImageFormat(info.GetDepthStencilBufferFormat());
            textureInfo.SetMipCount(1);
            textureInfo.SetDepth(1);

            MemoryPoolType type = MemoryPoolType_RenderTarget;
            size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, textureInfo);
            size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, textureInfo);

            int count = 1; //m_BufferCount;
            m_pDepthStencilBuffer = reinterpret_cast<nn::gfx::Texture*>(m_AllocateFunction(sizeof(nn::gfx::Texture) * count, NN_ALIGNOF(nn::gfx::Texture), m_pAllocateFunctionUserData));
            for (int i = 0; i < count; i++) {
                new(&m_pDepthStencilBuffer[i])nn::gfx::Texture;
                if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
                {
                    ptrdiff_t offset = m_MemoryPool[type].allocator.Allocate(size, alignment);
                    NN_SDK_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                    m_pDepthStencilBuffer[i].Initialize(&m_Device, textureInfo, &m_MemoryPool[type].object, offset, size);
                }
                else
                {
                    m_pDepthStencilBuffer[i].Initialize(&m_Device, textureInfo, nullptr, 0, 0);
                }
            }
        }

        // DepthStencilView
        {
            nn::gfx::DepthStencilView::InfoType depthStencilViewInfo;
            depthStencilViewInfo.SetDefault();
            depthStencilViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);

            int count = 1; //m_BufferCount;
            m_pDepthStencilView = reinterpret_cast<nn::gfx::DepthStencilView*>(m_AllocateFunction(sizeof(nn::gfx::DepthStencilView) * count, NN_ALIGNOF(nn::gfx::DepthStencilView), m_pAllocateFunctionUserData));
            for (int i = 0; i < count; i++) {
                new(&m_pDepthStencilView[i])nn::gfx::DepthStencilView;
                depthStencilViewInfo.SetTexturePtr(&m_pDepthStencilBuffer[i]);
                m_pDepthStencilView[i].Initialize(&m_Device, depthStencilViewInfo);
            }
        }
#endif
    }

    m_IsInitialized = true;

    // その他組み込み GFX オブジェクト
    {
        // ViewportScissorState
        {
            nn::gfx::ViewportStateInfo viewportInfo;
            viewportInfo.SetDefault();
            viewportInfo.SetWidth(static_cast< float >(m_DisplayWidth));
            viewportInfo.SetHeight(static_cast< float >(m_DisplayHeight));

            nn::gfx::ScissorStateInfo scissorInfo;
            scissorInfo.SetDefault();
            scissorInfo.SetWidth(m_DisplayWidth);
            scissorInfo.SetHeight(m_DisplayHeight);

            nn::gfx::ViewportScissorState::InfoType viewportScissorInfo;
            viewportScissorInfo.SetDefault();
            viewportScissorInfo.SetScissorEnabled(true);
            viewportScissorInfo.SetViewportStateInfoArray(&viewportInfo, 1);
            viewportScissorInfo.SetScissorStateInfoArray(&scissorInfo, 1);

            NN_SDK_ASSERT(nn::gfx::ViewportScissorState::GetRequiredMemorySize(viewportScissorInfo) == 0);
            m_ViewportScissorState.Initialize(GetDevice(), viewportScissorInfo);
        }

        // VertexState
        {
            nn::gfx::VertexState::InfoType vertexInfo;

            // VertexStateType_Float3_Float2
            vertexInfo.SetDefault();
            ptrdiff_t stride = sizeof(float) * 5;
            nn::gfx::VertexAttributeStateInfo attribs[2];
            {
                attribs[0].SetDefault();
                //attribs[0].SetNamePtr("i_Position");
                attribs[0].SetBufferIndex(0);
                attribs[0].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
                attribs[0].SetOffset(0);
                attribs[0].SetShaderSlot(0);
            }
            {
                attribs[1].SetDefault();
                //attribs[1].SetNamePtr("i_TexCoord");
                attribs[1].SetBufferIndex(0);
                attribs[1].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
                attribs[1].SetOffset(sizeof(float) * 3);
                attribs[1].SetShaderSlot(1);
            }
            nn::gfx::VertexBufferStateInfo buffer;
            {
                buffer.SetDefault();
                buffer.SetStride(stride);
            }
            vertexInfo.SetVertexAttributeStateInfoArray(attribs, 2);
            vertexInfo.SetVertexBufferStateInfoArray(&buffer, 1);

            InitializeVertexState(&m_VertexState[VertexStateType_Float3_Float2], vertexInfo);
        }

        // RasterizerState
        {
            nn::gfx::RasterizerState::InfoType rasterizerInfo;

            // RasterizerStateType_FillSolid_CullNone
            rasterizerInfo.SetDefault();
            rasterizerInfo.SetCullMode(nn::gfx::CullMode_None);
            rasterizerInfo.SetScissorEnabled(true);
            rasterizerInfo.SetDepthClipEnabled(false);
            m_RasterizerState[RasterizerStateType_FillSolid_CullNone].Initialize(GetDevice(), rasterizerInfo);
        }

        // BlendState
        {
            nn::gfx::BlendState::InfoType blendInfo;
            nn::gfx::BlendTargetStateInfo targetInfo;

            // BlendStateType_Disabled
            blendInfo.SetDefault();
            targetInfo.SetDefault();
            targetInfo.SetBlendEnabled(false);
            blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
            InitializeBlendState(&m_BlendState[BlendStateType_Disabled], blendInfo);

            // BlendStateType_Alpha
            blendInfo.SetDefault();
            targetInfo.SetDefault();
            targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue);
            targetInfo.SetBlendEnabled(true);
            targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
            targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
            targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
            blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
            InitializeBlendState(&m_BlendState[BlendStateType_Alpha], blendInfo);
        }

        // DepthStencilState
        {
            nn::gfx::DepthStencilState::InfoType depthStencilInfo;

            // DepthStencilStateType_Disabled
            depthStencilInfo.SetDefault();
            depthStencilInfo.SetDepthTestEnabled(false);
            depthStencilInfo.SetDepthWriteEnabled(false);
            m_DepthStencilState[DepthStencilStateType_Disabled].Initialize(GetDevice(), depthStencilInfo);
        }

        // Sampler
        {
            nn::gfx::Sampler::InfoType samplerInfo;

            // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Repeat
            samplerInfo.SetDefault();
            samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
            samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Repeat);
            samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Repeat);
            samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Repeat);
            m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Repeat].Initialize(GetDevice(), samplerInfo);

            // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Mirror
            samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
            samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
            samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
            m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Mirror].Initialize(GetDevice(), samplerInfo);

            // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp
            samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
            samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
            samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
            m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp].Initialize(GetDevice(), samplerInfo);

            // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border
            samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToBorder);
            samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToBorder);
            samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToBorder);
            m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border].Initialize(GetDevice(), samplerInfo);
        }
    }
}// NOLINT(impl/function_size)

void GraphicsFramework::Initialize(
    const FrameworkInfo& info
) NN_NOEXCEPT
{
    Initialize(
        info,
        DefaultAllocateFunction,
        DefaultFreeFunction,
        nullptr
    );
}

void GraphicsFramework::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // GFX 終了
    {
        // m_MemoryPoolAllocator, m_DescriptorPoolAllocator は Finalize() するので
        // 確保していた offset, index の個々の Free() は不要

        for (int i = 0; i < SamplerType_End; i++) {
            m_Sampler[i].Finalize(GetDevice());
        }
        for (int i = 0; i < DepthStencilStateType_End; i++) {
            m_DepthStencilState[i].Finalize(GetDevice());
        }
        for (int i = 0; i < BlendStateType_End; i++) {
            FinalizeBlendState(&m_BlendState[i]);
        }
        for (int i = 0; i < RasterizerStateType_End; i++) {
            m_RasterizerState[i].Finalize(GetDevice());
        }
        for (int i = 0; i < VertexStateType_End; i++) {
            FinalizeVertexState(&m_VertexState[i]);
        }
        m_ViewportScissorState.Finalize(GetDevice());
        m_SwapChain.Finalize(&m_Device);

        //for (int i = 0; i < 1/*m_BufferCount*/; i++) {
        //    m_pDepthStencilView[i].Finalize(&m_Device);
        //    m_pDepthStencilBuffer[i].Finalize(&m_Device);
        //    m_pColorTargetView[i].Finalize(&m_Device);
        //    m_pColorBuffer[i].Finalize(&m_Device);
        //}
        int count = m_BufferCount;
        for (int i = 0; i < count; i++) {
            m_pCommandBuffer[i].object.Finalize(&m_Device);
            m_FreeFunction(m_pCommandBuffer[i].pControlMemory, m_pAllocateFunctionUserData);
        }
        //m_FreeFunction(m_pDepthStencilView, m_pAllocateFunctionUserData);
        //m_FreeFunction(m_pDepthStencilBuffer, m_pAllocateFunctionUserData);
        //m_FreeFunction(m_pColorTargetView, m_pAllocateFunctionUserData);
        //m_FreeFunction(m_pColorBuffer, m_pAllocateFunctionUserData);
        m_FreeFunction(m_pCommandBuffer, m_pAllocateFunctionUserData);
        m_FreeFunction(m_pShaderScratchMemory, m_pAllocateFunctionUserData);

        for (int i = 0; i < nn::gfx::DescriptorPoolType_End; i++)
        {
            if (m_DescriptorPoolSlotCount[i] > 0) {
                m_DescriptorPool[i].Finalize(&m_Device);
                m_DescriptorPoolAllocator[i].Finalize();
            }
            m_DescriptorPoolSlotCount[i] = 0;
        }

        for (int i = 0; i < MemoryPoolType_End; i++)
        {
            if (m_MemoryPool[i].size == 0) {
                continue;
            }
            m_MemoryPool[i].allocator.Finalize();
            m_MemoryPool[i].object.Finalize(&m_Device);
            m_MemoryPool[i].size = 0;
            m_FreeFunction(m_MemoryPool[i].pMemory, m_pAllocateFunctionUserData);
        }
        m_pFence.Finalize(&m_Device);
        m_Queue.Finalize(&m_Device);
        m_Device.Finalize();
        nn::gfx::Finalize();
    }

    // VI 終了
    {
        nn::vi::DestroyLayer(m_pLayer);
        nn::vi::CloseDisplay(m_pDisplay);
        nn::vi::Finalize();
    }

    m_AllocateFunction = nullptr;
    m_FreeFunction = nullptr;
    m_pAllocateFunctionUserData = nullptr;

    m_IsInitialized = false;
}

bool GraphicsFramework::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

//------------------------------------------------------------------------------
// 初期化パラメータ取得
//------------------------------------------------------------------------------
int GraphicsFramework::GetDisplayWidth() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_DisplayWidth;
}

int GraphicsFramework::GetDisplayHeight() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_DisplayHeight;
}

int GraphicsFramework::GetBufferCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_BufferCount;
}

nn::AlignedAllocateFunctionWithUserData GraphicsFramework::GetAllocateFunction() const NN_NOEXCEPT
{
    //NN_SDK_REQUIRES(IsInitialized());
    return m_AllocateFunction;
}

nn::FreeFunctionWithUserData GraphicsFramework::GetFreeFunction() const NN_NOEXCEPT
{
    //NN_SDK_REQUIRES(IsInitialized());
    return m_FreeFunction;
}

void* GraphicsFramework::GetAllocateFunctionUserData() const NN_NOEXCEPT
{
    //NN_SDK_REQUIRES(IsInitialized());
    return m_pAllocateFunctionUserData;
}

//------------------------------------------------------------------------------
// メモリアロケータ(初期化時に渡したアロケータ関数を使用)
//------------------------------------------------------------------------------
void* GraphicsFramework::AllocateMemory(size_t size, size_t alignment) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_AllocateFunction(size, alignment, m_pAllocateFunctionUserData);
}

void GraphicsFramework::FreeMemory(void* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_FreeFunction(ptr, m_pAllocateFunctionUserData);
}

//------------------------------------------------------------------------------
// 内部 GFX オブジェクト取得
//------------------------------------------------------------------------------
nn::vi::Display* GraphicsFramework::GetDisplay() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pDisplay;
}

nn::vi::Layer* GraphicsFramework::GetLayer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_pLayer;
}

nn::gfx::Device* GraphicsFramework::GetDevice() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_Device;
}

nn::gfx::Queue* GraphicsFramework::GetQueue() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_Queue;
}

nn::gfx::SwapChain* GraphicsFramework::GetSwapChain() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_SwapChain;
}

nn::gfx::Fence* GraphicsFramework::GetFence() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_pFence;
}

nn::gfx::ViewportScissorState* GraphicsFramework::GetViewportScissorState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_ViewportScissorState;
}

nn::gfx::VertexState* GraphicsFramework::GetVertexState(VertexStateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_VertexState[type].object;
}

nn::gfx::RasterizerState* GraphicsFramework::GetRasterizerState(RasterizerStateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_RasterizerState[type];
}

nn::gfx::BlendState* GraphicsFramework::GetBlendState(BlendStateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_BlendState[type].object;
}

nn::gfx::DepthStencilState* GraphicsFramework::GetDepthStencilState(DepthStencilStateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_DepthStencilState[type];
}

nn::gfx::Sampler* GraphicsFramework::GetSampler(SamplerType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_Sampler[type];
}

//------------------------------------------------------------------------------
// カラーバッファ, 深度ステンシルバッファ
//------------------------------------------------------------------------------

nn::gfx::ColorTargetView* GraphicsFramework::GetColorTargetView() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return m_SwapChain.AcquireNextScanBufferView();
}

#if 0
nn::gfx::Texture* GraphicsFramework::GetDepthStencilBuffer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_pDepthStencilBuffer[0];
}

nn::gfx::DepthStencilView* GraphicsFramework::GetDepthStencilView() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_pDepthStencilView[0];
}
#endif

//------------------------------------------------------------------------------
// メモリプール
//------------------------------------------------------------------------------
nn::gfx::MemoryPool* GraphicsFramework::GetMemoryPool(MemoryPoolType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_MemoryPool[type].size > 0);
    return &m_MemoryPool[type].object;
}

nn::gfx::util::MemoryPoolAllocator* GraphicsFramework::GetMemoryPoolAllocator(MemoryPoolType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_MemoryPool[type].size > 0);
    return &m_MemoryPool[type].allocator;
}

ptrdiff_t GraphicsFramework::AllocatePoolMemory(MemoryPoolType type, size_t size, size_t alignment) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_MemoryPool[type].size > 0);
    return m_MemoryPool[type].allocator.Allocate(size, alignment);
}

void GraphicsFramework::FreePoolMemory(MemoryPoolType type, ptrdiff_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_MemoryPool[type].size > 0);
    m_MemoryPool[type].allocator.Free(offset);
}

//------------------------------------------------------------------------------
// デスクリプタプール
//------------------------------------------------------------------------------
nn::gfx::DescriptorPool* GraphicsFramework::GetDescriptorPool(nn::gfx::DescriptorPoolType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[type] > 0);
    return &m_DescriptorPool[type];
}

nn::gfx::util::DescriptorPoolAllocator* GraphicsFramework::GetDescriptorPoolAllocator(nn::gfx::DescriptorPoolType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[type] > 0);
    return &m_DescriptorPoolAllocator[type];
}

int GraphicsFramework::AllocateDescriptorSlot(nn::gfx::DescriptorPoolType type, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[type] > 0);
    return m_DescriptorPoolAllocator[type].Allocate(count);
}

void GraphicsFramework::FreeDescriptorSlot(nn::gfx::DescriptorPoolType type, int indexSlot) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[type] > 0);
    m_DescriptorPoolAllocator[type].Free(indexSlot);
}

void GraphicsFramework::SetBufferViewToDescriptorPool(int indexSlot, const nn::gfx::GpuAddress& gpuAddress, size_t size)
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_BufferView] > 0);

    nn::gfx::DescriptorPool& bufferDescriptorPool = m_DescriptorPool[nn::gfx::DescriptorPoolType_BufferView];
    bufferDescriptorPool.BeginUpdate();
    {
        bufferDescriptorPool.SetBufferView(indexSlot, gpuAddress, size);
    }
    bufferDescriptorPool.EndUpdate();
}

void GraphicsFramework::SetTextureViewToDescriptorPool(int indexSlot, const nn::gfx::TextureView* pTextureView)
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_TextureView] > 0);

    nn::gfx::DescriptorPool& textureViewDescriptorPool = m_DescriptorPool[nn::gfx::DescriptorPoolType_TextureView];
    textureViewDescriptorPool.BeginUpdate();
    {
        textureViewDescriptorPool.SetTextureView(indexSlot, pTextureView);
    }
    textureViewDescriptorPool.EndUpdate();
}

void GraphicsFramework::SetSamplerToDescriptorPool(int indexSlot, const nn::gfx::Sampler* pSampler)
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_Sampler] > 0);

    nn::gfx::DescriptorPool& samplerDescriptorPool = m_DescriptorPool[nn::gfx::DescriptorPoolType_Sampler];
    samplerDescriptorPool.BeginUpdate();
    {
        samplerDescriptorPool.SetSampler(indexSlot, pSampler);
    }
    samplerDescriptorPool.EndUpdate();
}

//------------------------------------------------------------------------------
// ルートコマンドバッファ
//------------------------------------------------------------------------------
nn::gfx::CommandBuffer* GraphicsFramework::GetRootCommandBuffer(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return &m_pCommandBuffer[bufferIndex].object;
}

void GraphicsFramework::ResetRootCommandBuffer(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    GraphicsFramework::CommandBuffer& commandBuffer = m_pCommandBuffer[bufferIndex];

    commandBuffer.object.Reset();

    MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
    commandBuffer.object.AddCommandMemory(
        GetMemoryPool(memorypoolType),
        commandBuffer.commandMemoryOffset,
        commandBuffer.commandMemorySize
    );

    commandBuffer.object.AddControlMemory(
        commandBuffer.pControlMemory,
        commandBuffer.controlMemorySize
    );
}

void GraphicsFramework::BeginRootCommandBuffer(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::CommandBuffer& commandBuffer = m_pCommandBuffer[bufferIndex].object;

    commandBuffer.Begin();
    commandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor);

#if NN_GFX_IS_TARGET_NVN
    int scratchMemoryIndex = bufferIndex;
    ShaderScratchMemoryInfo& shaderScratchMemory = m_pShaderScratchMemory[scratchMemoryIndex];

    if (shaderScratchMemory.size != m_ShaderScratchMemorySize) {
        CreateShaderScratchMemory(scratchMemoryIndex);
    }

    if (shaderScratchMemory.size > 0) {
        nvnCommandBufferSetShaderScratchMemory(commandBuffer.ToData()->pNvnCommandBuffer,
            m_MemoryPool[MemoryPoolType_RenderTarget].object.ToData()->pNvnMemoryPool,
            shaderScratchMemory.offset, shaderScratchMemory.size);
    }
#endif

    if (m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_BufferView] > 0) {
        commandBuffer.SetDescriptorPool(&m_DescriptorPool[nn::gfx::DescriptorPoolType_BufferView]);
    }
    if (m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_TextureView] > 0) {
        commandBuffer.SetDescriptorPool(&m_DescriptorPool[nn::gfx::DescriptorPoolType_TextureView]);
    }
    if (m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_Sampler] > 0) {
        commandBuffer.SetDescriptorPool(&m_DescriptorPool[nn::gfx::DescriptorPoolType_Sampler]);
    }
}

void GraphicsFramework::EndRootCommandBuffer(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_pCommandBuffer[bufferIndex].object.End();
}

//------------------------------------------------------------------------------
// シェーダスクラッチメモリ
//------------------------------------------------------------------------------
void GraphicsFramework::SetShaderScratchMemorySize(size_t size) NN_NOEXCEPT
{
#if NN_GFX_IS_TARGET_NVN
    NN_SDK_REQUIRES(IsInitialized());
    int granularity;
    nvnDeviceGetInteger(m_Device.ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY, &granularity);

    m_ShaderScratchMemorySize = nn::util::align_up(size, granularity);
#else
    NN_UNUSED(size);
#endif
}

void GraphicsFramework::CreateShaderScratchMemory(int bufferIndex) NN_NOEXCEPT
{
#if NN_GFX_IS_TARGET_NVN
    NN_SDK_REQUIRES(IsInitialized());
    ShaderScratchMemoryInfo& shaderScratchMemory = m_pShaderScratchMemory[bufferIndex];

    MemoryPoolType memorypoolType = MemoryPoolType_RenderTarget;
    shaderScratchMemory.size = m_ShaderScratchMemorySize;

    int alignment;
    nvnDeviceGetInteger(m_Device.ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &alignment);

    m_MemoryPool[memorypoolType].allocator.Free(shaderScratchMemory.offset);
    shaderScratchMemory.offset = m_MemoryPool[memorypoolType].allocator.Allocate(
        shaderScratchMemory.size, static_cast<size_t>(alignment));
    NN_SDK_ASSERT(shaderScratchMemory.offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
#else
    NN_UNUSED(bufferIndex);
#endif
}

void GraphicsFramework::BeginFrame(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    ResetRootCommandBuffer(bufferIndex);
    BeginRootCommandBuffer(bufferIndex);
}

void GraphicsFramework::EndFrame(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    //// カラーバッファからスキャンバッファへコピー
    //{
    //    nn::gfx::CommandBuffer* commandBuffer = GetRootCommandBuffer(bufferIndex);
    //    nn::gfx::Texture* pScanBuffer = GetSwapChain()->AcquireNextScanBuffer();
    //    nn::gfx::Texture* pColorBuffer = GetColorBuffer();

    //    nn::gfx::TextureCopyRegion region;
    //    region.SetDefault();
    //    region.SetWidth(GetDisplayWidth());
    //    region.SetHeight(GetDisplayHeight());
    //    commandBuffer->BlitImage(pScanBuffer, region, pColorBuffer, region, 0);
    //}

    EndRootCommandBuffer(bufferIndex);
}

void GraphicsFramework::ExecuteCommand(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::Queue* queue = GetQueue();
    queue->ExecuteCommand(GetRootCommandBuffer(bufferIndex), GetFence());
    queue->Flush();
}

void GraphicsFramework::Present() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    nn::gfx::Queue* queue = GetQueue();
    //GetFence()->Sync(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / 60));
    queue->Present(GetSwapChain(), 1);
    queue->Sync();
}

//------------------------------------------------------------------------------
// メモリプールアロケータ, デスクリプタプールアロケータ用アロケータ関数
//------------------------------------------------------------------------------
void* GraphicsFramework::MemoryPoolAllocatorMalloc(size_t size, void* pUserData)
{
    GraphicsFramework* pFramework = reinterpret_cast<GraphicsFramework*> (pUserData);
    return pFramework->GetAllocateFunction()(size, 1, pFramework->GetAllocateFunctionUserData());
    //return pFramework->AllocateMemory(size, 1);
}

void GraphicsFramework::MemoryPoolAllocatorFree(void* ptr, void* pUserData)
{
    GraphicsFramework* pFramework = reinterpret_cast<GraphicsFramework*> (pUserData);
    pFramework->GetFreeFunction()(ptr, pFramework->GetAllocateFunctionUserData());
    //pFramework->FreeMemory(ptr);
}

void* GraphicsFramework::DescriptorPoolAllocatorMalloc(size_t size, void* pUserData)
{
    GraphicsFramework* pFramework = reinterpret_cast<GraphicsFramework*> (pUserData);
    return pFramework->GetAllocateFunction()(size, 1, pFramework->GetAllocateFunctionUserData());
    //return pFramework->AllocateMemory(size, 1);
}

void GraphicsFramework::DescriptorPoolAllocatorFree(void* ptr, void* pUserData)
{
    GraphicsFramework* pFramework = reinterpret_cast<GraphicsFramework*> (pUserData);
    pFramework->GetFreeFunction()(ptr, pFramework->GetAllocateFunctionUserData());
    //pFramework->FreeMemory(ptr);
}

//------------------------------------------------------------------------------
// 外部 gfx オブジェクト コマンドバッファ
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeCommandBuffer(GraphicsFramework::CommandBuffer* pOutCommandBuffer, nn::gfx::CommandBuffer::InfoType& info, size_t commandMemorySize, size_t controlMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // CommandBuffer
    pOutCommandBuffer->object.Initialize(GetDevice(), info);

    // CommandMemory
    MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
    size_t alignment = nn::gfx::CommandBuffer::GetCommandMemoryAlignment(GetDevice());
    size_t size = commandMemorySize;
    pOutCommandBuffer->commandMemoryOffset = AllocatePoolMemory(memorypoolType, size, alignment);
    NN_SDK_ASSERT(pOutCommandBuffer->commandMemoryOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
    pOutCommandBuffer->commandMemorySize = size;

    // ControlMemory
    alignment = nn::gfx::CommandBuffer::GetControlMemoryAlignment(GetDevice());
    size = controlMemorySize;
    pOutCommandBuffer->pControlMemory = AllocateMemory(size, alignment);
    pOutCommandBuffer->controlMemorySize = size;
}

void GraphicsFramework::FinalizeCommandBuffer(GraphicsFramework::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // CommandBuffer
    pCommandBuffer->object.Finalize(GetDevice());

    // CommandMemory
    nns::gfx::GraphicsFramework::MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
    FreePoolMemory(memorypoolType, pCommandBuffer->commandMemoryOffset);

    // ControlMemory
    FreeMemory(pCommandBuffer->pControlMemory);
}

void GraphicsFramework::ResetCommandBuffer(GraphicsFramework::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    pCommandBuffer->object.Reset();

    nns::gfx::GraphicsFramework::MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
    pCommandBuffer->object.AddCommandMemory(
        GetMemoryPool(memorypoolType),
        pCommandBuffer->commandMemoryOffset,
        pCommandBuffer->commandMemorySize
    );

    pCommandBuffer->object.AddControlMemory(
        pCommandBuffer->pControlMemory,
        pCommandBuffer->controlMemorySize
    );
}

//------------------------------------------------------------------------------
// gfx オブジェクト 頂点ステート
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeVertexState(GraphicsFramework::VertexState* pOutVertexState, nn::gfx::VertexState::InfoType& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize(info);
    if (size) {
        pOutVertexState->pMemory = AllocateMemory(size, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
        pOutVertexState->object.SetMemory(pOutVertexState->pMemory, size);
    }
    else {
        pOutVertexState->pMemory = nullptr;
    }

    pOutVertexState->object.Initialize(GetDevice(), info);
}

void GraphicsFramework::FinalizeVertexState(GraphicsFramework::VertexState* pVertexState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    pVertexState->object.Finalize(GetDevice());
    if (pVertexState->pMemory) {
        FreeMemory(pVertexState->pMemory);
    }
}

//------------------------------------------------------------------------------
// gfx オブジェクト ブレンドステート
//------------------------------------------------------------------------------
void GraphicsFramework::InitializeBlendState(GraphicsFramework::BlendState* pOutBlendState, nn::gfx::BlendState::InfoType& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize(info);
    if (size) {
        pOutBlendState->pMemory = AllocateMemory(size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        pOutBlendState->object.SetMemory(pOutBlendState->pMemory, size);
    }
    else {
        pOutBlendState->pMemory = nullptr;
    }

    pOutBlendState->object.Initialize(GetDevice(), info);
}

void GraphicsFramework::FinalizeBlendState(GraphicsFramework::BlendState* pBlendState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    pBlendState->object.Finalize(GetDevice());
    if (pBlendState->pMemory) {
        FreeMemory(pBlendState->pMemory);
    }
}

//------------------------------------------------------------------------------
// デフォルトのメモリアロケータ
//------------------------------------------------------------------------------
void* GraphicsFramework::DefaultAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
        NN_UNUSED(pUserData);
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    return aligned_alloc(alignment, size);
#else
    return _aligned_malloc(size, alignment);
#endif
}

void GraphicsFramework::DefaultFreeFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    free(ptr);
#else
    _aligned_free(ptr);
#endif
}

void* GraphicsFramework::DefaultReallocateFunction(void* ptr, size_t newSize, void* pUserData)
{
    NN_UNUSED(pUserData);
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    return realloc(ptr, newSize);
#else
    return _aligned_realloc(ptr, newSize, 1);
#endif
}


} // namespace gfx
} // namespace nns
