﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/font.h>
#include <nn/font/font_TextureCache.h>
#include <nn/font/font_ScalableFont.h>
#include <nn/gfx.h>

namespace framework{

    class TextWriter;
    class TextWriterPool;
    class PoolTextWriterHolder;

    class TextWriter
    {
        friend class TextWriterPool;
    public:
        void Reset() NN_NOEXCEPT;

        void SetViewportSize(float w, float h) NN_NOEXCEPT;
        void SetColor(const nn::util::Color4fType& color) NN_NOEXCEPT;
        void SetPosition(float x, float y) NN_NOEXCEPT;
        void SetFontSize(float size) NN_NOEXCEPT;
        void Print(const char* str) NN_NOEXCEPT;

        void MakeCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT;

    private:
        void Initialize(TextWriterPool* pTextWriterPool) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

    private:
        TextWriterPool* m_pTextWriterPool;
        nn::font::DispStringBuffer m_DisplayStringBuffer;
        nn::font::GpuBuffer m_GpuBuffer;

        float m_ViewportSizeX;
        float m_ViewportSizeY;

        nn::util::Color4fType m_Color;
        float m_PositionX;
        float m_PositionY;
        float m_FontSize;

        void* m_pDrawBufferMemory;
        ptrdiff_t m_GpuBufferOffset;
    };

    struct TextWriterPoolInfo
    {
        // プール内に用意する Writer の数
        int writerCount;

        // 1 回で描ける文字数の上限
        int charCountMax;

        // Writer で使用するフォントバイナリ(ttf フォーマット)
        const void* pFontData;
        // Writer で使用するフォントバイナリの大きさ(バイト)
        size_t fontDataSize;
    };

    class TextWriterPool
    {
        friend class TextWriter;
        friend class PoolTextWriterHolder;
    public:
        void Initialize(const TextWriterPoolInfo& info) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        std::shared_ptr<PoolTextWriterHolder> Acquire() NN_NOEXCEPT;

    private:
        nn::font::TextureCache m_TextureCache;
        nn::font::ScalableFont m_ScalableFont;
        nn::font::RectDrawer m_RectDrawer;
        void* m_pRectDrawerWorkBuffer;
        int m_CharCountMax;
        int m_WriterCount;

        struct Entry
        {
            TextWriter writer;
            bool isUsed;
        };
        std::vector<Entry> m_EntryList;
    };

    class PoolTextWriterHolder
    {
    public:
        PoolTextWriterHolder(int index, TextWriter* pWriter, TextWriterPool* pPool) NN_NOEXCEPT;
        ~PoolTextWriterHolder() NN_NOEXCEPT;

        TextWriter* Get() NN_NOEXCEPT;
    private:
        int m_Index;
        TextWriter* m_pWriter;
        TextWriterPool* m_pPool;
    };

}
