﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TextWriterPool.h"

#include "Framework.h"
#include "../Config.h"
#include <nn/util/util_Color.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_Matrix.h>

namespace framework{

    namespace{
        static const uint32_t TextureCacheWidth = 512;
        static const uint32_t TextureCacheHeight = 512;

        static const int CacheFontSize = 40;

        void* MemoryAllocate(size_t size, size_t alignment, void*) NN_NOEXCEPT
        {
            return Framework::Allocate(size, alignment);
        }

        void MemoryFree(void* ptr, void*) NN_NOEXCEPT
        {
            Framework::Free(ptr);
        }

        bool AllocateAndRegisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutSlot, const nn::gfx::TextureView& textureView, void*) NN_NOEXCEPT
        {
            int index = Framework::AllocateTextureDescriptorSlot();
            auto pPool = Framework::GetTextureDescriptorPool();
            pPool->SetTextureView(index, &textureView);
            pPool->GetDescriptorSlot(pOutSlot, index);
            return true;
        }

        void UnregisterAndFreeTextureViewSlot(nn::gfx::DescriptorSlot* pSlot, const nn::gfx::TextureView&, void*) NN_NOEXCEPT
        {
            auto pPool = Framework::GetTextureDescriptorPool();
            auto index = pPool->GetDescriptorSlotIndex(*pSlot);
            Framework::FreeTextureDescriptorSlot(index);
        }

        bool AllocateAndRegisterSamplerSlot(nn::gfx::DescriptorSlot* pOutSlot, const nn::gfx::Sampler& sampler, void*) NN_NOEXCEPT
        {
            int index = Framework::AllocateSamplerDescriptorSlot();
            auto pPool = Framework::GetSamplerDescriptorPool();
            pPool->SetSampler(index, &sampler);
            pPool->GetDescriptorSlot(pOutSlot, index);
            return true;
        }

        void UnregisterAndFreeSamplerSlot(nn::gfx::DescriptorSlot* pSlot, const nn::gfx::Sampler&, void*) NN_NOEXCEPT
        {
            auto pPool = Framework::GetSamplerDescriptorPool();
            auto index = pPool->GetDescriptorSlotIndex(*pSlot);
            Framework::FreeSamplerDescriptorSlot(index);
        }
    }

    void TextWriterPool::Initialize(const TextWriterPoolInfo& info) NN_NOEXCEPT
    {
        if(info.writerCount == 0)
        {
            m_WriterCount = 0;
            return;
        }

        NN_SDK_REQUIRES_NOT_NULL(info.pFontData);
        NN_SDK_REQUIRES_GREATER(info.fontDataSize, 0u);
        auto pDevice = Framework::GetDevice();

        // テクスチャキャッシュの初期化
        {
            nn::font::TextureCache::InitializeArg arg = {};
            arg.SetDefault();
            arg.allocateFunction = MemoryAllocate;
            arg.pUserDataForAllocateFunction = nullptr;
            arg.textureCacheWidth = TextureCacheWidth;
            arg.textureCacheHeight = TextureCacheHeight;
            arg.fontCount = 1;
            arg.fontListCount[0] = 1;
            arg.pFontDatas[0][0] = info.pFontData;
            arg.pFontDataSizes[0][0] = info.fontDataSize;
            arg.pFontDataTypes[0][0] = nn::font::TextureCache::FontDataType_Ttf;

            m_TextureCache.Initialize(pDevice, arg);
            m_TextureCache.RegisterTextureViewToDescriptorPool(AllocateAndRegisterTextureViewSlot, nullptr);
        }

        // スケーラブルフォントの初期化
        {
            nn::font::ScalableFont::InitializeArg arg;
            arg.SetDefault();
            arg.pTextureCache = &m_TextureCache;
            arg.fontSize = CacheFontSize;
            arg.fontFace = 0;
            arg.isAlternateCharSpaceWithOriginalWidthForNotReadyChar = true;
            m_ScalableFont.Initialize(arg);
        }

        // ASCII 文字を登録
        for(uint16_t c = 0; c < 127; c++)
        {
            m_ScalableFont.RegisterGlyph(c);
        }
        m_TextureCache.UpdateTextureCache();
        m_TextureCache.CompleteTextureCache();

        // RectDrawer の初期化
        {
            size_t workBufferSize = nn::font::RectDrawer::GetWorkBufferSize(pDevice, info.charCountMax);
            void* pWorkBuffer = Framework::Allocate(workBufferSize, NN_ALIGNOF(std::max_align_t));
            //NN_SDK_LOG("[devovl] RectDrawer WorkSize=%lld\n", workBufferSize);
            NN_ABORT_UNLESS(m_RectDrawer.Initialize(pDevice, pWorkBuffer, info.charCountMax));
            m_RectDrawer.RegisterSamplerToDescriptorPool(AllocateAndRegisterSamplerSlot, nullptr);
            m_pRectDrawerWorkBuffer = pWorkBuffer;
        }

        m_CharCountMax = info.charCountMax;
        m_WriterCount = info.writerCount;

        m_EntryList.resize(info.writerCount);
        for(auto& e : m_EntryList)
        {
            e.writer.Initialize(this);
            e.isUsed = false;
        }
    }

    void TextWriterPool::Finalize() NN_NOEXCEPT
    {
        if(m_WriterCount == 0)
        {
            return;
        }

        auto pDevice = Framework::GetDevice();

        for(auto& e : m_EntryList)
        {
            NN_ASSERT(!e.isUsed);
            e.writer.Finalize();
        }

        m_RectDrawer.UnregisterSamplerFromDescriptorPool(UnregisterAndFreeSamplerSlot, nullptr);
        m_RectDrawer.Finalize(pDevice);
        m_ScalableFont.Finalize(pDevice);
        m_TextureCache.UnregisterTextureViewFromDescriptorPool(UnregisterAndFreeTextureViewSlot, nullptr);
        m_TextureCache.Finalize(pDevice, MemoryFree, nullptr);

        Framework::Free(m_pRectDrawerWorkBuffer);
    }

    std::shared_ptr<PoolTextWriterHolder> TextWriterPool::Acquire() NN_NOEXCEPT
    {
        int index = -1;
        TextWriter* pWriter = nullptr;
        for(int i = 0; i < m_WriterCount; i++)
        {
            auto& e = m_EntryList[i];
            if(!e.isUsed)
            {
                index = i;
                pWriter = &e.writer;
                e.isUsed = true;
                break;
            }
        }

        if(index < 0)
        {
            NNT_VI_FW_LOG("[TextWriterPool] Acquire text writer failed\n");
            return nullptr;
        }

        // 状態をクリアしてから返す。
        pWriter->Reset();
        return std::make_shared<PoolTextWriterHolder>(index, pWriter, this);
    }

    //-----------------------------

    void TextWriter::Initialize(TextWriterPool* pTextWriterPool) NN_NOEXCEPT
    {
        auto pDevice = Framework::GetDevice();
        m_pTextWriterPool = pTextWriterPool;
        {
            nn::font::DispStringBuffer::InitializeArg dsbArg = {};
            dsbArg.SetCharCountMax(pTextWriterPool->m_CharCountMax);

            size_t drawBufferSize = nn::font::DispStringBuffer::GetRequiredDrawBufferSize(dsbArg);
            void* pDrawBufferMemory = Framework::Allocate(drawBufferSize, NN_ALIGNOF(std::max_align_t));
            //NN_SDK_LOG("[devovl]DsipStringBuffer drawBufferSize=%lld\n", drawBufferSize);
            dsbArg.SetDrawBuffer(pDrawBufferMemory);
            m_pDrawBufferMemory = pDrawBufferMemory;

            nn::font::GpuBuffer::InitializeArg gbArg = {};
            gbArg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
            size_t constantBufferSize = nn::font::DispStringBuffer::GetRequiredConstantBufferSize(pDevice, dsbArg);
            //NN_SDK_LOG("[devovl]DsipStringBuffer constantBufferSize=%lld\n", constantBufferSize);
            auto offset = Framework::AllocateConstantBufferPoolMemory(constantBufferSize, NN_ALIGNOF(std::max_align_t));
            gbArg.SetBufferSize(constantBufferSize);
            gbArg.SetBufferCount(1);
            gbArg.SetMemoryPool(Framework::GetConstantBufferMemoryPool());
            gbArg.SetMemoryPoolOffset(offset);
            gbArg.SetAllocator(MemoryAllocate, nullptr);
            gbArg.SetAllocateSyncFlag(false);
            NN_ABORT_UNLESS(m_GpuBuffer.Initialize(pDevice, gbArg));
            m_GpuBufferOffset = offset;

            dsbArg.SetConstantBuffer(&m_GpuBuffer);
            NN_ABORT_UNLESS(m_DisplayStringBuffer.Initialize(pDevice, dsbArg));
        }

        Reset();
    }

    void TextWriter::Finalize() NN_NOEXCEPT
    {
        auto pDevice = Framework::GetDevice();

        m_GpuBuffer.Finalize(pDevice, MemoryFree, nullptr);
        m_DisplayStringBuffer.Finalize(pDevice);

        Framework::Free(m_pDrawBufferMemory);
        Framework::FreeConstantBufferPoolMemory(m_GpuBufferOffset);
    }

    void TextWriter::Reset() NN_NOEXCEPT
    {
        m_ViewportSizeX = 1280;
        m_ViewportSizeY = 720;
        m_PositionX = 0;
        m_PositionY = 0;
        m_FontSize = 10;
        m_Color = { 1, 1, 1, 1 };
        m_DisplayStringBuffer.SetCharCount(0);
    }

    void TextWriter::SetViewportSize(float w, float h) NN_NOEXCEPT
    {
        m_ViewportSizeX = w;
        m_ViewportSizeY = h;
    }

    void TextWriter::SetPosition(float x, float y) NN_NOEXCEPT
    {
        m_PositionX = x;
        m_PositionY = y;
    }

    void TextWriter::SetFontSize(float size) NN_NOEXCEPT
    {
        m_FontSize = size;
    }

    void TextWriter::SetColor(const nn::util::Color4fType& color) NN_NOEXCEPT
    {
        m_Color = color;
    }

    void TextWriter::Print(const char* str) NN_NOEXCEPT
    {
        nn::font::TextWriter writer;
        writer.SetDispStringBuffer(&m_DisplayStringBuffer);
        writer.SetFont(&m_pTextWriterPool->m_ScalableFont);
        writer.SetCursor(m_PositionX, m_PositionY);
        writer.SetScale(m_FontSize / CacheFontSize, m_FontSize / CacheFontSize);
        nn::util::Unorm8x4 color = nn::util::Color4u8(m_Color);
        writer.SetTextColor(color);

        writer.StartPrint();
        {
            writer.Print(str);
        }
        writer.EndPrint();

        m_GpuBuffer.Map(0);
        {
            nn::util::MatrixT4x4fType m = NN_UTIL_MATRIX_T4X4F_INITIALIZER(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);
            nn::util::MatrixOrthographicOffCenterRightHanded(
                &m,
                0,
                m_ViewportSizeX,
                m_ViewportSizeY,
                0,
                0,   // nearZ
                1000 // farZ
            );
            m_DisplayStringBuffer.BuildConstantBuffer(m);
        }
        m_GpuBuffer.Unmap();
        m_GpuBuffer.SetGpuAccessBufferIndex(0);
    }

    void TextWriter::MakeCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
    {
        m_pTextWriterPool->m_RectDrawer.Draw(*pCommandBuffer, m_DisplayStringBuffer);
    }

    //----------------------------------

    PoolTextWriterHolder::PoolTextWriterHolder(int index, TextWriter* pWriter, TextWriterPool* pPool) NN_NOEXCEPT
        : m_Index(index)
        , m_pWriter(pWriter)
        , m_pPool(pPool)
    {
    }

    PoolTextWriterHolder::~PoolTextWriterHolder() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pPool);
        NN_SDK_ASSERT_NOT_NULL(m_pWriter);

        NN_SDK_ASSERT(m_pPool->m_EntryList[m_Index].isUsed);
        m_pPool->m_EntryList[m_Index].isUsed = false;
    }

    TextWriter* PoolTextWriterHolder::Get() NN_NOEXCEPT
    {
        return m_pWriter;
    }

}
