﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Framework.h"
#include "FrameworkImpl.h"

#include <memory>
#include <vector>
#include <array>
#include <nn/nn_Assert.h>
#include <nn/pl.h>
#include <nn/util/util_ScopeExit.h>

#include "gfx_GraphicsFramework.h"
#include "Hid.h"
#include "../panel/PanelPreparationTaskQueue.h"

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

namespace framework{
    nns::gfx::GraphicsFramework g_Gfw;
    FrameworkImpl g_Impl;
    panel::PanelPreparationTaskQueue g_PanelPreparationTaskQueue;
}

namespace {
    using namespace framework;

    int g_SamplerDescriptorSlotIndex;
    nn::gfx::DescriptorSlot g_DefaultSamplerDescriptorSlot;
}


void Framework::Initialize(const framework::FrameworkParameter& param) NN_NOEXCEPT
{
    g_Impl.parameter = param;
    {
        nns::gfx::GraphicsFramework::FrameworkInfo info;
        //info.SetDebugMode(nn::gfx::DebugMode_Enable);
        info.SetBufferCount(param.rootCommandBufferCount);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetDisplayWidth(param.width);
        info.SetDisplayHeight(param.height);
        info.SetSwapChainBufferCount(param.frameBufferCount);
        info.SetSwapChainFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
        info.SetRootCommandBufferCommandMemorySize(param.rootCommandMemorySize);
        info.SetRootCommandBufferControlMemorySize(param.rootControlMemorySize);
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_RenderTarget  , param.memoryPool.texturePoolSize       );
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_CommandBuffer , param.memoryPool.commandBufferPoolSize );
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_ConstantBuffer, param.memoryPool.constantBufferPoolSize);
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_Shader        , param.memoryPool.shaderPoolSize        );
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_Data          , param.memoryPool.dataPoolSize          );
        g_Gfw.InitializeGraphicsSystem(param.firmwareMemorySize);
        g_Gfw.Initialize(info);
    }
    {
        auto pPool = g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler);
        pPool->BeginUpdate();
        {
            g_SamplerDescriptorSlotIndex = g_Gfw.GetDescriptorPoolAllocator(nn::gfx::DescriptorPoolType_Sampler)->Allocate(1);
            NN_ASSERT_NOT_EQUAL(g_SamplerDescriptorSlotIndex, -1);
            int index = g_SamplerDescriptorSlotIndex;
            pPool->SetSampler(index, g_Gfw.GetSampler(nns::gfx::SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border));
            pPool->GetDescriptorSlot(&g_DefaultSamplerDescriptorSlot, index);
        }
        pPool->EndUpdate();
    }

    g_Impl.memoryPoolManager.Initialize(param.memoryPool);


    // TexturePool を初期化
    {
        g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->BeginUpdate();
        NN_UTIL_SCOPE_EXIT{
            g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->EndUpdate();
        };

        framework::TexturePoolInfo info;
        info.textureWidth  = param.poolTextureWidth;
        info.textureHeight = param.poolTextureHeight;
        info.textureFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
        info.textureCount  = param.poolTextureCount;

        g_Impl.texturePool.Initialize(info);
    }

    // TextWriterPool を初期化
    {
        g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->BeginUpdate();
        g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->BeginUpdate();
        NN_UTIL_SCOPE_EXIT{
            g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->EndUpdate();
            g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->EndUpdate();
        };
        // 共有フォントを取得
        auto fontType = nn::pl::SharedFontType_Standard;
        nn::pl::RequestSharedFontLoad(fontType);
        while(nn::pl::GetSharedFontLoadState(fontType) != nn::pl::SharedFontLoadState_Loaded)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }

        framework::TextWriterPoolInfo info;
        info.charCountMax = param.poolTextWriterCharCountMax;
        info.writerCount  = param.poolTextWriterCount;
        info.pFontData    = nn::pl::GetSharedFontAddress(fontType);
        info.fontDataSize = nn::pl::GetSharedFontSize(fontType);
        NN_SDK_ASSERT_NOT_NULL(info.pFontData);
        NN_SDK_ASSERT_GREATER(info.fontDataSize, 0u);

        g_Impl.textWriterPool.Initialize(info);
    }

    // 描画履歴を初期化
    {
        g_Impl.renderedRegion.resize(param.frameBufferCount);
        for(auto& e : g_Impl.renderedRegion)
        {
            e.clearColor = nn::util::Color4f(0, 0, 0, 0);
        }
    }


    g_Impl.commandResourceLock.Initialize(param.rootCommandBufferCount);
    g_Impl.rootCommandMemoryUsedSizeMax = 0;
    g_Impl.rootControlMemoryUsedSizeMax = 0;

    // Hid を初期化
    Hid::Initialize();
}

void Framework::Finalize() NN_NOEXCEPT
{
    Hid::Finalize();

    g_Impl.textWriterPool.Finalize();
    g_Impl.texturePool.Finalize();
    g_Impl.commandResourceLock.Finalize();
    {
        g_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, g_SamplerDescriptorSlotIndex);
    }
    g_Gfw.Finalize();
}


void Framework::MakeInitializeCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor);
    //pCommandBuffer->SetDescriptorPool(g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_BufferView));
    pCommandBuffer->SetDescriptorPool(g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView));
    pCommandBuffer->SetDescriptorPool(g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler));

    // フレームバッファのクリア
    {
        nn::gfx::ColorTargetView* viewList[2];
        int n = g_Gfw.GetSwapChain()->GetScanBufferViews(&viewList[0], g_Impl.parameter.frameBufferCount);
        for(int i = 0; i < n; i++)
        {
            nn::gfx::ColorTargetView* pView = viewList[i];
            pCommandBuffer->SetRenderTargets(1, &pView, nullptr);
            MakeSetViewportScissorCommand(pCommandBuffer, GetScreenRectangle());
            MakeClearCommand(pCommandBuffer, nn::util::Color4f(0, 0, 0, 0));
        }
    }
}

//-----------------------------------------

framework::Rectangle Framework::GetScreenRectangle() NN_NOEXCEPT
{
    return Rectangle(0, 0, g_Impl.parameter.width, g_Impl.parameter.height);
}

//-----------------------------------------

void Framework::MakeSetViewportCommand(nn::gfx::CommandBuffer* pCommandBuffer, const Rectangle& rect) NN_NOEXCEPT
{
    nn::gfx::ViewportStateInfo vinfo;
    vinfo.SetDefault();
    vinfo.SetOriginX(static_cast<float>(rect.x));
    vinfo.SetOriginY(static_cast<float>(rect.y));
    vinfo.SetWidth(static_cast<float>(rect.width));
    vinfo.SetHeight(static_cast<float>(rect.height));
    pCommandBuffer->SetViewports(0, 1, &vinfo);
}

void Framework::MakeSetScissorCommand(nn::gfx::CommandBuffer* pCommandBuffer, const Rectangle& rect) NN_NOEXCEPT
{
    nn::gfx::ScissorStateInfo sinfo;
    sinfo.SetDefault();
    sinfo.SetOriginX(rect.x);
    sinfo.SetOriginY(rect.y);
    sinfo.SetWidth(rect.width);
    sinfo.SetHeight(rect.height);
    pCommandBuffer->SetScissors(0, 1, &sinfo);
}

void Framework::MakeSetViewportScissorCommand(nn::gfx::CommandBuffer* pCommandBuffer, const Rectangle& rect) NN_NOEXCEPT
{
    Framework::MakeSetViewportCommand(pCommandBuffer, rect);
    Framework::MakeSetScissorCommand(pCommandBuffer, rect);
}

void Framework::MakeClearCommand(nn::gfx::CommandBuffer* pCommandBuffer, const nn::util::Color4f& color) NN_NOEXCEPT
{
    float c[4];
    std::memcpy(c, &color, 4 * sizeof(float));
    nvnCommandBufferClearColor(
        pCommandBuffer->ToData()->pNvnCommandBuffer,
        0, /* colorTargetIndex */
        c,
        NVN_CLEAR_COLOR_MASK_RGBA
    );
}

//-----------------------------------------

nn::gfx::Device* Framework::GetDevice() NN_NOEXCEPT
{
    return g_Gfw.GetDevice();
}

nn::gfx::ColorTargetView* Framework::GetCurrentColorTargetView() NN_NOEXCEPT
{
    return g_Gfw.GetColorTargetView();
}

nn::gfx::ViewportScissorState* Framework::GetCurrentViewportScissorState() NN_NOEXCEPT
{
    return g_Gfw.GetViewportScissorState();
}

size_t Framework::GetFirmwareMemorySize() NN_NOEXCEPT
{
    return g_Impl.parameter.firmwareMemorySize;
}

nn::gfx::CommandBuffer* Framework::GetCommandBuffer(int index) NN_NOEXCEPT
{
    return g_Gfw.GetRootCommandBuffer(index);
}

size_t Framework::GetRootCommandMemorySize() NN_NOEXCEPT
{
    return g_Impl.parameter.rootCommandMemorySize;
}

size_t Framework::GetRootControlMemorySize() NN_NOEXCEPT
{
    return g_Impl.parameter.rootControlMemorySize;
}

size_t Framework::GetRootCommandMemoryUsedSizeMax() NN_NOEXCEPT
{
    return g_Impl.rootCommandMemoryUsedSizeMax;
}

size_t Framework::GetRootControlMemoryUsedSizeMax() NN_NOEXCEPT
{
    return g_Impl.rootControlMemoryUsedSizeMax;
}

const nn::gfx::DescriptorSlot& Framework::GetDefaultSamplerDescriptorSlot() NN_NOEXCEPT
{
    return g_DefaultSamplerDescriptorSlot;
}

//-----------------------------------------

void* Framework::Allocate(size_t size, size_t alignment) NN_NOEXCEPT
{
    auto p = g_Gfw.AllocateMemory(size, alignment);
    NN_SDK_ASSERT_NOT_NULL(p);
    return p;
}

void Framework::Free(void* p) NN_NOEXCEPT
{
    g_Gfw.FreeMemory(p);
}

//-----------------------------------------

size_t Framework::GetTexturePoolMemorySize() NN_NOEXCEPT
{
    return g_Impl.parameter.memoryPool.texturePoolSize;
}

size_t Framework::GetCommandBufferPoolMemorySize() NN_NOEXCEPT
{
    return g_Impl.parameter.memoryPool.commandBufferPoolSize;
}

size_t Framework::GetConstantBufferPoolMemorySize() NN_NOEXCEPT
{
    return g_Impl.parameter.memoryPool.constantBufferPoolSize;
}

size_t Framework::GetShaderPoolMemorySize() NN_NOEXCEPT
{
    return g_Impl.parameter.memoryPool.shaderPoolSize;
}

size_t Framework::GetDataPoolMemorySize() NN_NOEXCEPT
{
    return g_Impl.parameter.memoryPool.dataPoolSize;
}

#define FRAMEWORK_MEMORYPOOLHOLDER_PTR(Type) g_Impl.memoryPoolManager.Get##Type##MemoryPoolHolder()

#define FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(Type) \
    nn::gfx::MemoryPool* Framework::Get##Type##MemoryPool() NN_NOEXCEPT         \
    {                                                                           \
        return FRAMEWORK_MEMORYPOOLHOLDER_PTR(Type)->GetPool();                 \
    }                                                                           \
    ptrdiff_t Framework::Allocate##Type##PoolMemory(size_t size, size_t alignment) NN_NOEXCEPT \
    {                                                                           \
        auto offset = FRAMEWORK_MEMORYPOOLHOLDER_PTR(Type)->Allocate(size, alignment); \
        NN_ABORT_UNLESS_GREATER_EQUAL(offset, 0);                               \
        return offset;                                                          \
    }                                                                           \
    ptrdiff_t Framework::Allocate##Type##PoolMemoryNoCheck(size_t size, size_t alignment) NN_NOEXCEPT \
    {                                                                           \
        auto offset = FRAMEWORK_MEMORYPOOLHOLDER_PTR(Type)->Allocate(size, alignment); \
        return offset;                                                          \
    }                                                                           \
    void Framework::Free##Type##PoolMemory(ptrdiff_t offset) NN_NOEXCEPT        \
    {                                                                           \
        FRAMEWORK_MEMORYPOOLHOLDER_PTR(Type)->Free(offset);                     \
    }                                                                           \
    size_t Framework::GetTotalAllocated##Type##PoolMemorySize() NN_NOEXCEPT     \
    {                                                                           \
        return FRAMEWORK_MEMORYPOOLHOLDER_PTR(Type)->GetTotalAllocatedSize();   \
    }                                                                           \
    size_t Framework::GetTotalAllocated##Type##PoolMemorySizeMax() NN_NOEXCEPT  \
    {                                                                           \
        return FRAMEWORK_MEMORYPOOLHOLDER_PTR(Type)->GetTotalAllocatedSizeMax(); \
    }

FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(Texture);
FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(CommandBuffer);
FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(ConstantBuffer);
FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(Shader);
FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(Data);

#undef FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS
#undef FRAMEWORK_MEMORYPOOLHOLDER_PTR

//-----------------------------------------

nn::gfx::DescriptorPool* Framework::GetTextureDescriptorPool() NN_NOEXCEPT
{
    return g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView);
}

int Framework::AllocateTextureDescriptorSlot() NN_NOEXCEPT
{
    auto index = g_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    NN_ABORT_UNLESS_NOT_EQUAL(index, -1);
    return index;
}

int Framework::AllocateTextureDescriptorSlotNoCheck() NN_NOEXCEPT
{
    auto index = g_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    return index;
}

void Framework::FreeTextureDescriptorSlot(int index) NN_NOEXCEPT
{
    g_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, index);
}

nn::gfx::DescriptorPool* Framework::GetSamplerDescriptorPool() NN_NOEXCEPT
{
    return g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler);
}

int Framework::AllocateSamplerDescriptorSlot() NN_NOEXCEPT
{
    auto index = g_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    NN_ABORT_UNLESS_NOT_EQUAL(index, -1);
    return index;
}

int Framework::AllocateSamplerDescriptorSlotNoCheck() NN_NOEXCEPT
{
    auto index = g_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    return index;
}

void Framework::FreeSamplerDescriptorSlot(int index) NN_NOEXCEPT
{
    g_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, index);
}

//--------------------------------------

std::shared_ptr<framework::PoolTexture> Framework::AcquirePoolTexture(int width, int height, nn::gfx::ImageFormat format) NN_NOEXCEPT
{
    return g_Impl.texturePool.Acquire(width, height, format);
}

std::shared_ptr<framework::PoolTextWriterHolder> Framework::AcquirePoolTextWriter() NN_NOEXCEPT
{
    return g_Impl.textWriterPool.Acquire();
}

//--------------------------------------

panel::PanelPreparationTaskQueue* Framework::GetPanelPreparationTaskQueue() NN_NOEXCEPT
{
    return &g_PanelPreparationTaskQueue;
}

