﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/vi/fbshare/vi_SharedTextureMemoryPool.h>
#include <nn/vi/fbshare/vi_SharedTexturePool.h>
#include <nn/vi/fbshare/vi_SharedNativeWindow.h>

#include "testVi_NvnUtility.h"

static NN_ALIGNAS(4096) char g_CommandStorage[10 * 4096];
static NN_ALIGNAS(4096) char g_ControlStorage[4096];

static const int DefaultLoopCount  = 10;
static const int DefaultFrameCount = 10;

NNT_VI_TEST_SHIM(SharedNativeWindowImpl_Basic)
{
    static const int LoopCount = DefaultLoopCount;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;
    static const int FrameCount = DefaultFrameCount;
    NN_LOG("SharedNativeWindow>Basic-------------------------------\n");

    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);

        ContextExt context0({"server"});
        Context context1({"client"});

        context0.ConnectService();
        context1.ConnectService();

        // setup SharedLowLevelLayer
        auto layerId = context0.GetDefaultLayerId();
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

        // setup SharedBuffer
        auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

        // setup SharedLayer
        nn::vi::fbshare::SharedLayerHandle hLayer = {};
        nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

        // attach
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));

        // export SharedBuffer to me
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.RegisterSharedBufferImporterAruid(hBuffer, nn::applet::GetAppletResourceUserId()));

        // get hMem id
        nn::vi::native::NativeMemoryHandleId memId = {};
        size_t memSize = 0;
        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedBufferMemoryHandleId(&memId, &memSize, &layout, hBuffer));

        NN_LOG("importing memory:\n");
        NN_LOG("  id   : %u\n", memId);
        NN_LOG("  size : %llu(%lluKB)\n", memSize, memSize / 1024);
        NN_LOG("  count: %d\n", layout.count);
        for(int i = 0; i < layout.count; i++)
        {
            auto& e = layout.entries[i];
            NN_LOG("    texture[%d]: %dx%d %llu-%llu(%llu bytes)\n", i, e.width, e.height, e.offset, e.offset + e.size, e.size);
        }

        {
            // init nvn and device
            NNT_VI_SCOPED_DEVICE           (device);
            NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, g_CommandStorage, sizeof(g_CommandStorage));
            NNT_VI_SCOPED_QUEUE            (queue, device);
            NNT_VI_SCOPED_COMMANDBUFFER    (commandBuffer, device);
            NNT_VI_SCOPED_FRAMETEXTUREVIEW (texView);

            NN_LOG("Creating shared memory pool\n");
            nn::vi::fbshare::SharedTextureMemoryPool sharedMemoryPool;
            NN_ABORT_UNLESS_RESULT_SUCCESS(sharedMemoryPool.Initialize(&device, memId, memSize, nn::vi::fbshare::SharedTextureMemoryPoolOption_None));

            NN_LOG("Creating shared texture pool\n");
            nn::vi::fbshare::SharedTexturePool sharedTexturePool;
            NN_ABORT_UNLESS_RESULT_SUCCESS(sharedTexturePool.Initialize(&device, &sharedMemoryPool, layout, NVN_FORMAT_RGBA8_SRGB));

            NN_LOG("Creating shared native window\n");
            nn::vi::fbshare::SharedNativeWindow sharedNativeWindow;
            NN_ABORT_UNLESS_RESULT_SUCCESS(sharedNativeWindow.Initialize(context1.GetSystemService(), hBuffer, hLayer));

            // setup index exchange table
            int fbCount = 4;
            int fbList[4] = {0, 1, 2, 3};
            sharedNativeWindow.ResetExchangeTable(fbList, fbCount);

            // create window
            NVNwindow window;
            {
                NN_LOG("Creating window\n");
                NVNtexture* textureList[4] = {};
                for(int i = 0; i < fbCount; i++)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(sharedTexturePool.AcquireTexture(&textureList[i], fbList[i]));
                }

                NVNwindowBuilder builder;
                nvnWindowBuilderSetDevice(&builder, &device);
                nvnWindowBuilderSetDefaults(&builder);
                nvnWindowBuilderSetNativeWindow(&builder, sharedNativeWindow.GetNativeWindowHandle());
                nvnWindowBuilderSetPresentInterval(&builder, 1);
                nvnWindowBuilderSetTextures(&builder, fbCount, textureList);
                nvnWindowInitialize(&window, &builder);
            }

            for(int frame = 0; frame < FrameCount; frame++)
            {
                NNT_VI_SCOPED_SYNC(acqSync, device);
                int index = -1;
                nn::vi::fbshare::SharedLayerTextureIndexList indexList;
                while(sharedNativeWindow.PreAcquireTexture(&indexList, nn::TimeSpan::FromMilliSeconds(16)).IsFailure())
                {
                    NN_LOG("waiting for pre-acquire texture\n");
                }
                NN_ABORT_UNLESS_EQUAL(nvnWindowAcquireTexture(&window, &acqSync, &index), NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);
                NN_LOG("Frame %d / index = %d\n", frame, index);

                NVNtexture* pTargetTexture = nullptr;
                NN_ABORT_UNLESS_RESULT_SUCCESS(sharedTexturePool.AcquireTexture(&pTargetTexture, index));

                nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, sizeof(g_CommandStorage));
                nvnCommandBufferAddControlMemory(&commandBuffer, g_ControlStorage, sizeof(g_ControlStorage));
                nvnCommandBufferBeginRecording(&commandBuffer);

                // wait display
                nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

                // clear
                {
                    float angle = 0.5f * 2.f * 3.14f * frame / 60.f;
                    float r = 0.5f * std::cos(angle               ) + 0.5f;
                    float g = 0.5f * std::cos(angle + 3.14f       ) + 0.5f;
                    float b = 0.5f * std::cos(angle + 3.14f * 0.5f) + 0.5f;
                    float clearColor[4] = { r, g, b, 1.f };
                    NVNcopyRegion clearRegion;
                    clearRegion.xoffset = 0;
                    clearRegion.yoffset = 0;
                    clearRegion.zoffset = 0;
                    clearRegion.width  = nn::vi::fbshare::SharedFrameBufferWidth;
                    clearRegion.height = nn::vi::fbshare::SharedFrameBufferHeight;
                    clearRegion.depth  = 1;
                    nvnCommandBufferClearTexture(&commandBuffer, pTargetTexture, &texView, &clearRegion, clearColor, NVN_CLEAR_COLOR_MASK_RGBA);
                }

                // clear
                {
                    float origX = nn::vi::fbshare::SharedFrameBufferWidth / 2;
                    float origY = nn::vi::fbshare::SharedFrameBufferHeight / 2;
                    float length = 300;
                    float edge = 50;
                    float angle = 0.5f * 2.f * 3.14f * frame / 60.f;
                    float x = length * std::cos(angle) + origX;
                    float y = length * std::sin(angle) + origY;
                    float clearColor[4] = { 1.f, 1.f, 1.f, 1.f };
                    NVNcopyRegion clearRegion;
                    clearRegion.xoffset = x - edge * 0.5f;
                    clearRegion.yoffset = y - edge * 0.5f;
                    clearRegion.zoffset = 0;
                    clearRegion.width  = edge;
                    clearRegion.height = edge;
                    clearRegion.depth  = 1;
                    nvnCommandBufferClearTexture(&commandBuffer, pTargetTexture, &texView, &clearRegion, clearColor, NVN_CLEAR_COLOR_MASK_RGBA);
                }


                auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);

                NN_LOG("submitting\n");
                nvnQueueSubmitCommands(&queue, 1, &hCommand);
                nvnQueueFinish(&queue);

                NN_LOG("presenting\n");
                nvnQueuePresentTexture(&queue, &window, index);
            }

            context0.SafeDetachSharedLayerFromLowLevelLayer(hLayer);

            nvnWindowFinalize(&window);
            sharedNativeWindow.Finalize();
            sharedTexturePool.Finalize();
            sharedMemoryPool.Finalize();
        }

        context0.DisconnectService();
        context1.DisconnectService();
        context0.CleanSharedBuffer(SlotIndex);
    }
}// NOLINT(impl/function_size)
