﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <memory>
#include <random>
#include <list>
#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/vi/fbshare/vi_SharedLayerWindowImpl.h>

#include "testVi_NvnUtility.h"
#include "testVi_SceneUtility.h"

#define NNT_VI_LOG_MGR(...) NN_LOG("[mgr]" __VA_ARGS__)
#define NNT_VI_LOG_APT(format, ...) NN_LOG("[%s]" format, m_Name, __VA_ARGS__)

namespace {

    static const size_t AppletCommandMemorySize = 32 * 1024;
    static const size_t AppletControlMemorySize =  8 * 1024;

    enum RunMode
    {
        RunMode_PreAcquire,
        RunMode_CancelTexture,
    };

    enum MessageTag
    {
        MessageTag_Exit = 0,
        MessageTag_NotifyGoingToForeground,
        MessageTag_AcceptGoingToForeground,
        MessageTag_NotifyGoingToBackground,
        MessageTag_AcceptGoingToBackground,
    };

    struct Message
    {
    public:
        static Message MakeNotifyGoingToForeground(int lastFgIndex) NN_NOEXCEPT
        {
            Message msg = {};
            msg.tag = MessageTag_NotifyGoingToForeground;
            msg.value0 = lastFgIndex;
            return msg;
        }

        static Message MakeAcceptGoingToForeground() NN_NOEXCEPT
        {
            return {MessageTag_AcceptGoingToForeground};
        }

        static Message MakeNotifyGoingToBackground() NN_NOEXCEPT
        {
            return {MessageTag_NotifyGoingToBackground};
        }

        static Message MakeAcceptGoingToBackground() NN_NOEXCEPT
        {
            return {MessageTag_AcceptGoingToBackground};
        }

    public:
        MessageTag tag;
        int value0;
    };

    class MessageQueue
    {
    public:
        nn::os::MutexType  m_Mutex;
        nn::os::ConditionVariableType m_Condition;
        std::list<Message> m_MessageList;

    public:
        MessageQueue() NN_NOEXCEPT
        {
        }

        void Initialize() NN_NOEXCEPT
        {
            nn::os::InitializeMutex(&m_Mutex, false, 0);
            nn::os::InitializeConditionVariable(&m_Condition);
        }

        void Finalize() NN_NOEXCEPT
        {
            nn::os::FinalizeConditionVariable(&m_Condition);
            nn::os::FinalizeMutex(&m_Mutex);
        }

        void PushMessage(const Message& msg) NN_NOEXCEPT
        {
            {
                nn::os::LockMutex(&m_Mutex);
                NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
                m_MessageList.push_back(msg);
            }
            nn::os::SignalConditionVariable(&m_Condition);
        }

        bool TryPopMessage(Message* pOutMessage) NN_NOEXCEPT
        {
            {
                nn::os::LockMutex(&m_Mutex);
                NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
                if(m_MessageList.empty())
                {
                    return false;
                }
                *pOutMessage = m_MessageList.front();
                m_MessageList.pop_front();
            }
            nn::os::SignalConditionVariable(&m_Condition);
            return true;
        }

        void PopMessage(Message* pOutMessage) NN_NOEXCEPT
        {
            {
                nn::os::LockMutex(&m_Mutex);
                NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
                while(m_MessageList.empty())
                {
                    nn::os::WaitConditionVariable(&m_Condition, &m_Mutex);
                }
                *pOutMessage = m_MessageList.front();
                m_MessageList.pop_front();
            }
            nn::os::SignalConditionVariable(&m_Condition);
        }
    };

    struct MessageChannel
    {
        MessageQueue toApplet;
        MessageQueue toManager;
    };

    class PseudoApplet
    {
    public:
        explicit PseudoApplet(const nn::vi::ProxyName& proxyName) NN_NOEXCEPT
            : m_Context(proxyName)
        {
        }

        void Initialize(
            const char* name,
            RunMode runMode,
            size_t threadStackSize,
            int threadPriority,
            int threadCoreNumber,
            MessageChannel* pMessageChannel,
            NVNdevice* pDevice,
            nn::vi::fbshare::SharedBufferHandle bufferHandle,
            nn::vi::fbshare::SharedLayerHandle layerHandle
        ) NN_NOEXCEPT
        {
            m_Name = name;
            m_RunMode = runMode;

            // メインスレッド
            {
                m_MainThread.m_pThreadStackMemory = aligned_alloc(nn::os::ThreadStackAlignment, threadStackSize);
                NN_SDK_ASSERT_NOT_NULL(m_MainThread.m_pThreadStackMemory);
                nn::os::CreateThread(&m_MainThread.m_Thread, ThreadFunction, this, m_MainThread.m_pThreadStackMemory, threadStackSize, threadPriority, threadCoreNumber);
                m_MainThread.m_ThreadStackSize = threadStackSize;
                m_MainThread.m_ThreadPriority = threadPriority;
                m_MainThread.m_ThreadCoreNumber = threadCoreNumber;
            }

            m_FaderState.Initialize(name);

            m_pMessageChannel = pMessageChannel;
            m_pDevice = pDevice;
            m_BufferHandle = bufferHandle;
            m_LayerHandle = layerHandle;
        }

        void Finalize() NN_NOEXCEPT
        {
            // メインスレッド
            {
                nn::os::DestroyThread(&m_MainThread.m_Thread);
                free(m_MainThread.m_pThreadStackMemory);
            }
        }

        void Start() NN_NOEXCEPT
        {
            nn::os::StartThread(&m_MainThread.m_Thread);
        }

        void WaitForExit() NN_NOEXCEPT
        {
            nn::os::WaitThread(&m_MainThread.m_Thread);
        }

    private:
        static void ThreadFunction(void* p) NN_NOEXCEPT
        {
            reinterpret_cast<PseudoApplet*>(p)->ThreadFunctionImpl();
        }

        void ProcessMessage(bool* pOutIsExitRequested) NN_NOEXCEPT
        {
            *pOutIsExitRequested = false;

            Message msg = {};
            if(m_pMessageChannel->toApplet.TryPopMessage(&msg))
            {
                if(msg.tag == MessageTag_Exit)
                {
                    *pOutIsExitRequested = true;
                    return;
                }
                switch(msg.tag)
                {
                case MessageTag_NotifyGoingToForeground:
                    {
                        NNT_VI_LOG_APT("received NotifyGoingToForeground\n", 0);
                        NN_ASSERT(!m_HasForeground);
                        m_HasForeground = true;
                        NVNtexture* pLastFg = nullptr;
                        m_Window.GetSharedTexture(&pLastFg, msg.value0);
                        m_FaderState.Start(pLastFg);
                        m_pMessageChannel->toManager.PushMessage(Message::MakeAcceptGoingToForeground());
                        break;
                    }
                case MessageTag_NotifyGoingToBackground:
                    {
                        NNT_VI_LOG_APT("received NotifyGoingToBackground\n", 0);
                        NN_ASSERT(m_HasForeground);
                        m_HasForeground = false;
                        if(m_FaderState.IsActive())
                        {
                            m_FaderState.End();
                        }
                        m_pMessageChannel->toManager.PushMessage(Message::MakeAcceptGoingToBackground());
                        break;
                    }
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
        }

        void ThreadFunctionImpl() NN_NOEXCEPT
        {
            switch(m_RunMode)
            {
            case RunMode_PreAcquire:
                return ThreadFunctionImpl_PreAcquire();
            case RunMode_CancelTexture:
                return ThreadFunctionImpl_CancelTexture();
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        void ThreadFunctionImpl_PreAcquire() NN_NOEXCEPT
        {
            m_Context.ConnectService();
            NN_UTIL_SCOPE_EXIT{ m_Context.DisconnectService(); };

            m_Window.Initialize(m_Context.GetSystemService(), m_pDevice, m_BufferHandle, m_LayerHandle, 2, NVN_FORMAT_RGBA8_SRGB);
            NN_UTIL_SCOPE_EXIT{ m_Window.Finalize(); };

            auto& device = *m_pDevice;
            NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, m_CommandMemory, sizeof(m_CommandMemory));
            NNT_VI_SCOPED_QUEUE            (queue, device);
            NNT_VI_SCOPED_COMMANDBUFFER    (commandBuffer, device);
            NNT_VI_SCOPED_FRAMETEXTUREVIEW (texView);

            m_FrameCount = 0;
            m_HasForeground = false;
            for(;;)
            {
                bool isExitRequested = false;
                ProcessMessage(&isExitRequested);
                if(isExitRequested)
                {
                    break;
                }

                // FG でなければ飛ばす
                if(!m_HasForeground)
                {
                    NNT_VI_LOG_APT("no foreground\n", 0);
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
                    continue;
                }

                // バッファが取れるかチェック
                auto preAcqResult = m_Window.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(100));
                if(nn::vi::ResultNotReady::Includes(preAcqResult))
                {
                    // バッファが割り当てられていなければ飛ばす
                    NNT_VI_LOG_APT("no buffer\n", 0);
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
                    continue;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(preAcqResult);

                NNT_VI_SCOPED_SYNC(acqSync, device);
                int index = -1;
                NVNtexture* pTargetTexture = nullptr;
                m_Window.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);
                EXPECT_NE(nullptr, pTargetTexture);

                m_FrameCount++;
                NNT_VI_LOG_APT("frame %d\n", m_FrameCount);

                nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, AppletCommandMemorySize);
                nvnCommandBufferAddControlMemory(&commandBuffer, m_ControlMemory, AppletControlMemorySize);
                nvnCommandBufferBeginRecording(&commandBuffer);

                nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

                Draw(&commandBuffer, pTargetTexture, &texView);
                DrawFader(&commandBuffer, pTargetTexture, &texView);

                auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);
                nvnQueueSubmitCommands(&queue, 1, &hCommand);
                m_Window.PresentTexture(&queue, index);
                nvnQueueFinish(&queue);
                nvnSyncWait(&acqSync, NVN_WAIT_TIMEOUT_MAXIMUM);
            }
        }

        void ThreadFunctionImpl_CancelTexture() NN_NOEXCEPT
        {
            m_Context.ConnectService();
            NN_UTIL_SCOPE_EXIT{ m_Context.DisconnectService(); };

            m_Window.Initialize(m_Context.GetSystemService(), m_pDevice, m_BufferHandle, m_LayerHandle, 2, NVN_FORMAT_RGBA8_SRGB);
            NN_UTIL_SCOPE_EXIT{ m_Window.Finalize(); };

            auto& device = *m_pDevice;
            NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, m_CommandMemory, sizeof(m_CommandMemory));
            NNT_VI_SCOPED_QUEUE            (queue, device);
            NNT_VI_SCOPED_COMMANDBUFFER    (commandBuffer, device);
            NNT_VI_SCOPED_FRAMETEXTUREVIEW (texView);

            m_FrameCount = 0;
            m_HasForeground = false;
            for(;;)
            {
                bool isExitRequested = false;
                ProcessMessage(&isExitRequested);
                if(isExitRequested)
                {
                    break;
                }

                NNT_VI_SCOPED_SYNC(acqSync, device);
                int index = -1;
                NVNtexture* pTargetTexture = nullptr;
                if(m_Window.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(16)).IsFailure())
                {
                    NNT_VI_LOG_APT("failed to pre-acquire texture\n", 0);
                }
                m_Window.AcquirePreAcquiredTexture(&acqSync, &index, &pTargetTexture);

                // FG でなければ飛ばす
                if(!m_HasForeground)
                {
                    NNT_VI_LOG_APT("no foreground\n", 0);
                    m_Window.CancelTexture(index);
                    //nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // PreAcquireTexture で待っているはずなので Sleep しない
                    continue;
                }

                m_FrameCount++;
                NNT_VI_LOG_APT("frame %d\n", m_FrameCount);

                nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, AppletCommandMemorySize);
                nvnCommandBufferAddControlMemory(&commandBuffer, m_ControlMemory, AppletControlMemorySize);
                nvnCommandBufferBeginRecording(&commandBuffer);

                nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

                Draw(&commandBuffer, pTargetTexture, &texView);
                DrawFader(&commandBuffer, pTargetTexture, &texView);

                auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);
                nvnQueueSubmitCommands(&queue, 1, &hCommand);
                m_Window.PresentTexture(&queue, index);
                nvnQueueFinish(&queue);
                nvnSyncWait(&acqSync, NVN_WAIT_TIMEOUT_MAXIMUM);
            }
        }

        void Draw(
            NVNcommandBuffer* pCommandBuffer,
            NVNtexture* pTexture,
            NVNtextureView* pTexView
        ) NN_NOEXCEPT
        {
            if(pTexture == nullptr)
            {
                return;
            }
            float time = m_FrameCount * (1.f / 60.f);
            nnt::vi::scene::RotationBoxParameter param = {};
            param.SetPreset();
            nnt::vi::scene::DrawRotationBoxScene(pCommandBuffer, pTexture, pTexView, time);
        }

        void DrawFader(
            NVNcommandBuffer* pCommandBuffer,
            NVNtexture* pTexture,
            NVNtextureView* pTexView
        ) NN_NOEXCEPT
        {
            if(!m_FaderState.IsActive())
            {
                return;
            }
            if(pTexture == nullptr)
            {
                return;
            }

            float moveRatePerSec = 2;
            float time = m_FaderState.m_FadeFrameCount / 60.f;

            float dyRate = moveRatePerSec * time;
            float hRate = 1 - dyRate;

            if(hRate <= 0)
            {
                m_FaderState.End();
                return;
            }

            nnt::vi::scene::CopyTextureParameter param;
            param.dstX = 0;
            param.dstY = static_cast<int>(dyRate * nvnTextureGetHeight(pTexture));
            param.dstWidth  = nvnTextureGetWidth(pTexture);
            param.dstHeight = static_cast<int>(hRate * nvnTextureGetHeight(pTexture));
            param.pDstView = pTexView;
            param.srcX = 0;
            param.srcY = 0;
            param.srcWidth  = nvnTextureGetWidth(m_FaderState.m_pLastForegroundTexture);
            param.srcHeight = static_cast<int>(hRate * nvnTextureGetHeight(m_FaderState.m_pLastForegroundTexture));
            param.pSrcView = nullptr;
            nnt::vi::scene::CopyTexture(pCommandBuffer, pTexture, m_FaderState.m_pLastForegroundTexture, param);

            m_FaderState.m_FadeFrameCount++;
        }


    private:
        const char* m_Name;
        RunMode m_RunMode;

        struct {
            size_t m_ThreadStackSize;
            int m_ThreadPriority;
            int m_ThreadCoreNumber;
            nn::os::ThreadType m_Thread;
            void* m_pThreadStackMemory;
        } m_MainThread;

        MessageChannel* m_pMessageChannel;

        NVNdevice* m_pDevice;

        Context m_Context;
        nn::vi::fbshare::SharedBufferHandle m_BufferHandle;
        nn::vi::fbshare::SharedLayerHandle  m_LayerHandle;
        nn::vi::fbshare::SharedLayerWindowImpl m_Window;

        struct {
        public:
            void Initialize(const char* name) NN_NOEXCEPT
            {
                m_Name = name;
                m_pLastForegroundTexture = nullptr;
                m_FadeFrameCount = -1;
            }

            void Start(NVNtexture* pLastFg) NN_NOEXCEPT
            {
                NNT_VI_LOG_APT("fader start\n", 0);
                NN_ASSERT(m_pLastForegroundTexture == nullptr);
                m_pLastForegroundTexture = pLastFg;
                m_FadeFrameCount = 0;
            }

            void End() NN_NOEXCEPT
            {
                NNT_VI_LOG_APT("fader end\n", 0);
                NN_ASSERT_NOT_NULL(m_pLastForegroundTexture);
                m_pLastForegroundTexture = nullptr;
                m_FadeFrameCount = -1;
            }

            bool IsActive() const NN_NOEXCEPT
            {
                return m_pLastForegroundTexture != nullptr;
            }

        public:
            const char* m_Name;
            NVNtexture* m_pLastForegroundTexture;
            int m_FadeFrameCount;
        } m_FaderState;

        bool m_HasForeground;

        int m_FrameCount;
        NN_ALIGNAS(4096) char m_CommandMemory[AppletCommandMemorySize];
        NN_ALIGNAS(4096) char m_ControlMemory[AppletControlMemorySize];
    };

}

class TestTransitionState
{
public:
    static const int SharedBufferSlotIndex = 0;
    static const int SharedBufferFrameBufferCount = 6;

    static const int LastForegroundBufferIndex = 0;
    static const int FrontBufferIndex0 = 2;
    static const int FrontBufferIndex1 = 3;

    static const int AppletCount = 2;

    static const size_t PseudoAppletStackSize = 32 * 1024;
public:
    std::shared_ptr<PseudoApplet> m_pApplet0;
    std::shared_ptr<PseudoApplet> m_pApplet1;

    nn::vi::LayerId m_LowLayerId;

    ContextExt m_Context;
    nn::vi::fbshare::SharedBufferHandle m_hBuffer;
    nn::vi::fbshare::SharedLayerHandle m_hLayer[AppletCount];
    nn::vi::LayerStackFlagType m_Stacks[AppletCount];
    int m_CurrentApplet;

    MessageChannel m_MessageChannel[AppletCount];

public:
    TestTransitionState() NN_NOEXCEPT
        : m_Context({"manager"})
    {
    }

    ~TestTransitionState() NN_NOEXCEPT
    {
    }

    void Run(RunMode runMode) NN_NOEXCEPT
    {
        NNT_VI_LOG_MGR("init\n");
        auto& context = m_Context;
        context.ConnectService();
        NN_UTIL_SCOPE_EXIT{ context.DisconnectService(); };

        // setup SharedBuffer
        auto hBuffer = context.CreateSharedBuffer(SharedBufferSlotIndex, SharedBufferFrameBufferCount);
        NN_UTIL_SCOPE_EXIT{ context.DestroySharedBuffer(SharedBufferSlotIndex, hBuffer); };
        m_hBuffer = hBuffer;
        // export SharedBuffer to me
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.RegisterSharedBufferImporterAruid(hBuffer, nn::applet::GetAppletResourceUserId()));
        NN_UTIL_SCOPE_EXIT{ context.UnregisterSharedBufferImporterAruid(hBuffer, nn::applet::GetAppletResourceUserId()); };

        // create SharedLayer;
        nn::vi::fbshare::SharedLayerHandle hLayer0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateSharedLayer(&hLayer0, nn::applet::GetAppletResourceUserId()));
        nn::vi::fbshare::SharedLayerHandle hLayer1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateSharedLayer(&hLayer1, nn::applet::GetAppletResourceUserId()));
        m_hLayer[0] = hLayer0;
        m_hLayer[1] = hLayer1;

        nn::vi::LayerStackFlagType stacks0 = (1 << nn::vi::LayerStack_Default);
        nn::vi::LayerStackFlagType stacks1 = (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_Screenshot);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.SetSharedLayerLayerStack(hLayer0, stacks0));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.SetSharedLayerLayerStack(hLayer1, stacks1));
        m_Stacks[0] = stacks0;
        m_Stacks[1] = stacks1;

        // create message channel
        for(int i = 0; i < AppletCount; i++)
        {
            m_MessageChannel[i].toApplet.Initialize();
            m_MessageChannel[i].toManager.Initialize();
        }
        NN_UTIL_SCOPE_EXIT{
            for(int i = 0; i < AppletCount; i++)
            {
                m_MessageChannel[i].toApplet.Finalize();
                m_MessageChannel[i].toManager.Finalize();
            }
        };


        NNT_VI_SCOPED_DEVICE(device);
        m_pApplet0 = std::make_shared<PseudoApplet>(nn::vi::ProxyName({"app0"}));
        m_pApplet1 = std::make_shared<PseudoApplet>(nn::vi::ProxyName({"app1"}));

        m_pApplet0->Initialize("app0", runMode, PseudoAppletStackSize, nn::os::DefaultThreadPriority, 1, &m_MessageChannel[0], &device, hBuffer, hLayer0);
        m_pApplet1->Initialize("app1", runMode, PseudoAppletStackSize, nn::os::DefaultThreadPriority, 2, &m_MessageChannel[1], &device, hBuffer, hLayer1);

        // 開始
        m_pApplet0->Start();
        m_pApplet1->Start();

        // SharedBuffer をデフォルトレイヤに接続
        auto layerId = context.GetDefaultLayerId();
        m_LowLayerId = layerId;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.BindLowLevelLayerToManagedLayer(layerId));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

        m_CurrentApplet = -1;
        // 最初は Applet0 に。
        AttachBuffer(0);

        for(int t = 0; t < 10; t++)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            int next = (m_CurrentApplet + 1) % AppletCount;

            DetachBuffer();
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            AttachBuffer(next);
        }


        NNT_VI_LOG_MGR("finalize\n");
        // 終了
        {
            Message msg = {};
            msg.tag = MessageTag_Exit;
            for(int i = 0; i < AppletCount; i++)
            {
                m_MessageChannel[i].toApplet.PushMessage(msg);
            }
            m_pApplet0->WaitForExit();
            m_pApplet1->WaitForExit();
        }

        m_pApplet0->Finalize();
        m_pApplet1->Finalize();

        NNT_VI_LOG_MGR("done\n");
    }// NOLINT(impl/function_size)

    void AttachBuffer(int appletIndex) NN_NOEXCEPT
    {
        NN_ASSERT(m_CurrentApplet < 0);
        NN_ASSERT_RANGE(appletIndex, 0, static_cast<int>(AppletCount));
        m_CurrentApplet = appletIndex;

        // FG に行くことを通知
        {
            m_MessageChannel[m_CurrentApplet].toApplet.PushMessage(Message::MakeNotifyGoingToForeground(LastForegroundBufferIndex));
            {
                Message msg = {};
                m_MessageChannel[m_CurrentApplet].toManager.PopMessage(&msg);
                NN_ABORT_UNLESS_EQUAL(msg.tag, MessageTag_AcceptGoingToForeground);
            }
        }

        // アタッチ
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Context.AttachSharedLayerToLowLevelLayer(m_hLayer[appletIndex], m_LowLayerId, FrontBufferIndex0, FrontBufferIndex1));
    }

    void DetachBuffer() NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(m_CurrentApplet, 0, static_cast<int>(AppletCount));
        auto& context = m_Context;

        // デタッチ
        {
            context.SafeDetachSharedLayerFromLowLevelLayer(m_hLayer[m_CurrentApplet]);

            // 最終出画フレームをコピー
            int displayIndex = -1;
            context.SynchronizeLowLevelLayer(&displayIndex, m_LowLayerId);
            if(displayIndex < 0)
            {
                NNT_VI_LOG_MGR("no buffer is displayed\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.FillDetachedBufferColor(m_hBuffer, LastForegroundBufferIndex, 0, 0, 0, 255));
            }
            else
            {
                NNT_VI_LOG_MGR("copy buffer %d to %d\n", displayIndex, LastForegroundBufferIndex);
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.CopyDetachedBufferImage(m_hBuffer, LastForegroundBufferIndex, displayIndex));
            }

            // コピー先バッファのレイヤスタックを確認
            {
                nn::vi::LayerStackFlagType stacks = {};
                nn::vi::CropRegion crop = {};
                int32_t scaling = 0;
                uint32_t transform = 0;
                int32_t presentInt = 0;
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSharedFrameBufferContentParameter(&stacks, &crop, &scaling, &transform, &presentInt, m_hBuffer, LastForegroundBufferIndex));
                EXPECT_EQ(m_Stacks[m_CurrentApplet], stacks);
            }

            // コピー先を遷移レイヤとして表示
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.PresentDetachedBufferToLowLevelLayer(m_hBuffer, m_LowLayerId, LastForegroundBufferIndex));
        }

        // BG に行くことを通知
        {
            m_MessageChannel[m_CurrentApplet].toApplet.PushMessage(Message::MakeNotifyGoingToBackground());
            {
                Message msg = {};
                m_MessageChannel[m_CurrentApplet].toManager.PopMessage(&msg);
                NN_ABORT_UNLESS_EQUAL(msg.tag, MessageTag_AcceptGoingToBackground);
            }
        }

        m_CurrentApplet = -1;
    }



};

NNT_VI_TEST_INTEGRATED(Transition_PreAcquire)
{
    auto p = std::make_shared<TestTransitionState>();

    NN_LOG("------------------------------------\n");
    NN_LOG("  RUNMODE: PreAcquire\n");
    NN_LOG("------------------------------------\n");
    p->Run(RunMode_PreAcquire);
}

NNT_VI_TEST_INTEGRATED(Transition_CancelTexture)
{
    auto p = std::make_shared<TestTransitionState>();

    NN_LOG("------------------------------------\n");
    NN_LOG("  RUNMODE: CancelTexture\n");
    NN_LOG("------------------------------------\n");
    p->Run(RunMode_CancelTexture);
}
