﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <string>
#include <sstream>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt.h>

#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"


namespace {

    static const bool isForcedPrintThumb = false;

    uint8_t GetChannelValue(int x, int length, uint8_t colorMax) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER(length, 0);
        float v = std::floorf(static_cast<float>(colorMax) * static_cast<float>(x) / static_cast<float>(length - 1));
        if(v < 0)
        {
            return v;
        }
        if(v > 255)
        {
            return 255;
        }
        return static_cast<uint8_t>(v);
    }

    std::vector<uint8_t> GetTestImageBuffer(int x, int y, int w, int h, int width, int height, nn::vi::ImageTransform transform, uint8_t bgColor, uint8_t bgAlpha, uint8_t colorMax, uint8_t alpha) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(x, 0, width);
        NN_ASSERT_RANGE(y, 0, height);
        NN_ASSERT_MINMAX(w, 0, width);
        NN_ASSERT_MINMAX(h, 0, height);
        NN_ASSERT_MINMAX(x + w, 0, width);
        NN_ASSERT_MINMAX(y + h, 0, height);
        std::vector<uint8_t> out;
        out.resize(4 * width * height);

        // 背景色
        for(int y = 0; y < height; y++)
        {
            for(int x = 0; x < width; x++)
            {
                uint8_t* p = out.data() + 4 * (x + (width * y));
                p[0] = bgColor;
                p[1] = bgColor;
                p[2] = bgColor;
                p[3] = bgAlpha;
            }
        }

        // 書込み部分
        uint8_t b = 0;
        for(int dy = 0; dy < h; dy++)
        {
            int py = y + dy;
            uint8_t g = GetChannelValue(
                (transform == nn::vi::ImageTransform_YFlip) ? (h - 1 - dy) : dy,
                h,
                colorMax
            );

            for(int dx = 0; dx < w; dx++)
            {
                int px = x + dx;
                uint8_t r = GetChannelValue(dx, w, colorMax);

                uint8_t* p = out.data() + 4 * (px + (width * py));
                p[0] = r;
                p[1] = g;
                p[2] = b;
                p[3] = alpha;
            }
        }

        return out;
    }

    void CheckImageBuffer(
        const std::vector<uint8_t>& a,
        const std::vector<uint8_t>& b,
        int width,
        int height,
        int tolerance
    ) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_EQUAL(a.size(), 4 * width * height);
        NN_ABORT_UNLESS_EQUAL(b.size(), 4 * width * height);

        bool isOk = true;
        for(size_t i = 0; i < a.size(); i++)
        {
            if(std::labs(a[i] - b[i]) > tolerance)
            {
                isOk = false;
                NN_LOG("check img buffer: diff = %d (i=%lld)\n", static_cast<int>(a[i] - b[i]), i);
                break;
            }
        }

        if(!isOk || isForcedPrintThumb)
        {
            static const int ThumbDeltaX = 100;
            static const int ThumbDeltaY = 100;
            auto addChar = [](std::stringstream& ss, uint8_t r, uint8_t g)
            {
                if(r < 128)
                {
                    if(g < 128)
                    {
                        ss << ".";
                    }
                    else
                    {
                        ss << "|";
                    }
                }
                else
                {
                    if(g < 128)
                    {
                        ss << "-";
                    }
                    else
                    {
                        ss << "+";
                    }
                }
            };

            NN_LOG("expected image:\n");
            for(int y = 0; y < height; y += ThumbDeltaY)
            {
                std::stringstream ss;
                for(int x = 0; x < width; x += ThumbDeltaX)
                {
                    auto* p = a.data() + 4 * (x + y * width);
                    addChar(ss, p[0], p[1]);
                }

                auto str = ss.str();
                NN_LOG("  %s\n", str.c_str());
            }

            NN_LOG("actual image:\n");
            for(int y = 0; y < height; y += ThumbDeltaY)
            {
                std::stringstream ss;
                for(int x = 0; x < width; x += ThumbDeltaX)
                {
                    auto* p = b.data() + 4 * (x + y * width);
                    addChar(ss, p[0], p[1]);
                }

                auto str = ss.str();
                NN_LOG("  %s\n", str.c_str());
            }
        }

        EXPECT_TRUE(isOk);
    }

    void CheckContentParameter(
        ContextExt& context,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        int index,
        nn::vi::LayerStackFlagType expectedStacks,
        const nn::vi::CropRegion& expectedCrop,
        uint32_t expectedTransform,
        int32_t expectedPresentInterval
    ) NN_NOEXCEPT
    {
        nn::vi::LayerStackFlagType stacks = 0;
        nn::vi::CropRegion crop = {};
        int32_t scalingMode = 0;
        uint32_t transform = 0;
        int32_t presentInterval = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSharedFrameBufferContentParameter(&stacks, &crop, &scalingMode, &transform, &presentInterval, hBuffer, index));
        EXPECT_EQ(expectedStacks, stacks);
        if(std::memcmp(&expectedCrop, &crop, sizeof(nn::vi::CropRegion)) != 0)
        {
            NN_LOG("  expected crop = {%d, %d; %d, %d}\n", expectedCrop.x, expectedCrop.y, expectedCrop.width, expectedCrop.height);
            NN_LOG("  actual   crop = {%d, %d; %d, %d}\n", crop.x, crop.y, crop.width, crop.height);
            EXPECT_EQ(expectedCrop.x, crop.x);
            EXPECT_EQ(expectedCrop.y, crop.y);
            EXPECT_EQ(expectedCrop.width, crop.width);
            EXPECT_EQ(expectedCrop.height, crop.height);
        }
        EXPECT_EQ(0, scalingMode);
        EXPECT_EQ(expectedTransform, transform);
        EXPECT_EQ(expectedPresentInterval, presentInterval);
    }
}

// テストの実行関数
namespace {

    struct TestSetSubImage
    {
        static const int Width = nn::vi::fbshare::SharedFrameBufferWidth;
        static const int Height = nn::vi::fbshare::SharedFrameBufferHeight;
        static const int FrameBufferCount = 4;
        static const int SlotIndex = 0;
        static const nn::vi::LayerStackFlagType LayerStacks = nn::vi::ValidLayerStackFlags;
        static const int DestTextureIndex = 0;

        struct Parameter
        {
        public:
            Parameter() NN_NOEXCEPT
            {
                this->srcTransform = nn::vi::ImageTransform_None;
                this->dstTransform = nn::vi::ImageTransform_None;

                this->srcAlpha     = 255;
                this->alphaOption  = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
                this->colorOption  = nn::vi::fbshare::SharedTextureColorOption_None;

                this->subX = 0;
                this->subY = 0;
                this->subW = Width;
                this->subH = Height;
                this->bgColor = 0;
                this->bgAlpha = 255;
            }

        public:
            nn::vi::ImageTransform srcTransform;
            nn::vi::ImageTransform dstTransform;

            uint8_t srcAlpha;
            nn::vi::fbshare::SharedTextureAlphaOption alphaOption;
            nn::vi::fbshare::SharedTextureColorOption colorOption;

            int subX;
            int subY;
            int subW;
            int subH;
            uint8_t bgColor;
            uint8_t bgAlpha;
        };

        static void Test(Parameter& param) NN_NOEXCEPT
        {
            ContextExt context0({"server"});
            context0.ConnectService();

            // setup SharedBuffer
            auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);

            auto srcImg    = GetTestImageBuffer(0, 0, param.subW, param.subH, param.subW, param.subH, param.srcTransform, 0xCC, 0xCD, 255, param.srcAlpha);
            auto expectImg = GetTestImageBuffer(
                param.subX,
                param.subY,
                param.subW,
                param.subH,
                Width,
                Height,
                nn::vi::ImageTransform_None, // 最終的に読込む画像は左上原点になる
                (param.colorOption == nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha) ? (param.bgAlpha * param.bgColor) / 256 : param.bgColor,
                (param.alphaOption == nn::vi::fbshare::SharedTextureAlphaOption_Opaque) ? 255 : param.bgAlpha,
                (param.colorOption == nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha) ? param.srcAlpha : 255,
                (param.alphaOption == nn::vi::fbshare::SharedTextureAlphaOption_Opaque) ? 255 : param.srcAlpha
            );

            // 書込み
            nn::vi::fbshare::SharedTextureOption dstOption = {};
            dstOption.alphaOption = param.alphaOption;
            dstOption.colorOption = param.colorOption;
            dstOption.transform   = param.dstTransform;
            dstOption.stacks      = LayerStacks;
            nn::vi::ImageTransform srcTransform = {};
            srcTransform = param.srcTransform;

            uint32_t bgColor = 0;
            bgColor |= (param.bgColor << 0);
            bgColor |= (param.bgColor << 8);
            bgColor |= (param.bgColor << 16);
            bgColor |= (param.bgAlpha << 24);

            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetDetachedSharedBufferSubImage(
                hBuffer,
                DestTextureIndex,
                param.subX,
                param.subY,
                param.subW,
                param.subH,
                bgColor,
                srcImg.data(),
                srcImg.size(),
                dstOption,
                srcTransform
            ));

            // 読み出し
            std::vector<uint8_t> dstImg;
            dstImg.resize(expectImg.size());
            size_t imgSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetDetachedSharedBufferImage(&imgSize, dstImg.data(), dstImg.size(), hBuffer, DestTextureIndex));

            // 許容誤差
            int tol = 0;
            if(param.colorOption == nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha)
            {
                tol = 1;
            }

            // チェック
            CheckImageBuffer(expectImg, dstImg, Width, Height, tol);
            CheckContentParameter(context0, hBuffer, DestTextureIndex, LayerStacks, { 0, 0, 0, 0 }, param.dstTransform, 1);

            context0.DisconnectService();
            context0.CleanSharedBuffer(SlotIndex);
        }
    };

    struct PartialRectangle
    {
        int x;
        int y;
        int w;
        int h;
    };

    std::vector<PartialRectangle> GetPartialRectList() NN_NOEXCEPT
    {
        std::vector<PartialRectangle> v;
        v.push_back({   0,   0, 400, 220});
        v.push_back({ 880,   0, 400, 220});
        v.push_back({   0, 500, 400, 220});
        v.push_back({ 880, 500, 400, 220});
        return v;
    }

    std::vector<std::pair<uint8_t, uint8_t>> GetBgColorList() NN_NOEXCEPT
    {
        std::vector<std::pair<uint8_t, uint8_t>> v;
        v.push_back({   0, 255});
        v.push_back({   0,   0});
        v.push_back({ 255, 255});
        v.push_back({ 128, 128});
        return v;
    }

    template<typename ParamModificationFunction>
    void TestForeachSubImageRegion(ParamModificationFunction f)
    {
        {
            NN_LOG("full\n");
            TestSetSubImage::Parameter p = {};
            f(p);
            TestSetSubImage::Test(p);
        }
        for(auto& bgc : GetBgColorList())
        {
            for(auto& rect : GetPartialRectList())
            {
                NN_LOG("sub{%d,%d;%d,%d}, bg{%d,%d}\n", rect.x, rect.y, rect.w, rect.h, bgc.first, bgc.second);
                TestSetSubImage::Parameter p = {};
                p.subX = rect.x;
                p.subY = rect.y;
                p.subW = rect.w;
                p.subH = rect.h;
                p.bgColor = bgc.first;
                p.bgAlpha = bgc.second;
                f(p);
                TestSetSubImage::Test(p);
            }
        }
    }
}

NNT_VI_TEST_FUNCTIONAL(SetSubImage_Default)
{
    TestForeachSubImageRegion([](TestSetSubImage::Parameter&){});
}

NNT_VI_TEST_FUNCTIONAL(SetSubImage_SourceYFlip)
{
    TestForeachSubImageRegion([](TestSetSubImage::Parameter& p){
        p.srcTransform = nn::vi::ImageTransform_YFlip;
    });
}

NNT_VI_TEST_FUNCTIONAL(SetSubImage_DestinationYFlip)
{
    TestForeachSubImageRegion([](TestSetSubImage::Parameter& p){
        p.dstTransform = nn::vi::ImageTransform_YFlip;
    });
}

NNT_VI_TEST_FUNCTIONAL(SetSubImage_SourceDestinationYFlip)
{
    TestForeachSubImageRegion([](TestSetSubImage::Parameter& p){
        p.srcTransform = nn::vi::ImageTransform_YFlip;
        p.dstTransform = nn::vi::ImageTransform_YFlip;
    });
}

NNT_VI_TEST_FUNCTIONAL(SetSubImage_SourceAlpha)
{
    TestForeachSubImageRegion([](TestSetSubImage::Parameter& p){
        p.srcAlpha = 100;
        p.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_None;
    });
}

NNT_VI_TEST_FUNCTIONAL(SetSubImage_Opaque)
{
    TestForeachSubImageRegion([](TestSetSubImage::Parameter& p){
        p.srcAlpha = 100;
        p.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
    });
}

NNT_VI_TEST_FUNCTIONAL(SetSubImage_SourceColor)
{
    TestForeachSubImageRegion([](TestSetSubImage::Parameter& p){
        p.srcAlpha = 100;
        p.colorOption = nn::vi::fbshare::SharedTextureColorOption_None;
    });
}

NNT_VI_TEST_FUNCTIONAL(SetSubImage_PreMultipledAlpha)
{
    TestForeachSubImageRegion([](TestSetSubImage::Parameter& p){
        p.srcAlpha = 100;
        p.colorOption = nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha;
    });
}

