﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include <nnt.h>

#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"

static const int DefaultLoopCount  = 10;
static const int DefaultFrameCount = 10;

NNT_VI_TEST_FUNCTIONAL(Loop_CLAcquire_CLPresent)
{
    static const int LoopCount = DefaultLoopCount;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;
    static const int FrameCount = DefaultFrameCount;
    NN_LOG("CLAcquire>CLPresent-------------------------------\n");

    ContextExt context0({"server"});
    Context context1({"client"});

    context0.ConnectService();
    context1.ConnectService();

    // setup SharedLowLevelLayer
    auto layerId = context0.GetDefaultLayerId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

    // setup SharedBuffer
    auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

    // setup SharedLayer
    nn::vi::fbshare::SharedLayerHandle hLayer = {};
    nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

    // attach
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));

    // open SharedLayer
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.OpenSharedLayer(hLayer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.ConnectSharedLayer(hLayer));
    nn::os::SystemEventType acquirableEvent = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedFrameBufferAcquirableEvent(&acquirableEvent, hLayer));
    NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&acquirableEvent); };


    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);

        for(int frame = 0; frame < FrameCount; frame++)
        {
            nn::os::WaitSystemEvent(&acquirableEvent);
            int index = -1;
            nn::util::Color4u8 color;
            color.SetR(static_cast<int>(128 * std::sin(0.0314 * frame / 3) + 128));
            color.SetG(static_cast<int>(128 * std::sin(0.0628 * frame / 3) + 128));
            color.SetB(static_cast<int>(128 * std::sin(0.0157 * frame / 3) + 128));
            color.SetA(255);

            nn::vi::native::NativeSync dispSync = {};
            nn::vi::native::NativeSync appSync = {};

            nn::vi::fbshare::SharedLayerTextureIndexList indexList;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&index, &dispSync, &indexList, hLayer));
            NN_LOG("    sync %d-%d | %d-%d | %d-%d | %d-%d\n",
                reinterpret_cast<int&>(dispSync._data[4 * 1]), reinterpret_cast<int&>(dispSync._data[4 * 2]),
                reinterpret_cast<int&>(dispSync._data[4 * 3]), reinterpret_cast<int&>(dispSync._data[4 * 4]),
                reinterpret_cast<int&>(dispSync._data[4 * 5]), reinterpret_cast<int&>(dispSync._data[4 * 6]),
                reinterpret_cast<int&>(dispSync._data[4 * 7]), reinterpret_cast<int&>(dispSync._data[4 * 8])
            );
            nnt::vi::WaitSync(dispSync);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.FillSharedFrameBufferColor(hLayer, index, color));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.PresentSharedFrameBuffer(hLayer, index, appSync));
        }
    }

    context0.DisconnectService();
    context1.DisconnectService();
    context0.CleanSharedBuffer(SlotIndex);
}

NNT_VI_TEST_FUNCTIONAL(Loop_CLAcquire_CLCancel)
{
    static const int LoopCount = DefaultLoopCount;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;
    static const int FrameCount = DefaultFrameCount;
    static const int CancelCount = 4;
    NN_LOG("CLAcquire>CLCancel-------------------------------\n");

    ContextExt context0({"server"});
    Context context1({"client"});

    context0.ConnectService();
    context1.ConnectService();

    // setup SharedLowLevelLayer
    auto layerId = context0.GetDefaultLayerId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

    // setup SharedBuffer
    auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

    // setup SharedLayer
    nn::vi::fbshare::SharedLayerHandle hLayer = {};
    nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

    // attach
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));

    // open SharedLayer
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.OpenSharedLayer(hLayer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.ConnectSharedLayer(hLayer));
    nn::os::SystemEventType acquirableEvent = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedFrameBufferAcquirableEvent(&acquirableEvent, hLayer));
    NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&acquirableEvent); };


    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);

        for(int frame = 0; frame < FrameCount; frame++)
        {
            nn::os::WaitSystemEvent(&acquirableEvent);
            int index = -1;
            nn::util::Color4u8 color;
            color.SetR(static_cast<int>(128 * std::sin(0.0314 * frame / 3) + 128));
            color.SetG(static_cast<int>(128 * std::sin(0.0628 * frame / 3) + 128));
            color.SetB(static_cast<int>(128 * std::sin(0.0157 * frame / 3) + 128));
            color.SetA(255);

            nn::vi::native::NativeSync dispSync = {};
            nn::vi::native::NativeSync appSync = {};

            // まず最初に Acquire
            nn::vi::fbshare::SharedLayerTextureIndexList indexList;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&index, &dispSync, &indexList, hLayer));
            NN_LOG("    sync %d-%d | %d-%d | %d-%d | %d-%d\n",
                reinterpret_cast<int&>(dispSync._data[4 * 1]), reinterpret_cast<int&>(dispSync._data[4 * 2]),
                reinterpret_cast<int&>(dispSync._data[4 * 3]), reinterpret_cast<int&>(dispSync._data[4 * 4]),
                reinterpret_cast<int&>(dispSync._data[4 * 5]), reinterpret_cast<int&>(dispSync._data[4 * 6]),
                reinterpret_cast<int&>(dispSync._data[4 * 7]), reinterpret_cast<int&>(dispSync._data[4 * 8])
            );

            // Cancel と Acquire を繰り返す
            int calcelIndex = index;
            for(int c = 0; c < CancelCount; c++)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(context1.CancelSharedFrameBuffer(hLayer, calcelIndex));

                int reacqIndex = -1;
                nn::vi::native::NativeSync reacqSync = {};
                nn::vi::fbshare::SharedLayerTextureIndexList reacqIndexList;
                nn::os::WaitSystemEvent(&acquirableEvent);
                NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&reacqIndex, &reacqSync, &reacqIndexList, hLayer));

                EXPECT_EQ(calcelIndex, reacqIndex);
            }

            nnt::vi::WaitSync(dispSync);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.FillSharedFrameBufferColor(hLayer, index, color));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.PresentSharedFrameBuffer(hLayer, index, appSync));
        }
    }

    context0.DisconnectService();
    context1.DisconnectService();
    context0.CleanSharedBuffer(SlotIndex);
}

NNT_VI_TEST_FUNCTIONAL(CLCancel_CheckAcquirableEvent)
{
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 2;
    static const int CancelCount = 8;

    ContextExt context0({"server"});
    Context context1({"client"});

    context0.ConnectService();
    context1.ConnectService();

    // setup SharedLowLevelLayer
    auto layerId = context0.GetDefaultLayerId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

    // setup SharedBuffer
    auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

    // setup SharedLayer
    nn::vi::fbshare::SharedLayerHandle hLayer = {};
    nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

    // attach
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1));

    // open SharedLayer
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.OpenSharedLayer(hLayer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.ConnectSharedLayer(hLayer));
    nn::os::SystemEventType acquirableEvent = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedFrameBufferAcquirableEvent(&acquirableEvent, hLayer));
    NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&acquirableEvent); };


    // 1 回目でチェック。
    // それ以降は 2 枚バッファがないと回らない。
    {
        bool isLastBuffer = true;

        // 最終バッファになるように 1 枚残して Acquire しておく
        int keptIndex = -1;
        NN_LOG("  isLastBuffer = %s\n", isLastBuffer ? "yes" : "no");
        if(isLastBuffer)
        {
            nn::vi::native::NativeSync sync;
            nn::vi::fbshare::SharedLayerTextureIndexList indexList;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&keptIndex, &sync, &indexList, hLayer));
        }

        {
            nn::os::WaitSystemEvent(&acquirableEvent);
            int index = -1;
            nn::util::Color4u8 color = { 128, 128, 128, 255 };;

            nn::vi::native::NativeSync dispSync = {};
            nn::vi::native::NativeSync appSync = {};

            // まず最初に Acquire
            nn::vi::fbshare::SharedLayerTextureIndexList indexList;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&index, &dispSync, &indexList, hLayer));
            NN_LOG("    sync %d-%d | %d-%d | %d-%d | %d-%d\n",
                reinterpret_cast<int&>(dispSync._data[4 * 1]), reinterpret_cast<int&>(dispSync._data[4 * 2]),
                reinterpret_cast<int&>(dispSync._data[4 * 3]), reinterpret_cast<int&>(dispSync._data[4 * 4]),
                reinterpret_cast<int&>(dispSync._data[4 * 5]), reinterpret_cast<int&>(dispSync._data[4 * 6]),
                reinterpret_cast<int&>(dispSync._data[4 * 7]), reinterpret_cast<int&>(dispSync._data[4 * 8])
            );

            // Cancel と Acquire を繰り返す
            int calcelIndex = index;
            for(int c = 0; c < CancelCount; c++)
            {
                EXPECT_EQ(!isLastBuffer, nn::os::TryWaitSystemEvent(&acquirableEvent));

                NN_ABORT_UNLESS_RESULT_SUCCESS(context1.CancelSharedFrameBuffer(hLayer, calcelIndex));
                EXPECT_TRUE(nn::os::TryWaitSystemEvent(&acquirableEvent));

                int reacqIndex = -1;
                nn::vi::native::NativeSync reacqSync = {};
                nn::vi::fbshare::SharedLayerTextureIndexList reacqIndexList;
                NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&reacqIndex, &reacqSync, &reacqIndexList, hLayer));
                EXPECT_EQ(!isLastBuffer, nn::os::TryWaitSystemEvent(&acquirableEvent));

                EXPECT_EQ(calcelIndex, reacqIndex);
            }

            nnt::vi::WaitSync(dispSync);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.FillSharedFrameBufferColor(hLayer, index, color));
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.PresentSharedFrameBuffer(hLayer, index, appSync));
        }

        // キープしておいたバッファを返却
        if(keptIndex >= 0)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.CancelSharedFrameBuffer(hLayer, keptIndex));
        }
    }

    context0.DisconnectService();
    context1.DisconnectService();
    context0.CleanSharedBuffer(SlotIndex);
}

NNT_VI_TEST_FUNCTIONAL(CLCancel_CheckDetachReadyEvent)
{
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;

    ContextExt context0({"server"});
    Context context1({"client"});

    context0.ConnectService();
    context1.ConnectService();

    // setup SharedLowLevelLayer
    auto layerId = context0.GetDefaultLayerId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

    // setup SharedBuffer
    auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

    // setup SharedLayer
    nn::vi::fbshare::SharedLayerHandle hLayer = {};
    nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

    // attach
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));
    nn::os::SystemEventType detachReadyEvent = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetSharedLayerDetachReadyEvent(&detachReadyEvent, hLayer));
    NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&detachReadyEvent); };

    // open SharedLayer
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.OpenSharedLayer(hLayer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.ConnectSharedLayer(hLayer));
    nn::os::SystemEventType acquirableEvent = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedFrameBufferAcquirableEvent(&acquirableEvent, hLayer));
    NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&acquirableEvent); };

    {
        // 全バッファ Acquire
        NN_LOG("  acquiring all buffers by client\n");
        for(int i = 0; i < FrameBufferCount; i++)
        {
            int index;
            nn::vi::native::NativeSync sync;
            nn::vi::fbshare::SharedLayerTextureIndexList indexList;
            NN_LOG("  acquiring (%d/%d)\n", i + 1, FrameBufferCount);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&index, &sync, &indexList, hLayer));
        }

        // デタッチ開始
        NN_LOG("  request safe-detach\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.StartDetachSharedLayerFromLowLevelLayer(hLayer));

        // 1 枚ずつバッファを返却
        for(int i = 0; i < FrameBufferCount; i++)
        {
            // 返却前はデタッチ準備できていない
            NN_LOG("  check safe-detach is not ready\n");
            EXPECT_FALSE(nn::os::TryWaitSystemEvent(&detachReadyEvent));

            // ちなみに返却する順番は acquire した順序の逆順にする必要はない。
            NN_LOG("  canceling (%d/%d)\n", i + 1, FrameBufferCount);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.CancelSharedFrameBuffer(hLayer, i));
        }

        // すべて返却するとデタッチ可能
        NN_LOG("  check safe-detach is ready\n");
        EXPECT_TRUE(nn::os::TryWaitSystemEvent(&detachReadyEvent));

        NN_LOG("  finish safe-detach\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.FinishDetachSharedLayerFromLowLevelLayer(hLayer));
    }

    context0.DisconnectService();
    context1.DisconnectService();
    context0.CleanSharedBuffer(SlotIndex);
}

NNT_VI_TEST_FUNCTIONAL(CLCancel_CheckForceDetached)
{
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;
    static const int LoopCount = 5;

    ContextExt context0({"server"});
    Context context1({"client"});

    context0.ConnectService();
    context1.ConnectService();

    // setup SharedLowLevelLayer
    auto layerId = context0.GetDefaultLayerId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

    // setup SharedBuffer
    auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

    // setup SharedLayer
    nn::vi::fbshare::SharedLayerHandle hLayer = {};
    nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

    // attach
    nn::os::SystemEventType detachReadyEvent = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetSharedLayerDetachReadyEvent(&detachReadyEvent, hLayer));
    NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&detachReadyEvent); };

    // open SharedLayer
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.OpenSharedLayer(hLayer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.ConnectSharedLayer(hLayer));
    nn::os::SystemEventType acquirableEvent = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedFrameBufferAcquirableEvent(&acquirableEvent, hLayer));
    NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&acquirableEvent); };

    NN_LOG("Checking ForceDetach...\n");
    for(int t = 0; t < LoopCount; t++)
    {
        // アタッチ
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));

        // 全バッファ Acquire
        NN_LOG("  acquiring all buffers by client\n");
        for(int i = 0; i < FrameBufferCount; i++)
        {
            int index;
            nn::vi::native::NativeSync sync;
            nn::vi::fbshare::SharedLayerTextureIndexList indexList;
            NN_LOG("  acquiring (%d/%d)\n", i + 1, FrameBufferCount);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&index, &sync, &indexList, hLayer));
        }

        // デタッチ開始
        NN_LOG("  request safe-detach\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.StartDetachSharedLayerFromLowLevelLayer(hLayer));

        // 1 枚ずつバッファを返却
        NN_LOG("  canceling (%d/%d)\n", 0, FrameBufferCount);
        NNT_VI_EXPECT_SUCCESS(context1.CancelSharedFrameBuffer(hLayer, 0));

        NN_LOG("  canceling (%d/%d)\n", 1, FrameBufferCount);
        NNT_VI_EXPECT_SUCCESS(context1.CancelSharedFrameBuffer(hLayer, 1));

        NN_LOG("  force detach\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ForceDetachSharedLayerFromLowLevelLayer(hLayer));

        // 強制デタッチされていたら Denied が返る。
        NN_LOG("  canceling (%d/%d)\n", 2, FrameBufferCount);
        NNT_VI_EXPECT_ERROR(nn::vi::ResultDenied, context1.CancelSharedFrameBuffer(hLayer, 0));

        NN_LOG("  canceling (%d/%d)\n", 3, FrameBufferCount);
        NNT_VI_EXPECT_ERROR(nn::vi::ResultDenied, context1.CancelSharedFrameBuffer(hLayer, 1));
    }

    NN_LOG("Checking ForceDetach->Attach...\n");
    // アタッチ
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));
    for(int t = 0; t < LoopCount; t++)
    {

        // 全バッファ Acquire
        NN_LOG("  acquiring all buffers by client\n");
        for(int i = 0; i < FrameBufferCount; i++)
        {
            int index;
            nn::vi::native::NativeSync sync;
            nn::vi::fbshare::SharedLayerTextureIndexList indexList;
            NN_LOG("  acquiring (%d/%d)\n", i + 1, FrameBufferCount);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&index, &sync, &indexList, hLayer));
        }

        // デタッチ開始
        NN_LOG("  request safe-detach\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.StartDetachSharedLayerFromLowLevelLayer(hLayer));

        // 1 枚ずつバッファを返却
        NN_LOG("  canceling (%d/%d)\n", 0, FrameBufferCount);
        NNT_VI_EXPECT_SUCCESS(context1.CancelSharedFrameBuffer(hLayer, 0));

        NN_LOG("  canceling (%d/%d)\n", 1, FrameBufferCount);
        NNT_VI_EXPECT_SUCCESS(context1.CancelSharedFrameBuffer(hLayer, 1));

        NN_LOG("  force detach\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ForceDetachSharedLayerFromLowLevelLayer(hLayer));

        NN_LOG("  attach again\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));

        // 強制デタッチされていたら Denied が返る。
        NN_LOG("  canceling (%d/%d)\n", 2, FrameBufferCount);
        NNT_VI_EXPECT_ERROR(nn::vi::ResultDenied, context1.CancelSharedFrameBuffer(hLayer, 0));

        NN_LOG("  canceling (%d/%d)\n", 3, FrameBufferCount);
        NNT_VI_EXPECT_ERROR(nn::vi::ResultDenied, context1.CancelSharedFrameBuffer(hLayer, 1));
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ForceDetachSharedLayerFromLowLevelLayer(hLayer));

    context0.DisconnectService();
    context1.DisconnectService();
    context0.CleanSharedBuffer(SlotIndex);
}
