﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt.h>

#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"

NNT_VI_TEST_FUNCTIONAL(BufferContentParameter_Present)
{
    static const int LoopCount = 10;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;

    ContextExt context0({"server"});
    Context context1({"client"});

    for(int t = 0; t < LoopCount; t++)
    {
        context0.ConnectService();
        context1.ConnectService();

        // setup SharedLowLevelLayer
        auto layerId = context0.GetDefaultLayerId();
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

        // setup SharedBuffer
        auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

        // setup SharedLayer
        nn::vi::fbshare::SharedLayerHandle hLayer = {};
        nn::vi::LayerStackFlagType initialStacks = (1 << nn::vi::LayerStack_Default);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, initialStacks));

        // attach
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1));

        // open SharedLayer
        NN_ABORT_UNLESS_RESULT_SUCCESS(context1.OpenSharedLayer(hLayer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context1.ConnectSharedLayer(hLayer));
        nn::os::SystemEventType acquirableEvent = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedFrameBufferAcquirableEvent(&acquirableEvent, hLayer));
        NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&acquirableEvent); };

        auto checkContentParameter = [&](
            int index,
            nn::vi::LayerStackFlagType expectedStacks,
            const nn::vi::CropRegion& expectedCrop,
            uint32_t expectedTransform,
            int32_t expectedPresentInterval
            ) -> void
        {
            nn::vi::LayerStackFlagType stacks = 0;
            nn::vi::CropRegion crop = {};
            int32_t scalingMode = 0;
            uint32_t transform = 0;
            int32_t presentInterval = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetSharedFrameBufferContentParameter(&stacks, &crop, &scalingMode, &transform, &presentInterval, hBuffer, index));
            EXPECT_EQ(expectedStacks, stacks);
            EXPECT_EQ(0, std::memcmp(&expectedCrop, &crop, sizeof(nn::vi::CropRegion)));
            EXPECT_EQ(0, scalingMode);
            EXPECT_EQ(expectedTransform, transform);
            EXPECT_EQ(expectedPresentInterval, presentInterval);
        };

        auto acquireTexture = [&](int* pOutFrameIndex) -> void
        {
            nn::os::WaitSystemEvent(&acquirableEvent);
            nn::vi::native::NativeSync dispSync = {};
            nn::vi::fbshare::SharedLayerTextureIndexList indexList = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(pOutFrameIndex, &dispSync, &indexList, hLayer));
            nnt::vi::WaitSync(dispSync);
        };

        auto presentTexuture = [&](
            int index,
            const nn::vi::native::NativeSync& sync,
            const nn::vi::CropRegion& crop,
            uint32_t transform,
            int32_t presentInterval
            ) -> void
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.PresentSharedFrameBuffer(hLayer, index, sync, crop, transform, presentInterval));
        };

        nn::vi::native::NativeSync appSync = {};
        // 初期状態を確認
        NN_LOG("Checking Initial\n");
        for(int i = 0; i < FrameBufferCount; i++)
        {
            checkContentParameter(i, 0, {}, 0, 1);
        }

        NN_LOG("Checking LayerStack\n");
        {
            nn::vi::LayerStackFlagType stacksList[] ={
                (1 << nn::vi::LayerStack_Default),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame) | (1 << nn::vi::LayerStack_Screenshot),
            };
            for(int k = 0; k < sizeof(stacksList) / sizeof(stacksList[0]); k++)
            {
                auto stacks = stacksList[k];
                nn::vi::CropRegion crop = {0, 0, 1280, 720};
                uint32_t transform = 0;
                int32_t presentInterval = 1;
                int index = -1;
                acquireTexture(&index);
                NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));
                presentTexuture(index, appSync, crop, transform, presentInterval);
                checkContentParameter(index, stacks, crop, transform, presentInterval);
            }

        }
        // LayerStack を戻す
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, initialStacks));

        NN_LOG("Checking CropRegion\n");
        {
            nn::vi::CropRegion cropList[] ={
                { 0, 0, 1280, 720 },
                { 0, 0, 640, 360 },
            };
            for(int k = 0; k < sizeof(cropList) / sizeof(cropList[0]); k++)
            {
                nn::vi::LayerStackFlagType stacks = initialStacks;
                nn::vi::CropRegion crop = cropList[k];
                uint32_t transform = 0;
                int32_t presentInterval = 1;
                int index = -1;
                acquireTexture(&index);
                presentTexuture(index, appSync, crop, transform, presentInterval);
                checkContentParameter(index, stacks, crop, transform, presentInterval);
            }
        }

        NN_LOG("Checking Transform\n");
        {
            uint32_t transformList[] = { nn::vi::ImageTransform_None, nn::vi::ImageTransform_YFlip };
            for(int k = 0; k < sizeof(transformList) / sizeof(transformList[0]); k++)
            {
                nn::vi::LayerStackFlagType stacks = initialStacks;
                nn::vi::CropRegion crop = {0, 0, 1280, 720};
                uint32_t transform = transformList[k];
                int32_t presentInterval = 1;
                int index = -1;
                acquireTexture(&index);
                presentTexuture(index, appSync, crop, transform, presentInterval);
                checkContentParameter(index, stacks, crop, transform, presentInterval);
            }
        }

        NN_LOG("Checking PresentInterval\n");
        {
            uint32_t presentIntervalList[] = { 1, 2, 3 };
            for(int k = 0; k < sizeof(presentIntervalList) / sizeof(presentIntervalList[0]); k++)
            {
                nn::vi::LayerStackFlagType stacks = initialStacks;
                nn::vi::CropRegion crop = {0, 0, 1280, 720};
                uint32_t transform = 0;
                int32_t presentInterval = presentIntervalList[k];
                int index = -1;
                acquireTexture(&index);
                presentTexuture(index, appSync, crop, transform, presentInterval);
                checkContentParameter(index, stacks, crop, transform, presentInterval);
            }
        }

        context0.DisconnectService();
        context1.DisconnectService();
        context0.CleanSharedBuffer(SlotIndex);
    }
}// NOLINT(impl/function_size)

NNT_VI_TEST_FUNCTIONAL(BufferContentParameter_Detached)
{
    static const int LoopCount = 10;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;

    ContextExt context0({"server"});
    Context context1({"client"});

    for(int t = 0; t < LoopCount; t++)
    {
        context0.ConnectService();
        context1.ConnectService();

        // setup SharedLowLevelLayer
        auto layerId = context0.GetDefaultLayerId();
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

        // setup SharedBuffer
        auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

        // setup SharedLayer
        nn::vi::fbshare::SharedLayerHandle hLayer = {};
        nn::vi::LayerStackFlagType initialStacks = (1 << nn::vi::LayerStack_Default);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, initialStacks));

        auto checkContentParameter = [&](
            int index,
            nn::vi::LayerStackFlagType expectedStacks,
            const nn::vi::CropRegion& expectedCrop,
            uint32_t expectedTransform,
            int32_t expectedPresentInterval
            ) -> void
        {
            nn::vi::LayerStackFlagType stacks = 0;
            nn::vi::CropRegion crop = {};
            int32_t scalingMode = 0;
            uint32_t transform = 0;
            int32_t presentInterval = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetSharedFrameBufferContentParameter(&stacks, &crop, &scalingMode, &transform, &presentInterval, hBuffer, index));
            EXPECT_EQ(expectedStacks, stacks);
            EXPECT_EQ(0, std::memcmp(&expectedCrop, &crop, sizeof(nn::vi::CropRegion)));
            EXPECT_EQ(0, scalingMode);
            EXPECT_EQ(expectedTransform, transform);
            EXPECT_EQ(expectedPresentInterval, presentInterval);
        };

        auto resetContentParameters = [&]() -> void{
            for(int i = 0; i < FrameBufferCount; i++)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(context0.FillDetachedBufferColor(hBuffer, i, 255, 255, 255, 255, 0));
            }
        };

        auto testCopy = [&](int srcIndex) -> void{
            nn::vi::LayerStackFlagType srcStacks = 0;
            nn::vi::CropRegion         srcCrop = {};
            int32_t                    srcScalingMode = 0;
            uint32_t                   srcTransform = 0;
            int32_t                    srcPresentInterval = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetSharedFrameBufferContentParameter(&srcStacks, &srcCrop, &srcScalingMode, &srcTransform, &srcPresentInterval, hBuffer, srcIndex));

            int dstIndex = (srcIndex + 1) % FrameBufferCount;
            NN_ASSERT(dstIndex < FrameBufferCount);
            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CopyDetachedBufferImage(hBuffer, dstIndex, srcIndex));

            nn::vi::LayerStackFlagType dstStacks = 0;
            nn::vi::CropRegion         dstCrop = {};
            int32_t                    dstScalingMode = 0;
            uint32_t                   dstTransform = 0;
            int32_t                    dstPresentInterval = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetSharedFrameBufferContentParameter(&dstStacks, &dstCrop, &dstScalingMode, &dstTransform, &dstPresentInterval, hBuffer, dstIndex));

            EXPECT_EQ(srcStacks, dstStacks);
            EXPECT_EQ(0, std::memcmp(&srcCrop, &dstCrop, sizeof(nn::vi::CropRegion)));
            EXPECT_EQ(srcScalingMode, dstScalingMode);
            EXPECT_EQ(srcTransform, dstTransform);
            EXPECT_EQ(srcPresentInterval, dstPresentInterval);
        };

        // 初期状態を確認
        NN_LOG("Checking Initial\n");
        for(int i = 0; i < FrameBufferCount; i++)
        {
            checkContentParameter(i, 0, {}, 0, 1);
        }

        NN_LOG("Check Fill\n");
        {
            nn::vi::LayerStackFlagType stacksList[] ={
                (1 << nn::vi::LayerStack_Default),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame) | (1 << nn::vi::LayerStack_Screenshot),
            };
            for(int k = 0; k < sizeof(stacksList) / sizeof(stacksList[0]); k++)
            {
                auto stacks = stacksList[k];
                int index = k % FrameBufferCount;
                NN_ABORT_UNLESS_RESULT_SUCCESS(context0.FillDetachedBufferColor(hBuffer, index, 255, 255, 255, 255, stacks));
                checkContentParameter(index, stacks, {}, 0, 1);
                testCopy(index);
            }
        }
        resetContentParameters();

        std::vector<uint8_t> buf;
        buf.resize(4 * 1280 * 720);
        NN_LOG("Check SetImage\n");
        {
            nn::vi::LayerStackFlagType stacksList[] ={
                (1 << nn::vi::LayerStack_Default),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame) | (1 << nn::vi::LayerStack_Screenshot),
            };
            for(int k = 0; k < sizeof(stacksList) / sizeof(stacksList[0]); k++)
            {
                auto stacks = stacksList[k];
                int index = k % FrameBufferCount;
                NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetDetachedSharedBufferImage(hBuffer, index, buf.data(), buf.size(), stacks));
                checkContentParameter(index, stacks, {}, 0, 1);
                testCopy(index);
            }
        }
        resetContentParameters();

        NN_LOG("Check SetSubImage(FullScreen)\n");
        {
            nn::vi::LayerStackFlagType stacksList[] ={
                (1 << nn::vi::LayerStack_Default),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame) | (1 << nn::vi::LayerStack_Screenshot),
            };
            for(int k = 0; k < sizeof(stacksList) / sizeof(stacksList[0]); k++)
            {
                auto stacks = stacksList[k];
                int index = k % FrameBufferCount;
                NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetDetachedSharedBufferSubImage(hBuffer, index, 0, 0, 1280, 720, 0xFFFFFFFF, buf.data(), buf.size(), stacks));
                checkContentParameter(index, stacks, {}, 0, 1);
                testCopy(index);
            }
        }
        resetContentParameters();

        NN_LOG("Check SetSubImage(ZeroRegion)\n");
        {
            nn::vi::LayerStackFlagType stacksList[] ={
                (1 << nn::vi::LayerStack_Default),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame),
                (1 << nn::vi::LayerStack_Default) | (1 << nn::vi::LayerStack_LastFrame) | (1 << nn::vi::LayerStack_Screenshot),
            };
            for(int k = 0; k < sizeof(stacksList) / sizeof(stacksList[0]); k++)
            {
                auto stacks = stacksList[k];
                int index = k % FrameBufferCount;
                NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetDetachedSharedBufferSubImage(hBuffer, index, 0, 0, 0, 0, 0xFFFFFFFF, nullptr, 0, stacks));
                checkContentParameter(index, stacks, {}, 0, 1);
                testCopy(index);
            }
        }
        //resetContentParameters(); // 最後はリセットしないでおく

        context0.DisconnectService();
        context1.DisconnectService();
        context0.CleanSharedBuffer(SlotIndex);
    }
}// NOLINT(impl/function_size)
