﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testVi_SceneUtility.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nvn/nvn_FuncPtrInline.h>

namespace nnt{ namespace vi{ namespace scene{

    namespace {
        static const int GryphWidth  = 8;
        static const int GryphHeight = 8;

        char Gryph_Unknown[] = {
            "* * * * "
            " * * *  "
            "* * * * "
            " * * *  "
            "* * * * "
            " * * *  "
            "* * * * "
            " * * *  "
        };

        // 約物

        char Gryph_SP[] = {
            "        "
            "        "
            "        "
            "        "
            "        "
            "        "
            "        "
            "        "
        };

        char Gryph_Dot[] = {
            "        "
            "        "
            "        "
            "        "
            "        "
            "**      "
            "**      "
            "        "
        };

        char Gryph_Comma[] = {
            "        "
            "        "
            "        "
            "        "
            "        "
            "**      "
            " *      "
            "*       "
        };

        char Gryph_ParenL[] = {
            "     ** "
            "    *   "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "    *   "
            "     ** "
        };

        char Gryph_ParenR[] = {
            "**      "
            "  *     "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "  *     "
            "**      "
        };

        char Gryph_BraceL[] = {
            "     ** "
            "    *   "
            "    *   "
            "  **    "
            "  **    "
            "    *   "
            "    *   "
            "     ** "
        };

        char Gryph_BraceR[] = {
            "**      "
            "  *     "
            "  *     "
            "   **   "
            "   **   "
            "  *     "
            "  *     "
            "**      "
        };

        char Gryph_BracketL[] = {
            "   **** "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "   **** "
        };

        char Gryph_BracketR[] = {
            "****    "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "****    "
        };


        // 数字

        const char Gryph_0[] = {
            "  ***   "
            " *   *  "
            "*     * "
            "*     * "
            "*     * "
            "*     * "
            " *   *  "
            "  ***   "
        };

        const char Gryph_1[] = {
            "   *    "
            "  **    "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            "   *    "
            " *****  "
        };

        const char Gryph_2[] = {
            " *****  "
            "*     * "
            "      * "
            "     *  "
            "    *   "
            "   *    "
            "  *     "
            "******* "
        };

        const char Gryph_3[] = {
            " *****  "
            "*     * "
            "      * "
            "  ****  "
            "      * "
            "      * "
            "*     * "
            " *****  "
        };

        const char Gryph_4[] = {
            "    **  "
            "   * *  "
            "  *  *  "
            " *   *  "
            "*    *  "
            "******* "
            "     *  "
            "     *  "
        };

        const char Gryph_5[] = {
            "******* "
            "*       "
            "*       "
            "******  "
            "      * "
            "      * "
            "*     * "
            " *****  "
        };

        const char Gryph_6[] = {
            " *****  "
            "*     * "
            "*       "
            "******  "
            "*     * "
            "*     * "
            "*     * "
            " *****  "
        };

        const char Gryph_7[] = {
            "******* "
            "      * "
            "     *  "
            "    *   "
            "    *   "
            "   *    "
            "   *    "
            "   *    "
        };

        const char Gryph_8[] = {
            " *****  "
            "*     * "
            "*     * "
            " *****  "
            "*     * "
            "*     * "
            "*     * "
            " *****  "
        };

        const char Gryph_9[] = {
            " *****  "
            "*     * "
            "*     * "
            " ****** "
            "      * "
            "      * "
            "*     * "
            " *****  "
        };

        const char* GryphList_Number[] = {
            Gryph_0,
            Gryph_1,
            Gryph_2,
            Gryph_3,
            Gryph_4,
            Gryph_5,
            Gryph_6,
            Gryph_7,
            Gryph_8,
            Gryph_9,
        };

        void DrawCharImpl(NVNcommandBuffer* pCommandBuffer, NVNtexture* pTexture, NVNtextureView* pTexView, const char* gryph, float posX, float posY, float size, const float* color) NN_NOEXCEPT
        {
            float texWidth  = nvnTextureGetWidth(pTexture);
            float texHeight = nvnTextureGetHeight(pTexture);
            float tileSizeX = size / GryphWidth;
            float tileSizeY = size / GryphHeight;
            for(int i = 0; i < GryphHeight; i++)
            {
                float tilePosY = posY + tileSizeY * i;
                int tileTop    = static_cast<int>(std::max<float>(0, tilePosY));
                int tileBottom = static_cast<int>(std::min<float>(texHeight, tilePosY + tileSizeY));
                int tileHeight = tileBottom - tileTop;
                if(tileHeight <= 0)
                {
                    continue;
                }

                for(int j = 0; j < GryphWidth; j++)
                {
                    char g = gryph[j + i * GryphWidth];
                    if(g == ' ')
                    {
                        continue;
                    }

                    float tilePosX = posX + tileSizeX * j;
                    int tileLeft  = static_cast<int>(std::max<float>(0, tilePosX));
                    int tileRight = static_cast<int>(std::min<float>(texWidth, tilePosX + tileSizeX));
                    int tileWidth = tileRight - tileLeft;
                    if(tileWidth <= 0)
                    {
                        continue;
                    }

                    NVNcopyRegion rect = {};
                    rect.xoffset = tileLeft;
                    rect.yoffset = tileTop;
                    rect.zoffset = 0;
                    rect.width   = tileWidth;
                    rect.height  = tileHeight;
                    rect.depth   = 1;
                    nvnCommandBufferClearTexture(pCommandBuffer, pTexture, pTexView, &rect, color, NVN_CLEAR_COLOR_MASK_RGBA);
                }
            }
        }
    }

    void DrawChar(
        NVNcommandBuffer* pCommandBuffer,
        NVNtexture* pTexture,
        NVNtextureView* pTexView,
        char c,
        float posX,
        float posY,
        float size,
        const nn::util::Color4f& color
    ) NN_NOEXCEPT
    {
        const char* gryph = Gryph_Unknown;
        if(c >= '0' && c <= '9')
        {
            gryph = GryphList_Number[c - '0'];
        }
        else
        {
            switch(c)
            {
            case ' ': gryph = Gryph_SP; break;
            case '(': gryph = Gryph_ParenL; break;
            case ')': gryph = Gryph_ParenR; break;
            case '{': gryph = Gryph_BraceL; break;
            case '}': gryph = Gryph_BraceR; break;
            case '[': gryph = Gryph_BracketL; break;
            case ']': gryph = Gryph_BracketR; break;
            case '.': gryph = Gryph_Dot; break;
            case ',': gryph = Gryph_Comma; break;
            default: gryph = Gryph_Unknown; break;
            }
        }
        float colorArray[4] = {};
        nn::util::VectorStore(colorArray, color);
        DrawCharImpl(pCommandBuffer, pTexture, pTexView, gryph, posX, posY, size, colorArray);
    }

    void DrawString(
        NVNcommandBuffer* pCommandBuffer,
        NVNtexture* pTexture,
        NVNtextureView* pTexView,
        const char* str,
        float posX,
        float posY,
        float size,
        const nn::util::Color4f& color
    ) NN_NOEXCEPT
    {
        for(int i = 0; str[i] != '\0'; i++)
        {
            char c = str[i];
            DrawChar(pCommandBuffer, pTexture, pTexView, c, posX + i * size, posY, size, color);
        }
    }

    void DrawString(
        NVNcommandBuffer* pCommandBuffer,
        NVNtexture* pTexture,
        NVNtextureView* pTexView,
        const char* str,
        const DrawStringParameter& param
    ) NN_NOEXCEPT
    {
        DrawString(pCommandBuffer, pTexture, pTexView, str, param.posX, param.posY, param.size, param.color);
    }

}}}

