﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/nn_Common.h>

#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/fs.h>
#include <nv/nv_ServiceName.h>
#include <nn/vi/detail/vi_SystemServiceProxy.private.h>
#include <nnt.h>
#include <nnt/vi/testVi_TestServerProxy.h>
#include <nn/am/am_Shim.h>

#include <nn/vi/manager/vi_Manager.h>

#include "testVi_MemoryManagement.h"
#include "testVi_Context.h"

extern "C" void nninitStartup() NN_NOEXCEPT
{
    InitializeAllocator();
}

const int Context::SharedBufferCountMax;


TEST(SharedBufferTest, PresentToManagedLayer)
{
    const int LoopCount  = 10;
    const int FrameCountMax = 10;
    NN_LOG("PresentToManagedLayer-------------------------------\n");
    const int slot = 0;
    const int bufferCount = 4;
    Context context;

    auto memFill = [&](void* pDst, size_t size)->void
    {
        uint8_t* p = reinterpret_cast<uint8_t*>(pDst);
        for(int i = 0; i < nn::vi::fbshare::SharedFrameBufferSize; i += 4)
        {
            p[0] = 128; // R
            p[1] = 128; // G
            p[2] = 128; // B
            p[3] = 255; // A
            p += 4;
        }
        for(int i = 0; i < nn::vi::fbshare::SharedFrameBufferSize; i += 4)
        {
            p[0] = 128; // R
            p[1] = 0; // G
            p[2] = 0; // B
            p[3] = 255; // A
            p += 4;
        }
        for(int i = 0; i < nn::vi::fbshare::SharedFrameBufferSize; i += 4)
        {
            p[0] = 0; // R
            p[1] = 128; // G
            p[2] = 0; // B
            p[3] = 255; // A
            p += 4;
        }
        for(int i = 0; i < nn::vi::fbshare::SharedFrameBufferSize; i += 4)
        {
            p[0] = 0; // R
            p[1] = 0; // G
            p[2] = 128; // B
            p[3] = 255; // A
            p += 4;
        }
    };

    std::mt19937 rand(0);
    std::uniform_int_distribution<int> dist(0, FrameCountMax);

    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        context.ConnectService();
        auto h = context.CreateSharedBuffer(slot, bufferCount, memFill);
        auto layerId = context.GetDefaultLayerId();
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.BindLowLevelLayerToManagedLayer(layerId));

        nn::os::SystemEventType synchronizedEvent = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetLowLevelLayerSynchronizedEvent(&synchronizedEvent, layerId));
        NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&synchronizedEvent); };

        NN_ABORT_UNLESS_RESULT_SUCCESS(context.ConnectLowLevelLayerToSharedBuffer(layerId, h));
        int frameCount = dist(rand);
        for(int frame = 0; frame < frameCount; frame++)
        {
            int next = frame % bufferCount;
            nn::os::WaitSystemEvent(&synchronizedEvent);

            int displayedIndex = -1;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.CheckLowLevelLayerSynchronized(&displayedIndex, layerId));

            NN_LOG("    displayed %d | nextPresent %d\n", displayedIndex, next);

            NN_ABORT_UNLESS_RESULT_SUCCESS(context.FillDetachedBufferColor(h, next, 255, 128, 64, 255));

            NN_ABORT_UNLESS_RESULT_SUCCESS(context.PresentDetachedBufferToLowLevelLayer(h, layerId, next));
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.DisconnectLowLevelLayerFromSharedBuffer(layerId));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.UnbindLowLevelLayer(layerId));
        context.DestroySharedBuffer(slot, h);
        context.DisconnectService();
    }
}
