﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/am/am_Shim.h>
#include <nn/util/util_Color.h>

#include <nn/vi/vi_Result.h>
#include <nn/vi/manager/vi_Manager.h>
#include <nn/vi/native/vi_NativeType.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>
#include <nn/vi/fbshare/vi_SharedFrameBufferSize.h>
#include <nn/vi/fbshare/vi_SharedTextureOption.h>

#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"

#include <nnt.h>

enum SharedBufferCreationMode
{
    SharedBufferCreationMode_TransferMemory,
    SharedBufferCreationMode_ProcessHeap,
};

class Context
{
public:
    static const int SharedBufferCountMax = 4;
    static const SharedBufferCreationMode Mode = SharedBufferCreationMode_ProcessHeap;

private:
    nn::vi::ProxyName m_ProxyName;
    nn::vi::DisplayName m_DisplayName;

    nn::vi::manager::DisplayManagerService m_Service;

    struct SharedBufferResource
    {
        int m_BufferCount;
        void* m_pBufferMemory;
        nn::os::TransferMemoryType m_BufferTransferMemory;
        uint64_t m_ProcessHeapBlockId;
    } m_SharedBufferResource[SharedBufferCountMax];

public:

    Context() NN_NOEXCEPT
        : m_SharedBufferResource()
    {
        InitializeGraphics();
        m_ProxyName = { "mgr" };
        m_DisplayName = { "Default" };
    }

    explicit Context(const nn::vi::ProxyName& proxyName) NN_NOEXCEPT
        : m_SharedBufferResource()
    {
        InitializeGraphics();
        m_ProxyName = proxyName;
        m_DisplayName = { "Default" };
    }

    ~Context() NN_NOEXCEPT
    {
        bool isClean = true;
        for(int i = 0; i < SharedBufferCountMax; i++)
        {
            if(m_SharedBufferResource[i].m_pBufferMemory != nullptr || m_SharedBufferResource[i].m_ProcessHeapBlockId != 0)
            {
                NN_LOG("TestCodeError: SharedBuffer slot %d is not cleaned up. Call CleanSharedBuffer().\n", i);
                isClean = false;
            }
        }
        NN_ABORT_UNLESS(isClean);
    }

    void ConnectService() NN_NOEXCEPT
    {
        m_Service.Initialize(nn::vi::PolicyLevel_Composition, &m_ProxyName);
    }

    void DisconnectService() NN_NOEXCEPT
    {
        m_Service.Finalize();
    }

    nn::sf::SharedPointer<nn::visrv::sf::IManagerDisplayService> GetManagerService() NN_NOEXCEPT
    {
        return m_Service.GetManagerDisplayService();
    }

    nn::sf::SharedPointer<nn::visrv::sf::ISystemDisplayService> GetSystemService() NN_NOEXCEPT
    {
        return m_Service.GetSystemDisplayService();
    }

    nn::sf::SharedPointer<nn::visrv::sf::IApplicationDisplayService> GetApplicationService() NN_NOEXCEPT
    {
        return m_Service.GetApplicationDisplayService();
    }

    // void (*MemoryFillFunction)(void* p, size_t size)
    template<typename MemoryFillFunction>
    nn::vi::fbshare::SharedBufferHandle CreateSharedBuffer(int slot, int bufferCount, MemoryFillFunction memFillFunc) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slot, 0, SharedBufferCountMax);
        SharedBufferResource& res = m_SharedBufferResource[slot];

        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};
        layout.count = bufferCount;
        for(int i = 0; i < bufferCount; i++)
        {
            layout.entries[i].offset = nn::vi::fbshare::SharedFrameBufferSize * i;
            layout.entries[i].size   = nn::vi::fbshare::SharedFrameBufferSize;
            layout.entries[i].width  = nn::vi::fbshare::SharedFrameBufferWidth;
            layout.entries[i].height = nn::vi::fbshare::SharedFrameBufferHeight;
        }

        size_t memorySize = bufferCount * nn::vi::fbshare::SharedFrameBufferSize;
        nn::vi::fbshare::SharedBufferHandle h = {};

        if(NN_STATIC_CONDITION(Mode == SharedBufferCreationMode_TransferMemory))
        {
            auto pMemory = aligned_alloc(4096, memorySize);

            memFillFunc(pMemory, memorySize);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateTransferMemory(&res.m_BufferTransferMemory, pMemory, memorySize, nn::vi::fbshare::SharedBufferMemoryPermission));
            nn::os::NativeHandle hTransferMemory = nn::os::DetachTransferMemory(&res.m_BufferTransferMemory);

            NN_LOG("  CreateSharedBuffer On TransferMemory\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->CreateSharedBufferTransferMemory(
                &h,
                nn::sf::NativeHandle(hTransferMemory, true),
                memorySize,
                layout
            ));

            res.m_BufferCount = bufferCount;
            res.m_pBufferMemory = pMemory;
        }
        if(NN_STATIC_CONDITION(Mode == SharedBufferCreationMode_ProcessHeap))
        {
            uint64_t blockId = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->AllocateProcessHeapBlock(&blockId, memorySize));

            NN_LOG("  CreateSharedBuffer On ProcessHeap #%lld\n", blockId);
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->CreateSharedBufferProcessHeap(
                &h,
                blockId,
                layout
            ));

            res.m_BufferCount = bufferCount;
            res.m_ProcessHeapBlockId = blockId;
        }

        return h;
    }

    nn::vi::fbshare::SharedBufferHandle CreateSharedBuffer(int slot, int bufferCount) NN_NOEXCEPT
    {
        return CreateSharedBuffer(slot, bufferCount, [](void*,size_t){});
    }

    // 720p, 1080p 対応
    // レイアウトは
    //
    // |720|720|720|720|
    //         | 1080  | 1080  |
    //
    // 2,3 (720p) と 4 (1080p) が同じメモリ
    nn::vi::fbshare::SharedBufferHandle CreateSharedBufferEx720p1080p(int slot) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slot, 0, SharedBufferCountMax);
        SharedBufferResource& res = m_SharedBufferResource[slot];

        size_t memoryBlockSize = 34 * 128 * 1024;

        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};
        layout.count = 6;
        for(int i = 0; i < 4; i++)
        {
            layout.entries[i].offset = memoryBlockSize * i;
            layout.entries[i].size   = memoryBlockSize;
            layout.entries[i].width  = 1280;
            layout.entries[i].height = 720;
        }
        for(int i = 0; i < 2; i++)
        {
            layout.entries[i + 4].offset = 2 * memoryBlockSize * i + 2 * memoryBlockSize;
            layout.entries[i + 4].size   = 2 * memoryBlockSize;
            layout.entries[i + 4].width  = 1920;
            layout.entries[i + 4].height = 1080;
        }

        size_t memorySize = 6 * memoryBlockSize;
        nn::vi::fbshare::SharedBufferHandle h = {};

        if(NN_STATIC_CONDITION(Mode == SharedBufferCreationMode_TransferMemory))
        {
            auto pMemory = aligned_alloc(4096, memorySize);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateTransferMemory(&res.m_BufferTransferMemory, pMemory, memorySize, nn::vi::fbshare::SharedBufferMemoryPermission));
            nn::os::NativeHandle hTransferMemory = nn::os::DetachTransferMemory(&res.m_BufferTransferMemory);

            NN_LOG("  CreateSharedBuffer On TransferMemroy\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->CreateSharedBufferTransferMemory(
                &h,
                nn::sf::NativeHandle(hTransferMemory, true),
                memorySize,
                layout
            ));

            res.m_BufferCount   = 6;
            res.m_pBufferMemory = pMemory;
        }
        if(NN_STATIC_CONDITION(Mode == SharedBufferCreationMode_ProcessHeap))
        {
            uint64_t blockId = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->AllocateProcessHeapBlock(&blockId, memorySize));

            NN_LOG("  CreateSharedBuffer On ProcessHeap #%lld\n", blockId);
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->CreateSharedBufferProcessHeap(
                &h,
                blockId,
                layout
            ));

            res.m_BufferCount = 6;
            res.m_ProcessHeapBlockId = blockId;
        }

        return h;
    }


    void DestroySharedBuffer(int slot, nn::vi::fbshare::SharedBufferHandle h) NN_NOEXCEPT
    {
        NN_LOG("  DestroySharedBuffer\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->DestroySharedBuffer(h));

        CleanSharedBuffer(slot);
    }

    // Destroy を呼ばなかった場合に代わりに呼ぶ
    void CleanSharedBuffer(int slot) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(slot, 0, SharedBufferCountMax);
        SharedBufferResource& res = m_SharedBufferResource[slot];

        if(NN_STATIC_CONDITION(Mode == SharedBufferCreationMode_TransferMemory))
        {
            nn::os::DestroyTransferMemory(&res.m_BufferTransferMemory);
            free(res.m_pBufferMemory);
        }
        if(NN_STATIC_CONDITION(Mode == SharedBufferCreationMode_ProcessHeap))
        {
            if(m_Service.GetManagerDisplayService() != nullptr)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->FreeProcessHeapBlock(res.m_ProcessHeapBlockId));
            }
            // 切断済の場合は自動回収されているので何もしなくて良い
        }

        res.m_BufferCount = 0;
        res.m_pBufferMemory = nullptr;
        res.m_ProcessHeapBlockId = 0;
    }

    nn::Result RegisterSharedBufferImporterAruid(nn::vi::fbshare::SharedBufferHandle hBuffer, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_LOG("  RegisterImporterAruid\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->RegisterSharedBufferImporterAruid(hBuffer, importerAruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result UnregisterSharedBufferImporterAruid(nn::vi::fbshare::SharedBufferHandle hBuffer, nn::applet::AppletResourceUserId importerAruid) NN_NOEXCEPT
    {
        NN_LOG("  UnregisterImporterAruid\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->UnregisterSharedBufferImporterAruid(hBuffer, importerAruid));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetSharedBufferMemoryHandleId(nn::vi::native::NativeMemoryHandleId* pOutHandleId, size_t* pOutSize, nn::vi::fbshare::SharedMemoryPoolLayout* pOutLayout, nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
    {
        NN_LOG("  GetBufferMemoryHandleId\n");
        uint64_t size = 0;
        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->GetSharedBufferMemoryHandleId(pOutHandleId, &size, &layout, hBuffer, nn::applet::GetAppletResourceUserId()));
        *pOutSize = static_cast<size_t>(size);
        *pOutLayout = layout;
        NN_RESULT_SUCCESS;
    }

    nn::vi::DisplayId OpenDisplay() NN_NOEXCEPT
    {
        nn::vi::DisplayId displayId = {};
        NN_LOG("  OpenDisplay\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetApplicationDisplayService()->OpenDisplay(&displayId, m_DisplayName));
        return displayId;
    }

    void CloseDisplay(nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_LOG("  CloseDisplay\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetApplicationDisplayService()->CloseDisplay(displayId));
    }

    nn::vi::LayerId CreateManagedLayer(nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_LOG("  CreateManagedLayer\n");
        nn::vi::LayerId layerId = {};
        nn::vi::LayerSettings settings;
        nn::vi::SetLayerSettingsDefaults(&settings);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->CreateManagedLayer(&layerId, displayId, reinterpret_cast<nn::vi::LayerSettingsType&>(settings), nn::applet::GetAppletResourceUserId()));
        return layerId;
    }

    void DestroyManagedLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_LOG("  DestroyManagedLayer\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Service.GetManagerDisplayService()->DestroyManagedLayer(layerId));
    }

    nn::vi::LayerId GetDefaultLayerId() NN_NOEXCEPT
    {
        NN_LOG("  GetDefaultLayerId\n");
        nn::vi::LayerId layerId;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::am::GetSelfController()->CreateManagedDisplayLayer(&layerId));
        return layerId;
    }

    nn::Result BindLowLevelLayerToManagedLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_LOG("  BindLowLevelLayerToManagedLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->BindSharedLowLevelLayerToManagedLayer(layerId, m_DisplayName, nn::applet::GetAppletResourceUserId()));
        NN_RESULT_SUCCESS;
    }

    nn::Result UnbindLowLevelLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_LOG("  UnbindLowLevelLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->UnbindSharedLowLevelLayer(layerId));
        NN_RESULT_SUCCESS;
    }

    nn::Result ConnectLowLevelLayerToSharedBuffer(nn::vi::LayerId layerId, nn::vi::fbshare::SharedBufferHandle hSb) NN_NOEXCEPT
    {
        NN_LOG("  ConnectLowLevelLayerToSharedBuffer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->ConnectSharedLowLevelLayerToSharedBuffer(layerId, hSb));
        NN_RESULT_SUCCESS;
    }

    nn::Result DisconnectLowLevelLayerFromSharedBuffer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_LOG("  DisconnectLowLevelLayerFromSharedBuffer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->DisconnectSharedLowLevelLayerFromSharedBuffer(layerId));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetLowLevelLayerSynchronizedEvent(nn::os::SystemEventType* pOutEvent, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_LOG("  GetLowLevelLayerSynchronizedEvent\n");
        nn::sf::NativeHandle h = {};
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->GetSharedLowLevelLayerSynchronizedEvent(&h, layerId));

        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
        h.Detach();
        NN_RESULT_SUCCESS;
    }

    nn::Result CheckLowLevelLayerSynchronized(int* pOutIndex, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_LOG("  CheckLowLevelLayerSynchronized\n");
        int64_t index = -1;
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->CheckSharedLowLevelLayerSynchronized(&index, layerId));
        *pOutIndex = static_cast<int>(index);
        NN_RESULT_SUCCESS;
    }

    nn::Result PresentDetachedBufferToLowLevelLayer(nn::vi::fbshare::SharedBufferHandle hSb, nn::vi::LayerId layerId, int frame) NN_NOEXCEPT
    {
        NN_LOG("  PresentDetachedBufferToLowLevelLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->PresentDetachedSharedFrameBufferToLowLevelLayer(hSb, layerId, frame));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetDetachedSharedBufferImage(
        size_t* pOutSize,
        void* buffer,
        size_t bufferSize,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        int index
    ) NN_NOEXCEPT
    {
        NN_LOG("  GetDetachedBufferImage\n");
        uint64_t outSize = 0;
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->GetDetachedSharedFrameBufferImage(&outSize, nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize), hBuffer, index));
        *pOutSize = static_cast<size_t>(outSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result FillDetachedBufferColor(nn::vi::fbshare::SharedBufferHandle hSb, int index, int r, int g, int b, int a, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT
    {
        NN_LOG("  FillDetachedBufferColor\n");
        uint32_t color =
            ((r <<  0) & 0x000000FF) |
            ((g <<  8) & 0x0000FF00) |
            ((b << 16) & 0x00FF0000) |
            ((a << 24) & 0xFF000000);

        nn::vi::fbshare::SharedTextureOption option = {};
        option.stacks = stacks;
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->FillDetachedSharedFrameBufferColor(hSb, index, color, option));
        NN_RESULT_SUCCESS;
    }

    nn::Result FillDetachedBufferColor(nn::vi::fbshare::SharedBufferHandle hSb, int index, int r, int g, int b, int a) NN_NOEXCEPT
    {
        return FillDetachedBufferColor(hSb, index, r, g, b, a, (1 << nn::vi::LayerStack_Default));
    }

    nn::Result SetDetachedSharedBufferImage(
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        int index,
        const void* buffer,
        size_t size,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::ImageTransform srcTransform
    ) NN_NOEXCEPT
    {
        NN_LOG("  SetDetachedBufferImage\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->SetDetachedSharedFrameBufferImage(
            hBuffer, index, nn::sf::InBuffer(reinterpret_cast<const char*>(buffer), size), dstOption, srcTransform)
        );
        NN_RESULT_SUCCESS;
    }

    nn::Result SetDetachedSharedBufferImage(nn::vi::fbshare::SharedBufferHandle hBuffer, int index, const void* buffer, size_t size, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT
    {
        nn::vi::fbshare::SharedTextureOption option = {};
        option.stacks = stacks;
        return SetDetachedSharedBufferImage(hBuffer, index, buffer, size, option, nn::vi::ImageTransform_None);
    }

    nn::Result SetDetachedSharedBufferSubImage(
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        int index,
        int x,
        int y,
        int w,
        int h,
        uint32_t bgColor,
        const void* buffer,
        size_t size,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::ImageTransform srcTransform
    ) NN_NOEXCEPT
    {
        NN_LOG("  SetDetachedBufferSubImage\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->SetDetachedSharedFrameBufferSubImage(
            hBuffer, index, x, y, w, h, bgColor, nn::sf::InBuffer(reinterpret_cast<const char*>(buffer), size), dstOption, srcTransform)
        );
        NN_RESULT_SUCCESS;
    }

    nn::Result SetDetachedSharedBufferSubImage(nn::vi::fbshare::SharedBufferHandle hBuffer, int index, int x, int y, int w, int h, uint32_t bgColor, const void* buffer, size_t size, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT
    {
        nn::vi::fbshare::SharedTextureOption option = {};
        option.stacks = stacks;
        return SetDetachedSharedBufferSubImage(hBuffer, index, x, y, w, h, bgColor, buffer, size, option, nn::vi::ImageTransform_None);
    }

    nn::Result CopyDetachedBufferImage(nn::vi::fbshare::SharedBufferHandle hSb, int dst, int src) NN_NOEXCEPT
    {
        NN_LOG("  CopyDetachedBuffer(%d <- %d)\n", dst, src);
        nn::vi::fbshare::SharedTextureOption option = {};
        option.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
        option.transform   = nn::vi::ImageTransform_None;
        nn::vi::LayerStackFlagType nullStacks = (1 << nn::vi::LayerStack_Null);
        return CopyDetachedBufferImage(hSb, dst, hSb, src, option, nn::vi::ValidLayerStackFlags, nullStacks);
    }

    nn::Result CopyDetachedBufferImage(
        nn::vi::fbshare::SharedBufferHandle hDstSb,
        int dst,
        nn::vi::fbshare::SharedBufferHandle hSrcSb,
        int src,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::LayerStackFlagType maskStacks,
        nn::vi::LayerStackFlagType nullStacks
    ) NN_NOEXCEPT
    {
        NN_LOG("  CopyDetachedBuffer(%d <- %d)\n", dst, src);
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->CopyDetachedSharedFrameBufferImage(hDstSb, dst, hSrcSb, src, dstOption, maskStacks, nullStacks));
        NN_RESULT_SUCCESS;
    }

    nn::Result CreateSharedLayer(nn::vi::fbshare::SharedLayerHandle* pOutHandle, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT
    {
        NN_LOG("  CreateSharedLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->CreateSharedLayer(pOutHandle, userAruid));
        // 状態チェック
        EXPECT_EQ(0, GetSharedLayerLayerStack(*pOutHandle));
        NN_RESULT_SUCCESS;
    }

    nn::Result DestroySharedLayer(nn::vi::fbshare::SharedLayerHandle hSl) NN_NOEXCEPT
    {
        NN_LOG("  DestroySharedLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->DestroySharedLayer(hSl));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetSharedLayerLayerStack(nn::vi::LayerStackFlagType* pOutValue, nn::vi::fbshare::SharedLayerHandle hSl) NN_NOEXCEPT
    {
        NN_LOG("  GetSharedLayerLayerStack(h=%lld)\n", hSl._value);
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->GetSharedLayerLayerStacks(pOutValue, hSl));
        NN_RESULT_SUCCESS;
    }

    nn::vi::LayerStackFlagType GetSharedLayerLayerStack(nn::vi::fbshare::SharedLayerHandle hSl) NN_NOEXCEPT
    {
        nn::vi::LayerStackFlagType v = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetSharedLayerLayerStack(&v, hSl));
        return v;
    }

    nn::Result SetSharedLayerLayerStack(nn::vi::fbshare::SharedLayerHandle hSl, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT
    {
        NN_LOG("  SetSharedLayerLayerStack(h=%lld,stacks=%X)\n", hSl._value, stacks);
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->SetSharedLayerLayerStacks(hSl, stacks));
        EXPECT_EQ(stacks, GetSharedLayerLayerStack(hSl));
        NN_RESULT_SUCCESS;
    }

    nn::Result OpenSharedLayer(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_LOG("  OpenSharedLayer\n");
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->OpenSharedLayer(h, nn::applet::GetAppletResourceUserId()));
        NN_RESULT_SUCCESS;
    }

    nn::Result CloseSharedLayer(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_LOG("  CloseSharedLayer\n");
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->CloseSharedLayer(h));
        NN_RESULT_SUCCESS;
    }

    nn::Result ConnectSharedLayer(nn::vi::fbshare::SharedLayerHandle h)  NN_NOEXCEPT
    {
        NN_LOG("  ConnectSharedLayer\n");
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->ConnectSharedLayer(h));
        NN_RESULT_SUCCESS;
    }

    nn::Result DisconnectSharedLayer(nn::vi::fbshare::SharedLayerHandle h)  NN_NOEXCEPT
    {
        NN_LOG("  DisconnectSharedLayer\n");
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->DisconnectSharedLayer(h));
        NN_RESULT_SUCCESS;
    }

    nn::Result AttachSharedLayerToLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer,
        nn::vi::LayerId managedLayerId,
        const nn::vi::fbshare::SharedLayerTextureIndexList& iBufList
    ) NN_NOEXCEPT
    {
        NN_LOG("  AttachSharedLayerToLowLevelLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->AttachSharedLayerToLowLevelLayer(
            hSharedLayer,
            managedLayerId,
            iBufList
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result StartDetachSharedLayerFromLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_LOG("  StartDetachSharedLayerFromLowLevelLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->StartDetachSharedLayerFromLowLevelLayer(hSharedLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result FinishDetachSharedLayerFromLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_LOG("  FinishDetachSharedLayerFromLowLevelLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->FinishDetachSharedLayerFromLowLevelLayer(hSharedLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetSharedLayerDetachReadyEvent(
        nn::os::SystemEventType* pOutEvent,
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_LOG("  GetSharedLayerDetachReadyEvent\n");
        nn::sf::NativeHandle h = {};
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->GetSharedLayerDetachReadyEvent(&h, hSharedLayer));
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
        h.Detach();
        NN_RESULT_SUCCESS;
    }

    nn::Result ForceDetachSharedLayerFromLowLevelLayer(
        nn::vi::fbshare::SharedLayerHandle hSharedLayer
    ) NN_NOEXCEPT
    {
        NN_LOG("  ForceDetachSharedLayerFromLowLevelLayer\n");
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->ForceDetachSharedLayerFromLowLevelLayer(hSharedLayer));
        NN_RESULT_SUCCESS;
    }

    nn::Result AcquireSharedFrameBuffer(
        int* pOutIndex,
        nn::vi::native::NativeSync* pOutSync,
        nn::vi::fbshare::SharedLayerTextureIndexList* pOutIndexList,
        nn::vi::fbshare::SharedLayerHandle hLayer
    ) NN_NOEXCEPT
    {
        NN_LOG("  AcquireSharedFrameBuffer\n");
        int64_t index = -1;
        nn::vi::native::NativeSync sync = {};
        auto indexList = nn::vi::fbshare::SharedLayerTextureIndexList::GetInvalidValue();
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->AcquireSharedFrameBuffer(&index, &sync, &indexList, hLayer));
        *pOutIndex = static_cast<int>(index);
        *pOutSync = sync;
        *pOutIndexList = indexList;
        NN_RESULT_SUCCESS;
    }

    nn::Result PresentSharedFrameBuffer(
        nn::vi::fbshare::SharedLayerHandle hLayer,
        int index,
        const nn::vi::native::NativeSync& sync,
        const nn::vi::CropRegion& cropRegion,
        nn::vi::ImageTransformType transform,
        int presentInterval
    ) NN_NOEXCEPT
    {
        NN_LOG("  PresentSharedFrameBuffer\n");
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->PresentSharedFrameBuffer(hLayer, static_cast<int64_t>(index), sync, cropRegion, transform, presentInterval));
        NN_RESULT_SUCCESS;
    }

    nn::Result PresentSharedFrameBuffer(
        nn::vi::fbshare::SharedLayerHandle hLayer,
        int index,
        const nn::vi::native::NativeSync& sync
    ) NN_NOEXCEPT
    {
        nn::vi::CropRegion crop = {};
        crop.x = 0;
        crop.y = 0;
        crop.width = nn::vi::fbshare::SharedFrameBufferWidth;
        crop.height = nn::vi::fbshare::SharedFrameBufferHeight;
        int presentInterval = 1;
        nn::vi::ImageTransformType transform = nn::vi::ImageTransform_None;
        NN_RESULT_DO(PresentSharedFrameBuffer(hLayer, index, sync, crop, transform, presentInterval));
        NN_RESULT_SUCCESS;
    }

    nn::Result CancelSharedFrameBuffer(
        nn::vi::fbshare::SharedLayerHandle hLayer,
        int index
    ) NN_NOEXCEPT
    {
        NN_LOG("  CancelSharedFrameBuffer\n");
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->CancelSharedFrameBuffer(hLayer, static_cast<int64_t>(index)));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetSharedFrameBufferAcquirableEvent(
        nn::os::SystemEventType* pEvent,
        nn::vi::fbshare::SharedLayerHandle hLayer
    ) NN_NOEXCEPT
    {
        NN_LOG("  GetSharedFrameBufferAcquirableEvent\n");
        nn::sf::NativeHandle h;
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->GetSharedFrameBufferAcquirableEvent(nn::sf::Out<nn::sf::NativeHandle>(&h), hLayer));
        nn::os::AttachReadableHandleToSystemEvent(pEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
        h.Detach();
        NN_RESULT_SUCCESS;
    }

    nn::Result FillSharedFrameBufferColor(
        nn::vi::fbshare::SharedLayerHandle hLayer,
        int index,
        nn::util::Color4u8 color
    ) NN_NOEXCEPT
    {
        NN_LOG("  FillSharedFrameBufferColor\n");
        NN_RESULT_DO(m_Service.GetSystemDisplayService()->FillSharedFrameBufferColor(hLayer, static_cast<int64_t>(index), color.GetR(), color.GetG(), color.GetB(), color.GetA()));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetSharedFrameBufferContentParameter(
        nn::vi::LayerStackFlagType* pOutStacks,
        nn::vi::CropRegion* pOutCrop,
        int32_t* pOutScalingMode,
        uint32_t* pOutTransform,
        int32_t* pOutPresentInterval,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        int index
    ) NN_NOEXCEPT
    {
        NN_LOG("  GetSharedFrameBufferContentParameter\n");
        nn::vi::LayerStackFlagType stacks = 0;
        nn::vi::CropRegion crop = {};
        int32_t scalingMode = 0;
        uint32_t transform = 0;
        int32_t presentInterval = 0;
        NN_RESULT_DO(m_Service.GetManagerDisplayService()->GetSharedFrameBufferContentParameter(&stacks, &crop, &scalingMode, &transform, &presentInterval, hBuffer, index));
        *pOutStacks = stacks;
        *pOutCrop = crop;
        *pOutScalingMode = scalingMode;
        *pOutTransform = transform;
        *pOutPresentInterval = presentInterval;
        NN_RESULT_SUCCESS;
    }
};


// utility

class ContextExt
    : public Context
{
public:
    ContextExt() NN_NOEXCEPT
        : Context()
    {
    }

    explicit ContextExt(const nn::vi::ProxyName& proxyName) NN_NOEXCEPT
        : Context(proxyName)
    {
    }

    nn::Result AttachSharedLayerToLowLevelLayer(nn::vi::fbshare::SharedLayerHandle hLayer, nn::vi::LayerId layerId, int buffer0, int buffer1) NN_NOEXCEPT
    {
        return Context::AttachSharedLayerToLowLevelLayer(hLayer, layerId, nn::vi::fbshare::SharedLayerTextureIndexList::Get(buffer0, buffer1));
    }

    nn::Result AttachSharedLayerToLowLevelLayer(nn::vi::fbshare::SharedLayerHandle hLayer, nn::vi::LayerId layerId, int buffer0, int buffer1, int buffer2) NN_NOEXCEPT
    {
        return Context::AttachSharedLayerToLowLevelLayer(hLayer, layerId, nn::vi::fbshare::SharedLayerTextureIndexList::Get(buffer0, buffer1, buffer2));
    }

    nn::Result AttachSharedLayerToLowLevelLayer(nn::vi::fbshare::SharedLayerHandle hLayer, nn::vi::LayerId layerId, int buffer0, int buffer1, int buffer2, int buffer3) NN_NOEXCEPT
    {
        return Context::AttachSharedLayerToLowLevelLayer(hLayer, layerId, nn::vi::fbshare::SharedLayerTextureIndexList::Get(buffer0, buffer1, buffer2, buffer3));
    }

    void SafeDetachSharedLayerFromLowLevelLayer(nn::vi::fbshare::SharedLayerHandle hLayer) NN_NOEXCEPT
    {
        nn::os::SystemEventType detachReadyEvent = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetSharedLayerDetachReadyEvent(&detachReadyEvent, hLayer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(StartDetachSharedLayerFromLowLevelLayer(hLayer));
        nn::os::WaitSystemEvent(&detachReadyEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(FinishDetachSharedLayerFromLowLevelLayer(hLayer));
        nn::os::DestroySystemEvent(&detachReadyEvent);
    }

    void SynchronizeLowLevelLayer(int* pOutDisplayedBufferIndex, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        nn::os::SystemEventType synchronizedEvent = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetLowLevelLayerSynchronizedEvent(&synchronizedEvent, layerId));
        nn::os::WaitSystemEvent(&synchronizedEvent);
        nn::os::DestroySystemEvent(&synchronizedEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(CheckLowLevelLayerSynchronized(pOutDisplayedBufferIndex, layerId));
    }
};
