﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/omm/omm_Api.h>
#include <nn/vi.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/vi_DisplayEvents.h>
#include <nn/vi/manager/vi_Manager.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/viUtil/testVi_PowerFixture.h>

void nnt::vi::util::PowerFixture::TearDown() NN_NOEXCEPT
{
    nn::vi::manager::DisplayManagerService service;
    nn::vi::ProxyName proxy;
    std::strncpy(proxy.value, "vi:m", sizeof(proxy.value));
    service.InitializeMinimum(nn::vi::PolicyLevel_Composition, &proxy);
    NN_UTIL_SCOPE_EXIT
    {
        service.FinalizeMinimum();
    };

    nn::omm::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::omm::Finalize();
    };

    auto pSystem = service.GetApplicationDisplayService();

    nn::vi::DisplayName display;
    std::strncpy(display.value, "Internal", sizeof(display.value));
    nn::vi::DisplayId internalDisplayId;
    bool isInternalDisplayValid = pSystem->OpenDisplay(&internalDisplayId, display).IsSuccess();
    NN_UTIL_SCOPE_EXIT
    {
        if( isInternalDisplayValid )
        {
            pSystem->CloseDisplay(internalDisplayId);
        }
    };

    nn::vi::DisplayId externalDisplayId;
    std::strncpy(display.value, "External", sizeof(display.value));
    bool isExternalDisplayValid = pSystem->OpenDisplay(&externalDisplayId, display).IsSuccess();
    NN_UTIL_SCOPE_EXIT
    {
        if( isExternalDisplayValid )
        {
            pSystem->CloseDisplay(externalDisplayId);
        }
    };

    auto pManager = service.GetManagerDisplayService();

    switch( nn::omm::GetOperationMode() )
    {
        case nn::omm::OperationMode::Handheld:
            if( isExternalDisplayValid )
            {
                NNT_ASSERT_RESULT_SUCCESS(pManager->SetDisplayPowerState(externalDisplayId, nn::vi::PowerState_Off));
            }

            ASSERT_TRUE(isInternalDisplayValid);
            NNT_ASSERT_RESULT_SUCCESS(pManager->SetDisplayPowerState(internalDisplayId, nn::vi::PowerState_On));
            break;
        case nn::omm::OperationMode::Console:
            {
                if( isInternalDisplayValid )
                {
                    NNT_ASSERT_RESULT_SUCCESS(pManager->SetDisplayPowerState(internalDisplayId, nn::vi::PowerState_Off));
                }

                ASSERT_TRUE(isExternalDisplayValid);

                nn::vi::HotplugStateType hotplug = nn::vi::HotplugState_Disconnected;
                NNT_EXPECT_RESULT_SUCCESS(pManager->GetDisplayHotplugState(&hotplug, externalDisplayId));
                if( hotplug == nn::vi::HotplugState_Connected )
                {
                    NNT_ASSERT_RESULT_SUCCESS(pManager->SetDisplayPowerState(externalDisplayId, nn::vi::PowerState_On));
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(pManager->SetDisplayPowerState(externalDisplayId, nn::vi::PowerState_Blank));
                }
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}
