﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include "glvapp/glvapp_RootSurface.h"

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
// For declearation of 'nv::InitializeGraphics' and 'nv::SetGraphicsAllocator'
#include <nv/nv_MemoryManagement.h>
#endif // defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

using namespace glv;

// 有効時 : SIGLOランタイムエンジンの初期化/終了を明示的に行います。
// 無効時 : SIGLOランタイムエンジンの初期化/終了を行わず、オリジナルGLVが提供するテストコード相当の実行を行います。
#define ENABLE_GLV_FRAMEWORK_SETUP

namespace
{
//!============================================================================
//!
//!============================================================================
void ntSetBool(const Notification& n)
{
    bool& b = *(reinterpret_cast<bool*>(n.receiver()));
    b = true;
}

#if defined( ENABLE_GLV_FRAMEWORK_SETUP )

//!============================================================================
//! @brief グラフィクスペリフェラル初期化
//!============================================================================
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

static void* NvnAllocate( size_t size, size_t alignment, void* userPtr )
{
    NN_UNUSED( userPtr );
    return ::aligned_alloc( alignment, size );
}

static void NvnDeallocate( void* addr, void* userPtr )
{
    NN_UNUSED( userPtr );
    ::free( addr );
}

static void* NvnReallocate( void* addr, size_t newSize, void* userPtr )
{
    NN_UNUSED( userPtr );
    return ::realloc( addr, newSize );
}

static void InitializeGraphicsPeripheral() NN_NOEXCEPT
{
    static bool s_OnceInitialized = false;
    if ( false == s_OnceInitialized )
    {
        // NVNグラフィクス稼働予約メモリ領域サイズ
        const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
        // this memory allocation will be used from the nvn graphics systems at runtime.
        nv::SetGraphicsAllocator( NvnAllocate, NvnDeallocate, NvnReallocate, NULL );
        nv::SetGraphicsDevtoolsAllocator( NvnAllocate, NvnDeallocate, NvnReallocate, NULL );
        nv::InitializeGraphics( NvnAllocate( GraphicsSystemReservedMemorySize, sizeof( uintptr_t ), nullptr ), GraphicsSystemReservedMemorySize );
        s_OnceInitialized = true;
    }
}

#else

static void InitializeGraphicsPeripheral() NN_NOEXCEPT
{
    // do nothing.
}

#endif // defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )



//!============================================================================
//! @brief      シーケンシャルテスト用ルートルーパー
//! @details    最初のフレームで即時アプリケーションループを抜ける宣言を行います。
//!============================================================================
class TestRootSurface : public glvapp::RootSurfaceContext
{
public:
    virtual const glv::RequiredRestoration OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        glv::ApplicationFrameworkExit();    // フレーム終了時にアプリケーションループを抜ける宣言。
        return glvapp::RootSurfaceContext::OnLoopAfterSceneRenderer( context, events );
    }
};

//!============================================================================
//! @brief 初期化/終了フレームワーク搭載テストフィクスチャ
//!============================================================================
class TestGlvFixture : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        InitializeGraphicsPeripheral();

        static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration(
            glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary,
            glv::HidInitialConfiguration::PadSamplingVariation_Normal
        );

        // GLVランタイムエンジンの初期化及びアプリケーションフレームワークの初期化
        glv::ApplicationFrameworkInitialize( LocalHidConfiguration );

        // ルートサーフェイスコンテキストの作成
        TestRootSurface* const pContext = new TestRootSurface();
        m_pTestRootSurface = pContext;

        // メインループコールバックを登録
        glv::ApplicationFrameworkRegisterLoopCallback( pContext );

        // メインループ.
        glv::Application::run();
    }

    static void TearDownTestCase()
    {
        // ルートサーフェイスコンテキストの解放
        TestRootSurface* pContext;
        if ( nullptr != ( pContext = m_pTestRootSurface ) )
        {
            m_pTestRootSurface = nullptr;
            delete pContext;
        }

        // エンジンコンテキストの解放
        glv::Application::quit();
    }

private:
    static TestRootSurface* m_pTestRootSurface;
};

TestRootSurface* TestGlvFixture::m_pTestRootSurface = nullptr;

//!============================================================================
//! @brief 内部テストシンボル作成用マクロ
//!============================================================================
#define INNER_TEST_CLASS_NAME_( _test_case_name_, _test_name_ ) \
    TestCase##_test_case_name_##_##_test_name_

//!============================================================================
//! @brief TEST, TEST_F ラッピングマクロ
//!============================================================================
#define UNIT_SEQUENCE( _test_case_name_, _test_name_ )                          \
class INNER_TEST_CLASS_NAME_( _test_case_name_, _test_name_ )                   \
{                                                                               \
public:                                                                         \
    static void RunTestSequence() NN_NOEXCEPT;                                  \
};                                                                              \
                                                                                \
TEST_F( TestGlvFixture, _test_case_name_##_##_test_name_ )                      \
{                                                                               \
    INNER_TEST_CLASS_NAME_( _test_case_name_, _test_name_ )::RunTestSequence(); \
}                                                                               \
                                                                                \
void INNER_TEST_CLASS_NAME_( _test_case_name_, _test_name_ )::RunTestSequence() NN_NOEXCEPT

#else // defined( ENABLE_GLV_FRAMEWORK_SETUP )

#define UNIT_SEQUENCE   TEST

#endif // defined( ENABLE_GLV_FRAMEWORK_SETUP )

} // unnamed namespace


UNIT_SEQUENCE( DataObject, Basics )
{
    // basics
    const int s1 = 5, s2 = 4, s3 = 3, s4 = 2;
    Data d( Data::INT, s1, s2, s3, s4 );

    EXPECT_TRUE( d.hasData() );
    EXPECT_EQ( Data::INT, d.type() );
    EXPECT_EQ( sizeof( int ), d.sizeType() );
    EXPECT_TRUE( d.isNumerical() );
    EXPECT_EQ( 0, d.offset() );
    EXPECT_EQ( 4, d.order() );
    EXPECT_EQ( 1, d.stride() );

    EXPECT_EQ( s1, d.size( 0 ) );
    EXPECT_EQ( s2, d.size( 1 ) );
    EXPECT_EQ( s3, d.size( 2 ) );
    EXPECT_EQ( s4, d.size( 3 ) );
    EXPECT_EQ( s1*s2, d.size( 0, 1 ) );
    EXPECT_EQ( s1*s2*s3, d.size( 0, 1, 2 ) );
    EXPECT_EQ( d.size(), d.size( 0, 1, 2, 3 ) );

    EXPECT_EQ( 0, d.indexFlat( 0, 0, 0, 0 ) );
    EXPECT_EQ( 1, d.indexFlat( 1, 0, 0, 0 ) );
    EXPECT_EQ( s1 * 1, d.indexFlat( 0, 1, 0, 0 ) );
    EXPECT_EQ( s1 * 2, d.indexFlat( 0, 2, 0, 0 ) );
    EXPECT_EQ( s2 * s1 * 1, d.indexFlat( 0, 0, 1, 0 ) );
    EXPECT_EQ( s2 * s1 * 2, d.indexFlat( 0, 0, 2, 0 ) );
    EXPECT_EQ( s3 * s2 * s1 * 1, d.indexFlat( 0, 0, 0, 1 ) );
    EXPECT_EQ( s3 * s2 * s1 * 2, d.indexFlat( 0, 0, 0, 2 ) );

    for ( int i = 0; i < d.size(); ++i )
    {
        int i1, i2, i3, i4;
        d.indexDim( i1, i2, i3, i4, i );
        EXPECT_TRUE( d.inBounds( i1, i2, i3, i4 ) );
        EXPECT_EQ( i, d.indexFlat( i1, i2, i3, i4 ) );
    }

    for ( int i = 0; i < d.size(); ++i )
    {
        d.assign( i, i );
    }

    for ( int i = 0; i < d.size(); ++i )
    {
        EXPECT_EQ( i, d.elem<int>( i ) );
    }

    // resizing
    {
        Data e( Data::INT, 8 );
        //int N; // resize change, in bytes

        for ( int i = 0; i < e.size(); ++i )
        {
            e.assign( i, i );
        }

        int N = e.resize( 16 );
        EXPECT_EQ( N, e.sizeType() * 8 );

        // old elements should be copied over; extras should be zero
        for ( int i = 0; i < 8; ++i )
        {
            EXPECT_EQ( i, e.at<int>( i ) );
        }

        for ( int i = 8; i < 16; ++i )
        {
            EXPECT_EQ( 0, e.at<int>( i ) );
        }

        N = e.resize( 0 );
        EXPECT_EQ( N, e.sizeType() * -16 );

        N = e.resize( 8 );
        EXPECT_EQ( N, e.sizeType() * 8 );

        for ( int i = 0; i < e.size(); ++i )
        {
            EXPECT_EQ( 0, e.at<int>( i ) );
        }
    }

    // cloning data
    {
        Data e = d;
        EXPECT_EQ( d, e );

        e.clone();
        EXPECT_NE( d.elems<int>(), e.elems<int>() );
        EXPECT_EQ( d, e );
    }

    // reversed slice
    {
        Data e = d.reversed();

        for ( int i = 0; i < e.size(); ++i )
        {
            EXPECT_EQ( d.elem<int>( d.size() - 1 - i ), e.elem<int>( i ) );
        }

        e.clone();
        for ( int i = 0; i < e.size(); ++i )
        {
            EXPECT_EQ( d.elem<int>( d.size() - 1 - i ), e.elem<int>( i ) );
        }

        e += d;
        for ( int i = 0; i < e.size(); ++i )
        {
            EXPECT_EQ( e.elem<int>( i ), e.size() - 1 );
        }
    }
}

UNIT_SEQUENCE( DataObject, ReferenceCountingOther )
{
    // reference counting
    {
        Data d1( Data::INT, 4, 4 );
        EXPECT_EQ( 1, Data::references( d1.elems<int>() ) );

        {
            Data d2 = d1;
            EXPECT_EQ( d1.elems<int>(), d2.elems<int>() );
            EXPECT_EQ( 2, Data::references( d1.elems<int>() ) );
        }
        EXPECT_EQ( 1, Data::references( d1.elems<int>() ) );

        {
            Data d2 = d1.slice( 1 );
            Data d3 = d2.slice( 1 );
            EXPECT_EQ( 3, Data::references( d1.elems<int>() ) );
        }
        EXPECT_EQ( 1, Data::references( d1.elems<int>() ) );
    }

    const float cf = 10;
    float f = 10;
    std::string s = "hello";

    Data d( cf );

    EXPECT_TRUE( d.hasData() );
    EXPECT_EQ( 1, d.size() );
    EXPECT_EQ( Data::FLOAT, d.type() );
    EXPECT_FLOAT_EQ( cf, d.at<float>( 0 ) );
    EXPECT_EQ( "10", d.toString() );
    EXPECT_EQ( "10", d.at<std::string>( 0 ) );

    // Getting/setting values
    d.assign( 11.f );
    EXPECT_FLOAT_EQ( 11.f, d.at<float>( 0 ) );

    d.clone();
    EXPECT_TRUE( d.hasData() );
    EXPECT_EQ( 1, d.size() );
    EXPECT_EQ( Data::FLOAT, d.type() );

    d.assign( 11.f );
    EXPECT_FLOAT_EQ( 11.f, d.at<float>( 0 ) );

    d.assign( 100 );
    EXPECT_EQ( 100, d.at<float>( 0 ) );

    d.set( f );
    EXPECT_EQ( &f, d.elems<float>() );
    EXPECT_FLOAT_EQ( f, d.at<float>( 0 ) );

    d.set( s );
    EXPECT_EQ( Data::STRING, d.type() );
    EXPECT_EQ( 1, d.size() );
    EXPECT_EQ( &s, d.elems<std::string>() );
    EXPECT_EQ( s, d.at<std::string>( 0 ) );

    d.clone();
    EXPECT_NE( &s, d.elems<std::string>() );
    EXPECT_EQ( s, d.at<std::string>( 0 ) );

    d.set( "hello" );
    EXPECT_EQ( Data::STRING, d.type() );
    EXPECT_EQ( "hello", d.at<std::string>( 0 ) );

    {
        std::string str_d2 = "d2";
        Data d2( str_d2 );

        d.set( "d" );
        d2.assign( d );
        EXPECT_EQ( d2.at<std::string>( 0 ), d.at<std::string>( 0 ) );
    }

    {
        Data a( Data::FLOAT, 3 );
        a.assign( 10, 0 );
        a.assign( 20, 1 );
        a.assign( 30, 2 );
        float b[ 3 ];
        a.copyTo( b );
        for ( int i = 0; i < 3; ++i )
        {
            EXPECT_EQ( a.at<float>( i ), b[ i ] );
        }
    }
}

UNIT_SEQUENCE( DataObject, CheckingElements )
{
    Data::Type types[] = { Data::BOOL, Data::INT, Data::FLOAT, Data::DOUBLE };
    for ( int i = 0; i < 4; ++i )
    {
        Data a( types[ i ], 8 );
        a.assignAll( 0 );
        EXPECT_TRUE( a.isZero() );
        a.assign( 1, 1 );
        EXPECT_TRUE( !a.isZero() );
        a.assign( 1, 3 );
        a.assign( 1, 5 );
        a.assign( 1, 7 );
        EXPECT_TRUE( a.slice( 0, 4, 2 ).isZero() );
    }
}

UNIT_SEQUENCE( DataObject, Searching )
{
    Data e( Data::STRING, 3 );
    const int npos = Data::npos;
    std::string s1 = "test1", s2 = "test2", s3 = "test3";
    e.assign( s1, 0 );
    e.assign( s2, 1 );
    e.assign( s3, 2 );

    EXPECT_EQ( 0, e.indexOf( s1 ) );
    EXPECT_EQ( 1, e.indexOf( s2 ) );
    EXPECT_EQ( 2, e.indexOf( s3 ) );
    EXPECT_EQ( npos, e.indexOf( std::string( "invalid" ) ) );
}

UNIT_SEQUENCE( DataObject, MultipleElementAssignment )
{
#define ASSERT_EQUALS(a,b,c,d,e)\
            EXPECT_EQ(a, d1.at<int>(0));\
            EXPECT_EQ(b, d1.at<int>(1));\
            EXPECT_EQ(c, d1.at<int>(2));\
            EXPECT_EQ(d, d1.at<int>(3));\
            EXPECT_EQ(e, d1.at<int>(4))

    Data d1( Data::INT, 5 );
    d1.assignAll( -1 );
    ASSERT_EQUALS( -1, -1, -1, -1, -1 );

    d1.assignAll( 0 );
    ASSERT_EQUALS( 0, 0, 0, 0, 0 );

    {
        float t[] = { 1,2,3,4,5 };
        d1.assignFromArray( t, 2 );
        ASSERT_EQUALS( 1, 2, 0, 0, 0 );

        d1.assignFromArray( t + 2, 2, 1, 2 );
        ASSERT_EQUALS( 1, 2, 3, 4, 0 );

        d1.assignAll( 0 );
        d1.assignFromArray( t, 3, 2 );
        ASSERT_EQUALS( 1, 3, 5, 0, 0 );

        d1.assignFromArray( t + 1, 2, 2, 3 );
        ASSERT_EQUALS( 1, 3, 5, 2, 4 );
    }
#undef ASSERT_EQUALS
}

UNIT_SEQUENCE( StringUtil, Token )
{
#define SET4(x, a,b,c,d) x[0]=a; x[1]=b; x[2]=c; x[3]=d

#define EQ4(x, a,b,c,d) \
        EXPECT_EQ(a, x[0]);\
        EXPECT_EQ(b, x[1]);\
        EXPECT_EQ(c, x[2]);\
        EXPECT_EQ(d, x[3])

#define EQB4(x, a,b,c,d) \
        EXPECT_EQ(a, static_cast<int>(x[0]));\
        EXPECT_EQ(b, static_cast<int>(x[1]));\
        EXPECT_EQ(c, static_cast<int>(x[2]));\
        EXPECT_EQ(d, static_cast<int>(x[3]))

#define EQF4(x, a,b,c,d) \
        EXPECT_FLOAT_EQ(a, x[0]);\
        EXPECT_FLOAT_EQ(b, x[1]);\
        EXPECT_FLOAT_EQ(c, x[2]);\
        EXPECT_FLOAT_EQ(d, x[3])

#define EQD4(x, a,b,c,d) \
        EXPECT_DOUBLE_EQ(a, x[0]);\
        EXPECT_DOUBLE_EQ(b, x[1]);\
        EXPECT_DOUBLE_EQ(c, x[2]);\
        EXPECT_DOUBLE_EQ(d, x[3])

    bool b1;
    float f1;
    double d1;
    std::string s1;

    bool b4[ 4 ];
    float f4[ 4 ];
    double d4[ 4 ];
    std::string s4[ 4 ];

    int N;    // number of elements converted

    N = fromToken( f1, "1e-2" );
    EXPECT_EQ( 1, N );
    EXPECT_FLOAT_EQ( float( 1e-2 ), f1 );

    N = fromToken( f1, "1000" );
    EXPECT_EQ( 1, N );
    EXPECT_FLOAT_EQ( float( 1000 ), f1 );

    N = fromToken( d1, "1e-2" );
    EXPECT_EQ( 1, N );
    EXPECT_DOUBLE_EQ( double( 1e-2 ), d1 );

    N = fromToken( d1, "1000" );
    EXPECT_EQ( 1, N );
    EXPECT_DOUBLE_EQ( double( 1000 ), d1 );

    N = fromToken( b1, "0" );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( 0, static_cast<int>( b1 ) );

    N = fromToken( b1, "1" );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( 1, static_cast<int>( b1 ) );

    N = fromToken( s1, "\"test\"" );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( "test", s1 );

    N = fromToken( b4, 4, 1, "{1,0,1,1}" );
    EXPECT_EQ( 4, N );
    EQB4( b4, 1, 0, 1, 1 );

    N = fromToken( b4, 4, 1, "{0,  1, 0,0}" );
    EXPECT_EQ( 4, N );
    EQB4( b4, 0, 1, 0, 0 );

    N = fromToken( f4, 4, 1, "{1, -1.2, 1e10, +.1}" );
    EXPECT_EQ( 4, N );
    EQF4( f4, 1.f, -1.2f, 1e10f, +.1f );

    N = fromToken( d4, 4, 1, "{1, -1.2, 1e10, +.1}" );
    EXPECT_EQ( 4, N );
    EQD4( d4, 1, -1.2, 1e10, +.1 );

    N = fromToken( s4, 4, 1, "{\"one\", \"two\", \"three\", \"four\"}" );
    EXPECT_EQ( 4, N );
    EQ4( s4, "one", "two", "three", "four" );

    N = toToken( s1, 1000.f );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( "1000", s1 );

    N = toToken( s1, 2000.0 );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( "2000", s1 );

    N = toToken( s1, true );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( "1", s1 );

    N = toToken( s1, false );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( "0", s1 );

    N = toToken( s1, "test" );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( "\"test\"", s1 );

    N = toToken( s1, std::string( "test" ) );
    EXPECT_EQ( 1, N );
    EXPECT_EQ( "\"test\"", s1 );

    SET4( b4, 1, 0, 1, 1 );
    N = toToken( s1, b4, 4, 1 );
    EXPECT_EQ( 4, N );
    EXPECT_EQ( "{1, 0, 1, 1}", s1 );

    SET4( f4, -1, 0.1, 3, 1e10 );
    N = toToken( s1, f4, 4, 1 );
    EXPECT_EQ( 4, N );
    EXPECT_TRUE( ( s1 == "{-1, 0.1, 3, 1e+10}" || s1 == "{-1, 0.1, 3, 1e+010}" ) );

    SET4( d4, -1, 0.1, 3, 1e10 );
    N = toToken( s1, d4, 4, 1 );
    EXPECT_EQ( 4, N );
    EXPECT_TRUE( ( s1 == "{-1, 0.1, 3, 1e+10}" || s1 == "{-1, 0.1, 3, 1e+010}" ) );

    SET4( s4, "one", "two", "three", "four" );
    N = toToken( s1, s4, 4, 1 );
    EXPECT_EQ( 4, N );
    EXPECT_EQ( "{\"one\", \"two\", \"three\", \"four\"}", s1 );

#undef SET4
#undef EQ4
#undef EQB4
#undef EQF4
#undef EQD4
}

UNIT_SEQUENCE( ViewObject, LinkList )
{
    View n00, n10, n11, n20;

    n00.add( n10 );
    n00.add( n11 );
    n10.add( n20 );

    // check all of our links
    EXPECT_EQ( &n10, n00.child );
    EXPECT_EQ( nullptr, n00.parent );
    EXPECT_EQ( nullptr, n00.sibling );
    EXPECT_EQ( &n00, n10.parent );
    EXPECT_EQ( &n11, n10.sibling );
    EXPECT_EQ( &n20, n10.child );
    EXPECT_EQ( &n00, n11.parent );
    EXPECT_EQ( nullptr, n11.child );
    EXPECT_EQ( nullptr, n11.sibling );
    EXPECT_EQ( &n10, n20.parent );
    EXPECT_EQ( nullptr, n20.child );
    EXPECT_EQ( nullptr, n20.sibling );

    n10.makeLastSibling();
    EXPECT_EQ( nullptr, n10.sibling );
    EXPECT_EQ( &n10, n11.sibling );
    n11.makeLastSibling();

    n10.remove();
    EXPECT_EQ( nullptr, n10.parent );
    EXPECT_EQ( nullptr, n10.sibling );
    EXPECT_EQ( nullptr, n00.child->sibling );

    n11.remove();
    EXPECT_EQ( nullptr, n00.child );

    EXPECT_EQ( &n20, n10.child );
}

UNIT_SEQUENCE( ViewObject, MemoryManagement )
{
    View * v0d = new View;
    View v1s;
    View * v1d = new View;

    EXPECT_TRUE( v0d->dynamicAlloc() );
    EXPECT_FALSE( v1s.dynamicAlloc() );

    *v0d << v1s << v1d;

    delete v0d;
}

UNIT_SEQUENCE( NotifierObject, Basic )
{
    bool bv1 = false, bv2 = false, bf = false;
    Notifier n;

    EXPECT_EQ( 0, n.numObservers( Update::Value ) );

    n.attach( ntSetBool, Update::Value, &bv1 );
    n.attach( ntSetBool, Update::Value, &bv2 );
    n.attach( ntSetBool, Update::Focus, &bf );

    EXPECT_EQ( 2, n.numObservers( Update::Value ) );
    EXPECT_EQ( 1, n.numObservers( Update::Focus ) );

    n.notify( Update::Value );
    EXPECT_TRUE( bv1 );
    EXPECT_TRUE( bv2 );
    EXPECT_FALSE( bf );

    n.notify( Update::Focus );
    EXPECT_TRUE( bf );

    bv1 = bv2 = false;
    n.detach( ntSetBool, Update::Value, &bv1 );
    n.notify( Update::Value );
    EXPECT_FALSE( bv1 );
    EXPECT_TRUE( bv2 );

    bv1 = bv2 = false;
    n.detach( ntSetBool, Update::Value, &bv2 );
    n.notify( Update::Value );
    EXPECT_FALSE( bv1 );
    EXPECT_FALSE( bv2 );
}

UNIT_SEQUENCE( ButtonObject, Basic )
{
    bool b = false;
    Button w;
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( true );
    EXPECT_TRUE( b );
    EXPECT_TRUE( w.getValue() );

    // Setting to current value should NOT send notification to avoid infinite loops
    b = false;
    w.setValue( w.getValue() );
    EXPECT_FALSE( b );

    // Boolean model variable
    bool v = false;
    w.attachVariable( v );
    w.setValue( true );
    EXPECT_TRUE( v );
    w.setValue( false );
    EXPECT_FALSE( v );

    v = true;
    w.onDataModelSync();
    EXPECT_TRUE( w.getValue() );

    w.setValue( true );

    EXPECT_EQ( "1", w.data().toToken() );
    w.setValue( false );
    EXPECT_EQ( "0", w.data().toToken() );
    w.setDataFromString( "0" );
    EXPECT_FALSE( w.getValue() );
    w.setDataFromString( "1" );
    EXPECT_TRUE( w.getValue() );

    EXPECT_EQ( 0, w.setDataFromString( "invalid" ) );
}

UNIT_SEQUENCE( ButtonsObject, Basic )
{
    Buttons w;
    bool b = false;
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( true, 0 );
    EXPECT_TRUE( b );
    EXPECT_TRUE( w.getValue( 0 ) );

    w.data().resize( 2, 2 );
    w.data().assignAll( 0 );

    bool v1 = false;
    bool v2 = false;
    w.attachVariable( v1, 0 );
    w.attachVariable( v2, 2 );

    w.setValue( true, 0 );
    EXPECT_TRUE( v1 );
    w.setValue( true, 2 );
    EXPECT_TRUE( v2 );
    w.setValue( false, 0 );
    EXPECT_FALSE( v1 );
    w.setValue( false, 2 );
    EXPECT_FALSE( v2 );

    w.setValue( false, 0 );
    w.setValue( false, 1 );
    w.setValue( false, 2 );
    w.setValue( false, 3 );
    EXPECT_EQ( "{0, 0, 0, 0}", w.data().toToken() );

    v1 = v2 = false;
    w.setDataFromString( "{1,1,1,1}" );
    //assert(w.getValue(0) && w.getValue(1) && w.getValue(2) && w.getValue(3));
    EXPECT_TRUE( w.getValue( 0 ) );
    EXPECT_TRUE( w.getValue( 1 ) );
    EXPECT_TRUE( w.getValue( 2 ) );
    EXPECT_TRUE( w.getValue( 3 ) );

    //assert(v1 && v2);
    EXPECT_TRUE( v1 );
    EXPECT_TRUE( v2 );
}

UNIT_SEQUENCE( SliderObject, Basic )
{
    bool b = false;
    Slider w;
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( 0.99f );
    EXPECT_TRUE( b );
    EXPECT_FLOAT_EQ( 0.99f, static_cast<float>( w.getValue() ) );

    // Setting to current value should NOT send notification to avoid infinite loops
    b = false;
    w.setValue( w.getValue() );
    EXPECT_FALSE( b );

    float v;
    w.attachVariable( v );
    w.setValue( 0.99 );
    EXPECT_FLOAT_EQ( 0.99f, v );

    v = 0;
    w.onDataModelSync();
    EXPECT_DOUBLE_EQ( 0, w.getValue() );

    std::string s;
    w.setValue( 0.25 );
    EXPECT_EQ( "0.25", w.data().toToken() );
    w.setDataFromString( "0.5" );
    EXPECT_DOUBLE_EQ( 0.5, w.getValue() );
    EXPECT_FLOAT_EQ( 0.5, v );

    EXPECT_EQ( 0, w.setDataFromString( "invalid" ) );
    EXPECT_DOUBLE_EQ( 0.5, w.getValue() );
}

UNIT_SEQUENCE( SlidersObject, Basic )
{
    bool b = false;
    Sliders w( Rect( 1 ), 2, 2 );
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( 0.99f, 3 );
    EXPECT_TRUE( b );
    EXPECT_FLOAT_EQ( 0.99f, static_cast<float>( w.getValue( 3 ) ) );

    float v1, v2;
    w.attachVariable( v1, 0 );
    w.attachVariable( v2, 1 );

    w.setValue( 0.1f, 0 );
    EXPECT_FLOAT_EQ( 0.1f, v1 );
    w.setValue( 0.2f, 0 );
    EXPECT_FLOAT_EQ( 0.2f, v1 );

    v1 = 0.8f;
    v2 = 0.9f;
    w.onDataModelSync();
    EXPECT_FLOAT_EQ( 0.8f, static_cast<float>( w.getValue( 0 ) ) );
    EXPECT_FLOAT_EQ( 0.9f, static_cast<float>( w.getValue( 1 ) ) );

    w.setValue( 0.1f, 0 );
    w.setValue( 0.2f, 1 );
    w.setValue( 0.3f, 2 );
    w.setValue( 0.4f, 3 );
    EXPECT_EQ( "{0.1, 0.2, 0.3, 0.4}", w.data().toToken() );

    v1 = v2 = 0;
    w.setDataFromString( "{0.4,0.3,0.2,0.1}" );
    EXPECT_DOUBLE_EQ( 0.4, w.getValue( 0 ) );
    EXPECT_DOUBLE_EQ( 0.3, w.getValue( 1 ) );
    EXPECT_DOUBLE_EQ( 0.2, w.getValue( 2 ) );
    EXPECT_DOUBLE_EQ( 0.1, w.getValue( 3 ) );
    EXPECT_FLOAT_EQ( v1, static_cast<float>( w.getValue( 0 ) ) );
    EXPECT_FLOAT_EQ( v2, static_cast<float>( w.getValue( 1 ) ) );
}

UNIT_SEQUENCE( Slider2DObject, Basic )
{
    bool b = false;
    Slider2D w;
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( 0.01f, 0 );
    EXPECT_TRUE( b );
    EXPECT_FLOAT_EQ( 0.01f, static_cast<float>( w.getValue( 0 ) ) );

    // Setting to current value should NOT send notification to avoid infinite loops
    b = false;
    w.setValue( 0.01f, 0 );
    EXPECT_FALSE( b );

    b = false;
    w.setValue( 0.00f, 0 );
    w.setValue( 0.01f, 1 );
    EXPECT_TRUE( b );
    EXPECT_FLOAT_EQ( 0.01f, static_cast<float>( w.getValue( 1 ) ) );

    b = false;
    w.setValueMax();
    EXPECT_TRUE( b );
    EXPECT_DOUBLE_EQ( 1.0, w.getValue( 0 ) );
    EXPECT_DOUBLE_EQ( 1.0, w.getValue( 1 ) );

    float v1, v2;
    w.attachVariable( v1, 0 );
    w.attachVariable( v2, 1 );

    w.setValue( 0.5f, 0 );
    EXPECT_FLOAT_EQ( 0.5f, v1 );
    w.setValue( 0.6f, 1 );
    EXPECT_FLOAT_EQ( 0.6f, v2 );

    v1 = 0.1;
    v2 = 0.2;

    w.onDataModelSync();
    EXPECT_FLOAT_EQ( v1, static_cast<float>( w.getValue( 0 ) ) );
    EXPECT_FLOAT_EQ( v2, static_cast<float>( w.getValue( 1 ) ) );

    w.setValue( 0.2, 0 );
    w.setValue( 0.3, 1 );
    EXPECT_EQ( "{0.2, 0.3}", w.data().toToken() );

    v1 = v2 = 0;
    w.setDataFromString( "{0.7, 0.8}" );
    EXPECT_DOUBLE_EQ( 0.7, w.getValue( 0 ) );
    EXPECT_DOUBLE_EQ( 0.8, w.getValue( 1 ) );

    EXPECT_FLOAT_EQ( v1, static_cast<float>( w.getValue( 0 ) ) );
    EXPECT_FLOAT_EQ( v2, static_cast<float>( w.getValue( 1 ) ) );
}

UNIT_SEQUENCE( SliderGridObject, Basic )
{
    bool b = false;
    SliderGrid<4> w;
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( 0.01f, 0 );
    EXPECT_TRUE( b );
    EXPECT_FLOAT_EQ( 0.01f, static_cast<float>( w.getValue( 0 ) ) );

    float v1, v2;
    w.attachVariable( v1, 0 );
    w.attachVariable( v2, 1 );

    w.setValue( 0.1f, 0 );
    EXPECT_FLOAT_EQ( 0.1f, v1 );
    w.setValue( 0.2f, 0 );
    EXPECT_FLOAT_EQ( 0.2f, v1 );

    v1 = 0.8f; v2 = 0.9f;
    w.onDataModelSync();
    EXPECT_FLOAT_EQ( 0.8f, static_cast<float>( w.getValue( 0 ) ) );
    EXPECT_FLOAT_EQ( 0.9f, static_cast<float>( w.getValue( 1 ) ) );

    w.setValue( 0.1f, 0 );
    w.setValue( 0.2f, 1 );
    w.setValue( 0.3f, 2 );
    w.setValue( 0.4f, 3 );
    EXPECT_EQ( "{0.1, 0.2, 0.3, 0.4}", w.data().toToken() );

    v1 = v2 = 0;
    w.setDataFromString( "{0.4,0.3,0.2,0.1}" );
    EXPECT_DOUBLE_EQ( 0.4, w.getValue( 0 ) );
    EXPECT_DOUBLE_EQ( 0.3, w.getValue( 1 ) );
    EXPECT_DOUBLE_EQ( 0.2, w.getValue( 2 ) );
    EXPECT_DOUBLE_EQ( 0.1, w.getValue( 3 ) );

    EXPECT_FLOAT_EQ( v1, static_cast<float>( w.getValue( 0 ) ) );
    EXPECT_FLOAT_EQ( v2, static_cast<float>( w.getValue( 1 ) ) );
}

UNIT_SEQUENCE( SliderRangeGridObject, Basic )
{
    bool b = false;
    SliderRange w;
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( 0.01f, 0 );
    w.setValue( 0.02f, 1 );
    EXPECT_TRUE( b );
    EXPECT_FLOAT_EQ( 0.01f, static_cast<float>( w.getValue( 0 ) ) );
    EXPECT_FLOAT_EQ( 0.02f, static_cast<float>( w.getValue( 1 ) ) );
    EXPECT_FLOAT_EQ( 0.015f, static_cast<float>( w.center() ) );
    EXPECT_FLOAT_EQ( 0.01f, static_cast<float>( w.range() ) );

    w.endpoints( 0, 1 );
    EXPECT_DOUBLE_EQ( 0, w.getValue( 0 ) );
    EXPECT_DOUBLE_EQ( 1, w.getValue( 1 ) );

    w.centerRange( 0.5, 0.25 );
    EXPECT_DOUBLE_EQ( ( 0.5 - 0.25 / 2 ), w.getValue( 0 ) );
    EXPECT_DOUBLE_EQ( ( 0.5 + 0.25 / 2 ), w.getValue( 1 ) );

    float v1, v2;
    w.attachVariable( v1, 0 );
    w.attachVariable( v2, 1 );

    w.setValue( 0.2, 0 );
    w.setValue( 0.3, 1 );
    EXPECT_EQ( "{0.2, 0.3}", w.data().toToken() );

    v1 = v2 = 0;
    w.setDataFromString( "{0.7, 0.8}" );
    EXPECT_DOUBLE_EQ( 0.7, w.getValue( 0 ) );
    EXPECT_DOUBLE_EQ( 0.8, w.getValue( 1 ) );

    EXPECT_FLOAT_EQ( v1, static_cast<float>( w.getValue( 0 ) ) );
    EXPECT_FLOAT_EQ( v2, static_cast<float>( w.getValue( 1 ) ) );
}

UNIT_SEQUENCE( LabelObject, Basic )
{
    bool b = false;
    Label w;

    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( "test" );
    EXPECT_TRUE( b );
    EXPECT_EQ( "test", w.getValue() );

    EXPECT_EQ( "\"test\"", w.data().toToken() );

    w.setValue( "" );
    w.setDataFromString( "\"test\"" );
    EXPECT_EQ( "test", w.getValue() );
}

UNIT_SEQUENCE( NumberDialerObject, Basic )
{
    bool b = false;
    NumberDialer w( 1, 2, 1, 0 );
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( 0.75 );
    EXPECT_TRUE( b );
    EXPECT_DOUBLE_EQ( 0.75, w.getValue() );

    // Setting to current value should NOT send notification to avoid infinite loops
    b = false;
    w.setValue( w.getValue() );
    EXPECT_FALSE( b );

    double v = 0;
    w.attachVariable( v );
    w.setValue( 0.5 );
    EXPECT_DOUBLE_EQ( 0.5, v );

    v = 0.25;
    w.onDataModelSync();
    EXPECT_DOUBLE_EQ( v, w.getValue() );

    w.setValue( 0.2 );
    EXPECT_EQ( "0.2", w.data().toToken() );

    v = 0;
    w.setDataFromString( "0.8" );
    EXPECT_DOUBLE_EQ( 0.8, w.getValue() );
    EXPECT_DOUBLE_EQ( v, w.getValue() );
}

UNIT_SEQUENCE( TextViewObject, Basic )
{
    bool b = false;
    TextView w;
    w.attach( ntSetBool, Update::Value, &b );
    w.setValue( "hello" );
    EXPECT_TRUE( b );
    EXPECT_EQ( "hello", w.getValue() );

    EXPECT_EQ( "\"hello\"", w.data().toToken() );

    std::string v = "test";
    w.attachVariable( v );

    w.onDataModelSync();
    EXPECT_EQ( "test", w.getValue() );

    w.setDataFromString( "\"world\"" );
    EXPECT_EQ( "world", w.getValue() );
    EXPECT_EQ( "world", v );
}

namespace {

void ForceHalfWrapText(Label& label)
{
    label.size(25.f);
    label.paddingX(0.f);
    label.paddingY(0.f);
    label.pos(Place::CC, 0, 0).anchor(Place::CC);
    label.WrapText(label.width() / 2.f);
}

template<typename CharType, typename StringType>
void InsertAndErase(const StringType& source,
    const StringType& expectAtInserted,
    const char32_t insertCharacter,
    const typename Ucs4Editor<CharType, StringType>::size_type sourceLength,
    const typename Ucs4Editor<CharType, StringType>::size_type insertPosition
)
{
    Ucs4Editor<CharType, StringType> e(source);
    EXPECT_EQ(sourceLength, e.size());

    e.insert(insertPosition, 1, insertCharacter);
    EXPECT_EQ(sourceLength + 1, e.size());
    EXPECT_TRUE(expectAtInserted == e.GetValue());

    e.erase(insertPosition, 1);
    EXPECT_EQ(sourceLength, e.size());
    EXPECT_TRUE(source == e.GetValue());
}

}

UNIT_SEQUENCE(UTF8, Label)
{
    bool b = false;
    Label w;

    const auto pValue = GLV_TEXT_API_STRING_UTF8("テストUTF8日本語");
    w.attach(ntSetBool, Update::Value, &b);
    w.setValue(pValue);
    EXPECT_TRUE(b);
    EXPECT_EQ(pValue, w.getValue());

    w.setValue("");
    w.setDataFromString(GLV_TEXT_API_STRING_UTF8("\"テストUTF8日本語\""));
    EXPECT_EQ(pValue, w.getValue());
}

UNIT_SEQUENCE(UTF8, WrapText)
{
    Label utf8(GLV_TEXT_API_STRING_UTF8("ああああああああああああああああああああ"), false);
    ForceHalfWrapText(utf8);
    EXPECT_EQ(GLV_TEXT_API_STRING_UTF8("ああああああああああ\nああああああああああ"), utf8.getValue());

    Label utf16(GLV_TEXT_API_STRING_UTF16("ああああああああああああああああああああ"), false);
    ForceHalfWrapText(utf16);
    EXPECT_TRUE(*utf16.getRawTextOnWide() == GLV_TEXT_API_STRING_UTF16("ああああああああああ\nああああああああああ"));
}

UNIT_SEQUENCE(UTF8, Ucs4Editor)
{
    const auto insertChar = GLV_TEXT_API_STRING_UTF32('あ');
    const auto baseU8 = GLV_TEXT_API_STRING_UTF8("基準文字列");
    InsertAndErase<char, std::string>(baseU8, GLV_TEXT_API_STRING_UTF8("あ基準文字列"), insertChar, 5, 0);
    InsertAndErase<char, std::string>(baseU8, GLV_TEXT_API_STRING_UTF8("基あ準文字列"), insertChar, 5, 1);
    InsertAndErase<char, std::string>(baseU8, GLV_TEXT_API_STRING_UTF8("基準あ文字列"), insertChar, 5, 2);
    InsertAndErase<char, std::string>(baseU8, GLV_TEXT_API_STRING_UTF8("基準文あ字列"), insertChar, 5, 3);
    InsertAndErase<char, std::string>(baseU8, GLV_TEXT_API_STRING_UTF8("基準文字あ列"), insertChar, 5, 4);
    InsertAndErase<char, std::string>(baseU8, GLV_TEXT_API_STRING_UTF8("基準文字列あ"), insertChar, 5, 5);

    const auto baseU16 = GLV_TEXT_API_STRING_UTF16("基準文字列");
    InsertAndErase<WideCharacterType, WideString>(baseU16, GLV_TEXT_API_STRING_UTF16("あ基準文字列"), insertChar, 5, 0);
    InsertAndErase<WideCharacterType, WideString>(baseU16, GLV_TEXT_API_STRING_UTF16("基あ準文字列"), insertChar, 5, 1);
    InsertAndErase<WideCharacterType, WideString>(baseU16, GLV_TEXT_API_STRING_UTF16("基準あ文字列"), insertChar, 5, 2);
    InsertAndErase<WideCharacterType, WideString>(baseU16, GLV_TEXT_API_STRING_UTF16("基準文あ字列"), insertChar, 5, 3);
    InsertAndErase<WideCharacterType, WideString>(baseU16, GLV_TEXT_API_STRING_UTF16("基準文字あ列"), insertChar, 5, 4);
    InsertAndErase<WideCharacterType, WideString>(baseU16, GLV_TEXT_API_STRING_UTF16("基準文字列あ"), insertChar, 5, 5);
}
