﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nnt.h>
#include <nn/edid/edid_DisplayInfo.h>
#include <nn/edid/edid_SyncType.h>
#include <detail/edid_DisplayInfo.h>

TEST(DisplayInfo, Initialize)
{
    nn::edid::DisplayInfo display;
    nn::edid::detail::Initialize(&display);

    for( int i = 0; i < sizeof(display.manufacturerCode); ++i )
    {
        EXPECT_EQ(0, display.manufacturerCode[i]);
    }

    EXPECT_EQ(0, display.productCode);
    EXPECT_EQ(0, display.serialNumber);
    EXPECT_EQ(0, display._date);

    EXPECT_EQ(nn::edid::VideoInterface_Digital, display.interfaceType);
    EXPECT_EQ(nn::edid::SyncLevel_Digital, display.sync);
    EXPECT_EQ(nn::edid::BlankLevel_Digital, display.blankType);

    EXPECT_EQ(nn::edid::ColorDepth_Undefined, display.depth);
    EXPECT_EQ(nn::edid::ColorType_Undefined, display.colorSupport);

    EXPECT_TRUE(display.contentTypes.IsAllOff());
    EXPECT_TRUE(display.syncTypes.IsAllOff());

    EXPECT_FALSE(display.isSrgbDefaultColorSpace);
    EXPECT_FALSE(display.isContinuousFrequency);
    EXPECT_FALSE(display.hasVsyncSerrations);
    EXPECT_FALSE(display.isStandbyModeSupported);
    EXPECT_FALSE(display.isSuspendModeSupported);
    EXPECT_FALSE(display.isVeryLowPowerSupported);
    EXPECT_FALSE(display.isYccQuantizationSelectable);
    EXPECT_FALSE(display.isRgbQuantizationSelectable);
}

TEST(DisplayInfo, ToAscii)
{
    // testing boundaries
    EXPECT_EQ(0, nn::edid::detail::ToAscii(0));

    // this is small enough to test exhaustively
    for( std::uint8_t i = 27; i < std::numeric_limits<decltype(i)>::max(); ++i )
    {
        EXPECT_EQ(0, nn::edid::detail::ToAscii(i));
    }
    EXPECT_EQ(0, nn::edid::detail::ToAscii(std::numeric_limits<std::uint8_t>::max()));

    EXPECT_EQ('A', nn::edid::detail::ToAscii(1));
    EXPECT_EQ('B', nn::edid::detail::ToAscii(2));
    EXPECT_EQ('C', nn::edid::detail::ToAscii(3));
    EXPECT_EQ('D', nn::edid::detail::ToAscii(4));
    EXPECT_EQ('E', nn::edid::detail::ToAscii(5));
    EXPECT_EQ('F', nn::edid::detail::ToAscii(6));
    EXPECT_EQ('G', nn::edid::detail::ToAscii(7));
    EXPECT_EQ('H', nn::edid::detail::ToAscii(8));
    EXPECT_EQ('I', nn::edid::detail::ToAscii(9));
    EXPECT_EQ('J', nn::edid::detail::ToAscii(10));
    EXPECT_EQ('K', nn::edid::detail::ToAscii(11));
    EXPECT_EQ('L', nn::edid::detail::ToAscii(12));
    EXPECT_EQ('M', nn::edid::detail::ToAscii(13));
    EXPECT_EQ('N', nn::edid::detail::ToAscii(14));
    EXPECT_EQ('O', nn::edid::detail::ToAscii(15));
    EXPECT_EQ('P', nn::edid::detail::ToAscii(16));
    EXPECT_EQ('Q', nn::edid::detail::ToAscii(17));
    EXPECT_EQ('R', nn::edid::detail::ToAscii(18));
    EXPECT_EQ('S', nn::edid::detail::ToAscii(19));
    EXPECT_EQ('T', nn::edid::detail::ToAscii(20));
    EXPECT_EQ('U', nn::edid::detail::ToAscii(21));
    EXPECT_EQ('V', nn::edid::detail::ToAscii(22));
    EXPECT_EQ('W', nn::edid::detail::ToAscii(23));
    EXPECT_EQ('X', nn::edid::detail::ToAscii(24));
    EXPECT_EQ('Y', nn::edid::detail::ToAscii(25));
    EXPECT_EQ('Z', nn::edid::detail::ToAscii(26));
}

TEST(DisplayInfo, GetMaufacturerCode)
{
    nn::edid::DisplayInfo display;
    nn::edid::detail::Initialize(&display);

    // maps to ABC
    // See Section 3.4.1 in EDID spec
    std::uint16_t code = (1 << 14) | (1 << 9) | (1 << 8) | (1 << 2);
    nn::edid::detail::GetManufacturerCode(&display, code);

    EXPECT_STREQ("ABC", display.manufacturerCode);
    // make sure this is null terminated
    EXPECT_EQ(0, display.manufacturerCode[sizeof(display.manufacturerCode) - 1]);
}

TEST(DisplayInfo, GetVideoInterface)
{
    // Check Table 3.11 in EDID spec for bit definitions

    // first 7 bits can be any value
    for( std::uint8_t i = 0; i <= 0x7F; ++i )
    {
        EXPECT_EQ(nn::edid::VideoInterface_Analog, nn::edid::detail::GetVideoInterface(0x00 | i));
    }

    for( std::uint8_t i = 0; i < 8; ++i )
    {
        // bits 4-6 could be any value
        std::uint8_t unusedBits = (i << 4);
        EXPECT_EQ(nn::edid::VideoInterface_Digital, nn::edid::detail::GetVideoInterface(0x80 | unusedBits));
        EXPECT_EQ(nn::edid::VideoInterface_Dvi, nn::edid::detail::GetVideoInterface(0x81 | unusedBits));
        EXPECT_EQ(nn::edid::VideoInterface_HdmiA, nn::edid::detail::GetVideoInterface(0x82 | unusedBits));
        EXPECT_EQ(nn::edid::VideoInterface_HdmiB, nn::edid::detail::GetVideoInterface(0x83 | unusedBits));
        EXPECT_EQ(nn::edid::VideoInterface_Mddi, nn::edid::detail::GetVideoInterface(0x84 | unusedBits));
        EXPECT_EQ(nn::edid::VideoInterface_DisplayPort, nn::edid::detail::GetVideoInterface(0x85 | unusedBits));
    }
}

TEST(DisplayInfo, GetSignalLevel)
{
    // Check Table 3.11 in EDID spec for bit definitions

    // first 5 bits can be any value
    for( std::uint8_t i = 0; i <= 0x1F; ++i )
    {
        EXPECT_EQ(nn::edid::SyncLevel_300, nn::edid::detail::GetSignalLevel(0x00 | i));
        EXPECT_EQ(nn::edid::SyncLevel_286, nn::edid::detail::GetSignalLevel(0x20 | i));
        EXPECT_EQ(nn::edid::SyncLevel_400, nn::edid::detail::GetSignalLevel(0x40 | i));
        EXPECT_EQ(nn::edid::SyncLevel_000, nn::edid::detail::GetSignalLevel(0x60 | i));
    }

    // for digital signals, only care about bit 7
    for( std::uint8_t i = 0; i <= 0x7F; ++i )
    {
        EXPECT_EQ(nn::edid::SyncLevel_Digital, nn::edid::detail::GetSignalLevel(0x80 | i));
    }
}

TEST(DisplayInfo, GetVideoSetup)
{
    // Check Table 3.11 in EDID spec for bit definitions

    // bits 0-3 can be any value
    for( std::uint8_t i = 0; i <= 0x0F; ++i )
    {
        // bits 5, 6 can be any value
        for( std::uint8_t j = 0; j < 4; ++j )
        {
            std::uint8_t unusedBits = i | (j << 5);

            EXPECT_EQ(nn::edid::BlankLevel_Equal, nn::edid::detail::GetVideoSetup(0x00 | unusedBits));
            EXPECT_EQ(nn::edid::BlankLevel_Setup, nn::edid::detail::GetVideoSetup(0x10 | unusedBits));
        }
    }

    // for digital signals, only care about bit 7
    for( std::uint8_t i = 0; i <= 0x7F; ++i )
    {
        EXPECT_EQ(nn::edid::BlankLevel_Digital, nn::edid::detail::GetVideoSetup(0x80 | i));
    }
}

TEST(DisplayInfo, GetSyncTypes)
{
    // Check Table 3.11 in EDID spec for bit definitions

    // bits 4-6 can be any value
    for( std::uint8_t i = 0; i < 8; ++i )
    {
        // bit 0 can be any value
        for( std::uint8_t j = 0; j < 2; ++j )
        {
            std::uint8_t unusedBits = j | (i << 4);
            nn::edid::SyncTypeSet sync;

            std::uint8_t value = 0x00 | unusedBits;
            sync = nn::edid::detail::GetSyncTypes(value);
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::CompositeSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SeparateSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());

            value = 0x02 | unusedBits;
            sync = nn::edid::detail::GetSyncTypes(value);
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::CompositeSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SeparateSync>());
            EXPECT_TRUE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());

            value = 0x04 | unusedBits;
            sync = nn::edid::detail::GetSyncTypes(value);
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
            EXPECT_TRUE(sync.Test<nn::edid::SyncType::CompositeSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SeparateSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());

            value = 0x08 | unusedBits;
            sync = nn::edid::detail::GetSyncTypes(value);
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::CompositeSync>());
            EXPECT_TRUE(sync.Test<nn::edid::SyncType::SeparateSync>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
            EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());
        }
    }

    // for digital signals, only care about bit 7
    for( std::uint8_t i = 0; i <= 0x7F; ++i )
    {
        nn::edid::SyncTypeSet sync;

        sync = nn::edid::detail::GetSyncTypes(0x80 | i);
        EXPECT_FALSE(sync.Test<nn::edid::SyncType::BipolarCompositeSync>());
        EXPECT_FALSE(sync.Test<nn::edid::SyncType::CompositeSync>());
        EXPECT_FALSE(sync.Test<nn::edid::SyncType::SeparateSync>());
        EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnGreen>());
        EXPECT_FALSE(sync.Test<nn::edid::SyncType::SyncOnRgb>());
    }
}

TEST(DisplayInfo, HasVsyncSerrations)
{
    // Check Table 3.11 in EDID spec for bit definitions

    // bits 1-6 can be any value
    for( std::uint8_t i = 0; i < 64; ++i )
    {
        std::uint8_t unusedBits = i << 1;
        EXPECT_FALSE(nn::edid::detail::HasVsyncSerrations(unusedBits));
        EXPECT_TRUE(nn::edid::detail::HasVsyncSerrations(0x01 | unusedBits));
    }

    // bits 0-6 can be any value
    for( std::uint8_t i = 0; i < 128; ++i )
    {
        EXPECT_FALSE(nn::edid::detail::HasVsyncSerrations(0x80 | i));
    }
}

TEST(DisplayInfo, GetColorDepth)
{
    // Check Table 3.11 in EDID spec for bit definitions

    // bits 0-6 can be any value
    for( std::uint8_t i = 0; i < 128; ++i )
    {
        EXPECT_EQ(nn::edid::ColorDepth_Undefined, nn::edid::detail::GetColorDepth(i));
    }

    // bits 0-3 can be any value
    for( std::uint8_t i = 0; i <= 0x0F; ++i )
    {
        EXPECT_EQ(nn::edid::ColorDepth_Undefined, nn::edid::detail::GetColorDepth(0x80 | i));
        EXPECT_EQ(nn::edid::ColorDepth_6Bits, nn::edid::detail::GetColorDepth(0x90 | i));
        EXPECT_EQ(nn::edid::ColorDepth_8Bits, nn::edid::detail::GetColorDepth(0xA0 | i));
        EXPECT_EQ(nn::edid::ColorDepth_10Bits, nn::edid::detail::GetColorDepth(0xB0 | i));
        EXPECT_EQ(nn::edid::ColorDepth_12Bits, nn::edid::detail::GetColorDepth(0xC0 | i));
        EXPECT_EQ(nn::edid::ColorDepth_14Bits, nn::edid::detail::GetColorDepth(0xD0 | i));
        EXPECT_EQ(nn::edid::ColorDepth_16Bits, nn::edid::detail::GetColorDepth(0xE0 | i));
        EXPECT_EQ(nn::edid::ColorDepth_Undefined, nn::edid::detail::GetColorDepth(0xF0 | i));
    }
}

TEST(DisplayInfo, GetColorSupport)
{
    // Check Table 3.14 in EDID spec for bit definitions

    // only valid for analog signals
    for( std::uint8_t i = 0; i < std::numeric_limits<decltype(i)>::max(); ++i )
    {
        EXPECT_EQ(nn::edid::ColorType_Undefined, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_Digital, i));
        EXPECT_EQ(nn::edid::ColorType_Undefined, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_Dvi, i));
        EXPECT_EQ(nn::edid::ColorType_Undefined, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_HdmiA, i));
        EXPECT_EQ(nn::edid::ColorType_Undefined, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_HdmiB, i));
        EXPECT_EQ(nn::edid::ColorType_Undefined, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_Mddi, i));
        EXPECT_EQ(nn::edid::ColorType_Undefined, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_DisplayPort, i));
    }

    // bits 0-3 can be any value
    for( std::uint8_t i = 0; i < 8; ++i )
    {
        // bits 5-7 can be any value
        for( std::uint8_t j = 0; j < 8; ++j )
        {
            std::uint8_t unusedBits = i | (j << 5);

            EXPECT_EQ(nn::edid::ColorType_Monochrome, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_Analog, unusedBits));
            EXPECT_EQ(nn::edid::ColorType_RgbColor, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_Analog, 0x08 | unusedBits));
            EXPECT_EQ(nn::edid::ColorType_NonRgbColor, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_Analog, 0x10 | unusedBits));
            EXPECT_EQ(nn::edid::ColorType_Undefined, nn::edid::detail::GetColorSupport(nn::edid::VideoInterface_Analog, 0x18 | unusedBits));
        }
    }
}

TEST(DisplayInfo, IsStandbyModeSupported)
{
    // Check Table 3.14 in EDID spec for bit definitions

    // bits 0-6 can be any value
    for( std::uint8_t i = 0; i < 128; ++i )
    {
        EXPECT_FALSE(nn::edid::detail::IsStandbyModeSupported(i));
        EXPECT_TRUE(nn::edid::detail::IsStandbyModeSupported(0x80 | i));
    }
}

TEST(DisplayInfo, IsSuspendModeSupported)
{
    // Check Table 3.14 in EDID spec for bit definitions

    // bits 0-5 can be any value
    for( std::uint8_t i = 0; i < 64; ++i )
    {
        // bit 7 can be any value
        EXPECT_FALSE(nn::edid::detail::IsSuspendModeSupported(i));
        EXPECT_FALSE(nn::edid::detail::IsSuspendModeSupported(i | 0x80));
        EXPECT_TRUE(nn::edid::detail::IsSuspendModeSupported((1 << 6) | i));
        EXPECT_TRUE(nn::edid::detail::IsSuspendModeSupported((1 << 6) | i | 0x80));
    }
}

TEST(DisplayInfo, IsVeryLowPowerSupported)
{
    // Check Table 3.14 in EDID spec for bit definitions

    // bits 0-4 can be any value
    for( std::uint8_t i = 0; i <= 0x0F; ++i )
    {
        // bits 6, 7 can be any value
        for( std::uint8_t j = 0; j < 4; ++j )
        {
            std::uint8_t unusedBits = j << 6;

            EXPECT_FALSE(nn::edid::detail::IsVeryLowPowerSupported(i | unusedBits));
            EXPECT_TRUE(nn::edid::detail::IsVeryLowPowerSupported((1 << 5) | i | unusedBits));
        }
    }
}

TEST(DisplayInfo, IsSrgbDefaultColorSpace)
{
    // Check Table 3.14 in EDID spec for bit definitions

    // bits 0, 1 can be any value
    for( std::uint8_t i = 0; i < 4; ++i )
    {
        // bits 3-7 can be any value
        for( std::uint8_t j = 0; j < 0x1F; ++j )
        {
            std::uint8_t unusedBits = i | (j << 3);

            EXPECT_FALSE(nn::edid::detail::IsSrgbDefaultColorSpace(unusedBits));
            EXPECT_TRUE(nn::edid::detail::IsSrgbDefaultColorSpace(0x04 | unusedBits));
        }
    }
}

TEST(DisplayInfo, IsContinuousFrequency)
{
    // Check Table 3.14 in EDID spec for bit definitions

    // bits 1-7 can be any value
    for( std::uint8_t i = 0; i < 128; ++i )
    {
        std::uint8_t unusedBits = i << 1;

        EXPECT_FALSE(nn::edid::detail::IsContinuousFrequency(unusedBits));
        EXPECT_TRUE(nn::edid::detail::IsContinuousFrequency(1 | i));
    }
}

TEST(DisplayInfo, VideoCapability)
{
    // Check Table 8-23 in HDMI 1.4b spec for bit definitions

    // bits 0-6 can be any value
    for( std::uint8_t i = 0; i < 128; ++i )
    {
        nn::edid::DisplayInfo display;
        nn::edid::detail::Initialize(&display);

        std::uint8_t capability[] = { 0xE2, 0x00, static_cast<std::uint8_t>(0x80 | i) };
        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(capability, sizeof(capability), nn::edid::cea861::BlockTag::VideoCapability, &display));
        EXPECT_TRUE(display.isYccQuantizationSelectable);

        capability[2] = i;
        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(capability, sizeof(capability), nn::edid::cea861::BlockTag::VideoCapability, &display));
        EXPECT_FALSE(display.isYccQuantizationSelectable);
    }

    // bits 0-5 can be any value
    for( std::uint8_t i = 0; i < 64; ++i )
    {
        nn::edid::DisplayInfo display;
        nn::edid::detail::Initialize(&display);

        std::uint8_t capability[] = { 0xE2, 0x00, static_cast<std::uint8_t>(0x40 | i) };
        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(capability, sizeof(capability), nn::edid::cea861::BlockTag::VideoCapability, &display));
        EXPECT_TRUE(display.isRgbQuantizationSelectable);

        capability[2] = i;
        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(capability, sizeof(capability), nn::edid::cea861::BlockTag::VideoCapability, &display));
        EXPECT_FALSE(display.isRgbQuantizationSelectable);
    }
}

TEST(DisplayInfo, VendorSpecificVideo)
{
    // Check Table 8-16 in HDMI 1.4b spec for bit definitions

    // bits 4-7 can be any value
    for( std::uint8_t i = 0; i < 0x0F; ++i )
    {
        nn::edid::DisplayInfo display;
        nn::edid::detail::Initialize(&display);

        std::uint8_t unusedBits = i << 4;

        // content types stored in the 9th byte (if available)
        std::uint8_t block[9];
        block[8] = unusedBits;

        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(block, sizeof(block), nn::edid::cea861::BlockTag::VendorSpecific, &display));
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Cinema>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Game>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Graphics>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Photo>());

        block[8] = 0x01 | unusedBits;
        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(block, sizeof(block), nn::edid::cea861::BlockTag::VendorSpecific, &display));
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Cinema>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Game>());
        EXPECT_TRUE(display.contentTypes.Test<nn::edid::ContentType::Graphics>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Photo>());

        block[8] = 0x02 | unusedBits;
        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(block, sizeof(block), nn::edid::cea861::BlockTag::VendorSpecific, &display));
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Cinema>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Game>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Graphics>());
        EXPECT_TRUE(display.contentTypes.Test<nn::edid::ContentType::Photo>());

        block[8] = 0x04 | unusedBits;
        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(block, sizeof(block), nn::edid::cea861::BlockTag::VendorSpecific, &display));
        EXPECT_TRUE(display.contentTypes.Test<nn::edid::ContentType::Cinema>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Game>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Graphics>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Photo>());

        block[8] = 0x08 | unusedBits;
        EXPECT_TRUE(nn::edid::detail::DisplayDataBlockVisitor(block, sizeof(block), nn::edid::cea861::BlockTag::VendorSpecific, &display));
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Cinema>());
        EXPECT_TRUE(display.contentTypes.Test<nn::edid::ContentType::Game>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Graphics>());
        EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Photo>());
    }
}

TEST(DisplayInfo, PhysicalAddressDataBlockVisitor)
{
    nn::edid::detail::PhysicalAddressData data;
    data.address = 0;
    data.isFound = false;

    EXPECT_TRUE(nn::edid::detail::PhysicalAddressDataBlockVisitor(nullptr, 0, nn::edid::cea861::BlockTag::VendorSpecific, &data));

    std::uint8_t block[6] =
    {
        0x66,
        0x00,
        0x00,
        0x00,
        0x12,
        0x34,
    };

    ASSERT_FALSE(nn::edid::detail::PhysicalAddressDataBlockVisitor(block, sizeof(block), nn::edid::cea861::BlockTag::VendorSpecific, &data));
    EXPECT_TRUE(data.isFound);
    EXPECT_EQ(0x1234, data.address);
}
