﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nnt.h>
#include <nn/edid.h>
#include <nnt/edidUtil/testEdid_FullModeInfo.h>

typedef std::vector<nnt::edid::FullModeInfo> ModeList;

static const std::uint8_t s_pEdidRaw[] =
{
    0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00,
    0x4D, 0xD9, 0x09, 0xEC, 0x00, 0x00, 0x00, 0x00,
    0x1B, 0x10, 0x01, 0x03, 0x80, 0x46, 0x28, 0x78,
    0x0A, 0x0D, 0xC9, 0xA0, 0x57, 0x47, 0x98, 0x27,
    0x12, 0x48, 0x4C, 0x20, 0x00, 0x00, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3A,
    0x80, 0x18, 0x71, 0x38, 0x2D, 0x40, 0x58, 0x2C,
    0x45, 0x00, 0xDF, 0xA4, 0x21, 0x00, 0x00, 0x1E,
    0x01, 0x1D, 0x80, 0x18, 0x71, 0x1C, 0x16, 0x20,
    0x58, 0x2C, 0x25, 0x00, 0xDF, 0xA4, 0x21, 0x00,
    0x00, 0x9E, 0x00, 0x00, 0x00, 0xFC, 0x00, 0x48,
    0x44, 0x4D, 0x49, 0x20, 0x4C, 0x4C, 0x43, 0x0A,
    0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xFD,
    0x00, 0x3A, 0x3E, 0x0F, 0x46, 0x0F, 0x00, 0x0A,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x01, 0x83,

    0x02, 0x03, 0x3B, 0x71, 0x53, 0x94, 0x13, 0x05,
    0x03, 0x04, 0x11, 0x10, 0x1F, 0x20, 0x22, 0x3C,
    0x3E, 0x12, 0x02, 0x01, 0x16, 0x15, 0x07, 0x06,
    0x23, 0x09, 0x07, 0x07, 0x83, 0x01, 0x00, 0x00,
    0x7A, 0x03, 0x0C, 0x00, 0x10, 0x00, 0xA8, 0x2D,
    0x21, 0xC0, 0x10, 0x01, 0x41, 0x01, 0x12, 0x20,
    0x28, 0x10, 0x66, 0x00, 0x08, 0x10, 0x76, 0x96,
    0x90, 0xA0, 0xB0, 0x8C, 0x0A, 0xD0, 0x8A, 0x20,
    0xE0, 0x2D, 0x10, 0x10, 0x3E, 0x96, 0x00, 0xDF,
    0xA4, 0x21, 0x00, 0x00, 0x18, 0x8C, 0x0A, 0xD0,
    0x8A, 0x20, 0xE0, 0x2D, 0x10, 0x10, 0x3E, 0x96,
    0x00, 0x30, 0xA4, 0x21, 0x00, 0x00, 0x18, 0x8C,
    0x0A, 0xA0, 0x14, 0x51, 0xF0, 0x16, 0x00, 0x26,
    0x7C, 0x43, 0x00, 0x30, 0xA4, 0x21, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xCE,
};

TEST(ReferenceEdid, DisplayInfo)
{
    nn::edid::Edid edid;
    ASSERT_EQ(nn::edid::Error_None, nn::edid::OpenEdid(&edid, s_pEdidRaw, sizeof(s_pEdidRaw)));

    nn::edid::DisplayInfo display;
    nn::edid::GetDisplayInfo(&display, &edid);

    EXPECT_STREQ("SNY", display.manufacturerCode);
    EXPECT_EQ(0xEC09, display.productCode);
    EXPECT_EQ(0, display.serialNumber);

    int date;
    EXPECT_TRUE(nn::edid::GetDisplayWeekOfManufacture(&date, &display));
    EXPECT_EQ(27, date);
    EXPECT_TRUE(nn::edid::GetDisplayYearOfManufacture(&date, &display));
    EXPECT_EQ(2006, date);

    EXPECT_EQ(nn::edid::VideoInterface_Digital, display.interfaceType);
    EXPECT_EQ(nn::edid::SyncLevel_Digital, display.sync);
    EXPECT_EQ(nn::edid::BlankLevel_Digital, display.blankType);

    EXPECT_EQ(nn::edid::ColorDepth_Undefined, display.depth);
    EXPECT_EQ(nn::edid::ColorType_Undefined, display.colorSupport);

    EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Cinema>());
    EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Game>());
    EXPECT_TRUE(display.contentTypes.Test<nn::edid::ContentType::Graphics>());
    EXPECT_FALSE(display.contentTypes.Test<nn::edid::ContentType::Photo>());
    EXPECT_TRUE(display.syncTypes.IsAllOff());

    EXPECT_FALSE(display.isSrgbDefaultColorSpace);
    EXPECT_FALSE(display.isStandbyModeSupported);
    EXPECT_FALSE(display.isContinuousFrequency);
    EXPECT_FALSE(display.hasVsyncSerrations);
    EXPECT_FALSE(display.isStandbyModeSupported);
    EXPECT_FALSE(display.isSuspendModeSupported);
    EXPECT_FALSE(display.isVeryLowPowerSupported);
    EXPECT_FALSE(display.isYccQuantizationSelectable);
    EXPECT_FALSE(display.isRgbQuantizationSelectable);
}

static bool GetDisplayModes(const nn::edid::DisplayModeInfo* pMode, const nn::edid::DisplayTimingInfo* pTimingInfo, const nn::edid::ImageSizeInfo* pSizeInfo, void* pUserData) NN_NOEXCEPT
{
    ModeList* list = static_cast<ModeList*>(pUserData);

    list->push_back(nnt::edid::FullModeInfo(pMode, pTimingInfo, pSizeInfo));

    return true;
}

TEST(ReferenceEdid, DisplayModeInfo)
{
    nn::edid::Edid edid;
    ASSERT_EQ(nn::edid::Error_None, nn::edid::OpenEdid(&edid, s_pEdidRaw, sizeof(s_pEdidRaw)));

    ModeList list;
    nn::edid::VisitDisplayModes(&edid, GetDisplayModes, &list);

    ASSERT_EQ(25, list.size());

    // Established Timing
    EXPECT_EQ(640, list[0].pMode->width);
    EXPECT_EQ(480, list[0].pMode->height);
    EXPECT_EQ(60.00f, list[0].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[0].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[0].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[0].pMode->stereoMode);
    EXPECT_FALSE(list[0].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[0].pTimingInfo);
    EXPECT_EQ(nullptr, list[0].pSizeInfo);

    // No Standard Timings
    // DTD #1
    EXPECT_EQ(1920, list[1].pMode->width);
    EXPECT_EQ(1080, list[1].pMode->height);
    EXPECT_EQ(60.00f, list[1].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[1].pMode->imageAspectRatio);
    EXPECT_EQ(63.f / 64.f, list[1].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[1].pMode->stereoMode);
    EXPECT_FALSE(list[1].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[1].pTimingInfo);

    EXPECT_EQ(148500, list[1].pTimingInfo->pixelClock);

    EXPECT_EQ(1920, list[1].pTimingInfo->hactive);
    EXPECT_EQ(280, list[1].pTimingInfo->hblank);
    EXPECT_EQ(88, list[1].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(44, list[1].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(1080, list[1].pTimingInfo->vactive);
    EXPECT_EQ(45, list[1].pTimingInfo->vblank);
    EXPECT_EQ(4, list[1].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(5, list[1].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[1].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(list[1].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(list[1].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(list[1].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[1].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Positive, list[1].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Positive, list[1].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[1].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[1].pSizeInfo);

    EXPECT_EQ(735, list[1].pSizeInfo->imageWidth);
    EXPECT_EQ(420, list[1].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[1].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[1].pSizeInfo->verticalBorder);

    // DTD #2

    EXPECT_EQ(1920, list[2].pMode->width);
    EXPECT_EQ(1080, list[2].pMode->height);
    EXPECT_NEAR(60.00f, list[2].pMode->refreshRate, 0.1f);
    EXPECT_EQ(16.f / 9.f, list[2].pMode->imageAspectRatio);
    EXPECT_EQ(63.f / 64.f, list[2].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[2].pMode->stereoMode);
    EXPECT_TRUE(list[2].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[2].pTimingInfo);

    EXPECT_EQ(74250, list[2].pTimingInfo->pixelClock);

    EXPECT_EQ(1920, list[2].pTimingInfo->hactive);
    EXPECT_EQ(280, list[2].pTimingInfo->hblank);
    EXPECT_EQ(88, list[2].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(44, list[2].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(540, list[2].pTimingInfo->vactive);
    EXPECT_EQ(22, list[2].pTimingInfo->vblank);
    EXPECT_EQ(2, list[2].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(5, list[2].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Positive, list[2].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Positive, list[2].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[2].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[2].pSizeInfo);

    EXPECT_EQ(735, list[2].pSizeInfo->imageWidth);
    EXPECT_EQ(420, list[2].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[2].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[2].pSizeInfo->verticalBorder);

    // CEA-861 block
    // Video Data Block
    EXPECT_EQ(1920, list[3].pMode->width);
    EXPECT_EQ(1080, list[3].pMode->height);
    EXPECT_EQ(50.00f, list[3].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[3].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[3].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[3].pMode->stereoMode);
    EXPECT_TRUE(list[3].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[3].pTimingInfo);
    EXPECT_EQ(nullptr, list[3].pSizeInfo);

    EXPECT_EQ(1280, list[4].pMode->width);
    EXPECT_EQ(720, list[4].pMode->height);
    EXPECT_EQ(50.00f, list[4].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[4].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[4].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[4].pMode->stereoMode);
    EXPECT_FALSE(list[4].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[4].pTimingInfo);
    EXPECT_EQ(nullptr, list[4].pSizeInfo);

    EXPECT_EQ(1920, list[5].pMode->width);
    EXPECT_EQ(1080, list[5].pMode->height);
    EXPECT_EQ(60.00f, list[5].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[5].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[5].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[5].pMode->stereoMode);
    EXPECT_TRUE(list[5].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[5].pTimingInfo);
    EXPECT_EQ(nullptr, list[5].pSizeInfo);

    EXPECT_EQ(720, list[6].pMode->width);
    EXPECT_EQ(480, list[6].pMode->height);
    EXPECT_EQ(60.00f, list[6].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[6].pMode->imageAspectRatio);
    EXPECT_EQ(32.f / 27.f, list[6].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[6].pMode->stereoMode);
    EXPECT_FALSE(list[6].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[6].pTimingInfo);
    EXPECT_EQ(nullptr, list[6].pSizeInfo);

    EXPECT_EQ(1280, list[7].pMode->width);
    EXPECT_EQ(720, list[7].pMode->height);
    EXPECT_EQ(60.00f, list[7].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[7].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[7].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[7].pMode->stereoMode);
    EXPECT_FALSE(list[7].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[7].pTimingInfo);
    EXPECT_EQ(nullptr, list[7].pSizeInfo);

    EXPECT_EQ(720, list[8].pMode->width);
    EXPECT_EQ(576, list[8].pMode->height);
    EXPECT_EQ(50.00f, list[8].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[8].pMode->imageAspectRatio);
    EXPECT_EQ(16.f / 15.f, list[8].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[8].pMode->stereoMode);
    EXPECT_FALSE(list[8].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[8].pTimingInfo);
    EXPECT_EQ(nullptr, list[8].pSizeInfo);

    EXPECT_EQ(1920, list[9].pMode->width);
    EXPECT_EQ(1080, list[9].pMode->height);
    EXPECT_EQ(60.00f, list[9].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[9].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[9].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[9].pMode->stereoMode);
    EXPECT_FALSE(list[9].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[9].pTimingInfo);
    EXPECT_EQ(nullptr, list[9].pSizeInfo);

    EXPECT_EQ(1920, list[10].pMode->width);
    EXPECT_EQ(1080, list[10].pMode->height);
    EXPECT_EQ(50.00f, list[10].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[10].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[10].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[10].pMode->stereoMode);
    EXPECT_FALSE(list[10].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[10].pTimingInfo);
    EXPECT_EQ(nullptr, list[10].pSizeInfo);

    EXPECT_EQ(1920, list[11].pMode->width);
    EXPECT_EQ(1080, list[11].pMode->height);
    EXPECT_EQ(24.00f, list[11].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[11].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[11].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[11].pMode->stereoMode);
    EXPECT_FALSE(list[11].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[11].pTimingInfo);
    EXPECT_EQ(nullptr, list[11].pSizeInfo);

    EXPECT_EQ(1920, list[12].pMode->width);
    EXPECT_EQ(1080, list[12].pMode->height);
    EXPECT_EQ(30.00f, list[12].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[12].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[12].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[12].pMode->stereoMode);
    EXPECT_FALSE(list[12].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[12].pTimingInfo);
    EXPECT_EQ(nullptr, list[12].pSizeInfo);

    EXPECT_EQ(1280, list[13].pMode->width);
    EXPECT_EQ(720, list[13].pMode->height);
    EXPECT_EQ(24.00f, list[13].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[13].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[13].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[13].pMode->stereoMode);
    EXPECT_FALSE(list[13].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[13].pTimingInfo);
    EXPECT_EQ(nullptr, list[13].pSizeInfo);

    EXPECT_EQ(1280, list[14].pMode->width);
    EXPECT_EQ(720, list[14].pMode->height);
    EXPECT_EQ(30.00f, list[14].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[14].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[14].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[14].pMode->stereoMode);
    EXPECT_FALSE(list[14].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[14].pTimingInfo);
    EXPECT_EQ(nullptr, list[14].pSizeInfo);

    EXPECT_EQ(720, list[15].pMode->width);
    EXPECT_EQ(576, list[15].pMode->height);
    EXPECT_EQ(50.00f, list[15].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[15].pMode->imageAspectRatio);
    EXPECT_EQ(64.f / 45.f, list[15].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[15].pMode->stereoMode);
    EXPECT_FALSE(list[15].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[15].pTimingInfo);
    EXPECT_EQ(nullptr, list[15].pSizeInfo);

    EXPECT_EQ(720, list[16].pMode->width);
    EXPECT_EQ(480, list[16].pMode->height);
    EXPECT_EQ(60.00f, list[16].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[16].pMode->imageAspectRatio);
    EXPECT_EQ(8.f / 9.f, list[16].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[16].pMode->stereoMode);
    EXPECT_FALSE(list[16].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[16].pTimingInfo);
    EXPECT_EQ(nullptr, list[16].pSizeInfo);

    EXPECT_EQ(640, list[17].pMode->width);
    EXPECT_EQ(480, list[17].pMode->height);
    EXPECT_EQ(60.00f, list[17].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[17].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[17].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[17].pMode->stereoMode);
    EXPECT_FALSE(list[17].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[17].pTimingInfo);
    EXPECT_EQ(nullptr, list[17].pSizeInfo);

    EXPECT_EQ(720, list[18].pMode->width);
    EXPECT_EQ(576, list[18].pMode->height);
    EXPECT_EQ(50.00f, list[18].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[18].pMode->imageAspectRatio);
    EXPECT_EQ(64.f / 45.f, list[18].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[18].pMode->stereoMode);
    EXPECT_TRUE(list[18].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[18].pTimingInfo);
    EXPECT_EQ(nullptr, list[18].pSizeInfo);

    EXPECT_EQ(720, list[19].pMode->width);
    EXPECT_EQ(576, list[19].pMode->height);
    EXPECT_EQ(50.00f, list[19].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[19].pMode->imageAspectRatio);
    EXPECT_EQ(16.f / 15.f, list[19].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[19].pMode->stereoMode);
    EXPECT_TRUE(list[19].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[19].pTimingInfo);
    EXPECT_EQ(nullptr, list[19].pSizeInfo);

    EXPECT_EQ(720, list[20].pMode->width);
    EXPECT_EQ(480, list[20].pMode->height);
    EXPECT_EQ(60.00f, list[20].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[20].pMode->imageAspectRatio);
    EXPECT_EQ(32.f / 27.f, list[20].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[20].pMode->stereoMode);
    EXPECT_TRUE(list[20].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[20].pTimingInfo);
    EXPECT_EQ(nullptr, list[20].pSizeInfo);

    EXPECT_EQ(720, list[21].pMode->width);
    EXPECT_EQ(480, list[21].pMode->height);
    EXPECT_EQ(60.00f, list[21].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[21].pMode->imageAspectRatio);
    EXPECT_EQ(8.f / 9.f, list[21].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[21].pMode->stereoMode);
    EXPECT_TRUE(list[21].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[21].pTimingInfo);
    EXPECT_EQ(nullptr, list[21].pSizeInfo);

    // DTD #1
    EXPECT_EQ(720, list[22].pMode->width);
    EXPECT_EQ(480, list[22].pMode->height);
    EXPECT_NEAR(59.94f, list[22].pMode->refreshRate, 0.002f);
    EXPECT_EQ(1.5f, list[22].pMode->imageAspectRatio);
    EXPECT_EQ(7.f / 6.f, list[22].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[22].pMode->stereoMode);
    EXPECT_FALSE(list[22].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[22].pTimingInfo);

    EXPECT_EQ(27000, list[22].pTimingInfo->pixelClock);

    EXPECT_EQ(720, list[22].pTimingInfo->hactive);
    EXPECT_EQ(138, list[22].pTimingInfo->hblank);
    EXPECT_EQ(16, list[22].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(62, list[22].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(480, list[22].pTimingInfo->vactive);
    EXPECT_EQ(45, list[22].pTimingInfo->vblank);
    EXPECT_EQ(9, list[22].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(6, list[22].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[22].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(list[22].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(list[22].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(list[22].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[22].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Negative, list[22].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Negative, list[22].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[22].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[22].pSizeInfo);

    EXPECT_EQ(735, list[22].pSizeInfo->imageWidth);
    EXPECT_EQ(420, list[22].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[22].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[22].pSizeInfo->verticalBorder);

    // DTD #2
    EXPECT_EQ(720, list[23].pMode->width);
    EXPECT_EQ(480, list[23].pMode->height);
    EXPECT_NEAR(59.94f, list[23].pMode->refreshRate, 0.002f);
    EXPECT_EQ(1.5f, list[23].pMode->imageAspectRatio);
    EXPECT_EQ(8.f / 9.f, list[23].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[23].pMode->stereoMode);
    EXPECT_FALSE(list[23].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[23].pTimingInfo);

    EXPECT_EQ(27000, list[23].pTimingInfo->pixelClock);

    EXPECT_EQ(720, list[23].pTimingInfo->hactive);
    EXPECT_EQ(138, list[23].pTimingInfo->hblank);
    EXPECT_EQ(16, list[23].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(62, list[23].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(480, list[23].pTimingInfo->vactive);
    EXPECT_EQ(45, list[23].pTimingInfo->vblank);
    EXPECT_EQ(9, list[23].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(6, list[23].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[23].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(list[23].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(list[23].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(list[23].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[23].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Negative, list[23].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Negative, list[23].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[23].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[23].pSizeInfo);

    EXPECT_EQ(560, list[23].pSizeInfo->imageWidth);
    EXPECT_EQ(420, list[23].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[23].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[23].pSizeInfo->verticalBorder);

    // DTD #3
    EXPECT_EQ(1440, list[24].pMode->width);
    EXPECT_EQ(480, list[24].pMode->height);
    EXPECT_NEAR(60.00f, list[24].pMode->refreshRate, 0.1f);
    EXPECT_EQ(3.0f, list[24].pMode->imageAspectRatio);
    EXPECT_EQ(4.f / 9.f, list[24].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[24].pMode->stereoMode);
    EXPECT_TRUE(list[24].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[24].pTimingInfo);

    EXPECT_EQ(27000, list[24].pTimingInfo->pixelClock);

    EXPECT_EQ(1440, list[24].pTimingInfo->hactive);
    EXPECT_EQ(276, list[24].pTimingInfo->hblank);
    EXPECT_EQ(38, list[24].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(124, list[24].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(240, list[24].pTimingInfo->vactive);
    EXPECT_EQ(22, list[24].pTimingInfo->vblank);
    EXPECT_EQ(4, list[24].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(3, list[24].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[24].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(list[24].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(list[24].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(list[24].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[24].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Negative, list[24].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Negative, list[24].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[24].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[24].pSizeInfo);

    EXPECT_EQ(560, list[24].pSizeInfo->imageWidth);
    EXPECT_EQ(420, list[24].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[24].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[24].pSizeInfo->verticalBorder);
} // NOLINT(impl/function_size)

TEST(ReferenceEdid, GetSourcePhysicalAddress)
{
    nn::edid::Edid edid;
    ASSERT_EQ(nn::edid::Error_None, nn::edid::OpenEdid(&edid, s_pEdidRaw, sizeof(s_pEdidRaw)));

    std::uint16_t address;
    ASSERT_TRUE(nn::edid::GetSourcePhysicalAddress(&address, &edid));
    EXPECT_EQ(0x1000, address);
}
