﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nnt.h>
#include <nn/edid.h>
#include <nnt/edidUtil/testEdid_FullModeInfo.h>

typedef std::vector<nnt::edid::FullModeInfo> ModeList;

static const std::uint8_t s_pEdidRaw[] =
{
    0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00,
    0x4D, 0x10, 0xDB, 0x0F, 0x00, 0x00, 0x00, 0x00,
    0xFF, 0x11, 0x01, 0x03, 0x80, 0x52, 0x2E, 0x78,
    0x2A, 0x1B, 0xBE, 0xA2, 0x55, 0x34, 0xB3, 0x26,
    0x14, 0x4A, 0x52, 0x20, 0x00, 0x00, 0x1D, 0x40,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x8F, 0x0A,
    0xD0, 0x8A, 0x20, 0xE0, 0x2D, 0x10, 0x10, 0x3E,
    0x96, 0x00, 0x20, 0x58, 0x32, 0x00, 0x00, 0x18,
    0x8F, 0x0A, 0xD0, 0x8A, 0x20, 0xE0, 0x2D, 0x10,
    0x10, 0x3E, 0x96, 0x00, 0x20, 0x58, 0x32, 0x00,
    0x00, 0x18, 0x00, 0x00, 0x00, 0xFC, 0x00, 0x53,
    0x48, 0x41, 0x52, 0x50, 0x20, 0x48, 0x44, 0x4D,
    0x49, 0x0A, 0x20, 0x20, 0x00, 0x00, 0x00, 0xFD,
    0x00, 0x37, 0x4C, 0x0F, 0x41, 0x0B, 0x00, 0x0A,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x01, 0x99,

    0x02, 0x03, 0x16, 0x71, 0x42, 0x01, 0x03, 0x23,
    0x09, 0x07, 0x01, 0x83, 0x01, 0x00, 0x00, 0x66,
    0x03, 0x0C, 0x00, 0x10, 0x00, 0x80, 0xC4, 0x09,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x8F, 0x0A, 0xD0, 0x8A, 0x20, 0xE0, 0x2D, 0x10,
    0x10, 0x3E, 0x96, 0x00, 0x20, 0x58, 0x32, 0x00,
    0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xCE,
};

TEST(Only480p, DisplayInfo)
{
    nn::edid::Edid edid;
    ASSERT_EQ(nn::edid::Error_None, nn::edid::OpenEdid(&edid, s_pEdidRaw, sizeof(s_pEdidRaw)));

    nn::edid::DisplayInfo display;
    nn::edid::GetDisplayInfo(&display, &edid);

    EXPECT_STREQ("SHP", display.manufacturerCode);
    EXPECT_EQ(0, display.manufacturerCode[3]);
    EXPECT_EQ(0x0FDB, display.productCode);
    EXPECT_EQ(0, display.serialNumber);

    int date;
    EXPECT_FALSE(nn::edid::GetDisplayWeekOfManufacture(&date, &display));
    EXPECT_TRUE(nn::edid::GetDisplayYearOfManufacture(&date, &display));
    EXPECT_EQ(2007, date);

    EXPECT_EQ(nn::edid::VideoInterface_Digital, display.interfaceType);
    EXPECT_EQ(nn::edid::SyncLevel_Digital, display.sync);
    EXPECT_EQ(nn::edid::BlankLevel_Digital, display.blankType);

    EXPECT_EQ(nn::edid::ColorDepth_Undefined, display.depth);
    EXPECT_EQ(nn::edid::ColorType_Undefined, display.colorSupport);

    EXPECT_TRUE(display.contentTypes.IsAllOff());
    EXPECT_TRUE(display.syncTypes.IsAllOff());

    EXPECT_FALSE(display.isSrgbDefaultColorSpace);
    EXPECT_FALSE(display.isStandbyModeSupported);
    EXPECT_FALSE(display.isContinuousFrequency);
    EXPECT_FALSE(display.hasVsyncSerrations);
    EXPECT_FALSE(display.isStandbyModeSupported);
    EXPECT_FALSE(display.isSuspendModeSupported);
    EXPECT_TRUE(display.isVeryLowPowerSupported);
    EXPECT_FALSE(display.isYccQuantizationSelectable);
    EXPECT_FALSE(display.isRgbQuantizationSelectable);
}

static bool GetDisplayModes(const nn::edid::DisplayModeInfo* pMode, const nn::edid::DisplayTimingInfo* pTimingInfo, const nn::edid::ImageSizeInfo* pSizeInfo, void* pUserData) NN_NOEXCEPT
{
    ModeList* list = static_cast<ModeList*>(pUserData);

    list->push_back(nnt::edid::FullModeInfo(pMode, pTimingInfo, pSizeInfo));

    return true;
}

TEST(Only480p, DisplayModeInfo)
{
    nn::edid::Edid edid;
    ASSERT_EQ(nn::edid::Error_None, nn::edid::OpenEdid(&edid, s_pEdidRaw, sizeof(s_pEdidRaw)));

    ModeList list;
    nn::edid::VisitDisplayModes(&edid, GetDisplayModes, &list);

    ASSERT_EQ(8, list.size());

    // Established Timing
    EXPECT_EQ(640, list[0].pMode->width);
    EXPECT_EQ(480, list[0].pMode->height);
    EXPECT_EQ(60.00f, list[0].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[0].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[0].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[0].pMode->stereoMode);
    EXPECT_FALSE(list[0].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[0].pTimingInfo);
    EXPECT_EQ(nullptr, list[0].pSizeInfo);

    // Standard Timing
    EXPECT_EQ(480, list[1].pMode->width);
    EXPECT_EQ(640, list[1].pMode->height);
    EXPECT_EQ(60.00f, list[1].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[1].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[1].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[1].pMode->stereoMode);
    EXPECT_FALSE(list[1].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[1].pTimingInfo);
    EXPECT_EQ(nullptr, list[1].pSizeInfo);

    // DTD #1
    EXPECT_EQ(720, list[2].pMode->width);
    EXPECT_EQ(480, list[2].pMode->height);
    EXPECT_NEAR(60.00f, list[2].pMode->refreshRate, 0.05f);
    EXPECT_EQ(1.5f, list[2].pMode->imageAspectRatio);
    EXPECT_EQ(8.f / 9.f, list[2].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[2].pMode->stereoMode);
    EXPECT_FALSE(list[2].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[2].pTimingInfo);

    EXPECT_EQ(27030, list[2].pTimingInfo->pixelClock);

    EXPECT_EQ(720, list[2].pTimingInfo->hactive);
    EXPECT_EQ(138, list[2].pTimingInfo->hblank);
    EXPECT_EQ(16, list[2].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(62, list[2].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(480, list[2].pTimingInfo->vactive);
    EXPECT_EQ(45, list[2].pTimingInfo->vblank);
    EXPECT_EQ(9, list[2].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(6, list[2].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[2].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Negative, list[2].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Negative, list[2].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[2].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[2].pSizeInfo);

    EXPECT_EQ(800, list[2].pSizeInfo->imageWidth);
    EXPECT_EQ(600, list[2].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[2].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[2].pSizeInfo->verticalBorder);

    // DTD #2

    EXPECT_EQ(720, list[3].pMode->width);
    EXPECT_EQ(480, list[3].pMode->height);
    EXPECT_NEAR(60.00f, list[3].pMode->refreshRate, 0.05f);
    EXPECT_EQ(1.5f, list[3].pMode->imageAspectRatio);
    EXPECT_EQ(8.f / 9.f, list[3].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[3].pMode->stereoMode);
    EXPECT_FALSE(list[3].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[3].pTimingInfo);

    EXPECT_EQ(27030, list[3].pTimingInfo->pixelClock);

    EXPECT_EQ(720, list[3].pTimingInfo->hactive);
    EXPECT_EQ(138, list[3].pTimingInfo->hblank);
    EXPECT_EQ(16, list[3].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(62, list[3].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(480, list[3].pTimingInfo->vactive);
    EXPECT_EQ(45, list[3].pTimingInfo->vblank);
    EXPECT_EQ(9, list[3].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(6, list[3].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[3].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(list[3].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(list[3].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(list[3].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[3].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Negative, list[3].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Negative, list[3].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[3].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[3].pSizeInfo);

    EXPECT_EQ(800, list[3].pSizeInfo->imageWidth);
    EXPECT_EQ(600, list[3].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[3].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[3].pSizeInfo->verticalBorder);

    // CEA-861 block
    // Video Data Block
    EXPECT_EQ(640, list[4].pMode->width);
    EXPECT_EQ(480, list[4].pMode->height);
    EXPECT_EQ(60.00f, list[4].pMode->refreshRate);
    EXPECT_EQ(4.f / 3.f, list[4].pMode->imageAspectRatio);
    EXPECT_EQ(1.f, list[4].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[4].pMode->stereoMode);
    EXPECT_FALSE(list[4].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[4].pTimingInfo);
    EXPECT_EQ(nullptr, list[4].pSizeInfo);

    EXPECT_EQ(720, list[5].pMode->width);
    EXPECT_EQ(480, list[5].pMode->height);
    EXPECT_EQ(60.00f, list[5].pMode->refreshRate);
    EXPECT_EQ(16.f / 9.f, list[5].pMode->imageAspectRatio);
    EXPECT_EQ(32.f / 27.f, list[5].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[5].pMode->stereoMode);
    EXPECT_FALSE(list[5].pMode->isInterlaced);
    EXPECT_EQ(nullptr, list[5].pTimingInfo);
    EXPECT_EQ(nullptr, list[5].pSizeInfo);

    // DTD #1
    // for some reason, this is garbage data, but it's correct...
    EXPECT_EQ(0, list[6].pMode->width);
    EXPECT_EQ(0, list[6].pMode->height);
    EXPECT_EQ(0.f, list[6].pMode->refreshRate);
    EXPECT_EQ(0.f, list[6].pMode->imageAspectRatio);
    EXPECT_EQ(0.f, list[6].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[6].pMode->stereoMode);
    EXPECT_FALSE(list[6].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[6].pTimingInfo);

    EXPECT_EQ(25000, list[6].pTimingInfo->pixelClock);

    EXPECT_EQ(0, list[6].pTimingInfo->hactive);
    EXPECT_EQ(0, list[6].pTimingInfo->hblank);
    EXPECT_EQ(0, list[6].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(0, list[6].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(0, list[6].pTimingInfo->vactive);
    EXPECT_EQ(0, list[6].pTimingInfo->vblank);
    EXPECT_EQ(0, list[6].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(0, list[6].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[6].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_TRUE(list[6].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_FALSE(list[6].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_TRUE(list[6].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[6].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Undefined, list[6].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Undefined, list[6].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[6].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[6].pSizeInfo);

    EXPECT_EQ(0, list[6].pSizeInfo->imageWidth);
    EXPECT_EQ(0, list[6].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[6].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[6].pSizeInfo->verticalBorder);

    // DTD #2
    EXPECT_EQ(720, list[7].pMode->width);
    EXPECT_EQ(480, list[7].pMode->height);
    EXPECT_NEAR(60.00f, list[7].pMode->refreshRate, 0.05f);
    EXPECT_EQ(1.5f, list[7].pMode->imageAspectRatio);
    EXPECT_EQ(8.f / 9.f, list[7].pMode->pixelAspectRatio);
    EXPECT_EQ(nn::edid::StereoMode_None, list[7].pMode->stereoMode);
    EXPECT_FALSE(list[7].pMode->isInterlaced);

    ASSERT_NE(nullptr, list[7].pTimingInfo);

    EXPECT_EQ(27030, list[7].pTimingInfo->pixelClock);

    EXPECT_EQ(720, list[7].pTimingInfo->hactive);
    EXPECT_EQ(138, list[7].pTimingInfo->hblank);
    EXPECT_EQ(16, list[7].pTimingInfo->horizontalFrontPorch);
    EXPECT_EQ(62, list[7].pTimingInfo->horizontalSyncPulse);

    EXPECT_EQ(480, list[7].pTimingInfo->vactive);
    EXPECT_EQ(45, list[7].pTimingInfo->vblank);
    EXPECT_EQ(9, list[7].pTimingInfo->verticalFrontPorch);
    EXPECT_EQ(6, list[7].pTimingInfo->verticalSyncPulse);

    EXPECT_FALSE(list[7].pTimingInfo->analogSync.Test<nn::edid::SyncType::BipolarCompositeSync>());
    EXPECT_FALSE(list[7].pTimingInfo->analogSync.Test<nn::edid::SyncType::CompositeSync>());
    EXPECT_TRUE(list[7].pTimingInfo->analogSync.Test<nn::edid::SyncType::SeparateSync>());
    EXPECT_FALSE(list[7].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnGreen>());
    EXPECT_FALSE(list[7].pTimingInfo->analogSync.Test<nn::edid::SyncType::SyncOnRgb>());

    EXPECT_EQ(nn::edid::Polarity_Negative, list[7].pTimingInfo->vsyncPolarity);
    EXPECT_EQ(nn::edid::Polarity_Negative, list[7].pTimingInfo->hsyncPolarity);
    EXPECT_FALSE(list[7].pTimingInfo->hasSyncSerrations);

    ASSERT_NE(nullptr, list[7].pSizeInfo);

    EXPECT_EQ(800, list[7].pSizeInfo->imageWidth);
    EXPECT_EQ(600, list[7].pSizeInfo->imageHeight);
    EXPECT_EQ(0, list[7].pSizeInfo->horizontalBorder);
    EXPECT_EQ(0, list[7].pSizeInfo->verticalBorder);
} // NOLINT(impl/function_size)

TEST(Only480p, GetSourcePhysicalAddress)
{
    nn::edid::Edid edid;
    ASSERT_EQ(nn::edid::Error_None, nn::edid::OpenEdid(&edid, s_pEdidRaw, sizeof(s_pEdidRaw)));

    std::uint16_t address;
    ASSERT_TRUE(nn::edid::GetSourcePhysicalAddress(&address, &edid));
    EXPECT_EQ(0x1000, address);
}
