﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket.h>

#include "testFixtures.h"
#include "utils.h"

namespace {
    const int DummyUserData = 27;

    bool sockOptCalled = false;
    bool openCalled = false;
    bool closeCalled = false;
}

int sockOptCallback(void *clientp, curl_socket_t curlfd, curlsocktype purpose)
{
    sockOptCalled = true;

    // Check that we got the correct user data
    int *userData = reinterpret_cast<int*>(clientp);
    EXPECT_EQ(DummyUserData, *userData);

    // Don't actually modify the socket, just return
    return CURL_SOCKOPT_OK;
}

curl_socket_t openSocketCallback(void *clientp, curlsocktype purpose, curl_sockaddr *address)
{
    openCalled = true;

    // Check that we got the correct user data
    int *userData = reinterpret_cast<int*>(clientp);
    EXPECT_EQ(DummyUserData, *userData);

    // Get socket parameters
    nn::socket::Family domain = static_cast<nn::socket::Family>(address->family);
    nn::socket::Type type = static_cast<nn::socket::Type>(address->socktype);
    nn::socket::Protocol protocol = static_cast<nn::socket::Protocol>(address->protocol);

    // Create the socket
    int socketDescriptor = nn::socket::Socket(domain, type, protocol);
    if (socketDescriptor == -1)
    {
        return CURL_SOCKET_BAD;
    }

    return socketDescriptor;
}

int closeSocketCallback(void *clientp, curl_socket_t item)
{
    closeCalled = true;

    // Check that we got the correct user data
    int *userData = reinterpret_cast<int*>(clientp);
    EXPECT_EQ(DummyUserData, *userData);

    // Close the socket
    int result = nn::socket::Close(item);
    if (result == -1)
    {
        return 1;
    }

    return 0;
}

TEST_F(CurlTest, SocketCallbackTest)
{
    CURLcode cResult;

    const char *testUrl = "http://natf.com/headers.php";

    // Don't care about the actual data returned from this request
    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, NullWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    // Set up the socket callbacks and userdata pointers
    cResult = curl_easy_setopt(cHandle, CURLOPT_SOCKOPTFUNCTION, sockOptCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_SOCKOPTDATA, &DummyUserData);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_OPENSOCKETFUNCTION, openSocketCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_OPENSOCKETDATA, &DummyUserData);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_CLOSESOCKETFUNCTION, closeSocketCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_CLOSESOCKETDATA, &DummyUserData);
    ASSERT_EQ(CURLE_OK, cResult);

    // Force the connection to close after getting the response
    cResult = curl_easy_setopt(cHandle, CURLOPT_FORBID_REUSE, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    // Set the URL and perform the request
    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    // Check the response code
    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    // Verify that the callbacks were called
    ASSERT_TRUE(sockOptCalled);
    ASSERT_TRUE(openCalled);
    ASSERT_TRUE(closeCalled);
}
